# High level interface to Launchpad API to manage credentials
# Derived from bits of ubuntutools in ubuntu-dev-tools and Arsenal

import os, sys, socket
import httplib2

from launchpadlib.credentials import Credentials
from launchpadlib.launchpad import Launchpad, STAGING_SERVICE_ROOT, EDGE_SERVICE_ROOT
from launchpadlib.errors import HTTPError

from lpltoolkit.debug import *

class LaunchpadService:
    ''' High level interface to the launchpad service API '''

    def __init__(self, config_path=None, cache_path=None, consumer='lpltoolkit'):
        ''' Construct Launchpad object, including obtaining launchpad credentials '''
        self._project_name = None
        self._configdir = None
        self._cachedir = None
        self._service_root = 'https://api.launchpad.net/beta/'
        self.name = consumer
        self.project = None
        self.launchpad = None

        ''' Debugging '''
        if "LPDEBUG" in os.environ:
            httplib2.debuglevel = os.getenv("LPDEBUG", None)
            sys.stdout = DebugStdOut()

        ''' Service root '''
        if "LPSTAGING" in os.environ:
            self._service_root = STAGING_SERVICE_ROOT
        elif "LPEDGE" in os.environ:
            self._service_root = EDGE_SERVICE_ROOT

        home = os.path.expanduser('~')

        ''' Config dir '''
        if "LPCONFIG" in os.environ:
            self._confdir = os.getenv("LPCONFIG", None)
        elif config_path:
            self._confdir = config_path
        else:
            self._confdir = os.path.join(home, '.config', self.name)
            if not os.path.exists(self._confdir):
                dbg("Creating config directory %s" %(self._confdir))
                os.makedirs(self._confdir)
        assert os.path.exists(self._confdir)

        ''' Cache dir '''
        if "LPCACHE" in os.environ:
            self._cachedir = os.getenv("LPCACHE", None)
        elif cache_path:
            self._cachedir = cache_path
        else:
            self._cachedir = os.path.join(home, '.cache', self.name)
        if not os.path.exists(self._cachedir):
            dbg("Creating cache directory %s" %(self._cachedir))
            os.makedirs(self._cachedir,0700)
        assert os.path.exists(self._cachedir)

        try:
            self._get_creds()
        except:
            # Launchpad could not be accessed.  Maybe better luck next time.
            sys.exit(7)
        return None

    def _get_creds(self):
        ''' Credentials '''
        retrieve_credentials = False
        self._credentials_file = os.path.join(self._confdir, "lp-credentials.txt")
        if not os.path.exists(self._credentials_file):
            retrieve_credentials = True
        else:
            try:
                credentials = Credentials()
                dbg("Loading credentials from %s" %(self._credentials_file))
                credentials.load(open(self._credentials_file))
                self.launchpad = Launchpad(credentials, self._service_root, self._cachedir)
            except socket.error:
                excType, excValue, excTraceback = sys.exc_info()
                dbg("  %s\n" %(excValue.response))
                dbg("  %s\n" %(excValue.content))
                sys.stderr.write("Launchpad service connection timed out")
                sys.exit(1)
                
            except HTTPError:
                # Can I use e.code here?
                excType, excValue, excTraceback = sys.exc_info()
                err("Failed to retrieve credentials from Launchpad")
                dbg("  %s\n" %(excValue.response))
                dbg("  %s\n" %(excValue.content))
                retrieve_credentials = True

        if retrieve_credentials:
            dbg("Retrieving token and login")
            try:
                self.launchpad = Launchpad.get_token_and_login(self.name, self._service_root, self._cachedir)
                credfd = open(self._credentials_file, "w")
                os.chmod(self._credentials_file, 0600)
                dbg("Saving retrieved credentials to %s" %(self._credentials_file))
                self.launchpad.credentials.save(credfd)
                credfd.close()
            except:
                sys.stderr.write("Could not set up Launchpad")
                sys.exit(1)

        return True

    def reset(self):
        self._get_creds()

        if self._project_name:
            dbg("Reloading project %s" %(self._project_name))
            self.load_project(self._project_name)

    def load_project(self, project):
        try:
            self.project = self.launchpad.load("%s%s" % (self.launchpad._root_uri, project))
        except:
            err("Exception cast loading project %s" % (project))
            self.project = None

        return self.project

    def load_distro_series(self, distro, series):
        try:
            self.distro_series = self.launchpad.load("%s%s/%s" % (self.launchpad._root_uri, distro, series))
        except:
            err("Exception cast loading distro %s and series %s" % (distro, series))
            self.distro_series = None
            sys.exit(1)

        return self.distro_series
