import backend as real_backend
import math
import random

backend = None

#============================ SCREEN FUNCTIONS ==============================

class Screen:
    """
    Defines the graphics window in which objects can be created
    """
    def __init__(self, width=640, height=480,
                 title="Gasp", background=(255,255,255), back_end=None):
        global backend
        backend = back_end
        self.width = width
        self.height = height
        self.title = title
        self.background = background

        # backend.check_screen_atts(self)  # checks the variables
        backend.create_screen(self)

    def __str__(self):
        return 'Screen: \n width = %d \n height = %d \n title = %s \n' % \
               (self.width, self.height, self.title)


def begin_graphics(width=640, height=480, title="Gasp",
                   background=(255,255,255), back_end=real_backend):
    """
    Initialize a graphics window in GASP
    """
    Screen(width, height, title, background, back_end)


def end_graphics():
    """
    Close the graphics window
    """
    backend.end()


def clear_screen():
    """
    Removes all objects from the graphics window
    """
    backend.clear_screen()


def remove_from_screen(obj):
    """
    Removes a specified object from the graphics window
    """
    backend.remove(obj)


#================================ Point =====================================

class Point:
    """
    Creates a point object with an x and y attribute
    """
    def __init__(self, *args):
        if len(args) == 1:
            self.x, self.y = args[0]
        else:
            self.x, self.y = args
            
    def __str__(self):
        return "A Point instance at (%i, %i) \n" %(self.x, self.y)
    
    def distance(self, other):
        ''' The distance between two points '''
        return math.sqrt(float((other.x - self.x))**2 +
                                (float(other.y - self.y))**2) 
    

def make_it_a_point(point):
    """
    Takes something and makes it into a point if it isn't one already
    """
    if isinstance(point, Point):
        return point
    else:
        return Point(point)


# =============================== Shapes ====================================

class Plot:
    """
    Allows screen objects to be placed in the graphics window
    """
    def __init__(self, pos, color=(0,0,0), size=1):
        self.pos = make_it_a_point(pos)
        self.color = color
        self.size = size
        backend.plot(self)

    
class Line:
    """
    Allows the creation of lines in the graphics window
    """
    def __init__(self, start, end, color=(0,0,0)):
        self.start = make_it_a_point(start)
        self.end = make_it_a_point(end)
        self.color = color

        backend.create_line(self)

    def __repr__(self):
        return "A Line instance from (%d, %d) to (%d, %d)" % \
                (self.start.x, self.start.y, self.end.x, self.end.y)
    

class Shape:
    """
    Allows the creation of shapes in the graphics window
    """
    def __init__(self, center, filled=False, color=(0,0,0), thickness=1):
        self.center = make_it_a_point(center)
        self.filled = filled
        self.color = color
        self.thickness = thickness
        self.angle = 0

    def move_to(self, center):
        """
        Moves a shape to a designated point
        """
        self.center = make_it_a_point(center)
        backend.move_to(self, self.center)

    def move_by(self, dx=0, dy=0):
        """
        Moves a shape by specified X and Y amounts
        """
        self.center.x += dx
        self.center.y += dy
        backend.move_to(self, self.center)

    def rotate_to(self, angle):
        """
        Rotates a shape to an angle
        """
        change = angle - self.angle
        self.angle = angle
        backend.rotate_by(self, change)

    def rotate_by(self, angle):
        """
        Rotates a shape by an angle
        """
        self.angle += angle
        backend.rotate_by(self, angle)


class Box(Shape):
    """
    Allows the creation of squares and rectangles in the graphics window
    """
    def __init__(self, lower_left_corner, width, height,
                 filled=False, color=(0,0,0), thickness=1):
        self.center = Point((int(lower_left_corner[0] + width/2.0),
                             int(lower_left_corner[1] + height/2.0)))
        self.height = height
        self.width = width
        Shape.__init__(self, self.center, filled, color, thickness)
        
        backend.create_box(self)

    def move_to(self, lower_left_corner):
        new_center = Point((int(lower_left_corner[0] + self.width/2.0),
                            int(lower_left_corner[1] + self.height/2.0)))
        Shape.move_to(self, new_center)

    def __repr__(self):
        return "Box instance at (%d, %d) with width %d and height %d" % \
               (self.center.x, self.center.y, self.width, self.height)


class Polygon(Shape):
    """
    Allows the creation of polygons in the graphics window
    """
    def __init__(self, points, filled=False, color=(0,0,0)):
        self.points = points
        x_values = []
        y_values = []    
        for point in self.points:
            x_values.append(point[0])
            y_values.append(point[1])
        self.height = (max(y_values) - min(y_values))
        self.width = (max(x_values) - min(x_values))
        self.center = Point(self.width/2 + min(x_values), self.height/2 +
                                                              min(y_values))
        Shape.__init__(self, self.center, filled, color)
        backend.create_polygon(self)
            
    def __repr__(self):
        return "Polygon instance at (%i, %i) with the points:\n %t" % \
                (self.center.x , self.center.y, self.points)
        

class Circle(Shape):
    """
    Allows the creation of circles in the graphics window
    """
    def __init__(self, center, radius,
                 filled=False, color=(0,0,0), thickness=1):
        Shape.__init__(self, center, filled, color, thickness)
        self.radius = radius
        
        backend.create_circle(self)

    def __repr__(self):
        return "Circle instance at (%d, %d) with radius %d" % \
               (self.center.x, self.center.y, self.radius)


class Arc(Shape):
    """
    Allows the creation of arcs in the graphics window
    """
    def __init__(self, center, radius, start_angle, end_angle,
                 filled=False, color=(0,0,0), thickness=1):
        Shape.__init__(self, center, filled, color, thickness)
        self.radius = radius
        self.start_angle = start_angle
        self.end_angle = end_angle
        
        backend.create_arc(self)

    def __repr__(self):
        return "Arc instance at (%d, %d) with start angle %d and end angle %d" \
             % (self.center.x, self.center.y, self.start_angle, self.end_angle)


class Oval(Shape):
    """
    Allows the creation of circles in the graphics window
    """
    def __init__(self, center, width, height,
                 filled=False, color=(0,0,0), thickness=1):
        Shape.__init__(self, center, filled, color, thickness)
        self.width = width
        self.height = height

        backend.create_oval(self)

    def __repr__(self):
        return "Oval instance at (%d, %d) with width %d and height %d" % \
               (self.center.x, self.center.y, self.width, self.height)

        
class Image(Shape):
    """
    Allows the placing of images in the graphics window
    """
    def __init__(self, file_path, center, width=0, height=0):
        self.name = ''.join(file_path.split('.')[:-1]).split('/')[-1]
        self.path_name = file_path
        self.x = center[0]
        self.y = center[1] 
        self.center = Point(center)
        self.width = width
        self.height = height
        backend.create_image(self)

    def __repr__(self):
        return "Image instance at (%i, %i) from the file %s" % \
               (self.center.x, self.center.y, self.path_name)

 
def move_to(obj, pos):
    """
    Moves a screen object to a specified location
    """
    obj.move_to(pos)


def move_by(obj, dx, dy):
    """
    Moves a screen object by specified X and Y amounts
    """
    obj.move_by(dx, dy)


def rotate_to(obj, angle):
    """
    Rotates a screen object to a specified angle
    """
    obj.rotate_to(angle)


def rotate_by(obj, angle):
    """
    Rotates a screen object by a specified angle
    """
    obj.rotate_by(angle)


# =============================== Text ======================================

class Text:
    """
    Allows the creation of text in the graphics window
    """
    def __init__(self, text, pos, color=(0,0,0), size=12):
        self.text = text
        self.pos = make_it_a_point(pos)
        self.color = color
        self.size = size

        backend.create_text(self)


# =============================== Sound =====================================

class Sound:
    """
    Allows the creation and playback of sounds
    """
    def __init__(self, file_path):
        self.path = file_path
        self.name = ''.join(file_path.split('.')[:-1]).split('/')[-1]
        backend.create_sound(self)

    def play(self):
        backend.play_sound(self)

    def stop(self):
        backend.stop_sound(self)
        

def create_sound(file_path):
    """
    Creates a sound object from a file
    """
    return Sound(file_path)
        

def play_sound(sound):
    """
    Starts to play a specified sound object
    """
    sound.play()


def stop_sound(sound):
    """
    Stops the playback of a specified sound object
    """
    sound.stop()


# =========================== Event Functions ==============================

def mouse_position(): 
    """
    Returns a Point() at the mouse's current position
    """
    return Point(backend.mouse_pos())


def mouse_buttons():  
    """
    Returns a dictionary of the current state of the mouse buttons
    """
    return backend.mouse_pressed()


def keys_pressed(): 
    """
    Returns a list of all of the keys being pressed
    """
    return backend.keys()


def key_pressed(key, wait_for_upstroke=False): 
    """
    Returns a boolean of whether a key has been pressed
    """
    if wait_for_upstroke:
        return key in backend.waiting_keys()
    else:
        return key in keys_pressed()


    
# ============================= GASP TOOLS ==================================

def screen_shot(filename):
    """
    Saves a screenshot to a specified file name
    """
    backend.screen_picture(filename)

random_choice = random.choice


def random_between(num1, num2):
    if (num1 == num2):
        return num1
    elif (num1 > num2):
        return random.randint(num2, num1)
    else:
        return random.randint(num1, num2)

read_string = raw_input


def read_number(prompt='Please enter a number: '):
    """
    Prompts the user to enter a number
    """
    while True:
        result = input(prompt)
        if isinstance(result, (float, int)):
            return result
        print "But that wasn't a number!"


def read_yesorno(prompt='Yes or no? '):
    """
    Asks the user to enter yes or no
    """
    while True:
        result = raw_input(prompt)
        result = result.lower()
        if result=='yes' or result=='y': return True
        if result=='no' or result=='n': return False
        print "Please answer yes or no."

# =============================== Time =====================================

def set_speed(speed):
    """
    Sets program frame rate in frames per second 
    """
    backend.set_frame_rate(speed)


def update_when(event_type):
    """
    Event Types include 
    'key_pressed' 
    'mouse_clicked'
    'next_tick'
    """
    backend.update_when(event_type)


def sleep(duration):
    """
    Sleeps for a length of time
    """
    backend.sleep(duration)


# ============================== main =====================================

if __name__ == "__main__":
    """
    If api.py is run independently, this runs the test suite
    """
    import doctest
    doctest.testmod()
