
##############################################################################
#
# Copyright (c) 2003-2018 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2018 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
Test suite for the escript.symbolic module

:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""

__author__="Cihan Altinay"

from esys.escript import *
from esys.escript.symbolic import *
import esys.escriptcore.utestselect as unittest
from esys.escriptcore.testing import *
import numpy

sympyavail=True
try:
    import sympy as sp
    spVer=sp.__version__
    spl=spVer.split('.')
    if int(spl[0]) == 0 and int(spl[1]) < 7:
        sympyavail=False
except ImportError as e:
    sympyavail=False

    
@unittest.skipIf(not sympyavail, 'sympy not available')
class Test_SymbolicTestCase(unittest.TestCase):

    # number of digits that have to match for results to be considered equal
    TOL_DIGITS=8

    def test_Evaluator(self):
        e=Evaluator()
        self.assertEqual(len(e), 0, "empty evaluator returns wrong length")
        self.assertEqual(e.evaluate(), (), "result of evaluate() not empty")

        x=Symbol('x')
        e=Evaluator(x*x, x**3)
        self.assertEqual(len(e), 2, "evaluator returns wrong length")
        self.assertEqual(e[0], x*x, "first expression wrong")
        self.assertEqual(e[1], x**3, "second expression wrong")

        f=e.addExpression(x**4)
        self.assertEqual(len(e), 3, "wrong length after addExpression()")
        self.assertEqual(e, f, "addExpression() did not return self")
        self.assertEqual(e[2], x**4, "third expression wrong")

        e+=x**5
        self.assertEqual(len(e), 4, "wrong length after += operator")
        self.assertEqual(e[3], x**5, "fourth expression wrong")

        self.assertRaises(RuntimeError, e.evaluate)
        f=e.subs(x=2)
        self.assertEqual(e, f, "subs() did not return self")
        self.assertEqual(e.evaluate(), (4,8,16,32), "wrong result after subs()")
        self.assertEqual(e(x=3), (9,27,81,243), "wrong result after __call__")

        VALUE=2.718942
        xx=Data(VALUE, FunctionSpace())
        ref=[VALUE**2, VALUE**3, VALUE**4, VALUE**5]
        res=e(x=xx)
        for i in range(len(ref)):
            self.assertTrue(isinstance(res[i], Data), "substituted expression not a Data object")
            self.assertAlmostEqual(Lsup(res[i]-ref[i]), 0.0, self.TOL_DIGITS, "wrong result after substitution with Data object")

        x=Symbol('x', (3, 2))
        e=Evaluator(x**2, x**3, x**4, x**5)
        xx=numpy.array([[9.2518231214758302, -5.4150410540605654], [-1.5781869039394181, -0.36601352461017989], 
[6.5486813875509249, -8.6886755722383953]])
        ref=[xx**2, xx**3, xx**4, xx**5]
        res=e(x=xx)
        for i in range(len(ref)):
            self.assertTrue(isinstance(res[i], numpy.ndarray), "substituted expression not a numpy array object")
            self.assertAlmostEqual(Lsup(res[i]-ref[i]), 0.0, self.TOL_DIGITS, "wrong result after substitution with Data object")
        u = Symbol('u')
        p=2*u**2+3*u+1
        p2=sin(u)
        p3 = p.diff(u)
        evalu = Evaluator()
        evalu.addExpression(p)
        evalu.addExpression(p2)
        evalu.addExpression(p3)
        evalu.subs(u=2*symconstants.pi)
        evaluated=evalu.evaluate(evalf=True)
        sumEvaluated=sum(evaluated)
        self.assertAlmostEqual(numpy.float64(sumEvaluated),126.939132358972,self.TOL_DIGITS,"evalf not working properly")



    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acos_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=acos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.926229092304)
        ref=acos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acos_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=acos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.27315403586573495])
        ref=acos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acos_Symbol_rank2(self):
        shape=(4, 3)
        x=Symbol('x', shape)
        y=acos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.37568218691765387, 0.34977584999862543, -0.80869238615355665], [-0.61035116375592269, 
-0.45335550273935099, -0.49960450460056949], [0.60717099273041053, 0.40289325250233188, -0.60444576587503174], 
[-0.89811930442918819, -0.60483960658905933, -0.63278204925929105]])
        ref=acos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acos_Symbol_rank3(self):
        shape=(2, 2, 2)
        x=Symbol('x', shape)
        y=acos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.28294591921545531, -0.87338330651803453], [-0.50966631448739208, -0.082647248089423142]], 
[[-0.45342151180416645, 0.36377996422283188], [0.99314106052662066, 0.72462685364781954]]])
        ref=acos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acos_Symbol_rank4(self):
        shape=(6, 6, 4, 3)
        x=Symbol('x', shape)
        y=acos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.9698639936970419, 0.6837726039759382, 0.90742758798111844], [-0.97227663936751973, 
0.12941312303794317, 0.34064388973836879], [-0.75153064493473032, 0.77955915027583744, -0.86217224512120127], 
[0.89122414332908018, 0.19241639846981062, 0.93352976903707874]], [[0.66129652297119668, 0.084581076574701264, 
0.62382889555490384], [0.95478101865901732, -0.41259469992458597, 0.44151899127334437], [0.70898860844260803, 
0.53380541271000181, -0.92226566444061664], [0.029474738319820615, 0.69109257408120373, 0.65679968113840981]], 
[[0.86106944670139396, 0.6300234583142148, 0.076880670859732714], [0.2253925237077703, 0.76923724689273043, 
0.77263960256789566], [0.53028082323498582, -0.78512526886701672, 0.49519041035562483], [0.63880816755864855, 
0.29687000041718359, 0.38658795117197275]], [[-0.076534653557333687, -0.3967262582044091, -0.60920407381650032], 
[-0.87959884137085198, 0.027639242327631708, -0.78441365752273318], [-0.6728736320700619, -0.47539626079363306, 
-0.16413443643547598], [-0.45357877112754674, 0.87819571510931449, -0.67962166457285433]], [[0.46475540215850364, 
0.53038929855347328, 0.38533054335427375], [-0.056933414398614879, 0.87909780297773343, -0.60784012891042138], 
[-0.11782523852976623, 0.32841583417825904, -0.79189397256603256], [0.16815502412904237, -0.21871279026588697, 
-0.35147548432928799]], [[-0.13023045018721024, 0.035884880737155012, -0.66955208894774043], [-0.4807859528270737, 
-0.51316577202791391, 0.36601609262240298], [0.91360389271710862, -0.64791045153437898, 0.19751409598025171], 
[0.97667411471292342, -0.84263066888080251, -0.46191358772467117]]], [[[-0.55350843875470956, 0.21770206487016952, 
0.68727230392019445], [-0.50609180103825424, 0.68825091403321026, -0.7760339154688991], [-0.46193558256344125, 
0.11789372150431743, -0.74085434055396937], [0.56848212359494266, -0.10376746859901886, -0.010021864289275761]], 
[[0.93048623596253388, 0.19148360464509762, -0.79925004487412599], [0.48480498404063566, 0.25393078928193069, 
0.51222681362263045], [0.72310425057996586, 0.44717754486239314, 0.16741059458654095], [-0.20555454230013925, 
0.63872811821577735, -0.34919818535405667]], [[0.70110068908338463, -0.38126175940887674, 0.70159062182272502], 
[0.052641384763922661, 0.16786429825581717, -0.14733654537642482], [0.60386749029708264, 0.8529942795078016, 
-0.46948405188166675], [-0.92077654666794828, 0.16169780784958543, -0.2030650375519758]], [[0.36750882282830855, 
-0.4359578151446144, 0.35912842832970782], [0.93699953036917893, 0.97654834215823794, -0.68367197487413156], 
[-0.30295985288220306, -0.40941499606856757, -0.94908608296417718], [-0.64059055491426808, 0.90284530756180836, 
0.55061631301733427]], [[0.53295709723999352, -0.71116893562367367, -0.58471862851176715], [0.52375462400044159, 
0.51461672037330386, 0.26483541966586177], [0.19896875118278379, 0.25198231321156705, 0.79292931653054155], 
[-0.17621633454316732, -0.33325927293358903, -0.076981041753686918]], [[-0.26658065892257654, -0.75339387717303641, 
0.079831561811479768], [0.90472315300895434, -0.14365988178733358, 0.17638660656020311], [-0.53327932412847168, 
0.87658604789509398, 0.49936165264109089], [-0.92816938072602495, -0.29934242401516031, -0.64588774632416901]]], 
[[[-0.38355843377305709, 0.78563582398487619, 0.32382887859826437], [0.59275912501572647, -0.78402355965068704, 
0.62764171778814215], [-0.82237802750360101, -0.28097885679558621, 0.39572637315879389], [0.15918621805898292, 
-0.91447170817921886, -0.16502238896616994]], [[-0.51195619696033834, 0.076711319730985883, 0.10989025726043811], 
[-0.094582819535743878, 0.98170060791815494, -0.97518768279462109], [0.26496280614824497, -0.31971174173444217, 
-0.0096909350490550494], [-0.4231838002298538, -0.62291992635285887, -0.53553226076173854]], [[-0.99833582883762895, 
-0.99239226558123073, -0.78980897383771476], [-0.35394818355812951, -0.41568067535445263, -0.24120137319706592], 
[0.23098415690008811, 0.23058900574488961, -0.39429167965123368], [-0.65040520384228806, -0.51269216512870308, 
0.26692185656325296]], [[0.16049932697272107, 0.32829773928445638, -0.95095524784147534], [0.86403210822314058, 
0.70248869313527162, 0.478559297535045], [0.20887212178393688, -0.84276685154561548, 0.53338899953159813], 
[0.34042418198746804, -0.69470118627831257, 0.85834654145397593]], [[0.24465142369773996, -0.0053667556731387123, 
-0.78600963705702731], [0.81385501018918949, 0.045771927607326068, -0.0077607227734122208], [-0.68173005485970228, 
-0.031852333063610461, 0.33875569251682669], [0.83379653372429341, 0.82781334032487885, -0.31942053761903022]], 
[[-0.61390777120181661, -0.97740208556502028, 0.91610185576894643], [0.55012454593713311, -0.50211380007600082, 
-0.090480801507946351], [-0.49464402692937837, 0.90886554646299, 0.22650291911600795], [-0.53061207494879903, 
-0.054855613395369263, -0.76065777515270816]]], [[[0.53176236326759674, -0.62242500642560805, -0.55393000722339791], 
[0.70154449591876422, -0.62071654498640383, 0.13569263309222124], [-0.69094438993254492, -0.16841676940093997, 
-0.3195168638881436], [-0.89191308614526155, -0.62987357786396769, -0.70207182359669096]], [[-0.76251254075829156, 
-0.89540085990381613, 0.90523684171573926], [-0.87869886613999748, 0.81393545030251602, 0.12051959168250681], 
[0.5730084015564636, 0.85895024044720891, 0.084888272475384774], [0.12810559392423326, -0.73433637468352586, 
0.19375397797704097]], [[-0.39200634144699364, -0.27469748116255488, 0.80693440427441487], [-0.51533102885391169, 
-0.083905779978496575, 0.76583379652294092], [0.59256287128575247, 0.9757663898417348, -0.84811182880816394], 
[-0.23663314288010984, 0.47289469732275236, -0.97680400334209994]], [[0.84655260410375255, -0.54409562453085858, 
0.8029289207595951], [-0.96769439690341219, 0.80951596327566144, 0.022566075663047691], [-0.54845077396508923, 
-0.62604583144930825, -0.38508197458689453], [0.43055942936382707, 0.33545191565526977, -0.30548376107252273]], 
[[-0.40009263827234975, 0.13736272104150316, -0.70646692991901094], [-0.07873960687646786, -0.32245114605079528, 
0.69243975538348779], [-0.15163716182844245, 0.31425769040540819, 0.12209935640047509], [0.63444587658478957, 
-0.40930735656533135, 0.5427556064802932]], [[-0.50794750224795671, -0.86735696196028766, 0.2259668336910452], 
[-0.01045378343849146, -0.48003357906858657, 0.59085313572942999], [-0.93826585091279613, -0.80422961068039833, 
0.45316700325817316], [-0.44884080170125795, 0.021080459042064126, -0.79742694310000584]]], [[[-0.37634929703803133, 
-0.36435677308011072, -0.047421589140866471], [0.56532279889731263, -0.9677301256843851, -0.35144240500933632], 
[0.79247993192712163, 0.6316628128384989, -0.63222620033419896], [-0.64876375917649609, -0.96033549284807873, 
-0.23080965502955619]], [[0.63643164518928219, -0.83661082774574047, -0.51033612037930642], [-0.22732331709005571, 
-0.62599410478940221, 0.46632499504196723], [0.87235078585116699, -0.53279598404304229, 0.94628181910376252], 
[0.35586185162417738, -0.46061267693601993, -0.67028988378505683]], [[-0.2577884572570226, -0.13324993186931944, 
0.87861212909113062], [-0.42299667637258276, 0.79034343514564087, -0.37673364574589829], [-0.69471921100553979, 
0.097941387840716043, -0.47529672213699636], [-0.0630527691934335, -0.77906838973189596, 0.74999106701507157]], 
[[0.97743232286012405, 0.52446504043805464, 0.052557036531267087], [-0.70223910941029954, 0.34212881997278899, 
0.13995878619138713], [-0.84943011272910995, -0.48392810765098049, -0.84803247155622974], [0.0065150542355560948, 
-0.87491406063401889, -0.53834270198323875]], [[-0.063749987050949031, 0.95285667156463605, 0.43438669676634123], 
[0.94125038073203871, -0.56630348520510365, 0.057499590406511869], [-0.59062834627401872, 0.26812168473300568, 
-0.09153769873034534], [0.7501319331741676, -0.37439330052687714, 0.48464143074284594]], [[0.33268979229742168, 
0.74726793169682315, 0.027036921186866847], [-0.46212037500198311, 0.20441352644308552, 0.16484658883088943], 
[-0.37999391239234148, -0.33630721725795154, -0.54616203920730388], [-0.2193683558727324, 0.71838716749399123, 
-0.66927333500651232]]], [[[-0.80002646471787608, 0.91802088185054842, -0.25964283976295088], [-0.39721940798886202, 
0.99098636136416984, 0.79196037285068854], [0.56818272938210379, -0.91793846026906722, 0.17647057804356692], 
[-0.075656004171821056, 0.5320883413093358, 0.59845023387586949]], [[0.38067576143131676, 0.56227008435509185, 
-0.0067675343525166465], [0.82181986806009988, 0.58200462805504749, -0.28727971847043365], [-0.85060565716037573, 
-0.96122488786034466, 0.29634788021521774], [0.4896935412365504, 0.25180556342234084, -0.64796733264795203]], 
[[-0.65249244450565969, -0.79760790638293533, 0.357837060059226], [-0.62425749657772411, 0.36321572614140685, 
-0.075239548982791726], [-0.10261430349384515, 0.73778682020857156, -0.8451572925727977], [-0.15404490705401019, 
0.30409875617503612, -0.64993341323761356]], [[-0.47709028022190036, -0.53054883897661709, -0.89094046524654713], 
[-0.14949700674790289, -0.45480636961790144, 0.11198808626339063], [0.22710172582033716, 0.77767159491805393, 
-0.84553933847096219], [-0.29051950700472418, -0.71835822266783977, -0.21738708175396226]], [[0.5393057866871358, 
0.23973330786728941, 0.10686108397635663], [-0.19221190383674047, 0.095173732476013351, 0.68770744950220686], 
[-0.37251953689431949, 0.11948226430709274, 0.47525628456678048], [0.88178969567661425, -0.85649316785802321, 
0.9167561289501529]], [[-0.2812308369877583, 0.0043126017498942293, 0.89518002369774519], [-0.46527866855928113, 
0.15331038774092076, 0.43046177301164268], [-0.75811258328949749, 0.038368298715327542, -0.89770359937192357], 
[0.86073358749110196, -0.93249409969206787, -0.36845812863748328]]]])
        ref=acos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asin_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=asin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.431730238814)
        ref=asin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asin_Symbol_rank1(self):
        shape=(2,)
        x=Symbol('x', shape)
        y=asin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.92145319614811427, 0.94465663841879088])
        ref=asin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asin_Symbol_rank2(self):
        shape=(6, 5)
        x=Symbol('x', shape)
        y=asin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.56634296783306604, 0.037805628085990195, -0.57469140462117219, -0.30438670067179352, 
0.041456806103785215], [0.40789838857518612, 0.12260000792836978, -0.10403114180169171, 0.21312345417816214, 
0.26231205267127322], [0.36261359461200882, 0.88723883233865397, -0.62745800606911351, -0.57613437801666034, 
-0.5425382866399604], [0.46387588603562158, -0.85284525665191624, -0.21503031045810395, -0.014771027999943342, 
0.059429031543783983], [0.26003868713177258, -0.75725385670275114, 0.0083671406201912824, -0.73448123821531119, 
0.6341620118378164], [0.66733319694134541, -0.16796095486269813, 0.78372713738320754, -0.74613864128829999, 
-0.52712917854103369]])
        ref=asin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asin_Symbol_rank3(self):
        shape=(4, 1, 4)
        x=Symbol('x', shape)
        y=asin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.094501833474667007, -0.55309441094429967, -0.79845169181526643, -0.47718075788077474]], 
[[-0.53893650589396858, 0.40619817971097305, 0.8323193433377194, -0.45748425513214586]], [[-0.27446979757932066, 
0.15585816836170929, 0.17908000628525578, -0.30478315185262828]], [[-0.46563333114039285, -0.95138520727883846, 
0.21982311155339862, 0.8023588658475711]]])
        ref=asin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asin_Symbol_rank4(self):
        shape=(4, 5, 3, 3)
        x=Symbol('x', shape)
        y=asin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.057716021525408445, -0.12949830236203796, 0.73380240825767729], [-0.81691890394422151, 
-0.52650289918927307, 0.34552929788374298], [-0.21245192113992162, -0.17161243160430972, -0.51511961260999506]], 
[[0.90756021579506063, -0.3107205591563913, 0.33295134258241132], [0.85370200679922514, 0.5366193189709676, 
-0.14274255028441996], [0.88890078056431965, 0.68562222822738672, 0.018555900167712691]], [[-0.7677210558059544, 
-0.42987450834181873, 0.34062555441769993], [-0.55986658192047134, 0.15201515248109487, 0.45041146571033996], 
[-0.5615338718338585, -0.016823107202138088, 0.75387972448354068]], [[0.49649541591284962, -0.35999032210058712, 
-0.038906185984071806], [-0.060990532417816157, 0.70237226968236466, -0.24468722193265968], [0.3769159898756107, 
-0.036594122044091959, -0.052124621375065239]], [[-0.86466788348798485, -0.98759430543087645, -0.092667065015815897], 
[0.50723899358724212, 0.14824318547867787, -0.91636836243450159], [-0.75941113428576323, -0.40903062862030071, 
-0.12642886840516221]]], [[[0.20641896211603683, 0.37030953085752882, -0.028314336414237751], [-0.7024858212833982, 
-0.97151210901123553, -0.49926440954650242], [0.89580677410305576, -0.051703255102513168, 0.33378884684669541]], 
[[0.69866737079972241, -0.075194555680634156, 0.82089277627832091], [-0.52509953409334398, -0.73815072391398506, 
0.8691935555335939], [0.08322686715068417, -0.41221913620800055, -0.14739675677292707]], [[0.61836796685745554, 
0.057258532063050538, 0.29646817711629558], [0.1277168091915617, -0.20905568063414237, 0.71190299719173566], 
[0.043137800002834403, -0.43544306961057555, -0.56880050119154912]], [[0.15363789130748984, -0.87928329129345784, 
0.26228230280018416], [-0.10284743343442915, 0.26084056205453554, 0.64607086769574185], [0.39996598284453677, 
-0.43519684659670088, -0.74666422478079308]], [[0.94451459850895336, 0.28672311634628977, 0.890469820462791], 
[0.28150852647765356, 0.15920460597687747, 0.52768406334169571], [0.60989499786113188, -0.3127701449008693, 
-0.84694414681764907]]], [[[0.69088309678063275, -0.79357690073172571, -0.88062617029480639], [-0.3510177957761309, 
0.37650805807444532, -0.44489301131961856], [-0.79698264433300481, -0.2947858634015319, 0.12179939737236545]], 
[[0.21472352548606888, 0.092305123063669559, 0.72862365773967452], [-0.7252024997813098, 0.40589966976130842, 
0.61047021972883075], [0.68268381479533069, -0.2117854713910845, 0.16479372825960037]], [[-0.88350805916233321, 
-0.30369110745425987, 0.64148927354975238], [-0.15409842869327095, -0.7027962672332273, -0.9726797472599682], 
[0.26773083610372339, 0.86981707216870219, 0.8473443304424364]], [[0.20234644123389067, 0.95164080933576645, 
0.73359703977810131], [0.82610063371896603, -0.90544532318544824, 0.050233038329991375], [0.79963014423104095, 
0.72587100976899066, -0.24779234856866794]], [[-0.70795182286468683, -0.47767622395988951, -0.41990254704795249], 
[0.72314563407639754, 0.1984011315530918, -0.71010184351361305], [-0.10518522890243842, 0.779430085611863, 
-0.86796817185238195]]], [[[-0.77238710354022722, 0.67881736256439318, 0.80411363253476265], [-0.23948917098484679, 
0.59349479185689868, 0.10091987770374122], [-0.58641485741372334, -0.38160458888832083, 0.34151293051834397]], 
[[-0.8705668303161318, -0.32899171892963719, -0.49180267630003915], [-0.78194382617690428, 0.12818271051860974, 
-0.20357098708991894], [-0.68478196286534532, -0.79382378836099554, -0.91424331527699709]], [[-0.33035665697444538, 
-0.042976304458228753, -0.08120501312507189], [0.18235536224202509, -0.082449095168907638, 0.9737312492776431], 
[-0.41017758740869636, -0.1194143991466301, -0.34183530125513828]], [[-0.15576032635901416, 0.59308642123742561, 
0.81860342334818514], [0.44857249004056587, 0.85907673925536554, -0.85813671882627918], [-0.31543849869426555, 
-0.96095095792464202, -0.38398598431621012]], [[0.53036157753360347, -0.94490191155822756, 0.77575232170974773], 
[0.43056993349822936, -0.17278636508625977, -0.54535280058321156], [0.13325563745299163, -0.693394107791939, 
0.055462899052198589]]]])
        ref=asin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atan_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=atan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.860141973955)
        ref=atan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atan_Symbol_rank1(self):
        shape=(2,)
        x=Symbol('x', shape)
        y=atan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.98993674934990938, -0.99236167254265473])
        ref=atan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atan_Symbol_rank2(self):
        shape=(2, 1)
        x=Symbol('x', shape)
        y=atan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.037619457109643761], [0.00034285243007525779]])
        ref=atan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atan_Symbol_rank3(self):
        shape=(5, 1, 3)
        x=Symbol('x', shape)
        y=atan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.31311567842844967, -0.264660574719773, 0.043138279716100625]], [[-0.63763538643211959, 
-0.12071031457376002, 0.7459399913878979]], [[-0.51656973815942453, 0.15844528717305906, -0.45363944849829152]], 
[[0.81554506872022881, 0.32406399554069254, -0.25300113720577122]], [[0.44230550766475218, -0.95146381418271075, 
-0.82685975829456293]]])
        ref=atan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atan_Symbol_rank4(self):
        shape=(1, 5, 6, 1)
        x=Symbol('x', shape)
        y=atan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.66215807646287272], [0.90857105447376663], [0.25062042740113455], [-0.095192598690087005], 
[0.8179547721935736], [-0.52572733482612399]], [[0.32073399601918573], [0.91149209435461565], [-0.38892607664183743], 
[-0.82477298300384772], [-0.10874676985131093], [0.60175485711854981]], [[-0.25995382604523121], [0.30528071758531405], 
[-0.65680378071090839], [-0.091887506307082534], [-0.35664958452656514], [-0.75112744155853428]], [[0.61178494692087937], 
[0.35334759767150659], [-0.020133112539715059], [-0.3329636393021953], [-0.38148797496239983], [-0.076854477822588585]], 
[[0.79500012025345179], [0.41606950304631951], [-0.29298890383567544], [-0.066123991907781621], [0.8273618358441388], 
[0.058979895072739597]]]])
        ref=atan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_abs_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=abs(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.775328614779)
        ref=abs(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_abs_Symbol_rank1(self):
        shape=(3,)
        x=Symbol('x', shape)
        y=abs(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.32610345481660996, 0.083835733535441381, 0.52697893637877491])
        ref=abs(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_abs_Symbol_rank2(self):
        shape=(2, 6)
        x=Symbol('x', shape)
        y=abs(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.99031083132616637, -0.017559781029514498, 0.1615399089922378, 0.80509817818269669, 
-0.85022093902582818, 0.47881000856377964], [-0.18498358265294268, 0.86385463770587423, 0.4528408182463266, 
0.32053834700733508, 0.78323323668801614, 0.84222731937048878]])
        ref=abs(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_abs_Symbol_rank3(self):
        shape=(4, 5, 2)
        x=Symbol('x', shape)
        y=abs(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.53038245271077034, 0.15634028373634679], [-0.14566088317349579, -0.92430387504548661], 
[0.86485801245184257, 0.91451258710846073], [-0.67077388692313367, 0.3425396887241885], [-0.12915299531467062, 
-0.84515163412427352]], [[-0.17440154995442758, 0.41254655372410687], [-0.56244248983035128, 0.71780251479519341], 
[-0.52077657278269918, -0.65440787554808533], [0.12618225766279956, -0.2825363557187941], [0.74606380643387293, 
-0.17788922693678377]], [[-0.020655919907325959, 0.66019132198123165], [-0.20857147602043535, 0.47516900113687233], 
[-0.35246808744941327, -0.13077210928452754], [0.35576809100973428, 0.58154456805135069], [0.76830979394863141, 
0.24360515506673308]], [[0.77272664278528236, 0.2058585059178244], [0.27536427404951191, -0.28018371036318546], 
[-0.72462987561483239, -0.95954451982133571], [0.90611242106785794, -0.66419849276092213], [-0.31239883222499998, 
0.3452030039902394]]])
        ref=abs(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_abs_Symbol_rank4(self):
        shape=(5, 5, 3, 1)
        x=Symbol('x', shape)
        y=abs(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.49336528797761026], [0.88507912406502709], [-0.82039739585199012]], [[0.21512209674028404], 
[0.67653568933664299], [-0.15613763416680815]], [[-0.76449472365328641], [0.16889840387675381], [0.53742253752596469]], 
[[0.1595421248149862], [0.089710393917641573], [-0.63320888587500446]], [[0.83754487648720866], [0.87900606451270691], 
[-0.44530434715397993]]], [[[-0.042313016418267679], [-0.11679391746043866], [-0.99871724466365497]], [[-0.067108125922443085], 
[-0.41368649823246151], [0.73510045767960763]], [[0.5322709135003576], [-0.36518440645839712], [0.21286470886734854]], 
[[-0.72111547394277586], [0.24290166871673224], [-0.068189617325046692]], [[-0.93272411267838273], [-0.76489811517705952], 
[-0.63684337090706822]]], [[[-0.98763470890192506], [-0.52146866985667439], [-0.71621840499134803]], [[-0.90948417135004189], 
[0.068051675706532366], [-0.27965484696909981]], [[0.23500234455218627], [0.032018805194052646], [-0.98736077180874449]], 
[[-0.51745451000646092], [0.41903709932018729], [0.018184680793931651]], [[-0.96016764667564858], [0.72706365949627227], 
[-0.35976000226958482]]], [[[0.58849083226548027], [-0.99920338622368576], [-0.69769506230129785]], [[0.51653413721430841], 
[0.97132313513481106], [-0.83423684846982615]], [[0.73791643479451885], [-0.7102239467832816], [0.17187068302369291]], 
[[0.19423839039149748], [-0.95163872504969071], [-0.41296303043287819]], [[0.20053488824142351], [-0.61109423114398931], 
[0.89624170350900223]]], [[[-0.5618392655558353], [-0.85048295823139242], [-0.43065853192755643]], [[-0.31629294731062751], 
[0.59471784097844549], [-0.18802993577888105]], [[0.48802383148818262], [-0.7652243456355492], [0.38768425175370158]], 
[[-0.89592391116840164], [-0.48962523093855292], [0.31131958720382502]], [[0.42839470245229072], [-0.054372701340328788], 
[-0.71496329963814498]]]])
        ref=abs(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acosh_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=acosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(6.72382470778)
        ref=acosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acosh_Symbol_rank1(self):
        shape=(4,)
        x=Symbol('x', shape)
        y=acosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([4.4418753677648386, 9.1524347698545263, 5.587338108385989, 8.7296387798215722])
        ref=acosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acosh_Symbol_rank2(self):
        shape=(5, 2)
        x=Symbol('x', shape)
        y=acosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[6.4559785877944842, 9.8740724270298479], [8.7200118058068696, 1.2390063653757846], 
[4.1282413912798077, 9.2321720965034757], [3.4337749987007511, 1.2379431819313753], [8.3269533329508043, 3.2777640368117211]])
        ref=acosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acosh_Symbol_rank3(self):
        shape=(5, 5, 3)
        x=Symbol('x', shape)
        y=acosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[6.0152026903794651, 9.4585362600850704, 2.1036572249496355], [2.8641630199523775, 7.1516554653015705, 
3.6103103465334292], [9.1068166972732811, 3.5409833896784288, 5.2030471034872701], [8.715781487064227, 2.680074300359339, 
4.8169564939819018], [8.5846648446168174, 4.8645025977459628, 5.8352661350837334]], [[3.9064063433950449, 7.8746178953272921, 
1.1760997755490548], [3.4198643147185832, 8.3703307069653334, 3.4393591569429343], [7.6862875799102675, 4.9969016062086897, 
8.1585598795916567], [5.2998765478150149, 5.6029343230277977, 9.9965526645506113], [3.4902270734397449, 1.0170873066216086, 
5.5996348599482175]], [[7.0397343330225652, 4.1456383609425353, 4.0890909211623754], [8.8744511247068711, 7.4334387550260335, 
7.16948368809412], [9.7897326727220335, 9.4348015706735442, 5.7567890238536652], [2.6216242190098327, 5.2669144511342241, 
8.3108049400860828], [8.7677570541464256, 5.8891127662816736, 4.2279499791171267]], [[7.1880173169113686, 6.5932606805263525, 
9.0212619891564714], [7.3596153224034833, 1.578590752827461, 5.8919114986073637], [4.0912644414625072, 1.8474712826264885, 
3.7039419780782357], [9.873057695656998, 7.9011086271931505, 2.788397711610247], [7.89746023875799, 4.0160513288162338, 
7.2726330156941152]], [[5.9810703245800498, 7.5495908540274668, 9.0865369611322411], [3.8417212858625192, 8.7008795677984097, 
8.7035575591937899], [2.2251450363908121, 8.229707491988183, 9.1147663508626309], [2.0576178219797132, 9.1803765730631035, 
4.5556102335562043], [1.6061620436629711, 1.0235823006529636, 5.8788874665146666]]])
        ref=acosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_acosh_Symbol_rank4(self):
        shape=(4, 5, 4, 3)
        x=Symbol('x', shape)
        y=acosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[6.9002642431462471, 4.023864450357614, 5.271788526409229], [7.0139723450119975, 6.1653596269769704, 
5.2559327323743696], [8.6478227887772938, 5.2907896815249407, 5.2057400674489394], [5.2705189230730305, 9.0190721171796557, 
9.9793618657528107]], [[3.933562455134846, 4.007784362411746, 7.6216242444650488], [8.5725703663362793, 1.2652674678784197, 
7.7441703030551787], [7.7165518032684997, 4.7650126873619829, 2.5850262464501452], [6.8833155914907289, 7.9919723517332732, 
8.7897887179453278]], [[5.1980066707815364, 2.541154945231157, 2.1467892737252123], [2.7257493124117245, 4.2151583649078805, 
5.4393368030384517], [7.1662036040643011, 8.0739111106256871, 2.2559977272849565], [3.4077211357113599, 1.0492471348491872, 
8.675277588885919]], [[9.7125080553737533, 7.7915487686088216, 9.0807968358432625], [8.6705584613690228, 2.9806888909998728, 
4.5160099869398627], [9.5229297356795364, 6.9394074945742634, 7.3253891893115686], [7.6688411691524339, 1.5091772718757848, 
6.7135945195518971]], [[7.9748696252069609, 1.5302908004157565, 1.4115856946463001], [9.7506908066032487, 5.4668572315506552, 
8.2493053816589565], [7.0336967544616904, 6.2051842422522991, 8.9167252840217159], [5.6762985089722457, 4.1475364846648155, 
6.7575275010101734]]], [[[6.9361900384269486, 2.3922365155613985, 4.9546244391428225], [8.9531009701795305, 8.1421929234745001, 
1.0139267363999811], [5.6334675755054429, 2.8932745641361244, 4.7189566783991932], [1.4276519606158355, 2.8873223577771325, 
8.8154816464954404]], [[3.1282012187302604, 7.8948446281371503, 3.8560504833636635], [8.0762601160872691, 6.0939900756504395, 
6.4452354516509933], [8.5495990173820218, 6.1795197714529717, 5.2986338347436828], [2.214535963462128, 5.9392069162386152, 
2.5116447675734879]], [[3.7834430371447372, 5.9663799785524319, 1.9602188302414825], [2.9756085519100832, 8.784642024338801, 
8.3918357159177326], [3.8150199276222807, 1.1734225521577613, 9.0590797664235279], [9.5253601099610901, 4.9792155221349157, 
2.3499938495967667]], [[7.0455879246857087, 4.7539124617840045, 6.0545684070500982], [7.5041530321558412, 6.1528933090981734, 
9.6551719375838303], [4.9023787266601184, 6.2457126459977443, 5.5862069259778364], [3.4215326388663341, 7.8061449245102139, 
6.856321217898155]], [[6.7400782003873365, 3.9250047999184674, 2.2308431330118959], [3.1350703675234479, 3.9484188687573445, 
8.952323993579288], [6.756205405557437, 6.3246920331199092, 5.4649387271667615], [1.3831791598588217, 9.2758532673631198, 
2.9547932307108322]]], [[[2.8613682329650225, 8.1996849279018669, 4.1182462559058202], [3.1771740197911855, 2.7782090288230488, 
6.627436674614942], [9.2734767938740426, 6.9832877839482732, 7.7205976290513796], [5.922247827240338, 1.1604362104968517, 
5.1375919901808738]], [[4.257577907777458, 1.0459645067035588, 7.2429466774308944], [2.8601030431128973, 8.4845317263783198, 
6.4323666143930769], [5.0075280698945308, 4.8644150752109701, 4.6035942085214376], [3.1151044926871396, 7.4649407868000459, 
8.937884567233322]], [[9.7177329651489242, 4.1760634881191949, 3.1990219643846869], [7.3220294788471341, 9.4763763739737215, 
8.4084095876470126], [5.7893759341468876, 2.1261279137706861, 3.3604322810005085], [5.994719361612022, 7.9654415797896645, 
7.854342597029313]], [[7.2779883650888761, 7.9151829806347669, 6.5829210097881905], [1.1843103224173512, 6.7645722754536424, 
3.6602294955652868], [9.8368695002377322, 6.7934398856819156, 2.6320414102704], [9.6288134486172154, 2.74781590694433, 
7.4257810599430494]], [[9.7307655383745768, 6.8192774686112925, 9.4537497966902873], [2.8228406367739041, 1.9707933630613783, 
3.0269165139224476], [9.138663911257936, 6.7006152579054321, 8.8218088563980288], [9.0150127611852575, 3.6351516467497458, 
7.4040798195776141]]], [[[9.4508820926167214, 9.5393542159366653, 9.3858450418607777], [3.5459159309516401, 9.0826317720128316, 
7.1421709359669157], [2.5503548815978978, 5.4933120809838698, 2.4915124906451571], [8.1683190952280391, 1.8061262870076487, 
7.6317026705050122]], [[1.227076544697288, 6.3630637020177199, 2.9976443165827744], [4.5845784267352592, 5.5904928711456305, 
2.5525451580526219], [8.9832392463646826, 6.2895147244752607, 2.4095810316450894], [9.5669079930218484, 8.4328142698416251, 
4.2156906317867069]], [[2.0166277461644753, 4.8094387993056094, 2.742288876689285], [7.3192873363389337, 2.4550446907579397, 
9.3769292144489], [5.404027531065192, 3.8496499562832582, 9.5760533893514115], [4.8553154895808932, 8.1849139888325553, 
9.4672408070298708]], [[8.667063905733702, 1.5623384217345557, 6.2792744981269326], [9.259867357703607, 4.6213548899944623, 
5.2974148879590039], [5.2168318654410433, 5.9498795913528575, 7.3954369070388521], [5.3864341710177133, 2.553144804117415, 
8.2536442117377149]], [[6.1531231141973226, 2.4092489462217079, 7.7710708019060819], [5.1563761490595308, 8.8828896437681308, 
1.5293841483889452], [1.5674499880091153, 8.4372090503985646, 1.9487247984772338], [2.2233717405495255, 7.3711370391825692, 
6.1206646374330873]]]])
        ref=acosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asinh_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=asinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.598228982253)
        ref=asinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asinh_Symbol_rank1(self):
        shape=(4,)
        x=Symbol('x', shape)
        y=asinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.14784368718445662, 0.46560157410741021, -0.76829595317465027, 0.23030894200339946])
        ref=asinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asinh_Symbol_rank2(self):
        shape=(5, 4)
        x=Symbol('x', shape)
        y=asinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.66884637140450121, 0.95642857948690918, 0.97890589342265844, -0.18209001755176502], 
[-0.80675954530781646, 0.52896241620018314, 0.5732277311269689, 0.056171723930511064], [0.45873026331250855, 
0.02554950186173266, -0.89124568646838376, -0.3661412755017337], [0.93151922153251898, 0.92979819289831722, 
-0.34031263262653111, -0.57814103158053176], [0.029767958951282925, 0.037128253243068343, 0.40892362341839483, 
0.029348138001230373]])
        ref=asinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asinh_Symbol_rank3(self):
        shape=(4, 5, 2)
        x=Symbol('x', shape)
        y=asinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.95342726148287715, 0.44705579758591263], [-0.89286602287873351, 0.60149771948801911], 
[0.59820107147864654, -0.37374102728556835], [-0.81161846073814825, -0.60436808434566602], [-0.11316307408521453, 
-0.51556764029260371]], [[-0.47756317482879451, -0.25124811872307351], [0.96930648565950905, 0.75531318099706857], 
[-0.95735145191647297, -0.54798923400914745], [-0.65134356862325582, 0.3900720896190697], [0.24321639778980408, 
-0.021383993041474536]], [[0.57454101227038845, 0.24902589627051896], [-0.79288075758546706, -0.19007563638541969], 
[-0.23064499836221564, -0.29697869891695139], [0.021230677459263436, -0.72199878849337251], [-0.071089303678373295, 
-0.60963555333328978]], [[-0.99370418439719299, -0.47170996058340364], [0.085212272263422761, -0.70113235935880946], 
[0.043596531425463825, -0.22479547967460656], [-0.38429988447838714, -0.92433253771131207], [-0.9237549325024752, 
-0.99106268060090574]]])
        ref=asinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_asinh_Symbol_rank4(self):
        shape=(6, 2, 2, 4)
        x=Symbol('x', shape)
        y=asinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.42035098785266833, -0.50778431023644743, 0.77497133342871649, 0.95898013564231999], 
[-0.38143020320576237, 0.076286333225312752, -0.13375494652561426, -0.44739293000483804]], [[-0.76352681886706697, 
0.92127891949297736, 0.30704945189980615, -0.28748968697773813], [-0.68053453955594922, -0.75613002920399652, 
0.58405172936008976, -0.10678418707380022]]], [[[0.62013742787709591, -0.83203516435875757, -0.65185106878223076, 
-0.63069164595597815], [0.85972884075343625, -0.77261992521611056, -0.54638314238045393, 0.27430277098076328]], 
[[-0.50394921636521239, -0.10708657712524894, -0.19087121846217325, 0.76144230533489488], [-0.37416664019788737, 
-0.18296624941410178, 0.63774499995510614, -0.3754493090942721]]], [[[0.72572753817086277, -0.67926711836090203, 
-0.96403435620328115, -0.67945607439120304], [0.42838332302945203, 0.92635428184704183, 0.54173257245863593, 
0.85089995038126442]], [[0.44971715025354975, 0.16810955340952227, 0.51468434244816308, 0.42709071390013564], 
[-0.7667508983299427, -0.096265958069885471, 0.96511656550002933, 0.92241869655573261]]], [[[0.28141629592206452, 
0.5600127797928991, 0.84702576906533111, -0.14922224690750574], [-0.29230493031675175, 0.1550391898544321, 0.42865591125132507, 
0.44243344042127242]], [[0.42703875689850701, 0.713699903182206, -0.76163439659338383, -0.19660579665148714], 
[-0.67544690133148033, 0.072200405294204684, 0.34279093259046345, 0.26649948167735582]]], [[[-0.075404402880603572, 
0.3781560772876924, 0.15611360275267261, -0.61625142766162644], [-0.76377704212014952, 0.62193411570684187, 
0.27371063745346702, -0.40365045645814401]], [[-0.50564045223104159, 0.22463102688835201, 0.90405914033368484, 
0.15958598687633074], [0.92040176694317033, -0.27378636583346716, -0.59708389661169314, 0.56133075582385938]]], 
[[[0.40163511134187524, 0.80568205285203831, -0.87601680322861086, -0.0258957454599269], [-0.25306916077509101, 
0.10025751053562248, -0.65445089856615968, -0.80201810243847538]], [[-0.34026461978069067, -0.66732965980653325, 
-0.88050450060310803, -0.28769825727349629], [-0.56522116115075072, 0.58138006376728368, 0.64141501365240794, 
-0.087517488663966247]]]])
        ref=asinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atanh_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=atanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.628778693085)
        ref=atanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atanh_Symbol_rank1(self):
        shape=(6,)
        x=Symbol('x', shape)
        y=atanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.23377007393415838, 0.85082427200389166, -0.3672361188323976, 0.95699236676679145, 
0.32329255767895848, -0.38277477043832997])
        ref=atanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atanh_Symbol_rank2(self):
        shape=(5, 5)
        x=Symbol('x', shape)
        y=atanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.79604197272552435, -0.71832440603900083, -0.39219883427249691, -0.32035896428214516, 
-0.3605081103133152], [-0.41481333820526656, -0.54343495874232373, 0.67230662132038033, -0.42991549666194606, 
-0.65897349668488836], [-0.89339993133389073, 0.069763867484796904, -0.82980439581318799, -0.5295288674178884, 
0.18073438514528828], [0.94879914790673658, -0.72717881858519351, 0.61680262100356154, -0.66995209943988976, 
0.78626676129120687], [-0.78305955423453888, -0.98343064085155563, -0.45613714001988592, 0.60653794115268211, 
0.18604295662031234]])
        ref=atanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atanh_Symbol_rank3(self):
        shape=(5, 5, 2)
        x=Symbol('x', shape)
        y=atanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.2505756275545592, 0.68329655810518286], [0.14404635298981106, -0.97489158010327293], 
[-0.45795183709617993, 0.60269725861900048], [-0.56039317935671473, -0.78521056827171232], [0.67544999032819764, 
0.95332621782119165]], [[-0.78468048507726018, -0.46026948904662213], [0.84829780700057067, -0.27697730443881752], 
[-0.22353692688457061, 0.14061063558953801], [0.30143955037123038, -0.44320585604229978], [-0.6291690150903968, 
-0.24285738695239201]], [[-0.77983556852210456, -0.52620696091088548], [-0.41127294772127354, 0.32396529787105011], 
[-0.025432166978220039, 0.38112840083359845], [-0.065146631765661045, 0.91275522290649014], [-0.015944424434559457, 
0.38383419672934749]], [[-0.15110516801664109, 0.65151139101479427], [0.044776428273270774, 0.46455522316971742], 
[-0.78650407174021386, 0.93566600454547233], [-0.74522434814606453, -0.73691439802718883], [0.46232903971263983, 
0.72289315441747903]], [[-0.3231902393414845, 0.1004468233938538], [0.79506641720685378, -0.41193338584394357], 
[0.23282874906430351, 0.8452513793209242], [-0.80912013900539437, -0.69506120669847027], [-0.27083398868358821, 
0.92235669490614591]]])
        ref=atanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_atanh_Symbol_rank4(self):
        shape=(5, 3, 6, 1)
        x=Symbol('x', shape)
        y=atanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.20971387407726128], [-0.41017784562343929], [0.92526679315453864], [-0.96604154709107237], 
[-0.88992278920919232], [-0.89757928997071623]], [[-0.75457723070136806], [-0.73501341204175952], [-0.054691368761987436], 
[0.093791543801928112], [-0.65557196448384225], [0.60019825060123111]], [[0.91233787400636412], [-0.5499959984043814], 
[0.99155155929892902], [-0.57370124127141864], [-0.20662784579353843], [0.75787303934343209]]], [[[0.79654936662500275], 
[-0.85177021351041193], [0.074753415012279056], [-0.2641206004735146], [0.11153832322535107], [0.15693811273347169]], 
[[0.92567752234134426], [-0.63740034307395055], [0.6487343176368785], [-0.55121237686193814], [-0.21329505519365632], 
[-0.44929207583425201]], [[0.57155412611977097], [0.87964691865786282], [0.6989429912547942], [0.94295165141537973], 
[0.001920697872647148], [-0.31997829037191239]]], [[[0.92371981536863679], [0.27838116581658556], [-0.056555061033674026], 
[0.4530367839907754], [-0.1893874618117577], [0.13122123745622916]], [[0.15842976217953764], [0.78718258116258566], 
[-0.20244099933371373], [0.048729565552966925], [-0.95385898077517495], [-0.19419025340281904]], [[0.10773941848498692], 
[0.34618706165061508], [0.78229446272764025], [0.75433888773270041], [-0.0024978625661338238], [0.016169681368193967]]], 
[[[0.097120067432626689], [0.144375667094365], [0.55503650616635114], [-0.72486550462586474], [0.083630122777541782], 
[-0.23107141934577902]], [[-0.18535629237550233], [0.043665093574946523], [0.11993297105437462], [0.21684171172366629], 
[0.56209137489464411], [-0.46315310043310309]], [[0.75398735355609503], [-0.1371133044213062], [0.38495914450868796], 
[-0.48529675864491639], [0.26292668486649107], [-0.66091882295572746]]], [[[0.42358160022377156], [-0.7278395216188438], 
[0.55786821012272148], [0.1867069149576901], [0.36128983658915859], [-0.78491987064307023]], [[0.82559112350987829], 
[-0.096220459574556827], [0.64380135888945356], [-0.19740961614989949], [-0.29584465102393387], [-0.65885431310797382]], 
[[-0.35786269260535208], [0.20617314114149266], [-0.35231051940200619], [0.81419458275840273], [-0.82227518410345657], 
[-0.26036455326162544]]]])
        ref=atanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cos_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=cos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.962736908515)
        ref=cos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cos_Symbol_rank1(self):
        shape=(5,)
        x=Symbol('x', shape)
        y=cos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.5766008963270195, 0.48134317621709255, -0.69099148955093792, -0.022499338398933588, 
0.64495835020537018])
        ref=cos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cos_Symbol_rank2(self):
        shape=(6, 1)
        x=Symbol('x', shape)
        y=cos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.34488447281026002], [-0.2156432164514368], [-0.50709251925268961], [-0.55502876901189802], 
[0.14009265888634115], [-0.40613042878272054]])
        ref=cos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cos_Symbol_rank3(self):
        shape=(1, 2, 1)
        x=Symbol('x', shape)
        y=cos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.13511862180958523], [-0.43753825785405476]]])
        ref=cos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cos_Symbol_rank4(self):
        shape=(4, 2, 3, 6)
        x=Symbol('x', shape)
        y=cos(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.19907982561050064, 0.30353084912481343, 0.53722096316688805, 0.84005546422152744, 
-0.48155631620576944, -0.74716985554050441], [-0.10945890161850658, 0.49233217846051169, 0.58288429083063154, 
0.57589773334033501, 0.91347177627136134, 0.82926042076398487], [-0.45050025042105002, -0.64163758228931944, 
-0.11867498581464697, -0.61428226242362349, -0.5681490740142765, 0.31182676814189914]], [[-0.18640484887126574, 
-0.29458350247786935, 0.31021822155024448, 0.15543101368975587, 0.94372810483738601, 0.94596061691367161], 
[0.77981903262407393, -0.1280769420882768, -0.91507670426075371, -0.31651675886156139, -0.18515942975231714, 
0.22118213876422232], [0.70080246911031452, 0.97100721922944788, 0.15232090415237343, -0.50830600844829621, 
-0.44299228598984564, 0.97015955714172186]]], [[[0.90052336040760483, -0.19483939885852997, -0.071995587555530705, 
0.72363256342604054, -0.74660493733607658, -0.19798046685496584], [-0.97779879034256956, -0.93659194549161229, 
0.55201880047182073, -0.53606579828624801, -0.29401179627649188, 0.43678927994627958], [0.083560652594002915, 
0.92009008040107254, -0.97255907829479549, -0.97967421428491952, 0.60354683571489409, 0.043141832577749506]], 
[[0.34483546830493461, 0.57344218277388048, 0.10962111147410725, 0.37176078745001195, 0.019358457577045574, 
0.78681662089181392], [-0.11148829570451269, 0.63248565232979814, 0.26641064854606356, 0.25051527613940383, 0.1645040198371428, 
-0.3093543217552821], [-0.35060208758737144, 0.46874322714901062, 0.82103471093154279, 0.65887801342729069, 
0.57173259609581306, -0.9019359004516625]]], [[[0.52364902857070916, -0.60862693024133407, 0.7079161176390536, 
-0.54753370235739474, -0.16157879084751081, 0.2347369013692997], [-0.30119311422568851, -0.67693029949165817, 
-0.44054777013294211, -0.17595510468230091, -0.3722573761659016, 0.65659148932273137], [0.61044366119250215, 
-0.18092684942538462, -0.13334316545753055, 0.67273477414384941, -0.32010645014433781, -0.17380363518066]], 
[[-0.31265178953405304, -0.049070594061846107, 0.86846200928806128, 0.95061018332202973, 0.46629594869434898, 
-0.69152902591122567], [0.43872521827099686, 0.43858746674163873, 0.34767628658906014, -0.018458574573531061, 
0.81758120632848064, 0.79301236650063633], [-0.82798904316023658, 0.4349709104192212, 0.023311070907567011, 
-0.30743832703234153, 0.13671396984233608, -0.83851956878206835]]], [[[-0.8734939357686784, 0.33732283609570723, 
0.41236518066874916, -0.95055809620416776, -0.98357746987052752, 0.95104084068664041], [0.88878898254669259, 
-0.16911847050551576, -0.24783333440009625, 0.82208119668585744, 0.0010431798911008094, 0.0082821615019470496], 
[0.1396509188825692, 0.094257822276048842, -0.13228750966572345, 0.89926170217127521, 0.077787334627457128, 
-0.17993373580123584]], [[-0.72769901784519031, 0.73016627869730022, 0.53694019977210439, 0.51397940947350396, 
0.52801017839145081, 0.074316759499168938], [-0.64851869908713899, 0.2788395070141696, -0.92092142853747871, 
0.13841382648580858, 0.14670057247971569, 0.24017939247346143], [-0.53458528754962442, -0.43934782222951374, 
0.79996580192339883, -0.80162215269264214, -0.028852977523383894, -0.86626759977804757]]]])
        ref=cos(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cosh_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=cosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.931620647295)
        ref=cosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cosh_Symbol_rank1(self):
        shape=(4,)
        x=Symbol('x', shape)
        y=cosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.91694419748148848, 0.56171068444245398, -0.1939553578454607, 0.81368744339677579])
        ref=cosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cosh_Symbol_rank2(self):
        shape=(6, 6)
        x=Symbol('x', shape)
        y=cosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.48776155430046098, 0.51838439075389875, -0.025949362624035688, 0.76367991246737388, 
0.05421265821947352, 0.31818128292100778], [0.85203189225961884, -0.23088542198435102, -0.45846863645622915, 
0.93382631282261341, 0.90336058939397912, 0.20142486952202332], [0.52949873270766634, -0.77221653035421634, 
0.74548788516908338, -0.85171273745031595, 0.057779925121308029, -0.38021502840326993], [0.58177405371136381, 
0.16088033704555493, 0.69483691557124461, 0.6756295135267214, -0.60312840314075, -0.53194963237386528], [0.12459222760988409, 
-0.61534279763692989, -0.53171300505442476, -0.51689144120023633, -0.84049372089324526, 0.83079152836396597], 
[-0.45021434764043256, 0.078051886698744966, -0.63691519882769332, 0.41482117547581998, 0.345396830355579, 
0.64290867709947563]])
        ref=cosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cosh_Symbol_rank3(self):
        shape=(1, 2, 4)
        x=Symbol('x', shape)
        y=cosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.99075905148006616, -0.556563736711716, 0.482948365904738, 0.61848063263263042], 
[-0.21160117123689282, 0.93380107058377382, -0.38203158244434676, 0.40818891686662773]]])
        ref=cosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_cosh_Symbol_rank4(self):
        shape=(3, 1, 5, 6)
        x=Symbol('x', shape)
        y=cosh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.64169029258586607, -0.39602130811642078, -0.87047269158743235, -0.70671915506067906, 
-0.23013892219203136, 0.051279741176573967], [-0.1815823273156254, 0.22362787424663488, 0.086531134128640774, 
0.2033832711169421, -0.29460998523122828, -0.29604077185914268], [0.17918530352350981, -0.95851230588757597, 
0.6089039291143532, -0.23757756476073078, -0.030912603179279685, -0.44366381486624329], [-0.069859395893596909, 
0.81133536902580095, 0.10767565447357419, 0.8824504020924937, -0.65509551955715173, 0.92272691816739782], [0.74133278082055698, 
0.40911929431277216, 0.1822539168616113, 0.049516641695215746, 0.30250549837342944, -0.82264380500945622]]], 
[[[-0.30440534124661278, 0.45772207772993267, -0.23307067348130106, -0.47299200985722356, -0.70713971065210002, 
0.63657827136991219], [-0.34568231546236672, -0.82773395989784282, 0.22061734040553627, -0.44947612583105667, 
-0.82579704972895351, 0.046140649241088294], [-0.20114859714917932, 0.62317698558045764, 0.43257530170170533, 
-0.51262532825183138, 0.046601230594965681, -0.95170817307557498], [-0.80642652621113942, -0.75405757465693624, 
-0.534964442795683, -0.050306363307079049, 0.86603326962096205, -0.30778937685069252], [-0.87863387509688473, 
0.85097807490368438, -0.25108326762247146, 0.4937910293257215, 0.056690519953803031, -0.82101973588398902]]], 
[[[0.17526396588556215, -0.075652957124554998, -0.69110848195606445, 0.24208861806210247, 0.24904774014376052, 
0.72231925877648662], [-0.68511614182752623, 0.041250777089497426, 0.4000072972000186, -0.87266655656284309, 
-0.4686214701437561, 0.86214782509815513], [0.34285298245488183, -0.072857735041811811, -0.75094545378399813, 
-0.8164507554195588, -0.63082644422203837, 0.30213883987979173], [0.55400445993054737, 0.48543645194975871, 
0.44249440951834762, 0.81459036603800894, -0.43944947288524649, -0.13212404044942572], [-0.49653039338796079, 
0.82084218457531311, 0.88279263592802959, 0.69105880560957389, 0.709818852462891, 0.1525990192290636]]]])
        ref=cosh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_exp_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=exp(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.00897111276758)
        ref=exp(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_exp_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=exp(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.47097987249741724])
        ref=exp(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_exp_Symbol_rank2(self):
        shape=(3, 4)
        x=Symbol('x', shape)
        y=exp(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.15599078831709989, 0.83530855752181044, -0.61853037841860181, -0.9049813364629431], 
[0.48126403663398643, 0.65331890812203364, -0.038516438386840379, -0.43146228667932118], [-0.84621703464988318, 
0.54969488534488598, 0.48884708476539385, -0.38979661635629914]])
        ref=exp(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_exp_Symbol_rank3(self):
        shape=(4, 3, 1)
        x=Symbol('x', shape)
        y=exp(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.0047010214133329509], [0.58497005822233938], [0.190531375657647]], [[0.40618460242888199], 
[-0.48479595911885331], [0.67308933034827656]], [[-0.85316164151848084], [0.75444007790920686], [-0.95179352370710779]], 
[[0.57810743647134744], [-0.014212029108792601], [-0.31757235628998748]]])
        ref=exp(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_exp_Symbol_rank4(self):
        shape=(6, 5, 6, 2)
        x=Symbol('x', shape)
        y=exp(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.30596292185592877, 0.30366663235901425], [0.6231722659051504, -0.054728526294849367], 
[-0.44139589051371075, -0.25120205722386713], [0.7446610696609921, 0.094724919048205125], [0.14650070934907045, 
0.70801091981612796], [-0.92050711984005651, 0.24685847906966507]], [[0.016532231248215012, -0.070321218023200016], 
[0.15423749362710448, -0.74674518498077402], [-0.87402020938257508, -0.75462133626457484], [0.032048778727314264, 
0.099037186704303615], [0.7307718585816565, 0.79054777284779743], [0.38242541669168162, 0.35850368942989852]], 
[[-0.7974167093641269, -0.73067586437853116], [0.97693678947125462, 0.76489469869146487], [-0.58677638460680348, 
0.19563004514794091], [0.20841425549350645, -0.15645084283457855], [-0.070711687011929314, -0.81559070077511575], 
[-0.21419661082485453, 0.45123310080509071]], [[-0.067287685193925917, 0.3753358617853324], [0.09214212746376016, 
-0.44585013366312976], [-0.51017734955982696, 0.95875741402337322], [0.76044897455281335, -0.98412720384937602], 
[0.20108459119786315, -0.80406413013483946], [-0.84990658490580007, -0.039581263271350586]], [[-0.10201832972678559, 
0.14901314132472443], [0.82460662405809915, -0.3277445601261022], [0.80429497544969863, 0.58503382154684558], 
[-0.9029622511070321, -0.94842471817599439], [-0.87424311988110603, -0.5221757541687011], [-0.70731975290265092, 
-0.86207424292162638]]], [[[-0.018286445553016062, 0.13426114801666578], [-0.86065592524997658, -0.18414707772021566], 
[0.12401626306004254, 0.47792494262585739], [-0.26746537783893931, 0.61287209753963512], [0.87839205675558762, 
-0.81706751142132883], [-0.88493506578054171, 0.83653201763925344]], [[-0.94826370683013517, 0.89631237021278398], 
[-0.29260053818029208, 0.019735658145988655], [-0.83393977636172489, -0.8304031388909725], [0.79918645083485074, 
0.028164076089578938], [0.099401675328516959, -0.73661730027692074], [-0.54617050015315072, -0.20618508714374184]], 
[[0.29500655335379022, -0.63914557204669897], [-0.78458797376840539, 0.75470073097403256], [0.98613948937213691, 
-0.46455905954917709], [0.44923419352389771, 0.34096106359171063], [0.90527793216545205, 0.68931172366431537], 
[0.50643534475243857, -0.99909209066321392]], [[0.23643933272399154, 0.42413493322088969], [0.92223926429637304, 
-0.13077833985235809], [-0.9867925158473887, 0.6674050590285765], [0.84603514612691333, -0.78345125440108832], 
[-0.26885985726915362, 0.29416519140636677], [-0.57881506763631174, 0.44904085829021878]], [[0.37679189347116226, 
0.53322127023021926], [0.20282442370186526, 0.90161357034603884], [0.0672068732531379, -0.60805739580442353], 
[0.62001773310375285, -0.37964492510738235], [0.25931816851780609, 0.39537332146251702], [0.93415525099142505, 
-0.61777234694360916]]], [[[-0.46407961603856274, -0.66157985016240173], [0.88792583878263098, -0.19592922807156876], 
[-0.25767252745309843, -0.29622441630507801], [-0.95723157511556889, -0.89555026038533314], [-0.84104343356145828, 
-0.091959765142363148], [0.91139107437952993, -0.94561206260990227]], [[-0.53141031225283797, 0.6553900664158494], 
[-0.7782132818838956, 0.90675858578939361], [-0.80975120643773746, 0.37549105272290362], [-0.40788420558627259, 
0.91264797523629126], [-0.57147655481112847, 0.37704319055136049], [-0.75715351320554047, -0.66149693595302361]], 
[[0.77741720769013178, 0.72110323844047852], [-0.85158161844112001, -0.090611850524293658], [0.93326323314347426, 
0.7726635627126055], [0.096281110382635315, 0.30340728420934093], [0.98608223159614794, -0.19239473769493354], 
[-0.40866198985687752, -0.16761911697427001]], [[0.55424558881285013, 0.93232913878921786], [-0.78146532823059345, 
-0.83624093432571733], [-0.35866792771792211, -0.61434788720274125], [0.044470089454589745, 0.79722737716855052], 
[-0.51967777990673092, 0.9805923265705967], [0.0078293675338785285, -0.96149168680409192]], [[-0.60930713897313771, 
0.70251075118522266], [0.35791335868253737, -0.73035308199412108], [0.82796146066830523, -0.92484044474111782], 
[0.054514328350172114, 0.84711134685166556], [0.83025616140982716, 0.34094003908064341], [0.45706712554547901, 
-0.9663983784464476]]], [[[0.64779801463869768, 0.58721995935270455], [-0.75329910729095451, -0.76682692418354059], 
[0.47506176675518885, 0.3948005109562045], [-0.058691608053078159, 0.9987090180383571], [-0.44551915179886969, 
0.94574964748483459], [0.17204189129879488, -0.98771974586774647]], [[0.67928082123746769, -0.05233122102290011], 
[-0.018892855055411939, 0.27850855290407428], [-0.55737341163384357, -0.98999600102735608], [-0.45428940104007975, 
-0.34857798317415933], [0.050137568675821198, 0.017012489885865234], [0.33268505891125932, 0.12771514339664125]], 
[[-0.11931273629412309, 0.9801586659410122], [0.92760042419988653, 0.39800507526656514], [-0.47982410148597343, 
-0.98073202711940177], [0.51619044075156806, -0.86880772774727366], [0.58945407296162178, 0.38510534483341852], 
[0.20364603194121722, 0.010652215325434256]], [[-0.14817838895124957, -0.4607469432281821], [-0.73819401543217866, 
-0.67666933215309033], [0.069595965942202964, 0.17910738949221994], [0.23009301566395224, 0.24941073286595539], 
[-0.19664453728095443, 0.04980497792297145], [0.6487390160450297, -0.07079289533351707]], [[0.12990862618698085, 
-0.79361849919311722], [0.40260059549511729, 0.31886179661338243], [-0.39466248189362174, 0.22814049762743593], 
[-0.13359160643719603, 0.72398092938172187], [-0.83794024634630637, 0.12365309093544918], [-0.48560967498252894, 
-0.77788096851502364]]], [[[0.85173658319575529, 0.577014551294043], [0.21361424424692532, 0.08429221555423938], 
[-0.6123838194145752, -0.49635328297554349], [-0.46383776641446772, 0.70001544428571028], [-0.12519883760550754, 
-0.22063456640416979], [-0.95006519142511259, -0.90605087088889102]], [[0.86863876058824752, -0.3278910447973975], 
[0.29364232163030501, 0.38885761932368479], [-0.1852571169901791, -0.63087246032151323], [0.29726745520525344, 
-0.89863370459558256], [-0.27063027983317189, 0.48310918893602217], [0.13289112275020032, -0.52736063420011248]], 
[[0.087504974214698494, 0.07823070592583603], [0.10809543261682686, 0.016525785820925121], [-0.0247510226856964, 
-0.52870869225632955], [0.19454693791622146, 0.78619539918072978], [-0.58854544664397435, -0.75068637898166757], 
[-0.79933324122526206, -0.51443287175476149]], [[0.40926067203193406, -0.76109702660474143], [-0.45025011020737327, 
-0.19737478860499191], [-0.62891504474281024, -0.68490643525957351], [0.21572191650916772, 0.93204453007061971], 
[0.79366966684285623, 0.59025377667120993], [0.48257449867527802, 0.53769741230172263]], [[0.092875342126348848, 
-0.63995797782070274], [0.42438374474894713, -0.93870603931761321], [-0.22407544965425785, 0.36096169772142561], 
[0.20757103573027025, 0.73693895922021624], [0.69392908125765773, 0.43630699013762242], [0.070299234971340052, 
-0.48962956941167923]]], [[[0.1392910791907398, -0.19184781072474832], [-0.57817350526221234, 0.67046017137213476], 
[0.84334097701803579, 0.92194490582093636], [0.7344610296633145, -0.71325048214812492], [0.81005466541197757, 
0.11462687735533406], [0.43147865789773499, -0.47540249062630613]], [[0.22146489672624026, -0.070224743042629267], 
[0.37190852159136001, -0.49038949960688982], [0.5768061787367067, -0.82397125662326132], [0.40992096386723653, 
-0.92741597936441122], [0.71608843694227464, -0.29014985382999003], [0.0087445773549872374, 0.0054133250537837796]], 
[[-0.45356421697562355, -0.82148436553156112], [0.86957033820002971, -0.64754595320426667], [0.65689985740452839, 
-0.025653863677893129], [0.15209545893793641, 0.35688892156821073], [-0.68881477588242679, -0.23401778209224866], 
[-0.87412906876778829, 0.028830446035241319]], [[0.90027677541748408, 0.10162761305094303], [-0.029923056747827914, 
0.64024050328453486], [-0.57223935620071487, 0.2580835332951712], [-0.85403803518041776, 0.40707347706215513], 
[-0.60372887967650457, -0.68103364819956358], [0.35800132324020217, 0.21743248362508916]], [[-0.50101144015350685, 
0.50676228110110633], [-0.86633010332152582, -0.23293592516880013], [0.79834894467327544, 0.97856055748655235], 
[0.38537515035061043, -0.032554076815270561], [-0.56571529738595117, -0.11411609009780688], [-0.045424841887348633, 
-0.91268440349210955]]]])
        ref=exp(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_length_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=length(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.0946238978632)
        ref=length(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_length_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=length(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.75101727786170946])
        ref=length(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_length_Symbol_rank2(self):
        shape=(3, 4)
        x=Symbol('x', shape)
        y=length(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.13260333155331105, 0.39525615038106587, -0.55326108909072813, 0.79354424989024275], 
[0.52891999129590883, 0.70333119333369232, -0.0552835107927796, -0.3210300128192034], [0.30319067763433094, 
0.38646115209166232, 0.32547988582039467, -0.89667353501088187]])
        ref=length(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_length_Symbol_rank3(self):
        shape=(3, 6, 1)
        x=Symbol('x', shape)
        y=length(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.093140445929148274], [0.97983436504786714], [0.84208144753830982], [0.82618104052595354], 
[-0.37862753498342494], [0.27940866408181542]], [[-0.83607886592584157], [-0.82456967253020941], [0.22264766877203246], 
[0.19469779479628335], [0.44849148482915768], [-0.92584590499595576]], [[0.80053922691304358], [-0.13965472658980693], 
[-0.2576168475142202], [-0.67500826543448533], [0.84520747498240878], [0.84339566285407508]]])
        ref=length(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_length_Symbol_rank4(self):
        shape=(6, 5, 2, 1)
        x=Symbol('x', shape)
        y=length(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.96227112410543603], [-0.12050638141735903]], [[0.87273325402806878], [-0.25584647257048876]], 
[[0.79160210110028761], [0.95874831319406595]], [[-0.25526277185811397], [0.12860093807007833]], [[0.77707917717967412], 
[-0.82241062274357857]]], [[[-0.083413231921356967], [0.10634547407277606]], [[0.42178224093611827], [-0.43085933146824962]], 
[[-0.52028132586319442], [-0.1786174821736457]], [[-0.33859767969258514], [-0.66491410813576635]], [[0.40688271279765575], 
[0.030334005353135129]]], [[[0.59878422774359952], [-0.43871932661498381]], [[-0.5133626205912607], [-0.19094099020008604]], 
[[-0.96044413926750805], [0.42238311141132967]], [[0.17652606549605521], [-0.26047797028713449]], [[-0.4904543686001881], 
[0.47631874521309747]]], [[[0.026806001819245084], [0.084735312407115737]], [[-0.76741011667885539], [-0.045743828487355032]], 
[[-0.47156297268381131], [-0.10296068162737204]], [[0.35593520818048408], [0.22865256456033967]], [[0.71754048308526985], 
[-0.66622298148824299]]], [[[-0.079660464506897677], [-0.91199764792949423]], [[0.47496979244333271], [-0.76143899805412585]], 
[[-0.48807312137736036], [-0.094324605017867436]], [[0.18672819320501999], [0.78370694714744671]], [[0.53410345624584354], 
[0.29336853974672805]]], [[[-0.99609543824443647], [0.062370694203262689]], [[-0.95883914251372371], [-0.9057079570956883]], 
[[0.24404376671817452], [0.092660484415446831]], [[0.36768810533455176], [-0.83983817558627605]], [[-0.93708621001717196], 
[-0.51718559191438107]]]])
        ref=length(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=log(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.107159184453)
        ref=log(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=log(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.71748703726710705])
        ref=log(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log_Symbol_rank2(self):
        shape=(4, 1)
        x=Symbol('x', shape)
        y=log(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.61131146595659991], [0.4018841521847335], [0.81353342105865289], [0.72018643914245828]])
        ref=log(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log_Symbol_rank3(self):
        shape=(3, 2, 6)
        x=Symbol('x', shape)
        y=log(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.12433529720706205, 0.49513931075732587, 0.85738319231370486, 0.38886084910667684, 
0.72697626312357977, 0.4798385362520069], [0.33656189784196133, 0.34549629760579526, 0.40961827440380838, 0.90136689434795281, 
0.54556037046914041, 0.054321654876574255]], [[0.49014475112872169, 0.58384583411152435, 0.28793797291503243, 
0.98558828341689531, 0.47394779571067724, 0.91693431782552004], [0.56842572231035071, 0.09460471131244319, 0.22119301483383746, 
0.68332830309557513, 0.13789251920418, 0.74379222666006339]], [[0.97593571771657284, 0.086828380361032576, 0.63819426725263118, 
0.91611825288980409, 0.22483917274871412, 0.88771042265188249], [0.3249774766351492, 0.40259671229467109, 0.55227266433238964, 
0.99638697603917548, 0.95026616262460017, 0.43307826205731914]]])
        ref=log(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log_Symbol_rank4(self):
        shape=(4, 1, 2, 5)
        x=Symbol('x', shape)
        y=log(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.40312628568383169, 0.78927471816219374, 0.5775266558195542, 0.27655259411969813, 
0.82765863328239186], [0.11476535063789561, 0.19059237573261978, 0.55510483652574394, 0.78191697417147665, 
0.94919156884023348]]], [[[0.80477601484601513, 0.21864036792012775, 0.89659476950097283, 0.61601221228894731, 
0.97345033196154296], [0.3184339673402341, 0.61723581856492982, 0.38779724121395009, 0.24140678678920258, 
0.10669408838757477]]], [[[0.71989806979858428, 0.72860837054242167, 0.089470671080997488, 0.51690977251205006, 
0.75464337597423148], [0.4869899172914971, 0.44781614962779659, 0.90959276060801564, 0.36524024237914154, 
0.088684506657893847]]], [[[0.15515320366637486, 0.70798821971098636, 0.83468781417459925, 0.5342410035792502, 
0.44830431756321187], [0.89470126436222164, 0.63679563517826232, 0.073884503153794134, 0.21032629030752192, 
0.97855521778496624]]]])
        ref=log(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log10_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=log10(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.498219542088)
        ref=log10(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log10_Symbol_rank1(self):
        shape=(3,)
        x=Symbol('x', shape)
        y=log10(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.80813470095459916, 0.41170559776422999, 0.9260128724072384])
        ref=log10(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log10_Symbol_rank2(self):
        shape=(3, 6)
        x=Symbol('x', shape)
        y=log10(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.92202406720474384, 0.67395082319814115, 0.75170452687880451, 0.47119294350212904, 
0.094763220450263086, 0.93099785282155867], [0.36748317679429243, 0.89841240489081309, 0.91273839155094505, 
0.33280124097597752, 0.88099429186526579, 0.28265470522966174], [0.59630149232099627, 0.60102609380135308, 0.38883593906194791, 
0.3353987113563639, 0.006458104437443879, 0.57399908608006966]])
        ref=log10(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log10_Symbol_rank3(self):
        shape=(5, 6, 4)
        x=Symbol('x', shape)
        y=log10(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.10186831498010007, 0.2135378272298486, 0.31887623832248724, 0.85136459127940955], 
[0.77153166093968795, 0.8652928608036673, 0.46016218707220957, 0.62890444485229324], [0.98869062585987, 0.31281721681265096, 
0.1906763574452276, 0.56515032637816931], [0.7348964862594094, 0.75036482091225354, 0.57360632859050775, 0.028144415523908783], 
[0.74733565640450517, 0.39742848180168677, 0.0060276061159015848, 0.2187742244789499], [0.57449014167080514, 
0.069653239933958333, 0.51890861215990913, 0.25933626043878222]], [[0.031646515221860683, 0.64487376072193425, 
0.45726389653281596, 0.78701446127216668], [0.66658897097152503, 0.023361053058920556, 0.80364203008965451, 
0.09981068764361356], [0.48849841526135229, 0.92824511652932273, 0.46801124296572805, 0.72367432307038315], 
[0.83799132329078574, 0.43689729832707169, 0.115925661579101, 0.025389205521198166], [0.60572885192125769, 0.23720439714228547, 
0.79494750705733219, 0.7785426859626976], [0.1487089803665852, 0.65051885658155617, 0.62360438855189892, 0.30416090266304407]], 
[[0.71661296904853167, 0.64021531993790359, 0.52935063966439966, 0.88314418302587583], [0.88609687349172017, 
0.89305619260106961, 0.9698460864899805, 0.10792208220968236], [0.63692222342691163, 0.72190870655932793, 0.43135282707676093, 
0.15031292581714806], [0.76335707739926162, 0.7812290266080858, 0.37360067739644476, 0.11187183050823757], 
[0.18079628330852127, 0.87980589608824433, 0.80966864186077359, 0.52227629692961164], [0.74373136855176603, 
0.56080991356987775, 0.025588609300598031, 0.7152144034319573]], [[0.65927102973313367, 0.96521136593413737, 
0.84141290263575941, 0.39101320837398756], [0.24199442992122577, 0.083280172678730668, 0.059628655773365291, 
0.45312774694270064], [0.071791848567499206, 0.58731034300167351, 0.13507446735484196, 0.63404459816759895], 
[0.064252976550993712, 0.86778730366242318, 0.24304038657150595, 0.80974135630859012], [0.86023351673257986, 
0.28798037573245827, 0.18206336760687325, 0.865748805047305], [0.62688482281847191, 0.30678950404931271, 0.90879845914837798, 
0.25264621185951885]], [[0.87929752218143675, 0.67613157227750442, 0.33207637942512758, 0.053658133777309658], 
[0.047498852091670818, 0.34361099621202074, 0.13033148088167945, 0.24614254127747925], [0.032630256788782952, 
0.7044910000509903, 0.56028056063442266, 0.2961220335509267], [0.22336001930263272, 0.85800396522318489, 0.39779517527224895, 
0.29852508428959745], [0.68954263727356824, 0.77664614962814849, 0.70091805915624272, 0.54407714611895186], 
[0.24346455168769077, 0.15214978129096468, 0.30266826105505595, 0.49516307475606802]]])
        ref=log10(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_log10_Symbol_rank4(self):
        shape=(6, 4, 4, 4)
        x=Symbol('x', shape)
        y=log10(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.97312304426062135, 0.50371673549785512, 0.49870441867539372, 0.59625629112644651], 
[0.42888281415509732, 0.28055418038266822, 0.4296807075614435, 0.62424267873607053], [0.90350985998662503, 
0.054198134071435033, 0.11616282480998508, 0.46501534991987659], [0.1287851494651574, 0.66200503345999651, 0.71112315756212041, 
0.33384528261264124]], [[0.10837906718437684, 0.19875300634263127, 0.65051862636948565, 0.91754958111895113], 
[0.14173672179569197, 0.87804513763209435, 0.68986536709156554, 0.3113445484731765], [0.98977049530489136, 0.50869263292346167, 
0.93257624255206584, 0.28855380644732709], [0.32702113175874259, 0.62353487506545668, 0.99899932773462718, 
0.032921795563313871]], [[0.18448698676767439, 0.13157492462247922, 0.32006295335594614, 0.43076675555860267], 
[0.10695016214699371, 0.93731742391031714, 0.21226381138561234, 0.52994216727644172], [0.56297525585828134, 
0.83577378985879514, 0.67319920774363795, 0.88477828907358214], [0.69281935923496452, 0.93552425811791695, 0.84326252536112545, 
0.66622129770632521]], [[0.16443323082227701, 0.56528531880131228, 0.86574203239515302, 0.50546407780568314], 
[0.78009877705008335, 0.90045042464017166, 0.29488767422917439, 0.6281037689097686], [0.80958221933004793, 0.18547986695556273, 
0.34020693804141555, 0.80066838808481733], [0.26448409097676451, 0.21616187093260331, 0.17662862220520992, 
0.28856819663532607]]], [[[0.62361277809576598, 0.18861573060671777, 0.77847278986377799, 0.85335109140807797], 
[0.93963975655730791, 0.35335993478614125, 0.52704405644282404, 0.091922289061672435], [0.51582102805455687, 
0.2532646190714688, 0.39871072269656349, 0.53867363489946352], [0.70228965464648918, 0.30192517401216468, 0.12976367025344371, 
0.6817212224046667]], [[0.44405943616873322, 0.41884685043497416, 0.32755622589773492, 0.4604437724614463], 
[0.12601428405739834, 0.82795233475270535, 0.71242308874675375, 0.22049695360764532], [0.14072345006800413, 
0.63508586856069948, 0.3475256153379892, 0.24873836656201642], [0.8378293360316218, 0.90789559631760264, 0.67889488825771249, 
0.12787966083913149]], [[0.60721920608567126, 0.26937480971793193, 0.071360957768064281, 0.89536518229145723], 
[0.16765910699083864, 0.53327625724280503, 0.42629930800958193, 0.92858203780749182], [0.048702709697830637, 
0.27507825424178212, 0.68774474358335602, 0.56691344449418679], [0.68184043090517088, 0.92225464618116249, 0.31925701817294916, 
0.73897665526599632]], [[0.54442688055787813, 0.30608150174334869, 0.32510361949852662, 0.17883613960790057], 
[0.1181904095798948, 0.86259263997056135, 0.80682913150521174, 0.076880450041311832], [0.86728806238220868, 
0.85316500174000975, 0.29001953875090858, 0.88121889862628899], [0.72007882595288464, 0.37978478238119562, 0.65676751089282648, 
0.068606123897136717]]], [[[0.93666797512755984, 0.96678305879304982, 0.36029356631245679, 0.92395670850800238], 
[0.78407410934143063, 0.60276020242395967, 0.39493636901797802, 0.136718123023709], [0.44357963223560382, 0.17466831124966198, 
0.5794146677883204, 0.53480803015502665], [0.84793399085466148, 0.82026048970834098, 0.51427051542252966, 
0.95595864436976785]], [[0.93407933329915294, 0.052272150458120614, 0.25779487498521214, 0.10837338763154591], 
[0.50553261102949909, 0.79819728052438987, 0.59419918297622898, 0.57481966358357472], [0.62769049843959612, 
0.73678065070682375, 0.078552428407051345, 0.38280660429075219], [0.83114948715709813, 0.61658034375557624, 
0.73828696605832278, 0.27473198529972709]], [[0.52465213046173209, 0.63570563035223027, 0.76982380183190746, 
0.68517307682910367], [0.56112386041730222, 0.75363407820979023, 0.69146617762795959, 0.50939854781017035], 
[0.061670720957187064, 0.4239568690769483, 0.50925781053442865, 0.060371309176233567], [0.81580264074308206, 
0.34130691984213835, 0.99293035736558632, 0.66027840394783122]], [[0.14279758406141252, 0.24618897041483667, 
0.91829033235925439, 0.62119038094091295], [0.38409516842277669, 0.44653959064808257, 0.34322660371962865, 
0.61374574958409256], [0.9322217407076836, 0.38946747476328158, 0.021048032763529623, 0.58962862788025583], 
[0.48450453389061188, 0.5524580527933648, 0.68791856345659252, 0.68692100861913852]]], [[[0.20288666471366812, 
0.21251468699432219, 0.11887844351379706, 0.2331265265972009], [0.43708430950625643, 0.013404275426153522, 0.52141662463283045, 
0.036465497571777195], [0.228587163615661, 0.095012204182163051, 0.83234434801514356, 0.24939455088823015], 
[0.9078003532174902, 0.19314684843395535, 0.30854339659531693, 0.22510267775199333]], [[0.25157928638119031, 
0.54788173275266938, 0.41441682674583036, 0.53263209018095936], [0.15647859461481839, 0.6163414519019067, 0.94367170678459, 
0.95068507087862486], [0.29684171117548386, 0.28665661411484022, 0.60247926465241519, 0.85811491419990515], 
[0.19521841666976958, 0.092821419609968925, 0.01425990608833827, 0.18166094551170653]], [[0.0033036073293121193, 
0.68287440340125827, 0.53602175234153759, 0.54063657348642191], [0.81420880186180022, 0.12890187581694856, 0.33448712242369882, 
0.72683888917771111], [0.19226341222916388, 0.066777094102526857, 0.077437378559695125, 0.52939971297816546], 
[0.61690895797045286, 0.068669013271507984, 0.72734490014589515, 0.84150217751157741]], [[0.078092859145982629, 
0.99832339494776334, 0.4379333777432215, 0.58662391622221299], [0.077977562698650305, 0.73999448359752207, 0.4398007813755217, 
0.10281504856819312], [0.86303228945558985, 0.97553035514179776, 0.61716208080203161, 0.90662715248490922], 
[0.1393701003478045, 0.076790012089785087, 0.2832171603046566, 0.77654529386538829]]], [[[0.21782796590226639, 
0.07512103214570276, 0.98455722959671599, 0.27344728641268701], [0.91857759076970991, 0.42291769455947803, 0.8521597897429537, 
0.56958559534799347], [0.77367575327646465, 0.25495852169270694, 0.12665151363318861, 0.14537389745394147], 
[0.25924138850455236, 0.44875033909235074, 0.9377404074205663, 0.23731283673217907]], [[0.37008450586672903, 
0.070185485995460262, 0.24743756901449432, 0.86604932122346667], [0.05690300064064302, 0.37593474682807937, 
0.16709682344146715, 0.94483922855975566], [0.059310439554258965, 0.73235040102988347, 0.99647669144487794, 
0.95963373810949226], [0.2347697836231295, 0.82537429527495143, 0.56565214937433028, 0.37851067595349241]], 
[[0.78152276710629842, 0.011366634534990427, 0.87740429923487273, 0.030998181459716667], [0.799466766551161, 
0.88221485460042204, 0.90011244476847396, 0.015604467911808029], [0.39219556976440761, 0.76619418926527305, 0.7477533848871275, 
0.48884860223006599], [0.69146799762943012, 0.48379532359780231, 0.047618270405578089, 0.99673032363512459]], 
[[0.62051358781916344, 0.078270774794253262, 0.2567249704646174, 0.023575417864789783], [0.87843429426353281, 
0.6384439792924067, 0.97689046994639417, 0.2650274153038098], [0.85671250627124729, 0.67276371060970652, 0.45410958421352454, 
0.26392228878947077], [0.38475161167572025, 0.48845766221009312, 0.36361276193592518, 0.22245828063603057]]], 
[[[0.84621797783094344, 0.5484481540327818, 0.76140596998492649, 0.87541105306250933], [0.45859718748312528, 
0.6620954059486196, 0.93159551635694915, 0.60410310545028545], [0.31346201372017846, 0.24678555343267194, 0.34008202134198862, 
0.35467637806398844], [0.38876223613015237, 0.024054960104140322, 0.81771625791740221, 0.70642102209610891]], 
[[0.70553790795534677, 0.16500773848825501, 0.41015170751678265, 0.086176818672664801], [0.68164676614638264, 
0.11651999466696039, 0.54400397137119683, 0.51766240590822998], [0.73019918691007357, 0.28673650032456788, 0.98076951933106249, 
0.15915593813986884], [0.57518459258950527, 0.21863322693173837, 0.70647793792547331, 0.78536521379867552]], 
[[0.10842481095741796, 0.99159547299722317, 0.20227277613907357, 0.15623673793097148], [0.53506617197832562, 
0.081032875381117475, 0.28776036934686611, 0.83160714159127624], [0.45782484407745527, 0.4778800373945864, 0.74308397002521243, 
0.068098712817207141], [0.70880168627669671, 0.22377695255510155, 0.72182850040282087, 0.57488717589324523]], 
[[0.89668771054920693, 0.451379233033814, 0.11786580262967772, 0.36808126131044228], [0.42570694822107091, 0.28325902091440869, 
0.60206040509084713, 0.60072114932904852], [0.39367069699174995, 0.64707556103482966, 0.33179150247234113, 
0.22249386846449937], [0.14082245423836326, 0.66402201361650104, 0.90508158294925567, 0.75885527515643492]]]])
        ref=log10(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sign_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=sign(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.353794463458)
        ref=sign(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sign_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=sign(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.0852015469195051])
        ref=sign(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sign_Symbol_rank2(self):
        shape=(3, 1)
        x=Symbol('x', shape)
        y=sign(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.38001318630212833], [0.088382898553667655], [-0.17689791146403966]])
        ref=sign(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sign_Symbol_rank3(self):
        shape=(4, 2, 1)
        x=Symbol('x', shape)
        y=sign(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.72378804122203766], [0.57252446403624169]], [[-0.5622682101967813], [0.82690553559220836]], 
[[0.39377930282697982], [0.099583244262002824]], [[-0.5926447731844755], [0.45345400584270745]]])
        ref=sign(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sign_Symbol_rank4(self):
        shape=(6, 3, 2, 3)
        x=Symbol('x', shape)
        y=sign(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.73675827862553889, 0.91483877403995462, 0.88544806038709822], [0.20289925604258285, 
0.49681253895088728, -0.12993315438114039]], [[-0.41051962546082166, 0.32972258960119438, 0.41612415640419931], 
[-0.3629990084703969, -0.7993222973959675, 0.90461900374117477]], [[0.87380421377509987, -0.64717971923231477, 
0.12041310931561333], [0.011454488872085999, -0.85964731104678238, -0.98824556182761669]]], [[[-0.40224123111678534, 
-0.017132779055906155, -0.86504935027256358], [-0.29728365316699867, -0.91267470135866202, 0.94361155600394131]], 
[[-0.10134896144611116, 0.15937237080487043, 0.38050069387662866], [-0.24698252247994912, 0.35615792130786672, 
-0.96614170258007848]], [[-0.36416287584006146, 0.85971161200977186, -0.043051019340186736], [-0.0021576092624762122, 
0.21350000314847861, 0.79220044489643704]]], [[[-0.73833628469383572, 0.57487778028930947, 0.9540897834401787], 
[-0.034279168411485816, -0.77873691901806796, 0.11719093621659105]], [[0.81893695504296726, -0.48035455150816397, 
0.7060058906946014], [0.47633055348625897, 0.9712024802861563, 0.33421842773194776]], [[0.52575961663641402, 
0.37781161222987669, -0.49864535672447152], [-0.56997962322850237, 0.050166891614832343, -0.27291002496026562]]], 
[[[-0.41595569588634995, 0.52999521006392092, -0.17170459060901866], [0.93873741356129536, 0.098996137061337919, 
0.78191026790783891]], [[-0.47540397412853808, -0.81682572557752753, -0.46468133201496742], [-0.3942392464403941, 
0.24897337387252083, -0.84311524023481055]], [[0.71528938597584202, -0.80049089776389226, 0.98596531426580269], 
[0.73191815193300069, -0.076983885753318271, 0.2817636222245723]]], [[[0.19247617425614472, -0.066366125321580816, 
-0.55279218562390464], [0.017469187199551595, 0.27398633923049531, 0.59435888154527006]], [[-0.62609777052479032, 
0.60980575646670121, 0.96396067133187602], [0.049214315134996411, -0.31355478197115882, 0.28382169982433281]], 
[[-0.92315240462472414, 0.96150500748190293, -0.6845346601166713], [0.75825820500621632, 0.8620004836212849, 
0.79272812163001483]]], [[[0.93710350559885525, 0.45394538309602117, 0.26129074628772386], [0.099486091312288139, 
-0.79989202448405528, -0.74523496658543098]], [[0.495499498536073, 0.4388822729917019, -0.80382922878486629], 
[0.2075710427779589, -0.089380121104465138, -0.19639060882269965]], [[0.26826700293085581, 0.57644629269118997, 
-0.50892587285306679], [-0.099397013894628516, -0.92030594627177797, -0.99008617084621808]]]])
        ref=sign(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sin_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=sin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.0533397015307)
        ref=sin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sin_Symbol_rank1(self):
        shape=(3,)
        x=Symbol('x', shape)
        y=sin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.46452827382534823, -0.16262326349062817, -0.57588880238767892])
        ref=sin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sin_Symbol_rank2(self):
        shape=(3, 1)
        x=Symbol('x', shape)
        y=sin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.1707851119380821], [0.33247864388804715], [0.99324813162455006]])
        ref=sin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sin_Symbol_rank3(self):
        shape=(4, 3, 4)
        x=Symbol('x', shape)
        y=sin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.12719341250253891, 0.94240805058335297, -0.86909746899297535, -0.30140578125466533], 
[0.81825173631703207, -0.43630012106697613, 0.83569604612838755, -0.81416580293443519], [-0.00020301755460416793, 
-0.15560407500011264, 0.54793528510549439, 0.86382840179995601]], [[0.49885493521351609, 0.81462242651477412, 
0.23275443469675228, -0.21674790180269099], [-0.7821624220018919, -0.88051866675451884, 0.65789295590027552, 
0.93234276414295225], [-0.25793802000447741, 0.60279182839018119, 0.50983979332071727, -0.98928307266384352]], 
[[0.66409681423819245, -0.1126276407936273, 0.86789106587410014, -0.81156662780578048], [0.04568617354929061, 
-0.082676650852101519, -0.28588547395675956, 0.84610563527160165], [0.21546862243589504, 0.026663031923359615, 
-0.48787866588477757, -0.34659781857768501]], [[-0.22843463423586319, 0.49676917265717457, -0.39524539046378382, 
-0.92279449190546536], [-0.19074534280274791, 0.89578986906775437, -0.33860659152453487, 0.099385397225141281], 
[0.92327715581396985, -0.29068143560427528, 0.61766952265221131, -0.26263716726458397]]])
        ref=sin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sin_Symbol_rank4(self):
        shape=(6, 5, 3, 2)
        x=Symbol('x', shape)
        y=sin(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.86072117846153251, -0.019885060663697107], [-0.75596420490317695, 0.32635974831722647], 
[-0.92700886052822917, -0.20295271149419603]], [[0.56087967275611028, -0.74593470753875568], [-0.87115021310505059, 
0.65703944484734844], [0.8452445692611863, 0.52556228616041656]], [[0.11340855624086399, -0.99113822454383094], 
[0.1482958797872409, -0.82905278835000318], [0.93562089630108169, -0.46224560103833556]], [[0.64277517260838746, 
-0.40274247637862648], [-0.80510217633866876, -0.5209708410699605], [-0.63301966565861534, -0.068342580535151853]], 
[[0.8263729251130838, 0.050211593544034816], [0.49238702440351956, -0.093240239446997331], [-0.23995676976991009, 
-0.083080595261635581]]], [[[0.61397129959987651, 0.40408480340977126], [0.086350805733938163, 0.15367708691986692], 
[0.27065837019542704, -0.43228842296737868]], [[0.70765273758478364, -0.36798368785345947], [-0.9899723962194924, 
-0.47290522770817822], [-0.62642431407007182, -0.48651185380472417]], [[-0.65402001961856904, 0.9065180648852682], 
[-0.78749946554104633, 0.45203097235082468], [0.97374565220766862, -0.24681403660502887]], [[0.56259917948190274, 
0.69755021533510231], [0.20770189548762086, -0.57023351652855969], [0.98270817482027506, -0.91376351712170134]], 
[[0.55583021063293603, -0.55817864083258839], [-0.33894604692507357, -0.95292619377606735], [0.42765375302141662, 
0.58898103376412037]]], [[[0.24341798478067833, -0.024645179148373053], [-0.67826428166543962, -0.91426078022945045], 
[0.3411072188458717, 0.70658098480681852]], [[-0.87681933497998621, -0.40489430363368761], [-0.32600422164756959, 
-0.58598334874061808], [0.90376876914414406, 0.029824078387123532]], [[0.58874466588853003, -0.82774962699531995], 
[0.91607363862422098, -0.79254174302484426], [-0.2644148749238695, 0.34053053372848652]], [[-0.91577964950737645, 
-0.69146674192578184], [-0.8977183107427702, 0.30048422166514044], [-0.16619420940308172, 0.54616054673573755]], 
[[0.84486754146636711, 0.68988790352389406], [0.46778140349805142, -0.71863324221692149], [0.10225625975272679, 
0.34182079104071539]]], [[[0.91072383294596571, -0.49590171495518365], [-0.038314582897158278, -0.68563330667437472], 
[-0.63313318194534141, -0.91038897945540276]], [[0.24916330989826108, -0.56534062734783541], [-0.17252719027825592, 
0.74957569378346589], [-0.63968063018884136, 0.50177550582088526]], [[0.93568476397885458, -0.93594419899311032], 
[-0.1817447116032751, -0.18999897705574575], [0.29765415309416277, -0.085041137255460697]], [[-0.3428987875787004, 
-0.61494312923538375], [0.77037994058704728, -0.57784965536584587], [0.019942654343778621, 0.31683116278372814]], 
[[0.17715614149827164, -0.78621638086609402], [0.39899437859837117, 0.51609297924211783], [0.65059708642966996, 
-0.094063078212815343]]], [[[0.96758424711675239, -0.89451370297444255], [0.45728487996661871, -0.55848557193517268], 
[0.77237147343588219, -0.68477204485039267]], [[0.41518673719657184, -0.22563771707467861], [-0.40284024326873147, 
0.046432822245995986], [-0.92854654670917691, -0.24510098891604559]], [[0.72928399948698708, 0.2133530398493455], 
[-0.50333016134640873, -0.85457494281524649], [0.82055627088886673, -0.51906877954233743]], [[-0.57871567599408968, 
0.11867295682968004], [-0.75066716147907231, 0.67255813066494197], [-0.35103424868504884, -0.76778391971732729]], 
[[-0.78519947150751657, -0.9261944511980289], [-0.055128778465886263, -0.15239107182086076], [-0.12461725975150895, 
0.56121942561894489]]], [[[-0.43815201173493357, -0.24257698008360418], [-0.29280500267491161, 0.064850361200631035], 
[0.2183869009065198, -0.13294396152029209]], [[0.12385697793008643, -0.91458521402051196], [0.71619413751089223, 
-0.5969526950005235], [-0.98863207961723676, -0.10186755453120067]], [[0.59585107831775264, 0.24631692427877816], 
[0.0068705467302270229, 0.247657304010932], [-0.95973641229592288, 0.68965345238510745]], [[0.24473766858854673, 
0.27037426996287039], [-0.73787209693898181, -0.056540963425711244], [0.62396998885054522, 0.41042344344095616]], 
[[-0.84642610310151167, 0.096445804050030226], [0.99848196110044296, -0.51915863199180023], [-0.13601993230775844, 
-0.60718971932510768]]]])
        ref=sin(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sinh_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=sinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.545758115177)
        ref=sinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sinh_Symbol_rank1(self):
        shape=(4,)
        x=Symbol('x', shape)
        y=sinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.48195185277943797, -0.96876323044125678, 0.30975387948513866, 0.056287466616526505])
        ref=sinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sinh_Symbol_rank2(self):
        shape=(2, 3)
        x=Symbol('x', shape)
        y=sinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.70084091868990073, -0.37122093760650854, -0.98542437006495609], [-0.85293314037311685, 
0.55234563280758864, 0.28910729615623465]])
        ref=sinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sinh_Symbol_rank3(self):
        shape=(3, 1, 6)
        x=Symbol('x', shape)
        y=sinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.6049409851077685, -0.060876864091317096, -0.91287636026437347, -0.42050089588498096, 
0.067752967213508208, -0.81956681283255439]], [[0.98432036228063402, -0.89186551144514703, 0.71981933267454967, 
0.47233832880070592, -0.54141614539879579, 0.30254425885779779]], [[-0.47418325474652412, 0.96433929261367002, 
0.3681263669352226, 0.036021997887072388, 0.18826942276317205, -0.61084148500472191]]])
        ref=sinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sinh_Symbol_rank4(self):
        shape=(1, 3, 2, 1)
        x=Symbol('x', shape)
        y=sinh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.25184670553734478], [-0.90193559353129849]], [[0.012157397905095735], [0.78463782528086501]], 
[[-0.99880314565239048], [0.73395648897790311]]]])
        ref=sinh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sqrt_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=sqrt(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.824688116082)
        ref=sqrt(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sqrt_Symbol_rank1(self):
        shape=(3,)
        x=Symbol('x', shape)
        y=sqrt(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.60033987577973147, 0.53929810926291055, 0.81544302583867656])
        ref=sqrt(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sqrt_Symbol_rank2(self):
        shape=(1, 1)
        x=Symbol('x', shape)
        y=sqrt(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.26884009957207955]])
        ref=sqrt(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sqrt_Symbol_rank3(self):
        shape=(2, 3, 1)
        x=Symbol('x', shape)
        y=sqrt(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.65496767364217046], [0.67783962327300917], [0.90007404601376473]], [[0.071997103388618311], 
[0.046235858883322378], [0.57362450572208434]]])
        ref=sqrt(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_sqrt_Symbol_rank4(self):
        shape=(5, 6, 5, 3)
        x=Symbol('x', shape)
        y=sqrt(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.72255651065698279, 0.83381610263629191, 0.088396309723070443], [0.79855742227758819, 
0.839338005541306, 0.81495790516402389], [0.62626755304522941, 0.078799158362552024, 0.33267140687945473], 
[0.43940707734260109, 0.51599082070096458, 0.84321011444082217], [0.24171707936068787, 0.0060558032715870569, 
0.56078845976765612]], [[0.57377382745965855, 0.49340859714655683, 0.2092972590061013], [0.68637889059321167, 
0.11029705930136668, 0.96396795238233834], [0.55179714688174419, 0.35108623014240958, 0.51181447099189581], 
[0.32324557912243723, 0.78654583988331572, 0.33712954059082634], [0.079321394582968718, 0.23774608497170258, 
0.69285790426559601]], [[0.85886317386456423, 0.13584169380100597, 0.24292242178587076], [0.61177645246068624, 
0.3036432086030022, 0.53035500209869535], [0.28628042513462981, 0.24043761219547199, 0.94891705021835726], 
[0.60395478182149109, 0.083245941455203476, 0.13027004597807579], [0.15048976641157319, 0.11401314017400843, 
0.18795600878276586]], [[0.09137780979901966, 0.32634976420570005, 0.61940335197091201], [0.37432581452719493, 
0.97387697903168757, 0.60713072535399282], [0.81126283730616289, 0.72286470992579144, 0.33277102567004557], 
[0.65725608926170365, 0.22113878439984025, 0.39873142337432677], [0.56458095772442962, 0.69984805698516706, 
0.13953969142708544]], [[0.14180888147181303, 0.6449607100547553, 0.31077677774512735], [0.78970211385749289, 
0.35440985857825957, 0.33586351342060605], [0.31861742504282387, 0.0091224852408655632, 0.72485922708603734], 
[0.19028038521297774, 0.813785636029978, 0.96553983147048672], [0.91561864871190557, 0.11116693352450835, 
0.70275695888285916]], [[0.93465367590426696, 0.08776495337568968, 0.92008576270779563], [0.55687479001255391, 
0.72198162368591801, 0.93723076623527124], [0.62999175799626039, 0.36064429891165684, 0.59775442438928683], 
[0.22690790215469459, 0.4685878404511844, 0.89223770066308428], [0.92972243126999998, 0.46025840602562273, 
0.40968201803367088]]], [[[0.94180597092282947, 0.96234744551267848, 0.87288463327476029], [0.41806750845644847, 
0.22788237798369737, 0.11966406067717961], [0.69733599374787714, 0.18851158381113808, 0.16370374653731978], 
[0.63211076175223446, 0.41154348987607725, 0.98533970125915882], [0.65773473183219433, 0.041250994596582657, 
0.96992667507584374]], [[0.40058859823800097, 0.12179652261476415, 0.99555017499337206], [0.68443796318239791, 
0.88112987266527754, 0.6935433018004582], [0.85006065726887914, 0.79136227813817728, 0.44775493806515188], 
[0.17365733710621323, 0.15978381135481545, 0.71519613537661453], [0.52661904411861471, 0.52676789546431813, 
0.48249638132196371]], [[0.19890220183563068, 0.6528046793768485, 0.39835999493439911], [0.30149513914196047, 
0.41004970280230157, 0.095723514003266286], [0.59342390611233331, 0.64188662999081492, 0.94573923409175364], 
[0.97182053826713977, 0.030426067513154798, 0.61174589100753962], [0.72835687654810322, 0.68713012593738343, 
0.31568564502713259]], [[0.95749206532668674, 0.93726678894579174, 0.4906774028313089], [0.38333713090398269, 
0.31180194657763527, 0.87930430515980695], [0.43976370526311137, 0.044494145039117527, 0.74864265600876034], 
[0.26436773182558582, 0.13762336840153333, 0.063845322270331906], [0.28527952524657374, 0.64364843195737631, 
0.47408670386667384]], [[0.61761106212360073, 0.62971040192959593, 0.49470735386519726], [0.41306026687735919, 
0.55507624155570312, 0.92853759222310106], [0.3274674257337109, 0.87896980125888136, 0.39331161497503286], 
[0.43611526427398251, 0.72534207729003164, 0.57962529848022637], [0.69210299406574527, 0.062460995174160816, 
0.93066659126857221]], [[0.083111660192682568, 0.99725328716841311, 0.27121278820489403], [0.73406289503925604, 
0.80538390625619516, 0.29315139024767889], [0.082676638871250652, 0.37206452764369702, 0.2649049794742897], 
[0.12408227322470267, 0.74472176475463692, 0.8174354535597258], [0.21686067449726132, 0.62883396701473271, 
0.86566439632322278]]], [[[0.89669530227407024, 0.19289830337776714, 0.089853089927319219], [0.98760388655662767, 
0.89008462181980053, 0.69570048836621301], [0.14682408248171763, 0.54479399309445797, 0.72720013814912943], 
[0.045069230893828971, 0.11461994792470509, 0.27430837116265094], [0.67604305313005575, 0.052652824417676269, 
0.63863799167616264]], [[0.74279794050475256, 0.098764285738637048, 0.90391176265205009], [0.031888689832916151, 
0.91362524779923027, 0.825151695764614], [0.013540853593327129, 0.13613017881117784, 0.94594185269170306], 
[0.0055776780082078536, 0.19975775247355665, 0.072259244456054361], [0.74617514477019764, 0.32471755948895109, 
0.083830790467844252]], [[0.66811203288133636, 0.27633938675577341, 0.040893602491139269], [0.51849181901456176, 
0.011501661678884445, 0.77217244062645096], [0.14375142366801896, 0.90381912060700553, 0.20190196091630164], 
[0.21322538371212196, 0.36585148392876865, 0.083727919690708918], [0.3097408597687018, 0.59723759835045875, 
0.4886767026543215]], [[0.16357973246609758, 0.92087932544946327, 0.3384708586629751], [0.067136871978869417, 
0.21776940016537627, 0.7705745192216632], [0.98259021799791124, 0.50961837355614692, 0.83555050647205176], 
[0.67608441880904768, 0.90312717890505212, 0.18053751246433791], [0.29700456036496337, 0.37000347516593268, 
0.96452501688322201]], [[0.32688708306221248, 0.8365877691645266, 0.17336317193714579], [0.90487915797856355, 
0.50717605535081722, 0.33815521397532988], [0.11780315669470454, 0.84781686200603812, 0.56387171418123161], 
[0.025302923773783292, 0.51725791550326283, 0.091383009756073719], [0.77550897972876764, 0.78383408486529027, 
0.39861904820998628]], [[0.38591849764482156, 0.88954837248020835, 0.94854305672295536], [0.57116041017796104, 
0.26025685153372258, 0.087398933964749181], [0.9500891765988958, 0.91514518514695409, 0.41455805101908916], 
[0.49709796581897214, 0.58623087645333538, 0.99567144777732952], [0.46118329193118057, 0.5775973942356375, 
0.50373425281250117]]], [[[0.50121029052250199, 0.97961710518990397, 0.25968884629076505], [0.34014901845291678, 
0.46162800321732056, 0.46244053814165542], [0.52473426277986934, 0.62035771009638596, 0.67250736503998665], 
[0.38672422756992952, 0.2779946669756711, 0.20218957593196996], [0.59604573638981928, 0.024040951338803018, 
0.61362301053778479]], [[0.83964586000872143, 0.99137647765888626, 0.3213278073048903], [0.50226612101424373, 
0.63425549308364282, 0.26181637288077153], [0.70988178201479624, 0.01605697643006665, 0.65300994192816364], 
[0.66902642711157012, 0.12496171226451158, 0.97775281702208727], [0.43836631773999302, 0.65765546467850677, 
0.98052750508062603]], [[0.90539506049457286, 0.5477116805032417, 0.65052354082147279], [0.24924938940701435, 
0.33302080748158636, 0.42271974362844567], [0.49401780858839472, 0.17866077442511874, 0.13806221412494846], 
[0.15760354547519873, 0.427519458168096, 0.53801604177513895], [0.94249603564964812, 0.12238866172772112, 
0.44347748909798668]], [[0.077420345199719076, 0.15127406852408509, 0.66676703504369916], [0.33541924877656193, 
0.58965539531548805, 0.59602448266722396], [0.74814540749654013, 0.3405615434346374, 0.23258421912054961], 
[0.34254483202651698, 0.50051875556983916, 0.7066991552934766], [0.51876294455266225, 0.23871624932785562, 
0.66835953935309655]], [[0.080238481120057381, 0.10002775155524224, 0.27219972402671833], [0.66024077280618931, 
0.78142461690690623, 0.02138659211584526], [0.1733046507979048, 0.83740599399143623, 0.24763289061444038], 
[0.69604609867940004, 0.62515136615997291, 0.27251249907669461], [0.072460996251230703, 0.45182139135661836, 
0.73082092555678158]], [[0.088423654573769661, 0.18339223736193833, 0.02175849452611256], [0.6058055700809617, 
0.50324964416184914, 0.16445448952327724], [0.79230850413129672, 0.027228331505205983, 0.18313393017353108], 
[0.40327962525921779, 0.53399983584420663, 0.2644936614840051], [0.95300193272232647, 0.18083330185835367, 
0.96908167133421397]]], [[[0.91385184246409168, 0.58113677477322401, 0.70231008005224593], [0.9474537951253309, 
0.62016324060777628, 0.17138498474532138], [0.34849671492753898, 0.46869766397564316, 0.73763896217058134], 
[0.87211066934174009, 0.48734126881749473, 0.64174237807970491], [0.79905289900338639, 0.42435051812848124, 
0.68150971117710202]], [[0.36588037184075373, 0.79537939361517529, 0.85181903232783007], [0.41141769837196351, 
0.86786051623070848, 0.22414117656064658], [0.81640895923110746, 0.12992365640477233, 0.2355496056791454], 
[0.45371835055421261, 0.78160552803709338, 0.092958530073615653], [0.41049364219195672, 0.42702222368276033, 
0.070574576044564519]], [[0.45831121140198006, 0.045085310497666065, 0.79294289698788456], [0.44302959532218689, 
0.65443298771574432, 0.68173823126947375], [0.39094486774242743, 0.066819108786377535, 0.6808460172331281], 
[0.5046311706772918, 0.32051470604316801, 0.77716245540800988], [0.75707151715004528, 0.67614765942702015, 
0.83284890164722847]], [[0.20347799733225258, 0.71006918737014257, 0.92588254441682205], [0.96207595455734829, 
0.49552591933007439, 0.12398037176865617], [0.68313730954291374, 0.10415236445564258, 0.45816046027438184], 
[0.36162415054339936, 0.99131438275747452, 0.22372131074708346], [0.67086419049901891, 0.47845701936987228, 
0.36985109558546647]], [[0.27775993248683417, 0.10092563911048058, 0.72711226482933478], [0.27741437292163063, 
0.097359636516875292, 0.81506667907392638], [0.92134062138716277, 0.42785167607992813, 0.23324025474365606], 
[0.34160362940981959, 0.41289016296485281, 0.78760263706808065], [0.0083578369819241605, 0.83144296140885887, 
0.33782688147606554]], [[0.46357765230192827, 0.13598683924850008, 0.99588735461627897], [0.84918521935812763, 
0.023911125125060662, 0.44141960863528895], [0.15630046849864843, 0.27771188177990347, 0.34981662538526515], 
[0.45994233771683468, 0.63477232786494087, 0.97573238156786168], [0.57208602704347389, 0.11631534583474779, 
0.86853091655212733]]]])
        ref=sqrt(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tan_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=tan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.804341597147)
        ref=tan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tan_Symbol_rank1(self):
        shape=(6,)
        x=Symbol('x', shape)
        y=tan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.14200146104850608, 0.55199436405695579, -0.72437980114165357, 0.54709933137129374, 
-0.87572550813546113, 0.54367761243658963])
        ref=tan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tan_Symbol_rank2(self):
        shape=(6, 1)
        x=Symbol('x', shape)
        y=tan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.28165724053839392], [-0.9979812387072815], [0.6196421842440949], [-0.55984839501305284], 
[-0.90025481208944447], [0.039912238206123662]])
        ref=tan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tan_Symbol_rank3(self):
        shape=(2, 2, 6)
        x=Symbol('x', shape)
        y=tan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.32601211340980618, 0.63454643711267789, -0.45601589073511661, 0.74928379052525806, 
0.044787945982150079, -0.84750426268279266], [0.59672024350466324, 0.60118070196120521, 0.73000920374303635, 
0.95615655945060252, -0.16391789424129, -0.93003036732378042]], [[0.33600995453074667, -0.33380626353926068, 
0.32296628695685414, -0.84507985878052083, 0.45550744326858905, 0.92956275280795997], [0.092215953473159473, 
-0.92624398799798224, 0.88877673878896624, -0.90947264228186619, 0.91763999125337503, -0.23639346745535073]]])
        ref=tan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tan_Symbol_rank4(self):
        shape=(4, 4, 5, 3)
        x=Symbol('x', shape)
        y=tan(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.39361167932986962, 0.74522581327046944, 0.60340274502183977], [-0.6594807460274732, 
0.58589637220463398, -0.11734763382586055], [0.80638763124208013, -0.64402850551432045, 0.36021742857944283], 
[0.021008293536134204, 0.97682781575076816, -0.10551371038954671], [0.45272674007823088, -0.85178595769433008, 
0.55624629341747567]], [[0.31391820874752718, 0.21075784758815663, 0.26314878859007695], [-0.98448333512712716, 
-0.72386430989711426, -0.26227979662440482], [0.96810162374605735, -0.83650196524727338, -0.59800258471380374], 
[0.23607297469872379, -0.24038831390187276, 0.11086313735031772], [-0.39471124201884034, -0.48352767341106229, 
0.21186754666535212]], [[-0.92540364716984591, 0.39613198860468768, 0.40648440480928327], [0.47970852348990078, 
-0.51595317096476734, 0.019379089520323856], [0.99883716946833001, 0.20405300322814135, -0.28044851950989402], 
[0.60005623011836895, -0.41740832183053578, 0.32623688963170117], [0.84656326181286734, -0.17647518491381819, 
0.82349636481198618]], [[0.16905962725378498, 0.98168860100078037, 0.9353114074077038], [0.19224595065824857, 
-0.066366531515243965, 0.29121472159397022], [0.86700433635139884, -0.79216700487801117, 0.69402792626896992], 
[-0.38719928456391628, 0.48160408888843276, 0.8230283146023567], [-0.034941545895473158, 0.11207736007529601, 
-0.76628638474929756]]], [[[0.8686966875749631, -0.62065531097846649, -0.46180638263063289], [0.11048319908775661, 
-0.94827689038381124, -0.55165165131652838], [-0.22050584520475791, 0.40277438568381108, 0.14008031400767562], 
[-0.99935387872751602, -0.71543940516080284, -0.43778571710569425], [0.37798157306313129, -0.94338234063989135, 
-0.42152333436894796]], [[0.43703301877468159, 0.39748667010044048, -0.48045382845548423], [-0.034882375838574564, 
0.93307896081154906, 0.5563628429503451], [-0.94652875131975178, 0.75880890900809206, -0.51068487819970043], 
[0.3221147597656433, -0.37159197684999445, 0.34493062162810562], [-0.60506673953023959, 0.51339586409976645, 
0.56922924643395412]], [[0.29716600159661311, -0.17816615862073992, -0.86445473447538435], [-0.35327015227085168, 
-0.56939353868570652, -0.085554286062446749], [0.2790841392787613, -0.34591132364831556, -0.72410488581237797], 
[0.37497587768910234, -0.35746717094914926, -0.056680969361326827], [0.18448099324282152, 0.14874815449357204, 
0.59801648660396767]], [[0.90371247560747991, -0.14391938108674629, -0.70974621137275418], [-0.40373973907758876, 
0.52028563903355174, -0.89764982436604335], [0.29560404854999356, 0.87638099062861352, 0.50647543392228189], 
[-0.62959804758651794, 0.76104749591985699, -0.65246519534456882], [-0.21062711278245239, -0.59382460014460814, 
-0.71289385582368081]]], [[[-0.057250620340783653, 0.77854761025428609, -0.38648357938612898], [0.45542175255587547, 
-0.19609171451182972, -0.63002638687212631], [-0.85247705442935584, -0.84366299228257202, 0.09426362678490352], 
[-0.016577883362343337, -0.74879437402571525, -0.68366817051255491], [-0.33604040313252814, -0.27164630945300972, 
-0.99475638768020791]], [[-0.70130149091787586, 0.021562173043616184, 0.028213936211842094], [-0.78481956059302371, 
0.15649596053059134, 0.46454870329897213], [-0.9149659414389737, -0.22891758886102731, 0.41037262794506679], 
[-0.21207892294129871, 0.98005671987894072, 0.20078126352133818], [-0.90786908894320839, -0.9292322436731113, 
-0.93804227944070973]], [[0.66602248301188327, -0.036918117482479529, 0.95756625797125583], [-0.98482796202997291, 
0.74646671050364821, 0.37761108311585767], [0.70069203580671968, -0.026580657560474341, -0.52863034488962501], 
[0.91109958034323801, 0.83013879029514004, 0.86314305237022082], [-0.15983859961928082, 0.26188047697492101, 
0.4520224761020788]], [[-0.23753974221656171, 0.69903406974916571, 0.078831077157341722], [-0.60754659217688189, 
0.87886737408616855, -0.16926573195254613], [-0.396356714566515, -0.45995800030317469, 0.46968566755059604], 
[0.004171700121168076, 0.18929502726741632, 0.74122834576831154], [-0.88950872921628754, 0.85812775846443068, 
0.090850855338491909]]], [[[0.94272405729076003, -0.18978264009146129, 0.71108734775582194], [-0.60356815477109649, 
0.2966157597336696, 0.60844941792654161], [0.00070885034533185731, -0.75060628565340282, -0.42600443605384442], 
[0.096238135067595154, 0.25901881694088336, 0.91590062410211792], [-0.39030807410637514, -0.75907117662165935, 
-0.20254736582842359]], [[0.58754460092193561, -0.24371471431563485, 0.58627489581326597], [0.88913896291886241, 
0.67488959799173909, 0.93377638758907922], [0.48859512249307491, -0.44057956114388985, 0.52751079353868247], 
[-0.88386673247457836, -0.92181365018497652, 0.87664934709731623], [-0.093747856591055889, 0.50375815759268705, 
-0.10854436426015468]], [[-0.36247514070986986, 0.28805372072027136, 0.0074580022781263189], [0.21178312012491851, 
0.17124067789062924, -0.81397797570729957], [-0.45432214958372841, -0.34287904328486429, 0.16559848179841796], 
[0.32622875399077267, 0.8831115716331901, 0.49293335318565323], [0.37786144819047274, -0.64318613413115933, 
0.37213148926932593]], [[0.45778076631363485, -0.45488393452463471, 0.86244580239817781], [-0.54667031890171836, 
0.56956513702652378, -0.29001978466158107], [-0.27371847261131044, -0.26805148200740891, -0.7138030282340837], 
[0.86535912062485276, -0.36626574115645316, 0.69491108145358949], [0.7486178910065886, -0.32082327504674524, 
-0.88028371669550376]]]])
        ref=tan(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tanh_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=tanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.831011926818)
        ref=tanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tanh_Symbol_rank1(self):
        shape=(6,)
        x=Symbol('x', shape)
        y=tanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.39196480149974411, -0.54251526921511051, -0.52139480864543208, 0.17868359648887799, 
0.0089005037204492599, 0.23043826836232895])
        ref=tanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tanh_Symbol_rank2(self):
        shape=(3, 5)
        x=Symbol('x', shape)
        y=tanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.48553795664137289, -0.54266228456823828, 0.28217202264662422, -0.29216615510382327, 
0.50459507430806538], [0.93381590221704958, 0.35428267816381576, -0.20203290604780344, -0.84060951155846686, 
-0.92373613333293636], [0.91344206146861673, 0.59459160694248836, -0.064996620389091264, 0.72847859506056656, 
0.41412421220478057]])
        ref=tanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tanh_Symbol_rank3(self):
        shape=(3, 5, 3)
        x=Symbol('x', shape)
        y=tanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.94545473952359593, -0.88260767783755489, -0.39608996855442458], [-0.38084063931365431, 
0.0084523510667395829, 0.48032048546370709], [0.52747664660496429, 0.24265492367668973, 0.18447217429205365], 
[0.84444752845755211, 0.09694699724475675, 0.93071570178439722], [0.78815805833653085, 0.2650987641752085, 
0.1453372965779669]], [[-0.70234909882964303, -0.54019981751523027, 0.75060083688122536], [0.43383860659626383, 
0.78214973033753021, 0.99438285869115606], [0.50897931090996273, -0.40361560071442049, 0.75175326664108044], 
[0.078491553080030974, 0.25640940637265919, 0.05849443439337354], [0.45036803442872086, 0.34400943330161837, 
-0.32026785810909519]], [[-0.17242359211073288, -0.76525329107995455, 0.1203010267264395], [-0.75378149034427544, 
-0.13784972986763488, 0.082743915100908838], [-0.11608570364014636, 0.6827892088482328, 0.82572499503393493], 
[-0.028093708749234114, -0.82979850639587394, 0.017586892786316755], [0.15875914618255837, -0.80165770170308548, 
0.98409276246266053]]])
        ref=tanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tanh_Symbol_rank4(self):
        shape=(5, 2, 1, 2)
        x=Symbol('x', shape)
        y=tanh(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.36925408088861333, -0.8166319668013422]], [[-0.89110935322929175, -0.51815528579516568]]], 
[[[0.062831611367165863, -0.97227775668697092]], [[0.10014974357021189, -0.41000660621777607]]], [[[0.94663237418617285, 
0.10102569077065815]], [[0.50958152199552398, 0.60908667764083657]]], [[[0.82223521505244812, 0.090202510262551217]], 
[[-0.25598596995961342, -0.087247739516351563]]], [[[-0.65399577676006637, 0.47061701394300859]], [[-0.066621211630382149, 
-0.28358502274278785]]]])
        ref=tanh(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNegative_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=whereNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.0803828681153)
        ref=whereNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNegative_Symbol_rank1(self):
        shape=(3,)
        x=Symbol('x', shape)
        y=whereNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.99636091172197294, -0.31212268186606806, -0.41102067649863772])
        ref=whereNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNegative_Symbol_rank2(self):
        shape=(4, 6)
        x=Symbol('x', shape)
        y=whereNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.14011610947358055, -0.38805302447048895, 0.47561730929995116, -0.77667238101643576, 
-0.37986729287196019, 0.72856527351504985], [-0.67583271624282593, 0.92113521837041623, -0.35285985999595937, 
-0.00072694303930620485, 0.30706411106352749, 0.26488405944565829], [0.051641850619284835, 0.26502732831505638, 
0.90212370337781222, 0.21647194253753432, -0.70321392303226737, 0.833807393173152], [0.43233299002919967, 0.45638736339005925, 
0.10986686261026546, -0.67368624490597528, 0.80637536196952175, -0.47101703933152583]])
        ref=whereNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNegative_Symbol_rank3(self):
        shape=(6, 1, 6)
        x=Symbol('x', shape)
        y=whereNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.49709022708318096, -0.9991658813647768, -0.68067770532082417, 0.74988986529507029, 
0.24231678461175354, 0.85378207208974777]], [[-0.76040345958316724, -0.059898982578544979, 0.10810456256312384, 
-0.91104887556788361, 0.40065022601288525, 0.9359618888837109]], [[-0.019273629089193278, 0.6143577869376613, 
0.4856067117008025, 0.30228820566695114, -0.9928344551428856, 0.66465766019824812]], [[0.2680662661555, -0.18993888196503383, 
0.70276125137285428, 0.332625441549254, -0.11206420205646839, -0.85817986798511914]], [[0.25050673309941041, 
-0.7934972165131271, 0.13564988832164571, 0.76169589157055451, 0.82021914074941371, 0.50506063253799782]], 
[[0.015631009039995147, -0.61471460656535504, -0.078217724751007411, -0.96164985262782943, 0.33996006854056282, 
-0.1453091316545736]]])
        ref=whereNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNegative_Symbol_rank4(self):
        shape=(6, 4, 3, 6)
        x=Symbol('x', shape)
        y=whereNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.18974799375992091, -0.89108848362168769, -0.99714838299326281, 0.15249233338699208, 
0.35957697555334356, 0.0026964635517325863], [0.81288676358349488, -0.55497556685445781, 0.14014349703379936, 
0.53500622875005144, -0.49468091449001084, -0.38934372479683033], [-0.70361562777255093, 0.52526544230973138, 
0.5236459438683021, 0.65030043386032688, 0.3879680498655822, -0.7772188274809384]], [[-0.89339417655282438, 
-0.75228567087285692, -0.37606670786279861, 0.91297244769514485, -0.19350022841894088, -0.87375837452688709], 
[0.46694468277904244, 0.30798220921990649, 0.31554988566138453, -0.75648202290336819, 0.48383147595272513, 
-0.81399300811832087], [0.60296097170486984, 0.021356348549628423, 0.2607642307185789, -0.94952078952058128, 
0.24260355482321927, 0.60906558053832982]], [[-0.38066433290396806, -0.94870340225840488, 0.16058623987334486, 
-0.99611411941266459, -0.82802023608626008, -0.11801781120206933], [-0.74951880715332653, 0.6038713537010032, 
0.54236727703212351, -0.70469285254505287, -0.040976379652611472, -0.43390087339746164], [0.68267152175592405, 
0.27835218888779001, 0.69871414599867321, -0.79259665551123226, -0.30008126454631867, -0.70180994888941872]], 
[[-0.68594092435762111, -0.053075832861085859, 0.07887458075822984, -0.14218846086926518, -0.51052555466914074, 
-0.32701355014331757], [-0.63649710059441955, -0.86040882988643008, -0.97385072027451192, 0.7917511606704617, 
-0.84187256165209789, -0.40101814099489896], [-0.077746097857161134, 0.30112420220774094, -0.76818619783445397, 
-0.35794543999325557, 0.33887985064692661, -0.82990621623696281]]], [[[-0.80690628106308626, 0.40109497567058439, 
0.52142561477784732, 0.24390447050316655, 0.87195861906037386, -0.090738828411722539], [0.45782011983131365, 
0.31251675977073212, -0.93522614783781988, 0.43991924129339566, 0.35726314361532774, -0.58360258407955734], 
[-0.51216694640659233, 0.51193580936832839, 0.64455230444360367, -0.67077362526142492, 0.096158543871722291, 
0.043800334104726169]], [[0.68096525264272167, -0.98242719399176193, -0.89313016799388811, -0.13915770300695951, 
0.96356700295681974, -0.71897653548488294], [0.9636830744321212, 0.63482782004923011, 0.010919100922413438, 
0.69878620109067091, -0.086215388227256495, 0.51046997837604824], [0.13411174149486715, -0.34521163824355261, 
-0.21894617654166271, 0.020868122216789686, -0.61656485953710005, -0.77343621011717145]], [[-0.9577139723603536, 
0.97035643539947269, -0.98463097936228117, -0.55864742528816325, 0.49901739448068172, -0.17619766385790103], 
[-0.55986434762291548, 0.41413001853387543, -0.22505413307320854, -0.92533317691440242, -0.75109592815075787, 
-0.49422235332136277], [-0.25134953219705047, 0.24150314538117468, -0.24748864789190872, -0.7284540531082766, 
0.9698307658796026, 0.10010963500185732]], [[0.67563784112332703, -0.90031344983507933, -0.048220490193214394, 
-0.36987816266014217, 0.6825046071919243, -0.37146472268073771], [-0.74424957729773533, 0.65990647561505877, 
0.81724935924684905, 0.7646269460748718, 0.46476390448806915, 0.30739813222244394], [-0.91206389966747525, 
-0.88333604683083222, -0.15746144446950261, 0.19693990233037995, -0.90169038562510972, -0.10237160713244786]]], 
[[[-0.97880009174036808, 0.13099119007059579, -0.11348428836676572, -0.25551815297419789, 0.40969232568856606, 
0.51579541143530339], [-0.12632132927937034, -0.94200103769019061, 0.69233945032092947, 0.94323068263860388, 
-0.670629898238428, 0.33667996562186575], [-0.50312943390465548, -0.77477287887478719, 0.54616957044912318, 
0.53101149021329586, 0.43875470195025401, 0.67413263293162506]], [[0.91274190318547022, 0.95377885620910474, 
-0.10452164583464052, -0.34478639159056756, 0.025354367755849649, -0.37141703424550832], [-0.25394650784661099, 
-0.12353479418147018, -0.21756821989854092, 0.7466304578848455, -0.62278337419276042, -0.35646304594085398], 
[-0.40933239644373853, 0.8718571933046062, 0.74817849767028122, 0.88801850462608267, -0.74106504733924905, 
-0.46883403300100168]], [[-0.08032948625767844, 0.0343491049345237, 0.10179424674738691, -0.281192978792977, 
-0.88090175041305319, 0.0068611680228787275], [0.19524094466058206, -0.53620296581105431, 0.38940209010553817, 
-0.88389309286710249, -0.65162896721296426, 0.46257338110776325], [-0.11939963998961844, -0.12459170323504432, 
0.56721445924492242, -0.5434934377046885, -0.94097319686236203, 0.10843818162615082]], [[0.94156569655681088, 
0.97937047999438565, 0.58328860247766023, 0.49999512882630359, 0.98498409635330897, -0.77617703005476235], 
[-0.79368894718950678, -0.49094875856870313, -0.47268575315274686, -0.48423087666296949, -0.15705360364959353, 
-0.092347853598065299], [-0.43773529882492035, -0.97330976684408221, 0.62275243482370546, 0.16243417577414521, 
-0.27746950274070858, -0.17634930756002976]]], [[[-0.57257563296277136, 0.051131913300164644, -0.095495549766847265, 
-0.19510866832630236, -0.71009423852663178, 0.76085183088330033], [0.060680252469367701, -0.77048545231645305, 
0.26366428280734966, -0.69041926752783755, 0.718088216916698, -0.49745431621938097], [0.12318995194046978, 0.86936537085873766, 
0.83918180976058321, 0.68380718722273914, 0.93333036673857861, -0.40802153642780503]], [[-0.39080649731083716, 
0.89846016192617917, -0.43753436018584635, -0.54164172029960556, 0.26500666447283927, -0.80442084034012318], 
[-0.61280888117488774, 0.19901497307242777, -0.38809942782108231, -0.15304848336510446, -0.89682030092543341, 
0.81682968047844762], [-0.37083147403347305, -0.5694861159060336, 0.82079574997354832, 0.16313708350728251, 
-0.96148263023930025, 0.61619631755008997]], [[-0.60218852847857085, -0.30606863782057858, -0.37854212049919034, 
-0.6189813873806298, -0.27466810355256932, 0.11312041334540823], [-0.24450376261524176, -0.3446492538039585, 
-0.11428337189518412, 0.86141817541072485, 0.01994360774629178, -0.9869163482668879], [0.83076429114401584, 
0.90701559122745246, -0.11515468013785246, -0.35636953652376158, 0.60173165224519298, -0.0041944503459219185]], 
[[-0.72486769047997224, 0.11125913628158135, -0.94325817369156395, -0.22227144314457647, 0.73184381474788585, 
-0.42269341360248136], [0.3322788694093366, 0.48675876476849655, 0.95199984348648936, 0.77336176266100121, 
-0.75298460572165227, -0.64919265433333395], [0.97951110380594075, 0.7606183719435895, -0.92931969801866887, 
-0.76656529512374871, -0.96565563744445226, -0.44206017089366356]]], [[[0.06437051353673251, 0.16064165673082509, 
0.66075872754206277, -0.024405885296694185, 0.83365693142502217, -0.68157234542080869], [-0.46145077162972359, 
0.44283796650491913, -0.25107173989879983, 0.99700063016208107, 0.63626965157409332, 0.67543716935101461], 
[-0.19144043467308913, 0.0084624137917301834, -0.011459481336658861, 0.020765982666997251, -0.66394566462368454, 
-0.13595799606345849]], [[0.48666927295430273, -0.54018652177965421, -0.60712253356220014, 0.86032968494357509, 
0.93646381542687074, 0.62229000542074231], [-0.48204694443669815, -0.043441074127007484, 0.52089066069776302, 
-0.14263968353974321, 0.67784661619051523, 0.67683914159762426], [-0.36406246921881258, 0.52484974557021125, 
-0.83945534497593077, -0.56626235962196247, -0.92874439106110906, -0.84316505863262137]], [[-0.61694738907429159, 
-0.84155413094960885, -0.059449787766562068, 0.75512682823773636, -0.47056126199970127, -0.94014349555090271], 
[0.38488296050771642, -0.68470636052993017, 0.9360344045616662, -0.38978549121193184, 0.25843508056533948, 
-0.4154997886443399], [0.98397174568093337, -0.17687534177156738, 0.15000998147481748, 0.67926300051950217, 
0.70188052710294779, 0.76571096090863833]], [[0.10520283712355649, 0.46358590529577093, -0.23107955221500487, 
-0.10221032014082643, -0.40083861723237568, -0.79077105783256307], [-0.45962678686990333, -0.37161803581831654, 
0.90407168311308617, -0.63118786505135183, -0.87881942461714035, -0.50043032112907881], [-0.40905017398749832, 
-0.78414396535965936, 0.52243010681437196, -0.34857967525022726, -0.79063821885581809, -0.44655427796158786]]], 
[[[-0.26830770663674852, 0.052309820978976962, -0.71260517345213703, 0.068649426504235889, -0.79584911074378639, 
-0.73341766864885805], [0.73182847261033501, -0.96117438640838193, 0.13525338817090482, -0.45877788351575455, 
-0.8514202874979051, -0.80721379845257357], [0.17064064304210924, 0.64156952246470644, 0.8579149441905618, 0.42505886722614017, 
-0.84192631535491813, 0.07347356074778566]], [[-0.0508167938616777, 0.79593473366420087, -0.18915387572251285, 
-0.78860288867907991, -0.78833850850429199, -0.034960906396380276], [0.63553534416742985, 0.54104290380014675, 
0.83137356210702995, 0.21568011240898222, -0.20580699671693359, 0.521832276659004], [0.87116328689755851, 0.92974282626786442, 
0.15936378842429577, 0.69133337125801075, 0.7608669199256719, -0.87784165352731791]], [[-0.75122215790102453, 
-0.11551204851061558, -0.95370414115103053, -0.59947328175015779, 0.30395159242315106, 0.13066616906441419], 
[0.061420488872834911, 0.30147702994470338, 0.27861956239959107, 0.28579025178671458, 0.36624849173491181, 
0.72786096607107176], [-0.949934981054112, -0.013206267446552067, -0.22928309991217377, 0.11095083320713495, 
-0.69546530710777343, 0.64638482716139301]], [[0.25305577703676563, 0.42897469852970049, -0.84969441201530493, 
0.49601628592368896, 0.18913452811455134, -0.42761115621334911], [-0.71085347752670947, -0.2320109793524352, 
0.56299071564628056, -0.47484796914866267, -0.11629577973670235, 0.2662526509891876], [-0.60078256931253216, 
-0.68333526085038865, 0.22683430437623353, 0.9676042368650819, 0.77154035736113635, -0.11928404003074666]]]])
        ref=whereNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonNegative_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=whereNonNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.967827875589)
        ref=whereNonNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonNegative_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=whereNonNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.097172211113862872])
        ref=whereNonNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonNegative_Symbol_rank2(self):
        shape=(1, 6)
        x=Symbol('x', shape)
        y=whereNonNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.16255465443162831, -0.075401548451550893, 0.61998992448153811, 0.98751665287362922, 
0.24334557776878274, 0.036731677371431237]])
        ref=whereNonNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonNegative_Symbol_rank3(self):
        shape=(1, 1, 5)
        x=Symbol('x', shape)
        y=whereNonNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.07183553646709262, 0.571643111411134, 0.12210418158332104, -0.47933340822013881, 
-0.77963131659386908]]])
        ref=whereNonNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonNegative_Symbol_rank4(self):
        shape=(4, 6, 1, 2)
        x=Symbol('x', shape)
        y=whereNonNegative(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.066993725840136076, -0.27545326516945057]], [[-0.42934072165626236, 0.42229356051693978]], 
[[-0.33267038317537279, 0.46893642688311576]], [[0.89266708946409734, 0.948473021117479]], [[0.52550545701405671, 
-0.16055292901173135]], [[0.52759295897665881, -0.54697795638017133]]], [[[3.661909080543424e-05, 0.2314045423354878]], 
[[-0.87463870704858038, 0.33691234404370474]], [[-0.24384623670510375, 0.76583873747605868]], [[-0.70439457329825306, 
0.79652728100315717]], [[0.89550406565185514, 0.056023942007641381]], [[-0.54812946125181439, 0.84663276137296695]]], 
[[[-0.40853943317595021, 0.57640997505014013]], [[-0.26371556266792195, -0.45625428604054785]], [[-0.74467882611750369, 
0.61126061359405348]], [[-0.49001100790246377, 0.29246177028397646]], [[0.29827578910576835, -0.60501268542676612]], 
[[-0.8927902525418363, 0.20127128252117599]]], [[[0.0906171701040972, -0.41160009957079779]], [[-0.90862039262623417, 
-0.55145680964277322]], [[-0.98000949136755855, 0.6511419686649127]], [[0.5222457343094018, 0.89406970239489447]], 
[[-0.39112557307084583, -0.3173281519283313]], [[-0.84876063366959853, 0.86272055297540673]]]])
        ref=whereNonNegative(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonPositive_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=whereNonPositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.114330980765)
        ref=whereNonPositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonPositive_Symbol_rank1(self):
        shape=(6,)
        x=Symbol('x', shape)
        y=whereNonPositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.90018358136070309, 0.30348641340688465, -0.85618120763971728, -0.53960506171544309, 
-0.17715605479196417, -0.65288288011449302])
        ref=whereNonPositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonPositive_Symbol_rank2(self):
        shape=(6, 4)
        x=Symbol('x', shape)
        y=whereNonPositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.52054333812889153, 0.93244502782155192, 0.23950589542474909, -0.27867700605384416], 
[0.136236165289082, -0.72973716446443526, 0.78345715642418656, -0.55231976912870806], [0.16068847831312683, 
0.69803624461720348, -0.56066314352940427, -0.048324609040210298], [-0.43917432100497389, 0.20836742086495308, 
-0.47887558228230453, -0.24601140217225681], [-0.36325465701947768, -0.7549630422808371, 0.54364541523192855, 
0.71680613627149858], [-0.018200507730612214, -0.70132178626132835, -0.067011377948262618, 0.93498344549668744]])
        ref=whereNonPositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonPositive_Symbol_rank3(self):
        shape=(2, 5, 3)
        x=Symbol('x', shape)
        y=whereNonPositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.27817206633072811, -0.081436436743882457, 0.75875807911236803], [-0.50343361180037705, 
-0.40952155515557176, 0.30633704565770836], [0.09713396455163692, 0.37971816691593219, -0.64528055605836387], 
[-0.40766608291627149, 0.42126635202766272, -0.36870570089878796], [-0.12999309462561448, -0.013452703166594748, 
0.62628550647076664]], [[0.92143294803926001, 0.87265908348850463, 0.72934818835787651], [0.72670061199951252, 
0.69405462234812743, -0.5375658770167584], [-0.21848892506997553, -0.1416308903572896, -0.18228425021307304], 
[0.59161630465569415, -0.59952258150667359, 0.073642971852158068], [0.20827259131225784, -0.16844724692462698, 
-0.29075535603128211]]])
        ref=whereNonPositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonPositive_Symbol_rank4(self):
        shape=(1, 6, 3, 4)
        x=Symbol('x', shape)
        y=whereNonPositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.76421090018782389, 0.22500582739784702, 0.19670981066640381, -0.029815196145000167], 
[-0.0023715023631671883, -0.25808105968817818, -0.77605684947970355, 0.02245024313203392], [-0.25729342994919935, 
0.54374427642860157, 0.98101283270914585, -0.46768270335386197]], [[0.70839406887585588, -0.37921199056531996, 
-0.027693666176255682, 0.18410753019338655], [0.9844664514717536, -0.68571594713891648, -0.87874566403081622, 
0.99836777995287473], [-0.60137381647439248, -0.96185559798925757, 0.26042164790889832, -0.21938871838079232]], 
[[-0.40465011497491798, -0.60126899021307523, 0.33212611684344351, 0.93945147833669984], [-0.17919578647888335, 
-0.29192839059809184, -0.16587643761468995, 0.5787318075152601], [0.088931359495969087, -0.49762262591243611, 
-0.8824559154533953, 0.69337053676631966]], [[-0.44259804575998496, -0.42643461551484818, 0.76308712257434186, 
0.52791080632128651], [0.43904901084960435, 0.58786748614643591, -0.89969054547864102, -0.81453674264286136], 
[-0.20388022961359886, 0.98704014958763309, 0.28779316308141101, 0.45902180644882029]], [[0.21488201234892346, 
0.15786087222573109, 0.14117129920118709, 0.89719639382806582], [0.44952238681525203, -0.20300718914847482, 
0.37907878593516453, 0.4012439149905449], [-0.053623080473312879, -0.47534453051260162, -0.73601106510348324, 
-0.1727295413441754]], [[0.65587612271205287, 0.061348381925462325, -0.6563598036663767, 0.22986828041081742], 
[-0.62435207387666525, -0.39642223631815066, -0.216712697339845, 0.7064737188000676], [0.63493925912402394, 
-0.97427339204813213, -0.60492536457553392, -0.79397528537512141]]]])
        ref=whereNonPositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonZero_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=whereNonZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.501142275845)
        ref=whereNonZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonZero_Symbol_rank1(self):
        shape=(1,)
        x=Symbol('x', shape)
        y=whereNonZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.00756416102854085])
        ref=whereNonZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonZero_Symbol_rank2(self):
        shape=(4, 6)
        x=Symbol('x', shape)
        y=whereNonZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.47272239608212874, 0.10051789240176356, 0.26272994868293043, -0.6159223154289033, 
-0.56667582585484544, -0.89253107859479286], [-0.55924657895182772, 0.93545079547557064, -0.34898656014938312, 
0.21953807462211961, 0.25492564264834683, -0.35726796840497133], [-0.40094701225180507, 0.1960349067873679, 
0.50731479699854276, -0.78686042102984066, 0.8672231296988846, -0.74804964207566349], [0.95531464602079019, 
-0.89226901941682035, -0.16304450424456007, -0.76454645659342102, -0.78708434678011052, 0.98801488744620647]])
        ref=whereNonZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonZero_Symbol_rank3(self):
        shape=(3, 5, 2)
        x=Symbol('x', shape)
        y=whereNonZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.7414892712283081, 0.41363030109393706], [-0.23348792044545874, 0.20376154349293718], 
[-0.39749859654170105, 0.60619331026813517], [0.19770196299714726, 0.47778531754573339], [-0.2785730022253452, 
0.74385409779061096]], [[0.39811958523569135, 0.98827673008103467], [-0.77742925379355432, -0.36327654260828868], 
[-0.24261210554539181, 0.76381491096289666], [0.71356756598505622, 0.42134472339801898], [-0.32787878885926935, 
0.36977573897534355]], [[0.81553089557162184, 0.65142084303833125], [0.26952185474651191, 0.23717538996324716], 
[-0.4178969152405565, 0.65966116581440826], [0.72266687659103757, 0.23151347657044385], [-0.93144901070650543, 
0.5533107522228673]]])
        ref=whereNonZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereNonZero_Symbol_rank4(self):
        shape=(4, 6, 5, 2)
        x=Symbol('x', shape)
        y=whereNonZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.90238883836103856, -0.078141430869900885], [0.96964248790199536, 0.86899132288638437], 
[-0.68707659301961588, -0.53073339851774448], [0.32688170956882767, -0.96921857548468071], [0.18196874278375952, 
0.57135339374463401]], [[0.25930891270158085, -0.315440727828324], [0.037520854381012825, 0.53261366081014083], 
[-0.15489495244200047, -0.84806312372317638], [0.092612945156565596, -0.25692276023115479], [0.17533235647978307, 
0.89313440080952344]], [[-0.64717308780928406, -0.75805104889803432], [-0.041739580260959475, -0.6205789153566017], 
[-0.25842404105910255, 0.23701518807315436], [0.31089303563347515, -0.51399810672806678], [-0.023306677944956888, 
0.81352506797068447]], [[0.76578954037946101, -0.30328270768759968], [-0.094894717069517798, 0.84307215381835277], 
[0.20655482690635352, -0.68118968257486001], [-0.031341709663070594, 0.7128253675049403], [0.61449090253099015, 
-0.41198959986501915]], [[0.40509351567157026, -0.69086923203639139], [0.43516889791102709, -0.3416724751663851], 
[-0.97964128140528683, -0.22791528480958068], [0.98482909258012752, -0.031469684482034577], [-0.2739148315804234, 
0.093180455447071431]], [[-0.7719005715950038, 0.43813213374298354], [-0.51396005960844726, 0.47137453974714116], 
[-0.46097779903586078, -0.063960915243740812], [0.26352848643891602, -0.48662867823453326], [0.66105464029555128, 
-0.87140048147775584]]], [[[-0.94916583974794411, -0.98499608201877042], [0.05418684797282558, 0.037422050745511637], 
[0.83557151410760477, 0.17226939821781873], [-0.58926990995905659, -0.97834331011266551], [0.30666874303114255, 
-0.2205888292045417]], [[0.33897197902963261, 0.71029540183574058], [0.087210612460627557, 0.8644231149968451], 
[0.38136782120964852, 0.78052625701905143], [0.8317510524713001, -0.014682435119764303], [0.75669358591385327, 
0.22159820191463875]], [[0.47631111031225082, -0.2432820015617243], [-0.056353663947721788, 0.94396356593922226], 
[0.093599462496724462, 0.98409700596138738], [0.51684918605680874, -0.20311323832637695], [0.031403436997070155, 
0.30752607757491823]], [[-0.66062234522460495, 0.98244061952736805], [0.71323974544343227, -0.99294274205841693], 
[0.085382602021304876, 0.34928281809030182], [0.95574038217827284, -0.63922033438378123], [-0.1510988417007515, 
-0.18232263718951058]], [[-0.86857459107290524, 0.27065376653408668], [0.34778882991636229, -0.69468078501349817], 
[-0.05621391859787872, -0.43113805766482893], [-0.21117099401216355, 0.47907566520267908], [0.99631903581124215, 
0.60545973609585002]], [[0.089327664796902129, -0.42328912150739639], [0.028808105423793418, -0.067461200204752769], 
[-0.86739525852770538, 0.60974321136693921], [0.78007851991957211, 0.52984603439955991], [0.82381822897434076, 
-0.096806791764524114]]], [[[-0.64847860728835283, -0.012817456365514346], [0.99972603703845064, -0.1654871349583027], 
[-0.48707077346066852, 0.38641150018763981], [-0.34858104546955748, -0.29831215451161652], [0.0013535663519075314, 
0.97429208816815671]], [[-0.66436036915097763, 0.7797349360947321], [-0.057540181599951445, 0.96084247891800678], 
[0.16745893815999846, -0.79480870410089799], [0.75887588500278524, 0.50015393183957935], [0.2201538536602945, 
-0.91465858939733891]], [[0.46221695878101121, -0.85543254542049985], [-0.79511261729439076, -0.78783008964239776], 
[-0.82020806490717169, -0.41979080265752078], [0.16681258015526867, -0.24042603202357271], [0.13735109423318947, 
0.74666019064984002]], [[-0.089558492701826919, 0.085142350465663208], [0.58353640055793665, -0.56820577095080083], 
[-0.93851521767397306, 0.12873063390745187], [-0.55662024343650884, 0.47890143961389775], [-0.77804687365059011, 
-0.28618046680866893]], [[-0.11520321604725736, -0.47181694802762109], [0.53172168116819973, 0.049579400427356424], 
[0.49159269264521122, 0.13546216707799741], [-0.11678849129062785, -0.76408076913005374], [-0.4191053917656975, 
0.48543625969583348]], [[-0.99253540206564139, 0.62751540233591219], [0.92479658199101622, -0.50793356938360823], 
[-0.66602828599508412, 0.64847360599262305], [0.1256913139099114, -0.070639983370629045], [0.52157580012214688, 
0.097535376596756374]]], [[[-0.57264487061083935, -0.52288880591820197], [-0.025742670244710775, -0.9170247460533445], 
[-0.48679517977387832, 0.64730955653708522], [-0.71406706864020308, 0.11107789589413164], [-0.34729260237235438, 
0.27372244225734632]], [[0.34789886396352321, 0.3054893282236173], [-0.43858235630977571, -0.93001449046583118], 
[-0.22582623588475315, -0.025163803596387035], [0.17634778066393886, -0.98755299203544178], [0.73172600806325594, 
0.59069588394537109]], [[-0.029584834339910104, -0.77713653122341264], [-0.7295543139272771, -0.99458301783387015], 
[-0.2150254830760292, -0.024387346615555527], [0.41364448121677078, 0.99185633963935982], [-0.17526406176681331, 
0.69496091879238597]], [[0.17731947095940614, 0.94603342650519973], [-0.77387354449747758, 0.29832428430509217], 
[0.94536466081511739, -0.32578078437216829], [0.32124342502797631, -0.38798755655698414], [0.90133877643029958, 
0.044111442161604897]], [[-0.14130265836940303, 0.81573218035119055], [-0.52307465861006031, 0.56184286344010603], 
[0.33004086040356784, -0.039891232212105443], [-0.85072814696072729, -0.53465904480572002], [-0.18414968349884031, 
-0.85314057271375754]], [[-0.27062891991592997, 0.81372863018676567], [0.4799755270102728, -0.4476182527576904], 
[0.69041788912015711, 0.36404717548231247], [0.94513419717329872, -0.71989397993530257], [-0.26780541921419498, 
-0.68483698187810105]]]])
        ref=whereNonZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_wherePositive_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=wherePositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.377437559815)
        ref=wherePositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_wherePositive_Symbol_rank1(self):
        shape=(5,)
        x=Symbol('x', shape)
        y=wherePositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.31183363348694781, -0.93037534848467796, 0.36068356078961328, -0.46454210720927702, 
0.25340754345513461])
        ref=wherePositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_wherePositive_Symbol_rank2(self):
        shape=(4, 3)
        x=Symbol('x', shape)
        y=wherePositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.13591127534951108, -0.12739880165333806, 0.14507359534802755], [0.59535385598268742, 
0.33953584270588988, -0.19906200104176852], [0.18362571608830569, -0.79899653225101086, 0.37942105104987678], 
[-0.47201581043531471, -0.77541914147114688, 0.96465645411216316]])
        ref=wherePositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_wherePositive_Symbol_rank3(self):
        shape=(4, 2, 1)
        x=Symbol('x', shape)
        y=wherePositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.40969347572725723], [0.72580022565361202]], [[-0.26443539125114923], [-0.91530007460262364]], 
[[0.14573116670359365], [-0.28164407682569004]], [[-0.46679193029472099], [-0.57181051152013929]]])
        ref=wherePositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_wherePositive_Symbol_rank4(self):
        shape=(3, 2, 6, 6)
        x=Symbol('x', shape)
        y=wherePositive(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.54939235178738421, 0.45983766515917179, -0.6768799220754369, -0.31120246648743488, 
0.13119139528703361, 0.93576894990156778], [0.75331042184998975, 0.86866146502192887, 0.94776229316272675, 0.96712428160924713, 
0.62154805391266121, 0.68121672514452269], [-0.43417781860910809, 0.65552746664802375, 0.71063461804308181, 
0.98778791451902781, -0.48232886555150478, -0.42649944923789374], [0.31062822963226555, -0.30711614218979921, 
0.38303258582904376, -0.59276849280094313, -0.3648303465358953, 0.70202576069688627], [-0.62819924058846577, 
-0.37419798273224547, -0.27130053284023625, 0.9072224174154484, -0.23462453856372756, 0.89263095914991464], 
[0.95858819953941832, -0.070679013453496919, -0.19253023047449003, -0.53174292805497925, 0.56862431614704834, 
0.25371483994251687]], [[0.95871997420741484, 0.01728114303581707, -0.53156136312545055, -0.98155837218866471, 
-0.10670820122709679, 0.26495340950333346], [-0.22967727261849369, 0.4375650694850719, -0.0042250610877931916, 
-0.5421661597598515, -0.84860859311034709, 0.59444229974550478], [0.97540045404987885, 0.65678666686341391, 
-0.035364867222089647, -0.3522032557572381, 0.61915266241839007, 0.94883330926640097], [0.84278744960115071, 
-0.55705943130933733, 0.32014782490667293, -0.32586020208420519, -0.78332186691390748, -0.76260991085881802], 
[-0.29737017596223536, -0.31086711994128269, -0.97532896318861551, -0.094277060177684113, 0.22788292175498737, 
0.7891252898086718], [0.97106594995803519, -0.93962773023973756, 0.52294109109502873, 0.86086141945254657, 0.34533706171782752, 
-0.052510031380444078]]], [[[0.67088753599260298, 0.60809788477592441, -0.77868218557713131, 0.50903109698323856, 
0.29858367870053315, -0.59306223460141338], [-0.47260467286240315, -0.39697435587335805, -0.89083357067986224, 
-0.10260983752890573, 0.56629157310250111, -0.023725279045225811], [-0.55232860144942308, 0.15188533926301861, 
0.64907089070588975, -0.90588370288039721, 0.36514909616019686, -0.55073669616405829], [-0.92748683215923933, 
-0.32679326500380479, 0.24029000646839771, -0.94828824884128604, 0.51573172499226461, 0.33426328315607057], 
[-0.33160559056410177, 0.1381088469968208, 0.81829873108118534, 0.19608670410260665, 0.43676662620148976, 
-0.49395935494495902], [-0.89744665081428976, -0.72520257870888472, -0.25699988567772225, 0.1452157451602043, 
0.85825985207490185, -0.17571346384249575]], [[0.20556011229018423, -0.4406987701618672, -0.35679132088310639, 
0.45463147548831095, 0.58981698516923786, 0.17201232788965282], [-0.060579641690733155, -0.39603147303547059, 
-0.58120873083145685, 0.14328246231837394, 0.07241743712817561, -0.98959634528852281], [-0.38295089355659506, 
0.061816079373679722, 0.26721144541178243, -0.11927829364986708, -0.051279697343907804, -0.47804261526385483], 
[-0.025809521544123193, -0.84617220153247308, 0.11499038006493412, 0.34486144600936797, 0.99238056859336488, 
0.287112601037254], [0.76038815575866914, 0.21802590045316728, 0.72634242544958671, -0.47094029961256623, -0.24638423765838557, 
-0.82286490046142857], [0.37163982157702424, 0.37654232010832778, 0.34791476598157045, 0.82143544452640405, 
0.59277828841993485, -0.45118263524964508]]], [[[0.29521007595604143, -0.63179603594099976, 0.077261193518442628, 
0.66774304786667282, -0.52226601400274797, -0.89958005256034101], [-0.54816558685324268, -0.64457692936298461, 
0.6236779652746034, -0.49567014905254259, -0.71656765859202665, -0.51340090886531975], [0.81122918603327743, 
0.67477141780547023, 0.98673641636939036, 0.9771761206805365, 0.0069183417410505488, -0.59540714802187855], 
[0.68867105619772562, -0.43177753685648002, 0.20120432038778735, -0.54767093106879772, -0.39249228174625972, 
0.057934672256320985], [-0.92217033048550312, -0.99762539640674675, 0.09083701809526401, -0.39936954891299159, 
-0.66466083743049476, -0.40207381070198411], [0.78877940780013556, -0.17183676675590709, 0.0093884540845652342, 
-0.43212358869427314, 0.76020883863712441, -0.46419855612306904]], [[0.40986152639742146, 0.20858406387860251, 
0.2223442693147093, -0.56678886373069481, 0.98618299637569695, 0.61763886209190932], [-0.22639122498865993, 
0.58118970991825791, -0.72547211951949686, 0.45228512524564946, 0.027889137077418136, 0.69723536325392699], 
[-0.63555290638233486, 0.87674986607485139, 0.80071488208678843, -0.69167780346176122, -0.79748774927404065, 
0.67552090311939672], [-0.23605617449770877, -0.24565927714151137, 0.41763899785877645, 0.49433701430402288, 
-0.35742877299856746, -0.84022180228469989], [0.85524675881098045, -0.73395332594232321, -0.69374858487743141, 
0.81140367658791446, -0.11215059436514774, -0.075936402122574354], [-0.28730947176915089, -0.90792495056576339, 
-0.53400548361351863, -0.74285157211058417, 0.48154969096498523, 0.063966517560076097]]]])
        ref=wherePositive(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereZero_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=whereZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.171420304482)
        ref=whereZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereZero_Symbol_rank1(self):
        shape=(4,)
        x=Symbol('x', shape)
        y=whereZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.52154649009116638, -0.44810975525862151, -0.029724338385659133, -0.71479460242802073])
        ref=whereZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereZero_Symbol_rank2(self):
        shape=(5, 2)
        x=Symbol('x', shape)
        y=whereZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.10622499961348031, -0.19082329294362066], [0.87822205186483915, -0.227570581816261], 
[0.66017937209601008, 0.43072185463506951], [-0.74991448789446014, -0.3100002758198217], [-0.51712579145451465, 
-0.70750035422006019]])
        ref=whereZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereZero_Symbol_rank3(self):
        shape=(4, 5, 5)
        x=Symbol('x', shape)
        y=whereZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.89987449420628129, 0.44176718894605771, 0.84278036910056686, 0.62192038982245967, 
0.37469813386654316], [0.44263670237461361, 0.47974259554063403, 0.21880355302135523, -0.99678239055867879, 
0.64599329362555924], [0.71665124130686442, 0.71606435130574031, -0.75650804974459507, 0.82859451747546475, 
-0.034729697394938919], [0.81957258409569311, 0.13604363619082505, -0.22386709023448881, -0.70124352476305707, 
0.29007292902149584], [0.017253258203504762, -0.97219754130253877, 0.11953387372384583, 0.0012251948724546491, 
-0.8433888133511056]], [[-0.29573415646179679, -0.16112108649231671, -0.65405436533907779, -0.76504430864747164, 
0.065591620731848765], [0.1230529204234696, 0.45750665883044639, -0.24863429421650451, 0.90805100101538661, 
0.43106369312103832], [-0.45255394102027768, -0.7708342898403413, 0.3569365489231302, 0.48515730469223373, 
-0.26660148832145936], [-0.15668590902657975, -0.30190701092645433, -0.1645703502153022, -0.81266608219504799, 
-0.49516234113184021], [-0.17495143651323897, 0.57202504858300274, 0.97362734085889935, 0.47097997558209292, 
-0.25753904085924262]], [[0.079668528057362131, 0.070373802041083566, 0.38037582302327744, 0.66277699527962497, 
-0.69081416648919203], [-0.19772575123040892, -0.084538607629316198, -0.92242025609493661, 0.68098269876058781, 
-0.89180009281883343], [0.077789712032067193, -0.60641025717530339, -0.028683412891698623, 0.22193601451721934, 
0.6513529374238638], [-0.22633428720737436, 0.85835116682624801, -0.19667612000958967, 0.34666016205682371, 
0.48687613932426577], [-0.73350490519888023, -0.76996015076308533, 0.20401724886134454, 0.97783986487668551, 
-0.097781308653667409]], [[-0.65651283187735432, -0.48754905420466255, -0.79475108913985437, -0.73064515993996038, 
0.46590720575206057], [-0.77492382103531865, 0.23868136345558622, 0.54530663160147674, -0.1735577582824015, 
0.25728144167719824], [0.60969501001983506, -0.02889586029527913, -0.1210183483182683, 0.14001630349981831, 
-0.36024256211146599], [0.8963576166338787, 0.56742574160442771, 0.96618862908484893, -0.45856528487907955, 
0.47724790543271212], [0.14235966019163016, 0.53124462653844851, 0.65221925434946604, -0.61131393378219001, 
-0.97528452689210066]]])
        ref=whereZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_whereZero_Symbol_rank4(self):
        shape=(3, 5, 4, 5)
        x=Symbol('x', shape)
        y=whereZero(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.87917924924094204, 0.92576082229381873, 0.31182535354339991, -0.86214749809692592, 
-0.27507709543943437], [-0.0016382939288797083, 0.43307986157908585, 0.52636371100155199, -0.043319908450309308, 
0.20091228183885401], [0.71583896980535289, 0.50199360637469881, -0.67746890499802959, 0.73499870395620159, 
0.63277338423504914], [-0.59257453161192131, 0.66091287836835622, -0.19325788610358763, 0.57712882367912877, 
-0.7563530936539633]], [[-0.48965026554891877, -0.32764373599077956, -0.5131590362892775, 0.85592981071649832, 
0.99561636195898173], [0.24222780295446933, 0.28730806112554474, -0.32041176685276618, 0.78323831462368121, 
-0.72040929168456769], [0.95423747440270379, 0.39687818036239664, -0.77878094662206365, 0.75880557194836884, 
-0.1218416672560394], [-0.53785770919122489, 0.67105013676368763, 0.42864708225385661, -0.78784834304583429, 
-0.2681253344477097]], [[0.46875429958618664, 0.38741628037005071, -0.081056386101901357, -0.23469920628069518, 
-0.40907581999438447], [0.065494837299773234, 0.38170940261789488, -0.79236472117163403, -0.45228389456840379, 
0.55163355897584609], [-0.22355146021048489, -0.93756490876746912, 0.9241257141019481, 0.60698441154913452, 
-0.21265210744597973], [0.21034749440796197, 0.1659692307697147, 0.78720887852180499, 0.71946077138975917, 
-0.85985855270766165]], [[0.34986386801299396, 0.35619084924105593, -0.1952077124592666, -0.22518203420372651, 
-0.59920836063901772], [-0.58104191295806817, 0.29689367250987697, -0.10221810014905208, 0.096526465218390367, 
0.055496030226582782], [0.12128925311183991, -0.36085110995947267, -0.69818945730712589, -0.54447063194246215, 
0.28238768912002654], [0.5685290343654088, 0.90491556875163925, -0.83800696608599567, -0.11641347846076111, 
0.44596200041220047]], [[-0.59532265019171771, -0.27066391467944517, -0.89311609332930963, -0.99850338961193308, 
0.15639571601440827], [0.41583353392337186, 0.030126889068769591, -0.18653046086180169, 0.71825314848840516, 
-0.59319353551728171], [0.35800182346025844, -0.6648117590030429, -0.26482259771960504, 0.54772429609472284, 
-0.58930805496179794], [-0.3705568042330003, 0.13914156728520855, 0.83616254382469846, 0.6609224092875603, 
0.12930292638032426]]], [[[-0.040652326769339941, -0.28000548584426599, -0.64476634965815371, -0.54049639115735393, 
-0.86161774681830083], [-0.61013140491056395, 0.54541436881577199, -0.54601847713291751, -0.31621881114237693, 
-0.61724019337500291], [-0.44644750286742663, 0.59518192305072182, -0.7693498325029422, -0.42060186241232422, 
-0.55489928209223538], [-0.010271097007732655, -0.22885352817942128, -0.23461127002541859, 0.2259662465741632, 
-0.75911861057241303]], [[-0.58916400291939897, 0.38944538474440882, -0.63829862105093849, 0.72019195488661647, 
0.52235809650215592], [0.13114169583575852, -0.58533255793898586, -0.68420879335952733, 0.92417522487531101, 
0.7272342344782925], [-0.56777601453995841, -0.99375451020494832, 0.23729078199002207, -0.37389917191977862, 
-0.6758356849508167], [0.45071814163626911, 0.80301641829374315, -0.34653672569319238, -0.33515189137311152, 
0.63975761694188726]], [[-0.78979414604272868, -0.43983713317632156, -0.74203230273291054, -0.88020557636880481, 
0.47328240527524712], [-0.82389677158683061, 0.7178390238645278, -0.11098357077701571, 0.73262520297238942, 
0.09735578528861577], [-0.88385341167453579, 0.61275720586751614, 0.76125858815934855, 0.14815442457054129, 
-0.54087876314080385], [0.85068577624506925, -0.46444971606055296, -0.62601913837363177, 0.2972222055112097, 
0.66067056423975701]], [[-0.72007073982328085, 0.90956585931155076, 0.88200178773846005, -0.69430533795630245, 
-0.49857629281722504], [0.58092394474421427, 0.23240811382528248, -0.16824649980781214, 0.55878471927179807, 
0.79568431021078023], [0.86166989839988894, 0.91296231064238254, 0.31164821231411866, -0.56046677494371044, 
0.0385512568580888], [0.93714843860938113, 0.12260670095667692, 0.41615058741030264, -0.53179227427253117, 
-0.26098397322251543]], [[0.0028558200394461775, -0.50060512460961681, -0.93283479021249138, 0.82618003523249883, 
0.21003657367784623], [-0.44588288887982475, -0.18118625551999723, -0.99177445676917753, -0.96873602172290019, 
-0.98701490005244841], [-0.93938608592020412, -0.016437629841765444, -0.36278147050642118, 0.52186707271265531, 
0.89250194233082958], [0.84314628295750893, -0.098344859482383917, 0.66383189141986532, -0.47591027885241921, 
0.59605402918250672]]], [[[-0.95794362834059377, -0.93767110890167471, -0.59649558326393914, 0.92707879675054206, 
-0.26646654731611896], [0.71313621784956482, 0.48496748630067921, -0.056901419366484518, 0.61280929565097964, 
0.74294037098969978], [-0.48859644796388979, -0.44214089667786016, 0.62644178280749219, -0.060536138800636019, 
-0.57599606335652731], [-0.83904456131168415, -0.74905532220546855, 0.7486912362306215, -0.22402222230020863, 
-0.067386476775930682]], [[0.3184385936276124, -0.48262522108105022, -0.15510478911798553, -0.88527377048903855, 
0.14896925645860337], [0.93368201508427573, 0.70361207151476601, 0.54827381187668389, -0.30181573690390695, 
0.93407788633099109], [-0.41953515397211016, -0.99699015474725328, 0.077039550676741619, 0.084209448590547131, 
0.76015066749097127], [0.54201663917636989, 0.12742530960610221, -0.12814330189633583, 0.51769077187580215, 
0.82486105615641492]], [[0.64354010561109676, -0.42473695374832476, 0.3063589610119255, 0.35974954581874252, 
-0.51335607714264087], [-0.83565577834576166, -0.4308190974476882, -0.17313285404785406, -0.70522642627212995, 
-0.64687941557652229], [-0.18599656731065095, -0.044712449778867569, 0.49716990327674271, 0.30430680138579724, 
0.08057899424959758], [0.58586637100455774, -0.13301988125684039, -0.044158021238169365, 0.053245830039011732, 
-0.53906389981826885]], [[0.21962406823739866, -0.71314553563905903, 0.54111730219890264, 0.19650615982926523, 
-0.059214339554341411], [-0.908481783495934, 0.33772345316258168, -0.86313443774430842, -0.57635827404614903, 
0.52176471220549625], [-0.96424795883461867, -0.31443781628771283, -0.86245198160612913, -0.6565037382017811, 
-0.35318228761977455], [0.14426706591786842, 0.58298102985131806, -0.49066513703465353, -0.9150068286814601, 
0.40374237858154149]], [[-0.69674134110121377, -0.71482950999283856, -0.59670958247227612, -0.91637682393291175, 
-0.25938985991666352], [0.45459801956464618, 0.74426900735359625, -0.22984550468363252, -0.43189830219709369, 
0.8058729071921289], [0.25819553521791616, -0.095869899831380279, 0.85491697659907051, -0.57530704702476676, 
-0.50441247934023292], [0.69625438900313252, 0.28897970313322419, 0.71872510620750329, -0.055458863026323524, 
-0.082229736356652872]]]])
        ref=whereZero(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_antisymmetric_Symbol_rank2(self):
        shape=(4, 4)
        x=Symbol('x', shape)
        y=antisymmetric(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.66708916764681492, -0.74238465201633974, -0.079353161755557622, 0.30257601442541904], 
[0.20925829383746208, -0.022009924374370327, 0.10502873236092491, -0.15884250966193902], [0.1797060494083087, 
0.023139755304512288, 0.45170178441767739, 0.48507784807517917], [0.059968719280828253, -0.41251906740163857, 
0.2910687977002866, -0.63445501785955627]])
        ref=antisymmetric(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_antisymmetric_Symbol_rank4(self):
        shape=(2, 3, 2, 3)
        x=Symbol('x', shape)
        y=antisymmetric(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.55376735565955926, -0.26986726052539045, 0.99540130065296051], [-0.66278909769325844, 
0.47418600478729656, -0.57224349831177945]], [[0.26869674765376628, 0.87439268666551895, -0.58960755596620462], 
[-0.40755988519135888, -0.12284324214732534, -0.31617642601932183]], [[-0.025792397075229179, 0.33857483183258519, 
-0.39641658275864589], [-0.13242160052866581, -0.025186857396592099, 0.95590380942335851]]], [[[0.27725575342422726, 
-0.50385473604694497, 0.11901129980645275], [0.75590460511710589, 0.096085673196859789, -0.72791893038598321]], 
[[0.38794533456732516, 0.68280080521959974, 0.95500219859151003], [-0.58249140643372388, -0.38374270163626578, 
0.49303266750299435]], [[-0.34681269192922426, -0.24778438052869323, 0.72024308366404277], [0.768229749916157, 
-0.79962778314547212, 0.70336464030375567]]]])
        ref=antisymmetric(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_symmetric_Symbol_rank2(self):
        shape=(2, 2)
        x=Symbol('x', shape)
        y=symmetric(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.61119718425127556, 0.9059918430328846], [0.22759887880296792, 0.98183091980677251]])
        ref=symmetric(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_symmetric_Symbol_rank4(self):
        shape=(2, 6, 2, 6)
        x=Symbol('x', shape)
        y=symmetric(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.47931228708865947, 0.10507685665916622, 0.1706306038031824, -0.48682627958350611, 
-0.052761050334725113, -0.61764395051469689], [0.33131346993926525, -0.17831168870543523, 0.83269324896567931, 
0.49711792481658645, -0.15160508118267435, 0.034823661795621907]], [[0.83488102623337146, 0.48444595120803413, 
-0.2438647943575607, 0.65263344181761029, -0.53266665587602979, 0.042392390454812912], [0.32082347612712003, 
0.24207481544942144, 0.90481421627507386, 0.58968892722797084, -0.37243554407799118, -0.95360987465922764]], 
[[0.54236303554410048, -0.77750655420008474, 0.86347524214181703, -0.60045464519880043, 0.6614103843852106, 
0.1661581628512927], [-0.27633818658200648, -0.0092044623033202821, -0.77064836472526466, 0.27939966240494951, 
0.71809970608669804, -0.0030250095390851328]], [[0.67629609745977559, 0.35017051906040519, 0.045207039627761736, 
-0.83837847095041584, 0.41784669604739855, 0.56574060434330353], [-0.085647568466385771, -0.70722504353285265, 
-0.2413437677613961, 0.65173300274030099, 0.49740733632840528, 0.78483628566812902]], [[-0.15655167722843855, 
-0.2643217130737634, -0.1907507722854267, 0.88643511791047569, 0.68916193655212843, -0.37723536389790091], 
[-0.55626566044838754, -0.98938031969368567, 0.72037970472566815, 0.08304233980247222, 0.57646971145679315, 
-0.53992799288622106]], [[-0.11144991273379024, 0.99118197843158296, 0.0042934206493501215, -0.61263222801310269, 
0.4938389815759765, -0.053018311339752655], [0.87098646055584861, 0.2188430651448523, -0.26804136146176893, 
-0.92535908367104325, 0.52121221487688008, -0.61711018990565591]]], [[[0.71270846704492241, -0.074743332130423612, 
0.12785251114370033, -0.79341710107657581, -0.017885731858772402, 0.68651610814691755], [0.063052501554288876, 
-0.35341399674809737, -0.22893402939136953, -0.28425123214766757, 0.66054323380507807, 0.061862036873020054]], 
[[0.67397793373178727, -0.34430769423381391, 0.32484940571520093, 0.073882587656904031, -0.69318017790826914, 
0.80042677264669448], [0.29304891549348078, -0.39990075961382399, -0.50334862380676815, -0.68412227199254638, 
-0.17985296091249903, 0.88016340392243464]], [[0.91132543158899537, 0.24474430946290204, 0.15332311503635343, 
0.15948665282678909, 0.18086292106026147, 0.7737255853795324], [-0.16063567286120239, -0.89954764701723167, 
-0.011648068495271113, 0.76021363407728981, 0.3648652547595066, -0.7163207516373884]], [[-0.5260529219985961, 
-0.5341824038899905, -0.24546004667603727, -0.68556445585168557, 0.00058699621817481784, 0.90805266220693048], 
[0.10346080847946149, 0.32661168806987351, -0.98860366824673807, 0.23210015251566563, -0.67778840377649785, 
-0.75819552887923702]], [[0.034695288275003344, -0.063434136501459459, 0.17626561461887391, -0.012597705074078869, 
-0.23908751605985867, 0.36337197419679801], [-0.40742067576725938, -0.28910284114307694, -0.034942103406036651, 
0.91056413009275716, -0.82944439530144787, 0.96920580528501543]], [[0.93536972885023206, 0.77182940845310055, 
0.027417593751139835, -0.26678735403398601, 0.49140092371614807, 0.46218190299367357], [0.79603807023119111, 
0.65706269041412702, -0.38729966802179794, 0.28924104766767744, 0.63848678811732884, -0.99073927196286404]]]])
        ref=symmetric(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maxval_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=maxval(x)
        
        xx=numpy.array(-0.134448776959)
        ref=maxval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maxval_Symbol_rank1(self):
        shape=(2,)
        x=Symbol('x', shape)
        y=maxval(x)
        
        xx=numpy.array([-0.29368513761235637, -0.9580644046832012])
        ref=maxval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maxval_Symbol_rank2(self):
        shape=(2, 5)
        x=Symbol('x', shape)
        y=maxval(x)
        
        xx=numpy.array([[0.34735121468535302, 0.31792373377370797, -0.34362504765406632, 0.74644673081162916, 
-0.057418925515936081], [0.6732366131700307, -0.95238446096007912, -0.21844674633886907, -0.18926547437334618, 
0.52984603152753795]])
        ref=maxval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maxval_Symbol_rank3(self):
        shape=(4, 3, 3)
        x=Symbol('x', shape)
        y=maxval(x)
        
        xx=numpy.array([[[0.42364784215934548, 0.66444692360687774, -0.29567189694092244], [0.99530022127084483, 
0.30806444685979906, 0.73704625719821393], [0.95488935047107915, -0.16103673114816508, 0.32508484387903103]], 
[[0.42646650140936937, 0.25758089523264083, -0.10299917300444994], [-0.32512913895080708, 0.60765404003443413, 
0.75746102375083146], [0.47427065451552686, -0.36709593744995073, -0.62234298674588739]], [[-0.51625772797208214, 
-0.17477544865889505, -0.2467213054045394], [0.21391484080459633, -0.11023541870594666, 0.52879222463793352], 
[-0.6115252143321559, -0.69548674713370429, -0.84959851655955743]], [[-0.047735520296660727, 0.63725644772456125, 
-0.81320293399414489], [0.033442410991393778, -0.52485955963669739, -0.87579195407828703], [0.099747411226758453, 
-0.48623897347358258, 0.18561580925432075]]])
        ref=maxval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maxval_Symbol_rank4(self):
        shape=(4, 5, 2, 3)
        x=Symbol('x', shape)
        y=maxval(x)
        
        xx=numpy.array([[[[-0.97772645022484239, 0.57304243484894912, -0.23114792233740067], [0.5009967216273834, 
0.98728285370921998, 0.88165326175944081]], [[-0.86554004095808312, 0.79967438864226104, -0.52356143827663781], 
[0.33910376076389492, -0.33715271820715276, 0.7471336395988164]], [[0.97807293621863911, -0.32426795551673915, 
0.97086761927014531], [0.69521998223272985, -0.50921868802924752, -0.4707875985687946]], [[0.32088835906711743, 
-0.67641005811719257, 0.6053555811404927], [-0.9755684596801586, -0.54801753900404271, -0.51569531087459231]], 
[[0.87859901272230245, -0.93819203169497767, 0.48552591270387691], [-0.90623293966225638, -0.56821898651640379, 
-0.70114717419616346]]], [[[0.7074153550653195, -0.065789330921568245, 0.1975906581408009], [-0.70031376520506416, 
-0.58588697093854769, 0.28784685846180036]], [[-0.37927702124416784, -0.25619886766039057, -0.75820003470201258], 
[0.75522605658504749, -0.86607222560760899, 0.77087852691254444]], [[0.78983241202496002, -0.93807825839461723, 
-0.64157008373209368], [-0.67265163687234764, -0.38384738076458658, 0.83095367437218925]], [[0.44769461663793142, 
0.4681548319926121, 0.89179461494471446], [0.79244020458833186, 0.69571626917307383, 0.91369455881021167]], 
[[-0.20838088102066532, 0.74001065048887726, -0.21044275890782416], [0.89884395702714248, 0.42077808513336179, 
0.89751742525902301]]], [[[0.13406499519690329, 0.93048539905404448, -0.88878801153635401], [-0.32682928056521643, 
-0.02669348688107398, 0.97150663926803249]], [[-0.97754157146727216, -0.52649319587800592, 0.51006871063667658], 
[-0.11775631034214085, -0.92540518843525632, -0.60798901233999914]], [[-0.26252763601616658, -0.10106038459184274, 
-0.35449159770172978], [-0.34352827137217257, 0.2131126361783362, -0.52225770770455338]], [[-0.99904658805863944, 
-0.03571103471541015, 0.34522288509344889], [-0.16466365212076028, -0.32272992229399655, -0.1104994415867564]], 
[[-0.58008507722437108, -0.41010112955007627, -0.77315836526542392], [0.20309194480149784, -0.46019343937075385, 
-0.25587082608895906]]], [[[0.87800190688704949, 0.14755820295840882, -0.66380548135485773], [-0.38373986766386214, 
0.35559704316537077, 0.42892933288907176]], [[-0.14611087776987608, 0.84852402939259775, -0.47292417128454423], 
[0.7686091241664712, 0.070435421419021704, -0.17088786425027247]], [[-0.85107214556959176, -0.069124539395263307, 
-0.84046317240118973], [0.59678265942099928, 0.88283476525309879, -0.24815533144527979]], [[-0.28723087974789396, 
-0.83621864123734024, 0.020689401256574813], [0.9232387272498741, -0.68810732653308104, -0.81093541699621685]], 
[[0.73105323790407883, 0.51454885036546982, -0.72394828813224632], [-0.25287960060139225, 0.64569479592808587, 
0.62813882863773229]]]])
        ref=maxval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minval_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=minval(x)
        
        xx=numpy.array(-0.30618310049)
        ref=minval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minval_Symbol_rank1(self):
        shape=(5,)
        x=Symbol('x', shape)
        y=minval(x)
        
        xx=numpy.array([-0.85135969885833096, 0.7532816318813329, -0.5882678068887619, 0.89119141508691402, 
-0.9379977266076569])
        ref=minval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minval_Symbol_rank2(self):
        shape=(6, 2)
        x=Symbol('x', shape)
        y=minval(x)
        
        xx=numpy.array([[0.04757935417938719, 0.56448960316435781], [-0.7967017037359494, -0.24401431507973848], 
[-0.59131630413733083, -0.52722327369086197], [-0.18374212137163393, -0.60315054526379708], [0.22196503091420383, 
0.12665575107549576], [0.71319493777219689, 0.2752127433396967]])
        ref=minval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minval_Symbol_rank3(self):
        shape=(6, 1, 1)
        x=Symbol('x', shape)
        y=minval(x)
        
        xx=numpy.array([[[-0.37545612635023939]], [[0.2230129304822781]], [[-0.45233309721956294]], [[-0.26620859057037216]], 
[[-0.30837618713582637]], [[0.78814029871507829]]])
        ref=minval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minval_Symbol_rank4(self):
        shape=(6, 1, 1, 2)
        x=Symbol('x', shape)
        y=minval(x)
        
        xx=numpy.array([[[[0.90578355615023698, -0.51094614506705494]]], [[[-0.63501200952398085, -0.46547690121911156]]], 
[[[0.67638924191025884, 0.55257795954409628]]], [[[0.68011812217900491, -0.53935030654427019]]], [[[0.51457282740049237, 
-0.65201714679111422]]], [[[-0.54031827299109603, -0.059011224051143341]]]])
        ref=minval(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inverse_Symbol_dim1(self):
        shape=(1, 1)
        x=Symbol('x', shape)
        y=inverse(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.707926990095334]])
        ref=inverse(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inverse_Symbol_dim2(self):
        shape=(2, 2)
        x=Symbol('x', shape)
        y=inverse(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.8054979921953962, -0.50425859166012033], [0.97357929148547639, 0.92004508523090989]])
        ref=inverse(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inverse_Symbol_dim3(self):
        shape=(3, 3)
        x=Symbol('x', shape)
        y=inverse(x)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.45028554514700891, -0.071572787871187593, -0.39441942665859409], [0.73229203641636409, 
-0.40855316211290282, -0.12348705744184918], [-0.47530902424927413, 0.45609397667651108, -0.3860382301700751]])
        ref=inverse(xx)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank0_offset0(self):
        shape=()
        x=Symbol('x', shape)
        y=transpose(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.47016751229)
        ref=transpose(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank1_offset0(self):
        shape=(3,)
        x=Symbol('x', shape)
        y=transpose(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.57978369223815918, 0.27681092533940088, -0.071291653268149568])
        ref=transpose(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank1_offset1(self):
        shape=(5,)
        x=Symbol('x', shape)
        y=transpose(x,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.60705531324714634, 0.62810480691473658, 0.77189612790979578, -0.71300060089285577, 
-0.35861924250677912])
        ref=transpose(xx,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank2_offset0(self):
        shape=(6, 6)
        x=Symbol('x', shape)
        y=transpose(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.29824722631373124, 0.63375418610212408, 0.21922442998100244, 0.12715532157523968, 
-0.29643893323918102, -0.16166333104335817], [-0.89328765219407691, 0.27683764917068565, 0.34017377147969219, 
-0.53274644593651321, -0.1422658361691147, 0.28161907229446315], [-0.38203701914488764, -0.2961521871919941, 
0.80071418498487024, 0.83878791977835077, -0.69075507497575783, -0.16169785100459899], [-0.032504150913533314, 
-0.64123342525770588, -0.084919983774251584, 0.92118199386482202, 0.26012548701114357, 0.72035325070640543], 
[-0.24685236184450643, 0.36815987147920048, -0.24045842732995948, -0.93481032202233783, 0.53659798717183871, 
-0.25927629660414153], [0.52973705424869744, 0.32004608474978991, 0.90380715550842816, -0.65392860402124442, 
0.63799552939147652, -0.27587752537198384]])
        ref=transpose(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank2_offset1(self):
        shape=(1, 6)
        x=Symbol('x', shape)
        y=transpose(x,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.73357955978002387, 0.32007537987130541, 0.80392128184540268, 0.075907889124529948, 
-0.039990110087071651, 0.12075338135391833]])
        ref=transpose(xx,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank2_offset2(self):
        shape=(3, 5)
        x=Symbol('x', shape)
        y=transpose(x,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.25551832867199376, 0.42008002231810315, 0.078215128725729111, 0.70039085616518548, 
-0.88204165217907637], [-0.34979785984659939, 0.08315537698343678, 0.43326454393298341, -0.062646735390029296, 
0.41602450028477334], [-0.41495962824919475, 0.64372525733542396, 0.31391836076533819, 0.4939984978067602, 
-0.053702510243994528]])
        ref=transpose(xx,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank3_offset0(self):
        shape=(3, 4, 6)
        x=Symbol('x', shape)
        y=transpose(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.26311042905201254, 0.35909401464904889, 0.90408965239472594, 0.064216765615245475, 
0.89018018616522032, 0.61515401166904038], [-0.30989721207991727, 0.01554721397885328, 0.52855204676559642, 
0.93771990426057239, -0.15268008182723269, -0.72186799095134035], [0.29533985198855883, 0.84120750629942131, 
-0.22819454447353227, 0.53533756476516925, -0.17828320712265366, -0.78412542334949542], [-0.60495885555969942, 
-0.25855652778532168, -0.15744952045553418, 0.5567093392784852, 0.23570396793559212, 0.19344156929131118]], 
[[-0.42782972938810349, 0.93496323651712765, -0.20729448187853028, 0.15464520183866215, 0.0043514616865014677, 
-0.51750038510802732], [-0.22811855804368841, -0.48794731937174163, 0.92444939190764486, -0.82797921663444662, 
-0.78083506656193813, -0.63163718067425112], [0.56047492548799616, -0.2744337823672458, -0.6547449031228918, 
0.56648635685738769, -0.11694804353070998, -0.038687755044935379], [-0.40048029851421107, 0.58145660627249818, 
-0.71158323850172178, -0.92609046734296485, 0.47425311825021166, 0.24907305314740569]], [[-0.26289340847029763, 
-0.6627578090201578, 0.62394979447651489, -0.47215367721990265, -0.64661615031822905, -0.57803380349174183], 
[-0.99219145036848722, 0.025564187125975968, -0.84489002798303048, 0.1859355185991034, -0.10171106325521784, 
-0.96678921742804902], [-0.047487603598241224, -0.32327046729535125, 0.59358325902723896, 0.2430229651630238, 
0.9924605637701025, -0.9095931432779174], [-0.14746705870069343, 0.8316033803487568, -0.51338609129446722, 0.26829256656137224, 
0.15800350230147164, 0.74083217352043351]]])
        ref=transpose(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank3_offset1(self):
        shape=(1, 5, 2)
        x=Symbol('x', shape)
        y=transpose(x,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.068231745020743695, 0.28497669381507706], [0.16579540496302525, -0.0073004211699192467], 
[0.048763180867146261, -0.90650544874903161], [-0.77560606065477233, -0.22408349827566187], [0.17941319486573315, 
0.95839772281795654]]])
        ref=transpose(xx,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank3_offset2(self):
        shape=(2, 1, 3)
        x=Symbol('x', shape)
        y=transpose(x,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.44441954295782415, 0.57951688465897666, 0.97759547928033852]], [[0.9694383324351068, 
0.95911385829077767, 0.34153788003972307]]])
        ref=transpose(xx,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank3_offset3(self):
        shape=(4, 6, 1)
        x=Symbol('x', shape)
        y=transpose(x,3)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.56750193647189295], [0.74203408066371557], [0.88359255309208407], [-0.1420561936189324], 
[-0.1832315325470717], [0.67350071639095455]], [[-0.12986823051754137], [-0.62214225719818117], [0.76051722748179884], 
[0.19372894258576401], [0.51431220823279777], [-0.55327737164253321]], [[-0.12774343291108359], [0.91719475963576147], 
[0.91573642478780415], [-0.61521915077800782], [-0.40787721375622787], [-0.0061586053369555582]], [[0.057062787602994103], 
[0.49344734797982026], [-0.82845047193545973], [-0.68097806860425036], [-0.53065790567095972], [-0.21476705851128997]]])
        ref=transpose(xx,3)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank4_offset0(self):
        shape=(1, 3, 6, 6)
        x=Symbol('x', shape)
        y=transpose(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.54824608108340733, -0.95881993351872175, 0.096047244291673994, 0.052360005545741606, 
0.12490898613666901, 0.37925568740922189], [-0.55124438933079856, -0.52935244195563946, 0.27908541506517448, 
-0.1830678527428955, -0.61244328503367873, -0.010607705370612397], [-0.80506898031795937, -0.21539423990251261, 
0.73483184561396819, 0.3218480929559373, 0.63228046571224295, -0.11544050682355489], [0.50999066506659219, 0.61608238345318789, 
0.85718098893486228, -0.82554569171674919, 0.67239454553509659, -0.77128236726181387], [0.77415761641002478, 
-0.21788051784728957, -0.79355077597956836, -0.1361323772856724, -0.86290682182908185, 0.02622965736996985], 
[0.74395099341159487, 0.13786029833658375, 0.64251511260194349, -0.30926165348941947, 0.066002055058048015, 
-0.18373875123993755]], [[0.80181502648535408, -0.31866470950275949, -0.33856607958158369, -0.60420044839061493, 
-0.48291079444954743, 0.24475260440995461], [0.22283617220107033, -0.6615458160126455, -0.66676148321169748, 
0.97999033040506101, 0.06108795758199248, 0.040226927587740491], [-0.67186168564203053, -0.14276728079247003, 
0.19541571657523216, -0.75136887419362108, -0.59956121332937018, 0.059908246353421513], [-0.20724545062697186, 
-0.17856278158186711, 0.047379025453474055, 0.15840963916227646, 0.13540917508687045, 0.85895067204573583], 
[-0.83872496960394871, 0.23265853232152711, 0.19453704633971625, -0.96538534645560392, -0.89514948083720092, 
0.52011941028029685], [-0.97139300748704027, 0.7819520140439693, -0.015898695567955201, 0.29189177705966207, 
0.7814405901040784, -0.027981993399570637]], [[0.024367179589735111, -0.92559202239623994, 0.0380332273061188, 
0.36881267312762245, 0.20795105074551024, 0.69371808897843201], [0.49435054361558683, 0.58132683830046772, 
-0.76010694324043038, 0.28394316877079939, -0.82021099055909619, 0.49870600029852086], [-0.51039201474247053, 
0.56336641890331585, -0.52323759045305485, 0.52986550038889946, 0.5034221907692511, 0.18849560067453153], [0.60090832298881591, 
0.12268166503962252, 0.65511543959119356, -0.97314981949878665, 0.93457616725728321, -0.36648294820963012], 
[-0.51170849654484218, -0.50879187831813777, 0.10542578094698452, 0.45077764740856896, -0.31197154971763363, 
-0.45296904314713959], [0.14168707080581178, 0.062913177405235166, 0.65635509363307154, -0.11073208661229672, 
-0.16755936854126485, 0.9561185477484222]]]])
        ref=transpose(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank4_offset1(self):
        shape=(6, 2, 2, 3)
        x=Symbol('x', shape)
        y=transpose(x,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.61981582085444975, -0.96131273801498329, 0.74219386268038434], [0.55966010365844987, 
0.71389079962454427, 0.8989592580659409]], [[-0.32362828871796334, -0.19464923533132406, -0.077071376688295556], 
[-0.31631546071958838, -0.63804519499196855, 0.034469021777733255]]], [[[-0.98013465627214558, 0.55692938312034257, 
-0.069876839824936088], [0.23107446680923793, 0.20970489689947236, 0.63734275566499932]], [[-0.38904775463304775, 
0.071793925322782615, -0.46938272402469572], [0.81376007574910103, 0.72212779041170472, -0.7900188086653761]]], 
[[[-0.21364296019115114, 0.44694321429953288, 0.013273984088666202], [-0.30688415384900836, -0.79173559959507545, 
0.21393561197226707]], [[0.11643782235065991, -0.91447688288862561, 0.8191757449265018], [0.18102464836223464, 
0.23160336066416609, 0.59871147260783464]]], [[[0.99733195033134581, -0.3095466388403092, 0.3954110325522926], 
[0.46142982683595646, -0.47003892539287939, 0.94950085174619647]], [[0.16984700471120551, 0.15805429609841348, 
0.24500674000556089], [-0.55165958732539777, -0.92936800791598762, -0.022951588393276046]]], [[[-0.47677497240186817, 
-0.29108181048886861, -0.46273092627976475], [0.1273672188319761, -0.45995292810695965, 0.63194581534140681]], 
[[-0.55300845786078279, 0.7533517791070008, -0.13448526871956634], [-0.37358334038389174, 0.20829676253681684, 
-0.45687485563287189]]], [[[-0.036755848169893302, 0.30727556288418967, -0.21041329338872861], [0.20182923263729791, 
0.65877866222054271, -0.24646229766737426]], [[-0.98252816636757379, -0.36713378601223945, -0.80808457042378978], 
[-0.26267367173420597, -0.2518970867644057, 0.011477505796322118]]]])
        ref=transpose(xx,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank4_offset2(self):
        shape=(3, 2, 6, 3)
        x=Symbol('x', shape)
        y=transpose(x,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.16214769600244305, 0.24217872120946082, -0.30028030966869657], [-0.48045998353105079, 
0.95693823823765212, 0.52916178178914697], [-0.48954360115331808, 0.85861012301924489, -0.81692988730865213], 
[-0.087644936882469437, -0.3842832680291377, 0.35011196871912009], [0.57129365131241583, -0.026003764280923436, 
-0.75091671858302056], [-0.74212966728321961, 0.12594020322095445, -0.49899956224796882]], [[-0.63238633088294094, 
-0.34904744665375165, -0.77820922334570719], [0.9223856580495613, -0.092757622542367635, 0.072822560962293004], 
[-0.070993587117452295, -0.14883635172390397, -0.76366672463695995], [-0.58165146796847944, 0.25485622479175296, 
-0.4385367103246629], [-0.6718851054540862, -0.54227121325719052, 0.22659211333469198], [-0.77256203431387149, 
0.32093113572510412, -0.024748143906698106]]], [[[-0.81065249686006191, 0.85744411886085925, -0.6926225063339142], 
[0.1359015377696986, 0.16676992366466048, -0.25508688732693452], [0.51534906080188891, -0.51302456371614524, 
-0.13403185580745269], [-0.83972439581808334, 0.43981616780551591, 0.32672995034070751], [0.65639618069998873, 
-0.89753193640920048, 0.85900236622478832], [0.38497532723462147, 0.83291046269513536, -0.80651708700513747]], 
[[-0.51373944818354556, -0.72195574816716812, 0.90186186159805426], [-0.56332521605329755, 0.49732930230749539, 
0.1484166126253168], [-0.72917908882955729, -0.95729599166703117, 0.72755275645534057], [0.4921919188713737, 
-0.96432237192190851, -0.036083259295467185], [0.1723714120524098, 0.040941543603711272, 0.62521746283269097], 
[-0.1690314543596303, 0.47940137593575449, 0.42640711858738989]]], [[[0.10390397381997918, -0.24977943617300435, 
-0.98869850717150864], [-0.96639652829702083, 0.35034662237955816, -0.56248828327408051], [0.96408180547845057, 
-0.23999573384656303, -0.81951160331181905], [-0.47170906450449324, 0.6419551236751726, -0.48542637867915861], 
[0.70176197798886486, 0.96442202464370985, 0.83401799166885082], [0.79732376621020729, 0.70856863055292596, 
-0.28971110032777569]], [[-0.44536694566283019, 0.53490318645652013, -0.040524017414241387], [0.8017156672864223, 
-0.035466342245163318, 0.14641512106385846], [0.38200951751843482, 0.55381128552992576, 0.62196907345612029], 
[-0.59415983609316458, -0.057552239669435235, -0.87044248211447695], [-0.20880226272887281, 0.89867393628607095, 
-0.27599039986863305], [0.38668431971946848, 0.77159353796456487, -0.049495448837690637]]]])
        ref=transpose(xx,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank4_offset3(self):
        shape=(4, 5, 1, 3)
        x=Symbol('x', shape)
        y=transpose(x,3)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.83238397427219324, -0.80304945166676078, 0.2854233137814719]], [[-0.91700546179705711, 
-0.22967008232906316, -0.60376458756856954]], [[0.07087208457013805, 0.23117593172042072, -0.12667944419566446]], 
[[-0.27200565347308125, 0.31727021914865827, 0.40540640410723383]], [[-0.53708975045053231, 0.91228973325189755, 
0.72515582140538837]]], [[[-0.61931193506441096, 0.7057311962617685, 0.2507426651584741]], [[0.17997388476323595, 
0.60139517535377163, -0.99877115073110456]], [[0.016442022227512787, 0.061371032402413483, 0.14647605667883545]], 
[[-0.92003654886571784, 0.93042703193588072, -0.50427650232776866]], [[0.8567830545963715, -0.70882639028041727, 
0.47142817123861858]]], [[[-0.79629613633692742, 0.13970958592350069, 0.66198023787987537]], [[-0.15629334358873193, 
0.67125849304731378, -0.92132493615015476]], [[-0.41171369866982177, -0.67940087016833228, 0.84573809886934348]], 
[[-0.78634627557816228, -0.6648718173586603, 0.21803065010397238]], [[0.37245027723075919, 0.29269247650633523, 
-0.0530539286124041]]], [[[-0.42592013319073696, -0.54209531077082307, -0.48001439083151332]], [[-0.60430944008617171, 
-0.037657491348902949, 0.24493847028405114]], [[0.62609446119907131, -0.07190756457311509, -0.45046379127096903]], 
[[0.81443692227546949, -0.7066437837821713, 0.35296959921526638]], [[-0.58194723647967472, 0.85647238833506845, 
0.81985576259105364]]]])
        ref=transpose(xx,3)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transpose_Symbol_rank4_offset4(self):
        shape=(3, 6, 4, 5)
        x=Symbol('x', shape)
        y=transpose(x,4)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.33229462774359297, 0.90247279128446456, 0.50592066073757391, 0.060785036392947633, 
0.42887145358406364], [-0.907257269597191, 0.52593749893545261, 0.15806565139728712, 0.4014828795632972, 0.98069834651678889], 
[0.78648392989100535, -0.44370613604769593, -0.72850353165390147, -0.41298403201285483, 0.025500239778559974], 
[0.78312787656340843, 0.46791222498024743, 0.95311416319375231, -0.19391900054967337, -0.63841532569780779]], 
[[0.055027042647902036, 0.64399634269251083, 0.043106313163026266, -0.42193734992166587, -0.092307643484398794], 
[0.10064133409920628, 0.99940644225073805, -0.089336766142547264, -0.061535743483655203, -0.087024495000410651], 
[0.62741569832910571, -0.13599676515695536, 0.60970511662460902, 0.35050057112562061, -0.97838088249442912], 
[-0.39701550275339881, -0.94985980809765569, -0.94899518147143125, 0.19270095654119879, 0.055309394501884768]], 
[[0.048051840217318231, -0.51611293546959436, -0.40344898509551674, 0.17995256572278184, 0.75323171075063233], 
[-0.99737547946521543, -0.32300767971023481, 0.77295013504894339, 0.085248835129112743, -0.79439340442316086], 
[0.28222606480472634, -0.69527230454542521, -0.983488098301734, 0.86535616608678056, 0.41114575238879292], 
[-0.74078317645750813, -0.97945625176002649, -0.12625344089155188, -0.18298243111472701, -0.58222353017599215]], 
[[-0.0073724431427701909, 0.23504978886316708, -0.83090125959908101, 0.66530051906939836, 0.70266107220036167], 
[0.093890127834725723, 0.95900150757180191, 0.51197073529543435, -0.78425850996472746, -0.94470603565335387], 
[-0.071388551913005971, -0.47544393822466469, -0.068033180782151481, -0.21648824917734188, -0.48983248930826551], 
[-0.2890450213375646, 0.18194665582133607, 0.02109813507365943, -0.016889345412850698, 0.23762415001400372]], 
[[-0.87281897425177246, -0.96798831153907439, 0.98829641983199701, 0.36852869737264848, 0.030443638702282838], 
[0.70009727153104762, 0.30750607403016739, -0.067944927837938707, 0.50497652483500488, 0.74142142074795814], 
[0.94524474872090281, 0.61664148567473354, 0.30011498538454373, -0.18702691685130746, 0.67586267221196716], 
[-0.15715611674830132, -0.2664838239013203, 0.53499529703408633, 0.68966212533316895, 0.31706456142302786]], 
[[-0.61361056925887447, 0.2651271574376759, -0.058034816405870826, 0.026096774527163191, 0.86449606511668353], 
[0.66167065604121955, -0.76850782791683336, -0.31082833544595068, 0.10214720450989412, -0.13996060580137493], 
[0.81897939427689592, 0.98054333564702012, 0.64032021643575998, 0.0095206241554930404, -0.50399470919082723], 
[0.90867727408752286, 0.22213931062196868, 0.95267282274457266, 0.86263400660543099, 0.7519551789241945]]], 
[[[0.38191401430698102, 0.37730075198834445, -0.79991143368809481, -0.49974624670391177, 0.066369343448481777], 
[-0.91707771405962335, 0.59062252835510964, 0.82758172000123942, 0.32449099313834506, -0.81055151761968358], 
[0.44989031480621278, -0.54521499045113608, -0.53121692022940681, 0.24072105071115835, -0.12817241596903073], 
[0.13904976457703833, 0.030046961917888471, 0.70753775663615404, -0.60469897584877352, -0.12383354192837315]], 
[[-0.17727864548962713, -0.29320776189295783, -0.33076432225775898, 0.28775293953035663, 0.070466399800353852], 
[-0.94538163707919876, -0.59277722582268999, -0.93478437032776429, 0.54592562272539324, -0.61030865305945481], 
[-0.78174718345966432, 0.053898699368549963, 0.26068252952423698, 0.02860510133045624, 0.53860692587466485], 
[-0.1677595962509808, 0.84891644487260676, 0.25757512667578952, 0.010676947915794655, -0.38341410260612996]], 
[[-0.21063887494948252, 0.17940535968361693, 0.010336624497043534, -0.36892140792062067, -0.60349140836207993], 
[0.94352144795039594, 0.90594742886621549, -0.032237236043826556, -0.72815081506583357, -0.50488589347928992], 
[-0.044996589584375801, -0.54777479364307013, -0.25880117240478318, 0.092166950955798077, -0.74453035849522009], 
[-0.0021105882174756729, 0.92030237977249985, 0.48667071082933555, -0.71770449050514529, 0.20374645355617482]], 
[[-0.32968158110806312, -0.15882891043306024, -0.98973554045080636, -0.46311489579039278, -0.28189430873894583], 
[0.9462604680209814, 0.92990752172800017, 0.53141975729159108, 0.80570831962889522, 0.80396606987612818], 
[-0.20932447128280707, 0.055029256719247854, -0.96667008775210883, 0.46228702000070787, 0.6804861239115374], 
[-0.68254922808630791, -0.22923051538212302, -0.37598228292009517, 0.01642607444030153, -0.48762298517017322]], 
[[-0.73601485167433878, -0.086222233111941549, -0.25490018975133566, 0.59236131424013672, 0.97329818123254719], 
[0.036033089069700042, 0.12180976819687284, 0.37160256484045684, -0.70025744765373799, -0.78626985060205223], 
[0.14244157766668519, 0.35013755700794147, -0.42932738749017552, -0.95409618877765179, 0.62460606190112], [0.42470339031954674, 
0.11776945347732748, 0.97030071735149681, -0.030508385181680531, 0.2341359040188209]], [[-0.93019751564333197, 
-0.45361092731092323, -0.19723516978348754, 0.11582714865699772, -0.70071075045657172], [-0.574365934807338, 
-0.58283445893977137, 0.85997922228632362, 0.6777778161939656, 0.57937573444265911], [-0.29846108057039755, 
-0.95162324677220944, -0.9197625947297654, 0.78405565495850182, -0.55548401533165159], [-0.52864799493321701, 
-0.18196487033621067, -0.90739419753799266, -0.37273831977874727, -0.94605963276711336]]], [[[0.48867050622665253, 
0.50121200272268362, -0.11522914048382393, 0.40787887450728166, -0.047539751893834525], [0.46759103431805826, 
0.28114873662918072, -0.17744521938483748, 0.58822856341940177, -0.38153988957565566], [-0.44061179634452663, 
-0.97565902758452139, 0.45256693803279813, 0.35926683335150811, -0.21176594268102877], [0.92085622317262916, 
-0.11550419843494253, -0.96096571736481673, -0.77038222079052243, -0.78634403588930968]], [[-0.64191213335186137, 
0.0012017391400909183, 0.31513898199573154, -0.08724292833766234, 0.69241486856808221], [0.99597325069704978, 
0.49476384913356641, 0.42693038367571745, 0.50553024031512317, 0.025567401569908332], [-0.014350826868655853, 
0.06809959353037498, -0.94297390373874101, -0.22267232157528749, -0.50795581870192175], [0.47382389590740637, 
0.095664306022194889, 0.89734507254336293, 0.006974189965401667, 0.15898671363577122]], [[0.69901858702680331, 
-0.18710464818378902, 0.96414709435312784, -0.41086102283106962, 0.53578954843965398], [-0.98270251036616241, 
-0.80015058555456253, 0.25054328458308972, -0.16571504695518979, 0.58952623827739048], [-0.43057416854909114, 
0.92712034950679234, -0.0069512131742137662, -0.70334861768288848, -0.02654826105951491], [0.92741698205395462, 
0.92240331651669449, -0.94369884276001037, -0.71376796410123156, -0.27929588390311211]], [[0.8272907790057249, 
-0.10993839660087423, 0.96961173322751715, 0.36192353096889596, -0.70514828393138296], [-0.61215992865588342, 
0.375812873280017, 0.15828578451417141, -0.34315695877440189, -0.066783092137497224], [-0.22753764985447877, 
-0.0064898197588376672, 0.5847156917989782, 0.4630485492203853, -0.22693788606996823], [-0.70770906782317744, 
0.067426433242679717, -0.0088942713036097576, -0.42422727266183835, -0.61539593803778936]], [[-0.74120217114854592, 
0.066393910745510931, -0.73166587465353961, -0.025741071124232073, -0.30193486264924707], [0.22982289274604351, 
0.055835313713102241, 0.34815861007438209, -0.74389621152296437, 0.0077697667670584281], [-0.21763783415548121, 
-0.53681100602365928, 0.73158981322736794, -0.82545419131046871, 0.173428993559126], [0.68555782326953962, -0.7475989526480995, 
-0.40053248998810997, -0.68343194660083739, 0.19963556636740898]], [[-0.3497760449808982, -0.1566899029471458, 
-0.34378168284104782, 0.9016146890123149, 0.55751379359429087], [-0.0790899710491757, -0.54746513804520935, 
-0.89109379870634475, -0.93370678186121547, 0.08942389999118161], [0.44361763785236397, -0.68300690757588245, 
0.73919581973812498, 0.64529392753214432, -0.55324878497848751], [-0.969086352230907, -0.09773940808175019, 
0.023440100726240143, 0.10655036124361694, 0.62932270875795182]]]])
        ref=transpose(xx,4)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_trace_Symbol_rank2_offset0(self):
        shape=(1, 1)
        x=Symbol('x', shape)
        y=trace(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.16495468224218679]])
        ref=trace(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_trace_Symbol_rank3_offset0(self):
        shape=(2, 2, 1)
        x=Symbol('x', shape)
        y=trace(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.42747587571785162], [-0.76206235449212611]], [[0.56756592791038085], [0.92001802185126258]]])
        ref=trace(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_trace_Symbol_rank3_offset1(self):
        shape=(4, 3, 3)
        x=Symbol('x', shape)
        y=trace(x,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.30432006371619469, -0.22905338913674456, 0.42351956170977667], [0.35214002575912429, 
0.88921105503504827, -0.42739439460751671], [0.82999770972542053, -0.24368152054346126, 0.76327903699750377]], 
[[-0.27689661683308042, -0.47821019640402174, 0.050972678260606807], [0.3188234216192698, 0.40086532317431378, 
-0.77800871533307103], [-0.18065414138947311, -0.69993699005811294, -0.34667089306001908]], [[0.99684849266554032, 
-0.26053787631755632, 0.95412756688176081], [0.59498862532584784, -0.69512502548550681, -0.064097498492544203], 
[-0.13649401503153857, -0.21781863588425487, -0.048509177218103883]], [[-0.39222703275865856, -0.93666463077435935, 
0.06728383106782565], [-0.97258445996510789, 0.010034442276707356, -0.90229604923295037], [0.69358418855440673, 
-0.53650098576171312, -0.50681953935116653]]])
        ref=trace(xx,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_trace_Symbol_rank4_offset0(self):
        shape=(6, 6, 2, 2)
        x=Symbol('x', shape)
        y=trace(x,0)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.3694750359037331, 0.71358743964939397], [-0.093742282652185427, 0.50896359122836432]], 
[[0.85347316428928743, 0.49118560683579515], [-0.25941831681141836, -0.4320010783967263]], [[-0.2760108419751206, 
0.66793883720063874], [0.46227421258471102, -0.72413722178984985]], [[-0.74883130435508782, 0.070076073887716417], 
[0.64130431490622786, -0.98394733537310208]], [[-0.95412264969001881, -0.51566417255248354], [-0.16307641236026793, 
0.98363829896164612]], [[0.88586020150573708, -0.3397016558961703], [-0.57491311360772857, -0.78690905828319946]]], 
[[[0.39836393700999206, -0.63270208738135136], [-0.2959641344970767, 0.1331199240600347]], [[0.051134315403124964, 
0.63834753921918486], [0.94123408446203127, 0.63148116557776945]], [[0.38719619871015465, 0.63773576445595159], 
[-0.15000442219899179, -0.96078834001896318]], [[-0.56830030967894141, 0.74951162375423319], [0.8395447319472884, 
-0.14752461139071049]], [[0.17528853308372816, -0.53219338427670637], [0.048377969228808881, 0.14990742625452413]], 
[[0.053265153324379755, 0.92122213425272759], [0.83732746437009009, 0.43814166275704092]]], [[[0.81171997323018741, 
0.050189284623568087], [0.64368659625527402, -0.031103407826538421]], [[-0.89197465872384885, 0.50337420157698687], 
[0.95355395692917422, -0.50173533160056949]], [[0.2227005087016205, 0.84604906711247607], [0.74404874157118295, 
-0.31652247299227598]], [[0.20936533302820881, -0.47981411883936986], [-0.79812429019827413, -0.85006412570612411]], 
[[-0.34835305494224622, -0.28092858143638266], [-0.67836053479335412, -0.47092592927447807]], [[-0.4762265854343517, 
0.81066118213106519], [-0.52882160024668434, 0.23628622262664289]]], [[[-0.57714452658067983, -0.94459848277577452], 
[-0.22672215011737284, 0.75032447956233628]], [[0.60122975251361677, -0.34985235574140461], [0.058731385889433385, 
0.15621376590694092]], [[-0.73200655163930595, 0.21280839946856744], [0.81080911750417894, 0.69177135210443019]], 
[[0.32599640139949404, 0.66077480225915508], [-0.13983375872693049, 0.22753385790507874]], [[-0.8682379214958087, 
-0.94723615876099809], [-0.87056385033891615, -0.33792034125295589]], [[-0.86656851371244348, -0.34621403404401452], 
[0.40223101468808031, -0.47519853367365861]]], [[[0.63748179252209658, -0.5832056708017519], [0.90776206571008866, 
-0.43989465495579316]], [[-0.016474025722121866, 0.02187634681734707], [0.51992814008881871, 0.10846874003557327]], 
[[0.25254642503400238, 0.081802611693857274], [-0.55153027830967716, 0.3166821429560871]], [[-0.36703105520601986, 
0.35544509617061326], [0.70282199240315446, -0.2300975445344966]], [[0.68435837894355567, 0.25451709779959808], 
[0.21240874629983164, 0.21352558990199499]], [[0.31146533234807361, 0.3408700862763987], [-0.504329395303029, 
-0.89868721112966266]]], [[[0.97319504931670986, 0.56038442308741265], [-0.98093512597846488, 0.86123144490484749]], 
[[-0.33044930087386093, 0.63303634393803776], [0.50876715774738623, -0.42559630596165055]], [[0.131034837386522, 
-0.60521301245014536], [0.74470564639832104, 0.24450946961218123]], [[-0.91043823411503166, 0.077426268565379486], 
[-0.60978162133764302, 0.96395292244762465]], [[-0.85800175044000815, 0.95347198454173121], [0.97624761215874734, 
0.52351282878229899]], [[0.38893272392032885, 0.78013022187631176], [0.48928755294691362, 0.018595406529023917]]]])
        ref=trace(xx,0)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_trace_Symbol_rank4_offset1(self):
        shape=(2, 5, 5, 4)
        x=Symbol('x', shape)
        y=trace(x,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.12378126686132851, 0.29155430683608241, 0.86271434614578979, -0.11742093960687883], 
[0.3096490754214456, 0.98424686084231072, 0.047682319007110952, 0.11037750647267308], [0.80979559930294309, 
-0.55558374103286456, 0.30562481889393234, -0.060107448876047798], [0.32137687306255303, 0.11409107193014689, 
0.68863630213987848, -0.162339956706401], [-0.39678271595530479, 0.30823105783625726, 0.12481730156956949, 
-0.65655838356788188]], [[0.37190278270105592, 0.78087211009681878, -0.5712115602293879, -0.65678410030306233], 
[0.85257932570352102, -0.31908654965102068, 0.63411344983405282, -0.07680893575290515], [0.58099397171973477, 
-0.71698105785074984, -0.24354660896668157, 0.054323298860495139], [0.68519067831836811, -0.51251533167560348, 
0.62512754825557981, 0.50202928337687514], [0.79097760009172324, 0.36549819541627904, -0.35203980568116777, 
0.20959919307404018]], [[-0.81855419336037771, 0.03743773635384362, 0.77393976718134372, 0.17800308193249093], 
[0.9013417133178967, 0.97515203300010778, 0.92165156323643171, -0.27478200800050034], [-0.48006943841831262, 
-0.27726682525385704, 0.95102150281345432, 0.14269614427468058], [-0.53685584840593581, -0.19315649116132594, 
0.93091078259406013, -0.63208654722143365], [-0.91336292112565109, 0.39960911985000824, 0.62774042665873453, 
-0.4427910366440122]], [[0.96824982888547328, -0.76501446377948046, -0.084497317308873887, 0.35032191290416548], 
[0.45443874475115043, -0.67614747915686824, 0.35805507604279252, -0.65059154003009678], [-0.41981881920489528, 
0.64465722182163154, -0.45026362107301221, 0.66400052134070697], [-0.77864604744808918, 0.71753354687874582, 
-0.12511081275772229, 0.44152114585158397], [0.64615286685206974, 0.081113596793409126, -0.65488336674976022, 
0.1023686069245946]], [[0.66379766906805271, 0.17680814597757299, 0.23909690168058795, 0.62590288468658084], 
[-0.25037965360293613, -0.36319608732506903, 0.69856637614502426, 0.4975322249035723], [-0.049546003812601214, 
0.38249739847050179, 0.79461708872500747, 0.62941103922900377], [0.21893696910986726, -0.56594695446454857, 
-0.18451669262766912, 0.21622622136160619], [0.96117830325624687, -0.40565786959688022, -0.96361376264333809, 
-0.053661942793668427]]], [[[-0.4143734903349654, 0.015399175391612063, 0.03832389528949709, -0.02385534187482885], 
[0.39815239331466024, -0.32861245029375752, 0.54439771480435373, 0.22421570472444685], [-0.82140951647577398, 
-0.4962265612251815, 0.032276579168338593, -0.49920067177959715], [0.45421036728853426, 0.48205242543613158, 
-0.38468060784405789, -0.14689287020958464], [0.3801524842284878, 0.98958020513308442, -0.43475282256103931, 
0.53216608336887194]], [[0.57932442496651171, 0.29630529144309459, 0.38848702703468097, -0.97610661839641688], 
[0.019511053194284367, 0.087635528798853057, -0.11436457458590699, 0.89227169675701457], [0.45163471627961216, 
0.66436067041036351, 0.12395604030213492, 0.67045108762381167], [0.68795232894921332, -0.71406167213591476, 
0.61953408439302238, 0.36251154108701789], [0.93441188109710827, 0.43556955049545421, 0.64070329546365312, 
0.65383091190779585]], [[-0.7016324464114323, 0.48505659903570231, -0.62335659611951, -0.82467710141799766], 
[-0.00042357129587644593, -0.20895930532523566, 0.812423626676406, 0.82611561790288879], [0.48411706244884156, 
-0.084240741576977918, -0.25716822245247184, 0.62651637027671736], [-0.40943584403797506, -0.031945014291086649, 
-0.0098778087467032982, 0.91894796215157282], [-0.49086739641756272, 0.1805687025513798, -0.1261966568390942, 
0.43189663935468636]], [[-0.3613020149755688, -0.9364911512735099, 0.87409915068366684, -0.88796435875076196], 
[0.68955547419937058, -0.81471022221303735, 0.43887314869493066, 0.18475746015895744], [-0.97239674671757115, 
-0.51396510884131552, 0.42370739391811751, -0.57011771919372456], [-0.7598240962451801, 0.92502422612822177, 
0.44171920084267935, 0.18702900909141928], [0.85402195042094697, 0.7478630873905423, -0.11755556573379056, 
0.34764067266925336]], [[-0.84257914804615286, -0.37674785057143101, -0.67583596409386693, -0.30733938462062049], 
[0.49784772219676698, -0.87003179461678926, -0.61042075186317679, 0.74471521471017676], [-0.78414158502422859, 
-0.48804683157127116, 0.57169548435639528, 0.41243380496985704], [0.82238557009799917, -0.72865911005203476, 
-0.2871712630429426, -0.78757071588335292], [0.85111951836744892, 0.5123591436083319, -0.40010480491793321, 
0.22046127662744452]]]])
        ref=trace(xx,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_trace_Symbol_rank4_offset2(self):
        shape=(3, 2, 5, 5)
        x=Symbol('x', shape)
        y=trace(x,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.80759772146823461, -0.77749861744786375, -0.36595164071965813, -0.81631394659803314, 
-0.14810466968639213], [-0.85826154530843368, 0.69247913678532447, -0.34130155644802551, -0.73341502595875263, 
-0.0914643600114462], [-0.77855489110093301, -0.84050012789235073, -0.76462787937381682, -0.49583788093553438, 
0.98048332336770727], [-0.75209929633666728, 0.34260157374994149, 0.53426676144942831, 0.12011368460100513, 
-0.84644564353711083], [0.026287723774372829, 0.18633217002159053, 0.88805126992937278, -0.69998853536070271, 
0.65453143592628193]], [[0.44174324160573564, 0.13692958187203041, 0.52908626786544621, -0.10784948343546508, 
-0.54076063779496697], [0.66414440464095459, -0.60211372729570511, 0.44322504876735969, 0.30101035764676243, 
-0.0031369519522421729], [0.51844737757318726, 0.15806501376834281, 0.03572671193184962, 0.33936151502151612, 
0.91189615006923086], [-0.070202251975905794, -0.024090133185081353, 0.14909060132528196, -0.42284673113254478, 
0.86689233535961008], [0.89518462896722628, 0.55912706646746324, 0.30751715010176417, -0.65082951465644889, 
-0.35285211896659385]]], [[[0.13355685545417728, -0.76447398712495573, -0.75275708507682015, 0.25289324554044201, 
-0.23477887444205758], [-0.56591153678118156, 0.55696970284422043, -0.93875893604439797, -0.58030480133380724, 
-0.44881726840947866], [-0.47896651472143992, 0.96733234055265327, 0.22450555530743599, 0.20275275286750349, 
-0.14672046479404388], [0.55810739266760678, 0.38728923543560922, -0.86195907056442356, 0.38933324570089245, 
0.89322529122130256], [-0.067355271858988486, -0.47069266613030591, -0.27865768601618401, 0.1621150654990855, 
0.23087701733529364]], [[-0.76957238147428808, 0.13333872982873141, 0.14104212988871034, -0.45945852289191635, 
0.10235086621034206], [0.48317281840581394, 0.25338338904061941, 0.78401889148861126, 0.90466242531001861, 
-0.54036480877108506], [-0.98592067414310591, 0.54031682768232914, -0.65897157998900235, -0.39557182567935523, 
-0.43755597450475703], [-0.061841106404723956, -0.80855648338399977, 0.40575481011327397, -0.20020432584711512, 
-0.35829280236071614], [-0.18091229644353302, -0.8617313563589577, 0.081250358775862841, 0.82543683808636481, 
-0.46275954729595115]]], [[[0.52904118424582314, -0.27778472800013376, 0.78648513487074712, -0.51787469503355288, 
0.8726766975667033], [0.61118628401174435, -0.51993331224961148, 0.46973320194971824, -0.62741806634073627, 
-0.44130540230435722], [0.34675802396120314, -0.37407774500111546, -0.68051508554535411, -0.44237084589475506, 
0.60317869275934832], [-0.21817712329795746, -0.14355135609728897, -0.059399064953033154, -0.14887123868647301, 
-0.35748295461293478], [-0.50063286591425138, 0.090733138155519644, -0.0080899746842466591, 0.70580739887597321, 
0.092779804413343214]], [[-0.41376158685247955, -0.22479097547489935, 0.24572370847516334, -0.2897291132055575, 
-0.18292178371587542], [-0.39504387425912779, -0.24695248569015971, -0.020747309272694858, -0.009351018834127256, 
0.58497154188352618], [-0.92936518339681973, -0.8084569235062955, 0.73792232848005979, -0.5566527627021709, 
0.070323249875602345], [0.64908585266213192, -0.64793850000516495, -0.51603478112265022, -0.31684981086117303, 
0.51484248141945299], [-0.042900486671852534, 0.31371428969503667, 0.14738096447521887, -0.99722958474241752, 
0.92237015717396842]]]])
        ref=trace(xx,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank2_axes01(self):
        shape=(3, 1)
        x=Symbol('x', shape)
        y=swap_axes(x,0,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.21769555110605809], [0.64952253456407738], [-0.25166769574820047]])
        ref=swap_axes(xx,0,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank3_axes01(self):
        shape=(6, 6, 6)
        x=Symbol('x', shape)
        y=swap_axes(x,0,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.10642981337746416, 0.93313736904123146, 0.82015748200042782, 0.71664856903537277, 
0.63315269542224817, -0.012314407047726794], [0.35900809625301577, -0.98983348932369064, -0.40989812502338441, 
0.14735350026838101, 0.84185671846725274, 0.85448636767568664], [0.55936381064468765, 0.6998148387583274, 
-0.083221259440219297, 0.21649023422653224, 0.51173535114955859, -0.0080810523724805794], [0.35178682958173324, 
-0.144095296998072, 0.19516093680960656, -0.85163658295682798, -0.1732303733117877, 0.05284531490486355], [0.87461473343300167, 
-0.69229292170398948, -0.31214281940860045, 0.50131552625254661, -0.29548794007453005, -0.95820495849456977], 
[-0.5629353012110605, 0.78911962598701457, -0.93690095281629082, 0.016891202833589691, 0.52907700442059591, 
-0.94680737627790879]], [[-0.44204560716794128, -0.68587170421684163, 0.24468436022591566, -0.52480503853905258, 
0.7294481545694953, -0.77165885264286138], [0.78767599563623358, 0.59654752618387796, -0.78737353812775202, 
-0.94243949981808761, -0.26902037004663515, -0.41777877061147994], [0.36816304107602438, -0.47873121779080852, 
-0.93916948357494245, -0.99073161343317473, 0.47548100710545205, 0.97683447916110921], [0.77977236205393829, 
-0.031342861256108412, 0.95422852940122893, 0.010338115024421501, -0.51599948122491934, 0.68942589542518307], 
[-0.20128992450262095, -0.04486657096022828, 0.15018607978999077, -0.37571761061665554, -0.020277538955634888, 
-0.046908976593640128], [0.92522781174188551, -0.38846641046788433, -0.51747455410444143, 0.71452191111396957, 
-0.14809389077680324, -0.01003624362003408]], [[-0.038728130831833241, 0.39356321015567586, -0.2298366697047185, 
-0.62153668210564272, 0.24912521244214481, -0.22735741543628052], [0.40717179769917733, -0.12405604147598193, 
0.48565183685349878, -0.078790229392866395, -0.099053493947452997, 0.70275380082762195], [0.92005406115813271, 
0.18311029457922312, 0.79671610805430482, -0.94413559287042315, -0.78181275719693177, -0.23242322804060311], 
[-0.96947492575447125, -0.14439627972236724, 0.41873983738193865, 0.64899941975595832, 0.059594798198005616, 
-0.87567563011549532], [0.38066524199644136, 0.5724925851653162, -0.62859311384448446, -0.17266471018717766, 
0.86381499680822404, -0.896411221316888], [0.89871462138059699, 0.94482702416356434, 0.42917294271974304, -0.50277987522260603, 
0.34383413250278805, 0.42981922158802344]], [[0.59302452605496692, -0.16964831717309559, -0.92219718169124487, 
0.56838282448134025, -0.4255801270410553, -0.075054748497668466], [0.37597626870097423, -0.36966720283676691, 
-0.65052689084002835, 0.10549768051596375, 0.099404372271977515, 0.12954293493145008], [-0.53668098391245533, 
-0.88063229979757773, 0.96174793460523822, 0.035120432094327914, -0.41250152181042354, 0.78832748617016013], 
[0.61466245144919229, 0.66567717513777813, 0.24003996500751912, -0.64540049835011493, 0.94269227551763013, 
-0.31794258606077119], [-0.1049339556011708, -0.72055270398427052, -0.37318985123779269, 0.29317613863050029, 
0.24422394149849458, -0.47108704174615745], [0.98054390565748806, -0.31863530538615725, 0.68986895085906674, 
-0.79624851063001145, -0.6383020151127019, -0.14854691405866105]], [[0.22116226554264173, -0.49810233836219342, 
0.0015458268404977549, 0.92316589254252857, -0.090649830290937317, 0.091664186157400795], [0.65077988335982928, 
-0.41791747897614506, 0.7623419767569235, 0.99291420852758261, -0.22543914567824452, -0.3454827090482091], 
[-0.10056092189574906, 0.23713451850214962, 0.4768941874743009, -0.87487428040373172, -0.73583657430605531, 
0.98831244316329636], [-0.61939229423378284, 0.33051855192420532, 0.44557189670191244, 0.47129725236761488, 
0.98814900932512906, -0.97036775733744762], [-0.1673381412794559, 0.87838622774462727, -0.14430804969154543, 
-0.9989340485256728, -0.90458185109800948, 0.2905877623494455], [0.14194025762886975, -0.27964869026749151, 
-0.77861151996060385, 0.5116601990250611, -0.67312526253587146, 0.65840066060677183]], [[-0.62430351948770424, 
-0.0071471677646108844, 0.23432830431415552, 0.19344239492153004, -0.54580482114431605, -0.48088275126546454], 
[-0.21887503383811646, -0.23203626350879514, -0.33073912696073027, -0.28666519592046158, 0.583387749899553, 
0.62562647285352657], [-0.31676957522573312, -0.51425925349198054, -0.0054990632935891792, -0.36568000953427915, 
0.16292360668081862, -0.93458577705127999], [-0.65443868492351998, -0.063834480505040814, 0.63675966720505373, 
0.17185110182334107, -0.41660343373496866, -0.1330562585518098], [-0.98171945486618739, 0.22067485612946203, 
0.29853753241602421, 0.35381182823888402, 0.506851126934174, 0.18043808005509243], [-0.4644184452940141, 0.50048031667003112, 
0.48171594629246184, -0.43247814482259206, 0.48258976183446389, -0.60409641280938242]]])
        ref=swap_axes(xx,0,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank3_axes02(self):
        shape=(3, 4, 3)
        x=Symbol('x', shape)
        y=swap_axes(x,0,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.78150692807886113, 0.35074875631418645, 0.31166283857098609], [-0.65233112393850146, 
0.7341415518848966, 0.46901885172185653], [0.91480659429973343, 0.86487046557589276, 0.25952196599158284], 
[-0.94308354353181278, -0.63396367482598448, -0.65048524872769198]], [[-0.5833241935608382, -0.45564761219038319, 
-0.1304610302526017], [-0.54242309520521381, 0.63606522960962186, -0.85593809383020769], [-0.63787322886693532, 
-0.26899767269556674, 0.78194079544220929], [-0.90452431040130432, 0.39860683886210535, 0.72139509658777867]], 
[[0.1729670920274835, 0.49426611440511925, 0.72287641158487936], [-0.035781019033239492, 0.77306433154521215, 
0.95436730092806465], [-0.23434525336432932, 0.049275649255643206, -0.29843346738431498], [-0.91854464190031426, 
0.24199702505404619, 0.060576881190838261]]])
        ref=swap_axes(xx,0,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank3_axes12(self):
        shape=(2, 5, 3)
        x=Symbol('x', shape)
        y=swap_axes(x,1,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.86606688141926691, -0.67147257017990913, 0.13303136638848101], [-0.73632125665457449, 
-0.84884764271071034, -0.24026507621894067], [-0.30028531071790043, 0.76111924612151327, -0.86361085985125641], 
[-0.047309673018421261, -0.19519123112948544, 0.4770177878515649], [0.7482633540805721, -0.24586670365608754, 
0.55454095062524589]], [[-0.48661879568392186, 0.54069584022189665, -0.9827050585470638], [-0.98986554425556705, 
0.83735103307763925, -0.35720997345846506], [0.22690695629428714, -0.3247363288398819, 0.79094968432584767], 
[-0.29938075620187887, 0.8828976588670121, 0.9321920760917366], [0.72012146484851725, -0.57686641883828704, 
0.91172274784867335]]])
        ref=swap_axes(xx,1,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank4_axes01(self):
        shape=(5, 3, 4, 2)
        x=Symbol('x', shape)
        y=swap_axes(x,0,1)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.063856941814865564, -0.094913266307980138], [0.63653857996991969, 0.33310832173295157], 
[0.93193333601172545, -0.49706577894562498], [-0.41499972630183635, -0.61608649644555191]], [[-0.10030800337238888, 
0.03055162818573498], [-0.20355903781447116, -0.83011687143771296], [-0.27391354975460858, -0.38460600693961911], 
[-0.09266027837392854, 0.83281500501746963]], [[0.70109788966667441, 0.98116800131149406], [-0.27633973304517334, 
-0.56763517722783252], [0.61328128315019548, -0.9319207299758332], [0.12266029597760664, -0.99356708033884766]]], 
[[[-0.0052924604383173257, 0.081969131020698027], [-0.34557614374807777, -0.35751659542465553], [0.86013005812164733, 
0.6468717616083044], [-0.12469461378601543, -0.87147370104693023]], [[-0.97375471723106055, 0.71167773140777779], 
[-0.0490018537443484, 0.99072667972664608], [0.27895495180184637, -0.019230215561527819], [0.60330684386953548, 
0.31892960794234604]], [[-0.93567470639700723, -0.11032138118245394], [0.11027924524267418, -0.59757429125316963], 
[-0.84110635565523029, 0.58423320507966259], [0.44148162354999521, -0.44792654319916414]]], [[[0.30238075918093865, 
-0.32112586335624305], [0.050097137035601147, 0.30275065482361674], [0.36342067327254957, 0.7970379139923407], 
[0.7999633468200289, 0.67636234706532194]], [[0.711365599522638, 0.31783026115397051], [-0.067988986703677146, 
0.74435576829730343], [0.88882492272283176, -0.41345644036886053], [0.63191751104096561, 0.28191141757256855]], 
[[-0.17169973189435761, 0.098986321052313198], [-0.79516398155274959, 0.33677146182147721], [-0.54174053106741482, 
0.5992940407382481], [-0.98295177693900704, 0.11155589323115001]]], [[[-0.29393858124872074, -0.37107261469951625], 
[-0.57627255581814163, -0.17427726663715903], [-0.70452354173337683, -0.72529638479774405], [-0.81493470452133887, 
0.22016925688161137]], [[0.39949951438753661, -0.67790980242657595], [0.8119622350269935, 0.35550548057137243], 
[-0.081656055239998304, 0.65328828082207946], [0.97444088271124141, -0.67414886229642246]], [[0.85794809382472903, 
0.58804155655312629], [0.31241996517333237, -0.0501225464077939], [-0.19286740684773584, 0.44503671557481184], 
[-0.99884223044964315, 0.12541323812748328]]], [[[0.64038989465966156, 0.9227685353773607], [-0.22456976311899046, 
0.70577230697364746], [-0.84996566550598307, -0.020288562525679676], [0.6759347617580258, -0.88006529281308032]], 
[[-0.48171315329095954, -0.80903012176993783], [0.3692883708844632, 0.79573589862083671], [0.3960717634030444, 
0.58938341812778172], [-0.055507065597342642, -0.5226620536612776]], [[0.60214359487407831, 0.59459452261371859], 
[-0.90015779991417055, 0.95782529723709176], [0.042819485409720803, -0.34942353666888026], [-0.51827270804959968, 
0.79912030944745882]]]])
        ref=swap_axes(xx,0,1)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank4_axes02(self):
        shape=(3, 6, 1, 4)
        x=Symbol('x', shape)
        y=swap_axes(x,0,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.032392152120468198, -0.87858540432830945, -0.20721356466346585, -0.072357648158782428]], 
[[0.46237263599117506, 0.67216785874003659, -0.23621466572040317, 0.097260656630647979]], [[-0.99566928172404134, 
0.14851348327741087, 0.74191000934385753, -0.5831653157085126]], [[-0.34391886970682939, 0.25644213327615906, 
-0.83161759153222703, -0.97555934055890359]], [[0.21645836686921505, 0.19042336344333144, 0.70444396527553699, 
0.68957109388274218]], [[0.48367753011427395, -0.058073023700566218, 0.1243821782224519, 0.11165517624356447]]], 
[[[-0.50833603579916575, 0.78741164988661239, 0.29419207247304624, -0.4212201307141803]], [[0.2805635561859392, 
0.077557384999537238, -0.17267011615837169, -0.31007843365782572]], [[0.95528812756928483, -0.62607647421789969, 
0.29517769171562303, 0.38010605335638092]], [[-0.79526532040827291, -0.31790444321594236, -0.66036616046982877, 
-0.11579279996544134]], [[-0.050286650919344034, -0.92068361055053072, 0.44589302424758759, -0.71061337833520688]], 
[[-0.1081509843895363, 0.13972891598280879, 0.83000851314674451, 0.83079505613802973]]], [[[-0.72824314297385029, 
-0.75552343770000485, 0.65092979197728895, -0.85696811134506712]], [[-0.77687389073349977, -0.61036565595494574, 
-0.98446945757495152, 0.20719359414207039]], [[-0.22156737278440919, -0.98811071154431418, 0.39014982748396032, 
-0.77703730458939524]], [[0.0024708759609239905, 0.68496380859756401, -0.82188732686452637, 0.59650851535336469]], 
[[0.43135688830284913, -0.73785151433135709, -0.44749477482555333, 0.95219938202484378]], [[0.063699782783157133, 
-0.44073221820066788, -0.23448852201157133, 0.91739771503013112]]]])
        ref=swap_axes(xx,0,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank4_axes03(self):
        shape=(2, 3, 2, 5)
        x=Symbol('x', shape)
        y=swap_axes(x,0,3)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.73655270349795665, 0.37320235450886474, -0.65416825324659511, 0.71371702718505903, 
0.42730707442165894], [0.11156226907845035, 0.4049916879275608, 0.38325500080524133, -0.37385996497036733, 
-0.40513124420511648]], [[-0.95981210825114882, -0.82481160481523808, -0.19851869432524039, 0.31652530092036457, 
-0.37524663524668278], [0.031399362550116994, 0.63551188597753816, -0.33698037325121399, 0.49323910117044401, 
-0.69822267377888569]], [[0.92864679115022675, 0.18560792215281152, 0.91100736517902536, -0.29214254141667495, 
-0.7145356588238212], [0.7518016330188344, 0.21987803698585795, -0.96891827778554052, 0.05798125450370728, 
-0.11404345109511449]]], [[[0.41774120703077378, -0.061168198668940743, -0.46122560725834294, -0.91909015905731462, 
0.86184234235636059], [-0.023655222303560874, 0.86016547242839381, 0.72540068832407756, 0.6558775220488231, 
0.15755500790847954]], [[-0.25974489499063891, 0.24790178891365833, -0.14088431424525294, 0.61844999192534122, 
-0.29292121843485175], [0.35947771810054774, -0.5977284519273629, 0.70100108081636647, 0.10882508673428171, 
0.22287681187593433]], [[0.22499439509756702, 0.18751827244666641, -0.50515259984874361, -0.70843458916516888, 
-0.71679399309060576], [0.83215425602836168, -0.0041162846569491318, 0.61052560285037405, -0.96041141320016599, 
-0.51700717736090573]]]])
        ref=swap_axes(xx,0,3)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank4_axes12(self):
        shape=(4, 5, 4, 4)
        x=Symbol('x', shape)
        y=swap_axes(x,1,2)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.73974500641024754, -0.83371982240784925, 0.21753680430931688, -0.41437529114886029], 
[-0.8546393086450621, -0.310149511836598, -0.94699954067416336, -0.78393370209535318], [0.293302632249824, 
-0.099179053910463111, 0.1319553346234541, 0.52830432139105765], [0.29578786329711049, 0.70461582452576121, 
-0.69096222960973797, -0.97238474734273783]], [[0.63336124232100466, 0.26717653299676214, 0.41388597339062616, 
0.10822172834463384], [0.93922149239328578, 0.76652386182769883, 0.48006326711879499, -0.26873834853763579], 
[-0.21276105239360588, -0.87393982623589017, -0.24824239356501021, 0.92063270656309992], [0.8845337986443027, 
-0.47281198052783679, 0.16376779852748102, -0.85603254527460027]], [[0.038005424874316907, -0.0011980050297917266, 
0.046377985109577802, 0.84355230349640875], [0.75793484306371606, -0.45858869097547461, 0.70203002155922745, 
-0.020920037371820133], [0.48983551784284707, -0.14471388322105927, -0.49256674092370978, -0.52293776385196167], 
[-0.74348057482577268, 0.27565829665552588, 0.22331614710907499, -0.35169551322675652]], [[0.083149533907559192, 
-0.40316031946630115, -0.53356228859268295, -0.28050801261687108], [-0.20414944263890256, -0.48241869508077406, 
0.2479668290160828, -0.49291886695503262], [-0.32355245698546087, -0.42979837068501947, 0.21473071575626634, 
-0.64785270811244455], [0.38046892933580256, -0.18284685569545256, 0.17745455791067499, -0.11017177303366954]], 
[[-0.10183648655871247, -0.77473072205885862, -0.36670385344032375, -0.77662662815196182], [0.61331785915189863, 
0.85095214371258732, 0.76269851060787586, -0.29269308493874879], [-0.030838668942959169, 0.17711382316902169, 
-0.10006193246772632, 0.66182707690578213], [-0.12417679908354406, -0.65193225088882412, -0.34420971949158741, 
-0.91584823737266352]]], [[[0.81698004747699016, 0.75728774485525241, -0.25596676262257811, -0.32938176486999771], 
[0.65180398437595133, -0.73116477897866017, -0.88935701183377325, 0.72645513235838366], [0.24572501930739343, 
-0.74384158205436446, -0.17281573027553598, -0.64584012788674405], [0.94901041997887714, 0.62044725307496318, 
-0.017374418495853083, -0.58928765812938799]], [[0.37756857122388743, -0.54062421994586973, 0.34776639978082735, 
-0.53166179238584577], [-0.65288384282471146, 0.81101512397636366, 0.97915287497332559, -0.64245111285169409], 
[0.27062300354688218, 0.95860478660345927, -0.21953487929888027, 0.1158152762656981], [0.40836425978630264, 
-0.95603653105492481, 0.67489154809920682, 0.1569373735186439]], [[0.23628960485060246, 0.80897927037271389, 
-0.33552255620708982, -0.24717169298748631], [0.19602605418458974, 0.60168665400598975, -0.038773023370106019, 
-0.0050449308592941389], [0.89512112413711109, 0.5102535250824507, 0.96557766121495048, 0.26414192911120082], 
[0.32355356620327513, -0.26328406660924863, 0.0011789903626557852, -0.07858518899748379]], [[0.66949068536729639, 
-0.86893520984962391, -0.0032838802662580324, 0.32352851509519365], [0.010115172450642085, 0.97163130262561825, 
-0.44590201183068578, 0.24580963882996465], [0.22385087710094242, -0.56575408063181998, -0.61201742642495605, 
0.70189454568734932], [0.81300586153878163, -0.80658772377859567, 0.60553018762765687, 0.8496666476520438]], 
[[0.81712573608010475, -0.11708459111171599, -0.14174721514661792, 0.7572466239718596], [-0.7197472592615819, 
-0.75916402966364216, 0.84637356216723925, -0.89590648340922541], [-0.90368337472571336, -0.45800072918794887, 
0.53620955153023853, 0.6535510668152642], [0.73017291499689985, 0.19057835247507926, -0.20098654417097195, 
-0.76591348229153833]]], [[[-0.1026808205590819, 0.068852924749491562, 0.2552944461633635, -0.26315883861592626], 
[0.0010214986146361849, 0.75954979407328072, -0.11358513540833282, 0.094333207566387411], [0.26217356414543724, 
0.7281561221191768, -0.59411920574728994, -0.55684494517855154], [0.34391126178580533, -0.11093233760422416, 
0.80883631238505327, 0.8044453319110858]], [[0.67844349054600883, -0.31720562092151994, -0.44173583569057451, 
-0.4975209421522071], [-0.15159270219238796, 0.51441394267641827, -0.052837900384890446, -0.72889238156893521], 
[-0.55842758530624459, -0.28639271688734058, -0.99861188413619284, 0.77276564728362729], [-0.78969792241261816, 
0.84255608782312663, 0.93819816788875054, 0.98170937185533602]], [[0.22354943547226513, -0.10724637716120111, 
-0.029324260192181928, -0.61448253179418244], [0.64235652166229529, 0.17637680251529275, 0.044949532213990562, 
0.077552239663591038], [-0.2982650488459142, 0.60714470975066037, 0.41795520192372626, -0.65610096892596648], 
[-0.54429798496678994, 0.22641634484580653, -0.11873617719878782, 0.45847464206323618]], [[-0.41214829821260901, 
-0.73930285460165202, -0.11833330429700806, -0.53355787607952054], [-0.69546345167732171, 0.11831194274386059, 
-0.19181983843824635, 0.40594337966338179], [-0.27109697909391528, 0.88434588359945487, -0.46246016482513563, 
0.82469500229339521], [-0.65716164755676854, 0.6528856036566999, -0.00048050833613566546, -0.28626032537464252]], 
[[0.61902081823055344, -0.38741187711123426, 0.72848605377799491, -0.46385954957752573], [0.20581650696375098, 
-0.5914179771306034, -0.94032551924523089, -0.50402170189231676], [0.85893017440073027, -0.95752872135892719, 
-0.54082405521629506, -0.73104239789176928], [-0.48926453296538774, 0.26339805862283017, -0.69161316915529181, 
-0.42145044889581817]]], [[[-0.46085216379648952, -0.59971502117944309, 0.82525377242678699, -0.74467651698952064], 
[-0.12656538796120387, 0.59457713079418628, -0.6281660208133435, 0.14138176616731313], [0.65776664806455232, 
-0.98856244331419196, -0.60770514495225569, -0.026114659759104653], [0.3139065522867428, -0.18272150455731562, 
0.55576408983896552, 0.99788186678329716]], [[0.9059284768235556, -0.51773615991899891, 0.85952900272762456, 
-0.95599431544334301], [0.68006800199022721, 0.41535414988166131, -0.90591998281172503, 0.44129964366448116], 
[0.3279901686246447, 0.64011715337637254, -0.86513608807714037, -0.61570427033884911], [-0.91240320852160828, 
0.42987085919696355, -0.51807895640201029, 0.10594516117114616]], [[0.89680386268605394, 0.42186315681156894, 
0.862612878663507, 0.503099487985319], [0.65181351414114608, -0.99080137421501258, -0.53711829660618204, -0.98852792694615532], 
[0.38305109465214549, -0.80123595756587784, -0.12987791505350632, 0.95584443219891724], [0.30600658489089083, 
0.70992922504956635, 0.034480901024991351, 0.42911425109955537]], [[-0.16391106960386237, -0.44380535969904855, 
0.27736852645546106, -0.94872286214502899], [-0.58692056445126828, 0.099555142930463525, 0.75003991388314839, 
-0.060041223841673652], [0.45111383625269563, 0.95010647424752182, -0.57942533463504931, -0.45734187144125227], 
[0.80688663247320602, 0.072141102168988613, -0.72879069264221585, -0.29714805863267224]], [[-0.11986412361868148, 
0.46590235054739204, -0.36128360623333067, 0.007445604685021312], [0.18064242065262648, 0.28540170426477629, 
-0.15935838344121578, -0.50096377480610355], [0.47593918685144199, 0.95400895143362852, 0.33781924829535193, 
-0.52010729368308861], [0.44266632196358069, 0.1189649005099187, -0.66124683079497526, -0.36664336909522022]]]])
        ref=swap_axes(xx,1,2)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank4_axes13(self):
        shape=(6, 5, 2, 4)
        x=Symbol('x', shape)
        y=swap_axes(x,1,3)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.98586884069051117, -0.56763904081463745, 0.62754668592953977, 0.83554025375044816], 
[-0.65403431473519658, -0.23806712981035716, -0.2620043793238036, 0.58294017677485832]], [[-0.069195971494764308, 
-0.2410814078922292, 0.54615125832104305, 0.69610142401529096], [-0.2367767974551136, -0.51061612535795708, 
-0.54744662928147347, -0.42851213271730781]], [[0.45669089547252772, 0.20755395634942553, -0.22368047296648941, 
0.61578602814763328], [-0.90670973474435934, -0.75998168100777486, 0.039269783985293349, -0.50804076341016957]], 
[[-0.98822222184209929, -0.66683173151525166, -0.15881345804068703, 0.39331928226341506], [-0.94925376795214378, 
0.50687654185979314, 0.67459149659057571, 0.42205412531344555]], [[-0.98885223664573663, 0.41124550454858144, 
-0.11899816064513247, 0.38461407677183002], [-0.33277146097126886, -0.94678949539421753, -0.10170803118762195, 
0.079609608504609186]]], [[[-0.68508798562536444, -0.20409179478859096, -0.83085400080504312, -0.52821247146605099], 
[-0.71756064800062269, -0.73232900951421209, -0.83013994293004578, 0.78576052750522041]], [[-0.056263255060658768, 
-0.75142796209087548, -0.29228826972740984, 0.31546520290033198], [0.10860579044569407, 0.2953595497633017, 
-0.96880921291096467, -0.58650488029566872]], [[-0.47514455444973236, -0.42497155435173184, -0.51735727021721156, 
-0.93224295780902611], [0.35303007259966201, -0.096089133539329019, 0.35021666395175099, 0.068942964804255658]], 
[[0.31117566695681154, 0.62977167338711704, -0.98653613800975903, 0.57454143427984983], [-0.27533824607370461, 
-0.72926665998886886, -0.21714216312152024, 0.48493593516734568]], [[-0.46499109648503012, -0.82060683718034877, 
0.64868871575108833, -0.76253817365867338], [-0.71640965644954879, 0.72375550536015432, 0.061821859198022189, 
0.75904498897839145]]], [[[0.77761562462771061, -0.47486525207849928, -0.57820539342091593, 0.4994870541369103], 
[0.18274635006921569, -0.22518254658484493, 0.74111419727651628, 0.56589512492482363]], [[0.53136304186244354, 
0.33672791017295811, -0.85437644050173533, 0.11955168710193376], [-0.71692420533247048, 0.23409682186080749, 
-0.49934554501597894, 0.84003982986202508]], [[0.027524303557869167, -0.68892382635088234, 0.23361830128247263, 
-0.51916691552699845], [0.85795217902842791, 0.81305902312186529, -0.50030437819576123, -0.80401988249127498]], 
[[-0.493813160161098, -0.46020211090420338, 0.015196816567369043, -0.35139795074050428], [0.09977983042678118, 
-0.93498345014429796, 0.55224770957282687, 0.049584450446019446]], [[0.099396059589944574, 0.89847668843394191, 
0.61409757576932211, -0.67851064580160902], [0.29940157819000013, 0.89518326628515954, -0.45142707292925444, 
0.53142173420290484]]], [[[0.69557156818840249, 0.64349908807483014, -0.5939614958688304, 0.12130911207605055], 
[-0.66179124678061108, -0.60255672498067403, 0.24869613027276127, -0.8562790376636189]], [[-0.99659743366929177, 
-0.62379864186615919, 0.75985232629153754, -0.85647494999768692], [0.4498873294495398, 0.077431505587107541, 
-0.63956360937173962, -0.31751072898792598]], [[-0.50431016910432369, 0.075311333114095547, 0.89796493959200685, 
0.22169987214110742], [-0.7018574397793742, 0.019644482801496777, 0.52020743386419088, 0.93817575883996929]], 
[[0.8920568815786134, 0.56772438457000529, -0.48778358487149842, 0.14379811375198215], [0.85216187592435411, 
-0.34249735734100506, 0.59160310686902973, -0.66574876391032656]], [[-0.29267392161463279, -0.12522039617714387, 
-0.68151218128674751, 0.92868539522801496], [0.85765930613332464, -0.073765946209471078, 0.59728530416625158, 
-0.92616601572826562]]], [[[-0.1465081734143816, 0.92338795107381855, -0.41002740212737376, -0.71504687410665269], 
[-0.39574583384533746, 0.50552964497469333, 0.39714983384696168, -0.13525363014309799]], [[0.31930723179882126, 
0.1817314789621427, 0.4208879886194663, -0.98069541290646112], [0.29033391859508906, 0.26335963712446242, 0.096519668302318617, 
-0.20361048217768896]], [[0.054821771505883188, -0.88886271472233513, -0.68351287341409606, -0.50683756997411833], 
[-0.89936849027475452, -0.080379935225404653, -0.34300733239518966, 0.24289410909149534]], [[-0.2633080941212973, 
-0.13094793516896752, -0.3559027871776681, -0.21944401124756929], [0.89188670486528165, -0.41873390903617413, 
-0.64877177161590383, 0.88338346440389293]], [[0.60751793712518443, -0.047010738501569893, 0.59847525628544052, 
0.09067318220541698], [-0.0023393142245689003, 0.40008638714461608, 0.83407163919823746, 0.033572058738858823]]], 
[[[0.48798818489853302, 0.42284790829385477, -0.51495123978061774, 0.91367949640230695], [-0.55660785369460708, 
0.35220466835487918, 0.59059552287834305, 0.14390255274561503]], [[0.30507003478071382, -0.54022512885864349, 
0.85451634671727494, 0.08594248667004134], [-0.38255745890897597, -0.21110045449530723, -0.0072568407794140199, 
0.0092424670668349584]], [[-0.48010168213077642, 0.58222853689064125, 0.22935563562136663, 0.81726550239834794], 
[0.18458944686444978, -0.98808622125447609, 0.64215157538470446, 0.087375559503421796]], [[-0.8353278498847434, 
0.30111102220698682, -0.075337140458557661, 0.87057774375420438], [-0.25972290161565814, 0.71453789889843566, 
-0.53545070126329364, -0.94943548222656493]], [[0.21861272299305501, 0.87468891566862483, 0.18995754714113855, 
0.40400967087496031], [0.6103551344483813, -0.23141052939819828, 0.49320663050936542, 0.36272358143538352]]]])
        ref=swap_axes(xx,1,3)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_swap_axes_Symbol_rank4_axes23(self):
        shape=(4, 1, 3, 4)
        x=Symbol('x', shape)
        y=swap_axes(x,2,3)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.24160383981535571, -0.81174970820976222, 0.38708922182200212, 0.62651640709430234], 
[0.36270038425702467, 0.97555545129408383, -0.74118917438158305, -0.23572973809237041], [0.30469270096686474, 
-0.23060488645631683, 0.1124922120724583, -0.50614930605602937]]], [[[0.86631231259899844, -0.33994352272710837, 
-0.75572595805088527, -0.042268642897813624], [-0.77046112814426748, 0.051732489710859753, -0.50614004746109131, 
-0.23789040494465841], [-0.72153793260161936, -0.1443348257964574, 0.26774449631048047, -0.80817909716605807]]], 
[[[0.95014118654256841, 0.70782960440776432, -0.8691158309689011, 0.28917484644095093], [0.12311541419603267, 
0.70697165618196944, -0.35966452047115482, -0.54121075064958557], [-0.82342237341037161, -0.23203015477661659, 
0.35515576903252732, -0.56105072143070522]]], [[[-0.14760720876814282, 0.033999093224418742, -0.88801114460949493, 
-0.83689554457136794], [0.5673107699256037, 0.76140808723884801, 0.51395290101161351, -0.72054785713713354], 
[-0.39109655845316293, 0.084937361306425707, 0.16390710715953305, 0.78133917514349061]]]])
        ref=swap_axes(xx,2,3)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_clip_Symbol_rank0(self):
        shape=()
        x=Symbol('x', shape)
        y=clip(x,0.335857,0.766737)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.0258064974758)
        ref=clip(xx,0.335857,0.766737)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_clip_Symbol_rank1(self):
        shape=(2,)
        x=Symbol('x', shape)
        y=clip(x,0.367217,0.541996)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.3449432513872388, -0.77644553351747536])
        ref=clip(xx,0.367217,0.541996)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_clip_Symbol_rank2(self):
        shape=(1, 6)
        x=Symbol('x', shape)
        y=clip(x,0.480419,0.899882)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.86464363016774293, 0.59495695633350199, -0.98247985534475069, 0.8702456454375127, 
-0.18462541914989106, -0.3138750179749672]])
        ref=clip(xx,0.480419,0.899882)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_clip_Symbol_rank3(self):
        shape=(6, 1, 4)
        x=Symbol('x', shape)
        y=clip(x,0.153327,0.938393)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.88501914363206269, 0.70633645202787854, 0.39294725977038425, -0.36747911609279837]], 
[[0.61581416134035494, 0.96665681074683496, 0.021543754027566475, 0.20510691680067894]], [[-0.78327196674320221, 
-0.078805526910647794, -0.70602961195450198, -0.82032450207928909]], [[-0.24033679038843725, -0.67161255084556482, 
-0.52610174323234005, -0.76855031941624907]], [[0.16609136471226393, -0.73561119079956971, -0.021137711308077733, 
-0.74578273589423194]], [[-0.086261104980276837, -0.26554987713573408, 0.64936137483581446, 0.0035197018053816365]]])
        ref=clip(xx,0.153327,0.938393)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_clip_Symbol_rank4(self):
        shape=(3, 1, 5, 6)
        x=Symbol('x', shape)
        y=clip(x,0.416630,0.717385)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.35242643946250474, -0.0035143342919330411, 0.10254480255632181, -0.080906077787140429, 
0.4131090081223423, 0.85669382687125184], [0.5852867602824805, -0.64294274125600381, -0.18547315298357425, 0.26924363970141085, 
0.65808443917026582, -0.54681771256506062], [0.15621135763156402, 0.13323890568790997, 0.37775552491546738, 
-0.50878571126020811, 0.39707844728879449, 0.18836739743975128], [-0.62224628406926596, -0.99305707475593197, 
-0.83491272721058696, 0.35340721498104166, -0.59427564893211349, -0.92108456165670183], [0.64026324991699002, 
-0.55911940567267715, 0.68485406136486726, -0.9986277718812977, 0.50067323602931335, -0.54765034183971184]]], 
[[[-0.68057062206994257, 0.18789019883468483, 0.49734828895211836, -0.1732510645817964, -0.39247932240672601, 
-0.4339065271996414], [-0.75418301262293985, -0.1424083683138635, 0.30876505271130217, 0.22306963951286929, 
-0.065611760603126124, -0.43310391309353058], [0.35269558770995246, 0.57861471019224786, -0.25739749090392516, 
0.073732381136675906, 0.38332099586879864, 0.51607524368785751], [-0.56873358202859858, -0.25518714635844653, 
-0.80098526077251919, -0.84958859038125789, 0.0068421945172432164, 0.98097200683205465], [-0.9378248712256696, 
0.56399795618977877, -0.50964062478356231, 0.91470945785762603, -0.9336540935000377, -0.026203166527157817]]], 
[[[-0.77023203872533563, 0.95519653782503489, -0.71652852824353364, 0.68878426171931384, 0.36693735356123236, 
-0.35020921405206207], [-0.26708357766673374, -0.61407526733830498, -0.097022720150582664, 0.35585503729268053, 
0.4890108950144032, 0.099405316214215222], [-0.019701625438284021, -0.78570225163175533, -0.52620904212567599, 
0.85615630808977827, 0.50512938458027601, 0.2891979005442562], [0.16775328151212632, 0.28256662254260023, 
-0.099660432515685571, 0.83265994982225955, -0.76557851180892755, 0.87327637878785613], [-0.74584746419995551, 
0.39712774914523874, 0.82217390401956369, -0.4519439895184556, -0.9570135722026647, 0.82407849176617254]]]])
        ref=clip(xx,0.416630,0.717385)
        res=Evaluator(y)(x=xx)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inner_Symbol_rank0(self):
        shape=()
        x,y=symbols('x,y', shape=shape)
        z=inner(x,y)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(-0.852329118658)
        yy=numpy.array(-0.75330429454)
        ref=inner(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inner_Symbol_rank1(self):
        shape=(2,)
        x,y=symbols('x,y', shape=shape)
        z=inner(x,y)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.1265795647780048, 0.75223673325034079])
        yy=numpy.array([0.10826046956183677, 0.89068297917980055])
        ref=inner(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inner_Symbol_rank2(self):
        shape=(5, 6)
        x,y=symbols('x,y', shape=shape)
        z=inner(x,y)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.91220996643602215, 0.37896801484913412, -0.089973363308325505, 0.91119991020561253, 
0.5134843437528136, -0.35468571176744579], [-0.52927286272582852, 0.92462630463429263, 0.36116047879346969, 
-0.038789120158563506, 0.9385818952858449, -0.91956499637363565], [-0.83097182411600623, -0.084758465892522228, 
0.099984469999697456, -0.12511146004857454, 0.42543721309954741, 0.56563214634401149], [0.37938960146329492, 
0.50271635814327831, -0.46318637177178812, -0.030208028615403171, 0.44829531719294335, 0.66341773718893893], 
[-0.041879882008717617, 0.97827558280950155, 0.46488294246963457, -0.27884998126557314, -0.7833585481279195, 
-0.87236385576036279]])
        yy=numpy.array([[-0.99438087282661303, -0.17351736200107482, -0.19704776174380845, -0.48885499490821882, 
-0.37188108803653863, -0.46561278953015739], [0.95095894429384487, 0.99423070661380586, 0.22785252729899175, 
0.54720858072644107, 0.5600529448128726, 0.72887165009776433], [-0.84644603941953744, -0.70952293074156469, 
-0.51909768326363825, -0.4071493116476419, -0.076074747880092719, -0.0069993656333380283], [-0.92195684101044284, 
-0.43893666819538679, -0.14718678892445891, 0.97181105037614302, 0.38047525958736572, -0.23537304512696444], 
[-0.6478142880883575, 0.98423814669730603, -0.049688529800366421, -0.058073047334822325, -0.15697302754138009, 
0.67486666943196827]])
        ref=inner(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inner_Symbol_rank3(self):
        shape=(2, 4, 1)
        x,y=symbols('x,y', shape=shape)
        z=inner(x,y)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.94689258518000896], [0.31785014911567089], [0.29954771093641064], [0.75836089990693067]], 
[[-0.18521659202147234], [-0.47845834722258984], [-0.16842080933931092], [0.79254630926389047]]])
        yy=numpy.array([[[0.87416078202044001], [0.98551949329504307], [-0.5019329098781764], [0.73665025975506926]], 
[[0.80203205501476527], [0.78639229317250181], [0.93691082412076909], [-0.3685014711410024]]])
        ref=inner(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_inner_Symbol_rank4(self):
        shape=(1, 2, 3, 2)
        x,y=symbols('x,y', shape=shape)
        z=inner(x,y)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.62947164283297807, 0.75941647180328542], [-0.20243781205072109, 0.92575559823090825], 
[-0.97325873393370421, 0.032871008352538533]], [[-0.90012690259220252, 0.00040530353428680499], [0.41505536535480991, 
0.65303836313609609], [0.83283558176960271, 0.42766968821964735]]]])
        yy=numpy.array([[[[0.18116447096700039, 0.70800893900326045], [0.69671559357243362, -0.41363712349687698], 
[0.1059476699590145, -0.41005686467010483]], [[0.3315678682805272, -0.76642898464430465], [-0.57474492778556496, 
0.16483129612377412], [-0.092268687113073344, 0.38533656246054337]]]])
        ref=inner(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank0_and_0(self):
        sh0=()
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.954179720561)
        yy=numpy.array(-0.691124889776)
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank0_and_1(self):
        sh0=()
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.135269138248)
        yy=numpy.array([0.078688469309617926, 0.49712759239387694])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank0_and_2(self):
        sh0=()
        sh1=(3, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(-0.722158218233)
        yy=numpy.array([[0.79117756039005926], [-0.74407099893658901], [0.17775692726093917]])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank0_and_3(self):
        sh0=()
        sh1=(2, 3, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.571280712918)
        yy=numpy.array([[[0.83441927204924715, -0.61131119436768411, 0.023673074477871925, -0.071182188023339643, 
-0.80747978649787844, -0.16478312960542074], [0.03563855946192751, 0.76439565816666755, -0.87284560984901094, 
0.01153398733591926, 0.20599204999412635, 0.68479369682226898], [-0.59103898481381045, -0.50079786036707064, 
0.25727447794288238, -0.4407448383290451, -0.14882486252244731, -0.79410939681176473]], [[0.75015040744067152, 
0.68995158924872535, -0.70311571412237583, -0.39559342629424998, 0.59033808593169379, -0.65487801350644426], 
[0.16822159240255474, 0.71547437026549199, -0.93933274504767916, 0.63810701323183805, -0.15595946179557929, 
0.48338013802290347], [0.20904576590804003, -0.3919276928346711, 0.96954375933988435, 0.8657620242058679, 0.6677153957010058, 
-0.042944686528293419]]])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank0_and_4(self):
        sh0=()
        sh1=(2, 5, 1, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array(0.130636252063)
        yy=numpy.array([[[[-0.46747255901922169, 0.53267213812794645]], [[-0.54732306023140342, 0.89927251629297356]], 
[[-0.21453656820815326, -0.4635356511906441]], [[-0.43320010646388485, 0.15538283418633148]], [[-0.74687061990868697, 
0.12169801200063013]]], [[[0.49896048718913733, 0.35203724327111541]], [[-0.26121157159255848, -0.05101476527294202]], 
[[-0.0028002673287026436, 0.30072582217424793]], [[-0.18758496946411718, -0.84805805419858116]], [[0.74429252191773498, 
-0.23758888889914243]]]])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank1_and_0(self):
        sh0=(6,)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.62264594143280472, 0.184592354670706, 0.95730135123808302, 0.1999538322218597, -0.75777787624367221, 
-0.60801280559203508])
        yy=numpy.array(-0.879025424384)
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank1_and_1(self):
        sh0=(4,)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.065718718247615771, 0.19629464767007687, 0.16563222983197523, -0.01618155117959108])
        yy=numpy.array([-0.70723511901000613, -0.30718456079414413, 0.12683857645449104, 0.25947627326733347, 
0.46122855158647846])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank1_and_2(self):
        sh0=(3,)
        sh1=(6, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([0.98546465194904886, 0.45639695296483951, 0.76744151615689149])
        yy=numpy.array([[0.97476369902957916, -0.5104537951467103], [0.71268517900244088, 0.13115360540750798], 
[-0.77959316648888999, -0.49758351302687687], [-0.29936902838767065, 0.6217732986455371], [-0.30945663995737238, 
-0.097062795038203697], [0.73676965130574046, -0.45219581358621852]])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank1_and_3(self):
        sh0=(3,)
        sh1=(1, 5, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([-0.39617099281921919, 0.028680449611468317, -0.35640802477814515])
        yy=numpy.array([[[0.5822446449752563, -0.60156551577693085], [0.0244300035516265, -0.57977027033412831], 
[0.59676911954205125, -0.69889447906574342], [-0.78005016781235459, -0.84074807454929168], [0.66907701075454251, 
0.37238632756016643]]])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank2_and_0(self):
        sh0=(2, 1)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.45897363190948437], [0.93913267847867887]])
        yy=numpy.array(0.91171413566)
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank2_and_1(self):
        sh0=(3, 1)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.031660169289180029], [0.84184991662149922], [0.022967879686092596]])
        yy=numpy.array([0.90635227673812357, -0.32567028776445128, -0.14548259773186611, 0.60586360359620661])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank2_and_2(self):
        sh0=(2, 2)
        sh1=(3, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.35129394890069299, 0.33119040938878674], [-0.98440102652658146, -0.49331122661663995]])
        yy=numpy.array([[0.039296556849623965], [0.47335373668524849], [-0.3727249413788456]])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank3_and_0(self):
        sh0=(5, 6, 5)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.77077302181457963, 0.30686232378832834, -0.75516074329220584, -0.14206129969563808, 
-0.11911792442284508], [0.9823901893720226, 0.66654220497886696, 0.8900709396801898, 0.48465790946427778, 
-0.12541211472715252], [-0.55035003106504643, 0.061522863301689767, -0.38543436769031159, 0.62374216844839103, 
0.24178499517091234], [-0.4398658946181222, 0.14878063197400859, -0.15726290593146675, 0.17874334977002815, 
-0.14555854387639022], [0.27500860411672234, -0.34555861663804732, -0.64990918644041185, -0.027649512126243891, 
-0.034182765223071288], [-0.70583992967255815, 0.27383541448433135, -0.17113098984866881, 0.52296730980967499, 
-0.4686968432683567]], [[0.30718965836360312, 0.35268731547046483, 0.052215953635355916, -0.91698621743681241, 
-0.61873415532911857], [-0.81799917307940095, 0.8651865013273734, 0.28597512357930621, -0.9282905266046102, 
0.72788604279080538], [0.39300722885613482, 0.89202238150531721, 0.39834824339413077, 0.19870130692994126, 
-0.82167017598112158], [0.64480250190214328, -0.542082664539423, 0.24832097854488366, 0.26309628572205934, 
0.89947806273934772], [-0.36856089725355656, -0.24824185314073866, 0.90131772430325485, 0.74205096292895356, 
-0.47005218351257683], [0.037054079066698442, -0.26671623184528026, -0.20497626960584969, -0.61645920276420907, 
-0.080632836226234961]], [[-0.14275724272771084, 0.22125605561069883, 0.46817480580298931, -0.54623118391431591, 
-0.26788857507537633], [0.61287975128044248, 0.86552517131173401, 0.48802144852244345, 0.54579764334227865, 
0.1726152228975657], [0.57218136874636905, -0.29325463518138983, -0.63218106644180394, 0.029120167263931629, 
0.38156820931843072], [0.013908982034637196, 0.66751973941750564, -0.55182774249381006, -0.8879163150039342, 
0.12045691134986769], [0.78732678015886948, 0.018339501670040725, 0.1311017902975693, -0.28528083301467433, 
-0.79952627141069921], [0.81603477298714555, 0.66465911725495785, -0.2181840149905645, -0.69265205984319311, 
-0.31952276668432678]], [[-0.80065510190298128, 0.86772838624419601, 0.29073517719613862, -0.16972208480859474, 
0.93593296506108925], [-0.56380314581796465, 0.45814016958259907, -0.8204687092420575, 0.57035686639841643, 
0.078155606530178057], [-0.03475511577536361, 0.32236340523892659, 0.34980472152441622, -0.52293095562738778, 
0.80075750251225242], [0.92782124550347289, -0.76202994608599339, 0.97114259462719099, 0.80811666483084843, 
0.52826282393733126], [-0.23374984700406265, -0.87097085706461019, -0.35363984277933747, -0.28077209726814711, 
-0.49007147404623264], [0.64066260367340688, 0.27085331122836842, -0.61288750789206348, 0.12267663876018275, 
-0.89012407642793967]], [[0.52548297884543138, 0.011189163583985362, -0.17015452153586952, -0.395753578579221, 
0.069726247238518946], [-0.19404739756413347, 0.96421074588268785, -0.44030492004008326, 0.42230209613627001, 
0.70935278168778804], [0.26780691415764246, -0.07647958449709491, -0.16813906348388974, -0.85460125888080918, 
0.78380154855255757], [-0.72800309045599443, 0.59440147476426386, -0.74850278912726531, 0.86833816888176307, 
0.33455417972686519], [-0.36591220384035794, -0.4196556255918602, 0.57500070375192203, -0.60182084297096883, 
0.13202895875870824], [-0.75525375369636727, 0.54642966256459125, -0.45583824688237073, -0.32933076620108537, 
-0.65489706515779589]]])
        yy=numpy.array(0.692645767992)
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank3_and_1(self):
        sh0=(4, 2, 4)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[0.5477636359057374, 0.27826501728853592, 0.83813087281954135, 0.54694225263845975], 
[-0.15440947135451966, -0.62110469614458963, 0.8200676137364391, -0.11582050096240293]], [[-0.060323156765071273, 
0.043770906193430381, 0.61829918932111405, -0.81490008084455434], [-0.73398295474895758, -0.40135360793182673, 
-0.56807033732145107, 0.22191679080525728]], [[0.46640023059569535, 0.22415351441311282, 0.34099491667795889, 
-0.80832968226935353], [0.81457958139709641, 0.36885203160918478, -0.57534770454888906, -0.89952535128324573]], 
[[-0.58002933084183739, -0.29833523735285206, -0.17181067100530867, -0.76993002746168471], [-0.22246731837291933, 
0.81721594538042153, 0.55309488824772135, 0.48984418481192593]]])
        yy=numpy.array([-0.33293712690311228, 0.70546496700914396, 0.6936664553668126, -0.60613970725489463])
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_outer_Symbol_rank4_and_0(self):
        sh0=(4, 1, 5, 6)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=outer(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.72104363070851907, -0.12679311039170416, -0.13514553676429042, 0.75399049489041858, 
0.11296171688840984, -0.75382306579734681], [0.72304239945221993, 0.33010147878833762, -0.68984806623008632, 
0.77898922837073692, 0.047920100680009403, 0.4327358548103144], [-0.051082066052373865, 0.90385190071399935, 
0.67460983708445732, 0.21469617076304837, -0.75344247503198436, -0.27186407673754975], [-0.05351294853556432, 
0.91693539432831361, -0.5829548315190054, -0.53895820936072303, 0.64019003543905728, -0.9685155775082015], 
[-0.40892363497697448, -0.055370472494696621, -0.16573376533872342, 0.695319402736986, -0.56170828209580126, 
0.35478308883812826]]], [[[-0.49210092429107122, -0.7279970160262772, -0.54092707950737862, 0.56841850622721668, 
0.29193333276238986, -0.0039960069015902988], [0.13035830134403459, 0.62350501329187202, -0.93635529393047756, 
-0.54858131590154047, 0.90191857875561499, -0.218270794794998], [0.57159465747079707, 0.60185462803130019, 0.51478436568769137, 
-0.54104413025437581, -0.43262406570476042, -0.26863797840654402], [0.076348593458945402, 0.82540043144546349, 
0.81816513411045633, 0.10805601237592022, -0.60936696841891935, 0.49230986613359651], [0.030684777079482028, 
-0.60522482427969027, -0.1672966600747221, 0.010019545306680255, -0.45726590934129985, 0.024610570108272389]]], 
[[[-0.40089126480586867, 0.80869681077686484, -0.45101297684195574, -0.68561575884680104, 0.33824069063382978, 
0.56848316048248448], [-0.72088699530784783, 0.6126508859236548, -0.60289270905326786, 0.15172360570613264, 
0.38156358179571326, -0.24805171429966277], [0.02233767458552971, -0.56842341984700751, -0.9370317323734878, 
0.23506261137206463, 0.75613873506838014, -0.49729822950880531], [0.83449152637320689, -0.38716412772708941, 
-0.62655914904595522, -0.46151883922325609, 0.23552262204669705, -0.44852310060824907], [0.26228184332475468, 
0.0015315510160027745, 0.64603350804234894, 0.74435017365045408, -0.69827062324696731, -0.73004526540319525]]], 
[[[0.38395138006768259, 0.10157606008122455, 0.42082359305802353, -0.08343698635134178, 0.5594636045977186, 
0.91111905214967059], [-0.726543931059912, 0.62117018611853592, -0.39978910956582525, -0.6429914842508917, 
-0.62035101115430646, -0.10647275017627011], [0.89193900351663125, 0.33861165551225381, 0.41741913241046835, 
0.52924709946967585, -0.17789664879978617, -0.11922292353536834], [-0.79815516153140709, -0.95141859053081723, 
0.070907253988844454, -0.99795676429680125, 0.81818097969214088, -0.051196704081730271], [0.9768334745893219, 
0.21712171250579737, -0.97022812390194213, -0.36053467834420894, -0.8219232975167341, -0.81539574278229154]]]])
        yy=numpy.array(0.396112416081)
        ref=outer(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank0_and_0(self):
        sh0=()
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array(0.819293895897)
        yy=numpy.array(-0.358031229894)
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank1_and_0(self):
        sh0=(6,)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([0.25962722419618234, -0.81227210208559564, -0.57891734066591694, 0.94496263003167291, 
0.11476687607247693, 0.32666037552809279])
        yy=numpy.array(0.60091711432)
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank1_and_1(self):
        sh0=(4,)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([-0.69681599322282128, -0.17191531554469885, -0.038854213490760792, 0.74088030549064587])
        yy=numpy.array([-0.65525949795188287, -0.70129548673768705, 0.75742395331142798, 0.99872283197931511])
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank2_and_0(self):
        sh0=(3, 1)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([[0.84200065243387945], [0.83994291696353129], [-0.8836936913297111]])
        yy=numpy.array(-0.566148184926)
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank2_and_2(self):
        sh0=(1, 4)
        sh1=(1, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([[0.69030647300386838, 0.0019686270679213358, -0.94305407861405177, 0.86949070673348072]])
        yy=numpy.array([[-0.14762563746049073, -0.8644653944934293, 0.27100933715587661, 0.73630298878443479]])
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank3_and_0(self):
        sh0=(4, 6, 4)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([[[-0.75613985904439729, -0.45694961282061852, 0.59758151073920551, 0.9272853357202111], 
[0.39015189183179833, -0.17708333397411913, 0.14575808628843956, 0.6941092103873987], [0.57443796231449462, 
-0.33881983252560333, 0.67567100713425776, 0.86300458741001473], [-0.65633399194294295, -0.29790140174538604, 
0.61234500894708321, -0.81135670066497267], [-0.61460415273523017, -0.8317100411175693, -0.084416948078252396, 
-0.78723276656711061], [0.82092708582961182, 0.16802532036903539, 0.51491867879132536, -0.67234626368885397]], 
[[-0.36192716513447221, 0.78472745453428683, 0.18047748367119754, 0.31108699395648975], [0.95359431919033444, 
0.16405727310695073, -0.050365423199315362, 0.33856077723147804], [0.086317234083251426, -0.18523145618025705, 
-0.055398988001626792, -0.74173637133532999], [0.98021326957799171, 0.87125448638310266, 0.58528207165851853, 
-0.07699531346779187], [-0.76152478090338183, -0.60119243609841422, -0.60918329463091525, -0.69823846769716558], 
[-0.13813970934072151, 0.17823095597244576, -0.87174483384837864, -0.94257950089879472]], [[-0.88069437387170213, 
-0.38369774129541812, 0.40916071462188786, -0.88983710140103667], [0.46888812143472158, -0.80205004446885142, 
0.042524874740612617, 0.56750738950131074], [0.97562603365788081, -0.7474357240947862, 0.26013836679165525, 
0.34741180248288051], [-0.48005616865242806, 0.025815216568898958, 0.57982619557200321, -0.99847461827862727], 
[-0.24443062401676374, 0.21883034231272536, 0.49320914434472929, -0.73406221685850293], [0.9982487976957064, 
-0.4006637041168819, 0.32089759130384832, -0.4601951041226453]], [[-0.98405300079943303, -0.43971116475012795, 
-0.32885991840900686, -0.65311116950342085], [-0.14996646922958945, 0.44160286963818018, 0.072457623982987496, 
0.84902867420503636], [0.92158276939603345, 0.90533383776294407, -0.57030347111498902, -0.87389182279609989], 
[0.37511800712071408, -0.087311879001895631, -0.92162601985681114, -0.41334503539288536], [0.57507965286900675, 
-0.67554750531833774, -0.066211926208951555, -0.76037605091742999], [0.40550054366727939, -0.38864784366654725, 
0.56366713933243418, -0.20367829388715175]]])
        yy=numpy.array(0.285106995443)
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank3_and_3(self):
        sh0=(3, 4, 3)
        sh1=(3, 4, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([[[-0.81698446907829836, -0.96140942967271692, 0.80529206656669605], [0.11573524128339918, 
0.17505176178065729, -0.02848743318770719], [0.05236681741283955, 0.47358470161422339, -0.45028412109546867], 
[0.24655977229532922, -0.97433531842160903, 0.2560468785505523]], [[-0.04126286920284028, -0.12506740415463535, 
0.46452345347110646], [0.75063292653462765, -0.9011117549143326, -0.83839866944824748], [-0.80584680348116078, 
0.78231458540020871, -0.33897475611783445], [-0.20834845613372743, -0.64064201073241422, -0.22900963106384964]], 
[[-0.91980101989120833, 0.13861969038193789, -0.22031690044051899], [0.2191536828870162, -0.34239432235861078, 
-0.48326914510547359], [0.68240681992122076, -0.12044931881036036, -0.74883525972258269], [-0.65067740773831462, 
0.016534372553552501, 0.72765953280738804]]])
        yy=numpy.array([[[0.0650869795657949, -0.13367589560824022, -0.01275841925489396], [0.2211248667525727, 
-0.083835740743561482, 0.78819745031660715], [-0.87082127227196526, -0.59557223550493021, -0.31950016526315439], 
[-0.17394834818796623, -0.38436793752850895, 0.21174810377103559]], [[0.58586631060543426, 0.3347165872165101, 
0.83342968992816924], [0.70021876008732331, -0.063789528684661256, -0.59103831311980271], [-0.4983506708850054, 
0.6275310961366094, 0.97607515569651437], [-0.60732533112338261, -0.10632584858623617, 0.38469364394519601]], 
[[0.67612450146593206, 0.50561363417212046, 0.98811584482196646], [0.38053521637021648, 0.2583664337242928, 
0.82411321159156259], [-0.51577478887162909, -0.5327079290096961, -0.92700145971283265], [0.27940355473740452, 
0.81697843030836004, -0.19017530224258095]]])
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank4_and_0(self):
        sh0=(4, 1, 2, 3)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([[[[0.58065007920829781, 0.90489647821744112, -0.52291599368329433], [0.71603069994853219, 
0.26038331953585825, -0.18864988685919371]]], [[[-0.22774453233675929, 0.90672299639197074, -0.65801248074525009], 
[0.38173887846818433, -0.56800863449295247, 0.36273511634727384]]], [[[-0.9652193469221253, 0.78944769440121987, 
0.69725458263148887], [0.10733775214663011, -0.83376484136984508, 0.81397738367756856]]], [[[-0.70856963996101263, 
0.065661698104239052, 0.94692082686508261], [-0.30403376276017546, -0.27380438611834657, -0.57470557934412403]]]])
        yy=numpy.array(-0.531662852947)
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_maximum_Symbol_rank4_and_4(self):
        sh0=(1, 1, 5, 1)
        sh1=(1, 1, 5, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=maximum(x,y)
        
        xx=numpy.array([[[[0.77727138124075212], [-0.11393982329540875], [0.46884356664211135], [-0.57117755887951205], 
[-0.14368514618916794]]]])
        yy=numpy.array([[[[-0.36402631929467244], [0.45854783396939269], [0.54201860532667379], [0.77178247704416969], 
[0.6744593027674135]]]])
        ref=maximum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank0_and_0(self):
        sh0=()
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array(0.0634627238722)
        yy=numpy.array(0.7448921557)
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank1_and_0(self):
        sh0=(5,)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([-0.50877822773773396, -0.69243732402827907, -0.81574496407265018, 0.62088483666168304, 
-0.5935309641956017])
        yy=numpy.array(0.717923977964)
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank1_and_1(self):
        sh0=(3,)
        sh1=(3,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([-0.015623993505315958, -0.5118446986466092, -0.84356941714476852])
        yy=numpy.array([0.33278800494510197, -0.24951297138457229, 0.59087969556348763])
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank2_and_0(self):
        sh0=(4, 5)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([[-0.85198421021137527, -0.31654785363904692, -0.37573509700150898, 0.066533266298060889, 
0.40507011429602602], [0.70974315684225209, -0.65458300910524247, 0.52203634364949214, -0.3716432376362464, 
0.33806889038873145], [-0.17132882415192863, -0.42599674764474527, -0.58394265286578095, 0.74258285699551529, 
0.59367112993006432], [-0.22447160209992889, 0.32099346716116761, 0.23623142794413643, 0.50563641507287382, 
-0.96811668829946895]])
        yy=numpy.array(-0.527625495982)
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank2_and_2(self):
        sh0=(5, 6)
        sh1=(5, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([[-0.80634268832630118, -0.93159589997123704, -0.12179227983483898, -0.30383512396846757, 
0.1484519541741518, 0.13570090707446592], [-0.22341025021031036, -0.079920813675078861, 0.7174485040704055, 
0.82711948662710522, 0.63000619172464623, 0.27558058277981434], [0.35247514993911766, 0.93245790030009168, 
-0.62645685454024358, -0.44170125128821858, -0.3054634802481897, -0.68030914007854681], [-0.0045682537039382076, 
0.60145390394287856, -0.86538245954356818, 0.36319807156916961, 0.47949747744222471, 0.042777570976650692], 
[0.98300181711995815, -0.46187490862963587, 0.3906191318537795, 0.84754594534753336, -0.69164791075478727, 
-0.55322595817938169]])
        yy=numpy.array([[0.57828665854117589, -0.97964100464800064, -0.91943610465726588, -0.86177092350687423, 
-0.29777338641866136, 0.40133673287954919], [0.50600604425468765, 0.092842651701914969, 0.9666473770660422, 
0.17886462586000107, -0.60709506340417119, 0.040865070411784776], [0.049339194681566667, -0.68631947346020961, 
0.57272321691386852, 0.58767851346059685, -0.26012781606666557, -0.20669983465654207], [-0.97702628533528024, 
-0.11419742367508068, -0.29265156692300076, 0.082958759304118113, -0.10109646693680552, -0.72951391985780356], 
[0.37543670120908157, 0.76463465018530385, -0.33364960414680955, 0.47294513529970361, 0.36412375264529806, 
-0.45609228775500443]])
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank3_and_0(self):
        sh0=(2, 5, 4)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([[[-0.36010235159447013, 0.88279917767986871, -0.48708406645564239, -0.74149744390318495], 
[0.2500191108753016, 0.46498659654288321, -0.57125432084076966, -0.24500725999920547], [0.79367870539555296, 
-0.34874837808001247, 0.18451695226447939, 0.0064720855621340778], [0.75583876184915155, -0.09636856663758353, 
-0.36576176078628664, -0.014147677093818034], [0.0041224226289937604, 0.2066632586023307, 0.5027559882225856, 
0.94069195783362702]], [[-0.67509144110455788, -0.27577028865967246, -0.082504851245257749, 0.63767197489814564], 
[0.33582360742932593, -0.42488358868337817, 0.084203133002601183, 0.89788828799455089], [-0.6153128562056347, 
-0.95038151000357352, 0.52976628308414653, 0.053299201714020139], [0.33748099725977521, 0.44293942638293538, 
-0.18757306870211865, -0.80030987965378397], [-0.54940587813697417, -0.091392004011353389, -0.24416499459514962, 
0.66650544255988575]]])
        yy=numpy.array(-0.92357795254)
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank3_and_3(self):
        sh0=(5, 5, 1)
        sh1=(5, 5, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([[[0.86865108166840876], [-0.43302835396569606], [0.25138827731586622], [-0.37410370374144097], 
[0.96860479837446412]], [[0.29006476476601328], [-0.50563287562180159], [-0.7533990173466929], [-0.47236777081889114], 
[-0.028512145132094657]], [[-0.51312248443832731], [0.38052601420283549], [-0.36649907539251236], [-0.64149763713082697], 
[-0.19376085401085819]], [[-0.78317651991917048], [0.50130671003203986], [-0.63906162536276678], [0.40839024173493255], 
[-0.49918007819432209]], [[0.71318320148218839], [0.11373683147005553], [0.49495843647723192], [0.55785064046315602], 
[-0.36290021293056784]]])
        yy=numpy.array([[[-0.87621735341711515], [0.76733662786479262], [-0.60315200101029465], [0.29838818261584366], 
[-0.70476885105464859]], [[-0.79353927215817177], [0.4769716693905659], [-0.12360655150139088], [-0.10752462910828209], 
[-0.78526566024932909]], [[-0.87308885906688949], [-0.95747637278293363], [0.1309291413810052], [0.98813243191066347], 
[0.54718130394348696]], [[0.44450986408303272], [0.047847060742982617], [-0.29393165387146758], [-0.53491237816068637], 
[0.031635597117939618]], [[-0.81142405780099236], [0.36235374367845297], [-0.074789268428598676], [-0.65962740207016735], 
[-0.59008498340170901]]])
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank4_and_0(self):
        sh0=(5, 1, 5, 1)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([[[[-0.73666294414097777], [-0.24513996507526126], [-0.77870425615000238], [-0.65422587826124379], 
[0.50862799377216783]]], [[[-0.36092235772249981], [-0.95219388965305951], [0.62572135796157924], [-0.79044401699074207], 
[0.2525955544626084]]], [[[0.88142807726350525], [0.86822248044165229], [0.5631146627342023], [0.40715178729955004], 
[0.1845232982059275]]], [[[0.17823180061206778], [-0.12818951140999446], [-0.37675467453210643], [0.87915934255358597], 
[0.51865042497908309]]], [[[0.93071878225890425], [0.068397126927967733], [0.10652993800353894], [-0.77873811134813486], 
[-0.5683620133607441]]]])
        yy=numpy.array(-0.843028869929)
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_minimum_Symbol_rank4_and_4(self):
        sh0=(3, 2, 3, 2)
        sh1=(3, 2, 3, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=minimum(x,y)
        
        xx=numpy.array([[[[-0.49439119504994422, -0.19565581866553083], [-0.58246140960676418, -0.47308775065754261], 
[-0.088238471234961535, 0.52952070613166735]], [[0.47047457718234842, 0.15630525152462038], [0.098051190203235272, 
-0.55753823428015936], [0.67987996807486906, -0.10438844508995127]]], [[[-0.18204411345282723, 0.28967933442163041], 
[0.73668495396513101, 0.92593262416194388], [-0.4636590526293356, 0.571745376575852]], [[-0.60167735049715843, 
-0.78042614142487299], [-0.00013111916540120561, 0.2911405021058413], [0.56811416028395834, 0.21671390107366206]]], 
[[[-0.35340797537094715, 0.60492901106382901], [-0.97082208833041528, 0.92531447089683505], [-0.59839235258109014, 
-0.3238990722131474]], [[-0.95219045077424602, 0.97789198175649839], [0.91556425827866117, 0.9096369832167579], 
[-0.83190351697038989, -0.059148212083001805]]]])
        yy=numpy.array([[[[-0.24318643922310423, -0.096795767170201774], [-0.14525946299051129, -0.74431486769064348], 
[-0.2440379883748045, 0.33339396513978858]], [[-0.84172107243916061, 0.34807415443425338], [-0.6246770074647241, 
-0.42748838916007759], [-0.74863494665128427, 0.87369455924347084]]], [[[0.0084536490983400103, -0.38664933642289845], 
[0.53150178574099427, 0.93433333866847113], [0.3025016795576434, 0.94312551387128996]], [[-0.0080506012739409716, 
0.69537134680709234], [0.60861004073447322, 0.2409775990336116], [0.98869596425749018, 0.22505056717158611]]], 
[[[0.99361242565403685, -0.2569287417179833], [-0.83781677153729484, -0.25739976723544777], [0.34416990456773222, 
0.56990607220945932]], [[-0.49595772815773875, 0.97994352354885583], [0.51673253642323114, -0.41189373318165101], 
[-0.79927066132000024, -0.57211729405384637]]]])
        ref=minimum(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_matrix_mult_Symbol_rank2_and_1(self):
        sh0=(3, 5)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=matrix_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.027309770178801429, 0.61658408388310382, -0.53548124433755451, 0.40388288090554658, 
-0.16244542719320898], [0.35070395151135014, 0.32390481541541183, -0.57154006152253212, -0.66261977637815517, 
-0.16877504462831072], [0.33364659059618118, 0.032700423977650495, 0.68426318905733874, -0.74705264103483549, 
0.80953782754352654]])
        yy=numpy.array([-0.36729936846046862, 0.59223902962896835, -0.66387057292641938, -0.81500980280129753, 
0.64207694556545314])
        ref=matrix_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_matrix_mult_Symbol_rank2_and_2(self):
        sh0=(4, 3)
        sh1=(3, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=matrix_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.95767801685805476, -0.79185926721440603, -0.83328784047810123], [-0.69812473023950838, 
0.81556162750151517, 0.91782846766007942], [-0.23200262772255664, 0.71838434816218388, 0.21337626544920862], 
[-0.27375549987867154, -0.61655340632028111, -0.43047574940839595]])
        yy=numpy.array([[-0.11422130000282604, 0.12055840973397869], [0.55639605130506964, -0.063036426157317971], 
[0.6329327976840855, -0.1158568472497572]])
        ref=matrix_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_matrix_transposed_mult_Symbol_rank2_and_1(self):
        sh0=(5, 5)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=matrix_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.52417079378462561, -0.90097895927789984, 0.45926552997103043, 0.29900144738399237, 
-0.39395086909688648], [0.39325511344753861, -0.78247368976590037, 0.52388968764921051, 0.22422917428767697, 
-0.039040699419247815], [0.12383111962733429, -0.7738202589295673, 0.13073129341297651, 0.89112803194033097, 
0.56501864055352624], [-0.58340528383579837, -0.64556523301828572, 0.72604270374218971, -0.53757952872478243, 
0.19923822514126144], [-0.19672552565749779, -0.99987460446294785, 0.55077940925838398, -0.73877635319753621, 
-0.26273117924469469]])
        yy=numpy.array([-0.90879739986345309, 0.067699656011914966, -0.75576603890374683, -0.92643727930977149, 
0.5028910621493019])
        ref=matrix_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_matrix_transposed_mult_Symbol_rank2_and_2(self):
        sh0=(4, 4)
        sh1=(2, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=matrix_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.91520984507452141, 0.96543478170531349, -0.88395141253816667, 0.59137634223596591], 
[-0.81781170486496113, -0.84138831886941334, 0.5199846052378323, 0.50777229879698882], [0.36151295588779031, 
0.98141100739403919, 0.94786030608232563, 0.64329087304791344], [0.83214878947505522, 0.18060348967021511, 0.12886302317778209, 
-0.32995792834745363]])
        yy=numpy.array([[-0.42540245142864164, 0.56404942239870448, 0.80921803272305071, 0.73760533584805943], 
[0.53304997707226764, 0.8305619245863276, 0.76061368985056221, -0.41620879282834311]])
        ref=matrix_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_matrix_mult_Symbol_rank2_and_1(self):
        sh0=(5, 3)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_matrix_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.99084536582593641, 0.36958900056934629, 0.88637598981933197], [-0.82895839831239715, 
0.11605636302306377, 0.2412987914145599], [0.053840186318937056, -0.37304648797780526, -0.74618832703811022], 
[-0.31712977205689352, -0.29441290133517994, -0.071924636978345502], [-0.91342156159861232, 0.27586101266744278, 
-0.93720712189502131]])
        yy=numpy.array([0.76027834581422948, 0.53509561900544256, 0.74441296299399773, -0.90433440075682636, 
0.16586300420919686])
        ref=transposed_matrix_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_matrix_mult_Symbol_rank2_and_2(self):
        sh0=(2, 2)
        sh1=(2, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_matrix_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.33008357180023795, -0.42453372352841456], [-0.034831089371839763, 0.49665812953924626]])
        yy=numpy.array([[-0.31139967816552527, -0.90527240888898897], [0.39305995528211612, 0.83856119078409286]])
        ref=transposed_matrix_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_mult_Symbol_rank2_and_1(self):
        sh0=(5, 1)
        sh1=(1,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.12065738514159841], [0.33982939453464001], [-0.051173585932761778], [-0.13921676487823786], 
[0.77757979127684496]])
        yy=numpy.array([0.90300944980562603])
        ref=tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_mult_Symbol_rank2_and_2(self):
        sh0=(1, 2)
        sh1=(2, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.23559282019710692, 0.97143924717113217]])
        yy=numpy.array([[-0.40318502937291045, 0.96625979962108488, -0.44035326091920224, -0.53307174397533874, 
-0.37036733769719121], [0.55704231394283288, 0.095458279362755416, 0.21694469103058389, 0.56241669200043454, 
-0.31597519017392939]])
        ref=tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_mult_Symbol_rank4_and_2(self):
        sh0=(6, 4, 3, 1)
        sh1=(3, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.53044506047610795], [0.82699279648504098], [-0.1160952959531254]], [[0.50587891024827281], 
[-0.6805339205056844], [0.97310380806405394]], [[0.56211556040809785], [-0.76882286835501557], [0.28029111288323993]], 
[[-0.061099498869066737], [-0.17236179614419589], [-0.83590874857353947]]], [[[0.55722541428992511], [0.90469088373304918], 
[-0.89474648638572063]], [[-0.0491539137613064], [-0.7965523287767271], [-0.29018798934374401]], [[-0.53866845069614722], 
[-0.45643613076902545], [0.40757170276065358]], [[-0.30521840888689855], [-0.0026132608685125458], [-0.95967006262189014]]], 
[[[0.47898667124758165], [0.89300933316897568], [-0.048776838239707354]], [[0.81670162800498636], [0.68157136250813277], 
[-0.31899933329442054]], [[-0.86125897155176867], [-0.87275139495721765], [0.86645055630594792]], [[-0.20233344615491133], 
[0.54511181802892295], [-0.26850013173758436]]], [[[-0.97857077432718986], [0.73619160594541033], [0.03663052251114518]], 
[[0.071737946108099138], [0.6175867629469296], [0.70846998835956487]], [[-0.35871331357324032], [-0.0065250718538631958], 
[0.61374165442154238]], [[-0.99669765560872015], [0.13203926575481417], [-0.5246809230306968]]], [[[-0.46301798249596682], 
[-0.85903476269986379], [-0.86618449026945399]], [[0.71927998930432935], [-0.88170589959412649], [0.39074539415456777]], 
[[-0.7679992173514163], [0.75979209750961751], [0.37500296160737934]], [[0.40850982483521969], [0.73379893282352615], 
[0.079938187194695365]]], [[[0.96891190888620771], [-0.42047739408907336], [-0.32426632170329261]], [[-0.68646300004291416], 
[-0.34520682892139898], [0.81339981488561564]], [[-0.36440809627288107], [0.14437108681226074], [0.67541223061404887]], 
[[-0.087337180945730841], [0.05399085213198429], [0.2294684979780881]]]])
        yy=numpy.array([[0.7501862062918041], [0.12010239926410882], [0.33793918323596839]])
        ref=tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_mult_Symbol_rank4_and_3(self):
        sh0=(2, 3, 6, 4)
        sh1=(6, 4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.55544540384858077, -0.28243376190716796, -0.011214060279147064, 0.66576125042435463], 
[0.55132601404500003, -0.20847968776920522, -0.85318698248617331, -0.94163053593492241], [-0.29827324478820527, 
0.19164913434156028, -0.75392935348865486, -0.90262002778980355], [0.84383533432398172, -0.018670833297221856, 
0.87820928307324353, 0.25247345720343017], [-0.54432424195329054, 0.4938591881362302, 0.40682529216146768, 
-0.21027593586065207], [0.5617625609426371, 0.6269866945656084, 0.89433985661561177, -0.34784537362608692]], 
[[0.71119943493442817, 0.85839038737888518, -0.24473962317937104, 0.31931709886606985], [0.19842381764040939, 
-0.34392588645202116, 0.67303399909487749, 0.80569541430662306], [0.68018393549412859, -0.99045997698548871, 
-0.20639386917677016, 0.11563573861702126], [0.87985666985153044, -0.16531146550746234, 0.64002067272022201, 
0.25765679971761468], [-0.62768447123520299, -0.36052809113376516, -0.92308936994817947, 0.39847680890228609], 
[-0.56047614252922351, -0.34548805339918909, 0.55372163994303003, -0.11473052824132379]], [[0.18025919381585309, 
0.78553037558118199, -0.80357847443307762, 0.37672812076915685], [0.74510646111576739, -0.40600490676220069, 
-0.5320001943689896, 0.11017701305469885], [0.9371294193104589, 0.45305989964044535, 0.42290943619762777, 0.25284440033701405], 
[-0.64704127764723252, 0.46897638390046792, 0.66892623924376626, -0.05980387576026347], [0.31517389288464681, 
-0.20413954734667827, 0.323798042210651, 0.82342340029801697], [0.25154340184933055, 0.51869323605852746, 0.002221142821502653, 
-0.54049733368638586]]], [[[0.84274864958397577, 0.87259363912418397, -0.32248869117922507, 0.60378859465825263], 
[0.86266119217185389, 0.4422913721527777, -0.39373340093919684, -0.57236099285700592], [0.31873592237393877, 
-0.27657534901494696, -0.30742744904833796, -0.18975201104678696], [-0.019492064421242628, -0.24435415799574023, 
0.3687992150929309, -0.77162229528797721], [-0.90397914849928651, 0.45372363224277246, -0.26983012512140747, 
-0.46064137191918819], [0.21515222634016595, -0.81192108260553097, -0.55948330756812648, -0.84636844320889093]], 
[[-0.6253842866081305, 0.044857007662039372, 0.55095187548478752, -0.44265924448261207], [-0.051767686847068006, 
-0.058632822980667409, 0.8985819708143028, -0.12663701821876439], [-0.56434614531853988, 0.99874786374983282, 
0.73612320788667418, -0.42169611603158175], [-0.72943376160999418, 0.75976377959232488, 0.87522199646109411, 
-0.63322564515648505], [-0.15837549948715068, -0.69195723760580741, 0.95695621486986537, 0.98085522510032663], 
[-0.55620444854166862, 0.6852755317023651, 0.18763490997491328, -0.63935174017468577]], [[0.69387143443385879, 
-0.61161784857903734, 0.71764900504852513, -0.046814111214755538], [-0.72108750278497191, -0.39039055379312182, 
-0.23057793574041074, -0.80548758632077377], [0.94413221660877578, -0.18735149810981833, 0.90582459631051226, 
0.63251131502468882], [0.21819431557653512, -0.82869903661920041, -0.82252148201443975, -0.81766894760847575], 
[-0.085524565681989628, 0.13487102680790564, 0.25928059594752595, 0.10356902460294548], [-0.85877733215139518, 
0.95367619567603712, -0.90196037915374161, 0.18986969543011112]]]])
        yy=numpy.array([[[0.98263285544279078, -0.46528188665024173, -0.75219073556009808, -0.75492540903076644, 
-0.68102541687502849], [0.08464762209171095, -0.9547187361508469, -0.29432974244985011, -0.65716877563394283, 
0.096245085070755243], [0.63116259973650557, -0.083055050798974905, -0.53265114546164249, -0.80722276144916782, 
0.39442604307879714], [0.43082233173586748, -0.87939937130791401, -0.27327606942453042, -0.94520429605293366, 
-0.29089588114575848]], [[-0.22287002899378705, 0.38709668712937773, -0.16521346572964557, 0.57263703274323063, 
-0.15252514378818893], [0.46859446139524241, -0.15671714438335393, -0.64895345327331144, 0.92338076740995367, 
0.17602688813768741], [0.77488155371181833, 0.22991643570787268, 0.33431955970124294, 0.24393619247329457, 
0.36033389489912704], [-0.65628714760806206, -0.79266831386913883, 0.10308759822463531, 0.64824331506749, 
-0.69471045338125692]], [[0.46121653558554487, -0.012790438080468869, -0.7233731832528425, 0.43494553790476709, 
-0.16047219197001517], [0.56278313521763956, 0.49976737808019278, 0.56775544321552074, -0.85846192914452901, 
0.49134190298662928], [0.72378760227785333, 0.47497985111800478, 0.31607109702200709, -0.26868844599996411, 
0.11406595147047738], [0.45639768514127499, 0.59182643430352155, -0.79549273356233785, -0.3454663223194665, 
0.83965103094446247]], [[-0.97552531382423169, 0.36009427851558895, 0.92509570829547161, 0.46248082451033534, 
0.72495862443123005], [0.19980271775629266, 0.66136165560159355, -0.6092360292863146, -0.40233871169634505, 
-0.91141997284168697], [0.19794660551808962, -0.88815308376047097, -0.29042123767179007, 0.83839537943277076, 
-0.13780847749988623], [-0.31609287133940045, -0.44451273238080646, 0.66977779654685943, -0.11339641420932312, 
0.83491437097869259]], [[0.35356831558017388, -0.56349004518221313, 0.81729083805970038, -0.49292919965608406, 
0.72675723204843545], [0.54851433301851449, 0.62306951562374224, -0.93515954379604005, 0.96880977033861715, 
0.31671636556648863], [-0.98523044227318102, -0.24175065432610832, 0.09132303796712038, -0.49212065962430418, 
-0.84859191504493259], [0.99945955218579341, -0.7080884687817095, 0.34280646357855749, 0.16693588458780084, 
-0.156132287305633]], [[0.30488759710018054, 0.29104959521488438, 0.25931875474973975, -4.7058417930845309e-05, 
-0.66688580327902724], [0.26474640583463493, -0.36953109160406417, 0.22107296234345419, -0.26012795468655336, 
-0.93120834096341887], [-0.78499225428726915, 0.27778700785010946, 0.044649613703098945, 0.19952934909663322, 
-0.80522226006044395], [0.23906163945877013, -0.52702141918779977, 0.34968033519421349, 0.35306902390095951, 
0.85469125147247138]]])
        ref=tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_mult_Symbol_rank4_and_4(self):
        sh0=(4, 2, 4, 1)
        sh1=(4, 1, 4, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.50255493694386799], [0.62540328553415625], [-0.33379049280801665], [-0.0335195389527152]], 
[[0.61592210456020902], [-0.44228179216874364], [-0.80788563688070525], [0.29323929067359722]]], [[[-0.057584707153953252], 
[0.0034522151310367732], [-0.010620463445897288], [0.073109298555587898]], [[0.68317399707636506], [-0.66360602164012295], 
[-0.9641736756767747], [-0.051390017177419978]]], [[[-0.99274288391939192], [-0.73151186193807205], [-0.4310217435822481], 
[0.88850255643798648]], [[-0.1173611179319447], [-0.30584088916131913], [-0.20937753423164862], [-0.8921865927393362]]], 
[[[-0.13893735687443565], [-0.44912979427985933], [0.38926797395386159], [0.64124633929693098]], [[-0.20220846452255103], 
[0.68623763094967383], [0.36698484255735608], [-0.69719634695458765]]]])
        yy=numpy.array([[[[-0.26669391468484593, -0.63804282054534522, -0.24881970437665291], [0.55190990780222671, 
0.96294915489374788, -0.089847556481072122], [0.20859533312537648, 0.034753071263605939, 0.97807398207317209], 
[0.56776214768513467, -0.73948101843679992, 0.35496623022646756]]], [[[-0.12403043691232751, -0.080683588245591187, 
0.3199090866512615], [-0.88503061522188942, -0.12799800821242724, 0.34788847046171889], [0.91350588030837043, 
0.52942058236927747, 0.9202861176488879], [0.32671727262928552, -0.73044336761439843, 0.89931310523560137]]], 
[[[0.63889299784229681, -0.1273109512060997, -0.095779306749175008], [0.23187870015367129, -0.86858692186358177, 
-0.95735655431435296], [-0.34105765244433606, -0.069299907673253225, -0.68244384335082642], [-0.19117207112028778, 
0.68579583705908265, -0.14298635013883043]]], [[[0.24457624297263059, -0.82604396280744941, 0.010423435593567509], 
[0.36101056843318458, 0.78149002895272224, 0.79910104831018436], [-0.81190233873726525, 0.45926148241649378, 
0.44068783148157231], [-0.80116706946323468, 0.85421316147897475, -0.008656579157609201]]]])
        ref=tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_transposed_mult_Symbol_rank2_and_1(self):
        sh0=(1, 5)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.35736646443243703, 0.6521405716498454, -0.11557437226405098, 0.17753443994217988, 
-0.87885955061118048]])
        yy=numpy.array([-0.0025936054461073166, 0.8515587499241406, -0.96924877366893947, -0.6519913100435526, 
0.014920632322150684])
        ref=tensor_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_transposed_mult_Symbol_rank2_and_2(self):
        sh0=(1, 6)
        sh1=(3, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[0.21299145700937649, 0.05982190595488146, -0.73523453908221459, 0.94772308691957141, 
-0.23414024459417182, 0.81335704606365411]])
        yy=numpy.array([[0.43599374983495287, -0.39829190672659331, -0.67356951043357083, 0.46253760021030366, 
-0.411236183300141, -0.50307885277982423], [0.96271011000936535, 0.14996146039343428, -0.67616218353328517, 
-0.73725476608272156, 0.15765965076809807, -0.10329533702585825], [0.22085881351155767, 0.12651032113906169, 
-0.052594642748230447, 0.086983910100482564, -0.024106511513036333, 0.24532913664242195]])
        ref=tensor_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_transposed_mult_Symbol_rank4_and_2(self):
        sh0=(4, 2, 1, 2)
        sh1=(2, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.15523647406117758, 0.68942944459857269]], [[0.64762174613297052, 0.40310881579671864]]], 
[[[0.43944985333003928, -0.084716354986196007]], [[-0.90064940726422171, 0.21575527692467089]]], [[[0.9125734314654006, 
-0.36459928785133155]], [[-0.22462276783111212, -0.42161393988755957]]], [[[0.26571160085874457, -0.62600818728809027]], 
[[-0.71097319952325466, -0.62527074156544704]]]])
        yy=numpy.array([[0.34480495362926078, 0.79127258239206455]])
        yy=transpose(yy)
        ref=tensor_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_transposed_mult_Symbol_rank4_and_3(self):
        sh0=(2, 2, 4, 1)
        sh1=(5, 4, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.63156414988818033], [0.14374715793603454], [0.1369809765188772], [-0.34684379216654926]], 
[[0.97616909400050522], [-0.52942621094070108], [-0.65139814508948568], [0.28291443940795302]]], [[[-0.11117131193282015], 
[0.30074345343250219], [-0.20576285675601813], [-0.73850769540045946]], [[-0.95854126156601294], [0.1102938178432602], 
[-0.17604987505453229], [-0.15090050296902469]]]])
        yy=numpy.array([[[0.88846916115296048], [-0.96317734417128742], [0.79167690588470974], [-0.24941216771149621]], 
[[0.087864710998970086], [0.49341938847237454], [-0.13877599700628718], [-0.052176356370096322]], [[-0.28554997108693425], 
[0.10263720767890283], [-0.63767504194896452], [0.19867488868995142]], [[-0.64019235478401404], [0.72740758490294355], 
[-0.32331560205534315], [0.82484990041943318]], [[-0.029810240230135676], [-0.72521420778422474], [-0.58095863639602152], 
[0.89481641097769726]]])
        ref=tensor_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_tensor_transposed_mult_Symbol_rank4_and_4(self):
        sh0=(3, 3, 6, 2)
        sh1=(3, 2, 6, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=tensor_transposed_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.84755597758829171, -0.67373571013781475], [0.21043676133887135, -0.42019634442646003], 
[-0.27927003529536032, 0.9724236580675083], [0.54826540531883028, -0.79085341461748482], [-0.76439774215506873, 
0.32966722921497049], [0.21686694105443083, -0.94006484918503919]], [[0.13123794981099901, -0.59689310679611474], 
[-0.27950359544601633, 0.86904136970741352], [0.0005809751209999714, 0.16209354451470848], [0.37852289760101776, 
0.44608366423725254], [0.33421202520812221, 0.3361184880179886], [-0.42414721161747315, -0.79164905497092164]], 
[[-0.4883136111277282, 0.74695664950391172], [-0.590115384811432, -0.25389617797013719], [-0.40524298105608825, 
0.83916600464065483], [-0.63713926345075089, -0.91076809630770938], [0.88239582860343102, -0.96255066935558586], 
[-0.3737542244227352, 0.31260861835050435]]], [[[0.31719948346710791, -0.90404929393770894], [0.40394653645405909, 
0.96401715364199791], [0.26483811824211889, -0.2286737748833334], [0.62923693343804388, 0.8818076232194143], 
[0.9219468007177789, -0.06113995314953935], [0.56824408042811902, 0.81222687234264157]], [[-0.77404417160897565, 
0.094061787328822311], [-0.037391409482322668, -0.67743234836468003], [0.88053405896669967, -0.5883146465296063], 
[-0.53795932115186229, 0.15641226699531807], [-0.3718264329750125, -0.64755399077162701], [-0.44386871317070242, 
0.27307732986445976]], [[0.49304526248446323, -0.77078500204951994], [0.28913308896138612, 0.54330367345655128], 
[-0.7002552941491893, -0.58366960521941702], [0.98436621079330822, -0.85506584919618867], [0.19792031933413745, 
0.74421609088927787], [-0.25457650695285494, 0.93108452783888906]]], [[[0.78444328338935199, 0.87209114906272478], 
[-0.97605996541145146, -0.95366943942080762], [-0.72403434481507589, 0.99504484188574538], [0.16726735376241275, 
-0.94241119426222952], [0.76022909298662977, 0.3108891698404741], [-0.59489154946132539, 0.26437007223471487]], 
[[-0.97255488138410517, -0.72817873296134161], [-0.60742485367804822, 0.47476432637749255], [0.56231071346633721, 
0.33394551892391022], [0.65786653039412757, -0.54330363983889418], [-0.98537499125207151, -0.9892661839913921], 
[0.081664510299984006, -0.7850381482690072]], [[0.16822752620663683, -0.23283211739424714], [0.72505115525540376, 
-0.022268602307847329], [0.36971833123609321, 0.89692935657389028], [0.55276452045138424, -0.76571147926757721], 
[-0.067408501304647928, -0.81043325338547789], [-0.60136170508519227, 0.034178490906835002]]]])
        yy=numpy.array([[[[0.59217612005261566, -0.88562907335683017], [0.43015794976645472, 0.9219098139178401], 
[-0.93079363880584554, -0.47522741829819259], [-0.61332625627147586, -0.38867759984846884], [-0.34480087455496822, 
-0.94717685262067119], [-0.37527476285180716, -0.65775767239232241]], [[-0.88286536835553742, 0.96863359016316108], 
[-0.42009335015570604, -0.14231263750781808], [0.91209995330135896, 0.55858703736302862], [0.65425779074215962, 
-0.83438608868996056], [-0.14197729332209996, -0.60251330536871173], [0.46971781054871897, 0.77590949918589747]]], 
[[[0.66900674136933724, -0.12897433890512877], [0.37629508026801539, 0.034005210800539531], [0.37790410137412933, 
0.60250828460879413], [0.57602450391710769, -0.18596795284555157], [-0.87104913521161631, -0.79886253396205675], 
[0.21174086393525693, 0.42623945796864127]], [[0.15326943991635233, -0.7282561250887909], [0.26512369738553732, 
-0.77591956405233886], [-0.88079994362486036, -0.068931319862178597], [-0.9096491784879015, 0.46360379441326338], 
[0.43521824917526208, 0.60919134556845989], [-0.79124624832694335, -0.6978927299361497]]], [[[0.67661904936974149, 
0.64819340679109971], [0.77670605562254069, 0.86138686786322638], [0.45479975171754394, 0.32582401054461085], 
[0.52010369459373318, -0.56118539798478295], [0.24476327833622014, -0.6981553237560687], [-0.53847394042890273, 
0.013481953147003622]], [[-0.52999774200400096, 0.87091188026302446], [0.18425759987328805, -0.47947102332270775], 
[-0.29300503189478011, -0.63674527669184289], [-0.95396215629940517, 0.7648874034698252], [0.19311887263710448, 
0.1303497102872202], [-0.21350631928500596, -0.28280079389662571]]]])
        ref=tensor_transposed_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_tensor_mult_Symbol_rank2_and_1(self):
        sh0=(6, 1)
        sh1=(6,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.43390217759457728], [0.32327856382427189], [-0.50172947055128359], [-0.60976765860665494], 
[0.4418797607924716], [0.26651190325008534]])
        yy=numpy.array([0.11968303683008963, 0.31390129815322121, -0.4360610357407968, 0.86630708319422189, 
0.63332707394127419, 0.39311088754349366])
        ref=transposed_tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_tensor_mult_Symbol_rank2_and_2(self):
        sh0=(5, 4)
        sh1=(5, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[-0.33102511968481774, -0.84473846630598226, 0.26152407687265833, 0.83063501922139027], 
[-0.90413228316915473, 0.28831283798377982, 0.12220579974363721, -0.6389258074958355], [-0.58922343941444089, 
-0.3698022289555043, -0.9031454845664173, -0.77316554033900986], [-0.1324174807504086, 0.18026804468377278, 0.722546015802076, 
0.0616824118342576], [-0.83741019414362139, 0.43464187324821335, -0.22125435165730956, 0.89680005402255336]])
        yy=numpy.array([[-0.5026789012223789, 0.28301589220722345, 0.050189456326468429, -0.74021358089109346, 
0.64757702445838317], [0.39460041655465328, 0.61208302307482887, -0.67750978834331765, 0.31989099905035556, 
0.86948462995060205], [0.46039507645068034, -0.18719057850799969, 0.64575890832704763, 0.1558667245758123, 
0.80320202113564743], [0.86331659598105515, 0.073958022278642899, -0.50278803576812936, -0.33734349497473315, 
-0.91254551386476912], [-0.73431321665560212, 0.63480474692407296, -0.44780657677723701, -0.60543831250995694, 
-0.34229589543240824]])
        ref=transposed_tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_tensor_mult_Symbol_rank4_and_2(self):
        sh0=(6, 3, 6, 6)
        sh1=(6, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.59396478441660339, -0.84902639954028158, 0.9831025631116439, 0.11908468054038224, 
0.33663397176067078, -0.70241909423653048], [-0.12117633522577487, 0.38273255799575145, -0.048154239169091229, 
0.2717608559474376, -0.33879359507462881, 0.60825442213206937], [-0.19377168615394136, 0.66800793276233517, 
0.41419804077735867, 0.11473363801837366, -0.80817218051928541, -0.51883589334230673], [0.21966246260911126, 
0.97613924749717063, -0.025324906522526724, -0.073049532188571353, -0.95182672228225318, 0.0031497131834514747], 
[-0.84733698034869631, -0.44899395601494274, 0.21680916040388487, -0.21186511231131, 0.42050828575114774, 
-0.72760387167017715], [0.43523655063746447, 0.82096750177819056, -0.067276317563208954, 0.73792830292311695, 
-0.81961219719820622, -0.42165788999476295]], [[0.074110940926232605, -0.019908536155385459, -0.16513579024866853, 
0.86586486583246014, -0.96716502527975545, -0.67567057938485697], [0.20727691765023648, 0.77067841286909178, 
-0.54451960157208834, -0.24509816474162749, -0.89385481816495815, 0.43997712124162147], [0.64140435720703848, 
0.81550217525721957, -0.17836363787041942, 0.84565408632377781, -0.51563211772568929, -0.94620016903769599], 
[0.069285731488672342, 0.167695054902681, 0.75178661026600913, 0.61665388469645777, -0.16788574768866771, 
-0.43164437562860614], [-0.028058703050098055, -0.57968091309253378, 0.75352808372051716, -0.3181944438866442, 
0.74292850028977497, 0.32328398480549136], [0.44682174308281764, 0.33886827032218081, -0.8691764129343118, 0.37924209442118606, 
-0.042853977864793613, -0.96071356717307443]], [[-0.17283415078846365, 0.61109328378347327, -0.87899902229611748, 
-0.6183569468518566, -0.019323588947317116, 0.89285680552581459], [-0.075105172688181954, -0.54824335335989449, 
0.40656873591981024, 0.15945800223523166, 0.87089715095591003, 0.40569754439636951], [0.36940583539875926, 0.74535827873566318, 
-0.23905355272826667, 0.47590576850495148, 0.73973913457493001, -0.55822096785926534], [0.37339238823045151, 
0.26214929296592082, 0.21887187220421001, 0.092366470089652264, -0.59112178124299986, -0.13798851974896942], 
[0.94892213751577015, 0.53799551660834366, -0.38992440042335752, 0.061453639680707717, -0.60111948972701335, 
-0.24280731199378081], [0.13044544549188219, -0.78243838800016441, -0.70357031026148897, 0.13325247903954129, 
-0.6672551363710324, -0.37452480655730191]]], [[[0.83956052979501372, 0.63614360136261183, 0.85580218921414497, 
0.58590198085942857, -0.75674984229105391, 0.75120746959541274], [-0.2700764838874663, 0.95315766484477593, 
0.27438410362396071, 0.52813014405330172, 0.92068419642615007, -0.17145960888711698], [-0.98069465235318565, 
-0.0047848642732466473, 0.88542319456602714, -0.14533575448824654, 0.7768791537687747, -0.24003190582080647], 
[-0.69009899985015988, -0.64845776292706581, 0.45525011426425266, -0.58196732481390945, 0.020024267592280598, 
-0.87189884723974442], [-0.18154306536517373, -0.46388971017202252, -0.94077839399563645, -0.56811000831144542, 
0.44191434012707287, 0.25590225284400003], [-0.98813887209689932, 0.8302187007491526, 0.13135045890248809, 0.78016782383706129, 
-0.95202696849765567, -0.59044467268450251]], [[-0.5057542322742139, 0.2320592882716328, -0.73260585574235271, 
0.38777932851463537, -0.61224394267885507, 0.87706608877641945], [-0.10783379532600024, -0.83286005466990121, 
-0.67346750731150173, 0.2275066999451818, 0.34235562425299193, 0.77935649398032836], [-0.13614130307483574, 
0.22843605658172939, 0.15522887348828451, 0.14534707140472092, 0.29297445787451326, 0.32042453361642087], [-0.9731945915814002, 
-0.42617227025066096, 0.15152563878254344, 0.56766943342825371, 0.65918213136336612, -0.4086511565562716], 
[-0.59829565941237872, 0.68780962676493429, 0.57679797502696828, -0.20884050603554538, 0.49580030493479832, 
-0.76526069362436444], [0.29639045006024323, -0.54364074480029712, -0.46599120798835525, -0.90869084463211625, 
0.16250552435453436, -0.9442461065206933]], [[0.93279085146522345, 0.34092726497963421, 0.63223559564121667, 
-0.13454646434036377, -0.49247639969886592, -0.74290838301715034], [0.4451804620609936, 0.90815529338994994, 
-0.27102772122856567, -0.015458796406649267, -0.72087691691582823, -0.93501794872599486], [0.23343753907813825, 
0.69001200070941748, 0.60540193366262529, 0.5530569286192597, 0.7845911556154872, -0.32627898546249789], [-0.27322813695732417, 
0.33114854618900003, -0.26740581106103845, 0.21019310597627916, 0.67983598292662761, 0.82405313302276451], 
[0.35796398012180641, -0.45399685940843737, 0.97731858368380187, -0.39108420317128867, -0.15370622449113225, 
0.31914662769186153], [0.085622569745243071, 0.15488590924990886, 0.25635913628152585, 0.091803997517173608, 
0.12317959249613764, -0.96282827606426724]]], [[[-0.47096560588910941, 0.46142664094190389, -0.23029528181660242, 
0.28377687755642778, -0.73062270188671707, -0.18787092581726395], [-0.35521135047693808, -0.53948389338694325, 
-0.34848632436183413, 0.29585005196953551, 0.014644349785388755, 0.90329105026759282], [0.0032995764298566765, 
0.88210195238402989, -0.96339186729067161, 0.8275448289566647, -0.19068143949361094, -0.39265272984992361], 
[-0.39211470285355809, -0.014191562381086209, 0.82181623997004416, -0.52787138813456047, -0.65033757604866671, 
-0.93891372867531087], [-0.72300826981979571, 0.41631547673122693, 0.72151487538758641, -0.083184490222169227, 
-0.28178018479545464, -0.26982085252694099], [-0.51539332731899945, -0.25931439448135429, 0.18965054089691336, 
-0.24822060228679121, 0.22258984232994128, 0.4371635652330037]], [[0.712374272784839, -0.37449899121096375, 
-0.6933702732563376, -0.77090584586739852, -0.71704221420882241, 0.49050534573124827], [0.96007645785247386, 
0.84064770213152951, -0.6326011752802585, 0.55801796806513293, 0.36209183190479965, -0.49646553468360621], 
[0.26255699627775808, 0.43581275603081471, 0.80630075191719897, 0.28832683949340043, -0.79134285099890178, 
0.47072963891797648], [0.82137918994464898, 0.90934612514111279, 0.17252366141970299, -0.53601019880537826, 
0.98409684810947806, -0.5528897602840579], [-0.68283908236330437, 0.57141921679576746, 0.7863657711292964, 0.15177542981914116, 
-0.6880113248458799, 0.18607384845526753], [0.50111032922329835, 0.54226587727298603, -0.74647031841432887, 
0.76083934097422956, -0.21731611313271526, 0.6924272307285475]], [[-0.92106691189123913, 0.9907912396334202, 
-0.43424832434091143, -0.6501658743263774, 0.74647276977471511, 0.41764601563808634], [0.73139994701329347, 
-0.53892505737409291, -0.83218278049513494, -0.90935515174218784, 0.31299863124228566, 0.19741765716825599], 
[-0.30645424654275688, 0.59655211432276989, -0.92473432537557021, -0.014119639615391311, 0.24945987434430927, 
0.1187430978436339], [0.21980149184758546, -0.094520999581238163, -0.91053383038483582, 0.83587745786524992, 
0.8676687319920291, -0.45272189201465785], [-0.080995211368490194, 0.18044510835375882, 0.67705790999384208, 
-0.96892320287551947, 0.70657486172717143, 0.27066372647167003], [-0.14968008483689199, -0.91506759420146211, 
0.51834598971888024, 0.57620296765557866, 0.29263727619388713, -0.22279742781538769]]], [[[0.50751003049402299, 
-0.23865850434508951, -0.44855152069162396, -0.038541315139235754, 0.024129871685861559, -0.4729277558349021], 
[0.16666286833403809, -0.29452944493492228, -0.038662768660178592, 0.11023066972070739, 0.19895631828445026, 
-0.46194091998121301], [-0.96433597347670807, 0.47637643434777677, -0.61691976646010249, 0.84907611798456206, 
-0.30529328501512176, 0.048346752145214777], [-0.96560471576072815, -0.61156369207472361, -0.32385668674142964, 
0.99587353854002814, -0.86113879186208475, 0.30081094349435689], [-0.40574477736870795, 0.65258875698121588, 
-0.18691758500118305, 0.15897553971937439, 0.52872974985270838, -0.070916176997700919], [-0.45636407330940698, 
0.024486502305997693, -0.90390269910629772, 0.35733088773849064, -0.50575112016651702, -0.66927292524092219]], 
[[-0.18573268033487356, 0.63189266516253317, -0.62719463876829895, 0.47029720837332922, 0.19747842057921527, 
0.60691395489038924], [0.52454744520348706, -0.89038589260333101, 0.95872458295085261, 0.56373972462997224, 
0.15632332004448202, 0.34372760552514348], [-0.9965052969931516, -0.23994609635595388, 0.59271443906043264, 
-0.39076517690800072, -0.74220351908808868, -0.26263369740493236], [-0.59811921381129118, 0.59607138103369195, 
0.14976732946370319, -0.82438584767136014, -0.87411156361716658, 0.41291066676618282], [0.89721606656511854, 
-0.10506719517844609, -0.91739212678713611, 0.69451591363087917, 0.53982736262487929, 0.55702031908647287], 
[-0.025747967296285479, 0.45332868842874352, -0.39451841322021353, 0.87199641133437633, 0.23096984481183247, 
0.82738111796231162]], [[0.92238480432403858, -0.41038326430649597, 0.54438290734779815, 0.15999425298837067, 
0.46906623121982416, 0.49272816839326739], [0.74626247801363732, 0.60562216869124352, -0.14893892614696047, 
-0.16271863611620097, 0.48574907759845676, -0.26515999274128998], [0.66675349276350038, 0.91504020202714775, 
0.065733817518587978, -0.63560961818352246, 0.2791706309476909, 0.052030590054426895], [0.59573608525261501, 
0.79501071261469702, 0.45286124267013572, 0.45312830451071151, -0.81043954769398607, 0.81251586073181237], 
[-0.52940352028138471, 0.69452534819876055, -0.3284817253997323, 0.35617773181025347, -0.012822917005427925, 
-0.18111582225453771], [-0.61990649316081869, -0.20781460407317742, 0.10158261300452831, 0.32819736633061125, 
0.93163328725854977, 0.36829284271582141]]], [[[-0.075113466219749903, 0.84288939194928592, -0.12300225564876266, 
-0.66897437852537123, -0.67345052644715997, 0.010652288715401914], [-0.64682848648991453, -0.50786587573631903, 
-0.59760154854402225, 0.83438931248814052, -0.62379018849518597, 0.46860933370416746], [-0.77156278391598487, 
0.87983964576528662, -0.74367871559391063, -0.030935511411390948, -0.78561354605950728, -0.88714490776036437], 
[0.065644035379332388, 0.23274987089606403, 0.2617209610084934, 0.24263313881632076, -0.27432302967574618, 
-0.43182645244835216], [0.22252625866959974, -0.21712707198934345, -0.85910298033932397, -0.68009395050165233, 
-0.45579177491126965, 0.57529066930785122], [0.55769151650692028, -0.71812205271077278, -0.046863120686942317, 
-0.80639711602184638, -0.87189837385904756, 0.2620124498297145]], [[0.061041975337126964, -0.67924956207990705, 
0.43326156577697983, -0.76964889056844332, 0.20617038081805039, 0.28615236373917297], [-0.51919670185048883, 
0.051249281124663204, -0.92566836252855467, 0.023349148806710129, 0.54837487876224911, -0.694785075763801], 
[0.98247408194917152, -0.95943126390100808, 0.37453470648401321, 0.6200085166645144, -0.049499113582657461, 
0.46570591205646172], [-0.7672850491306753, 0.511724545721713, 0.080607562902663243, -0.12546582798543815, 
-0.54697150435625863, 0.91981517931881207], [0.19143830359199732, -0.63728069210219385, 0.64105122503918288, 
0.06505733266668301, 0.068110619574247355, 0.10013423555163903], [0.35313527125491784, -0.6224414753098273, 
0.40082663606493574, -0.77153129685326971, -0.79884078896289612, 0.11887797092609231]], [[-0.96988342151358986, 
-0.47484723455675626, -0.038335597373389874, -0.79183309861283635, 0.67551543217808452, 0.93713726574683442], 
[-0.47396957499547754, 0.13271225194183067, -0.53855963967018461, -0.93812456761180107, -0.47947795995062159, 
0.52531419493321585], [0.9965469131529574, -0.039448929470022653, -0.15306044266130781, -0.61127657084539466, 
0.96249800874888436, -0.52644574770736696], [0.90235601685621059, 0.55038785467904661, 0.43389709548726851, 
-0.066795677169694034, 0.82130075368490574, 0.89341199141228689], [-0.38114987968910596, -0.11810746704620412, 
-0.91932318886784703, -0.50890298199323847, 0.97882161185774463, -0.27043963959633088], [0.70200923578333652, 
-0.17172338394471698, -0.51018193869286144, -0.72049879731352773, 0.62765772507200257, 0.40183712483550371]]], 
[[[-0.49159451785391317, 0.56229574622976419, 0.0055507787957327359, 0.44316194837719203, -0.064606090786241577, 
0.99805740927806164], [-0.85464811204242186, 0.17390639895648263, -0.57266944345991289, -0.4282213132497259, 
-0.77132814347956269, -0.97157962546387955], [-0.67040359348591583, 0.60162416898121251, -0.92473367006330531, 
0.41721042357788729, -0.6616994851589284, 0.83692337782903814], [0.048442325666924635, 0.39197933199447665, 
-0.22414215721764608, -0.48680368252013717, -0.30322778333095179, 0.48189690180800748], [-0.26526216471926545, 
-0.99001089603292569, 0.40218764299214649, 0.57959046899961386, 0.072858568167314175, -0.35094963232058274], 
[0.49855316896200064, -0.56545687402988709, -0.71355207106693608, 0.48173468271230169, 0.31200815312001251, 
-0.29375268524432996]], [[-0.72539356128495758, 0.67343767414527744, -0.37654594050049228, -0.96741492689710284, 
0.17642093271168835, 0.78536787895024074], [0.3642198672126653, -0.47341472994311329, 0.61885199540480951, 
-0.22905887473889708, -0.023266744122137917, -0.34845809699355201], [-0.57295822028303545, 0.97828985603082708, 
-0.17926960074386233, 0.5067952815497474, -0.11666160007333248, -0.53812419938578704], [0.35669787925418217, 
0.21529023368051581, -0.95692977641758037, 0.7711477014975554, -0.092819453597700496, -0.21763131393442214], 
[0.55400761662803699, 0.73472581252296032, 0.29320302885644689, -0.16350853189347303, -0.51906065026251857, 
0.31322443893749674], [-0.72437621210177272, 0.71814933757264998, -0.58956685296080269, -0.84843253550527642, 
-0.51828344361206891, -0.40189332144680656]], [[-0.76646469492990876, -0.1364910180228216, 0.30152842752542619, 
0.34449646595574301, -0.83066923828186323, -0.66246365280006581], [0.75713970212875692, 0.91663388047123551, 
0.90070402346687639, -0.14917931563928089, 0.25077446399913117, 0.57045510140921274], [0.57687861601170898, 
-0.33311602366357729, -0.76516644637918652, -0.56609843551570171, 0.66882278696803543, -0.061715072216673272], 
[0.16839005304903387, 0.0063295534008118715, 0.94120533184131072, -0.038805849832608708, 0.9643346649610296, 
0.22710250904595353], [-0.89402249377583187, -0.18839750572739367, 0.17563788694964821, 0.50951966711021468, 
-0.12036389385802027, 0.92746401314999405], [-0.31445831377681532, -0.29275730555208246, -0.40850287707772592, 
-0.053237917494146192, -0.47602881328841451, -0.96250680068121608]]]])
        yy=numpy.array([[0.76950063640876043, 0.072514217592794195, -0.98954267916025351], [-0.82629364589343757, 
-0.79881246733231315, 0.58341184826384973], [-0.91676104084603582, -0.081723523590191194, -0.0084562355101498454], 
[0.82861109572926983, 0.0029271405698971797, 0.44221079613980452], [0.58388353928668946, -0.53880767835315058, 
-0.63263981409283887], [0.65113071920261301, -0.61401284219313568, 0.95394194296281176]])
        ref=transposed_tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_tensor_mult_Symbol_rank4_and_3(self):
        sh0=(3, 1, 3, 5)
        sh1=(3, 1, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.28885831289000974, -0.61552402069391765, 0.3621279251683962, 0.087641278236779918, 
-0.23192478817798201], [-0.84351946009820078, 0.97237548329669443, -0.11960152445498951, 0.69814794325777996, 
0.62336815132468915], [0.98332543314479715, 0.023965420293253992, -0.19283180861583515, -0.50249936652915372, 
-0.74832622450935649]]], [[[-0.41534904282814411, 0.83074778802181681, -0.27534559583076912, -0.036140464833054553, 
-0.43213470849448621], [-0.8296970721993937, 0.62912714049065888, -0.015994756741034566, 0.87572129686333144, 
0.25302686775218519], [-0.0065964796310742368, -0.13329409611113774, -0.8846193542764369, 0.88813085997073649, 
-0.1582655350147506]]], [[[-0.70946421708246032, -0.86652281682846888, -0.80812508241216707, 0.56822782350363354, 
-0.26546282042281999], [-0.75704823888939532, 0.23363677181534093, 0.3570736752109227, 0.43551173841188051, 
-0.073974669933908332], [-0.43190863939644664, -0.50033179610509215, -0.40005675330379997, -0.45907250499841301, 
-0.24657165663741587]]]])
        yy=numpy.array([[[0.57926407439887662, 0.91865130758071301]], [[-0.40053496951092171, 0.032797806117900619]], 
[[-0.92395503551580815, -0.13511600909177579]]])
        ref=transposed_tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_transposed_tensor_mult_Symbol_rank4_and_4(self):
        sh0=(1, 2, 5, 1)
        sh1=(1, 2, 2, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=transposed_tensor_mult(x,y)
        self.assertTrue(isinstance(y, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.46500892479009504], [0.22811494169778612], [-0.8887822219252528], [0.0067951016725904534], 
[-0.01465143028454241]], [[-0.50502436062238365], [0.19946740621864945], [-0.13437031861118132], [-0.52441581058558384], 
[-0.86735962435773351]]]])
        yy=numpy.array([[[[-0.091225397806633302, 0.084527814373397181, 0.68315602802422415, -0.16843703813845612, 
0.33260606150828509], [-0.086031569934833518, 0.98946571733071376, -0.58137284193711158, -0.94740238613879701, 
-0.68581218540704203]], [[-0.81449984947335285, 0.054908251789116846, 0.84188185119184711, 0.83061234227846392, 
-0.67732946970193053], [0.81714912670845119, 0.96797787267296953, 0.81773026340316779, -0.38092842657011716, 
-0.67741122719231139]]]])
        ref=transposed_tensor_mult(xx,yy)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank0_and_0_offset0(self):
        sh0=()
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.778200899638)
        yy=numpy.array(-0.270446676276)
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank0_and_1_offset0(self):
        sh0=()
        sh1=(3,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.776286030941)
        yy=numpy.array([-0.69500864705058651, 0.4660117567144948, 0.23913893797782326])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank0_and_2_offset0(self):
        sh0=()
        sh1=(5, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.16177117131)
        yy=numpy.array([[-0.85972263231942625, 0.39727793832824232], [-0.82587275784494452, 0.94883334111123863], 
[-0.45750826176710468, 0.83843003952406803], [-0.83569405630592075, -0.029474194795837683], [0.83944423595751094, 
-0.51903386782811523]])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank0_and_3_offset0(self):
        sh0=()
        sh1=(2, 4, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.69185575656)
        yy=numpy.array([[[-0.41668685623610391, 0.40429562830070354, 0.70669734808953666], [0.7975483374213832, 
-0.71071237335599946, -0.05065742613564872], [-0.049169717678666958, 0.73216230935651372, -0.47167660075781526], 
[0.15953578467152862, -0.99186746163379236, 0.99892478476378654]], [[-0.13507597618836931, 0.28271624674847273, 
0.6502434750615167], [-0.12129555956885252, -0.37413119538889728, -0.69101761040789356], [-0.23619225079431283, 
0.097487168206114916, 0.52588739787935634], [-0.10327061051569419, 0.92578665198019272, 0.030115892818262102]]])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank0_and_4_offset0(self):
        sh0=()
        sh1=(1, 1, 5, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(-0.527015091251)
        yy=numpy.array([[[[0.18336055500057413, 0.6732788704369812, -0.20210103168246984, -0.13038720413422689, 
0.10285027488556353, 0.33136405142917225], [-0.22479506647877945, 0.3913370747825502, 0.98819767011473192, 
-0.58666931336840622, -0.84599709953683466, 0.13986487498739897], [0.61747149805078205, -0.39445559346843817, 
0.7104741306608271, 0.62943043661113962, 0.79486178906904459, -0.99266176700960207], [0.98258695357978465, 
-0.65909248235116569, -0.29662660122221318, -0.76793326614184942, -0.20440858203845091, 0.16039991979047308], 
[0.35895145127869443, 0.52556586226895674, 0.98996973349933093, -0.075217627580181468, 0.52923098255144452, 
-0.60734671909084348]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_0_offset0(self):
        sh0=(1,)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.73826198281454514])
        yy=numpy.array(0.286933371836)
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_1_offset0(self):
        sh0=(3,)
        sh1=(6,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.99996885635439137, -0.25721238506809785, 0.73251585570478772])
        yy=numpy.array([0.58731514666440621, 0.35073674431891844, -0.42047469398474679, -0.051365302279608116, 
0.30451294516660554, -0.96383457664080319])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_1_offset1(self):
        sh0=(5,)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.16449963760319419, 0.3350468359743306, -0.80126155814958677, -0.69735779177124679, 
-0.23366384035084775])
        yy=numpy.array([0.47543793714508453, -0.42149732632267622, 0.32935119028381599, -0.85060169624847437, 
0.95485222475951637])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_2_offset0(self):
        sh0=(4,)
        sh1=(1, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.30614791887958237, 0.016368501754689513, -0.77924877070678744, 0.99056115819369173])
        yy=numpy.array([[-0.98035063781811993, 0.77877874707635342, 0.99768543638204354]])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_2_offset1(self):
        sh0=(4,)
        sh1=(4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.12521558764909813, -0.15603238802983443, 0.80204601334002912, 0.072951097321355851])
        yy=numpy.array([[-0.40647533769178712, -0.96393729880360657, 0.11958654261432056, 0.35695685169675517, 
0.80178641854831567], [-0.99699678001478365, -0.76516943034077212, -0.74046800313420169, 0.29331813564033515, 
-0.47524207800592433], [-0.98231056345792389, -0.074679262054213602, -0.64603647487036997, -0.23850829202607171, 
-0.50497835355418985], [-0.43978773943742722, 0.95131193846143036, -0.0040369173752075049, -0.82536237287109837, 
0.9940112311423106]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_3_offset0(self):
        sh0=(2,)
        sh1=(3, 2, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.38381989621037449, 0.30530483307911194])
        yy=numpy.array([[[0.48250629267362344, 0.87321883662976929, -0.31777030438727638, -0.86383680895007653, 
-0.36337331425656472, -0.15750179034238743], [0.29619763343832894, 0.79299447667755163, -0.41098523135244935, 
-0.32897041303819341, -0.68235594089849694, -0.76565940756272943]], [[-0.72936794505215574, 0.14526075434170704, 
0.80604773833049781, 0.38894435280160722, -0.63689275585897698, 0.18837736600426558], [-0.9818562343110151, 
-0.16920930501602571, 0.65846068565491289, 0.746619915032948, 0.68020025534852468, -0.91633624326936358]], 
[[-0.96201499007184799, -0.55462412846810172, -0.45350704220169047, 0.87120970312428625, -0.47989849112207428, 
0.22011734371232139], [0.39824829233933889, -0.43719475842494293, 0.20845045651491079, -0.19988617200859338, 
0.064664626178630114, 0.6539086903102993]]])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_3_offset1(self):
        sh0=(1,)
        sh1=(1, 6, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.81951256235392922])
        yy=numpy.array([[[-0.012698689948340114], [0.5147769433349132], [-0.3395685483657247], [-0.55245131501182443], 
[0.099251472827513121], [-0.88109340497562672]]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank1_and_4_offset1(self):
        sh0=(4,)
        sh1=(4, 2, 4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.91546805889638661, 0.24469178939904412, -0.47162195334336676, -0.35062894564925351])
        yy=numpy.array([[[[-0.66669534553099097, 0.75898662519941729, -0.49239973647156954, -0.36530524564401201, 
-0.16013916676347995], [0.3361263244098518, 0.89935841667768224, 0.47161104254896791, -0.94365215736745145, 
0.61440008707884775], [0.88756433039337379, -0.71794379314236134, 0.40608115192433258, 0.57979740135525915, 
-0.67777022834511214], [-0.54393006270515731, 0.80643028448754372, 0.14911258317115816, -0.27303400262970112, 
0.24614458972154662]], [[-0.4203683058846337, 0.83950033236826682, 0.35907570764524555, -0.30247907641102945, 
0.76554273554344943], [-0.4012180458261474, 0.057281002847160645, 0.84412916891691125, 0.20484286385558215, 
-0.85424580178921117], [0.91207674869474564, -0.91816729828286081, 0.75094453550882001, 0.37979007807620069, 
0.31875692450217263], [-0.81656435198132593, -0.088798921377700291, 0.51896994990070122, -0.26158774457210088, 
-0.45865625970814183]]], [[[0.8762000977750628, -0.53341094820017809, -0.77752652566514291, 0.65880859743323561, 
0.42618402881255868], [-0.59730619228442738, 0.31904477949455634, -0.19818129495085635, 0.16221204834251934, 
0.71056633216925924], [-0.80920753135741608, 0.39751594863809259, -0.070913760841482487, 0.32369583784804079, 
-0.72873308299439143], [-0.51599165614126785, -0.40536443949367329, 0.62890046094847318, 0.19711690868032017, 
-0.83873428851251663]], [[-0.28320567096648319, 0.5208690699756795, -0.83539111333757621, -0.30102288980944381, 
-0.49293224885810205], [0.16960947936868798, -0.045194126462518858, -0.9579810315476418, -0.45973803747686759, 
-0.8861563360123863], [-0.61676569687885308, 0.090868705148684947, -0.90133302983403185, -0.44359006263416489, 
0.99712771244218135], [0.91687288441564574, 0.67558696375345884, 0.95486404637960343, -0.42847308428432451, 
0.89676333262581442]]], [[[-0.11341246990115406, -0.24969576755157918, 0.56908263316967922, -0.64770835583181752, 
0.41657637635311806], [-0.10425348304547222, -0.85332360344435143, -0.84275855446743941, 0.041281831980971351, 
0.93928409900728083], [0.12475539728746332, -0.57326001768690182, -0.34097898722513542, -0.81485807561429691, 
0.068399828951894781], [-0.22543564970257046, 0.020394873837392025, -0.21885560648273072, -0.19177591452869747, 
0.85677091796451399]], [[0.33037367273245732, -0.12120216544608176, -0.871598771446781, -0.52347487126137038, 
-0.99499281325700895], [-0.63017086142500323, -0.65363594673239112, -0.10168632246870324, 0.86122143029423515, 
0.1599821681323792], [0.65434564864681488, -0.46763231888772072, -0.11852738049053047, 0.56067756320632656, 
-0.64273294549833793], [-0.14406004015997209, -0.6156490721411092, 0.59483389084792804, 0.00016955875999102865, 
0.5794647863472997]]], [[[0.10694052353302252, 0.97564302960301341, 0.81225418717614217, 0.38442130590706358, 
0.50467694090447557], [0.66170565630000899, 0.0077585352989146816, 0.49500203124019437, -0.52689372726443895, 
-0.17551490494464717], [-0.65297322304433969, 0.22244153308969072, 0.24694382376883506, 0.70185883549334194, 
0.42496672111446276], [-0.6303606127727801, 0.088050805914949404, 0.93763194671316485, 0.262810375593441, 
0.38108603292585141]], [[0.4852956197801408, 0.38406064005604224, 0.13763806216350227, -0.10313727225367453, 
0.29218393362310069], [0.33787913508171008, -0.11388516702982843, -0.50836001257614494, 0.74998805751613418, 
-0.93077701584935224], [0.23050638298604675, 0.77281171686667283, 0.59537859644146351, 0.38289245012630602, 
-0.79182706737438746], [0.047391686793905041, -0.15764647268734677, -0.07316965171366574, -0.74699623961369954, 
-0.33576974660801095]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_0_offset0(self):
        sh0=(5, 5)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.32831433767592055, 0.4286111272258053, -0.93376232534905212, -0.011721851046949672, 
0.54742610553281756], [-0.93305296878330202, 0.57923749939644065, -0.79341178908172916, 0.010939900183319473, 
-0.70176590684694085], [-0.96413652228511681, 0.076681704593711153, -0.63539855623469266, 0.40683577855277453, 
0.049557908131688144], [-0.48352026198170028, -0.35503403895095254, -0.067225506936371859, -0.506924341673614, 
0.19165729409656573], [-0.37230779643845846, -0.69972071095280652, 0.43087296951431986, -0.68165970093609918, 
0.53230728532728677]])
        yy=numpy.array(0.913226284536)
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_1_offset0(self):
        sh0=(4, 1)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.98479539858488163], [0.87350492917539002], [0.61747051133581721], [-0.029629169421557355]])
        yy=numpy.array([0.52820276859841364, 0.13744538461535738, 0.45057075444011963, 0.88016649410525005, 
0.94889445663048888])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_1_offset1(self):
        sh0=(5, 6)
        sh1=(6,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.75856119720324133, -0.92703296980472261, -0.50709888794290459, -0.92530126451244699, 
-0.11106592452363895, -0.43633305021558999], [0.069537703906288328, 0.69297809165713931, -0.70955212499588893, 
0.27521267250665749, 0.1548383984960171, 0.029278449278885343], [0.1967546699287126, -0.24979392556034963, 
0.088321336385759563, 0.40719717360693553, 0.29920697058032042, -0.8576174381913142], [0.02857726898874513, 
0.14970342993965358, -0.081135362680279322, -0.58504524211759912, 0.39941645053046559, -0.10103955543896226], 
[0.23080918499626812, -0.65562613138135895, -0.64157092630807622, -0.25832792824283324, -0.58203915086947333, 
0.39129742034158133]])
        yy=numpy.array([-0.28031430857506145, -0.090813061038143461, 0.84169948054525623, -0.48866043139680149, 
-0.65250318827868803, 0.81372357858104172])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_2_offset0(self):
        sh0=(2, 1)
        sh1=(3, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.3580507239368691], [-0.53898761602289769]])
        yy=numpy.array([[0.76508197191309346, -0.5684608563596214], [-0.87970057487112396, 0.20046941369915938], 
[-0.16140160295365669, 0.54704403651550493]])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_2_offset1(self):
        sh0=(3, 2)
        sh1=(2, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.75688336094398445, -0.98101875789604009], [-0.48545508782843982, 0.42363153312807245], 
[-0.30946500603006677, -0.78507978601605855]])
        yy=numpy.array([[-0.40459192018135348, -0.33448847694827255, 0.83151169647604206, -0.99990094472353275], 
[-0.25979084499589655, 0.30568058855344282, 0.70446156517806835, -0.50537473630632435]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_2_offset2(self):
        sh0=(1, 3)
        sh1=(1, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.59323018944854944, -0.26582173297387057, -0.10206097228680977]])
        yy=numpy.array([[-0.52155895133831942, 0.67980681565044865, 0.24720927500033207]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_3_offset1(self):
        sh0=(6, 4)
        sh1=(4, 2, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.56789135083024678, -0.15302922972799182, -0.29334581391413206, -0.26195274704217697], 
[-0.24784249650552148, -0.65270118198654825, 0.79623475654675313, -0.30765633739797416], [-0.38902597262905814, 
0.26360758580216803, -0.11215786379917092, 0.037760015363922594], [0.0511305427723725, -0.99262810001050927, 
-0.90092187646007194, -0.53791334049817308], [0.6172523198169606, 0.80735537455260276, 0.89332759027623232, 
0.36527563784161643], [-0.084516889340532364, 0.97680948380068733, 0.1802554250454389, -0.019827016067265113]])
        yy=numpy.array([[[0.84226470752473048, 0.72261572021507892, -0.02573128014299142], [-0.70582197366008415, 
0.56947450316000592, -0.30077258620972436]], [[-0.43546766120188596, 0.029934501901479615, -0.19549658525336544], 
[0.69237391149840666, 0.4228987977333063, 0.73289725940903594]], [[-0.21627296920356565, 0.18158297059711459, 
-0.83770219188697759], [-0.96725561187625231, -0.22652657890222083, 0.38000729473228723]], [[0.98414645680569635, 
0.9747065008533724, 0.84290831752651507], [0.14539693231280149, -0.60738209333624327, -0.0015857109147463522]]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_3_offset2(self):
        sh0=(6, 1)
        sh1=(6, 1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.77952560868398968], [-0.21204901491122108], [0.0060402809681523451], [0.9263717034112593], 
[-0.96610002607719703], [0.44952721230521009]])
        yy=numpy.array([[[-0.90741198226459208, -0.71265784602697457, 0.10204874884029613, -0.73145019541625134, 
-0.86948869570308207, -0.14699545300364636]], [[-0.66126635458510652, 0.98302448352523575, 0.66506369858655989, 
-0.068528327902510178, 0.093457441643872485, 0.79005066771574128]], [[0.49456777046942801, -0.68150240928842298, 
0.73862134873274177, 0.54962224089259437, -0.88262960890209197, 0.7609423542445124]], [[0.14244305659186263, 
-0.20380584493155318, 0.37397168827839167, -0.61812670494292798, 0.91679399365653658, -0.7441271772333542]], 
[[0.65656112045860171, 0.10934059021586329, -0.31316670418388459, 0.24756849670261949, 0.44976569306764058, 
0.32213413009406677]], [[-0.51797824929149838, -0.19865839040925826, 0.80212221527627592, 0.098996756495719707, 
-0.65998623810944701, 0.62345242784580512]]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_4_offset1(self):
        sh0=(6, 5)
        sh1=(5, 6, 3, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.47172566061528709, 0.078935100541930092, -0.96210522208376603, 0.86759899961653208, 
0.65601235829865367], [-0.99089255717618974, 0.93860930957667743, 0.38179771145678654, 0.49887610182485709, 
-0.3866121315452542], [0.99422113436276671, 0.86802355456572067, 0.76424510223286068, -0.17303707510231159, 
0.87400040219358699], [0.67783971324158876, 0.81432678810234105, -0.97279957327957378, -0.87230070358292577, 
0.8988122867842272], [0.16728802483315297, 0.45839816631439767, -0.52775461378826694, 0.39112411914984335, 
-0.56520736218492029], [0.48436338527069456, 0.79959550501428023, -0.5885557280488587, 0.35973401756700629, 
0.26526108253756164]])
        yy=numpy.array([[[[0.090540573067726893, -0.021439658349278856, 0.0010015802900480431], [-0.91021841171763196, 
-0.72244952300579657, 0.7191454826128838], [0.29826609224268319, 0.23484645223864198, -0.84711792239821548]], 
[[-0.32043372014579807, 0.50299952098493494, -0.039756617896481661], [0.20975340311182622, -0.52049467511421454, 
0.87075348098703986], [-0.44831089848071137, -0.97850316806864623, 0.096021836963775087]], [[-0.12898224797332025, 
-0.98775355255827546, -0.98934513748534569], [-0.89638431195878421, 0.32558586594842209, -0.81688570673932714], 
[-0.01185744499255259, 0.29984253659256055, 0.85067235347756998]], [[0.24000625426078259, -0.74561864483904494, 
-0.23694287451520402], [-0.40510154813405386, 0.38519452676441146, -0.58973444712766798], [-0.64840940758493248, 
0.58240600317343794, -0.85010695215454812]], [[-0.88074093426560762, -0.77391928574296021, -0.14820962694529327], 
[-0.019025732244069893, 0.68016136915933956, -0.27162093721440672], [0.28172513212277073, 0.0102813948034437, 
-0.37435687589884203]], [[-0.8431347250995529, 0.053706582091624488, -0.28887509073319695], [-0.14857380054243841, 
-0.34412675263720782, -0.17925163959971679], [0.92315647872413775, -0.169709835410919, -0.72262129371434347]]], 
[[[-0.50196975412304967, -0.17202480449349622, -0.73147096983061988], [-0.46640664792493913, -0.69081627385030209, 
0.40899490991187304], [-0.062491383367106357, -0.027568901269507284, -0.71599088943832068]], [[-0.071918196642807741, 
-0.035545722842724947, 0.5281462075543375], [-0.25431448023613812, -0.04500355231529829, 0.23214520962436325], 
[0.42295098352445204, 0.52354893849379947, 0.066719344113555179]], [[-0.47049030197482922, 0.58106962708218579, 
-0.21814934679308329], [-0.34054634524351446, 0.28676759332442403, 0.46157446955031745], [-0.75325238791739868, 
-0.47614704441316968, -0.69813056265754958]], [[0.51569926524812848, 0.93009854339382447, -0.91952135877206165], 
[-0.059011449665577365, 0.91773778629838798, 0.61752646465576944], [0.50732110772221484, 0.22458187821527464, 
0.98894798584526122]], [[-0.59874021988734261, -0.7902474888502824, -0.77749366665573683], [-0.16879263654880305, 
0.29291279041288232, -0.18552051638338085], [0.90928188990348335, 0.059879352654672591, -0.59265433977005899]], 
[[-0.85125354004507492, -0.35617723858978212, 0.399817581079845], [-0.99777755225858167, 0.88543121632346078, 
-0.37466411439575098], [0.27795727136255888, 0.51954221009767121, 0.68582034150501769]]], [[[0.63429139750438934, 
0.803240391742158, -0.51247932579966604], [0.7766547260524701, -0.9961044703708144, -0.83355223990085814], 
[-0.27852246976961026, 0.69459280351639019, -0.059813543031333349]], [[0.12505039852370903, -0.4860329124679319, 
-0.033862285422065375], [0.93594248407895697, -0.81594055680724908, -0.33765657134191263], [0.066851877535606619, 
-0.70042267648975742, 0.22100937436105661]], [[0.16002133441806632, -0.32378626558135037, -0.13861289999460391], 
[0.13612559425692194, 0.74682282621444074, 0.59271313048892682], [-0.55349083527902532, 0.71461212716086897, 
-0.89387809280991859]], [[-0.8069367871373101, 0.39219873816519746, 0.71950287988728401], [-0.0034263728732759624, 
0.56408037193246319, 0.74365177145095784], [0.26780387059419608, 0.57893740395493509, -0.078685997749138092]], 
[[-0.25106075770610903, 0.38358934542753476, -0.28105682874160953], [-0.22412369537986621, 0.94983156686281411, 
0.014289505381869239], [0.99627027025428072, 0.53556201917826973, -0.50217528643731746]], [[-0.61015249964172913, 
0.95152208371434654, -0.74349471580800985], [-0.99754775051496614, -0.25652319510579757, 0.019577511256428703], 
[0.54617007341324886, -0.77621831020104182, -0.10367616964033077]]], [[[-0.34738521911448306, -0.94106578590704193, 
-0.28865723078711447], [-0.071715554608369425, -0.96789790103575757, -0.39056676395008871], [0.44905093577357524, 
-0.3520374628230496, -0.52386596574860844]], [[-0.87794425039308344, -0.0093450519941260524, -0.68290369124526107], 
[0.53703887021495778, -0.81974132854371606, -0.68713944715013109], [0.036962870031931994, 0.59416942748067458, 
-0.8867689774675509]], [[0.46177882589175967, 0.55863881148119265, 0.4662186762499767], [0.41723699675302117, 
-0.8328043601101256, -0.049981861765953317], [-0.044467947078313808, -0.71728287658457668, -0.64226979284521013]], 
[[0.71984876276525966, -0.80980420041523682, 0.30337301253840954], [-0.23031838222607348, -0.92198334468997434, 
-0.6378645003352672], [0.26527744050140889, 0.63871998995292989, 0.33353273412150175]], [[-0.65620821488953962, 
-0.35268721052735508, 0.36567251219657693], [-0.0017564193704233677, -0.84985262393509742, -0.86425459888817224], 
[-0.038132366512692029, 0.20298914922174882, 0.14647732890496634]], [[-0.75712032981843769, 0.4499598086364196, 
-0.8706311373994311], [-0.2962631932077866, -0.92634632777684556, 0.15780257872937065], [-0.6215094302760964, 
0.36651926504998911, 0.16104557463124936]]], [[[-0.96683832139952264, 0.66618952262684217, 0.53282949444692052], 
[0.5709846732258741, -0.88142581742887516, 0.97436711471416237], [0.60949016518593613, -0.82651373046085119, 
-0.45805343822570577]], [[-0.14335543060519451, 0.0552334778023853, -0.93219479493528823], [0.27338012426024361, 
0.29528033131405973, -0.069725358130424242], [0.76787019940918899, 0.58621862659599788, 0.92346766407533232]], 
[[-0.24482295276118027, 0.9504038867648994, 0.36246199111585176], [0.18852237386303194, -0.30071970735449383, 
0.61407959640810117], [-0.015719412599965121, -0.44806743982272845, -0.044744765439904466]], [[0.306082666587284, 
-0.37516895149071772, 0.47194403025627563], [0.57248529982865981, 0.84773649530527062, -0.63606537888521242], 
[-0.072121426180925763, 0.32991594395927515, 0.74158911557402907]], [[0.95579499865694295, -0.074255087503353856, 
-0.60944126810710419], [-0.82736601942539756, 0.51643359166162428, -0.58826137877076423], [-0.27602340091204636, 
0.052693022095508857, 0.89193987269821617]], [[-0.66780878786394737, -0.024872440896180459, -0.91372520408188174], 
[0.66151691358933795, -0.58842437189983099, 0.31560164484888342], [-0.12259961818782616, -0.18947440928555603, 
0.85217886879094218]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank2_and_4_offset2(self):
        sh0=(2, 5)
        sh1=(2, 5, 6, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.50667049246422757, -0.56508257037677612, -0.94848984390910873, -0.31750663178632643, 
-0.82865599848568317], [-0.93208808905517859, -0.40042002108399011, 0.22525943323786835, -0.18229479161967288, 
0.65807948180031017]])
        yy=numpy.array([[[[0.40414651645306177], [0.62741592426402737], [0.792051948767176], [0.50746003967543807], 
[-0.92636561274347251], [0.50621654779525294]], [[0.75938103139546587], [-0.3537834448306425], [0.63173503492801775], 
[-0.13317356057132712], [-0.039040830401476656], [-0.06362865176132515]], [[0.74514108484844854], [-0.16735254381155706], 
[0.36482380024470529], [0.2524521571228493], [0.86057322305243011], [-0.78175445640215013]], [[-0.18364691606510974], 
[0.076262553306061909], [-0.048497086654924804], [-0.50441391260187296], [-0.33257673118859055], [0.31770635197454666]], 
[[0.10996033753486079], [0.69102228357428253], [-0.13098993595512032], [0.64869110777111771], [-0.35421180388466489], 
[0.088395213803229788]]], [[[-0.21346061382438442], [0.98233244964611854], [-0.43411836297984485], [-0.54450544624629238], 
[0.47888352727010153], [-0.12892679809184737]], [[0.91533236215504066], [0.62594643976594111], [0.99543485928177655], 
[0.38941133463790933], [0.51654837963559208], [-0.26294204455507564]], [[-0.98953539610147256], [0.97002747745456386], 
[-0.91393612770114174], [-0.94571314605530277], [0.45486730879986359], [0.6480503159399984]], [[0.23153459988492653], 
[-0.38140895109654482], [0.14804295240219423], [-0.35111776030561992], [0.28967790618213729], [-0.48439329542340848]], 
[[0.056825424993940832], [-0.89380487638731432], [0.13828269263161741], [0.83332905530989998], [0.56407564639782226], 
[-0.68361472350798147]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_0_offset0(self):
        sh0=(4, 1, 4)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.030703485290683785, -0.80747199315356277, -0.93204704402451299, 0.46889843911156759]], 
[[0.20708526002069516, 0.74249474591344855, 0.41173921795684754, -0.91286835294801461]], [[0.68660540539865367, 
-0.7849410942279198, 0.15569991798627836, -0.67396663677733071]], [[-0.25137775717859223, 0.64836467503568018, 
0.35614597582157415, 0.77009358678503226]]])
        yy=numpy.array(0.174730634795)
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_1_offset0(self):
        sh0=(6, 3, 1)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.61130906116257266], [-0.77416633078217933], [-0.33559012575156077]], [[0.53402585098461208], 
[-0.95473878904666609], [-0.92584245518800268]], [[-0.57839604339554573], [-0.76788266385333337], [0.58550941085682306]], 
[[-0.76267081206400178], [0.053894668193436468], [0.63995814662772088]], [[-0.48296848098242107], [0.20584034634791681], 
[-0.28377868242675652]], [[-0.54594758690226186], [0.80126310026438397], [0.32132143706895566]]])
        yy=numpy.array([-0.2280793770357068, 0.9110193234845414, 0.91290767687018559, 0.18729626863004301, 
-0.50052876803468127])
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_1_offset1(self):
        sh0=(2, 6, 4)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.087613601453196965, -0.11127988610870587, 0.22837034220109098, 0.45850961884781483], 
[0.80001171062299647, 0.46491181181757457, -0.15673783297168664, -0.84842166536452956], [-0.060892928119377965, 
0.21331774464968567, -0.67062638309816913, 0.96747973507098695], [-0.63743201027820739, -0.6947861705088676, 
0.3412430684056591, 0.72585422378438724], [0.12141458359114288, 0.56374805719367216, 0.5012721844960133, 0.83735786266305579], 
[-0.26833882377795493, 0.90725348121937155, 0.27092746055016237, -0.72002358826931023]], [[0.52451030640251517, 
-0.95949109481163086, 0.0024323132308636808, -0.29085365198100788], [-0.79730690020825845, -0.0015092181660418813, 
0.16634566702058007, -0.099292642816858345], [0.51884546253506247, -0.57098594825831217, -0.83244780923137474, 
-0.43248312457244831], [-0.42976453572174567, -0.48835525080482833, 0.024127445539684489, -0.55613727641326127], 
[-0.59448294601992147, -0.046532881529117276, 0.14001729515596373, -0.73915423841763817], [0.60340029221359281, 
0.91728317647374724, 0.56738095359618534, 0.61091889969282165]]])
        yy=numpy.array([0.044628873027848792, -0.21970645647265608, 0.38567376885087112, 0.79288677220523551])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_2_offset1(self):
        sh0=(1, 4, 3)
        sh1=(3, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.098277207669267064, -0.57007722937133454, 0.65433650597029613], [0.61192199364019295, 
-0.71415983039764663, 0.8332897238291761], [0.046257200208623894, 0.40151884233380586, 0.3418329921669756], 
[-0.26187143299490545, -0.10358688103105229, 0.5002878087088185]]])
        yy=numpy.array([[-0.57508802747138632, -0.20046916179028451, 0.48532021906019662, 0.51586439732924161], 
[-0.61744111493756892, -0.75958240132105947, 0.2742585009148657, -0.48853826383749044], [0.075029564658172232, 
0.5793733764506519, -0.44816326887655022, -0.77556724937356103]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_2_offset2(self):
        sh0=(1, 5, 4)
        sh1=(5, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.59369944526899898, 0.89226772919147646, 0.49479123550595494, -0.25528471741928138], 
[-0.98497765198149878, 0.73439886177157487, 0.061908405020091095, -0.37146447648730119], [0.6671464047275073, 
-0.57513089375537985, 0.80775331218432656, -0.30252524140814097], [0.32463287815055919, 0.96409935426220694, 
-0.91951505599497629, 0.064801697571472028], [0.97490181177965485, -0.21381617903901384, 0.1518378923058048, 
-0.61335489004385169]]])
        yy=numpy.array([[-0.86634100457316832, -0.80230450460003322, -0.15667176480328915, 0.8818924631622298], 
[-0.2518085609332299, 0.77759251164524312, 0.18326779436077612, -0.20619870993086287], [0.46484612088044908, 
-0.73941983414764079, 0.572040348357989, -0.22134402792022168], [-0.41980546901479898, -0.11602289576555203, 
-0.25152286997815132, -0.77894586071151428], [-0.22058693972586596, -0.91192424209135226, -0.47801062569062691, 
-0.33827235245320963]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_3_offset1(self):
        sh0=(3, 2, 6)
        sh1=(6, 4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.57422057545271765, 0.75526653893052442, 0.81079370806402729, 0.93698616405991952, 
-0.54339365751711743, -0.25200139788724951], [-0.33253023867009546, 0.78515962501087788, -0.82131755572296994, 
-0.53677386910653335, 0.015303505067448375, 0.82518424267611223]], [[0.021396723183395583, -0.96024781080253296, 
0.68127557195838273, -0.52049140643058989, 0.27298812003301265, 0.0020240077632349696], [-0.46041198776468706, 
0.68971975949259834, -0.058050464182430206, 0.080249911235823257, 0.51876982357897061, 0.56761249200519681]], 
[[-0.22660244332154789, -0.57732238078932951, 0.083886973984427238, 0.96855099306060843, -0.43087775221643687, 
-0.54824301155548683], [0.24667821608774831, 0.35113438674547903, 0.983483027466731, -0.059384220370335639, 
0.91029040830438057, -0.6634627679471492]]])
        yy=numpy.array([[[0.89678849141267425, -0.70381808307355431, -0.52868279266882023, 0.53193545160829725, 
0.082838373929231324], [-0.99813438809249111, -0.66842507132828266, -0.80084183030934053, 0.95534010234518862, 
0.19539063270048995], [0.41248322801000636, 0.86067971236147045, 0.35239529505479861, -0.64073509420457575, 
0.61901520089816664], [-0.81088304615714324, 0.017763905310545391, -0.66334047099478988, -0.44811373052152259, 
0.83748882845843142]], [[0.47915024793678751, -0.82964033723519282, -0.88412659630808532, 0.57936253474466404, 
-0.43310025683602671], [0.73739563929733687, -0.84018737558592904, -0.33901402928833835, 0.37607520625791691, 
0.99507838839901241], [0.61078495979573422, 0.86290973827974882, 0.45968356473963556, -0.67020695437699906, 
-0.69940713088545436], [0.078685398761466629, 0.031384721994937026, 0.68131883693761108, -0.6390224455723359, 
0.57545710173862075]], [[0.73661153528221224, -0.0093753555674063183, -0.0060872893986509968, 0.21085736905354713, 
0.90147014005835091], [0.44475250899068586, -0.9374332416189366, -0.52535383503614952, -0.6319733290830325, 
0.91937973237489889], [0.11760157771047708, 0.6088138587044718, 0.64876499818101641, 0.37397168576644479, 
-0.99318588883695358], [-0.70507263017470789, 0.29173871449894739, -0.45716796330835718, -0.88450963213079148, 
-0.25778459703831125]], [[-0.55372305658085197, -0.61561498669912118, 0.030462836175874974, 0.13508774892946351, 
-0.54932385424029628], [-0.333890882626523, -0.18727182221537686, -0.46989696887511156, 0.14045439860891529, 
0.23812272243733523], [0.51596728918428902, 0.45391670662072658, -0.50435332319737403, 0.92717659535820185, 
0.92599111676958912], [-0.038938247399300963, -0.94779725394981162, 0.3372121829822321, -0.25785655510207439, 
-0.51887885784333743]], [[0.85409500809254157, 0.042086096894641489, -0.52227003113897985, -0.34776041449539563, 
-0.41238832948786741], [-0.94605499413534577, -0.51813750739307984, -0.16131166435461242, -0.79696078548312399, 
-0.1622200286778448], [0.043033311259488505, 0.17259155483018085, 0.4666818718690342, -0.3494986252021437, 
-0.10372039312488823], [-0.18917910285162387, -0.75695963675170375, -0.10453504873096198, 0.63263572286743219, 
0.37617595001814963]], [[-0.98040389162742247, 0.98925003500919639, -0.49597845248468198, -0.054905047431880849, 
0.43291364098491081], [-0.38177297979794278, 0.74474228600196435, 0.57607464421135735, 0.63109115783569369, 
0.90358924940702856], [0.15010116852126143, -0.40455183002381623, 0.11434665102777042, -0.6050871770677777, 
0.3679601690695784], [-0.68449545764539765, -0.24566723327920159, -0.25379875687119546, -0.015888973031557008, 
0.5842600301731764]]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_3_offset2(self):
        sh0=(3, 5, 5)
        sh1=(5, 5, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.97210493333177084, 0.020898400079180757, -0.92653261289841216, 0.42039662352587892, 
0.86765254063447617], [0.6907133559209524, -0.517553815485984, 0.85814529996004119, 0.50422112176023393, -0.29375070852832397], 
[0.37768930316151894, -0.90328817836758901, -0.037750496427062741, -0.83833859307160208, -0.9871325272868352], 
[0.84280147325222354, 0.45287477647420715, -0.33515871404088582, 0.78638153171519143, 0.95610235416784772], 
[0.59516164330744048, -0.51253508212931753, -0.1314541796178994, -0.60812256069306714, 0.005271114437214397]], 
[[0.1350474349377746, 0.92002814067983496, 0.11082772432523358, 0.95022672909808037, 0.087928751911553027], 
[-0.80743982390110869, -0.62035633341515983, -0.59659511471018112, 0.94536900928946221, -0.69470029455725313], 
[0.63159172652207407, 0.34712440512702702, 0.091400070718341997, -0.42507012653813669, 0.15641592062313481], 
[-0.69145932348965644, 0.80478112615624431, -0.63551751485727026, 0.40031821556124014, 0.23359777651201252], 
[0.77575199914239068, 0.88842113945921319, 0.41321396341016858, -0.53606989119223458, -0.3270835918186068]], 
[[-0.17251169495981156, 0.28539657744512525, 0.072575176492963189, 0.62765882694557917, 0.8899014503296323], 
[0.87793091491911746, -0.070357895563635475, 0.6724737377470893, -0.76222287971443747, 0.58042921195609787], 
[0.40156493619998557, 0.32039816314856884, 0.88684906641291716, -0.82619180258255409, 0.9414934871073124], 
[0.25419615063588696, -0.169198377126766, 0.32543793423400835, -0.46890728074721766, -0.78831575420793265], 
[-0.48561126508297581, 0.96335228269668471, -0.31353217649716791, -0.11291379120260259, 0.46221389116568479]]])
        yy=numpy.array([[[0.35937170811856478], [-0.75168195298596774], [0.21159408144193703], [-0.48925446136754203], 
[0.029234368307212266]], [[0.049016852645582087], [-0.94985864683552346], [-0.57914425575318229], [0.48059901737344868], 
[-0.57131824264096309]], [[0.00050497468508736709], [-0.96797102452003903], [0.86095571203602539], [-0.26789779591139351], 
[-0.3104760644168727]], [[-0.39198202162749718], [0.23144890810765051], [-0.36562880769889983], [0.26301366548420413], 
[0.15896355575974619]], [[0.78235387364617104], [0.23169582314919968], [0.046207586438298032], [0.033683598833781359], 
[0.28188126720011097]]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_3_offset3(self):
        sh0=(1, 4, 1)
        sh1=(1, 4, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.032653134355640878], [-0.12895343850860241], [-0.84621944587379838], [0.31726177168581904]]])
        yy=numpy.array([[[-0.47762057231087618], [-0.4852513198885906], [-0.30287498034802551], [0.83050657783509685]]])
        ref=generalTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_4_offset2(self):
        sh0=(5, 3, 5)
        sh1=(3, 5, 4, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.56904636100195027, 0.52417851065027277, -0.38820673702744801, 0.21142679362414163, 
-0.10992661017609273], [0.33787589047480626, -0.48273558048856846, -0.4257689005605223, -0.75433434590724935, 
0.59651952178654444], [-0.80405919790002711, -0.90291869418551474, 0.30219315265198299, -0.9601688901607115, 
0.66705585698271941]], [[0.85579538501262675, 0.57660381849119857, 0.80521718703882117, 0.6975489296844033, 
0.75211668864239978], [-0.86015727546158938, -0.51782657952014777, -0.41209396209748372, -0.60605357988059438, 
-0.52931461713241768], [-0.87123868525397996, -0.54932022045262441, 0.68477496538085125, 0.74362120626736883, 
-0.17176170590348794]], [[-0.021574897876409649, -0.62685704340254023, -0.58281867552537436, -0.088441989393366116, 
0.92107188712758115], [-0.088143626768824035, -0.82209369434337254, 0.74874237433308721, -0.59578644479118492, 
0.93264433936364077], [-0.89349266125629145, 0.85631049738195686, -0.94397840872975092, -0.73837071602760052, 
0.71401619176888409]], [[0.80671236682090419, 0.86203353773323266, 0.37608041825138749, 0.48838550857617458, 
0.55981916730660197], [-0.73096077846243057, -0.37222672808041435, 0.52541496455523129, 0.90132802581811067, 
-0.13651786011594913], [0.53755457509327154, -0.56490126460895396, -0.53615854843567212, -0.059644080107389197, 
-0.79806860600480478]], [[-0.82903078462037616, -0.91943938125141234, 0.30865742650070915, 0.19900441738155528, 
-0.38964592420655664], [-0.68134600272096857, 0.30043204893457931, -0.67394650058372796, 0.57483671554786198, 
0.23982499541022029], [0.67797420776884509, -0.71988605781825421, 0.63903671449142507, -0.63870892564507731, 
0.82045466961588454]]])
        yy=numpy.array([[[[0.95115913843809552, 0.4459413405123549], [-0.40791943259749441, 0.50708149483958143], 
[0.84273697175550755, 0.66432165552953459], [0.85588431381757979, -0.62942454987735541]], [[0.33869763918177953, 
0.025028012934277033], [0.52338887007319301, -0.033507856186596019], [0.32195081399087178, 0.087228701435298595], 
[-0.17993890319020189, 0.20636283870314376]], [[0.89517068076121609, 0.62589760562806407], [-0.71281947461757955, 
0.5893548787843732], [0.24961175265386015, -0.3848491591764458], [0.35698026189337995, 0.89646575158820685]], 
[[0.9886420453846736, -0.4260108877924409], [-0.29907754375089857, -0.085842752395373978], [-0.31773651482144261, 
-0.2032958412790018], [0.096117512707914798, -0.58613390797410636]], [[-0.18010068030738879, 0.24553654956211068], 
[-0.506655993361115, 0.64841443653232478], [-0.528753864554939, 0.27035036412934543], [0.46919949095155733, 
-0.50051463768647286]]], [[[-0.98657470005127745, -0.63491418556659074], [-0.96393604057790383, 0.10686283589063583], 
[0.63137682470323186, -0.012334691459104041], [0.54772544461423411, -0.4862254908694974]], [[-0.73397740926658517, 
-0.39883625541409384], [-0.58767881428061464, -0.93288881576468974], [0.29330836171655816, 0.88699793314173281], 
[0.44021439571566812, 0.83952582751853044]], [[-0.29828697057545317, 0.55113060337932773], [-0.2489514436148299, 
0.6021186755826109], [0.87180742005047773, -0.64428944262570731], [0.74677609827813907, -0.092843438949603785]], 
[[0.85163385934742375, 0.26750941090889735], [-0.65868408546455792, 0.68579269787292541], [0.28972435936610808, 
-0.597409212821846], [-0.66770337329851803, -0.33735981097984835]], [[-0.79970957341748172, 0.56769567857844483], 
[0.46853475648502951, -0.11442596074286704], [-0.36686272818937926, 0.58098106469309463], [0.54712148128665916, 
0.38785411577923612]]], [[[0.21319980799055815, -0.99347377862801167], [-0.76356539216567354, -0.57054101732827234], 
[-0.58164664995270177, -0.63915417313835743], [-0.96085944744171226, 0.14061814184312094]], [[-0.98026700711219505, 
0.18339653098320308], [-0.027209450108061528, -0.70610237746090632], [0.94653837420297937, 0.53464417391308561], 
[0.83552531198830815, -0.92790704230981791]], [[0.51229090374315089, -0.60785658799493114], [-0.7483534554415523, 
0.62677145001410506], [-0.67240232294928082, 0.060965354179338904], [0.39682948970729126, 0.51297826346296693]], 
[[0.34121985751814932, 0.3664579870348299], [-0.56852300204909167, 0.83743908452361993], [0.28786803355073665, 
-0.92127270916725035], [0.24992834818528764, -0.92821634996499514]], [[-0.31591502244605962, 0.97840752655130703], 
[-0.27189088833158115, -0.72970275507172122], [-0.17956314195654532, -0.67310603969743066], [0.18615248719069877, 
0.89659460917498479]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank3_and_4_offset3(self):
        sh0=(5, 4, 4)
        sh1=(5, 4, 4, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.9892455663385713, -0.47817238009045182, -0.3347997624690302, -0.19568713609011557], 
[-0.79245728868785892, 0.48189760826456962, 0.70615487299667201, 0.94693022296757645], [-0.50455863218618191, 
-0.79488810235998675, 0.42931608539342303, -0.91012543072745555], [-0.30607120165706481, 0.18974266505262594, 
-0.78308929167748675, -0.51915903307422662]], [[-0.67757692038733452, -0.095542852078508211, 0.9351295380841198, 
-0.98372640977715342], [0.61950266413186883, -0.70006653901729687, 0.99025554770390034, 0.57200430337836017], 
[0.59261549340058584, -0.47900633564953754, 0.68924348099605526, 0.66999282757105405], [-0.96138200131308449, 
-0.033430310542805319, 0.57128900841695107, -0.57084863327640978]], [[0.79914211207029706, 0.99071406695293063, 
-0.61659031253122265, -0.41338544275737332], [-0.6919954716398915, -0.50890406424925416, -0.6713918838082169, 
-0.49732504657246301], [0.30443836034546523, -0.39351310138638973, -0.11333366244816845, -0.44811307718502635], 
[0.036013581479172219, -0.78325753271264431, -0.9869783727185597, -0.34630851095670034]], [[0.91786219784151735, 
-0.75118942234664776, -0.20120362018872595, 0.78415058864692067], [-0.53773825295431266, 0.69677477934643983, 
0.75098076570703487, 0.42527246897461568], [0.49748713304708314, 0.87678559949778223, 0.9212949671381232, 0.96389600320834146], 
[0.73034790330530286, 0.42614023143587709, -0.70443927105492365, 0.85204804704078829]], [[-0.46070649600616265, 
-0.70097939461394709, 0.93560861656013627, -0.4635695299774758], [-0.18465728816672833, -0.9582485445071065, 
-0.063448052965652035, 0.10071444203327484], [-0.75958922248145533, 0.34758346712286503, 0.44414361017600923, 
0.58106974468680561], [0.74653810785092101, 0.58718499524904111, 0.88104738871557875, -0.53294119072031299]]])
        yy=numpy.array([[[[-0.085032969047221796, 0.68030086949926805], [-6.9898805543866516e-05, -0.53959347738099872], 
[0.4043448008361592, -0.350664632406118], [-0.40101787077740525, -0.16763535758449266]], [[-0.22743572380714161, 
0.82919599709665781], [0.75100703044325123, -0.87746697425078413], [-0.8098810656724138, -0.82570183074272152], 
[-0.075971390511780035, -0.62904623014594274]], [[-0.6017212077079146, -0.97049533577050684], [-0.087669308972781801, 
-0.61648415177773086], [0.034299204150757268, -0.7700778584271819], [0.18661681553399068, 0.26419755756971353]], 
[[0.37134268096334266, 0.95329107383261746], [-0.067859480077878231, -0.99619717119986695], [-0.91204933236023589, 
0.11157235387084952], [0.59488499305889908, 0.71874496148527078]]], [[[0.74124250182895723, 0.88221955604575641], 
[-0.70671123017938187, 0.26961734276794846], [-0.17600110524887169, -0.9237666170686456], [-0.51622091696284467, 
0.021958344315271994]], [[-0.077185732880006741, -0.77456233598565505], [-0.31758657231471399, -0.82660337481950208], 
[0.47431195997444164, 0.27794145254853397], [0.38104035773974942, 0.054165996959011808]], [[0.21772434546827579, 
-0.15335504351366058], [-0.67749995625423987, 0.59027526048984247], [-0.028412471225639946, -0.95421554453864843], 
[-0.71690305911861296, 0.97975644435287568]], [[-0.067143860448145354, 0.25981537120853915], [-0.79680461808521508, 
-0.98422267171091149], [0.90069625272026266, 0.63985884998914178], [-0.96209569697101593, -0.3787026812101808]]], 
[[[0.54963453965722064, -0.93551541826545748], [-0.11779265143641693, 0.61805499305525391], [-0.79494280437861087, 
-0.43103955695297547], [-0.6956327838655032, -0.65459982605370404]], [[0.86396692595192981, 0.48013174966040051], 
[-0.70376692733642066, 0.89024419779731256], [-0.37000014084753707, 0.28123641451571291], [0.88487732692102461, 
-0.44427068937801018]], [[0.49154055672765051, 0.59269544530813079], [-0.27647805471305453, -0.68197817802801097], 
[0.84248744404221609, 0.86436825894377334], [0.55524916207060615, -0.36060722970540859]], [[0.22125584226861861, 
0.57499120581011276], [0.64895315972803469, 0.66732145434612455], [0.85780060882052322, -0.63662546859004121], 
[-0.94600750098186048, 0.1789643470742639]]], [[[0.78541357408241375, 0.86665088671432566], [-0.48169942559864132, 
0.69523912356988138], [0.3171023733664573, -0.24370294129123038], [0.31389007207193842, 0.92617559753670164]], 
[[-0.80928773995481218, 0.5316222693340189], [0.60818161116375435, -0.20554591426140711], [-0.14770804861276976, 
0.28169550302985868], [-0.56944343282487098, 0.12218206883163774]], [[-0.9458386490631947, 0.3704339755597037], 
[0.86253599176558438, -0.74461478908119205], [-0.3514308568113762, 0.21873815546478848], [-0.3198876441569054, 
-0.041032888543660162]], [[0.65499414147986124, -0.576443397736355], [-0.4246138925726286, 0.28291265851711445], 
[-0.94649214120849168, -0.78199939752170033], [0.72651871238475696, -0.55629738350824387]]], [[[-0.61262774647375196, 
0.58939126291936739], [0.5047921262106474, -0.86780938269957297], [0.611053710467931, 0.84638260449111091], 
[0.12079667896274682, -0.18125799415895916]], [[0.8790391838757714, 0.56022651764458975], [-0.35026097353495356, 
0.44596355089263606], [-0.53280791930164528, 0.42131847198850036], [0.74058834714277766, -0.67941662348275345]], 
[[-0.084803487965648516, 0.39411810864038821], [0.60606354167182075, 0.56826985240689765], [0.25344261357126352, 
-0.28565544673047238], [0.3862548721799548, -0.71325294116356264]], [[0.71066060790441066, -0.2659121404186906], 
[-0.66846932059152553, -0.99356071393662404], [0.87058889457377475, -0.38926929520715725], [-0.7591890232558578, 
0.62823370748054952]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_0_offset0(self):
        sh0=(1, 5, 1, 6)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.84017639489458151, -0.84650109769987703, 0.42304624205505137, 0.70318660821058665, 
0.84702884382274313, -0.63873861518834141]], [[-0.0095131164166843973, 0.0069505263732572153, 0.68991254377545919, 
0.22730212546950135, 0.51275810808698852, -0.18459213780817629]], [[-0.75307891019007478, -0.68533231699144559, 
0.95086762153021831, -0.28230322047865752, -0.98650581785836033, 0.19719993434519822]], [[-0.73997648790750792, 
-0.70018555910390368, 0.2979590705650772, 0.75318249965036599, -0.42612463746155593, -0.96859400941423357]], 
[[0.43603961563035809, 0.44670826375564254, 0.21377017854462288, 0.5773708338701018, -0.74055953071850289, 
0.63156443059569489]]]])
        yy=numpy.array(0.618785225727)
        ref=generalTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_1_offset1(self):
        sh0=(2, 6, 5, 4)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.29113511026383976, 0.46229765554712321, 0.59088773786779392, 0.93042519177148186], 
[0.55679575912575752, 0.7970768548883842, -0.69237318468805564, -0.035742891371115748], [-0.49898765771516418, 
0.5731160052258264, -0.49592533032557462, 0.53021506833568277], [0.69259869765884274, 0.54487304825288274, 0.70183623207406476, 
-0.4726495707635614], [0.97084227765866071, -0.6253448180680512, -0.3183079449243893, 0.63958595371604043]], 
[[-0.31405632855037324, -0.97872601455236974, -0.98308394051420711, 0.21879372931482011], [0.40029583654119882, 
-0.067713495734489371, 0.4990292393138529, -0.90578923485653995], [0.91909355982122842, 0.83747985135012182, 
-0.08126146988971672, 0.91067712797336431], [-0.29621534346057077, -0.83173390224971522, -0.9930111474283092, 
0.48561335508921832], [0.236463311359979, -0.1568749316102469, -0.41185716413355444, 0.024392146394177416]], 
[[0.39085092006053923, 0.52822719145990016, 0.23626426214486473, 0.80797784754033275], [-0.016223836009182424, 
0.16724207173699446, -0.40236194850175955, 0.19024102127446474], [0.24856523062472613, -0.49489704680877122, 
-0.79124619261893558, -0.6130544727810272], [0.19302383228151099, 0.87759920621285148, 0.18096136522927231, 
-0.46446456176322704], [0.89599924793084385, 0.39709228277757225, -0.2401592090046778, -0.51371553690733252]], 
[[0.39313891690622582, -0.74960617342296176, 0.16269165535238739, 0.03289275705787964], [0.45774006787642474, 
0.27273319969787368, -0.07236564475189744, 0.30851917871757029], [-0.39571815407746391, 0.35998582581822292, 
0.70921534906311456, 0.34190303217430995], [0.60357310294624678, -0.43863268976677383, 0.92662854117982207, 
-0.94842656501413969], [-0.29959923461958882, 0.86253642091769667, 0.17927290250402095, -0.07637838389356677]], 
[[0.28221927869579977, -0.563122476343074, 0.78672295539261095, 0.14355116815185154], [0.11290798896640064, 
-0.27328643638671313, -0.91778473165543462, -0.20629665456397905], [-0.62371142259044299, 0.1719413016969884, 
-0.37068744050444624, -0.84443318578821813], [-0.4331137285190827, -0.96244159837776566, 0.93996721479167644, 
0.83218188905631196], [-0.8637416243990208, -0.94637064625000544, -0.36664252552577392, 0.94517638241514779]], 
[[0.49784871853992541, -0.66384737528794169, 0.18560831166652592, -0.078131273922674715], [-0.86142630144730314, 
-0.14534655881913761, 0.86763242528490814, -0.86009797138827948], [-0.70291169721448599, -0.73284719301472667, 
-0.40191319299472239, -0.15337677460800814], [-0.81166469995634638, 0.53426905548468251, -0.39044612478471485, 
0.70322288485022866], [0.71164575434034694, -0.91039565924927013, 0.4143669852487375, 0.84859638460495534]]], 
[[[0.016126940228555764, 0.60472121893598607, -0.36482118662184448, 0.92389057042931633], [0.34812089581028882, 
0.81991415043831095, -0.82141881706782294, -0.32280746388112092], [0.76520676818625, 0.33928558126896946, 
-0.094325065846385892, 0.023955447077503855], [0.4485377703002289, 0.84248815890227502, -0.58531657749214916, 
-0.82614698090151517], [-0.96689287584197148, -0.043330886354145282, -0.57346694519058716, -0.49954486762644978]], 
[[0.71703769676409435, -0.44105328011903877, 0.47577416638794423, 0.45159833959792084], [0.97313010139592815, 
-0.40373514201534988, 0.16774804945941502, -0.88279101095751589], [-0.7702441048529074, -0.031324075825988951, 
0.55289551566030193, 0.090151567425839518], [-0.45162588193928443, 0.21930976685485315, -0.93093141860155337, 
-0.47841671137204078], [0.92914043489782872, 0.43286881403402067, 0.85866199713675861, -0.98649358419473598]], 
[[-0.11243136751986493, 0.05669240408381393, 0.85868614923765363, -0.82466521138031657], [0.37646798726183439, 
0.81056254949627315, -0.45255797246658713, 0.89206966069759219], [-0.46667765220339863, -0.9089826452207479, 
0.48703960304854954, -0.7473438181451002], [-0.70435788803400423, -0.52617728546654208, -0.50659454268209148, 
-0.62714596794418176], [0.090366337918861062, 0.8677387186233525, -0.4909354209810235, -0.50661465530825356]], 
[[0.93748220244738878, -0.7440609156897422, -0.23314611343163683, 0.22188315209977416], [-0.95509248359663701, 
0.1006311423307189, -0.85519248386156277, -0.29522162160925203], [-0.15510446546468337, -0.2395223734380294, 
-0.11280351224663554, 0.58684583436701976], [0.97227025926897648, 0.75669771142232034, -0.54308971208818302, 
0.84407024110307804], [-0.83164296054164155, 0.1101592871193966, 0.62336111961314811, -0.4503454421669344]], 
[[0.18667356519390843, -0.13808052663606873, 0.82892528097320928, 0.2005946649581003], [-0.2541216170270848, 
0.6484068341713225, 0.78225277750252875, 0.30652885679399611], [-0.3523507237247363, -0.60191489588141245, 
-0.53094455463442514, 0.84877628619846313], [0.54275401273261048, 0.075401798483744953, 0.3545790840724683, 
0.81280673867963271], [0.74381648535298583, 0.83062855500710198, 0.64009074814710276, 0.49222289275089359]], 
[[0.42185586209015447, -0.72193902907960128, -0.68734569663430167, 0.33451005678645829], [-0.58186038830143283, 
-0.43737051666815141, 0.91017681826642849, 0.27345359855631579], [-0.83592882041288674, 0.67550681520961309, 
0.45085150334087198, 0.71399632661250201], [-0.33717827611345985, 0.63057886089484283, 0.99169901565395158, 
0.66928917807501809], [-0.092778539142545968, 0.45303171547496124, 0.068097806044977505, 0.015925925224931303]]]])
        yy=numpy.array([0.12165730124208629, -0.34859636041489117, 0.99118230855980083, -0.45784036085226965])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_2_offset1(self):
        sh0=(2, 2, 2, 6)
        sh1=(6, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.47766540219935094, -0.92295904215445268, -0.080652645559102343, 0.55025159357333808, 
-0.14950803869629192, -0.18603469973288833], [-0.87271588262406552, 0.76227902638745948, 0.4824551062013458, 
0.40382822270113938, -0.99397442167099381, -0.65676460910864609]], [[-0.75271558798917604, 0.90709984341532768, 
0.30077765197672335, 0.48447123619276655, 0.2872927025868266, 0.5136405687529273], [0.11556951050133724, 0.17700531043515366, 
0.14288267920823516, 0.61845629640103184, 0.58766076835289627, -0.79828484773810859]]], [[[-0.094530348405517328, 
-0.15619803707292235, -0.25244658831403899, -0.032420001967008893, -0.86286178469195685, 0.63244733283264987], 
[-0.90868814747847315, -0.064242037845542166, -0.92505615947500019, -0.91970631909490619, -0.11607241566418081, 
0.976319709472939]], [[-0.0024027602687943794, 0.8442480448934091, 0.057226919748089689, -0.92629468699436179, 
-0.93266953278681131, -0.83326098460887077], [-0.37153131803194839, 0.6583642092422346, -0.53154405373936897, 
-0.755434611564336, 0.035607052122304417, 0.47819402176090087]]]])
        yy=numpy.array([[0.24039286663589965, 0.57411837861251791, -0.69352155433896434, 0.7848460827139947], 
[0.398331753219483, 0.3660120997621259, 0.83369419666138134, 0.44960639558413829], [-0.56831176646686221, 0.52974141348190829, 
-0.21873163468210688, -0.37959380424710765], [-0.035060503748099991, 0.97086872564394988, 0.98625832418956105, 
-0.29790742184786145], [0.85113034633123097, 0.79668114237782506, 0.55549606502201376, 0.68029206259511055], 
[-0.69928180731998424, -0.72080446178836799, 0.22855342687351654, -0.49057649439771334]])
        ref=generalTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_2_offset2(self):
        sh0=(2, 3, 3, 4)
        sh1=(3, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.34027207236028678, -0.50838244985120129, 0.45790347210854998, 0.43798583054273599], 
[0.148887145639226, -0.52171809108898848, 0.45277366582884904, 0.8000287491578737], [-0.17613052241181593, 0.19373067549768108, 
0.86433625565149819, -0.20195098618790874]], [[-0.75270285017957694, 0.66672903299819275, 0.63907732518836657, 
0.96799908212063501], [0.33864087551343314, 0.25773501829976486, -0.52328913696881107, 0.20583615655628051], 
[-0.89571100164563688, -0.9399684471154901, -0.21017897434154564, 0.10677287810718994]], [[0.68139886858009846, 
-0.17454760007829018, 0.94663345545412603, 0.64178100666593463], [-0.81003656197198959, 0.072498985731899834, 
0.28912969985638792, 0.35134856414507443], [-0.9470688310707871, 0.28897434859947579, 0.49113192788253768, 
-0.22232185373588198]]], [[[0.096471090361194944, -0.77298817535872488, 0.42816282034681596, -0.70699737171059884], 
[-0.98369050459377871, -0.82361456602881811, -0.41892977250468233, 0.99775666448522493], [0.65712895586992937, 
0.44286191704268929, -0.42660613604553443, -0.92823616020476929]], [[0.87822373247910446, 0.16755293732722909, 
-0.59038263883234388, 0.093740562718347853], [0.64703971395678428, -0.86208085336084461, -0.75796389342610015, 
0.28764491345848131], [-0.36292727317667262, -0.31835376863793119, -0.01862898535334212, -0.33718157594441878]], 
[[0.51399337366187781, 0.27483775155242518, 0.82944694995462598, 0.62552523087356016], [0.017061282810787892, 
0.98268876418248619, -0.28457515009335621, -0.43339788412157354], [0.20235431736779663, 0.73898564732407102, 
-0.27964171131788929, -0.16370044723706023]]]])
        yy=numpy.array([[0.207324407053608, 0.13096606427880308, 0.42518782232005803, -0.91162839542543495], 
[0.94082585327110069, -0.020094662997587331, -0.28789525114129777, -0.42697224559203195], [0.6234471989079462, 
0.25616644419619461, -0.087014931383018412, -0.08605668078648776]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_3_offset2(self):
        sh0=(3, 2, 1, 6)
        sh1=(1, 6, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.88899824146708739, -0.20442479904655952, 0.98932509499368293, 0.37877232626868818, 
0.71124515446978509, 0.5615932331275828]], [[0.14375952203529807, 0.62515837073242309, 0.23951934428568622, -0.790764602834783, 
0.69906751899308883, 0.89080949569035162]]], [[[0.75462120574975389, -0.35168504663504208, 0.46886249229110666, 
-0.1923629079124074, 0.5752369839452971, -0.0092781587686867617]], [[-0.12252235916596521, 0.63692905618040441, 
0.83299165795024943, -0.32395143678767724, 0.88698154142843544, 0.021827771885164493]]], [[[-0.214813755594917, 
-0.75014213032712895, -0.11404588473182753, 0.95694606749320732, -0.030568964190160308, -0.073054421151125482]], 
[[-0.67615529566808763, 0.79661478958115839, -0.84272073541045889, -0.74012981558609936, -0.67522993443650803, 
-0.95898891225441751]]]])
        yy=numpy.array([[[-0.748548915978694, -0.74025482697757816, 0.53410990718395635, -0.76209555288462827, 
-0.9006127161998887], [0.75890539745585106, -0.47920189782119005, -0.89483286979477827, -0.27838832307819517, 
0.61420302254164438], [-0.46091816979179678, -0.37838959175297737, -0.2843510112114056, 0.93855493548670599, 
0.18620210527437209], [-0.70936487136462856, -0.81448845882387855, -0.2687512389403548, -0.3580807419564247, 
-0.78369736588589123], [-0.41143288625874308, 0.23540026095776478, -0.50336994160797244, -0.63642334399375411, 
-0.51051984809447104], [0.98959258092005187, -0.051173965543319877, -0.29936544012979582, -0.31152885184748391, 
0.19555678206820937]]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_3_offset3(self):
        sh0=(4, 2, 3, 4)
        sh1=(2, 3, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.94552672665065285, -0.88818521452880472, 0.9177667676054897, 0.80882640304489395], 
[-0.60377834525288887, 0.0045556163431130781, 0.32832767960184905, 0.19690007684504107], [0.57213494769701767, 
-0.80378473295462949, -0.15997912072500009, 0.8884613285500802]], [[0.20468082591282588, -0.55747931479868629, 
0.22214322056932434, 0.80737194050838612], [0.51592721196250113, -0.40604299220696682, 0.82295213719891125, 
-0.37580019169041479], [-0.29608402444211079, 0.038955026542308868, 0.19512515019578491, 0.8802808568649354]]], 
[[[-0.30969566752328226, 0.075181577258429666, -0.38782501960570004, 0.97547922844775137], [0.93685702410259353, 
0.45467330848946919, -0.94567155397618019, -0.57456674929826712], [-0.51658718140439563, 0.56714228006659861, 
0.51639719705108456, -0.60700514046283494]], [[-0.54859636709671133, 0.51809627037902817, -0.31066121474291597, 
-0.20429569807708958], [0.45226243950202116, 0.36825786279642791, 0.043125273375412831, -0.92497298277941731], 
[-0.22910259206011396, -0.58385335159277285, -0.24681799217005995, -0.179452323215886]]], [[[0.31167177604315688, 
-0.56359470210317508, -0.16644825178354328, 0.97385090692972187], [0.59256432933627368, 0.12326341923468842, 
0.79815656055475692, -0.71396167820944245], [-0.68509623422746468, -0.11036446719551529, -0.59484071817670525, 
0.79189294295235002]], [[0.29451370321991988, 0.87201603311462361, -0.84962723450785038, -0.32639787127722819], 
[0.39990674288352279, -0.22853598886656568, 0.43172656978047863, 0.56038390045088726], [-0.58907053938404119, 
-0.81773925097203581, -0.61527855944209398, -0.91383854187478897]]], [[[-0.33303573687327948, -0.33447950894749834, 
-0.59163636009511378, -0.26217886374314192], [0.89272237180431735, 0.98958551006122342, -0.092887975760077568, 
-0.083077686572530451], [-0.55222442842688713, 0.92848822133871334, 0.46752671566837223, -0.11496809802532226]], 
[[0.29989277399489667, -0.22308012439026803, -0.011614635013909691, -0.7313858702086895], [-0.680110015737454, 
0.62619884594693853, -0.89421660864698871, -0.45878643422242615], [-0.54257528236034647, -0.14530463024128926, 
-0.76573758581152696, 0.85678662373836723]]]])
        yy=numpy.array([[[0.42399187163939578, -0.41616154251832427, -0.15662271584075693, 0.56182513919190336], 
[-0.54445605979646161, 0.69490624371743825, -0.86386365434238188, -0.73484456052486968], [-0.52461588977480966, 
0.2375119004032813, -0.64203933251192669, 0.67248543166264874]], [[0.46631839064101221, 0.3110721595243815, 
-0.50980746557078116, 0.16210722867081451], [0.55972533235751842, 0.30100689351376975, -0.44001592420202384, 
0.36034931047928143], [0.26260582265832255, 0.79625782129920863, 0.60190131621554666, -0.32345887422388508]]])
        ref=generalTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_4_offset2(self):
        sh0=(4, 4, 2, 1)
        sh1=(2, 1, 3, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.90232367772589495], [-0.19745714631096023]], [[-3.6238194704774074e-05], [-0.72406790233806739]], 
[[-0.084330284904369002], [-0.22534259720121752]], [[0.4449286642048933], [0.98132436655464295]]], [[[-0.37780547624822747], 
[-0.99149216394784356]], [[-0.26455430984339334], [-0.28294498412217717]], [[0.96634163565233089], [-0.14392185634379429]], 
[[0.61607562970639407], [-0.39306351435163367]]], [[[-0.43767847994012143], [-0.80430761716423693]], [[-0.088779847319703675], 
[0.59041844527407106]], [[-0.4849847279065751], [-0.16214135332016633]], [[0.95315448744586351], [-0.026567048788048941]]], 
[[[-0.47146208734591433], [0.30498995066486145]], [[0.030141870572571561], [-0.48748936614843252]], [[-0.82644139491876834], 
[0.9396614701378736]], [[0.41876139073867047], [0.28894675292894467]]]])
        yy=numpy.array([[[[-0.45623106585276529, 0.38046970249045509, -0.31622814574304603, -0.66468820203967383, 
-0.42683487510255569], [-0.12698017183829369, -0.093113816418858297, 0.77722337527962404, -0.42820375527819299, 
-0.744909562047116], [0.29624697149553114, 0.57970295430496277, -0.32604839099336069, 0.58922194836120578, 
0.64954264071090861]]], [[[-0.66144862286816153, 0.605404996967422, 0.57929522566259495, -0.43151915103505467, 
-0.63205358346260421], [0.0031366675973272251, 0.22027027350290962, 0.50310434787166303, 0.057403968634963665, 
-0.49491069740184157], [-0.87224786992031333, -0.61823408750211439, 0.55656040096569481, 0.12598165808437889, 
0.080307745600085134]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_4_offset3(self):
        sh0=(6, 6, 1, 1)
        sh1=(6, 1, 1, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.29748224377954746]], [[-0.40002039306877579]], [[0.7514302226645877]], [[-0.37340400437081511]], 
[[0.0057004931898410049]], [[-0.29672482573892256]]], [[[-0.66400056472123792]], [[-0.35009868804512867]], 
[[0.19893628621475901]], [[-0.30672054370206281]], [[-0.47653241330271756]], [[0.72724334712170657]]], 
[[[0.97784144340052515]], [[-0.049946290218874534]], [[-0.83408984320643542]], [[0.91779042898565755]], 
[[0.069412355530750425]], [[0.18225918013835862]]], [[[0.6473517472432373]], [[0.59810170011626895]], [[0.18999729750206318]], 
[[-0.46446874517113867]], [[0.037150444536874216]], [[0.069283078435410816]]], [[[0.54927812259350661]], 
[[-0.99884933525936592]], [[0.89245643045703349]], [[-0.23779581948646955]], [[-0.92910298518903112]], 
[[0.26654945336237934]]], [[[-0.46982569865049006]], [[-0.66148311275339022]], [[-0.41770532713518227]], 
[[0.88751469595827426]], [[-0.44628143327122705]], [[0.70319495042352798]]]])
        yy=numpy.array([[[[0.5071630893024015, 0.029357680385102114]]], [[[0.42437388400829157, 0.10806933900267279]]], 
[[[0.33391151616662484, -0.40481365001808567]]], [[[0.68555307410635269, 0.82018276955123315]]], [[[0.86984274904337044, 
0.92641381699398484]]], [[[-0.9490604244582761, -0.86541812356128633]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorProduct_Symbol_rank4_and_4_offset4(self):
        sh0=(6, 2, 2, 1)
        sh1=(6, 2, 2, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorProduct(x,y,axis_offset=4)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.53342117484771623], [0.60636860709779539]], [[-0.65678307328192331], [0.85069348314681426]]], 
[[[-0.041921655942592784], [-0.41128178265480653]], [[-0.9415786290288306], [0.89364216276710051]]], [[[-0.35448926944783343], 
[-0.49946352027542673]], [[-0.1126651502834517], [0.66004473404082908]]], [[[0.75760919682408567], [0.28228464537993503]], 
[[-0.42896659408097593], [0.83901684512005903]]], [[[-0.35586985480044264], [-0.0099511375448759765]], [[-0.83985497234443063], 
[0.30321122558827973]]], [[[-0.99117713238215321], [0.023978865896797874]], [[0.058109748276461426], [0.17106346482465273]]]])
        yy=numpy.array([[[[0.26867199909598094], [-0.39580013361610122]], [[-0.91651199254059246], [-0.91139065414747122]]], 
[[[0.23204522340692102], [0.82559571709533008]], [[-0.53273497067768849], [0.98675489895798685]]], [[[-0.69266865234201425], 
[-0.1670837652475643]], [[0.37692944430508168], [-0.57464014038108946]]], [[[0.97648019289367527], [0.64523752939891255]], 
[[-0.41544653264216413], [-0.62490546381716117]]], [[[0.90463684599863359], [-0.66929120506526019]], [[0.71148506845242454], 
[0.40264119374156238]]], [[[-0.88659908603430804], [0.61372877563610762]], [[0.48500809761846098], [-0.14184221540078257]]]])
        ref=generalTensorProduct(xx,yy,axis_offset=4)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank0_and_0_offset0(self):
        sh0=()
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.814624220411)
        yy=numpy.array(0.767656675107)
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank0_and_1_offset0(self):
        sh0=()
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(-0.711465430896)
        yy=numpy.array([0.61737797324484323, 0.11903106474797021, 0.94551225288767604, -0.28295081373749142, 
-0.84563413204723559])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank0_and_2_offset0(self):
        sh0=()
        sh1=(4, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.0538363007341)
        yy=numpy.array([[-0.91319070264737956, 0.7332767395871318, -0.93802331184723786], [0.19907407893252116, 
-0.23590015185107482, 0.88269165520263648], [0.52293184215887001, 0.25498691701196119, -0.13282185570259286], 
[0.83318567198547733, -0.018432795256921874, -0.36915727633335194]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank0_and_3_offset0(self):
        sh0=()
        sh1=(1, 2, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.996879227576)
        yy=numpy.array([[[0.26734416785343074], [0.85577666623449944]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank0_and_4_offset0(self):
        sh0=()
        sh1=(6, 2, 3, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.399576197626)
        yy=numpy.array([[[[-0.59366128759548453, -0.61152717788396216, -0.76051157875064312, -0.49107325060518381, 
-0.86680971285731179, 0.60371155039808833], [-0.10935710785855068, -0.7999500614845716, -0.091787577669105058, 
-0.53975359862786032, 0.11992249695077262, -0.60589116464384429], [0.73186981362967551, 0.14589378539670728, 
-0.77654386343352333, 0.76012260748485527, -0.86898880174935078, 0.096131617008345716]], [[0.82631523208699287, 
0.87732777211798196, -0.2058782370410781, -0.60282298796710543, 0.40172601937886099, -0.61024927806278728], 
[-0.80906363279159788, 0.77990405684108244, 0.99716078653749429, -0.4538261786909723, -0.35359639334484427, 
0.91435014693286942], [-0.81729290594604409, 0.58854055260782423, 0.74582860240446891, 0.38709676894269962, 
0.64535053302910406, -0.54811459319321387]]], [[[0.5444896949561584, -0.4889221792104792, 0.42222970854063258, 
-0.41731770056053819, -0.18638953455010698, -0.53252666313967634], [0.97835981052328425, 0.095550883613026105, 
0.42984490115290264, 0.56937359919730901, 0.25745140210015749, 0.032504571857989495], [-0.90167334117518716, 
0.64248774576702194, -0.022630281654210993, 0.28850486136690501, -0.69263892374056124, 0.16765237303052949]], 
[[0.74724957113162049, -0.78646163660144297, 0.95321935757347998, -0.24484357445349691, 0.32897787902159958, 
0.68263503844488604], [-0.26891812429880457, -0.62923767710259892, 0.42883189040677006, -0.31262579794378986, 
-0.48647954603314059, -0.54861376202092837], [0.32449466400420235, 0.76491497563576605, 0.92840251095627901, 
-0.34415285602774492, -0.27411284981800588, -0.64855498671903833]]], [[[-0.99350244520648201, 0.27083195723596631, 
-0.77846208228313296, -0.55973845067796879, -0.36775464797695623, -0.88698221172694902], [0.0096979396552387964, 
0.547546832653236, -0.10369495157974362, -0.87969314811514998, -0.14330903201059964, 0.77729959136478022], 
[0.074680074580263822, -0.51687253125426968, 0.44017612886551749, 0.081660265435639756, -0.77431737160632896, 
-0.92344231970261226]], [[0.47872509688304965, 0.50187352414516795, -0.98588455918104945, -0.069822802366711789, 
0.34321622705875998, 0.69010897794227488], [0.032054859748860798, -0.73134011892099715, 0.94763110527810146, 
0.74622578603579548, -0.3354401531501332, -0.64816708602117368], [0.34667840812404083, 0.55178878060355863, 
0.78422555008776373, -0.6978691380713089, -0.64494211127050582, 0.030363478098864638]]], [[[0.30767138427596885, 
-0.70586913804483564, 0.92349500014072361, -0.4209488254238678, 0.054017381842049961, 0.83425322964830984], 
[-0.21727536775152823, -0.46468050597972144, -0.20381906089765911, -0.86989837726097363, -0.29684048829696064, 
0.43071813591077812], [-0.38153838714470312, 0.67846742812448513, -0.018959645298123773, -0.66428127699729189, 
0.88566755057403457, 0.95885247964376075]], [[0.88197237381141513, -0.69324434060896722, 0.17306040181809634, 
-0.14730527932265658, 0.63616234719802267, -0.70089049986866381], [-0.53923540380635604, -0.7616129521103705, 
-0.30565955248969945, 0.34538606588758558, 0.86952888090825553, 0.3161063434231326], [0.68216119629609651, 
-0.99065277599281742, -0.22237892373165313, 0.31043339786985968, 0.31783196103565681, -0.33157606899402436]]], 
[[[0.79255243882408366, -0.41100716350510402, -0.33656622200092179, 0.26462751155002229, -0.64338757263859381, 
-0.49542594302218923], [0.15621609478924525, -0.0023330585078806809, -0.99718617406841048, -0.7785852854902402, 
0.71980453301059, -0.67783498659647301], [0.099382014821135956, -0.17148131055187465, 0.5955186643600292, -0.95165848908044648, 
-0.89287765252850249, 0.34330081261940038]], [[0.042483607796466094, -0.49022968967520653, 0.82354208381578986, 
0.24453614730215234, 0.043377299684773218, -0.29555691508385973], [0.48125962363414998, 0.093775245538373353, 
-0.23948016879302214, -0.72278136798448056, 0.77975903263961088, -0.6445177050624542], [0.2458676616326636, 
0.88605409138332702, -0.707547873086255, 0.7509134561334998, -0.27590658840426907, 0.71830134979089899]]], 
[[[0.40440994730622948, -0.58642399850517601, -0.46088952932535276, 0.080178686445222969, -0.52026026872916464, 
-0.79727068567599479], [0.5264346479396933, 0.47686369754255442, -0.15574874871679967, 0.38352613291215887, 
0.55574355674996401, 0.41777705640924956], [-0.45045178944655873, 0.51291919413163978, 0.2679274386661834, 
-0.65198724795481633, 0.68544705787989368, -0.37001082759199133]], [[-0.26266375526562746, 0.32338055979917035, 
-0.53865514462889785, 0.43432831639940006, 0.0077751041253482089, -0.68543352389896395], [0.47769983807480743, 
0.41221785913080655, 0.92910644859335689, -0.62308170356683501, 0.31708966406890404, -0.70360990459393924], 
[0.62618405709840674, 0.65850977120249654, 0.48883020731288451, -0.1182019480851535, -0.96066035987971854, 
0.79284156720044741]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_0_offset0(self):
        sh0=(4,)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.70537979911120297, 0.77388217708162532, -0.68063484201180358, -0.6081715253425426])
        yy=numpy.array(-0.612869481699)
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_1_offset0(self):
        sh0=(4,)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.86375414105672732, -0.032138118359620771, -0.67889971914653402, -0.51806685301409305])
        yy=numpy.array([-0.23807299091099265, -0.85401131840750666, 0.98611005388708106, -0.16818956936434071])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_1_offset1(self):
        sh0=(2,)
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.22664616538527782, -0.61493097664101359])
        yy=numpy.array([-0.9601499438758434, -0.36024143594987579])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_2_offset0(self):
        sh0=(5,)
        sh1=(3, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.76772762666490402, 0.13121128419803663, -0.51124617471543776, 0.0068130033525448663, 
0.22431546449696627])
        yy=numpy.array([[0.3527959650580037, -0.17372725558912361, -0.31894723022525162, -0.15797865564135383, 
0.25497005682874985, -0.50408430099685608], [-0.69200086010595618, -0.10101343869464463, 0.52434928815085002, 
0.41296548082187345, -0.11055156858204929, 0.88403771405998444], [0.29703386475518179, 0.15890234225029531, 
0.47655793671102287, -0.67287137882496362, 0.82071192884708477, 0.53834441331935023]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_2_offset1(self):
        sh0=(2,)
        sh1=(2, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.76840643407624198, 0.075298478221837328])
        yy=numpy.array([[0.29695313003571111, -0.50624238859978465, -0.98600259509148058, 0.33692988784164535], 
[0.50101163546331495, -0.16567512472734403, 0.32388427325037261, 0.49248633654484841]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_3_offset0(self):
        sh0=(3,)
        sh1=(3, 6, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.36180818520691704, 0.48548672407104121, 0.42467181462626646])
        yy=numpy.array([[[-0.10261576288986718, -0.90569527628162305, -0.90226755617027399, -0.80382370995746077, 
0.12156718082653017, 0.4971490175038964], [-0.7048252437249618, -0.61598700050151645, -0.2588360446449911, 
-0.38836105800095955, -0.24655689159426752, -0.57698082682566554], [-0.59823151886728443, -0.26065134412037172, 
-0.95968774269515422, -0.64032400666635869, 0.64156842983768136, 0.42221782611885783], [-0.89910105260882522, 
-0.21495149356102616, 0.094689984005339456, 0.19108543293567481, 0.86807111913727653, 0.88462392996725736], 
[0.76835606662947176, -0.47248243675561841, -0.26962316022845956, 0.49828557243663574, -0.31017271094357124, 
0.20959199436886133], [0.55716098357680144, -0.67393152692718816, 0.16311218416790507, 0.028346085563404122, 
0.83405864746432301, 0.35930579962221354]], [[-0.11197625297834612, 0.76051023245396299, -0.78106156368440915, 
-0.22798334594830316, -0.80043232257344266, -0.538889245038221], [0.63562611900031118, 0.71805850270449811, 
0.078378667111100198, -0.026912688749275882, -0.76404279481136439, -0.70369037749663188], [0.10657033602045485, 
0.32032168929917426, -0.84622887756955101, 0.59978053083174832, 0.42927314073879042, 0.93977132223333171], 
[-0.30387110113239713, 0.63072454242318265, -0.70545167314525425, -0.56307903248006341, 0.15221798872215841, 
-0.13538702716680162], [-0.57900373044672837, 0.60850311366638343, 0.56481440937349303, -0.13185521836343694, 
-0.16270786505415513, -0.68942067735674861], [0.41567074389370817, -0.27479102205362427, 0.43727107033719315, 
0.81953957680566081, 0.52446683063212141, 0.17928980814937923]], [[-0.61539390422464568, -0.15050646795460909, 
-0.29947342977349556, -0.05118279679624238, -0.89330848865169377, 0.29107634038179864], [-0.79998147989479995, 
0.54101956344736335, 0.0687675762865263, 0.64652766190062549, 0.92491630225865906, 0.38821824799985216], [0.12283752929730918, 
0.53755586758658946, 0.57520255127929998, 0.33269778546967022, -0.20960827181090513, 0.039072430102570443], 
[0.57431388658235183, -0.72833912440681692, -0.27289875853639534, -0.55388250438235387, 0.99528161567745754, 
-0.26658658157280168], [0.35305195687533675, 0.80123642932147532, -0.51549860854927321, -0.734969656340972, 
0.58396618443091697, 0.070730123052987981], [0.94008157865223785, -0.42829608994183777, -0.50624369858698737, 
-0.52659511739691411, 0.9402506710921692, -0.57688629348293174]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_3_offset1(self):
        sh0=(6,)
        sh1=(6, 1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.053043109506544894, -0.76351722193724481, 0.52291867938239878, 0.52885013623176902, 
-0.24562905548673952, -0.72358061126331785])
        yy=numpy.array([[[-0.60040741890997595, -0.7466222178130959, 0.75652989531509562, 0.053925676928828992, 
-0.66236845530215915, -0.74521781992238045]], [[0.36552766907578804, 0.3664779583381037, 0.61917282739147494, 
-0.39740713857668464, 0.72546406487953874, 0.83165898845851105]], [[0.32299928952830093, -0.75904815610879361, 
-0.83158528071996729, -0.9745433933125256, 0.84572863438749968, -0.5001278492282426]], [[-0.67637590362705136, 
-0.55358191529659795, -0.30446956837320438, -0.70959708719671832, -0.31099657960090865, 0.049363505430713328]], 
[[0.058548829848487527, -0.26601052477906495, -0.46940307258466007, -0.22761773148757602, 0.91456506695688167, 
0.85060903944485955]], [[-0.90982098583995197, -0.34592053010500479, 0.7207207180373858, 0.11192287025766823, 
-0.67325634608621421, -0.43818066645573128]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank1_and_4_offset1(self):
        sh0=(3,)
        sh1=(3, 6, 3, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.37860205163790339, -0.3236624828761554, -0.75416579968462538])
        yy=numpy.array([[[[0.58500302694644657], [0.95863375796673522], [-0.30754104724044939]], [[0.6914539653803673], 
[0.044156548306602206], [-0.54936708650726751]], [[-0.77734536777501306], [-0.66605546526914527], [-0.17397153353636452]], 
[[-0.90389736540647259], [0.39178666465042333], [0.21704065145163431]], [[-0.88650825117933074], [-0.77008150611144544], 
[-0.13488445054772846]], [[0.3100855648318841], [0.8581577603562307], [0.40641791884573886]]], [[[-0.8494183897111014], 
[0.90200371543111668], [0.32670380588359782]], [[-0.78831651149059079], [0.73090870656426743], [0.689024144775646]], 
[[0.56240102467387332], [-0.94744370728840743], [0.90217253885173743]], [[-0.74260675598757753], [0.79534676772385215], 
[0.61746237510489443]], [[-0.86538627324512141], [0.062465634836654349], [0.68671058646969807]], [[0.46747373503608025], 
[-0.021198342629705103], [0.79435225219238248]]], [[[-0.96250923831615487], [-0.25691345663616594], [0.71578765125341404]], 
[[-0.40310605031340874], [-0.40557727344593797], [-0.93459253688564026]], [[0.9998403300715466], [-0.94717720616174095], 
[0.11068822589582528]], [[-0.14696375588883615], [0.46588731347758605], [0.78668928996706433]], [[-0.66682251675918014], 
[-0.62026119664196511], [-0.76523677552672353]], [[0.93396435471247186], [0.224630061140076], [-0.25521121571710315]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_0_offset0(self):
        sh0=(5, 1)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.33975194328516989], [-0.30237964137815654], [0.72619979051038275], [0.94850386746077175], 
[-0.26114828472085838]])
        yy=numpy.array(0.482213264901)
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_1_offset0(self):
        sh0=(1, 6)
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.5231398607496931, 0.19442768533985944, -0.12410720710220091, 0.47434178304912056, 
0.41822393492400711, 0.83619576878396895]])
        yy=numpy.array([0.19449450388720457, 0.17421553266943257, 0.018493316279244665, -0.14407202083320625])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_1_offset1(self):
        sh0=(2, 1)
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.95033686697396691], [0.45162498602208245]])
        yy=numpy.array([0.29876822299975525, -0.91689053658453323])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_2_offset0(self):
        sh0=(2, 6)
        sh1=(2, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.019461895630995985, 0.90952543846195955, 0.26788277014715134, -0.3020085292326713, 
0.22875418714537865, 0.84410619086528937], [0.79450574220066073, -0.17911214335259285, 0.88662529745494689, 
-0.21875803504659697, -0.39238191085552843, -0.71474387613878765]])
        yy=numpy.array([[-0.64144952784078546, -0.30426637421502911], [-0.8900498271873214, 0.18413418957748529]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_2_offset1(self):
        sh0=(1, 1)
        sh1=(1, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.95648354273973157]])
        yy=numpy.array([[-0.45277072127543816, 0.044925179246641411, -0.096558317828749907, -0.54337014282221108, 
0.78489413231084648]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_2_offset2(self):
        sh0=(1, 6)
        sh1=(1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.88231838359348669, -0.38940571787261513, -0.45726763660328795, -0.069877318694861135, 
0.094489084935554635, 0.17445645486666583]])
        yy=numpy.array([[0.19052683098171785, 0.16745425157047866, -0.50373706991080636, 0.56090907372370347, 
0.92279951614017564, 0.046667336505450852]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_3_offset1(self):
        sh0=(3, 6)
        sh1=(3, 1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.5378102088451624, -0.55267194735872649, 0.47251269247820704, 0.65334666902166161, 
0.49049864181560943, -0.46545558517232766], [0.8281428746482673, 0.35083917419360255, -0.89909694559797892, 
-0.49445900183288338, 0.49761925461394929, -0.46211514036226986], [-0.64219014894776527, -0.081653354250916355, 
-0.023951769968272973, 0.0041881554739167104, -0.29854405940664575, -0.91351176696438596]])
        yy=numpy.array([[[-0.73829552880069249, 0.21880413031519752, 0.12252289652468296, -0.89640218907241542, 
-0.63385355429382795, 0.36039006922747352]], [[-0.81399226933911861, -0.24512594429501, 0.26493274619181673, 
-0.19717646563715152, -0.0053874521454355673, 0.56898646432289657]], [[-0.73798429878018679, -0.25628637864891513, 
0.25015684807523297, 0.2822020664439131, -0.77228890853897125, 0.81693438741866187]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_3_offset2(self):
        sh0=(3, 2)
        sh1=(3, 2, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.42029819929083545, 0.13423529876869034], [-0.51742642695263497, -0.14060204855627356], 
[-0.96488090269126503, 0.42723096573241537]])
        yy=numpy.array([[[0.80248000129245201, 0.42910000332668097, -0.51142697077203847, 0.11480348636941606], 
[-0.02315846346659689, 0.60126541287874935, -0.31182212799941134, -0.073890487051837539]], [[-0.50735309562428665, 
0.26339010517155281, -0.97325145786476308, -0.98156687976530232], [-0.48653132367523844, -0.41294990908892415, 
-0.03451627703024629, -0.26377508095291491]], [[0.11496565190936892, -0.71388602708157078, 0.37183196898854654, 
-0.9575419810819763], [0.084731315624336334, 0.89449449036345596, 0.55629502214955129, -0.079727367698407381]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_4_offset1(self):
        sh0=(5, 3)
        sh1=(5, 1, 6, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.52141064336305654, -0.18563572922193816, 0.59015468778289515], [0.18666441473508844, 
-0.29444506095568634, 0.4201492436364489], [0.42684046902430639, -0.24253187364233741, -0.39476881975793421], 
[0.80949083419237966, -0.042072742230723659, -0.33040407698710994], [0.9348762714297465, -0.59729423036351714, 
0.21852711491511334]])
        yy=numpy.array([[[[0.75663283597201403], [-0.55299136426383866], [0.48299771731347874], [-0.55858595798692678], 
[-0.59257414946644404], [-0.25520827489208053]]], [[[-0.31794208203724117], [0.85105182135977686], [-0.80412530121415027], 
[-0.86622975190143126], [0.55659024122867851], [-0.90399470010261962]]], [[[-0.42372470517559191], [-0.11845975893929306], 
[0.44683535267888819], [0.47891982462011073], [0.86990785029468842], [0.25050287273493654]]], [[[-0.69879716111435064], 
[-0.95430032435309786], [0.7770957672885157], [0.38368635816630303], [-0.66000947572566648], [0.73909281875099153]]], 
[[[0.4007743253316649], [-0.17514039284214], [0.95932356270260466], [0.3021877871861709], [0.93778651402993529], 
[-0.32933872969599376]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank2_and_4_offset2(self):
        sh0=(5, 4)
        sh1=(5, 4, 2, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.31601440149840698, 0.46067335439643742, 0.77108971912026925, -0.1964765787937619], 
[0.79090650363342796, 0.70576525812109137, -0.66178116439633894, -0.34107682229086089], [0.99669106342727432, 
0.02862556620684531, -0.076403778644416498, -0.52189950296189136], [-0.24698749341937476, 0.38338320194594355, 
-0.78653038770823702, -0.47296198395641542], [0.031642330106748995, -0.98921039196100513, -0.97121315627158067, 
-0.021432626281755462]])
        yy=numpy.array([[[[-0.68091411903598154, -0.043533499589943236], [-0.21110176385203472, -0.31536677198396079]], 
[[-0.2246438215344102, -0.56800123196388097], [0.69152521493658736, 0.033921011382054278]], [[-0.81479631747119674, 
0.33195628993154824], [0.13945848592435595, 0.95388583370010727]], [[-0.24268638650112817, -0.673307258141566], 
[0.0031894949081026081, -0.1599376281260938]]], [[[-0.5722537342677807, 0.34774788542567192], [-0.23535062696001807, 
0.12818800875052161]], [[-0.31234330451422143, -0.60506742142434522], [-0.22883441024746132, -0.032930445783517337]], 
[[0.36432508598358915, 0.6711245077900907], [0.53169122651499512, -0.75982439696685478]], [[-0.015302699445884249, 
0.2123277445054057], [0.22478235601087726, 0.053974979268200274]]], [[[0.64841125766414676, -0.68128105996190147], 
[0.23583489535687296, -0.41244739521220519]], [[0.032203339339030412, 0.92324066042836628], [-0.65090476933691743, 
-0.23531322664234322]], [[-0.63378199939560598, -0.84740581885360977], [-0.90915662629667726, -0.99154426368755888]], 
[[0.36956366152592768, -0.40743792796181255], [0.61107825152563167, 0.11063933663515479]]], [[[-0.23820803958796599, 
-0.10986262105633227], [-0.91190856135740428, -0.56515741612693327]], [[-0.69243529508108392, -0.80256571442942271], 
[0.31210523362413189, -0.9255118718246147]], [[0.60806644727263537, 0.43147715016633992], [0.64615542475593779, 
0.58176575545294607]], [[-0.5519403908470546, -0.095729894287399642], [-0.50948778881477952, 0.73582276801353008]]], 
[[[-0.81492991515709301, 0.56064482939374183], [0.014950108749222091, -0.97715639361105788]], [[0.59168628386054345, 
0.5483259955636659], [0.41233103801636384, -0.43807400480920378]], [[0.85813919044547782, -0.3581658790148714], 
[0.29346577339921431, -0.39695831530056402]], [[0.73753837177558612, -0.42869966639936674], [0.85342459500579748, 
0.88967641665360042]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_0_offset0(self):
        sh0=(2, 5, 2)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.94493195167783028, -0.62678286440692621], [0.76507890094092201, 0.030074701870485443], 
[-0.13655080432754341, -0.62905870792503582], [0.26296246410690172, -0.49823036112769081], [-0.28979418871058416, 
-0.18584959438461857]], [[0.85355662235356022, 0.94649316210857348], [0.464853261553269, -0.26539750974201959], 
[-0.56605722047468099, -0.21422277336719686], [-0.56777827784759949, -0.3602782988414277], [-0.12940893302016154, 
0.65417059679055156]]])
        yy=numpy.array(0.339170332597)
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_1_offset0(self):
        sh0=(4, 1, 1)
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.76795404148813984]], [[-0.63379962009639801]], [[-0.8629465317963505]], [[0.0039193370822276918]]])
        yy=numpy.array([-0.546539474999443, 0.79644596112321087])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_1_offset1(self):
        sh0=(6, 2, 3)
        sh1=(6,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.3213461437649896, -0.43525090807707145, 0.23384995386677154], [0.63322813230730524, 
0.88717118623138114, -0.97175877008322753]], [[-0.97195544662281463, -0.33051902967182145, -0.63373848324676918], 
[0.47283056269517676, -0.91726583573803255, -0.84857567113688548]], [[0.033475683034253745, 0.91941442844471766, 
0.6995638221337801], [-0.082925290421518083, 0.022401974321274221, 0.54674788503132765]], [[-0.75101521150596873, 
0.48644705268158606, 0.46659395878716814], [-0.88232270369533694, 0.82433795602006055, 0.21378011019498655]], 
[[0.85723860291021992, 0.71807063629667289, -0.59087815160435997], [-0.019822303918352713, -0.55792029259390641, 
-0.32957722582939875]], [[-0.70219459383857563, 0.60892324216682914, 0.12423847881103733], [0.91582938046583928, 
0.91032957773463452, -0.19897246352940945]]])
        yy=numpy.array([-0.43985343495022811, 0.81894606584420115, 0.07690896509908729, 0.23502122135942916, 
-0.085139010157082584, -0.63787958408821699])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_2_offset1(self):
        sh0=(6, 2, 5)
        sh1=(6, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.24679623150289598, 0.95692166689802605, 0.006136812570472161, 0.66606029467485328, 
0.96026981335951511], [-0.86875356889220279, 0.70456830506054158, -0.22893257105803611, -0.5657761949038207, 
0.93582346407002981]], [[0.10458661509325595, -0.1037804862409637, -0.58430357494753316, -0.33081329850377283, 
-0.92861016252201711], [-0.70349282099508881, -0.67371773488040443, 0.14398105764647706, -0.0064622995109084869, 
0.26637273804368755]], [[0.98404432870088798, -0.93971317117058861, -0.0079407779715610083, 0.15150484543833542, 
-0.27236090938002033], [-0.62881382249800133, -0.507800467930553, -0.46929895693742996, 0.48426441407651222, 
0.39694237665051113]], [[-0.20245115799994617, -0.61104839880317097, -0.041653569605204499, 0.72799352265228268, 
-0.80984425890468392], [0.86479130569021212, -0.50384144591916291, -0.36965610122788006, 0.66610368243135598, 
0.023258256231476704]], [[0.079635427412542814, -0.28946994466192133, 0.18899108500375594, 0.30488814431210587, 
0.067911296164030022], [0.78707236181550644, -0.38044745512421363, 0.10477224008627761, -0.9654576674162898, 
0.65119611652111065]], [[0.47979669919634471, -0.63134871715282159, 0.083100244692191572, 0.94512758564272725, 
-0.22647126694569852], [-0.59994359854268575, 0.73487628729291488, 0.77643237899020923, 0.07720184717851053, 
-0.026455321193299852]]])
        yy=numpy.array([[-0.44996656098684351], [-0.99891661997778192], [0.29142701293040552], [-0.72597548426455849], 
[0.11152066731670995], [0.44487053049478309]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_2_offset2(self):
        sh0=(6, 1, 4)
        sh1=(6, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.31933662736403567, -0.36863741209880474, -0.66765677402791646, -0.39081571762402922]], 
[[0.27969617248887713, -0.50703642106192204, 0.25165131967597842, 0.94741571157342053]], [[0.014447158482012501, 
0.36514058208172906, -0.90106130466610068, 0.022577584439219756]], [[-0.7781384345197897, -0.61241204999358057, 
-0.86689396089540738, -0.56570786123653338]], [[-0.79325698089849728, -0.34736850780261141, 0.33248872965883791, 
0.52321816892261297]], [[0.97105528856858681, -0.9748121754638237, -0.0030094740971171596, -0.21212532839585418]]])
        yy=numpy.array([[-0.072938365284641815], [0.24579335756039633], [0.92060495302114709], [0.71185625702365685], 
[0.23874235100944285], [-0.30042793721873884]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_3_offset1(self):
        sh0=(1, 5, 1)
        sh1=(1, 4, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.97068371158245093], [-0.54365793806954144], [-0.097701331320395601], [0.70321690986494767], 
[0.57831955832322679]]])
        yy=numpy.array([[[0.72120740950478268, -0.0142449351959546, 0.91900433894889999, -0.33708269898921639, 
0.11962531141046262, -0.23206179481947919], [-0.80438867508142775, 0.66680943515948954, -0.057495486975172261, 
0.014235730087009513, -0.55068527204379469, 0.3486955218535035], [0.16388517938596969, -0.72798948990628931, 
-0.040416373288822527, -0.35817172222481286, -0.34985371425693801, -0.98405847398431057], [-0.0054725113422442462, 
0.58534888582643263, -0.49891489467449168, 0.17237174656139098, -0.096927471574500768, -0.62630549262478619]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_3_offset2(self):
        sh0=(2, 4, 2)
        sh1=(2, 4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.80271552107181421, 0.82001594448132908], [0.88124557604769871, -0.52916766331633824], 
[-0.2704322133746988, -0.84977581786794953], [-0.80902755202537313, 0.8449903204462188]], [[0.71498289186373243, 
0.47886718037746956], [-0.16142045612748279, -0.57361819824246529], [0.42097380007692964, -0.46921787468477993], 
[0.42290366492060261, -0.047832396339065664]]])
        yy=numpy.array([[[0.17580452819540238, 0.50233632977136278, -0.9472542794099934, 0.52935160984152563, 
-0.7217164314391491], [0.20915951961378876, 0.12232282026699703, -0.17663494164283877, 0.078055569963612736, 
0.75846401522705209], [-0.71412184567812798, 0.51422622794953132, -0.27355896180702199, -0.31173757607440478, 
-0.57868084139768827], [0.85212362780769069, 0.93430486209809871, -0.6220474691029787, -0.034250282740927229, 
-0.45083131727055603]], [[-0.21787781708403253, -0.83720982877494277, 0.28934109672976271, -0.71912149846983464, 
-0.14573720317849626], [-0.86437686752558274, -0.58546734713361004, -0.21917832560338191, -0.56308921568985193, 
-0.55509861412094863], [0.95916783502812319, -0.91938057012358576, -0.4675453968380332, -0.64704248217502003, 
-0.13066886783661147], [0.3436792576727008, 0.3844304408735999, -0.049120238966383178, 0.058292891849972772, 
-0.79980071642472317]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_3_offset3(self):
        sh0=(2, 1, 6)
        sh1=(2, 1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.6027741958045596, 0.44214419471870192, 0.4446379662463773, -0.52663625259150026, 
0.9283431338446595, 0.030626828443126364]], [[0.98315282629977174, -0.4562380710752052, 0.75678833838752535, 
0.41116860462410232, 0.032892525305242781, 0.9691103832366279]]])
        yy=numpy.array([[[0.24846096982985566, 0.88879966220456308, -0.91305582980913935, -0.99222651531273165, 
-0.028026569383039268, 0.62778961309992387]], [[-0.077156654953615567, -0.40091661902395215, -0.88032392723348019, 
-0.45664035944713754, -0.04840496232424063, -0.55070363321512272]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_4_offset2(self):
        sh0=(1, 2, 3)
        sh1=(1, 2, 2, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.55212229826505443, 0.87899619228041193, -0.089454845672060079], [-0.96336307651890007, 
-0.090625724111868422, -0.79272705525105169]]])
        yy=numpy.array([[[[0.29662215563575023], [-0.92831341547744772]], [[-0.29774833389682809], [0.99695500749966315]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank3_and_4_offset3(self):
        sh0=(3, 5, 5)
        sh1=(3, 5, 5, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.17904108479615854, 0.25017390108839921, -0.47803971505855691, 0.22389716684860894, 
-0.69407039402518156], [0.33276747843899157, -0.030009172249198635, 0.95861290070892591, -0.27715014225812706, 
0.56651794430038627], [-0.61877329650976098, 0.70567990777598655, 0.49007133190242924, 0.27854189431799936, 
-0.48779579903243531], [0.66383653559456346, 0.96316428257988651, 0.4482090381627668, 0.76245398719077961, 
-0.014509546242029492], [-0.73961855907924856, 0.38235396887645323, -0.44056110773528689, 0.25737180003734328, 
0.53112934499306119]], [[0.79755606764258014, 0.78284591248401392, -0.48420584860427351, 0.4122151445683111, 
0.017713518387848071], [0.18557197128103464, 0.10866246407321634, -0.96664505595410022, -0.65005226890585566, 
-0.7906549037722288], [-0.99531399265523723, -0.025554882343242102, -0.91504986060517601, -0.49816465061685711, 
-0.87876168924629483], [-0.074674039149984717, -0.47030703329938883, -0.27699108948128304, 0.76512033880203867, 
0.7955020051780719], [0.14701515496546036, 0.95419832555399564, -0.8990915962257815, 0.28759022985624871, 
-0.26706680581592024]], [[-0.66137344593840153, -0.60194428204937389, 0.049231921885965368, -0.45623210388279101, 
-0.88760539067937927], [-0.80260636530062857, 0.080630452460167978, -0.83812048679017859, 0.89149436684439509, 
0.14549804763195784], [0.34157116661025233, 0.7269023464416986, 0.80397868253168125, -0.5535272184838016, 0.37224024974642456], 
[0.19050292734135965, 0.57954494936806755, -0.31898471147611862, -0.1559305167540983, 0.45900547581999929], 
[0.91370346496319832, -0.75156125370623417, -0.46829618118157268, 0.16050502591336646, -0.51046756221393097]]])
        yy=numpy.array([[[[-0.16971273531710707, 0.19181233740233683, 0.91193218842112711, 0.1211911288351819, 
0.19293356756043845, 0.43866821840112702], [0.35993792761397758, -0.81372884614223517, 0.028054121751444283, 
-0.045101074184267231, -0.69092374917980548, -0.03677021898865096], [0.42392991606939079, -0.3663050705863673, 
-0.72088020297782318, -0.97566248450777238, -0.76504994862799536, -0.96699909036550191], [-0.54072152193379308, 
-0.22106809348199996, 0.89218224850400163, 0.85995010569004071, 0.27108458795609969, -0.019367758367003551], 
[0.03731133639387707, -0.59105091784892583, -0.32010740349362887, 0.83142731648681578, -0.90033743881713013, 
-0.85191293508259913]], [[0.88432469121618773, -0.22518755189906492, 0.93348882036364733, 0.2659810839378336, 
0.059316197812716576, -0.073564871857298098], [-0.48655474327529924, -0.033790114726071074, -0.92478739998889203, 
0.83274669782053135, -0.520461116314612, -0.042734081156052151], [0.83229317622234911, -0.51871320030408841, 
-0.38612304366624617, -0.017236850241836965, 0.5627568232996254, -0.18965120517459599], [-0.2385433200832483, 
-0.63487094432731994, -0.11340571456200532, -0.087487399056728199, -0.10918572742426114, 0.018449435733612018], 
[0.67508147855640455, -0.87327327391292608, -0.57458117386285457, -0.80985345948206855, 0.012714218942645905, 
-0.43092315106354806]], [[0.61490181142984768, 0.7114881510501383, 0.75673675697697407, 0.66977442129676024, 
0.95380307205078396, 0.5472460015020002], [0.5033190962870091, -0.72515498840536319, 0.99580021534178287, -0.3867107757566568, 
0.88166153936870484, 0.87363163126395715], [-0.68190342239704171, -0.033941284612304834, -0.91630610233560517, 
0.33476874094238984, -0.40702463772038944, 0.25566298338064475], [-0.040572202340821928, 0.058397155274422463, 
0.18541035679930085, -0.0057401604280438168, -0.25059564950428492, -0.69800580206417151], [0.93873060510697393, 
-0.3012074988233957, 0.24716865920733455, -0.68240076569591901, 0.34662174229029263, -0.56529158578653282]], 
[[-0.30079075412629797, -0.44369217202162825, -0.58330037006433644, -0.80282663131537468, -0.39298176591906908, 
-0.074567792401095234], [0.26225130403095487, -0.93711563626825489, -0.10577584585452615, -0.8936898565300333, 
0.033475477128490416, -0.14940283016331302], [-0.6231428572192681, -0.54337858879604006, -0.046643639410725246, 
0.57372451881305953, 0.58970202377280234, -0.83258587491129377], [0.33242078331534275, -0.044883548049232003, 
0.26765303274565033, 0.19760955011570691, -0.018680142473021855, -0.98252081102864786], [0.83594753565441282, 
0.33084871280181627, 0.11674740806036477, 0.63348797670585966, 0.77956377836202351, 0.056205017303361249]], 
[[-0.9299979920999244, -0.1770109440448322, 0.66568291662427259, 0.23039836773118161, -0.67760251672489114, 
0.75790135786238921], [0.22738877814074887, -0.42789132631331617, -0.54926316845706213, 0.69719077505679272, 
0.1639513558906649, 0.65013578466097188], [-0.55415242724780223, 0.26057337545571824, 0.75689819873299924, 
-0.31897046716604427, 0.36813940236232945, 0.3009628110063185], [0.54184888207382853, -0.55385264453687144, 
0.75576179732654225, 0.084606381288675925, 0.69490700780951031, -0.68171577099941238], [0.83058950988324609, 
0.79863811169541443, 0.19510025605681913, -0.73983163844047395, -0.62372644127764287, -0.39394791200871926]]], 
[[[-0.84837403439330217, -0.62344295384505144, 0.88163492805757171, -0.80301385700925532, 0.83022685921301265, 
-0.69024194606536216], [0.64019005886621239, 0.0011800576608740343, 0.42290884951683161, -0.75389766611844555, 
-0.94998856893094574, -0.28233069067751959], [0.53665717147780256, -0.28420937642668864, 0.26884510972609443, 
-0.010523410894559548, 0.8272053543426694, 0.43000607543254987], [-0.79729920932295295, -0.5508807231999735, 
0.32909159409720457, 0.50160262675163603, 0.92589554237531302, 0.24729023177449228], [0.17821021917825242, 0.46643913016826466, 
0.62604378476207301, -0.64959375323908408, 0.5546770134670429, -0.88907054891618853]], [[-0.53634652076211897, 
-0.77458972589544528, -0.64690470592235738, -0.00099435113634638306, -0.75453925969287661, -0.76219907157288103], 
[0.058769596282884473, 0.45315296506080771, -0.59179873720974152, 0.94842438720944755, -0.82839832884180886, 
0.46118726005727573], [0.77119630338515432, 0.40801628339624707, -0.45339212908632498, -0.54055847806693635, 
0.46653538127288807, 0.067349593601361946], [-0.56357727662718005, 0.1458933950945025, -0.63860427572938949, 
0.046552176939646861, 0.10071536977301987, 0.42947493440907714], [-0.8870086605703027, -0.047128348032707601, 
-0.28533782481610914, 0.14958094849749615, 0.68348758909419938, -0.60657519840816887]], [[0.30927200886633366, 
0.31454320915142286, 0.46420061051776917, 0.69630010628071437, 0.10521685270639791, -0.78403542325153719], [0.6276866802968819, 
0.53321985836349417, -0.50282834553430256, 0.7572944540393205, -0.83625908283960992, -0.32378655613694551], 
[0.12903994541706187, 0.054840078340502307, 0.74539816528725722, 0.49953415685464098, -0.54666855645789658, 
0.3634713619487151], [-0.96659826205011301, 0.052474062358784046, 0.66502652970432452, 0.7206296988769989, 0.63123359580976146, 
-0.6363447765166903], [-0.14647666698042983, -0.61032803423820892, -0.018464728738566416, -0.8357167411555575, 
0.86132470918053672, -0.9220938082950505]], [[0.52523163890066682, -0.84758619907416444, -0.24520442523303743, 
0.26605220319833434, 0.23911178942921474, -0.15264244001822824], [0.51107745615509126, 0.12668769741210206, 
0.98703220789229662, 0.34301425665340468, -0.43774690290505469, -0.2562603819063265], [-0.22992740449204363, 
-0.43309441524165826, 0.7824854260211771, -0.579364678855399, 0.72160624469026557, -0.165592290278745], [-0.56045494972995336, 
0.6007870838913496, -0.38385371845046889, -0.45154552839686635, -0.25712088743789363, -0.80188173766728887], 
[0.15439124952794203, 0.92513476457137256, 0.27997406260605873, -0.19561715864186335, 0.13984461830529771, 
0.050103965920999638]], [[0.2355521150249904, 0.51302717136166365, 0.79346470927346724, 0.23130794779438046, 
-0.29794824195900027, -0.83770444811335465], [0.041775455563043407, -0.24266397161570041, 0.81468333681354932, 
-0.69379011136233859, 0.33098813876420707, -0.92526460342407613], [-0.28737713951401234, 0.7622465827917102, 
0.8785980612011508, 0.92949957603107469, -0.78215958418384335, -0.89357450235651426], [0.85908048653690328, 
-0.46449276239941173, -0.28329835996991726, 0.82433642997853096, 0.70854406554719129, -0.023747422428462395], 
[0.6713720112757553, 0.42827831256117443, -0.056485613276181068, 0.57648329105079976, -0.95166092839723748, 
0.59962151257861551]]], [[[0.0053041753645135525, 0.35504225694062375, -0.34153192973672164, 0.72380140722849973, 
-0.93100812984186532, -0.3692690375984593], [0.073811597793496331, -0.62412350416168549, 0.92964932981794113, 
0.14991720823288146, 0.37951408755876548, 0.11973623701278902], [-0.39850341662168853, 0.97032327994682088, 
0.83508138157646594, -0.56943465312924912, 0.34332456432788838, 0.94169719510523264], [0.30510616871276386, 
-0.43928052494861936, 0.0031527103942015078, -0.85671082293071898, 0.8311286813294414, 0.00095811853696980975], 
[-0.38112644640830307, 0.026816332052293612, -0.69771305325356692, -0.85047286943544043, 0.16811782327426061, 
0.51724093264687032]], [[0.8358963448969734, 0.62179481283316584, -0.79409682325923048, -0.25090431125751422, 
0.24630474682416392, -0.221161125778357], [0.54470658337752331, -0.47585830517621686, -0.23366179228422057, 
-0.095268443211957932, 0.21179589391514608, 0.68088987614965424], [-0.076500012517358273, -0.33484112767672736, 
-0.15177812022945569, 0.57664009770562186, 0.089974234068942449, -0.4939309335698252], [-0.70737277711507729, 
-0.82413983245465472, -0.24113830317034757, -0.62925424793254914, -0.84965049853595853, 0.31937721810614983], 
[0.38487015268658542, -0.59494690480142909, 0.96481995359290873, 0.44948013278803178, -0.50423320854462395, 
-0.27538220619526088]], [[-0.33422580155992865, 0.20845238158026813, 0.564842462724505, 0.20926508029121949, 
0.76419796512896676, 0.63897289530722556], [0.137796754442159, 0.090304473593127099, -0.328322101486481, 0.52335908693725153, 
0.63805196116736518, 0.49082896358753048], [-0.63434978306165446, 0.12805262490508973, -0.61117509217191879, 
0.21574041199107841, -0.8045076173945529, 0.57289180788130678], [0.69612401210689079, 0.19566787894181825, 
-0.39461344093853845, 0.94570062334406524, -0.22231492106282569, 0.17917202055029735], [0.5987177573118565, 
-0.77093504371078714, -0.047266062357750105, -0.094109333481828594, 0.19573211069059493, -0.97670378456907225]], 
[[0.15871880367553537, -0.32478072187292506, -0.42018174643009809, -0.5075687631999628, 0.29798389980728301, 
0.25443959804637517], [-0.50265777340102735, -0.9223243757892361, -0.50206733445697593, -0.69731753740258817, 
-0.753585444772916, 0.61555006137871815], [-0.43206754937197878, -0.34030085975606283, 0.7674246811067158, 0.57784710803260819, 
-0.27899762095829073, 0.48391891201763215], [0.016148548554537401, -0.90470143639949541, -0.43469836770113002, 
0.65587627601759513, 0.36748466423523718, -0.42490819944089742], [-0.78148024441664066, -0.011625604982656856, 
0.18034086033998076, 0.62261311937473884, 0.65303047888420007, 0.86587316380485202]], [[0.57630148402844883, 
0.80116628064672901, 0.87930402865948243, 0.31355637014262383, 0.27856314229342516, -0.9868485069855395], 
[-0.041037186227189437, 0.96989708760740423, -0.36345827382564466, 0.66051307627042855, -0.88284451891756444, 
0.95252232553887972], [-0.41150814980582417, 0.21247066079624832, -0.92025327040974747, -0.76802109981703204, 
0.67036894372232347, -0.89449835271865719], [-0.68220646412569441, 0.78852101679619691, 0.44847799795625165, 
-0.36993302365568215, -0.93167824722010728, 0.069336579587963909], [-0.33413068106086219, 0.07881863435292713, 
-0.44236996657163785, -0.64779245584565914, 0.96899324591143321, -0.87062467591646264]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_0_offset0(self):
        sh0=(1, 3, 2, 3)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.34655252218455979, 0.18981605191760287, -0.13038120624391314], [0.3256154088206793, 
-0.94734293686770021, -0.59035204933528274]], [[0.70579580518613683, 0.11990382222697327, -0.64905878570333564], 
[-0.40913592579019098, 0.73231751294621361, 0.30178542659972529]], [[-0.4929867071599352, 0.25163766140223598, 
-0.56065838114923583], [-0.30132475933546754, -0.7773769114334903, -0.86199776711278187]]]])
        yy=numpy.array(-0.658096407776)
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_1_offset1(self):
        sh0=(2, 4, 4, 4)
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.3414584086712007, 0.72021422760286313, -0.19333816709384566, -0.18960200009516193], 
[0.53392088028162998, 0.55920282817622269, 0.49116288397586061, 0.039571567976897315], [0.30654048128088407, 
-0.75754734613687091, 0.91182701898404739, -0.06072922130233871], [0.036107083576853993, -0.51154192782126695, 
0.31944633617532547, 0.3564230562341193]], [[-0.15223358611651094, -0.54194236806539586, -0.92211565194915246, 
-0.23202867559731732], [-0.2247491647737403, -0.014597243298613449, -0.75275894632916374, 0.95999602455001498], 
[-0.888577751981519, 0.47728929909736451, -0.67910387231849523, 0.86170443586017087], [-0.8541400238500485, 
-0.4787307405824015, -0.50847006689392726, 0.98088536103841162]], [[0.38739015216948203, 0.019287729496189066, 
0.34253994619940964, -0.69200442232592763], [0.36082845549657638, -0.90989194236916049, 0.040586087447725072, 
0.53120673229395021], [0.039023561827339348, 0.88484302928785996, 0.46057100420680586, -0.24717888144548605], 
[-0.3023089853164751, 0.33227955726950342, 0.13051962211440626, 0.032755842185487438]], [[-0.48775558717737, 
0.88891225588965961, 0.50908469822966773, -0.70612660261075999], [-0.13705250388877532, 0.6951784355365842, 
0.92020208482056187, -0.51925232223239326], [0.60385791520702625, 0.71916508922506983, 0.7810750917413245, 
-0.73816402896505129], [0.14960329017079554, -0.41546346872595308, -0.5383232941741054, -0.84529273885601186]]], 
[[[0.82091745965372631, -0.35376495443639877, 0.80559695784321717, -0.2767108999679011], [-0.88422725691569437, 
-0.37750476215222228, 0.87871019080345314, -0.33032821731004702], [0.13858196847794435, -0.93109514668097759, 
0.41142766127322816, 0.43831467503543631], [-0.10729172492320571, -0.41135531089158461, 0.94165932127411667, 
0.82535114424788358]], [[-0.78455479196730527, 0.17051699498312356, -0.29231116184430306, 0.28295964128782636], 
[0.33363270757874441, 0.28378920937135521, -0.19171320889305599, 0.64059895216351848], [0.8394803114348941, 
-0.16682302893567935, -0.46471588168771349, 0.63581970118550912], [0.026639667208816054, 0.15135631571016095, 
0.83432904858619938, -0.59363860659462908]], [[-0.5885791981326558, -0.46394415266903111, -0.016543592206859969, 
-0.60845182744125337], [0.32009848118227224, 0.47610988466916182, -0.53959790584455614, -0.64116132593805264], 
[0.058953403759229861, -0.53998483937814989, -0.51240897880782099, -0.80333519690941602], [0.2796657599440866, 
0.098457370344107842, 0.95352002704689087, -0.22682183217434426]], [[0.92555168382056685, 0.89737088797115305, 
-0.92395025968214561, 0.58646334148440982], [0.5876318680305419, -0.88866065065219302, -0.80459156749196703, 
0.030296338620005914], [0.64027367829425597, -0.4815773409787234, -0.92292539608935087, 0.20063300600954137], 
[-0.41994592922861362, 0.010127929574038186, 0.70440925401182541, 0.13600416307747354]]]])
        yy=numpy.array([0.35328287805696901, 0.77827307115744082])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_2_offset1(self):
        sh0=(6, 4, 6, 6)
        sh1=(6, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.042162511343406317, 0.6377115730780254, -0.95966072285951243, -0.29991249939210696, 
-0.13839923964688738, 0.77346182575023503], [-0.049035482820126708, 0.70002570811341114, -0.72629745107028065, 
-0.65672359438505956, -0.84927732442443382, 0.010587262739725878], [-0.64473002828076464, -0.39419785003220387, 
0.89476855142321576, -0.10842099406652661, 0.48762123808222957, -0.15039749750313325], [0.41912149396850285, 
0.013347777330152022, 0.14761179920238776, 0.80438901386062067, 0.73513816172944879, 0.91151479365494392], 
[-0.88118735433917417, 0.17625814469174728, 0.049445386480393871, 0.25160815331804187, 0.39025001787002189, 
0.058849275998647865], [-0.69852924246854009, -0.25676236713302036, -0.42949571660992292, 0.87786088814731977, 
-0.65838053945245134, -0.92829232530860994]], [[-0.19781459092477527, 0.62708991678707959, 0.89723063789252033, 
0.25097187406839994, 0.2841452768095043, -0.97830116274721335], [0.28585660545457903, -0.55119774532500077, 
0.37276143784809657, -0.28797729795988025, 0.91992596096856749, -0.7092030476882154], [-0.65702343277398323, 
0.67237396957029549, -0.34955515792489544, 0.62263829259849279, 0.27931357128023482, -0.44997112120579796], 
[0.2374509574753243, 0.16143906595426016, 0.89856504649581148, -0.4108359832670192, -0.11310400679886312, 
-0.51295610977888706], [0.36814587323531045, -0.90977814061474849, 0.272544821898554, 0.53858977515974971, -0.6620367477433724, 
0.18378827033509681], [0.87099998346768626, 0.019142041136941179, 0.46042003634319872, -0.61937487018429982, 
-0.49009529658849416, -0.63019437393962585]], [[0.50614966898120084, 0.82622811927972628, 0.76838504112081663, 
0.38960506912923631, -0.16837270779745861, -0.81997113104036701], [-0.15755843296079219, -0.54972379616833056, 
0.40261255217205316, -0.022373237578056537, -0.88632644964237151, 0.042368225977630569], [-0.30670251020222761, 
0.4732911807548037, 0.78352322949797926, 0.064674448969492371, -0.14534077092092601, 0.1616024734287278], [0.70893880511298168, 
0.86325686379298139, -0.072500826232198445, 0.65720500496241496, 0.38465996322776208, -0.26643823945553913], 
[0.70196012384581752, -0.54382401356758647, -0.32918964977812193, 0.59705093476057014, 0.81783631384809485, 
0.27768518306571277], [-0.0053203789858897732, 0.1663071937786067, -0.84709251037973443, -0.30982434584148399, 
-0.04849514566901747, -0.81898787292141839]], [[0.14421826436135632, 0.055212980259119027, 0.97466741719092798, 
0.050821650045210109, -0.7443786381875015, 0.061776241095648388], [0.06461163008481785, 0.72423544450438238, 
-0.5762593977158188, 0.6015990281138428, 0.039784846410056529, 0.045960341600638444], [0.61972552975145168, 
-0.056200079274577153, -0.85948055667909951, -0.53046715167220393, -0.91982969688286231, 0.013297981736598796], 
[-0.6635720691138125, 0.44029912448045927, 0.8948551931147124, -0.28313106938786681, 0.9709563743448435, -0.43196100558879147], 
[-0.80541018243270557, -0.89891444367541107, 0.27392054648029318, -0.25673643392590706, 0.60143252170865646, 
0.035007201728611159], [0.69056966612182946, 0.097058201672195255, 0.91659941929338773, 0.64055195557722744, 
0.9578799654532606, 0.72105391436972899]]], [[[-0.96877135063613817, 0.85806666268765874, -0.44237448039937033, 
-0.87701517846216315, -0.98030169212425644, 0.35779615806126164], [0.29815760713754669, -0.29427413812056225, 
-0.7304269947724098, 0.35073120771199018, -0.65878374703414688, 0.074531060745254507], [-0.87654294247848985, 
-0.89716783040782899, 0.37785831777835877, -0.9938506208931186, 0.84416273760451133, -0.57763588939107646], 
[-0.0044769529022889998, -0.2136013310798448, 0.90429519128855285, -0.56198344173088555, 0.85371986768201102, 
-0.29024357947836998], [0.032558377734384303, -0.27849425628078195, 0.81437465271980702, -0.63138392847194558, 
0.83532000880556789, -0.14990036423028208], [0.29732732341883428, -0.55694709762000949, -0.85409029753181143, 
-0.40171486476800977, -0.29981149809240915, -0.14625948677011125]], [[0.15781865785381144, 0.58184384625012542, 
0.64072583554228846, 0.43039231640064912, 0.36244284727639586, -0.89162891754087248], [-0.11136389460690976, 
-0.54106430598855892, 0.42735950169373038, 0.011257544340948611, -0.58084993595227874, -0.75006479927526182], 
[0.72908328905043884, -0.48181775099760493, 0.86092820524936031, -0.1907258000856431, 0.45567609435987233, 
-0.87729532336413008], [-0.36006191004850363, -0.21865976130511733, -0.037744040442273974, -0.5813226914871612, 
-0.54849609690485046, -0.12566964155169025], [0.61210835500277305, 0.37703056105370236, -0.37976340743718451, 
-0.62537078299770488, 0.81547264387064167, 0.58115373894292621], [0.74643687858222352, -0.85580318360325314, 
-0.45746657597361495, -0.90199968284469123, 0.33208892436336024, 0.58094154556559174]], [[-0.48576288370397402, 
0.42225822795330292, -0.85879067817300769, 0.61626760429232075, 0.8200789640992685, -0.73528940168718537], 
[-0.16087994718691045, -0.2829104703354175, 0.0056699206606580788, -0.23614685206851482, 0.34223671452233417, 
0.22962441151958668], [-0.87719516985340173, -0.80421785796175138, 0.04397186710391443, -0.79491712624008759, 
-0.23493632486530847, -0.79772798610433759], [0.22493464717191691, -0.9371480791959359, 0.70036197122975885, 
0.92300570400863546, -0.039742320615192561, 0.52484713619700551], [-0.62563382981352555, -0.61696367772059846, 
0.38111099631036893, -0.49562408639605815, 0.22264577051664847, 0.41798518018216124], [0.18544409687331709, 
0.80507690575114177, 0.12874589032288619, 0.73063488077741479, 0.21726035565492996, -0.78319379569480851]], 
[[0.21734930373188188, -0.67865690219693886, 0.014537579060521999, 0.010583332485841312, -0.70582110074966931, 
-0.61395081739392565], [0.6592794617784139, 0.21971294633509686, -0.85723123529667444, -0.73104158836918609, 
0.92207088218631839, -0.80982235250279699], [0.012996452553325932, -0.57151508383551608, -0.86389299852949897, 
0.10827290556489833, -0.46835268794187757, -0.48608102005552256], [-0.63008485290764726, 0.55288436901080651, 
0.34534985614961333, 0.49169467037784553, -0.3615501685983844, -0.062741179215267895], [0.16925455477435047, 
-0.23894337660238363, -0.5133847433255887, -0.43157335731305668, -0.20018347737011877, 0.63232232232108587], 
[-0.80749546115289395, -0.85810364783803439, 0.9335834200166202, 0.64485297851396295, -0.51118924427461265, 
-0.93660014850874362]]], [[[0.19740368290751698, -0.14521855043557985, -0.43109296234295358, -0.86883061104253878, 
0.89988856848950771, -0.19493312856481393], [0.15110817653787234, -0.43733204023778915, 0.87728736290026621, 
0.48007921571592949, 0.99803350883656972, 0.91356792934572528], [-0.20356625552611218, -0.91453259857729385, 
-0.72933873370630775, 0.2364666724803397, 0.65212161549795766, 0.60252845440931502], [0.9387244006648956, 0.71808200709467918, 
0.85439603852850188, 0.55809398618191786, -0.41218792625551792, 0.4586029912755003], [-0.55246934334392139, 
-0.0092180528269476536, -0.99812989697209065, -0.27780119587654806, 0.64067350870445994, 0.96823523714415405], 
[0.17038210004528809, -0.0061651708711056852, 0.85537820729306491, -0.0095299496452943, 0.31765004669076213, 
0.78267237944899981]], [[0.8554908424410379, -0.53523699365532407, -0.053529681096079695, 0.86649063007326865, 
0.84735150602753651, -0.21468791614292004], [-0.70587591099191616, -0.4684978718988031, 0.83701711735473117, 
-0.49634564688608074, -0.11439251148523111, -0.84492704885393333], [0.21395372322921302, -0.26581928674883026, 
-0.31133859556976695, 0.71830738176449604, 0.8301545945383344, 0.5381016222938253], [-0.72780103693698517, 0.12859948921069897, 
0.10559476424350089, -0.41985888645427361, -0.3233112146720758, -0.27468535306860753], [0.68062625377050656, 
-0.20156132007998973, -0.68046145643440448, 0.92352570840259496, -0.97879090708968608, -0.65948684381020906], 
[-0.93441930225565928, 0.82373487477440532, -0.88663364786660837, 0.1135559702723179, -0.082544187746840914, 
-0.13403455153860211]], [[0.085534608721228089, -0.68359603734591023, -0.54614672603890857, -0.95412393953519237, 
-0.79293941870053009, -0.37789304994280593], [0.98598980396579194, 0.83826722035017842, -0.080830989782853413, 
0.89485471860016608, 0.11328635910350493, -0.40876231758556525], [0.49849032379352232, 0.76406181656369832, 
-0.054271463981376611, 0.55042060948131755, 0.19828630172202799, 0.86836617229536572], [0.90621194359899104, 
-0.61152046021588169, -0.61683198591128008, 0.10315414416990287, -0.0029800662206220796, 0.48069883950100389], 
[0.28090422897224343, -0.66193755608052118, 0.9103572100217916, -0.56973378164788557, 0.21635386329639972, 
-0.60494533742046031], [0.56098329532985369, 0.26031329875418607, -0.74853960027357469, 0.91763833328182542, 
-0.65339712064711386, -0.91014431120442141]], [[0.18759460455120669, 0.41154225875321604, -0.79818479795292552, 
-0.33827318340352441, -0.18001588147373915, -0.4792444579810422], [0.42982983325732316, -0.33155040902368893, 
0.42290450465282525, -0.83130715637759756, -0.3032188844864554, -0.32076840026535747], [0.76309763596897717, 
-0.16224259917231953, 0.15941745546495367, -0.48000416396167833, -0.51662007720067038, -0.16013023529096326], 
[-0.14551226482746693, -0.23868762789628617, 0.43760559780914088, -0.21403294145729168, 0.39050610138521646, 
0.027305986693560991], [0.68831153064682726, 0.40454026449346192, 0.45400060273514886, -0.58424538761250466, 
-0.76111681528727737, 0.27256331383667654], [0.1044332247399884, 0.2837991621031386, -0.001203020965017787, 
0.42943971468744824, -0.12224867720143306, 0.91580247420238114]]], [[[0.30981712080876034, -0.36461363844674954, 
0.25883776564780958, -0.34688945167117757, 0.040345424826182974, -0.06749832444068371], [-0.59105551888762986, 
0.69516128362058893, 0.26515800998959582, 0.59395820933112509, -0.95188121043888763, -0.043176224324436241], 
[0.074806347249416261, -0.37089306088992502, 0.8868053047477753, 0.22406455027035777, 0.88558318623287446, 
-0.048633060381149784], [-0.61012882139642399, 0.25806095788826244, 0.86588302539239126, -0.71237955335662217, 
-0.27929397861787764, -0.39826587067000041], [-0.15482183572226282, 0.1941180801783895, 0.396172098715869, 0.25696327959113985, 
0.63083195875557063, -0.021369726734949746], [0.69601966016372763, -0.084522963407117624, 0.041632530866564821, 
0.81361655503790375, 0.27544706507804895, -0.033891240365442776]], [[0.75004552156235227, 0.46831759183775734, 
0.30105823347929972, 0.65776586383338564, -0.82850270140457916, -0.36765087940084573], [0.6870863655414694, 
-0.88428011094818038, 0.35015265824894093, 0.86987640726315019, -0.28583378008722415, -0.16887142765052854], 
[-0.89213933381760668, -0.090914363084653615, -0.18502528958761433, -0.71744634023890197, -0.30218811909917043, 
0.19059736943777628], [0.2309838197094658, 0.61654339351992848, 0.93953945377266823, -0.99794399451758764, 0.10306486074324894, 
-0.30588044729010178], [0.96944831992506297, 0.43555570504661301, 0.67409592590958956, -0.73733621251980219, 
-0.27819589250300769, 0.97623067329564095], [-0.19875664545795457, 0.40016733425825723, 0.55053455538649265, 
-0.41542042540123059, -0.47755657862976575, 0.91166965504297037]], [[0.55465569222257893, 0.97801518537416565, 
-0.8384975405911741, -0.78893677175037857, -0.068837278369803467, 0.43298910997710416], [-0.89117061130346542, 
-0.49643455515898949, 0.20412441612280663, 0.20258976309177568, 0.66869866044785575, -0.73190022088298479], 
[0.30014196944494187, -0.98323354977000688, 0.61366923227760584, 0.9433600066705714, -0.73453068463833437, 
0.25975429241132009], [-0.45217958373518941, -0.018505584392769059, -0.54670764444194386, -0.23769720757579771, 
0.18925725348298461, -0.39177652648411843], [-0.8982516552003692, -0.46705650389443942, -0.70431373405436704, 
-0.77197332405835328, 0.61912038014716408, 0.35552133859893198], [-0.9074548436712726, 0.98084073586701348, 
0.75377940517744957, -0.0029510427732073463, -0.79558858762508322, 0.30955409625630081]], [[-0.13987273036615644, 
0.7527124519116839, -0.42952214112672604, -0.9199792435932983, 0.31299963895711103, -0.68850926629826525], 
[-0.93276653685217537, -0.6364468123563487, -0.64491273150300832, 0.35682825552127895, 0.3957488659093753, 
0.42114699035825409], [0.97392489895564882, -0.48662157646722193, -0.40327170782524902, -0.26138505710821169, 
-0.31059068850359739, 0.71238931649011494], [0.50425796335356177, 0.32690617734482408, 0.35774894798501822, 
0.46713289679387215, 0.57830551050541601, 0.60128508117768709], [-0.95699828479277893, 0.32147874780151464, 0.584535371003158, 
-0.80963008069875864, 0.46138534924120145, 0.06158733072193523], [0.57057357914499174, -0.38696862134399068, 
0.5898263650043043, -0.58047870667267243, 0.027690322946750889, -0.12972036067137638]]], [[[-0.73984548530094196, 
0.21729530014410625, -0.26379631268813908, -0.4261841814481957, -0.38220994009057363, 0.46077259115551872], 
[-0.86268868391703135, 0.59393767507038109, 0.48366312258474919, 0.43826892355571578, -0.12387619717030218, 
-0.84127908265918427], [0.51260811496892056, -0.36856743498475386, -0.92371152168924664, 0.059078514673276183, 
-0.90142315987201127, 0.90203914782271633], [0.46085674299457535, 0.51811670247114927, 0.79969250123322011, 
0.75192609570814417, -0.31612148976682031, -0.97644080596808513], [0.49479223700311037, -0.46536609648206806, 
0.77219839388352418, 0.18897428247961279, -0.55711191123380588, -0.5178460605712687], [-0.85111429221969503, 
0.99688411269626798, 0.24889909402941091, 0.52646665153290617, -0.84288852579381812, 0.095982918924786498]], 
[[0.41086399771638571, -0.32402913536636979, 0.41294509415445368, -0.79046183773338052, 0.30255346194134125, 
-0.54632934794436094], [0.14708212960941225, 0.88219210644472468, 0.40905218473126004, -0.07358844644983531, 
-0.13797832456204739, -0.77014012428157463], [0.76101806271230998, 0.016740225694247934, 0.67946175613441784, 
-0.43786494073285342, -0.44533883917625583, -0.3519667452524311], [0.95575487200495446, 0.67852186369000811, 
-0.97062795918612244, -0.47726635462436851, -0.094226390579245756, -0.27522974550385504], [-0.73917996379396977, 
0.31779421666342711, 0.22215571199713202, 0.68513784380791387, -0.73195172131436692, -0.42732482519613146], 
[-0.27002741717194434, 0.70452364644180099, 0.50310413146942068, -0.70939136170013795, -0.37467717659505295, 
-0.56434906180623345]], [[0.88883292507889777, 0.84335123909565746, 0.088611903545968707, 0.69165593298724315, 
0.10250411636641488, 0.96793214332795485], [0.50095630804735025, -0.74607571965301211, 0.51986034017520621, 
-0.45264550154774419, 0.038746546558982242, -0.0095903745022796194], [0.45151974988098509, -0.082032798524004136, 
-0.45562436000689677, 0.8234626074771596, -0.095955873987882967, 0.037288541709083356], [-0.30924230493076466, 
-0.31999268285109372, 0.39136247659988865, 0.93218659676514504, -0.5696837341558536, -0.95735108957544757], 
[-0.73135187733441587, -0.019874380167501871, -0.55656462336559254, -0.93826926452480652, -0.13527688334290944, 
-0.80999940401410742], [-0.86598956455746712, -0.72403171807487587, 0.78608521332121617, 0.59649000937297614, 
0.082554443500231667, 0.89195901106248332]], [[0.49000804715071267, 0.052809271156792015, 0.33351139136385877, 
0.37194395812974812, -0.64241588134954175, 0.38463896270194664], [-0.27152234452174229, -0.085805997788434007, 
0.55983845271132426, -0.70082501997797175, 0.59546595765732002, -0.93273307898439617], [0.081296569765666238, 
-0.90779142690080139, 0.81698221104923396, 0.14186188557911428, 0.16103080243959123, 0.78733049780850917], 
[0.85896207462250618, -0.28897004398366133, -0.68042695066009751, -0.92833928803987376, -0.57144509316773706, 
0.66066541830067282], [-0.35011016958754326, 0.308348658001665, -0.80711592082779493, 0.83789940682953068, 
-0.96205677005147483, -0.94521337605128797], [0.93185146298218724, 0.048126782467996732, -0.60247623496749214, 
0.36557293551570291, 0.060610484288959432, 0.71260441539900565]]], [[[0.88138261190717837, 0.026576605665631803, 
-0.83344429909895856, -0.27260635405986111, 0.39992951305512325, -0.63535013025300136], [-0.1878458136047314, 
-0.33921609848326484, 0.81524894412896587, -0.48702861016770571, 0.83636050002744011, 0.090205162520172077], 
[-0.95125917684979533, -0.010584371335448095, -0.00095065551970430207, -0.19991922816339125, 0.98912025173726836, 
0.71799259959497563], [0.12853254497222055, 0.078709584216896777, 0.7092513259326203, 0.77435675055228304, 0.75582810938673006, 
-0.13007717320699363], [-0.19462042911921373, -0.60354632553540344, -0.29301254903810658, 0.68227428445659299, 
0.96198371971229402, 0.72629829476527519], [-0.29121494184404417, 0.75712305322325202, 0.47397062780133581, 
-0.78470112091462041, -0.024797611638572636, 0.13061272480639974]], [[-0.79958644885478036, -0.3686918514395352, 
0.9582241585479907, 0.50978012627633418, 0.3805052193927918, -0.53145829710708914], [-0.93644514753081198, 0.9340767946289259, 
-0.38772893060475755, -0.92902267948540218, -0.24332948016173028, -0.63485994642641685], [0.19813166652330105, 
-0.88724097447397332, 0.048831239264311765, 0.28535030186934085, -0.81159584819410102, 0.90391954951859677], 
[-0.55627605603832286, 0.80652782171007931, -0.60441376433103433, -0.50468636529281952, -0.0011052890144653826, 
-0.14706717083262721], [0.88152003000248147, 0.88126628713750477, 0.045431085054144171, 0.029046165840170834, 
-0.79493945029139312, -0.19574334753322176], [-0.99143826403237867, 0.71936311293514321, 0.15063964704847943, 
-0.85422705959213396, -0.56329062123207296, 0.33763289589385503]], [[-0.64729748387907127, -0.6234614896998838, 
-0.68750220072353385, -0.9860460879815911, -0.99071076025284643, -0.576056999929629], [-0.14140421206763198, 
0.2666289040577976, -0.0093694807500968036, 0.2542860333439021, -0.42195016826447218, 0.2844927520917786], 
[0.44626470834219401, 0.004342136834977639, -0.28510739159369125, -0.42923561327962956, -0.7451105236288933, 
0.90040213558604187], [-0.70851069924878773, 0.2571440874375186, -0.42775240421299698, -0.92087030126833391, 
0.99908074897414534, 0.26247657271054314], [0.6277860257679484, 0.60402949682967644, 0.83917659069692574, 0.88835091473496175, 
0.26532522425903959, 0.84237176860108032], [0.64984450655947645, -0.33864724932560608, -0.066316752405876311, 
-0.99357691492671862, -0.37957332032038771, 0.044171614514355584]], [[0.6087021727462909, 0.9342521272949933, 
0.48153531026886376, 0.38740327696846943, -0.77667565435837349, 0.35142545128173164], [-0.61507226397388615, 
-0.74982723244207783, -0.21359607613656739, -0.61492739113511297, -0.421872873877708, 0.067416061929155946], 
[-0.014485538555476962, -0.19446990671526887, -0.85012621504925301, 0.36321452009219302, 0.7152817595857015, 
-0.7263983050506484], [0.6228875127536202, 0.25779359782233358, -0.91348251815638748, -0.92431888296092257, 
0.42726078108109866, 0.3898240710918548], [-0.48388863288686723, 0.11843714412218276, 0.81152339012279739, 
-0.97518058856722845, -0.28663357007671131, 0.60809370914656569], [-0.46504080406731707, -0.9929278407266322, 
-0.15213782813614585, 0.94419616663042549, -0.68356018342879632, 0.98660828991432914]]]])
        yy=numpy.array([[0.47556134085663526, -0.46564646655884534, 0.11376384661853112], [-0.41411533402237577, 
0.8259403407294581, -0.035138196964627832], [-0.31174879258766364, -0.64289117067285306, -0.32859956859166273], 
[0.25328053288051766, 0.78930945098426397, -0.20367646679769491], [-0.2402650481929558, -0.61095291516588812, 
0.10550861822363933], [-0.042553689185483856, -0.15747551337135102, -0.46490661017424251]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_2_offset2(self):
        sh0=(4, 6, 6, 1)
        sh1=(4, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.98127973655438017], [0.56705574313142626], [0.52416247050909948], [0.11873324519330697], 
[-0.21588096883363272], [0.4525986054701423]], [[0.35035607033841387], [-0.55479794819525319], [0.26951101186739534], 
[-0.21521351865306615], [0.2940319428993603], [0.34073802540587428]], [[-0.32772667717426107], [-0.22067748502819273], 
[0.87069974239612136], [0.58423914492651741], [-0.070106051026766369], [0.22820919778176152]], [[0.83815446080176814], 
[0.15740975932821399], [0.32680998647133497], [-0.15380299954436349], [-0.2823173955512932], [0.19201025700744645]], 
[[0.069381023724411905], [-0.10996561214877931], [-0.023474299649914432], [-0.35855037070669771], [0.29598840056404674], 
[0.71362813922484536]], [[-0.40517374601044098], [-0.89754115686905123], [0.50421181926911096], [0.8179109881106712], 
[-0.33958461371625437], [-0.71078562154940417]]], [[[0.74283132820206577], [-0.29266741060610602], [0.14619239334282708], 
[-0.61352059343549947], [0.69728792314157317], [-0.17308997004261073]], [[-0.19940156640121587], [0.57063482482847627], 
[0.61033400867487941], [-0.99630308383745647], [0.7390931826286522], [0.68790940383957544]], [[-0.28114056399620768], 
[-0.56615543640838117], [-0.75169110485056101], [0.49451023719404708], [0.22934209311908815], [-0.37226132243714605]], 
[[0.73987173195337719], [0.056785594801846084], [0.88748517455699738], [-0.95317159451145628], [-0.90711413143687847], 
[0.68072304598560529]], [[0.99612358739598594], [0.39720823579676834], [-0.92505634569836825], [0.90888792661999385], 
[-0.46816660932655618], [-0.59257079892148012]], [[-0.52227275422445518], [-0.24092858960196906], [0.77330299387634627], 
[-0.090804047638078922], [-0.94012998558468519], [-0.73428842939886874]]], [[[0.49147612502533522], [-0.50657800531700925], 
[-0.55787177199492111], [0.93558375250288783], [0.7335663414052167], [-0.094068797789865766]], [[-0.64836799994145289], 
[-0.75072250047975153], [0.40431771521844029], [0.039675465229186857], [0.90883212788072254], [-0.15744058030586516]], 
[[0.030732073041956687], [-0.93123429102872501], [-0.83830335969583003], [-0.97289431327013731], [-0.3269904759986999], 
[-0.5805274350397569]], [[0.29011670529906497], [-0.78612556514085474], [0.96190898801267477], [-0.44077902645897038], 
[0.92838110691612585], [-0.84648129705242137]], [[-0.27002674963050199], [-0.76537968303156845], [0.97986155800169006], 
[-0.67568480926563224], [0.62529028017416], [-0.69758267616303882]], [[0.14314091631168924], [0.50601578927215973], 
[-0.40960494452846974], [-0.50147319989870254], [0.83266787289332167], [0.24377699313960588]]], [[[-0.80040214979009661], 
[0.50861666566378161], [0.56936765369009867], [0.73190185652205364], [-0.78920666354293267], [0.29184859534732577]], 
[[0.88097950128303615], [-0.62757669770092517], [0.33845594156466352], [-0.073175538387391326], [0.59062330250380457], 
[-0.88680260004504263]], [[0.51205293974960919], [0.5909231873156604], [0.34495907961497041], [-0.29267785816646286], 
[0.34932548703977906], [-0.41311908695715061]], [[0.78584861017521579], [0.58944717357442311], [0.068968517210967972], 
[-0.53711969516165436], [0.94217922169882629], [0.80535847379068737]], [[0.37853799889466599], [0.77513127988361008], 
[-0.66305740716267292], [0.50583597503389122], [0.8346883921856898], [-0.67378694383684667]], [[0.11778302708031441], 
[-0.17669871377844748], [-0.4869571360769942], [-0.34137756231163396], [0.28336048689683846], [0.13500816923088799]]]])
        yy=numpy.array([[0.79335178651980254, 0.99789903701675975, 0.68174303418632687, -0.7962865046019092, 
0.9122885506092675, -0.44962997124911652], [-0.99824446979563053, 0.25349622041878295, -0.82921478893669853, 
-0.40291824641745722, -0.96016814153923269, 0.77926757833454485], [0.43027352753458059, -0.45630817295299875, 
-0.21939554305120712, 0.99671854985501795, 0.55243577677013578, -0.89638350258816324], [0.27331333161151927, 
0.66618708957573802, -0.58274054041781054, -0.34581266322132831, -0.50258922978513509, 0.0072889102404314343]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_3_offset2(self):
        sh0=(6, 6, 4, 4)
        sh1=(6, 6, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.58921608583228813, 0.4533538080572288, 0.37925541011552966, -0.29592204986434312], 
[-0.89545329748056224, -0.57806212894923625, -0.79979780574780679, -0.42406533377160094], [0.95727382823587903, 
0.13118163548389927, -0.89833293471379627, 0.08629361736970842], [0.12545248896500749, -0.83112923686712969, 
0.34047830848758731, -0.54455580971716988]], [[0.067230866819495638, 0.54063223888056378, 0.86630312179407287, 
0.33825740231942314], [0.37175817534846223, -0.79990870043111384, -0.7424778710920017, -0.7919578180394431], 
[0.8974726800028896, -0.43793971932517639, 0.90926998436637807, 0.4375677534162048], [0.20369453047053709, 
0.059124926564198432, 0.3064536467714023, 0.46729372783492318]], [[-0.67432221643637313, -0.97093125846548034, 
0.21974819456782879, -0.083343263222844577], [0.41600811078006128, 0.7928483180292154, -0.17470795913423398, 
0.012162767827991638], [-0.28942010861257339, 0.4502556911758131, 0.747268252377987, 0.31311048129631147], 
[-0.76481996783759576, -0.3364271858984067, 0.67239423573264334, 0.66333399297884621]], [[0.17810546404619587, 
-0.61707194427589473, 0.67438461744227607, 0.79714143453249275], [0.97673081057497413, 0.72461767749430805, 
-0.56470721935580404, -0.39987189507880472], [-0.66721719553489867, 0.44511521544135513, -0.72494069495483338, 
0.51103868789114615], [-0.76779884609315419, 0.29327051264095694, 0.31985280571383101, 0.93986896192840375]], 
[[0.35157726794336264, -0.906015971721712, 0.36666993599723186, 0.9031370483769452], [0.64553659330904645, 
-0.56422700291528338, 0.98653505338960068, 0.72108246581554658], [-0.97563219585623218, -0.87838462598174027, 
0.81951064672515583, 0.73686889370405217], [0.80835004892768603, 0.058169401683593414, 0.66962202720341746, 
0.96371999735121827]], [[-0.63100018258573565, -0.68234053575987397, -0.93783174312028628, -0.82063275684056891], 
[0.91119315759672537, -0.29334291059799278, -0.35882225323770811, 0.9073216486591642], [-0.069394683328207307, 
-0.23702623049058902, -0.58664471524901995, -0.020657395415513369], [-0.24029669001049836, -0.68227230687821661, 
-0.31551024169882025, -0.84923175466556278]]], [[[0.60131079602429938, 0.59955280075661332, 0.92108480746349319, 
0.73864280149165329], [0.11028401494116169, 0.53004396178035007, -0.38550792686306368, -0.93144909251268682], 
[-0.5242303659208214, 0.90160573450357462, 0.39879083698461559, 0.64125838159806059], [-0.1793936914390728, 
-0.28418214170846934, -0.7212699109336802, 0.45072195471172449]], [[0.36104003247835803, 0.57816191948854767, 
-0.30927234459263264, -0.42446508320917187], [-0.23548891244455405, -0.32792973108769696, 0.37571133870069295, 
-0.45904448520316987], [0.58169661316890364, 0.49924159705901494, -0.37100516090631097, 0.2430590289225838], 
[-0.54094061511151526, -0.64777902203476723, -0.47236375149126753, -0.015479218823038376]], [[-0.09056176887207168, 
0.22621954131879884, 0.21835410664186528, -0.87245247424384664], [-0.14457010686963123, -0.42927175747766011, 
0.49555610732414879, -0.35067487465756519], [-0.15185723905711779, -0.28198901529616172, 0.2173501323192919, 
-0.59631338020450309], [0.25461918751115942, 0.38213733937411432, -0.92803193448824861, -0.74787589321080361]], 
[[0.53888983210140773, 0.24821796049823019, 0.24706912519706581, -0.46083444748257407], [0.90140236728287815, 
-0.62870683445951636, 0.23681207543000937, 0.51955103700646399], [0.57075750239988432, -0.69057666628593894, 
-0.35655213288626464, 0.30750269553872123], [0.26951477817073588, -0.42027762809231772, -0.0039235621101343376, 
0.82352407995322263]], [[0.13556932151029577, -0.047650788641299568, -0.044991408288444168, 0.99774981761230586], 
[0.12237241180974201, -0.86217817070021341, -0.56672228389410062, 0.92586871663634263], [0.45925524924880601, 
-0.32975353483738967, -0.59959793714491516, 0.63026939628569445], [0.61755017859279149, -0.31668583261759742, 
0.81880288914646426, -0.22964709525799809]], [[-0.76988588608672237, 0.38441093290096995, 0.29736772285728907, 
-0.28248744181747143], [0.36001419093992304, 0.87422010740393197, -0.6278063242412959, -0.89668731670572366], 
[-0.79334498575568269, -0.24953094305081103, 0.04216924429105573, 0.52624274824727468], [0.14585995664614293, 
-0.23967255030242418, -0.27162486974214972, -0.43722715039069349]]], [[[0.38014220683635824, -0.89473401070262293, 
0.85440948256729699, -0.91186506070494722], [-0.39144647395956755, -0.26355355789221036, 0.24220822874357828, 
-0.54787157184360002], [-0.3811447998145765, 0.68906273929572737, -0.081910721631776262, -0.58429283528241105], 
[-0.25626663704244801, 0.92106179726629089, -0.43174192120428057, -0.20231193815201753]], [[-0.75570570493980127, 
-0.012964630047085146, 0.16448584105835629, 0.090228308341313346], [0.079512259343843539, -0.6369578411471144, 
0.20540673753344896, 0.082339593760119678], [-0.048407245015036349, 0.75729557685666449, -0.85774531124382936, 
0.22661772956896975], [0.15549930045850835, -0.38224585883665374, 0.4873932591745338, 0.17593724752072926]], 
[[0.41787128361998782, 0.65017610588309016, 0.75068199237499167, 0.97675947185256873], [-0.30101206347770959, 
0.0070118664953899845, 0.52944067431275066, 0.12049995365666155], [0.98344682907987258, -0.63720257818900894, 
0.58502873527295729, -0.1845012719837591], [-0.43121552761883564, -0.97248486848547744, -0.038819867737388103, 
0.55905040595721567]], [[0.66196871179909955, 0.64221518976518377, 0.058728662622099526, -0.86140970730781752], 
[-0.31108885608978132, -0.73587108287033298, 0.97550820170707397, -0.53605069491381152], [0.78838374933066135, 
0.44995497365458892, -0.84032216201267329, 0.47224296964530921], [-0.74315104992894221, 0.98288463980901764, 
0.89522593262579719, -0.90946787617475189]], [[-0.29811115235929675, 0.44792580167000273, -0.98899225638849697, 
0.066225805188364628], [0.43693689093799359, -0.07714809205307871, -0.70396709763816578, 0.41892809481759707], 
[0.22431258843385549, 0.41094626113705179, -0.50843392834620627, -0.78831157178013855], [-0.11524139701219172, 
-0.20156961213863767, -0.98291142692542355, -0.26156502990734776]], [[-0.1702310042043671, 0.26128063406808844, 
-0.32566561974358388, 0.80815848666230905], [-0.34921783088189828, -0.75338136725302651, 0.47151943332616963, 
-0.4233137588399416], [-0.88841850645639187, 0.88842493325840799, 0.22954403035673376, 0.063426850963324899], 
[0.23479799058470419, -0.78489194334660439, 0.29615717767414429, 0.89522166029999606]]], [[[-0.91328907223478284, 
-0.99931755455900584, -0.10040099668554214, 0.33929062019611056], [0.37962242483366793, -0.69781070395154088, 
0.12207107858218147, 0.69878762143244066], [-0.31721105498252866, -0.66431936214981935, -0.18923329927363919, 
-0.048055389783109526], [0.69490855244652305, 0.5342903383859261, -0.97144951414347469, 0.71260758227517251]], 
[[-0.16171129586332622, 0.31559644238329532, -0.11373614707875879, 0.15810078225707791], [0.98138813329239238, 
0.065422875669246761, -0.8731833981305448, 0.25134315757755532], [0.12002033829810443, -0.61915589639924762, 
0.14232125391690986, -0.92022851764276092], [-0.46212087661347123, -0.71206796366502401, -0.49165652448283792, 
-0.8539392550553051]], [[-0.94849753565461303, -0.041442389654873013, 0.85793106719495449, 0.95399798037283268], 
[-0.2054947537347156, -0.45043552307690349, -0.39902886085066158, 0.90284950242636208], [0.17374650276557202, 
0.59114290081617105, -0.27897984030780743, 0.92718473553071346], [0.903369382862385, 0.94265155043706317, 0.52665984753776862, 
-0.18721554013407715]], [[0.26339636428089719, -0.88589438378686736, -0.011999599170406272, 0.4437304692916364], 
[-0.32765466234920315, 0.22095429350686002, 0.71419504730817018, -0.1156321482478504], [0.060495367591797189, 
0.095975920635750667, 0.0080602104058311586, 0.4243971442325849], [-0.043392391955357601, 0.67725149030317078, 
-0.19055190815852474, 0.6487545256698517]], [[0.76742631279243256, -0.82166562415523603, -0.59043569244812355, 
0.82690924040532465], [0.79337810590577007, 0.23607720053032333, 0.98096216672426739, -0.87621297972599499], 
[-0.55532608305794318, 0.81943908580608849, -0.46611170254651868, -0.77839084667028491], [0.87837277064464936, 
0.66074360732912396, -0.79553634455927003, 0.72040427895338954]], [[0.3425521317156317, 0.64652477128130292, 
0.37686334374849384, -0.9822389818565791], [-0.99579608349495552, -0.7536809764077288, -0.53784513194677253, 
0.99080678756978657], [0.011816743813225727, -0.13510988009680336, -0.60028831565222251, 0.36743811878448152], 
[-0.15186966113342537, -0.43310118173362233, -0.8995642502112573, 0.063218366189984332]]], [[[0.14604061225068743, 
-0.68998735371389142, 0.56640635636405778, -0.32202791001139963], [-0.81188229898335806, 0.42310639457044519, 
-0.87356447805038129, 0.54307412971856017], [0.26613944082038721, -0.37924387627481604, 0.53096495869162275, 
0.90674353879967917], [0.64949374673184246, -0.25734004732929416, -0.017843220589131681, -0.14799905930306778]], 
[[-0.31544406289218507, 0.95821082822495485, 0.18196005974524998, -0.83218358053604868], [-0.84150514093041462, 
-0.89310047056945718, 0.91275019907711052, -0.90231741083912764], [0.18842446519652056, -0.72526693253884256, 
-0.55923271484178838, 0.47741722161639566], [0.70587124891041997, 0.027118401905467993, 0.86317570758897633, 
-0.53334850879224072]], [[0.56802052353195065, -0.14499677923839793, -0.47528374291429865, -0.099398816706534499], 
[0.33229490127747763, -0.41518528765433693, -0.028293035623634166, -0.23914267392571231], [0.13795465074528379, 
0.1715275225966495, 0.73618573986646996, 0.77413005528757894], [-0.41923251296915987, -0.55386876483412739, 
0.13238072298878123, 0.15357649985091548]], [[0.48708161156694607, 0.60273487971508, -0.28321106501929227, 
0.77950268327635786], [0.078377123093670953, -0.81271475285237771, -0.57651440047265612, 0.15812166379855852], 
[-0.55794738323133908, 0.16720153369456758, 0.35428465206250448, 0.74041917761851783], [-0.79282912737192301, 
-0.9739399690855699, 0.60428794398697505, 0.77924192424747418]], [[0.40320018762044585, 0.53036013267039106, 
-0.18709092688419071, 0.92727245542109582], [0.94775607696188291, 0.96374176581707549, -0.61077400196000586, 
0.95433446556611012], [-0.20692444957404188, 0.054102638514234691, -0.42054722588368332, 0.12794081621355313], 
[-0.18343943394045414, -0.40077951891516883, -0.42584005948058823, -0.23292627352422457]], [[0.36334568007053036, 
0.41039777420685075, -0.12889508340242695, 0.59673306997922659], [-0.48584701847888412, -0.23347167747972297, 
0.18989235677639127, 0.88772205698943862], [0.72526310626110413, -0.56223575862030817, -0.44959269883970543, 
0.92628124457227012], [0.38548402897255474, 0.85939466707328949, -0.84077786573364466, -0.96849077360291203]]], 
[[[0.12936769634070444, 0.18835860362721824, 0.53424478518723073, 0.032750022287083125], [0.75829147894483384, 
-0.44198728551480038, -0.8141646674507943, 0.76599302208779796], [-0.14900826186177607, -0.069931249326097955, 
0.2755613797240577, -0.47730476020311396], [-0.59571222398179469, 0.31468440042931745, 0.55423497073541972, 
0.41185303354213443]], [[-0.55384103989303157, 0.84520182577837422, -0.9866118636788006, -0.12994185647887102], 
[0.65190704923864451, -0.17560123028904306, 0.87227310849993778, 0.54231387563311184], [0.18854618150384561, 
-0.39281617043940287, -0.75636272629145673, -0.005405683479775325], [-0.19942043278674348, 0.23387571256038786, 
-0.070187828653202056, 0.7636968181065571]], [[-0.010285118314055541, 0.64367187404303072, 0.72958935578167261, 
-0.68249611939263022], [0.11084016740164948, 0.25132339764553646, 0.25791786545483797, -0.65898677029479069], 
[0.56359539211418608, -0.950966820517259, 0.43048937544985022, -0.62325190234495986], [-0.043914929622699272, 
-0.7088203562976565, 0.34971092212829546, -0.83932957134000685]], [[0.88258796379929905, 0.66943389677871612, 
0.17728147992587617, -0.75268784140135181], [0.63014233485038584, -0.03359096277258522, 0.69665330890230615, 
-0.47391095446434828], [-0.057643811335423845, 0.18896521777003383, 0.11937404611538227, 0.58062957915511682], 
[-0.44761766978470963, -0.89254225173932356, -0.21843763079866707, -0.26640504828221379]], [[0.053141273394807786, 
0.27580647995085639, 0.93792577275775613, 0.32189172455623249], [-0.89470207395930701, 0.30055705299653379, 
0.82094387250460743, -0.63066130007792309], [-0.28991938831520936, -0.55309964895035701, -0.80576380633126621, 
0.68895642079525343], [0.52870128972429264, -0.67059582107477178, 0.73613575435895418, -0.073777032517626351]], 
[[-0.78343651073724985, -0.14940513277441214, -0.84686039944015068, -0.43216423073345922], [0.99880729926478962, 
-0.8789281828012887, -0.73049952577103228, -0.8809139928559282], [-0.26408948715693703, -0.93382431525076859, 
-0.89956623789953039, 0.014860175231706352], [-0.31497594678443042, 0.34742686350640839, -0.83736284649039949, 
-0.48700054072893906]]]])
        yy=numpy.array([[[-0.10943261775212054, 0.065159928327866767, -0.5672548565427753, 0.40148027341505688, 
-0.052436386768943244, -0.048569461533888525], [0.076023736102825534, 0.090684176244226311, -0.50931177494362867, 
-0.3383407375650862, -0.59297949141156825, 0.019944796073826154], [-0.6635287302076569, -0.54827570786159963, 
0.27093840912556777, -0.21097720343403337, 0.82178352451781089, -0.36738868785953738], [-0.4436439703442836, 
-0.0042391517971547277, 0.99798435694134291, 0.51015605283833576, 0.82590939202418046, 0.80528562353482602], 
[-0.93497583548379937, 0.86567817831248739, 0.20850492600055071, 0.15911230643598784, 0.94873451605500358, 
-0.78712765800993734], [0.048866787359620467, -0.97484301308769261, -0.8289042817618355, 0.48417295754813194, 
-0.8703321313594079, -0.2018966107192357]], [[0.45200096027828796, -0.72938930125012602, -0.6068084920466732, 
-0.67505671817351875, -0.3522142664355985, 0.20804894752223668], [0.72532265706362709, -0.80002465099276932, 
-0.1680318465859798, -0.47942068516984104, 0.26152413503613836, -0.77916615505470288], [-0.35926109759425073, 
0.40128233592137752, 0.60676287619656089, -0.35882663540529935, -0.20558678304542943, -0.48335234434253316], 
[-0.63374991413427595, 0.37131701592521194, 0.080525342725519478, -0.06645896380367966, 0.97692354787558489, 
-0.59058806836570943], [-0.21942802369814052, 0.065974552604908698, -0.090176486907281639, -0.54322660867892214, 
0.97292578037119659, 0.73537808455095655], [-0.77120499844116064, -0.46530162087362004, -0.80596575231113254, 
-0.16885399690066349, 0.65436025158740985, -0.1662362584953363]], [[-0.70210262561396908, 0.72413610008760565, 
0.82944114416792991, -0.88510252677150825, -0.30056176076474039, -0.78390253228714157], [0.48404525626374673, 
0.86271986781885901, -0.82397012322769481, -0.80829418850975521, -0.40787180932697331, -0.05823594378075847], 
[-0.64460637595782355, -0.4652428520637113, 0.13678792766630687, 0.59022126586044799, 0.020826402054963333, 
-0.15031132492688437], [0.42816205663357665, 0.12903196869088318, -0.23109958532463493, -0.74478034364157586, 
0.41286558334939572, 0.064533538216738462], [0.13985324381302222, 0.12451533390994363, -0.057984137183421325, 
-0.083768355446046883, -0.036210702529995586, -0.57589691674894783], [0.65980912220684074, -0.94051996319169118, 
0.61862266269297228, 0.21746268602372898, 0.017558843947999447, -0.3386961553241179]], [[-0.37569938444597484, 
-0.31303510887712838, 0.59909456006044803, -0.34362768978079306, -0.98882431833797524, 0.43948218541046158], 
[0.28105717039681655, -0.51711669564619611, 0.61357538575433601, 0.68098474970520151, -0.77047535459937389, 
0.85712092617605551], [-0.92572567516910964, -0.43267096629254653, 0.71065856456840115, 0.62654611866522281, 
-0.15772486061867297, -0.5271381119053189], [0.73622767453559645, 0.069825141296371962, 0.76793540198928856, 
-0.46933056617002378, -0.47052593837997558, 0.97183121317014254], [-0.35710340862754397, -0.40575996043295892, 
-0.86911663864248401, 0.10558309138230371, -0.93638891598122487, 0.18441004023764984], [-0.14530264441340845, 
-0.68086481469970761, 0.94383511131769326, -0.44974064704108008, -0.74891870898217094, 0.43023459671017572]], 
[[-0.097109330203994482, 0.52254396906358336, -0.39112656153004277, 0.21855840368826374, -0.9963018494178173, 
0.79662921190109937], [-0.02465442416244068, 0.97246475820867162, -0.66733232076774995, -0.74297448873114424, 
-0.46279120826053255, 0.95973194490548308], [0.70386504613542877, 0.79209646743156248, -0.50920717751140132, 
0.6733362916647474, -0.76222221899179177, 0.29409942578353676], [-0.11007830626383486, 0.0082205789795026796, 
0.20370802792596043, -0.54725316728268547, 0.15123035243707039, 0.35821753767487663], [0.11152168216044545, 
-0.2183994818881243, -0.36529050797617901, -0.44604185831302079, -0.89721576279033144, 0.35473490225940862], 
[-0.40962593763125921, 0.59947258433375805, -0.065204000597408829, 0.069475616702755927, 0.26137501559438503, 
-0.23500232587481973]], [[0.3642803649864379, -0.83976733601275999, -0.27411660386100833, -0.83732543418613758, 
0.2942137265649738, 0.94629376413911559], [0.99690522160040018, 0.11665653788322539, -0.024031636747015694, 
0.10768571706850638, -0.95358825084792453, -0.25454263723517911], [-0.58811173112254944, 0.97337984503901254, 
-0.97365700158784585, 0.17063141318878761, -0.51659431876599915, -0.80652677647657289], [-0.56324269425012097, 
-0.29821500469440076, 0.61899206229709058, -0.51800394303048747, -0.35431923711702495, 0.096422286378439415], 
[0.72810260544824734, 0.70130424588106899, 0.48019960072261347, 0.8851063284697982, -0.47058043209747424, 0.89431858804104047], 
[0.23075200022738018, 0.78855281720822035, 0.92251308028039136, -0.88612600191063917, -0.61776216132314854, 
0.88555181719307674]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_3_offset3(self):
        sh0=(6, 3, 1, 2)
        sh1=(6, 3, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.16811535516881282, -0.31322675765514907]], [[0.096264472348501995, -0.75692077757225062]], 
[[0.18466940127632347, 0.0064851412179189882]]], [[[0.96032060848852185, -0.001286721289725179]], [[0.5802889732290808, 
-0.85511648805685669]], [[-0.70086269082448149, 0.35794160364429284]]], [[[0.32844279051758152, -0.23350037935670254]], 
[[0.20799748726049661, -0.36450529708759016]], [[0.32598451044480381, -0.59075428102525218]]], [[[0.56834205989240805, 
-0.82824208104447883]], [[0.32863563228588277, 0.7867608967089359]], [[-0.85224580389368576, -0.4140261034109014]]], 
[[[-0.11399493854578724, 0.44561886943089535]], [[0.066684290573578719, -0.79124187599380602]], [[0.66871552873684426, 
0.87758577365314783]]], [[[-0.88527325747186714, -0.26082056375095908]], [[-0.012864770208408594, -0.13264623299806155]], 
[[0.14691677196450414, -0.10730505403168]]]])
        yy=numpy.array([[[-0.73806222475249283], [0.11420823919675427], [0.30866057891825238]], [[-0.50513257035740766], 
[-0.79561864435049912], [0.35635334447404254]], [[0.17078526030743646], [0.33357429210195799], [-0.29162263339297656]], 
[[0.86721630427554253], [-0.3789583925349107], [0.023156510667068364]], [[0.5125422983851069], [0.33520359459197935], 
[-0.037565195097772897]], [[0.88585789162322381], [0.57563686575191331], [0.20345011939968316]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_4_offset2(self):
        sh0=(4, 2, 6, 6)
        sh1=(4, 2, 4, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.77770163561880601, -0.8265372679670262, 0.39784403965776916, 0.39075760629833955, 
0.31732103692040292, 0.067073704350251395], [0.22889790802047894, 0.026359766204381962, -0.71974901818561765, 
0.99804721026857801, 0.1704923286713278, -0.0078038153551931444], [0.40467216707294251, 0.81859876326217296, 
0.76915326175129084, 0.87642312409852363, 0.86599215868185242, -0.58101435376308364], [0.051850693553855853, 
0.20314487121762803, -0.58063506016063493, -0.6075462480865208, 0.3039776245715089, 0.40648255483782081], [0.31718137591073758, 
-0.56247889081019742, -0.59373757088750323, 0.53846020340441303, -0.91367876137409154, 0.94356943139394933], 
[0.10765475462100271, -0.31797806869448775, 0.056723416235310964, -0.66282712572194402, -0.92403699348653534, 
0.55507170522734484]], [[0.71885955025369852, -0.13941742161006143, 0.66712622488887341, -0.57109456137762771, 
-0.63585681848758568, 0.14175153552257824], [0.078780171750396866, 0.73944524159945879, 0.27997534690934778, 
-0.22012505573567087, 0.5257427088281772, -0.29530000287094982], [0.53757040494303054, 0.38074828389439008, 
-0.99240312959475085, -0.09093942644301678, 0.29537543720938064, -0.94896919307379735], [-0.56913764633152608, 
-0.71133217918848568, -0.25195384172988322, 0.45003743520776274, 0.25159595331808315, -0.39615837346788885], 
[0.12655626708677592, 0.25450083736391504, 0.76377550349660606, -0.3037889079597329, 0.24131034409732544, 0.99351266176141984], 
[0.22746400947199064, 0.40280690110759743, 0.90972302754905132, -0.20120613992214031, -0.95770750508713975, 
-0.65510501878147287]]], [[[-0.28784349063669201, 0.85006926325218624, -0.4077755284394291, -0.21832223451125166, 
0.34711873639651092, 0.012559572940557651], [-0.86511804714293583, -0.74293262346719713, -0.013566377994936873, 
-0.2118694134028245, 0.93198420733004506, -0.67470320892695024], [-0.7487760963232033, -0.48840811909307535, 
-0.84546411740352467, -0.72337679225235196, 0.96054659482657323, 0.1582781735903851], [0.079266070594715021, 
0.79227727811559623, 0.075671517072487315, 0.55040978622004322, 0.14862331648914107, 0.4975520231280246], 
[-0.42625004206737716, -0.2504940280138801, 0.016176309800306932, -0.38862533525903964, 0.97652441994287309, 
-0.22847913153819999], [-0.39093590977032355, 0.43030716932478663, 0.013658190714318286, 0.82824066761174508, 
0.50846187081555283, 0.30516689180638079]], [[0.67739359658354625, 0.27324868692688886, -0.046573879191458722, 
0.080789564823340099, -0.036971586868108197, -0.23060015903659692], [-0.94106698849508508, -0.69864216777677068, 
-0.3126891384519912, 0.95164324189458105, 0.10797403015211171, -0.07407222661363777], [-0.46525540227146744, 
-0.57221534702042476, 0.93293835259605995, -0.88932251145124264, -0.84883533276135159, -0.34635219573730747], 
[-0.42135105450830523, 0.96252066923709378, -0.079617504835290509, 0.93103424463131934, 0.12029511587391029, 
0.9195247588509039], [-0.020235166552427852, 0.56254033122292491, -0.61057985694746786, 0.42625154702298418, 
-0.39883670292512519, -0.39787273879707019], [0.05556225278114546, -0.86663912733799919, 0.26729726379801133, 
0.6035508371684879, 0.24529245731079685, -0.8423003157917146]]], [[[-0.037890787913175572, -0.89180994355618393, 
0.28789397466240474, -0.026738944810720033, 0.16296371646144592, -0.78195318863394436], [0.41433456657899592, 
0.88891348280258686, 0.0092383289099529087, -0.062667895574535004, 0.19193086722655406, 0.61333938171407909], 
[-0.07861183684216666, -0.89215961261083665, 0.49860890601122021, 0.70872770977059329, 0.73427361897386989, 
-0.59531108837468638], [0.87264355673720084, -0.35828368296930058, 0.84067021513310491, 0.71827480474081851, 
0.7183570215642372, 0.47588371504207605], [-0.64173228717377917, -0.33364411905595626, -0.44610266680751609, 
-0.83388301107256213, 0.71255681605606891, 0.30293231891972794], [-0.098253003590139221, -0.8833311804267634, 
0.39548015470275888, 0.073863756784558365, 0.27067463489215204, 0.57741595019316172]], [[0.25626754862668544, 
-0.40394923921148518, 0.4585141171299294, -0.29954339751444836, 0.39485835580701667, -0.022985281227973386], 
[-0.44611656118162535, 0.2935071996288221, 0.085718408304769511, -0.54817744323483608, 0.4997481761440532, 0.4495048512532851], 
[-0.96080866972404544, 0.77972792032077254, 0.27453242031871716, 0.17309110146291529, -0.27663846138400694, 
-0.87442283510100904], [0.068506762205459815, -0.85134546223466923, 0.73349843825779981, 0.11529327917273924, 
-0.004685543480134724, 0.015332515621374609], [-0.76055792969925307, -0.39578905658070362, -0.56810544871712931, 
0.78455458390543042, -0.50881760848808644, -0.95928171976549725], [0.20868361527258794, -0.33870101667265073, 
-0.021189063630032257, -0.26709828117766388, -0.8541498871630917, 0.70873769334420644]]], [[[-0.080996935515520008, 
0.29804732624398356, 0.30259282879137839, -0.13629080167767316, 0.093119101454820941, -0.75071829671596935], 
[-0.89346665045758034, -0.24101641393846851, -0.72761830058586874, -0.98818010197787598, 0.46128344128303533, 
-0.2044207705115284], [-0.25527265365060936, 0.68960005610826691, -0.52411639438033109, -0.75375283278739769, 
-0.15406542197880335, -0.70071723773090611], [-0.4078801302038273, -0.39134250479734933, 0.056602919592547929, 
0.4667174948745203, -0.31945559189507344, -0.025705303352011821], [-0.078616560872077912, 0.45962635170567201, 
-0.93725779269175313, 0.79388038213159229, -0.35146571583631903, 0.80966142918056128], [-0.24699775768829024, 
0.063093340634764905, 0.17124902337515779, 0.72941497707016167, 0.98269490906860635, -0.12924237586515375]], 
[[-0.11700034502905177, 0.23286880414729372, -0.24037926372512675, -0.64143224136062971, 0.289617814819632, 
0.61067786842632388], [0.81510998372160737, -0.91646385016297871, 0.85284623575971885, 0.52793820180219497, 
-0.085797776526315594, -0.79381700304282754], [0.82867386277682487, -0.78120904573213568, 0.81649488513644375, 
0.15121924766499117, -0.65572361927150213, 0.6258724168234624], [0.42696718841938908, -0.64335577112616593, 
0.71744208873589366, 0.92545064798280641, 0.35477586323621368, 0.95481501132729352], [-0.58960761387516647, 
0.11819340995085192, 0.38477336843690768, -0.81863906102062112, 0.92820410596409242, 0.91334810348750706], 
[0.96308083778558062, -0.93197376979568158, -0.48934800474080364, 0.86699738343158184, -0.3701670606764027, 
-0.14784678240789928]]]])
        yy=numpy.array([[[[0.24098519751894321, 0.23284390936996013, 0.59749490109623249], [0.29542650715394703, 
-0.67310624740056668, 0.14492944434630717], [0.57843554872574243, 0.76406914329713693, 0.87384826097577095], 
[-0.87101166301537236, -0.18541806152897022, -0.53330272414355595]], [[0.44702946862565263, 0.43669239862881981, 
-0.85559532318638754], [0.32002776548783118, 0.20156715855770835, 0.70145695973276911], [-0.79501854759282087, 
0.76823782981965105, -0.89873207710620062], [0.92793036448943966, 0.011931004459246219, -0.53809527158892423]]], 
[[[0.58793083628361464, 0.60068166616322083, -0.5650645395250633], [-0.30806303036227134, 0.018065028130206384, 
0.76659187340464841], [-0.80114211789020273, 0.25687414101971928, -0.048721642785012564], [-0.55495680717091056, 
-0.52870603370366509, 0.28311122841746195]], [[0.65070550257980098, 0.85981948153497156, 0.11870112463018567], 
[0.32938799279352149, -0.032061896584582916, 0.94606897745703966], [-0.22519763065479115, -0.022320237446119107, 
-0.77404632316660926], [0.36172022862723741, -0.056537474626539241, -0.61756577427796455]]], [[[0.95073930647419802, 
0.29837760614803743, -0.13449979948352642], [0.29560586392715749, 0.4284507388522083, 0.72112395501954407], 
[-0.35193042180925649, 0.63490506655688383, -0.30628695957816476], [0.18490885061273099, -0.3039831342218875, 
0.30169544606492682]], [[0.23521840658381565, 0.79513548388914046, 0.19099544154727166], [-0.81779171215333135, 
0.17300907840416402, -0.96683591517261291], [-0.97451715404879113, -0.94548694401819589, -0.98379599888512703], 
[0.37993421631914925, -0.89010602467599864, 0.56039582427275114]]], [[[-0.53595611753886963, 0.74689692996195012, 
0.73894238152071345], [-0.1879667476944471, 0.67132720726934791, -0.48826386985366854], [-0.036402447542734251, 
-0.89696840314505244, 0.020951097009541009], [0.71243154382327067, 0.79238770600970887, -0.60275884966448601]], 
[[-0.57931551783973156, 0.29985161910715208, -0.7125892189120111], [-0.23419171973429398, -0.52934109417924957, 
0.84530006424335702], [-0.42456361985085778, -0.98006187350368168, 0.38246030176514245], [0.12307707016725677, 
0.38944803951466245, -0.20618298804027013]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_4_offset3(self):
        sh0=(6, 5, 2, 1)
        sh1=(6, 5, 2, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.19451275633355247], [0.50906079024795581]], [[0.62428285310661535], [-0.074832715077812972]], 
[[-0.64832986271288862], [-0.31947333416701751]], [[-0.40010015892322537], [-0.092034758128705985]], [[-0.8291273324029933], 
[0.27859372615435585]]], [[[0.64333411444347455], [-0.2822448396784647]], [[-0.27079598096028934], [0.78022158057526769]], 
[[0.78563665023001827], [0.20221562633694679]], [[-0.42007400421049623], [-0.41118441332530353]], [[0.2499140504982349], 
[-0.22865668474862444]]], [[[0.94217480310447077], [-0.33207547816764871]], [[-0.34377163329131943], [-0.37324970716103256]], 
[[0.55382852708903307], [0.16641995106275886]], [[-0.37918491120260911], [0.20612451309811286]], [[-0.26973262065670256], 
[0.25864855635829986]]], [[[0.37435294260377905], [0.26346892564393021]], [[-0.14712812158503752], [-0.23073872197147249]], 
[[0.47968899997742809], [0.72354251095351318]], [[-0.81509126996739045], [-0.079753971902671861]], [[-0.031165832088541956], 
[-0.98885044894813912]]], [[[-0.87061711353818372], [-0.43557384539626409]], [[0.75124055302943371], [-0.47253716082017205]], 
[[0.57953651503357184], [-0.34508878443370072]], [[0.68343296842135959], [-0.19222815462402099]], [[-0.043671881204645491], 
[-0.44113564661862292]]], [[[0.36194491761512371], [0.45170594303806011]], [[-0.61024738875976281], [-0.72670168879575914]], 
[[0.62512121301270218], [-0.84594015334940575]], [[0.95342448009008463], [0.76860887784008791]], [[0.85394369300542738], 
[-0.0024905224741194054]]]])
        yy=numpy.array([[[[0.51293038493686471, -0.46248218064722413, -0.33552482269039974], [0.032339348432014692, 
-0.56278132590820729, 0.504675805175959]], [[0.48880516757073011, 0.80393023344467318, -0.48458553200634058], 
[-0.13023720143705897, 0.66147579123624722, 0.25038846260639036]], [[0.73673358396831001, 0.73167333429599934, 
-0.38949805946213001], [-0.8036269977948538, -0.17125524006150128, 0.82531435107658946]], [[0.10368629465913837, 
-0.99321769191190756, -0.50981576856095567], [0.50679538802266277, -0.17934164720519097, 0.50261392833778062]], 
[[-0.84218912721326, 0.82757547034714851, 0.56129319385636123], [0.85821565507841902, -0.087332368903009217, 
0.80708868822105506]]], [[[-0.54329654734676169, -0.7370314990983371, 0.27855701504814845], [0.75500655312155884, 
0.036756866241891339, 0.72343183630931884]], [[-0.48254905822725869, 0.48095859673511532, -0.28299247753900092], 
[-0.66691854755457602, 0.43067280045135314, 0.14201583859402378]], [[0.2339552880454141, 0.86167678786116353, 
-0.2386722728474433], [0.098484944787608386, -0.29945804861755088, -0.65188143630182238]], [[-0.41430484218668107, 
-0.070970074376035308, 0.028092013149029871], [-0.39723249182119491, -0.98126770835376953, -0.02239123034470647]], 
[[0.63347929880377607, -0.27087402458092935, -0.50377219449257371], [0.31042030929861886, -0.018351169016868951, 
-0.47663221485114837]]], [[[-0.73313461289048432, 0.55831101421226159, -0.55690393351046508], [-0.65478415679239865, 
0.39907373832020521, 0.79668928163710673]], [[0.71094440831827277, -0.70661155052277458, 0.07598072165740688], 
[0.20444740922915861, 0.5736762428836546, 0.11043232578865148]], [[0.67672910091899796, -0.43306439902322102, 
-0.29704816769501141], [-0.70783700803541927, 0.54358715423532078, 0.2958883974983253]], [[0.81664653099926698, 
0.81621730948567506, 0.23291123486130605], [0.82151965993686882, 0.22309819042186896, -0.55316931279441373]], 
[[-0.71848044035910785, 0.84915708357151232, -0.4899840485361413], [-0.73325693839532557, -0.51277992002551831, 
0.017428401139844452]]], [[[-0.97776437031776631, 0.18698360595518682, 0.72740531600609803], [0.45439430998554742, 
0.083238235814692363, -0.54038634253040918]], [[0.19546660726555198, -0.41181143501979345, -0.1759381924284289], 
[0.98490343816944437, -0.21896097502115541, 0.84876800350937431]], [[0.38860780902456105, 0.58046324555300988, 
-0.32003218061253147], [-0.68121304259513127, 0.79155973346974262, -0.90223360813692488]], [[0.10773840663935719, 
-0.46472226294091179, 0.71851819329296318], [0.87762159228969927, -0.47413278608730747, 0.64875394381868356]], 
[[-0.90097540198816795, 0.39500912174375991, -0.21450392213009417], [-0.82657026626794239, -0.38709812150519762, 
0.20376794039482315]]], [[[0.5945107198919255, -0.16447938937895268, 0.93328674222534991], [-0.31886150132057667, 
-0.33187857090598394, -0.045696806741569507]], [[-0.19290970086605608, -0.89930531401176905, 0.88610360571504354], 
[-0.51007003702795761, 0.70141618180286658, -0.83432016116703345]], [[0.67434082628088632, 0.59733571209554492, 
0.60763844673920042], [-0.099006315713944471, -0.48333880183323608, -0.27122949187623124]], [[0.15147680750921944, 
-0.46097956735960866, 0.22033079866242744], [0.77519565025039783, -0.87918986359294471, -0.17951916627133069]], 
[[0.40547464120889698, -0.2061712799063562, -0.21865750459238975], [0.12996584946211942, 0.6712399272089935, 
-0.87952145570232276]]], [[[-0.29876862936154414, -0.6101255483565442, -0.18069716950358305], [-0.5741411967294523, 
-0.4900231912763604, 0.70121082717922167]], [[0.17385397540373404, 0.14247141608647307, 0.31867738884813313], 
[0.94920308396374531, -0.27499237218662564, 0.010188618778938618]], [[-0.6661046517295317, 0.27779336376505737, 
-0.66632138763654747], [-0.62210611615023104, -0.9112107022765068, 0.95920036687499222]], [[0.20045757369031603, 
0.74730039208465815, -0.01817107855597877], [-0.53468005775121696, 0.69144413956807016, -0.467250722509853]], 
[[-0.021558888602753346, 0.046379253396290121, 0.90976181796342837], [0.66385170105998492, 0.63396886951606057, 
0.060859676392965412]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTransposedTensorProduct_Symbol_rank4_and_4_offset4(self):
        sh0=(6, 5, 1, 3)
        sh1=(6, 5, 1, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTransposedTensorProduct(x,y,axis_offset=4)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.12790787331446718, -0.12849806484100235, -0.1933101798746415]], [[-0.26255275922972299, 
-0.95686537036078856, -0.8050919236250047]], [[-0.32159885211331396, 0.29296074822889406, 0.55223552511164153]], 
[[-0.30173589350565799, -0.70601275745211023, 0.28499119398574102]], [[-0.2780462694497805, -0.38147325872773474, 
-0.57475983941098119]]], [[[-0.47308947311799043, 0.40244205036362746, 0.17657663916790711]], [[0.076545603008081331, 
-0.88708472431761431, 0.48011616862159334]], [[0.58196336261763859, 0.54845984175881379, 0.61556725545369062]], 
[[0.25772330655128051, -0.032544344889670507, -0.70492366013432561]], [[0.96179522644803184, 0.70518664983867363, 
-0.77833667920927141]]], [[[-0.59176106778859516, -0.56233804215956806, 0.70987915598398854]], [[0.70824190120435793, 
0.18003172845168613, 0.98863819741994252]], [[-0.89254104969311165, 0.057797454614478161, -0.82155753594754044]], 
[[0.36650348770159824, -0.25290303597092767, -0.75322625100791463]], [[-0.39624940803914543, 0.083634066829807008, 
0.68445565251301632]]], [[[0.5172326432316563, 0.14700234166489579, -0.85536954840833634]], [[-0.98465807792003845, 
0.11812693977483479, 0.84183998147949546]], [[-0.13102337037497769, 0.1876356197090383, -0.21333903768193396]], 
[[-0.11554558192150921, 0.16482694477881266, 0.24819582583126842]], [[0.76621740238179359, -0.60436411224898201, 
0.73554063961118432]]], [[[-0.036292474808631292, -0.9595442486321577, -0.86764996933664951]], [[-0.37991198542351068, 
0.86714774777301606, -0.35491579976364185]], [[0.035408922910725282, -0.61056844660145915, 0.25989535652609197]], 
[[0.86334395086585936, 0.1891438198721398, 0.23360162622085312]], [[-0.72343385091070056, -0.3197144252581543, 
-0.20809207375025984]]], [[[-0.14491056524757706, -0.15954083500236993, 0.71567753021938962]], [[-0.15684366945844208, 
-0.70326307204375782, -0.73343059747314898]], [[-0.84667615483725367, 0.15620204209780253, -0.55079351520220454]], 
[[-0.67127679219869085, -0.07655292406892733, -0.48986818762179585]], [[0.18331191241255862, 0.68210787341312251, 
-0.10990891720063201]]]])
        yy=numpy.array([[[[0.81338421665213323, 0.32816802650895638, 0.30693960623571592]], [[0.47956639162621473, 
0.68068326894678477, 0.6438481252735917]], [[-0.89278101705371249, 0.98732282201328658, -0.53771881133473109]], 
[[0.61236179415067182, -0.61109056102254433, 0.012587744880723228]], [[0.58435380933422509, -0.097165882865710707, 
0.50715318945623622]]], [[[-0.96236714768558329, -0.89366644337629841, 0.27763095226905432]], [[-0.31592803301617245, 
-0.47468720741133485, 0.4086816273964069]], [[0.70966858650171605, -0.64985697557044819, 0.32827419466499941]], 
[[-0.28963508942991223, -0.51303883491901292, -0.4442320784019238]], [[0.23382518534913022, -0.0097667562281906761, 
0.5068270564422519]]], [[[-0.84887120247751335, 0.51240069128931043, 0.65848096574388593]], [[0.82753208285351576, 
-0.10100152945985941, -0.010460741794096062]], [[-0.95348834405524152, 0.16002860629836535, -0.27750717886482668]], 
[[0.10650807150668928, -0.78399604270654155, -0.5506597516026106]], [[0.036373432236614933, 0.60966228690665125, 
-0.31780356813313237]]], [[[-0.31158229560100725, -0.15891170517402875, 0.41282097148314567]], [[-0.25829806387099774, 
0.42660236180176936, 0.015396675154741413]], [[0.81942571769807371, -0.92881319502610693, -0.50613465568326177]], 
[[0.20824885282256611, -0.43980540347976738, 0.5537088222808717]], [[-0.660706006160515, -0.99329692488653687, 
0.44308624329192892]]], [[[-0.14693291684468246, -0.38717403983241616, 0.61908383437899683]], [[0.26653555162990572, 
-0.20716177724193519, -0.23600883220527424]], [[0.67313978746424907, 0.24151305305589132, 0.73473564797133029]], 
[[0.35528072459185656, -0.84500095087481908, -0.71049571100005005]], [[-0.89275674175362085, -0.072292208048146733, 
-0.6655289889657634]]], [[[-0.78452912034707079, -0.33132646603998239, -0.10677388374727137]], [[0.80311999029773129, 
0.5346816074995413, 0.11586577754246141]], [[0.11396588557158394, 0.56718221181578343, -0.49362276982585374]], 
[[-0.72213434961679357, 0.45337782874347066, 0.60528407040385934]], [[-0.025574104537218201, 0.83625758577640208, 
0.55278347443416931]]]])
        ref=generalTransposedTensorProduct(xx,yy,axis_offset=4)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank0_and_0_offset0(self):
        sh0=()
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.471426163637)
        yy=numpy.array(0.232230425881)
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank0_and_1_offset0(self):
        sh0=()
        sh1=(4,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(-0.306421593496)
        yy=numpy.array([0.041799306077875631, -0.82798821410327306, -0.94984072547636345, -0.92384687226779039])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank0_and_2_offset0(self):
        sh0=()
        sh1=(4, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(-0.520011651413)
        yy=numpy.array([[0.64460347113842653, 0.36926675353230087, -0.62372298789207803, 0.86676302225886825], 
[-0.65960278095502178, -0.55644283754554191, 0.84437877039970077, 0.54302077636690327], [0.26230699021358661, 
0.8498588077959417, -0.8677688494792406, -0.50696622924350243], [-0.51388005255602343, 0.018405179782792436, 
-0.18878531345757943, -0.74769065758018138]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank0_and_3_offset0(self):
        sh0=()
        sh1=(6, 1, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(0.732709959639)
        yy=numpy.array([[[-0.13796206288941026, 0.63455023306914726]], [[-0.21958714279359293, -0.86577894564455482]], 
[[0.48241894064423918, -0.83377548161753778]], [[-0.40396724849216059, 0.086321703456439858]], [[0.056056531336905469, 
-0.8391711457868738]], [[0.38982291977538064, 0.88908459040536503]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank0_and_4_offset0(self):
        sh0=()
        sh1=(5, 5, 3, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array(-0.904659592155)
        yy=numpy.array([[[[0.87287895993873521, -0.90130591820262085, -0.73173523310828759, -0.94983191534773459], 
[0.49344856028538242, 0.70661494547647496, 0.99238679287055964, -0.82860185444626056], [-0.2964895988931231, 
-0.81358540760433251, 0.60571875536582609, 0.48773442326534955]], [[-0.22072107454529055, -0.85982898997110158, 
-0.85394839524315969, -0.82847754642423066], [-0.66682096901835841, 0.369629884238452, -0.10601896854458048, 
-0.66445864271118316], [-0.5748262909482873, 0.23405220438372698, 0.19712289925335225, -0.056650549019651164]], 
[[-0.47446491223846121, -0.89291138625948463, -0.89075336989161302, 0.082564296543275351], [-0.24396490556351824, 
-0.81160577541454293, -0.56154176219823282, 0.008790495146006494], [0.10319008728801404, 0.50221318570113294, 
0.58068352126953982, 0.56291728940963814]], [[0.92737352136691698, -0.4959730195270371, -0.64961095940522284, 
-0.13371269314008472], [0.74245316089690139, -0.4817138086242938, 0.73081613032367554, 0.080542980438863987], 
[-0.55817241006223139, -0.5719443726458715, -0.57420288508785111, -0.33102994719507395]], [[-0.03208381457038656, 
0.9625818024768078, 0.0185634285296159, 0.267856811620516], [-0.27905917423906734, 0.32047097392900503, -0.0071397009521496635, 
-0.22124371954855326], [0.092985487047691162, 0.63379621026481336, 0.25919215233259818, -0.96951850100651971]]], 
[[[0.23326720633874332, -0.46558905531944728, 0.13302905299696532, -0.067326859672633521], [0.66670235770162467, 
-0.38056102368292755, 0.7361148850310284, 0.10208381490639451], [-0.7717171505725402, 0.61097317257602324, 
-0.50892395078042107, -0.79081949690823161]], [[-0.56460833320180837, 0.89528260500998447, -0.90704812435060611, 
-0.61095765464522644], [-0.30008868281592882, -0.16384937389883603, -0.81147669751123752, -0.054561347736980403], 
[-0.87822851606856323, 0.53021988273121856, 0.53717047528745132, 0.655957110008349]], [[-0.79773160408303134, 
0.2805061956812851, -0.98786480015769551, -0.18893465946587096], [0.63282692728459389, 0.79781626522364157, 
-0.9116263226445056, -0.2896212062192467], [0.85962840583780364, -0.40078702648492359, 0.19640307246129529, 
-0.9857688740210715]], [[-0.1608590665396803, 0.98408941573044495, -0.29455801328964926, 0.35039185621841118], 
[0.039213111591590133, -0.83805143325981479, 0.26192392980843326, -0.52621146952069031], [0.063552391128338881, 
0.068076244899905936, 0.20397147506006608, 0.98173036048193185]], [[0.20694216691039102, -0.88463556054365133, 
-0.34157040351368662, -0.78332470778464369], [0.8141048376633242, -0.060939912709286448, 0.46345381652572959, 
-0.53205051332022824], [0.15586246034104789, 0.48847787164794343, 0.19789434473649892, -0.9727198242135624]]], 
[[[-0.96691715526210231, -0.10105153119987786, -0.33500252807527708, 0.89033385574129853], [-0.16315153402305227, 
0.13185714834928053, 0.40208944102518829, -0.22714509729727572], [-0.29309388115418078, -0.13639550288982627, 
-0.48215421539576808, -0.62273195503110856]], [[0.85560545805800481, 0.25188486653935915, -0.085943240913092689, 
0.2316053370392166], [0.21509090513737505, 0.18578858552475097, -0.63009413341301479, 0.10758100389121461], 
[0.91475020680401276, 0.98634504738540496, 0.12771337949278516, -0.28017049863494581]], [[0.28610425507558834, 
-0.28705224234683957, 0.018140065804505001, -0.97862765215005276], [0.99153591751940073, 0.0056829066650807558, 
0.60691975283882837, -0.28468050631853492], [-0.38665761735340798, 0.52467383695215508, -0.59824823903160929, 
0.9056579284021864]], [[0.36399379607146787, 0.27853020303043952, -0.65205866062348194, 0.54324613908814268], 
[0.81171228578576282, -0.46506248630811342, -0.25841418956661655, 0.46476342112007796], [0.84998200975338389, 
-0.4219751712534825, -0.66566484585473074, 0.51850676019384734]], [[-0.68704105811998728, -0.080615589700263612, 
-0.95779466538368507, -0.83866744714536612], [-0.058786147247552156, 0.71891835295901818, -0.26635280075833334, 
0.33719205173124522], [0.29457449304194183, -0.60695131014962134, 0.69619742486562686, 0.0016742533510953894]]], 
[[[-0.82674396848311771, 0.85546824638776897, -0.96695728315386531, -0.82430878359432991], [0.86715869732866824, 
-0.14027309299435897, -0.34256384803257478, -0.96705489186791671], [0.024644667011768373, 0.46927849779906539, 
-0.29988675915769569, -0.14933585475025102]], [[0.51028963532759586, -0.095542967424486758, -0.96991694225549119, 
-0.85465177820550942], [-0.28187189410986146, -0.60976362472330137, -0.99764192084835268, 0.87110828002645424], 
[0.50494722046574103, 0.83561221556430954, 0.82120409412705242, 0.60368825657978431]], [[0.80393664805598042, 
-0.045880280420188102, -0.038959831952714463, -0.33710143180705221], [0.92137070337263483, -0.93193858556088838, 
0.11566206985129157, 0.99162809329557589], [0.68946205583819387, -0.37007923366993056, -0.214569531987157, 
0.6751016543154762]], [[-0.22501430938760536, 0.76635982919382251, -0.44044122953307596, 0.81055776193741114], 
[0.86720766893686974, 0.35031883910111916, 0.19568456139152057, -0.7097363049939891], [-0.49344615958252236, 
-0.52554802706781079, 0.83964419357366005, 0.27098296585719228]], [[0.64141417087621644, -0.2000635850125938, 
0.12601732648571851, -0.57123846275701506], [0.29668686906038189, -0.35760586369117431, 0.92602415219056389, 
-0.48477451238726976], [0.96930406711823625, -0.46812423167572925, 0.5437422962511036, -0.40517840802003025]]], 
[[[0.31039472767360543, 0.58942232776771264, -0.97730991065532979, -0.44748998597574374], [0.88387492065735751, 
-0.028946675309387437, -0.60315969889236754, -0.44173258662408776], [-0.19125573490644521, 0.076135647893871061, 
-0.87564823206175002, -0.95192821303362973]], [[0.66284819931993222, 0.82143982130309667, -0.73838271411926382, 
0.65316621928173357], [0.59783262417593486, 0.71401068446879901, -0.10071834032950844, 0.4741774573948585], 
[0.41440849418239112, -0.86294319901787575, -0.3435258633829954, 0.085040434844048107]], [[0.12427580942617911, 
0.18149449507636972, -0.97303151826877854, 0.27506060134353771], [-0.96544873339020798, 0.59424553111083234, 
0.18429519204945755, 0.72796477826973072], [-0.4424098362933373, -0.97057198205469253, 0.91491677820758222, 
0.22168425115809076]], [[-0.2291705946775926, -0.12389887301081237, 0.46796120857760681, -0.72965477168921256], 
[0.35280709802372945, -0.66206127472072085, 0.33519115008839862, -0.16529228874263091], [-0.60137420013977572, 
0.98666748261249637, -0.36885358664581824, -0.87147437138402895]], [[-0.44954345282112662, 0.65002302778134968, 
0.86328346983353521, 0.094726841084572078], [-0.25980546373606184, -0.71352965793763889, 0.57106179526206713, 
-0.098323848007781045], [-0.093351872627125632, -0.88796989573322871, -0.35874406796429192, 0.63469214726051781]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_0_offset0(self):
        sh0=(6,)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.01083044248332099, 0.53470414040768643, -0.23482984235254212, -0.75856457331963312, 
-0.96561206576027536, -0.012624076745070711])
        yy=numpy.array(-0.148899423115)
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_1_offset0(self):
        sh0=(2,)
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.70271557696158027, -0.41334703792344807])
        yy=numpy.array([0.81139889309325897, 0.1092724055050196])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_1_offset1(self):
        sh0=(6,)
        sh1=(6,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.7218033178015264, 0.78190854856559167, 0.47464064960028707, -0.60883594107324424, 
0.84037220448977945, -0.31241672843250101])
        yy=numpy.array([0.51518279192280647, 0.62703373825238362, 0.68314824491400405, 0.40229307767836664, 
0.09380027860712703, -0.67844765828031184])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_2_offset0(self):
        sh0=(4,)
        sh1=(2, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.48912790916364957, -0.63245336918237283, -0.6311802811889704, -0.42750995342901432])
        yy=numpy.array([[0.28008525346336355, 0.49414457934904021, 0.29796245019163292], [-0.27849326017017506, 
0.99678068185414359, 0.22515347929661544]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_2_offset1(self):
        sh0=(6,)
        sh1=(1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([0.24579509131645572, 0.10548143046778935, 0.92129676460355991, 0.90951772378739348, 
-0.32688086169492414, -0.36315146393120545])
        yy=numpy.array([[-0.56922664322029548, 0.58126904303949045, 0.16298651099107131, 0.61658583013302715, 
-0.24837149719403451, -0.050762865554276448]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_3_offset0(self):
        sh0=(3,)
        sh1=(2, 2, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.76879697782833389, -0.30673589156871128, -0.067035486815998135])
        yy=numpy.array([[[0.3235235378765291], [-0.84985714903169818]], [[0.7291197183548348], [0.43161831092956215]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_3_offset1(self):
        sh0=(5,)
        sh1=(6, 2, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.86507794507684155, -0.38907085033501332, -0.94383385173687384, -0.49176137103734918, 
0.72087937744558372])
        yy=numpy.array([[[0.30786407744456401, -0.39858643899423907, -0.20931941859287062, -0.29693543393618937, 
0.9434381547331756], [-0.40327192996730821, -0.15311338011132491, 0.60623995872331671, 0.8097459513724492, 
-0.66606270466368711]], [[0.22485050331400491, 0.8307155080530424, -0.5421684572792842, -0.2933339080128301, 
0.83781640787689726], [-0.82720364523491829, 0.31882246717395923, -0.58124759069212129, -0.82927994983135789, 
-0.39849761206256096]], [[0.73402144012579695, -0.97632942859731853, -0.23459201821895692, -0.54132197197479459, 
0.281634063301061], [-0.95462547574067091, -0.51123124638429185, -0.28610719523532779, -0.46793295061366114, 
-0.025771416459326124]], [[-0.2713378396688273, 0.97442245016962481, 0.83859581221249546, 0.86723050182356842, 
0.95711909105199044], [0.013359354110134669, 0.82723664713483869, -0.89809685540508877, -0.243143183605675, 
-0.7416039909113894]], [[0.99492650280481909, 0.95688621860753686, -0.09406257390150663, 0.4126478628234469, 
-0.12920436416781289], [0.096054473214584712, -0.013243637931681373, -0.86700717203421163, -0.51876730363678591, 
0.67295230525192307]], [[0.80648383679683744, 0.54896626979351848, 0.38176943659982787, 0.41792635868186556, 
0.81001024727830151], [-0.7530626905600557, 0.64678449573379027, -0.18525926470925924, 0.35295179177087377, 
0.37032745070655859]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank1_and_4_offset1(self):
        sh0=(6,)
        sh1=(1, 6, 4, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([-0.91543652185123525, -0.64412872630350493, 0.14394782497163905, -0.34662477378189438, 
-0.76380594400665269, -0.29751562010530841])
        yy=numpy.array([[[[-0.12659156682931605, 0.55074822121644496, 0.33804953981606278, 0.70028257934924198, 
-0.10417887882079069, 0.17779351760357121], [0.69052552064021366, 0.66638340831298115, 0.70937920226523699, 
-0.49676286735032638, 0.48083821508472391, 0.24612126925825439], [0.45632832289765246, -0.16380411957048824, 
-0.42257729269930588, 0.55528498222296108, -0.77212117563239024, -0.17373584850163382], [-0.043954150813345638, 
-0.073184565207579544, 0.80443617385051325, 0.20103140020478127, -0.86552492160193828, 0.62532246934646696]], 
[[0.74224121156678802, -0.34160130399706912, 0.69073286049625571, 0.81823060946456372, -0.82754093654455962, 
-0.239859318389823], [0.16994057338042912, -0.47345606631208925, 0.0043112674019234465, 0.15507538590733327, 
-0.16215834902025361, -0.17260888741429992], [0.53128712665303546, -0.59704898332211775, -0.61207711956061384, 
0.72325386151052484, -0.17380252229819737, 0.86140255478183314], [-0.71666478319491866, 0.80935801372605698, 
0.75734695529146956, 0.03530710475557397, 0.28010756440235252, 0.6785457173405236]], [[0.57016361609009247, 
-0.065945204149945091, -0.68417435403843974, 0.31319345986839919, -0.68914789402748799, -0.18392575782329046], 
[0.26892302687106473, -0.64649888253994447, 0.76999411377453675, 0.053169916999408651, -0.49493748996954401, 
0.45448064258799636], [-0.34075065661906678, -0.0032033706606526824, 0.7069381761897473, 0.94236272066246674, 
-0.50608080954287993, -0.11439715693287322], [0.46211441711493717, -0.96159313347676356, 0.46125545792315714, 
0.074602603812749635, 0.041591206121663005, 0.30316393998358282]], [[0.98073204097608047, 0.94058847766756526, 
0.74495187429892851, -0.91728690388436429, -0.38337279252584988, -0.25029259294166706], [0.83705824957638275, 
-0.69585480664586297, 0.52413058059784157, 0.89769990232093777, -0.10905472230120128, 0.32753793435341327], 
[-0.20513483834094548, 0.80010769607326027, 0.39431302988446704, 0.86715948365165962, -0.70604062285926927, 
0.28424992099367774], [0.75138849513486483, -0.73556945898173187, -0.96677460330756326, 0.60940534784899669, 
-0.43820510455431805, 0.94586112634318775]], [[-0.40747253882042034, 0.60390776377605815, 0.90837187577227851, 
-0.33980631844457054, -0.26851705539536486, 0.10413680304937234], [0.090286458152448734, 0.78378566396394977, 
-0.74085867290829266, 0.098472837653960754, 0.87532685075669203, -0.25625430992909037], [0.12193853815855538, 
-0.51451978626808947, -0.24202226797175741, 0.0081260504654263599, 0.72251414423316773, -0.97230876895640495], 
[-0.30081982032787269, 0.66404987287582684, -0.73165989308266055, 0.098100109865459029, 0.77361268952880446, 
0.98388119642974847]], [[-0.28189186931535293, -0.91406693929595972, 0.67608123383527507, -0.58063725697063417, 
0.20142290340935487, -0.75048349160228245], [0.70751747341444227, 0.069081495074537091, 0.14745541566660836, 
-0.37317601861812455, -0.92373855945202488, 0.14748507139547051], [0.72229673892970192, 0.61993805723484741, 
0.8153380413513942, -0.35845647011854842, -0.24735858394810206, 0.7090758415234697], [-0.44831787547667346, 
0.26967356639704909, 0.014027002815995671, 0.99738510591927843, -0.23310587934175109, -0.85285645941061583]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_0_offset0(self):
        sh0=(3, 1)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.82035418364125445], [0.9056382005227146], [0.74796746401103764]])
        yy=numpy.array(-0.255587705821)
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_1_offset0(self):
        sh0=(1, 2)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.63561099327371973, 0.28573602026902689]])
        yy=numpy.array([0.22774404040635354, -0.20068723146754985, 0.41963504923029826, -0.86934664628827396, 
0.89857798677499057])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_1_offset1(self):
        sh0=(1, 5)
        sh1=(5,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.65501244912561574, -0.22778387715996407, 0.041659913974658958, -0.82524511408359724, 
-0.48673089729194108]])
        yy=numpy.array([0.99281373708023213, -0.74304863904357155, -0.97490997541506119, 0.84202848923736862, 
-0.9432566908434592])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_2_offset0(self):
        sh0=(1, 3)
        sh1=(6, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.38338942834934708, 0.48574468290585138, 0.92443944012671908]])
        yy=numpy.array([[-0.58351487326935514, -0.92478489386086715, 0.46286118814328359, -0.79721630027514889, 
0.57154956850879124, -0.59488192092439229], [0.19145655821827612, -0.15022383054258914, 0.31075049670864252, 
-0.42346425094008699, -0.69521476591403131, 0.52167792784165834], [0.11127849558234026, 0.81963569467739106, 
-0.41961274461290321, -0.74630213204426221, 0.33218333051705784, 0.086066323548166457], [-0.32408023111286499, 
-0.68075585721815246, -0.076526238969079285, -0.042267289837090161, -0.24596281150416166, 0.13611887083400975], 
[-0.13587635957331146, -0.2640289329754828, -0.7788855503480332, -0.81460119739205128, -0.73167590455021569, 
-0.36409142298518904], [-0.81189724358128634, -0.76186747440033975, 0.75124341962591101, 0.044009837613044445, 
-0.68059349663808488, -0.65223301860263905]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_2_offset1(self):
        sh0=(1, 4)
        sh1=(6, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.83488849084583494, 0.29725851854899776, -0.015880998756614195, 0.15128041343061671]])
        yy=numpy.array([[-0.56316883965592024, -0.60526843237626227, 0.57962859135559852, -0.11003346500824152], 
[-0.15913680964794685, 0.1734576610470131, 0.048812772513177727, -0.21188075983442545], [-0.22988011376193884, 
-0.12857307476857249, 0.747686164698256, -0.31066943033700078], [-0.58545526427281036, 0.19816630004768299, 
0.037639147916846305, -0.79839694155789309], [0.37711439359445253, -0.21344581266373308, -0.91557583012119403, 
0.63069787060404225], [-0.84860478001244055, -0.0028521921363926506, -0.36700760269535126, 0.84658482073657071]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_2_offset2(self):
        sh0=(4, 4)
        sh1=(4, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.5689913733377534, 0.14409084569049013, -0.33195329681931041, -0.060191174159764493], 
[0.28702452450006999, 0.96391289449124029, -0.12089711977593609, -0.47531191634629799], [-0.220885780762758, 
0.022965144249958991, 0.41008348706171205, 0.82916577629655164], [-0.72819819714333023, 0.84100850959222151, 
0.07744433306155396, 0.010306561794944269]])
        yy=numpy.array([[0.064712833859017271, -0.29298295609009495, -0.0059061243848470557, 0.40855566924419295], 
[-0.1064428597990017, -0.72329461169335985, 0.34323970252400926, 0.19311822772050924], [-0.36685960102162163, 
-0.19014835386227125, -0.20329079444830822, -0.58548372125715309], [0.213935765758545, -0.90762725840792258, 
0.21418913231762549, -0.072494565203732986]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_3_offset1(self):
        sh0=(6, 1)
        sh1=(3, 4, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.74436172859096539], [0.088545431314771772], [0.21274156679134038], [-0.92539685658185578], 
[-0.17062035603151915], [-0.6467423233092211]])
        yy=numpy.array([[[0.26370844676227279], [0.55954639699773745], [0.4374271233476652], [-0.54700093207020717]], 
[[-0.22391943319875773], [-0.76529145411690713], [-0.24079948772543625], [0.93686460290297746]], [[-0.98599262534367349], 
[-0.054470790927642287], [-0.13965539148440897], [-0.68285549735273254]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_3_offset2(self):
        sh0=(2, 3)
        sh1=(1, 2, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.19707932473628609, -0.88091149094161869, 0.065330167874279033], [0.95996794024921828, 
-0.13626210586042942, -0.62220773119409012]])
        yy=numpy.array([[[0.76116677899254537, 0.38929888507468635, 0.61000713496972891], [0.94624793595624235, 
-0.33001924247865744, -0.55061685962112228]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_4_offset1(self):
        sh0=(2, 4)
        sh1=(3, 2, 4, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[0.1741341604266966, -0.16690001240130981, 0.48417410329465849, -0.50897965594135486], 
[0.78948674460139001, 0.91148676396730366, 0.81739051467290058, -0.1958539680070337]])
        yy=numpy.array([[[[-0.6443733708675965, -0.70468182233819499, -0.80211743369388699, -0.55652658959898549], 
[0.85540402688262041, -0.31516975565357375, 0.25117505857161393, 0.035248871171389418], [-0.016450634152086252, 
0.2742082133521806, -0.13233953948675925, -0.27261483896917982], [0.8071474878589231, -0.19969682643648201, 
-0.22255270310213682, 0.070576568245937166]], [[-0.30626876061994301, -0.043048996939866857, -0.31872284779243287, 
0.21530796792217477], [0.23817726537172978, -0.86649449038266146, -0.57869549223663186, -0.54634032251323461], 
[-0.44370666158062955, -0.60723755015889846, -0.67902098149294354, 0.91897560024904568], [0.33403752858162328, 
0.80031228961597956, 0.53975933463171222, 0.74724937624364651]]], [[[0.040719887602647997, 0.072836349073226447, 
-0.83349122901161321, 0.92912888192245302], [0.10158081059980573, -0.0030751945514335688, 0.91763890671933757, 
0.046697816899473388], [-0.54543993738990482, -0.55718607038083823, -0.52069403892917587, 0.61609112665931054], 
[0.22368874890460821, 0.78299983187913158, -0.95337696945179018, 0.72429024847299206]], [[0.65925391613893458, 
0.62440736473917813, -0.82191060183745224, 0.30225711189305438], [0.34227987606324173, 0.50628579821589081, 
-0.79106560040297258, -0.45679203195140317], [0.99807877593273697, -0.48720379308621964, -0.1480831197338961, 
-0.026733828390276493], [0.14909886095064961, 0.18006548225614361, 0.37071096086946209, -0.7994704073744896]]], 
[[[-0.84306034329285273, 0.57992990788668775, -0.39260138103527242, -0.91402198393739398], [-0.9672178325173908, 
0.21693834101236975, 0.81570633947007898, 0.43665376671559186], [0.10855924306473508, 0.63307629532580556, 
-0.44703088576662697, 0.46352142449918188], [-0.6061699388029631, -0.13670208788191607, -0.54491666778298997, 
-0.99158175687218697]], [[-0.33204652325719453, 0.23078236946140285, 0.9829812246690377, 0.20782656107966879], 
[-0.61645882258508178, -0.040053115481294421, 0.51984576215207179, 0.89002370102972472], [0.57738833023239344, 
-0.74936789668954962, 0.84403788211240571, -0.44106802437154791], [0.97770317401411666, -0.58076863776674292, 
0.90697257301516787, 0.15882337664550472]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank2_and_4_offset2(self):
        sh0=(6, 4)
        sh1=(4, 6, 6, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[-0.5033639073037155, -0.44133848238602957, -0.59984173749797587, -0.48826387065996202], 
[0.45872643686426207, 0.52753514533047241, 0.23691115253211459, -0.50942201766991935], [-0.71519005101017319, 
0.2911957794153377, -0.23588655874586939, 0.10284531185530388], [-0.20549783521671294, -0.834597499610779, 
-0.025932241932195454, 0.14851965307310011], [-0.75478293029738763, -0.43650031226787434, -0.18355454657489578, 
0.099927788165152132], [-0.080601320401961551, -0.56245125806745344, 0.20239564145152134, -0.39270164033473476]])
        yy=numpy.array([[[[0.4601675138453889, -0.031649317828258861, -0.42298733856163384, -0.69475696827720546], 
[-0.29168289210822107, -0.94644962737121041, -0.91895979422857677, -0.62280271963826861], [0.36835704549858517, 
-0.85653277371146364, 0.10301175022218612, -0.48694942666694319], [-0.57886551829163735, -0.96634999709228109, 
-0.7985905559176445, -0.60711473168054497], [0.40826633022217451, 0.29766108320884888, 0.40436006896406518, 
-0.7495085484079429], [-0.53531762686551132, 0.41783347030060702, -0.20097151770153032, -0.75945048684257155]], 
[[0.17661159876154531, -0.57118180996588697, -0.87154338292579325, -0.95843956847510636], [-0.085430618955472992, 
0.69403894326204374, -0.719260393683824, 0.063025755240191828], [0.11490768714189192, 0.51676106349780748, 
-0.96301533471540601, 0.86321742878187968], [0.012451692577080786, 0.27404381749067652, -0.38482722001184411, 
-0.8108186342549597], [0.019440374922228054, 0.65681818233770173, 0.6444816621240741, 0.63816277732509086], 
[0.092291113388190738, 0.52667217263110544, 0.12596949734206686, -0.2799530422239831]], [[0.83996977560904784, 
-0.50300251863617884, 0.72405203171173493, -0.79947305750204234], [0.03656823806999987, 0.10886655361127073, 
-0.1027909055518168, -0.1383387577059505], [-0.26135376897619178, 0.6626687253404957, 0.84679505687729373, 
-0.79138635280541747], [-0.55946864679622932, 0.12561014574542995, 0.039614062176499321, 0.93922930394437754], 
[0.71271297279488133, -0.90889181244088046, 0.60506910468860897, 0.86666659487486242], [0.6446173720451045, 
0.84872381483454018, -0.0042911288090665689, 0.19573438467918125]], [[0.36133767365321279, -0.12401314738609326, 
0.76937769848751847, -0.72317920797539781], [0.76436827324419965, -0.57619436759641696, 0.88546650148077122, 
0.18894611398438776], [-0.5799670036808906, 0.84844238690182183, -0.905358946910791, -0.094941690550126578], 
[0.60108996938346637, -0.61021564869607081, -0.8261199164272226, 0.047365018655161739], [-0.20202906410408428, 
0.57513315415455146, 0.25727145311494182, 0.14886655847269181], [-0.53619101855431772, -0.85763256581880065, 
-0.45843399012895869, -0.37259702442877418]], [[-0.62640716487655501, -0.24110464707214896, -0.75721118019338562, 
-0.85796904813438335], [0.47438360735169405, -0.68035279804388482, 0.41093675004606123, 0.65818590643692998], 
[0.025530558428683126, -0.68106525895102599, 0.8254433275037476, -0.06828898602520872], [-0.69500789076231917, 
-0.6926497297926657, 0.53802164328732238, 0.12268665865671413], [0.91628700187795831, 0.01073035291657809, 
0.018403562642860916, 0.071605771623534764], [0.61061256266433594, 0.77658472767867592, 0.33560254452721905, 
0.0015988473851611396]], [[0.44966571256217147, 0.61651713640218819, -0.67493026176399074, -0.76365823726695625], 
[-0.56077596320852385, -0.98241833257028444, -0.94130420238900947, 0.54928451143761325], [0.92183497966954198, 
-0.36948869956298513, 0.9359633610363538, -0.58421578822089804], [-0.36993148767008988, 0.99299891530440476, 
0.056763180321585116, -0.70170467390416458], [-0.60571866177317402, -0.71466365685606936, 0.20034398531407827, 
-0.74703938795330505], [-0.83463627764738368, -0.49274370977004112, -0.60618992382423587, 0.98408893588443247]]], 
[[[0.29354183124466671, 0.18879218477400039, 0.76086789399694044, 0.49975305918980695], [0.23876225783468619, 
0.43627856560109191, -0.27742087744752886, -0.78048036728151793], [0.22327972933797113, -0.39615779938460793, 
0.52187706030939873, -0.8161080401643126], [0.77428229467407306, -0.48157161449411845, 0.80690503110836653, 
0.25433088108940316], [0.42331777415570948, 0.27108288152073445, -0.13517873938665659, 0.87987486773511447], 
[0.65702310984777612, 0.028200506461491637, -0.85372728256086394, -0.32318005704859454]], [[0.96916743885480838, 
-0.8429662477478681, 0.79387643290929333, -0.24272171998694048], [0.24287212776227918, 0.59997302990375179, 
-0.10418325667165784, 0.51772165752373134], [-0.81305395387961532, 0.091709716844496025, 0.82856710677621259, 
-0.88792780870535326], [0.40919515521570937, 0.099910324583336818, 0.768555279698792, -0.29573750165984758], 
[-0.17731171647463162, 0.78459626042409503, 0.5200817364287762, 0.31639593555735845], [-0.96253950637265651, 
0.43235064882496954, 0.73333670113554517, -0.72090816434772398]], [[-0.037704402674597848, 0.35664834828388692, 
0.4688309464936562, -0.33624949234442703], [-0.026492933477354264, -0.53196762062011738, 0.43286215155207008, 
0.79173308696968081], [0.46061651711146068, 0.92796407847247675, -0.15715477915252563, -0.51333373868286913], 
[0.085873575528234269, -0.89367877104920757, 0.15821171526388267, 0.19504567810181372], [-0.78348084339994117, 
0.17191004609130567, 0.35106496444627289, 0.2599371791043168], [0.60715518798707335, -0.38470084634865764, 0.16061429663167726, 
0.10790001929528525]], [[0.91239786435529258, 0.8873337644984336, -0.36260132967623138, 0.36188811097182394], 
[0.22423666565564471, -0.9617190640009412, 0.29567507033066409, 0.25593548898497587], [-0.85472206919801352, 
-0.6205517972812804, -0.17607887685841517, 0.9555108672797028], [-0.79386621090729781, 0.052755308694010461, 
-0.42602651469452257, -0.16070535951786957], [0.61139303224384012, -0.047872819108899289, -0.5332507436505054, 
-0.38698772370138523], [0.22812578419628449, -0.54874962171426289, 0.29939993610891391, 0.28206975221448261]], 
[[-0.91603075845272941, -0.44489422718242078, 0.52024673822488898, -0.4603079630432878], [-0.75705079672428877, 
0.89607116195766312, -0.75872153343428894, 0.69164689140996738], [-0.32339179993258504, -0.7499456546082226, 
0.10926750801165386, -0.30497049970350609], [-0.2944415421974711, 0.43815501933649781, -0.059909877931372257, 
-0.2627613539007474], [0.12779890532761828, -0.3797577466791775, 0.91329172888038679, 0.88983620570062394], 
[-0.44076924650940086, -0.93240662432405852, 0.54290523837611682, 0.62348677444577105]], [[-0.46470794602183974, 
0.91389358099884399, -0.0055839184016637233, -0.2202952000478684], [-0.8535995436735766, -0.49005620551819007, 
-0.86784039014409298, 0.48906880107983008], [-0.96708515669447648, 0.67619156362005151, -0.086876444379032636, 
0.8865509857688465], [-0.53545181127933117, 0.9378764786644247, -0.88950990898490323, 0.2967112698374974], [0.8249466344172065, 
-0.20818327595892416, -0.50236895446387386, -0.44391816539124118], [0.69591455086081866, 0.10626557606051668, 
-0.73681099059941046, -0.57810458839092016]]], [[[0.98696524124444096, 0.65824133033948273, 0.69080272921994124, 
-0.50365094982732983], [0.93307212557122687, -0.073153112895685934, -0.31135805931423666, 0.8031531061268975], 
[-0.002414565680207259, -0.79705546648443693, 0.65111459648676973, 0.72459285422629849], [0.054799691041136578, 
-0.92751695412244839, 0.64949471869241449, 0.68120353572513781], [0.56005498158507283, 0.022515043418911196, 
0.84999433291394255, 0.25741208937762239], [0.87709335886661588, 0.55176415555056701, 0.17040345943897561, 
0.86540274966144581]], [[0.41913756314288797, -0.93905743431488831, 0.6747821654830628, -0.38297166013453099], 
[-0.48431726864426627, 0.22579110946005154, 0.82985484623098227, -0.52885812914948471], [-0.57403127574747392, 
-0.91795713786736965, 0.22246937380736265, 0.80084813432338264], [0.71303399708782966, -0.19722809889855042, 
0.89567948474174552, -0.1603284100613902], [-0.77622162617511692, 0.83306034529375994, 0.46010886960692843, 
-0.77271162678421867], [0.10706464038427344, -0.75066932087151028, -0.95094018375419664, -0.92249051537055227]], 
[[-0.92657362933926835, 0.7164387793704563, 0.1003677862635155, 0.90918132276126973], [-0.55872244840158047, 
-0.50955637173002688, 0.069212754247733432, -0.66325133565114203], [-0.88878794202897926, -0.52662460952114398, 
0.012925670773745734, -0.55421974205857993], [0.58143547319283684, -0.51641932124740575, 0.1711239855798703, 
-0.86599680231359488], [-0.6376595918768877, -0.95705524403130982, 0.65334804605041907, 0.33689164667930682], 
[-0.99249162577364602, -0.81270103741208599, 0.94938184580307294, -0.055014657651899057]], [[-0.94779763512999371, 
-0.77597786910417743, -0.47471214037094001, -0.94853893144366341], [0.90012784093658449, -0.98060949874407743, 
-0.14160840593366353, 0.88573161231332809], [-0.36757638855501495, 0.69352551226339165, -0.042154967612349825, 
-0.87102116959181952], [-0.93772751983354463, -0.1429786872045633, 0.30643011237193929, 0.54877380686016397], 
[-0.88114137668998382, 0.53699449046534897, 0.44494272039686478, 0.61919329957770342], [-0.37118625717016851, 
-0.73105205642359672, 0.1424885243451659, -0.49257556746353814]], [[0.052144268222905854, -0.72277448848801429, 
0.27174916479258493, 0.79804028584383513], [0.95491634076755005, -0.52792364180629492, -0.78168770627686412, 
-0.055917481328625174], [-0.56549134489681152, 0.6546251089591455, 0.095608578842669267, 0.98059556829463812], 
[-0.2769886275335165, -0.35801511276610243, -0.71733195871610045, -0.87598770951761673], [-0.94354675933881693, 
-0.17829330728468218, -0.60139099772187032, 0.51807254930071056], [-0.02414200383437537, 0.55432842981408892, 
0.2869313649288987, -0.73280963328004534]], [[-0.4078306527289004, 0.49163223769362641, 0.20392038279721514, 
0.75566077059980885], [0.78179864262524856, 0.76996066946329877, -0.9440462869694326, -0.58654623755426005], 
[-0.14302829160489927, 0.047710825778722876, -0.59630881672308322, -0.55953175956612999], [-0.6694656373840604, 
-0.54317083438836589, -0.69922944480807692, -0.82006879582443015], [0.37973174720024216, -0.9902572812484034, 
-0.50220982070137521, 0.94508173500702819], [-0.88381803147548754, 0.33152894473921934, 0.52958227235271393, 
-0.76197819311216719]]], [[[-0.72288463411924719, -0.87064244044924344, -0.35121083336409953, -0.97219032292517382], 
[0.90004317732466532, 0.98232076236542643, -0.86816534006132717, 0.61080337730307765], [-0.34089766084301498, 
-0.58225591869899973, -0.93416816186509544, -0.59285276275102272], [0.84251887309938667, -0.013686926503804653, 
0.21804889165401908, 0.65532644086423741], [-0.79714721139264433, -0.31375944928054977, 0.42876544155444862, 
-0.63483518672602202], [-0.23513051617555392, -0.76917341379614856, -0.95183390312896687, -0.8728237932689924]], 
[[0.1971026738592172, -0.20467188557223315, 0.47938359809969477, 0.55492564684386614], [0.97695953218715381, 
0.92215203738347351, 0.52273205823419944, -0.56166212881229405], [0.93276926594732057, 0.39545984288825631, 
-0.81373854973779247, 0.76219888267337876], [0.34840228873093193, -0.79806942336182907, 0.34458105765306302, 
0.30355433882972949], [-0.20162877273131263, 0.64584059219990486, 0.0030748801958742078, -0.75634048139180377], 
[0.60602025977815477, 0.53204848058857901, -0.46894291434264268, -0.39837544016605531]], [[-0.4925072417502705, 
0.91187394854704507, -0.41856030317546633, -0.90812020367948643], [0.5366482921064788, -0.23328582184308466, 
0.67468807541006015, -0.31833761125219762], [0.74800269718288548, 0.79730510818674771, 0.4094400325745875, 
-0.11087704647538299], [0.93296421979585298, 0.58421573105444513, -0.15034120093276826, 0.18327829802553852], 
[-0.39813625891815052, 0.030781157123788816, -0.3438913330322686, -0.34362154498525466], [0.58525898397507103, 
-0.64895645920882106, 0.41734664322751258, 0.11354461576089969]], [[0.073766572850191725, 0.60579321665324204, 
0.86511599649209736, 0.6845607074560589], [-0.9344511188061968, -0.43446650744752269, -0.58937223256729099, 
-0.54658582419763135], [0.79179482148035718, -0.098282455194240725, -0.9161567933082444, 0.72201853793063586], 
[0.30645267942968557, -0.41124579877517986, 0.44646029231993922, -0.43687986735394513], [0.9415986334924924, 
0.74835329359135949, 0.69231248450236049, 0.55608764035808855], [0.50608378210507543, 0.65051867365172233, 
-0.77324721251161943, -0.41408802371974862]], [[0.98612751222265849, 0.73965686670399933, -0.73721416456178579, 
0.88153201828232564], [0.55738522355097975, 0.8624418072331772, 0.88237786060646961, 0.65536735858337725], 
[0.025673298204298201, 0.19909202314613794, -0.50428642740707375, -0.34061250431125134], [0.81131044124639473, 
0.73691721235398888, -0.19393800399298078, 0.24132598250349435], [0.44997187534756544, 0.67240435898603224, 
0.74064719442631421, 0.4067148649703789], [0.93673805021693823, -0.52488125602224689, 0.65727300612561668, 
0.043487226156371461]], [[0.15549122238130075, -0.76903826049025281, 0.18045700991649904, -0.37060581400476544], 
[0.22767181296035588, -0.87555767956137309, 0.37924330956131258, 0.87218832767300536], [0.23450425378675965, 
-0.37940483531186775, 0.79595336162464903, -0.3758245331376453], [-0.12090559982814431, -0.5514763134834022, 
-0.31508600373245566, 0.86019909748944601], [-0.91695179754935618, 0.82874052785283436, 0.27161276177326887, 
0.63976480833301763], [0.4005367297446274, 0.96504503235564498, 0.23886653677992098, 0.18247612903060162]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_0_offset0(self):
        sh0=(2, 2, 3)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.64035508765567539, -0.25122258889085614, 0.66951154000633939], [-0.55903242315998369, 
0.25529365060506226, 0.96524794120509405]], [[-0.92590274491370184, 0.60284205165899207, 0.051501278867664979], 
[0.82285377500844437, 0.020970123576399269, -0.83209369431751301]]])
        yy=numpy.array(-0.247577414181)
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_1_offset0(self):
        sh0=(6, 4, 2)
        sh1=(3,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.42144184173186128, 0.71830651955106917], [0.35185507895330814, -0.73021599407488935], 
[0.57910572687941242, -0.7812313929144088], [-0.67402312717734536, 0.42783843573712876]], [[-0.11861898014336969, 
-0.45975821998217858], [-0.86696768250501055, -0.41528118555026228], [-0.4018286942428273, 0.1215687602448392], 
[-0.33637496087618524, -0.63043732165181043]], [[-0.99390868950409028, -0.78809153609629234], [0.69994956391014851, 
-0.6490597130833764], [-0.34565792517121197, -0.97893326665578528], [-0.93045520461526321, 0.079137927867569235]], 
[[-0.36827036643368105, -0.88686648649213384], [0.62101407650691454, -0.81131556374997893], [0.76638176917512313, 
-0.94288181175930585], [-0.83189244951481678, -0.79445361899557421]], [[0.11187558796066366, -0.78896508575146007], 
[-0.99498437253962235, -0.58623045102666871], [-0.094343679498906763, 0.1965573657888644], [0.015599421163564609, 
-0.12754509259122959]], [[0.32976089238258788, -0.31198092844628711], [-0.7205998197627872, 0.50194290033843836], 
[-0.26909362723495911, 0.67756963536970827], [0.032439983275907025, 0.50453742630545539]]])
        yy=numpy.array([-0.68940406909485175, 0.95600721587805748, -0.37261160810060967])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_1_offset1(self):
        sh0=(5, 3, 1)
        sh1=(1,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.26550845431871606], [0.00089321056498037343], [0.89703766715155475]], [[0.91392385194080483], 
[0.83168044656262796], [0.30825754392085325]], [[-0.1908033334415391], [-0.41170570694988462], [-0.5386471974167959]], 
[[-0.89421849966658717], [-0.5246656598335202], [-0.66644953450722655]], [[-0.96512676501550976], [-0.37847115013778043], 
[0.023628194139760916]]])
        yy=numpy.array([0.50828379132858981])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_2_offset1(self):
        sh0=(6, 1, 6)
        sh1=(5, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.34202809196595041, 0.6050788541605594, 0.1375166390930016, -0.42162308610462285, 
0.78080886730129251, -0.080704453192573578]], [[0.97932253855306106, -0.17269293739628711, 0.87739789656661404, 
0.27527133202608423, 0.50944842315874528, 0.9513505345830342]], [[0.15682168124079432, 0.81268992205031498, 
0.62025781304047101, -0.058386607220287878, -0.71602664018103357, -0.6192751760125037]], [[0.18660153807783031, 
0.086888304385911663, -0.42180866736749323, 0.30369383461755795, -0.41968399938868362, 0.85706931768677008]], 
[[0.052600721907950287, -0.92855316368333041, 0.22153787512165701, 0.64644523933826403, -0.31909312482941621, 
0.77663139760195699]], [[-0.42151523959634551, 0.26742612216088735, -0.74884348931045208, -0.27772049924697151, 
-0.076193722859920987, 0.8877732822591835]]])
        yy=numpy.array([[0.64292277535383824, 0.66706319172776052, -0.26768059354611351, 0.097143471596036202, 
-0.28310274504764377, 0.51240877352659364], [-0.83449345150834642, -0.82599108946701749, 0.66949238142236434, 
0.65016891984815839, 0.86113345013202003, 0.18989437822893418], [0.21445748599268488, 0.035785402304244096, 
-0.45000704926104862, -0.14899048969983153, -0.51484859581537146, -0.14315934162930111], [0.71994131948639284, 
0.96641104030888325, -0.70400916715390305, -0.39684679212429197, 0.082971671181600426, 0.8313786480354215], 
[-0.53631127152416047, 0.68591598869276416, -0.74467516771109743, -0.89802300285875747, -0.41536950723609145, 
-0.74417930836865653]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_2_offset2(self):
        sh0=(6, 1, 6)
        sh1=(1, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.19354929115084096, -0.41937563495508501, -0.71259697348609152, -0.55674230557847326, 
0.41845223629576833, 0.76911863044494444]], [[0.041431892882880295, 0.60875051528556634, 0.76987318817965833, 
-0.77330919449956426, -0.24180425616710277, 0.84958466989090931]], [[-0.25855044769894753, -0.30283784596093244, 
0.73829429259222956, 0.070826619709828176, 0.14032285598144933, 0.36665878619781123]], [[-0.050302994504118681, 
-0.39117213891635871, 0.48851144561582394, -0.048888780285407885, -0.36286850683934824, -0.26888946131913039]], 
[[-0.2617328227084379, 0.63697356842330666, 0.47660351036428916, 0.41332203035338488, 0.40352900818820814, 
-0.52812856214878123]], [[0.03551079446992933, 0.93175410365667855, 0.72877751212161801, 0.055066222399361431, 
-0.23331296136249735, 0.37396370176158511]]])
        yy=numpy.array([[-0.97537196666732551, 0.71370751623823425, -0.98781068104718628, 0.27244410625765969, 
-0.46886574750093812, -0.35737241155615762]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_3_offset1(self):
        sh0=(2, 2, 5)
        sh1=(3, 4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.99802960876444979, -0.6017495314510648, -0.091171833586166651, 0.15546294446812503, 
-0.51393920020127326], [-0.21432330468096183, 0.84354226635213259, 0.34072239888338895, 0.16255176497091206, 
0.2279234952759075]], [[0.031684218151160515, -0.89966852682029796, 0.23808034173233295, -0.78229192636571443, 
0.043802061333848785], [-0.30268218596847518, -0.65042692732156149, -0.17594502584735316, 0.30939174896825472, 
0.79241543480253029]]])
        yy=numpy.array([[[-0.17806175908871014, -0.81445439282629972, -0.78910643547127135, 0.10261160718097817, 
-0.7308444015792368], [-0.99509720288499404, 0.98091217172258327, -0.10550529113030116, -0.42949675090773831, 
0.022415614578756626], [0.1976407689201245, 0.31117098702583212, 0.64983453332902252, 0.5766593982859689, 
-0.37192304425246769], [0.89184451838326373, -0.66359164081979571, 0.48955449450900623, 0.0035611716906231994, 
-0.18083246968302835]], [[-0.065282385435857293, 0.70382865506627335, 0.52556728261417951, -0.71798193223690543, 
0.58741184837911553], [0.70563180909404211, -0.15458213590554237, 0.33445563666834821, 0.15483131608922251, 
0.37388207512873128], [-0.27459791006136558, 0.38732618193485546, 0.28538480869655425, -0.035020544419192223, 
0.079244062668892612], [0.79393988552885575, -0.67116319178611272, -0.68584909615444323, 0.14511942496419739, 
-0.7425597584104997]], [[0.15848110918402192, -0.58829792814826853, 0.86924859776628027, -0.32285600060913078, 
0.89775033913463087], [0.42369134451637858, -0.61353492145393207, -0.30742171374211358, 0.4998210714138287, 
-0.8536948616529465], [0.27209758298582098, -0.90977887390332124, 0.83864135411752305, -0.26850854283243719, 
-0.0064163891799016159], [-0.28735174525936014, 0.1380764695581651, 0.91419523702377625, 0.62127362891478399, 
-0.22793909184059169]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_3_offset2(self):
        sh0=(3, 4, 1)
        sh1=(4, 4, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.068605208924411798], [0.1659460910837165], [0.24646650975567641], [-0.27043726055509776]], 
[[0.46411907955678555], [-0.2530534597894456], [0.48538800915881386], [0.53414661942188468]], [[-0.74884972792123294], 
[0.77997498288427813], [0.94920301618273184], [0.82234161181357757]]])
        yy=numpy.array([[[0.81943727018146628], [0.74359648582067739], [-0.21871352318309722], [-0.67352783767304913]], 
[[0.92988614849595486], [-0.87937873301454195], [0.48247053312702315], [0.72461910679878039]], [[-0.64584657921441879], 
[-0.42294040944352118], [0.12495731403079069], [-0.079118310375484668]], [[0.043235970635956855], [-0.90993426596490634], 
[-0.63076236058373958], [0.97526621727485652]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_3_offset3(self):
        sh0=(5, 5, 2)
        sh1=(5, 5, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.70111991700605536, 0.41173347800058102], [0.10854218729876641, 0.90199465710449767], 
[0.35799698615541242, -0.99732506435859647], [0.20581239686011421, -0.14156226313953124], [-0.43757439945904375, 
-0.021325276110910174]], [[0.11372221925174086, 0.73190291320271039], [0.38385233581841716, 0.98770252417964755], 
[-0.7299772353552787, 0.27236215704902245], [-0.34993838036004599, 0.036132404114575856], [-0.78038064596919843, 
-0.65632066596296412]], [[0.31008136521318308, -0.29781008391223374], [0.81506339444229225, 0.11473141006981979], 
[0.81477870298819588, -0.29883946114300142], [-0.89573901870528472, -0.28526355171726103], [0.36692558933277386, 
0.056716828476804482]], [[0.91403985400984022, 0.8829104844038107], [0.23509673964172983, 0.82696461822855216], 
[0.98208211192870576, 0.45136695846542696], [-0.6000293819760707, 0.43242215030475362], [-0.94800088190090714, 
0.94272296599935768]], [[-0.62480014572889786, -0.50459275843466322], [-0.23215589868824482, 0.72697242873595225], 
[-0.09669426527460967, -0.37115879987065536], [0.40710437709893776, 0.47366255261443024], [-0.10437953880798045, 
0.62806534996767516]]])
        yy=numpy.array([[[-0.59867091998951683, -0.062044589299270925], [-0.15619955418145737, 0.13871162263317149], 
[0.20106999246600776, -0.74359933670940848], [0.62768015699445301, 0.2760906559262557], [-0.91927679533696804, 
0.29361698159235683]], [[0.93724312116626418, 0.31172688044868724], [0.94791409785943692, 0.29577196681093354], 
[-0.18131904915248986, -0.085154158989659834], [-0.82418654831180516, -0.39958533114275596], [0.022753178670587682, 
-0.53412307901133405]], [[0.80605418775002402, -0.90099562577248693], [-0.51595653716107104, 0.5720655259644678], 
[0.14287524538284457, -0.7550642428098524], [0.87376665672365017, -0.92807368931809298], [-0.58513938837703683, 
0.8119772676057686]], [[0.47570035052314408, -0.063811999169514744], [-0.64992738581814802, 0.91157968854300697], 
[-0.28495284876968308, -0.75633445540279687], [0.51562432135348168, -0.28265830356998123], [-0.0050469718490631088, 
-0.44828961540847767]], [[0.032012780070115987, -0.64009186984193978], [0.051329568597343611, -0.13068778362006483], 
[-0.67203545894740646, 0.74741666533914608], [0.96947012262464227, -0.041533362634079252], [0.6243354992681085, 
-0.45088723534460984]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_4_offset2(self):
        sh0=(2, 6, 6)
        sh1=(3, 3, 6, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[-0.71905043164395255, 0.57124043582416384, -0.90759628596904696, 0.74674455035750054, 
0.19602852629763867, -0.28070685391288253], [0.20876110512405566, -0.90532081472839376, -0.094405846737827703, 
-0.15571697722146061, -0.79668400169618803, 0.40765595975618996], [-0.51513438990557492, -0.58659451581686839, 
0.94433184845397866, -0.13275442606724686, -0.371175847917254, -0.33352462281291828], [0.7139155849245522, 
-0.32677267455676495, 0.012271908652224939, 0.92812288826356126, 0.61991817434004659, -0.018659597138313], 
[0.58777648088462642, -0.77027106018598568, 0.79326648749005613, 0.30891597787867542, -0.87781570853465696, 
0.045861488192678213], [0.77377054822466418, 0.076158220897316387, -0.084046481649086457, 0.86927797438816312, 
-0.21289417919257891, -0.92693660958081114]], [[0.1915926652784874, -0.50695404278329281, 0.63764345681669221, 
0.50648774226380744, -0.097461706538350512, -0.50902686238780426], [0.80405499048626727, 0.69501215091541524, 
0.76194672147228126, 0.91851606239205497, -0.88022610052935835, 0.80739940433569846], [0.90507933193239332, 
-0.37272941294243811, 0.35071713283710704, -0.52463312167070519, -0.15153742894217803, 0.40286556462205581], 
[-0.53031868428424378, 0.57292678385701867, 0.36063975645994728, 0.47066221722301727, 0.53180180688489309, 0.5258602480824861], 
[-0.24025428195005194, 0.40937317990912825, -0.063455860097873407, 0.33543232450283345, 0.20936298322297509, 
0.85617354845758165], [-0.34914878562533458, 0.64961252218962651, -0.2851593666413561, 0.97765357077321791, 
-0.94114383277505032, -0.40701343587186267]]])
        yy=numpy.array([[[[0.75896013637139625, 0.50004684748629091, 0.39728314426909273, 0.80227334945791462, 
0.071111784875541373, -0.58386898983920443], [-0.7949260530525224, 0.22143566115239355, 0.91156625542343361, 
-0.62121839820464886, 0.45787067102526002, -0.17556326041558301], [0.62123719634960306, -0.99713101663496495, 
0.79475719857833504, -0.15017740802059842, -0.19516467699924234, -0.80175704700354666], [0.16068623384681957, 
-0.87084495109577009, 0.16527723435928232, -0.78977268546799095, 0.57153924933331601, 0.95716256975769043], 
[0.54972822765236451, -0.62238869965316423, 0.78971694496972855, 0.79847183081387585, -0.76230803161622651, 
-0.90531165140618497], [-0.31508848514372656, -0.53238865425206039, -0.13027127086094148, 0.92152370825346352, 
0.98160325505498736, 0.45498583431735939]], [[-0.36976473082568906, 0.55639485169410263, 0.898080937277979, 
0.36417666075360144, 0.18923158877015589, 0.88194661905652771], [-0.57285764585926735, -0.32259339325746939, 
0.9584350507959174, -0.76452886168627487, 0.20011853577495931, 0.46208505197528416], [0.25009161372207855, 
-0.45125683613310996, -0.16915016184675369, -0.17473858989426794, 0.15678445927231799, 0.16880698334052169], 
[-0.45968945138043305, -0.17191636572441382, 0.40601162339359531, -0.80760367944392231, -0.057405072049476535, 
0.36305986903527709], [-0.16329276756899458, 0.7040768573238505, 0.42755329930984121, -0.25148926318318177, 
-0.16157273429262742, 0.15883465407582098], [0.46023596641298914, -0.42890150180826248, -0.25222961088019713, 
0.29538775559483077, 0.41520678502194786, 0.65759443831947517]], [[0.034134066338769253, 0.81688481177421735, 
0.86605550789767793, 0.016312780813153127, -0.86857893394173447, 0.62829391093181686], [-0.39545709789418515, 
-0.24847507071235042, -0.72092425664149973, -0.19945356937010827, 0.37477609937620948, 0.88802167540371779], 
[-0.80585043372030096, 0.068102748643353417, -0.34042470882114428, 0.44707644755519005, -0.53236264852423876, 
-0.75149661616699137], [-0.29988611967749002, -0.25295955547307702, 0.98694078475626856, 0.055019047724870873, 
-0.64451325037846741, 0.94068414517929999], [-0.6658544467802141, 0.19748671888361646, -0.82567284196143409, 
-0.70972202966912734, 0.95026959634522523, -0.46043011205121265], [0.89034354262662063, -0.49015331215937641, 
0.24101037212667409, -0.96623674814543103, -0.76227780468892004, -0.8048881827491372]]], [[[-0.65151233670155517, 
0.99645264887778739, -0.34071427733048654, -0.8557600101222449, 0.92523589379030069, -0.2506220073167138], [0.8100721689862429, 
-0.30729849832634315, -0.39887615480684802, 0.79841874662581835, 0.40512827614838764, -0.45179475283033632], 
[-0.40189717973276418, 0.067791454324709788, 0.11388879815727626, -0.59517619379377362, 0.96914404762699036, 
-0.098241872025716992], [-0.065064162399236736, 0.25737192657112695, -0.62842827567993331, -0.51085262797288933, 
-0.049190101497266436, -0.14475328619180505], [-0.29921254110525775, -0.4906261145091495, -0.30765188172603142, 
-0.096013576580243898, -0.27517992677373315, -0.79496802708904157], [-0.63730125932494475, -0.53402282154332825, 
0.11906178376691678, -0.31908654961235694, -0.57224373865196254, 0.39324902649314519]], [[-0.11572540889671812, 
-0.89797110653422241, 0.2876085921684679, 0.081540717975045363, 0.13885986945931417, -0.95237263403434413], 
[0.19348120691730064, 0.47758328110428683, -0.40076587079487713, 0.50746701686015694, 0.30107649277973825, 
-0.16281157065194085], [-0.076899123133251646, 0.62189319127667497, -0.6255334791898024, -0.50854412969716978, 
0.32448281172638294, -0.98992075362082299], [-0.97626123724155733, -0.1868158829157498, 0.48647408000780401, 
0.79737063276323572, -0.036223842341729817, 0.65695120035363641], [-0.45051195390093346, -0.46772349799972579, 
0.79047215225464051, -0.98694408234489917, -0.94807106159139565, -0.80530987113395081], [0.95732954547070515, 
0.04301196437991206, -0.64240017375509084, 0.5682911240458377, 0.75412457439649239, -0.44626684354340895]], 
[[-0.37590060600289155, 0.67987754585051063, -0.64386445077827736, -0.032589877440152071, 0.078601983298098199, 
-0.15943930973769627], [-0.41915868470728856, 0.94879131865947741, -0.035751347536119527, -0.076540554296550933, 
0.72902418710015815, -0.77123544789486687], [-0.28561304132494625, 0.68453420025451894, -0.80129367885690206, 
0.97436669705274315, 0.93783492516808686, -0.27758512664647461], [0.56640019729730318, -0.48102508381365605, 
-0.092113197959390991, 0.86998376393048349, 0.07239123483647858, -0.056042230797117787], [0.15968837433110905, 
-0.43717169009689605, -0.81528176175835609, 0.13635174110364989, 0.8366158784923583, -0.42287363984031412], 
[-0.21922021260782776, 0.35159563798567373, -0.034361813759890003, -0.90895590974636842, -0.55293317691421184, 
0.50538925666316548]]], [[[-0.83935542247254391, 0.93551715436035598, -0.55833918079501421, -0.055628065759604661, 
-0.19305477380218061, -0.095944721423229185], [-0.57157150600391815, -0.7237982658005413, -0.88192529502433326, 
0.53119098654956032, 0.44575660147925866, -0.63664740934389297], [-0.71436495836862335, 0.56929299715200887, 
0.77255490240385805, -0.28341096103301711, -0.81626891262868684, 0.33483781977762384], [-0.47529114274306816, 
0.47491013650607883, 0.062637921497958438, 0.79793100751344692, 0.3064350871164796, 0.35415253227068355], [0.70975424877195015, 
-0.68263314771699801, -0.48303533914831309, -0.85751575000845914, 0.41029985310543338, 0.77437326860552935], 
[-0.66870017605594967, 0.63601458971501335, -0.41632415486912744, -0.22953276842894943, 0.97421646214068947, 
0.53130066113705077]], [[0.087527634310498037, 0.71219678898892358, -0.84376752150116152, -0.95644733994059195, 
0.33502990288505474, -0.54105133386346949], [-0.84009107549236361, -0.19877940653017601, 0.38645019697231686, 
-0.21719112459938561, -0.19992121782823191, 0.0040619416793195029], [0.63036416300565867, -0.23861683205234585, 
-0.54260377589114439, -0.80132618803712052, -0.87908181830562437, -0.72093622845927374], [0.87543702717600413, 
-0.73953707864723239, 0.669642102934354, 0.52082210185170985, -0.4013172661676454, 0.020829066937562812], [0.89038602885181506, 
0.19994889777575442, 0.25659116377980817, 0.019429580106713473, 0.54250480220596198, -0.78095214011695013], 
[-0.7175619609421191, -0.1619105560129086, 0.6078032758130556, 0.43232770916404806, -0.057389404971255997, 
0.81267769278512225]], [[-0.88924578216942796, -0.82346714253807329, -0.36519190289864856, -0.78587889907361985, 
-0.56581697793344699, -0.99533029600171941], [0.50657718864453272, -0.20809250567964854, 0.65855636671785778, 
0.11551312541608527, -0.29900270320234257, -0.45456734318118297], [-0.70856779714130824, -0.25690211492956672, 
0.80341962321706073, 0.3677817318500225, -0.098804313992064907, 0.14176038700946947], [-0.84825999004975183, 
-0.35371730677599311, -0.02408158570172092, -0.63837226231008848, -0.51681229980229348, 0.55552311426628997], 
[-0.72508384338515208, 0.58840849287281882, -0.27673196666401134, 0.11679516158713144, -0.46890198119848403, 
0.67214909564214542], [-0.89645550543545172, 0.19178764820615024, 0.80504072365283341, 0.86966863498020608, 
0.0041912916579451842, -0.79069130742598182]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank3_and_4_offset3(self):
        sh0=(3, 6, 3)
        sh1=(4, 3, 6, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[0.38262654298686161, 0.78791982715524456, -0.27190954587414362], [-0.20173406074885714, 
-0.11967369322714538, -0.47945092804827305], [-0.78692761313883985, -0.38311142517509289, 0.53156505620826988], 
[-0.32459697270173926, -0.60145183605057317, -0.016399809006024224], [0.98893112392921667, 0.58976180889099505, 
-0.96240232146780369], [0.18085388778180578, -0.60701413573451624, -0.61639917640979913]], [[-0.33110575522136365, 
0.52746297776922257, -0.97831577159785543], [0.99504513875524969, 0.61124247971430612, -0.49285333195425696], 
[-0.16418880261734747, -0.10593157583585078, 0.21086375425825277], [0.29356976380002275, 0.76987667844655872, 
-0.52762953243548139], [0.68443809715915926, 0.89411406632469115, 0.51687358956697382], [-0.75625669408230656, 
0.24992134257839482, 0.92158967836137129]], [[-0.73629674464918815, -0.29074518527615001, -0.024707311398679099], 
[-0.02155659293474721, -0.88127698766793849, 0.46307234395168573], [0.74943610124164772, 0.77628457126745998, 
0.16130476819523731], [-0.53312403149280496, 0.10151673975608722, -0.98405510832124543], [-0.31875051901218532, 
0.29740155400036405, 0.86811727942569172], [-0.39292978005466339, 0.51433548268336438, -0.48950537490809398]]])
        yy=numpy.array([[[[0.27352363194261309, -0.60887390160273425, -0.038439886041722771], [-0.92586134054981106, 
0.25418395021428153, 0.36276865224159049], [0.78625125990679923, -0.14315041650477944, -0.4085580158236124], 
[0.10454384558598417, -0.80808948317185947, 0.38634717519193384], [-0.73169824421511276, 0.84974441089566399, 
-0.61284342800034275], [-0.16524570079052281, -0.77772727834150146, -0.24887695835586676]], [[-0.020134795953986151, 
-0.093326514706296049, 0.17153937186092238], [0.89535239112508003, -0.94041273036269901, 0.97171398805683507], 
[0.92555284613487432, -0.68328048793492857, 0.44369167298393863], [0.28107762437239825, 0.10307808011406139, 
-0.75808306816183624], [-0.56249492048924621, 0.0337622624124847, -0.043432844950205718], [0.89398183872843728, 
-0.13305267985890001, -0.60174961029334595]], [[0.70370430258331806, -0.41053725897521165, 0.49540427555957556], 
[0.9974299170380907, 0.071299053360297293, 0.25140270780728802], [-0.33766978048814389, 0.45728019753288507, 
0.056372449933461022], [-0.67890233180996984, 0.82119185852410292, 0.84332294386255002], [0.59246799625506252, 
-0.86476996614704493, -0.24771311845174515], [-0.97143618544662291, 0.58943182830892527, -0.55081414787060656]]], 
[[[0.73132048487208579, -0.77588592556754699, 0.59913460401160767], [0.15316532550017103, 0.14890161785189804, 
0.14400399950673859], [-0.93844406549699633, 0.1153338262296979, 0.19750287979984038], [-0.84200771132577068, 
0.97875544693243199, 0.61678647297961597], [0.09765897044371652, -0.6153340912680394, 0.89954633735900624], 
[-0.3709027417630828, 0.18359103258777321, -0.90828930806270303]], [[0.45693266423640488, -0.79265208871081283, 
0.15636119900105228], [0.039784475464153202, -0.51967662957927496, 0.088087347938756499], [0.23917986753852216, 
0.012978576868522218, -0.41559815473829809], [0.17892590577587697, -0.95733471322871133, -0.27760649256286785], 
[0.17457212960542057, 0.92798634483547082, 0.27133291103737034], [-0.03824655843319591, 0.32370489872675945, 
0.23898083375399248]], [[-0.65245497105552563, -0.71458555570328186, 0.37612555970811434], [0.066698161451475002, 
-0.11938125491967821, -0.71055728895345371], [-0.020615509654541953, 0.060157236916275503, -0.036964010974019024], 
[0.35150767058071097, 0.35346734009384306, -0.34195897576052725], [-0.26169055013430809, -0.74791727487846837, 
-0.74684287242087111], [-0.52015453650766563, -0.40892099943469429, -0.76022125648476968]]], [[[-0.33429812347608023, 
-0.79601869309229989, -0.92321893026500268], [0.71356003505966004, 0.28669758872504469, 0.57636169107304314], 
[0.78536490506026113, -0.27769235324748798, -0.92256525968438963], [-0.4592491250485975, -0.096039826017131791, 
-0.58251911356922914], [0.66039623829191729, -0.931982709210599, 0.70699631511153815], [-0.34845492316154547, 
-0.89254731001836474, -0.49562115610482982]], [[-0.8641484301847191, -0.15968348784883424, -0.31912552649521886], 
[0.55428880491997301, 0.69019519762040105, -0.88718482028928536], [0.021756401314493523, 0.57373378110459705, 
-0.72738266665702689], [0.092484222487484224, 0.40672867579942773, 0.45682287473368266], [-0.275623561859204, 
-0.52961004548952673, -0.67925749786375778], [0.94342406044476901, -0.9210906149972613, -0.69605732780060969]], 
[[0.86071126833352474, -0.4444750377110267, -0.91177929424016346], [-0.40154555163723393, 0.8332519989834839, 
-0.37752244997896489], [-0.81790448966892204, 0.18686475049637452, -0.42958946299448364], [-0.47158720587177272, 
-0.42238275571133621, 0.56565190772224949], [-0.14356994023927294, 0.94070131380385691, 0.53417195858459743], 
[0.94304900829960592, 0.57250265901288722, -0.67818072777762639]]], [[[-0.83864416444040302, -0.35321381910368022, 
0.29347208992035689], [-0.25752351846366817, 0.2268696369431884, 0.38714431519955617], [0.64226313987805539, 
0.43237839062912165, 0.20315585546202275], [-0.6147779919052514, 0.01942682631773418, -0.34640979156039209], 
[-0.57933253763850612, 0.84359177482420056, -0.8638604283186706], [-0.61521729920945578, 0.85640843783722587, 
0.18877322832390631]], [[-0.68166372462353575, 0.71914430111251715, -0.1099465631902441], [0.19138351162590905, 
0.032690671310474428, -0.39144362086777051], [0.54870015786186377, 0.59432436783740816, -0.37684165543359871], 
[0.54831625294714237, 0.16290258816157666, -0.90682361529286459], [0.13422341198140564, 0.86778182259571035, 
0.53109157292309317], [-0.2777232991042029, -0.66748734290555989, 0.14482578423186232]], [[0.70130429769554636, 
-0.8813534752363541, 0.2560655628827786], [0.1952252225731268, 0.47630870603302622, 0.613215784728391], [-0.27722914247520869, 
-0.85843073879193255, 0.98746121778035789], [-0.86369566465635539, -0.80251922275149146, -0.63883001789575977], 
[0.78352913564507376, -0.11120040702351619, -0.84529773586815837], [-0.18706996851021085, -0.69097827017435054, 
0.30315460311603371]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_0_offset0(self):
        sh0=(4, 5, 1, 2)
        sh1=()
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=0)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.13639899516668108, -0.22304238400248066]], [[-0.29240609251946981, -0.31430586856074072]], 
[[-0.10264489596172144, 0.88584808467882725]], [[-0.16771816139563378, 0.50452095878414327]], [[0.77825657740490106, 
0.78432974663269261]]], [[[-0.94794488364793517, 0.13533633624763564]], [[-0.25051618999604486, -0.66985598583648565]], 
[[0.64407285405591552, -0.12894794645504892]], [[0.38178000527002132, 0.3636620660780383]], [[0.35420360177874333, 
0.24917948362044351]]], [[[-0.076893501611223325, 0.10639624613975851]], [[-0.59860362581112359, -0.41363716421548014]], 
[[-0.1519125931590235, 0.82103948443147212]], [[0.29747403326082611, 0.86222497361462191]], [[-0.65756229747034922, 
-0.53930384154288524]]], [[[0.27257331904856352, -0.55490428315800178]], [[-0.023608441465114982, 0.11684166238467131]], 
[[0.7368912600500741, 0.22559804857599941]], [[0.062384334363025706, 0.2745346510411919]], [[-0.66035305573163861, 
0.14051397070795169]]]])
        yy=numpy.array(-0.808556827778)
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=0)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_1_offset1(self):
        sh0=(5, 3, 5, 2)
        sh1=(2,)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.028112796109055704, -0.52494315243512846], [-0.47761123092820523, -0.53002270237637106], 
[0.51545598553586558, 0.73258952517540088], [0.99867934544572723, -0.37173132710912871], [0.22291753007123316, 
-0.13939518248185756]], [[-0.096864026537502701, -0.53482542204922345], [-0.25133745802873864, -0.54825808262885412], 
[0.83553930794759235, -0.24950698007421201], [0.29733162817327874, 0.45683238969530149], [-0.17907474988503558, 
0.73048243310877559]], [[-0.44247471798919835, -0.78482117528581741], [-0.52791973565073502, 0.13065057078009357], 
[-0.008158475596824255, 0.021402513119199496], [0.88266489580191076, 0.44119114845392593], [-0.45369109218959225, 
0.52333202361978781]]], [[[0.36042859009075157, 0.92142585658943399], [0.068134828521093205, 0.12208025489843766], 
[-0.37336278102389242, 0.9473716141259787], [-0.60806583593653651, -0.41379743762984988], [0.82454804883284849, 
0.014425947975733378]], [[-0.74055589442208314, 0.90670379231283249], [-0.45367971442641752, -0.51954442332742001], 
[-0.83127448061961906, -0.51854429756169806], [0.32539265115858251, -0.91029497866008691], [0.18969614177462502, 
-0.079713140484046097]], [[0.98239303252102395, 0.88712352427683649], [-0.33396514173870018, -0.093566494265546973], 
[0.55900402206540956, -0.19878179695032805], [0.27692232623057067, 0.4601925223560992], [-0.22156230492220241, 
0.51671401775935233]]], [[[0.64977418527110586, -0.96031001081982525], [0.72697717239199822, -0.87122241079092655], 
[-0.07459196560909076, 0.7312083888949461], [-0.84452644427473378, -0.82488518612639505], [-0.82381484622798729, 
-0.055485442091665282]], [[0.93161992484551237, 0.63598442261254262], [0.45439789065120184, 0.87699887053895265], 
[0.08859129198042015, 0.13516538350501905], [0.9383001668099249, 0.4422540399609749], [-0.98343051989148567, 
0.59522064064770408]], [[0.63733008953020387, 0.19287266432201911], [0.31804508946620103, 0.31397876141188075], 
[-0.79160669597198474, 0.20570041519400051], [-0.99427882731071837, -0.84530648072363324], [-0.10125930551928031, 
-0.4818437034881351]]], [[[0.38957088121771144, -0.23984089401310915], [0.51660317032640202, -0.78339787214821865], 
[-0.63347718641867701, 0.91741386672010838], [0.79765907801747638, 0.46259699280104694], [0.62191333149865002, 
0.89541461013105472]], [[-0.51042535753533413, 0.56519824607205993], [0.066608855862572369, 0.5099656034103921], 
[-0.86576992661358387, 0.54087527624513765], [0.15816298768145298, -0.58980918680255767], [-0.24893453666332332, 
0.35605235371645794]], [[-0.84109658995346681, 0.69797739126550185], [0.1552721655252387, -0.75480762357803455], 
[-0.37937161326849544, 0.50892346277132638], [0.98695569919827841, 0.63249497822226175], [-0.53458780031964714, 
-0.41832912114814813]]], [[[-0.027375338959985784, 0.48040520839854084], [0.17141567403111968, 0.42258060681619747], 
[-0.64618123794885762, 0.82361959409586016], [-0.76032297083351463, -0.59466495900325445], [-0.61540116977413595, 
-0.36399344746370943]], [[0.42484669576932532, 0.13922237547263405], [0.40994300617596702, 0.45558502048074612], 
[-0.95791273271477628, 0.19660600226637159], [-0.61721390305234802, 0.65923458717205508], [-0.93769150798096024, 
0.49091871049674807]], [[0.49634488126477905, -0.15368435122675494], [0.25703268094895315, 0.3236931200243589], 
[-0.20994042380895528, 0.4297747739819322], [0.050560306390537768, -0.31454287533719172], [-0.1079000359691018, 
-0.69434244577770099]]]])
        yy=numpy.array([-0.70581702915383393, 0.84738570414660641])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_2_offset1(self):
        sh0=(1, 2, 6, 5)
        sh1=(3, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=1)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.53860710110856669, 0.19341467708276827, -0.53110162527227534, -0.0034235010491499018, 
-0.93638559197423832], [0.13113913699190372, -0.90262683705608104, 0.65960297130969225, 0.6186371100369159, 
-0.90910733266682975], [0.47106938091126915, -0.55591367598378594, 0.80312797469636044, 0.29907305241616511, 
-0.065198306414174567], [-0.68406137144931667, 0.29651702407373204, -0.31132250964027652, 0.96710106666107643, 
0.36739275816459194], [0.92588108835379068, 0.33823398143931227, 0.6854992948914056, 0.64394201748493263, 0.94000172608925769], 
[0.85527511268696776, -0.71424911819324199, -0.66605125510837104, -0.16314648372523965, -0.36592006505366825]], 
[[0.35436761240394055, 0.62901194905682556, 0.17141287131590466, 0.49402948248177947, -0.65000726442287249], 
[-0.10161830836734653, 0.95323937024662353, 0.47203862648234063, 0.7093086815275349, -0.80626293473613431], 
[0.90687384204965116, 0.20142561612132481, -0.42847501254403353, -0.091378683070095557, -0.54583134802833522], 
[-0.16563225437281592, 0.27461591605166924, -0.64396688482886266, -0.92806154615972036, -0.48604626970081166], 
[0.020902467585537154, 0.025224283304103867, 0.092453101943310578, 0.50791243494212512, 0.73263512113616835], 
[-0.93065267562656162, 0.50544703438419747, -0.80485121073500587, 0.98520404310146703, -0.31935548923758272]]]])
        yy=numpy.array([[0.78313389790832422, -0.99683166766219222, 0.99619864900151933, -0.78022627463348715, 
-0.75467020864364276], [0.95287302253812212, 0.91206527282561711, -0.6110303159024173, 0.01765522113721385, 
0.9852102007820549], [-0.27338487483387408, 0.55927282962817304, 0.12991446162440301, -0.73700683213030915, 
0.35131326812554664]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=1)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_2_offset2(self):
        sh0=(2, 2, 4, 5)
        sh1=(4, 5)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.58163799480014511, -0.27536157201116418, -0.54044716881704979, 0.40327912248849396, 
0.063729314162940032], [-0.094238741647578061, 0.012469599340006354, -0.99145484575261955, 0.043797023722451867, 
-0.41301302517509031], [-0.94321250185184891, -0.70206943764305696, -0.061940399289910353, -0.037280170838496662, 
-0.90464504324939132], [0.94454383166784073, -0.74765718546075299, -0.81254516130735421, -0.42768258690407612, 
0.22580897518446208]], [[0.29985373577471131, -0.68575535414501676, 0.28358587004035263, -0.22165760116175615, 
0.91799287441692479], [0.089978844470657737, 0.58740948458438647, 0.55812935918951467, 0.19203986555711872, 
-0.018181001380035866], [-0.33861680885360701, 0.32712139957292585, 0.33531206649152479, -0.44869835173098216, 
-0.025980931548958308], [-0.15424375867119866, 0.14772014335866945, 0.46936293382936056, 0.9544661757216486, 
0.1320584955677695]]], [[[-0.55363098007492551, 0.86905373743223535, 0.91936036186839853, 0.90017100841973963, 
-0.77874284578004827], [0.81936609063598542, -0.26293268723883201, 0.069528288473563071, 0.0050239338139055345, 
0.57313079874002737], [0.58160146080872677, -0.57080164120443211, 0.36390894688506292, 0.52307577676362227, 
-0.36411882748354563], [0.15039251904995909, -0.69391364288529811, -0.063482724200391694, -0.033384291734013471, 
0.28536012066581451]], [[0.16159716309149186, 0.072891837550062455, -0.31547322951962986, 0.84171736823848398, 
-0.58662139486086473], [-0.93137072780661478, 0.5149672896635149, -0.51776503125024331, 0.97609740124406863, 
-0.41416315467373721], [0.9230318217827258, 0.38747963768842197, -0.35880893296510585, 0.8857249720158471, 
-0.88283565641644701], [-0.7460724162061263, -0.91116214735020984, 0.63033461203661645, -0.65804158254457445, 
0.80986391051176265]]]])
        yy=numpy.array([[-0.6500178371891614, -0.22782727388312929, -0.41594672766728102, 0.35612237653610701, 
-0.37133747552652085], [0.94745061529945374, -0.13391624855657658, 0.29795974276526471, 0.72478337691385897, 
-0.96863424338259252], [-0.8015722994602954, -0.7391704104740342, -0.39774563541428787, -0.56888300420334437, 
-0.35860218556647872], [0.73939503227042547, 0.53265416451686809, -0.90428584621088426, -0.85133583162684556, 
-0.94737996519926204]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_3_offset2(self):
        sh0=(6, 1, 6, 6)
        sh1=(2, 6, 6)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.050319407511412617, -0.7606469067792776, -0.98724726750518421, -0.13517106605262286, 
0.42618218057125823, 0.23749899722648293], [-0.87274429062641068, -0.70668053977304446, -0.46360114951493969, 
0.8872213640978186, -0.27461629396311626, 0.10508469944939902], [-0.87897554010751544, 0.19524415222965441, 0.4486724616314941, 
-0.44710905728208394, -0.45319374652653832, -0.23849944726581707], [-0.98200013222405991, -0.59059295928488531, 
0.1855442724079821, 0.21920436212893901, 0.3865667375852857, 0.55772451944013568], [-0.37503533064637185, 0.059280970209244721, 
-0.50598743338175733, 0.90545834953892901, -0.2689329244589822, 0.74891621406100017], [-0.93339725281591579, 
-0.85464220749663888, 0.10258594858403169, 0.77935229543724427, -0.47590384954560072, 0.04015989069042214]]], 
[[[0.014451680192207661, 0.80867685757967811, 0.44677493812130975, -0.94153123261098592, -0.76008266987352746, 
-0.36723005992711055], [-0.071134233319928519, -0.68596823052699896, -0.20555904707616368, 0.52010992910069098, 
0.81503436560738951, 0.81352189778755601], [-0.67837532998675587, -0.30661869041269907, -0.2800980220051057, 
0.35746905679712371, -0.49769889721402238, -0.34666461206811161], [0.05681385070828493, -0.41577069565011637, 
-0.92780457835088437, -0.72984910282870175, 0.37042935549951594, -0.3202296035335539], [-0.52177221362162785, 
-0.070735536359984863, -0.80013492847915346, 0.27718670806202006, -0.49853435039704808, -0.79851231774474929], 
[0.18247654108419575, 0.11309551789969796, 0.71711786406994804, -0.76480617012475949, -0.25966458789904601, 
-0.88026739355660522]]], [[[0.6718505798121992, 0.057770590242040454, -0.29921258330220124, 0.22423129890440552, 
0.50808153716888449, 0.39707094168391621], [0.17000212955608229, -0.13105503396095641, 0.0028771026923337661, 
0.25327012611839672, 0.54679549236357516, 0.12617264403835327], [0.17492592666616869, 0.74518503093202848, 
-0.31001708679225692, -0.24155297938750997, 0.40875868400417259, -0.46268460490960406], [0.53991883005428809, 
-0.3625071400902482, -0.24490493988977313, 0.16983490710403615, -0.33428635153852682, 0.085633811440430474], 
[-0.56349017344680763, -0.84696687346065325, -0.72839064755298222, -0.96242037046962658, 0.90846122411483599, 
-0.53239403111464756], [0.46828196751825324, 0.59729000296259449, 0.85600827389307099, -0.41021891874610672, 
-0.96277132557137457, 0.033345087212425684]]], [[[0.76162650508990759, -0.64011786952514016, 0.81520562007405872, 
-0.28241780198309252, -0.037039984706354501, 0.68108560443038524], [-0.53018502108847954, -0.32789297543432827, 
-0.80919247730258625, 0.82876536110331456, 0.5104526964391356, -0.0030395115523158545], [-0.60306130641466482, 
0.26022576810231968, 0.89842544638769684, -0.21585925488083713, 0.1382235252080688, 0.056388222327959481], 
[-0.29677415235666404, 0.85938546388987014, -0.85462121776346112, -0.78284490510630445, 0.72231196310298396, 
0.31539002741294242], [-0.47236427908016121, -0.79014609787943146, 0.32083020699858555, 0.64306313129940484, 
0.079162607187303946, 0.78109100164230094], [0.19081600790612274, -0.31856994861555754, 0.641988009233178, 
-0.63907216273425549, -0.59450539268165437, 0.68983891349851101]]], [[[0.25112426979705527, -0.018480164641835328, 
-0.32150370689139618, -0.38426152424568905, -0.38700910666495481, -0.14294824996884015], [-0.92214956861316355, 
-0.6728056426554192, 0.71619981773624208, 0.20031022268118237, 0.94446420278993215, 0.23728974732565611], [0.36037469605509176, 
0.78899288072049867, -0.94090376588535163, -0.24058187175406753, 0.233554815395842, 0.37039012967399465], [0.31480078839369896, 
-0.77372666211419894, 0.97742042291483688, 0.29371828749921636, -0.42144326926657616, 0.8657029615976175], 
[0.83853231807267359, 0.21567798444964814, 0.67651110889933186, 0.21647592775267288, -0.40585726241766773, 
-0.95619531978055794], [-0.1716974612353015, -0.053326382069342371, 0.24357151540996935, 0.056097333899976176, 
0.070486960520566466, -0.37765586873386847]]], [[[0.84558092197378243, -0.62164367615479632, 0.57053871773058606, 
0.74225660377627656, 0.34484513469863565, 0.95570505294597585], [0.39491000647004415, 0.063871776922492129, 
-0.42228107556670347, -0.33732609478070175, -0.5023859018469945, 0.4419376048788739], [-0.1938380746768833, 0.4821446499094717, 
-0.44892474638463931, -0.63444443419713026, -0.53981614997030403, 0.72033247846927173], [0.3998823529815918, 
-0.68960630190297389, -0.76005367115969191, -0.27768070049338389, -0.93601928516073918, -0.26035239800356846], 
[-0.51300925329087432, -0.19770791472750537, -0.14320295591980803, -0.24429144368725986, -0.88779132294006469, 
0.43627085273413369], [-0.69150599868754803, 0.2810993966131623, 0.96042375218144671, -0.32368973707728932, 
-0.39135687853287404, 0.57546011878738312]]]])
        yy=numpy.array([[[-0.76163079384329135, 0.2319424521426583, 0.044922705497931537, 0.87053725223831036, 
0.22907460953998093, 0.69596598885915939], [0.061452319435067571, -0.71558712131320767, -0.28284091581460546, 
0.95875995521535229, -0.89909487778756314, -0.41691266819099071], [-0.59506819273931977, 0.55372314309315951, 
0.18943106870940007, 0.62640859465788279, -0.51780223074876575, -0.60597922967445683], [-0.97514019664341123, 
0.29968713669528912, -0.38931360507738533, 0.50321838525950913, -0.14592096431702117, 0.92553673215690146], 
[0.76877044695670316, -0.30093344146492518, 0.96385053581423219, -0.60354339587208972, -0.16073623554710359, 
-0.052289919027013188], [-0.26595353127580568, -0.40900204334434931, -0.85642328592163652, -0.15273098085208692, 
0.89350488008910567, 0.18462224445780606]], [[-0.42890396677119202, 0.24163511408384797, 0.74026962962646858, 
-0.4539040429531509, -0.64240157688846322, 0.47112311070770496], [-0.7297378924888378, -0.4017245343370579, 
0.93449748496986285, 0.03440628475723928, -0.42099746417542105, 0.71955511982415743], [0.98475878298811881, 
-0.52628641457147629, -0.92823503064246071, -0.16737182870694078, 0.64948852598896623, -0.72255771270826807], 
[-0.58212272727824943, 0.74992505497716011, 0.1846461137412021, -0.48083413993249957, 0.98937098742043839, 
-0.21287561396765975], [0.69145343639848389, 0.10731399466957092, 0.75389579028300591, -0.57242028694596225, 
0.7363705865742558, -0.36050780422435058], [0.32591212538249392, -0.19755509168914265, -0.091696461520458517, 
0.13007386900186479, 0.18876479982306993, 0.43518768024960242]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_3_offset3(self):
        sh0=(1, 6, 2, 2)
        sh1=(6, 2, 2)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.87211331802479886, -0.21569893185129407], [0.16670293926024371, 0.0084885354695529713]], 
[[-0.1180821290266707, 0.040368775172895166], [-0.96952509120624009, -0.63451448284147216]], [[0.25834920339279144, 
-0.83490874935155523], [0.37705677778343083, 0.032508397561626712]], [[-0.108042990730431, 0.050957130070077739], 
[0.31956982890903118, 0.99780465701261045]], [[0.52420838330088859, 0.071682640533676123], [0.24708908182308531, 
0.09832477320623223]], [[0.054182874394223868, -0.54908908315599958], [0.38083156787266992, 0.13139990639711363]]]])
        yy=numpy.array([[[-0.00041234406860635175, -0.063786867732705588], [0.19581497918260649, 0.12672848224128574]], 
[[0.048173947814174056, 0.69666696781077775], [0.30209951443252003, -0.81804842735007988]], [[-0.26183454678400042, 
0.011421092774443808], [0.83916091786334901, -0.7349725609016744]], [[-0.25362022919256955, 0.99365404824846792], 
[0.56241446409266715, 0.061086215599543303]], [[0.011019867108348347, -0.75331741133240571], [0.65348327840142639, 
-0.12120558546337934]], [[-0.20941660510553839, 0.67613743568997986], [0.045519376869203043, 0.18517349784182446]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_4_offset2(self):
        sh0=(6, 3, 4, 4)
        sh1=(3, 6, 4, 4)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=2)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[0.18056875419335827, -0.5582811509873804, -0.28742885076096991, 0.87726769720628894], 
[-0.50111878597207959, 0.022782200950089537, -0.55249536903817598, -0.30822228021330078], [0.22904318684461411, 
0.26904067955566768, 0.27243094503563015, -0.37140069915763041], [-0.85538394316602218, -0.65668737465428628, 
0.50355083888136964, 0.1168317287290308]], [[0.23698287725080447, -0.44679444138793278, -0.4355770512064987, 
-0.91791851673422009], [-0.49226339692979248, 0.43896806650148079, 0.92683678012823356, 0.15940342187963719], 
[0.37597577688040529, -0.093711436257440273, 0.76082963803730785, -0.52892589221603181], [-0.87690015200525218, 
-0.41901530719799274, 0.33549992484475988, -0.24102835333987715]], [[-0.040844385687434315, -0.79556575060034973, 
-0.1305252904408658, -0.050134077540767041], [-0.037120300372950332, 0.19813351386443645, 0.55584213994419751, 
0.69217090967273043], [0.11380515213572751, -0.34765420050704909, -0.70272970500023746, 0.75432508725404435], 
[-0.41457009457340588, 0.80846248937380416, 0.79139713471033879, -0.20996850336844841]]], [[[0.66932731350684316, 
-0.71598750431283609, 0.22750495859244668, -0.37720488863395429], [0.73350331254782231, -0.041786731040243774, 
-0.92367513005593005, -0.062168200035398913], [-0.54703164757482425, -0.22625739208982676, 0.92439998058484174, 
0.72563107598869214], [0.71824014181840434, 0.31103931192733736, -0.49799445025903566, -0.77908371775739749]], 
[[0.416658100671353, 0.51460831581965727, -0.10975096893952352, -0.41253210662765549], [0.14524181654340573, 
-0.52876005944264182, 0.21217331262326478, 0.73042873224485971], [0.19824070343188294, -0.32350016369120072, 
-0.079511349572281231, 0.39543599937944873], [-0.9455301882112479, -0.67424020742084734, -0.51080691390008748, 
0.92018305451567062]], [[0.41824314778292671, 0.81261749121312077, -0.13974260376809577, -0.20995250871355786], 
[-0.43320038815221995, 0.93422227076024722, 0.55938319971007111, 0.87092986973038622], [-0.84085031390034715, 
-0.3037612612870404, 0.24109254554939996, -0.68801535184027207], [0.50187949954458455, 0.11242262993581043, 
0.28207149963051048, 0.37929603821722968]]], [[[0.80088518456884072, -0.048510782959318322, 0.38927747077348629, 
0.13254955900556831], [-0.27167164471462502, -0.5576882555739644, 0.3191474280600064, -0.0099596864569242438], 
[-0.11889660976466021, -0.92918634942648026, 0.51849643231901421, -0.12026215646395078], [-0.19362177482155118, 
0.40135581719999691, -0.38218688615384977, 0.27436239792827344]], [[-0.080498709442198058, -0.39045168635873773, 
0.02312130626651876, -0.29303358264875201], [-0.34724793521370856, -0.52710296511267685, -0.43512013908502634, 
-0.94478668390614562], [0.1942300791611169, 0.88407195912325909, 0.22781337855487771, 0.55151919567001273], 
[-0.78099273522884105, 0.92555111383566135, 0.8642061206863203, 0.38225960972675965]], [[-0.56677880753745846, 
0.34667696342785148, 0.63390703047228958, -0.1958005385888808], [0.9108485909356403, 0.61041126397579459, -0.46811673984247859, 
0.53787642051996087], [0.54238407393428201, 0.61742126813085507, -0.48506663323145216, -0.51823203853509803], 
[0.23248096542498931, -0.68357811482202147, -0.26057605972489184, 0.96083429112994168]]], [[[0.1395981324364346, 
0.42803617345268452, 0.15796293413968465, -0.035445046942211178], [0.11103713918527558, 0.435942733906588, 
-0.75689305971730181, 0.8156433466052917], [-0.9603554947820423, 0.80598854405806653, -0.87462079819075522, 
0.1381051950744463], [0.074152176573451856, 0.88191478360466991, 0.093012968466941182, -0.28381958093924764]], 
[[-0.35514192363624053, -0.55345812913697712, -0.20766117642218673, 0.20961346276362391], [-0.58699054639622683, 
-0.38632143864457413, 0.95095514332608677, 0.46808237807671627], [-0.99391833963865084, -0.74061835422059752, 
-0.95379802416391701, 0.65417107576772526], [-0.54528496295121953, -0.5718344158797084, 0.10155275534330244, 
-0.38230151615383057]], [[-0.024580587322448721, -0.3533344407248511, -0.29915814521023676, 0.88304506305262476], 
[0.1930290131044794, -0.12796912903425861, -0.43864747990033437, -0.72620413413196139], [0.4567803065027829, 
0.024377856924195473, 0.37835456303670267, 0.40763680307032857], [0.68821477585882351, 0.83719821226040581, 
-0.05449788525161825, -0.61679982954605017]]], [[[-0.542752579054391, -0.7277786528382677, 0.76186656693020383, 
0.99633817224565946], [-0.86646908014985091, 0.98932380283806376, -0.24069268837042168, 0.49611488008985827], 
[0.5288503903706443, -0.50878830162262534, -0.038293381159126394, -0.70148903322311296], [-0.5805159395975894, 
0.18576864462234433, -0.073379105167064029, 0.25955887673782674]], [[-0.49280367790650081, 0.34028169690541521, 
0.84768493029316527, 0.60177232842740702], [-0.92178095442715735, -0.83819032383063963, 0.9238778268368566, 
0.87134671686540854], [0.56660690763178767, -0.56482646091366062, 0.55510441023020984, -0.24259133578479597], 
[-0.59374924170750143, 0.94965881145918996, 0.78322019229231854, -0.12016203016968929]], [[0.43748149278846871, 
-0.16222039733932792, 0.47754759399509972, -0.034189939345754317], [-0.39168585579264037, -0.3686085479443677, 
-0.8854888512284067, 0.51500497520570976], [-0.78079009667137389, -0.61063304341437696, -0.70427717400046341, 
-0.26179203468582624], [0.97240973607899006, 0.37868148589660677, 0.75722451386076051, 0.50448843841402358]]], 
[[[0.085672556431644731, 0.38731046382309153, -0.14467118154773351, -0.2365129120219418], [-0.65225164297008043, 
0.93346025057287152, -0.93285150317411558, -0.41303056619537459], [0.84328944212534984, -0.66212218773441545, 
-0.98855630417176421, -0.33668565226549485], [0.96376204316961256, 0.01394892390957736, 0.96264533774995376, 
0.58855261235438472]], [[0.048071794588259609, 0.96702773080791604, 0.52905979824793703, -0.44649886013325912], 
[-0.83755764034283864, 0.6016696421044474, 0.028951716206993883, 0.30388878834648803], [0.62902444519377698, 
0.34766738034675959, 0.12785128050122774, -0.23167986250941053], [0.43126987834562747, -0.020715382383207137, 
-0.27635602094988454, 0.51269445359163446]], [[0.84457376923501193, -0.47174689930026137, 0.63290021710566302, 
-0.20104009982377113], [0.58042189690164303, 0.076910632798805167, -0.83924222396292736, -0.41662783836807771], 
[0.862150578753198, -0.010595782366137119, -0.18451331057293863, -0.66355099276779361], [-0.091729554897370669, 
-0.88754253059815635, -0.51026709188162855, -0.83193402132503169]]]])
        yy=numpy.array([[[[-0.053358970936407735, 0.23322703754907015, 0.16807425152320143, -0.73530766464658814], 
[0.40967191048656204, 0.64819942856253876, -0.4865973935063721, -0.62121215203843505], [0.10842098075769502, 
0.99849501560455645, 0.45752613576351808, -0.34172254601396523], [0.8942035593281541, -0.80782946793767052, 
-0.43608563703062209, -0.80851243401753914]], [[0.064565704831883064, 0.92517835218518019, 0.6704867691060552, 
-0.38772856175015247], [0.70630642962287937, 0.75083474665259375, -0.58007380999238056, 0.32861016025252332], 
[0.46626814226680624, -0.26080126608156884, 0.60593135003753673, 0.79447219178264583], [0.9947497552782405, 
0.18091099360292073, 0.002417009868073805, -0.84787538373979832]], [[0.46536087225168687, -0.76242111351583874, 
0.17696941378154007, -0.2272041112987544], [-0.64291190012970034, 0.40540886005461063, 0.57834044247470495, 
-0.51513023272596814], [0.33875160864488896, 0.053537900746065459, -0.97789121819694103, -0.31839001003796308], 
[0.90451015416608382, 0.14006057723671006, 0.33076477793371484, -0.35621341661634132]], [[-0.18706058236051892, 
0.10534962562414951, -0.85512930094882611, -0.22092487910050851], [-0.68963062322171553, 0.34672602338724956, 
0.88231164204355439, 0.82587986073243691], [0.59658371137925847, -0.25100280279998399, -0.4156797927329452, 
0.74160746072202599], [0.25375585149442381, 0.082455322472048342, 0.61380072836664179, 0.97094819000414856]], 
[[-0.51848371562681406, 0.19230658480088403, -0.85606482748928658, -0.81551431711243016], [-0.85340906882410095, 
-0.25241348310142175, -0.38053681436777675, 0.453559740730612], [-0.65304340394900118, 0.40119072263213762, 
0.34194503185813296, 0.15434911881736113], [-0.62722487148231965, -0.10612589172419185, 0.79019234367232039, 
0.28179731834394639]], [[-0.91338738164476174, 0.66394092899841617, -0.65042331673436293, 0.001881776091624543], 
[0.49187239439599306, 0.57741523420410124, -0.024162401743606843, 0.44547230724260323], [-0.50793560911936986, 
0.82124530921413985, -0.15580279177447243, 0.94920612504163904], [-0.5449261787956563, 0.78555639247567122, 
0.24014002239550458, -0.87208230551553889]]], [[[0.57280466892714643, -0.89283174464912429, -0.29079928952467049, 
-0.026332428838778954], [0.73983816470901909, -0.66082037783045866, -0.21648800159804926, -0.29514135871249025], 
[0.090867827351879926, -0.68680062827433508, -0.49853372394466944, -0.7257933704395374], [-0.039090537649200119, 
0.74627470951361441, -0.7109612635939162, 0.49065787398200933]], [[0.16921951044842709, 0.99621796277731489, 
-0.071036078329213215, -0.86904091453527044], [0.8905983126345236, -0.86808410606985587, 0.93517478450327851, 
0.70929854255193825], [-0.93344253586587467, 0.33462185273918177, -0.24445745705547561, -0.23591799359266319], 
[-0.54451589271706746, 0.35028651694541324, -0.5781747904357859, 0.1235911380949124]], [[-0.014249440855339968, 
0.54760517459954716, 0.11958888563439118, -0.93021621326335979], [-0.31688983804772786, -0.40780280931316826, 
0.56753312262305822, 0.74445921416301797], [0.23805243842025181, -0.82131579077935357, -0.90118030936517424, 
0.075650207448711981], [0.28547070882280901, 0.81959695184978498, 0.85588359046751861, 0.44710198664328304]], 
[[-0.77604709337791133, -0.044625344416932888, 0.18637447766932347, 0.53726087651180632], [-0.93076844924371205, 
-0.52318486171007295, 0.0032663318214520842, 0.47981138228294284], [-0.36729332910185941, 0.35861010694012307, 
0.52599076082853413, -0.37256610605393226], [0.85135095878593336, 0.52940302941553052, -0.056908707508160905, 
0.95300194616884948]], [[0.12377972777765178, 0.2984938918653206, 0.066890197527302631, 0.10307838408106829], 
[0.89917057476298212, 0.74256370688037232, 0.14529475048858065, -0.94748801712822872], [-0.84392771413548529, 
0.60648570947595926, 0.0092068051589937472, 0.97958548072439444], [-0.92652047184606645, -0.43067167334389223, 
-0.39211284472905095, -0.50008075826440801]], [[-0.71587367657606737, -0.44709986372330679, 0.26826329499087476, 
0.26028136717314654], [0.92604831075227101, 0.72045216608304163, -0.56375459790845728, -0.8688908154050643], 
[-0.88203593850308892, -0.35847451081083737, -0.65222182401656226, 0.15929518134564402], [0.32617049436120604, 
-0.31729771517556205, 0.81021291969428577, 0.4807689085861242]]], [[[0.20149627332251452, -0.67808906172136507, 
-0.16570492668822334, 0.2916770413580354], [-0.019085382259784289, -0.69972933053753938, 0.035536618333866787, 
0.34367202459355206], [0.49035498810228795, -0.80065494254131564, -0.84361913131769528, -0.18894843298017028], 
[0.43482653966447793, -0.92401956496389825, -0.031261799768590892, 0.79686601046705396]], [[0.60883954337643331, 
-0.93841089431644664, -0.59425132671116354, 0.80362627513665053], [-0.50181336350809769, -0.52346255328774771, 
-0.73728970859557652, -0.42736788456687469], [-0.65250779334833386, 0.66301894305200371, 0.65248840924633944, 
-0.26894864942683228], [-0.40220322647538365, -0.83446613637513711, 0.16461375864007044, -0.3105925070889517]], 
[[0.58787788139913388, 0.56464986919110038, 0.28237701513531288, 0.35598293517493773], [0.67329128261335836, 
0.69601904507045043, -0.90937994831906965, -0.35013899138910221], [0.73043692182340991, -0.35687715747669535, 
0.43072474299059138, 0.98754152782018534], [-0.78281317239823811, 0.51518711069616052, -0.26912571493102044, 
-0.59948302455534241]], [[-0.41528905177058317, -0.86114138806005003, -0.26280537986202801, -0.086884036551261978], 
[0.40693143984254121, -0.40207004892488363, 0.38713514011029004, 0.96876022905329684], [0.70075226658416034, 
0.61046143599408009, 0.61337607761882196, -0.17650661251444388], [-0.92603422351480225, -0.97678283033102242, 
-0.36442775588730258, 0.23658955093879253]], [[-0.49943998537245005, -0.59012073857664427, -0.44180862503949703, 
0.17247090795886266], [-0.87350527798332478, -0.66121022236013216, -0.24952286719524897, 0.33523128980575589], 
[-0.34111376781909719, 0.77702975255933193, -0.25187485365484341, -0.7361963300403056], [0.28356981980043172, 
-0.48019588071448038, -0.23348932908146436, -0.95082687572463276]], [[0.11520387462638459, 0.2288492302153502, 
-0.73570897960943782, -0.028042770377183945], [-0.52291444094978612, 0.41390444769169421, 0.73042091257071018, 
0.50288722375595185], [0.030041805333981486, -0.30838278273121, -0.57362388107995854, 0.68141988597860426], 
[0.52077856493624042, -0.61431776760707701, 0.16545599139388067, 0.8866173503291388]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=2)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_4_offset3(self):
        sh0=(4, 1, 5, 1)
        sh1=(5, 1, 5, 1)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=3)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.75724102164472828], [0.94320628986675192], [-0.6753267670798071], [-0.71934741923350454], 
[-0.0070239184802021182]]], [[[-0.32687081216318425], [0.65628129576957694], [0.9000164714768486], [0.26879520002856783], 
[-0.42720528615830933]]], [[[0.33610510814778283], [0.91810813058125174], [0.28862086560446443], [0.40433224731209916], 
[-0.22658238128474784]]], [[[-0.97024980219144497], [-0.2695438095534306], [-0.17127874747585525], [-0.32037590715551345], 
[-0.22718610982552678]]]])
        yy=numpy.array([[[[0.96832352969655799], [0.49624209488788407], [-0.3615590445344834], [-0.40018847054524631], 
[0.41616119534979656]]], [[[-0.68505865441071867], [0.67849307458809394], [0.8541712442225784], [0.035477849492586921], 
[0.075058128083024922]]], [[[0.92918647653830178], [0.82403317671032372], [-0.25438802172701447], [-0.6432058650491872], 
[0.6660214505502946]]], [[[0.54790127223130525], [-0.57460792762864643], [-0.89352382549427745], [-0.58519661436516079], 
[0.69496534370265861]]], [[[0.75856254636488996], [0.057988202127798871], [0.62155707319731124], [0.8879511352982008], 
[0.98410437662287054]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=3)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")
    #++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_generalTensorTransposedProduct_Symbol_rank4_and_4_offset4(self):
        sh0=(6, 3, 2, 3)
        sh1=(6, 3, 2, 3)
        x=Symbol('x', sh0)
        y=Symbol('y', sh1)
        z=generalTensorTransposedProduct(x,y,axis_offset=4)
        self.assertTrue(isinstance(z, Symbol), "wrong type of result")
        xx=numpy.array([[[[-0.74746905122259544, -0.45721326421592701, -0.12289747638449589], [-0.23586754427724976, 
-0.65290774169155297, 0.027563259717484634]], [[-0.37411175843182431, -0.1086348336435099, -0.10767542819138054], 
[-0.68067820435609172, -0.32956496413991498, 0.29364797634740403]], [[0.82084986812707017, -0.28974538306507291, 
-0.8893346183676536], [0.069628506485047259, 0.15306516300161643, 0.98410908598730229]]], [[[0.58805514202788234, 
0.36093888020209342, -0.35685479901568651], [0.38481643025244683, 0.84201119286915316, 0.21541378255224863]], 
[[-0.1360050653897884, -0.2334932679549544, -0.95007895125203401], [0.93695505503756649, 0.36384871669869834, 
-0.18597927828693983]], [[0.17421651344789235, 0.54680895709781074, 0.83778490645035375], [0.26970224969778167, 
0.29669615222645396, -0.96617851668590315]]], [[[0.0089989644192671747, 0.44475299660490397, -0.97945111967867371], 
[-0.15526070784677826, 0.82356055071917833, 0.9989433724314789]], [[0.15527792892604531, -0.21220941449319319, 
-0.76730252498846774], [-0.62625453069600412, 0.61460551529224916, -0.49172282861638705]], [[0.1827796837241098, 
-0.47570964203352717, 0.35030256418421968], [0.60952941556234053, -0.70596863085221395, 0.41432408519857766]]], 
[[[0.14967681488714102, 0.50790863199666858, 0.43960898248735458], [-0.39827821092075699, 0.6594457415342061, 
0.61483641607611372]], [[0.80371749656134406, -0.61570166115882352, 0.44550339683107265], [0.90858696537278605, 
-0.61120065160184667, -0.67767869230298139]], [[0.58154771621100387, -0.77153062307252651, 0.7864368473491965], 
[0.0081077380853038505, -0.10671043708102124, 0.43726215778192468]]], [[[0.75934405643278358, -0.4426362870637377, 
-0.8550705625444972], [0.63423054083343855, -0.84038348994484191, -0.089261009024083293]], [[0.76301538265994417, 
0.54147899931611176, -0.40279309854614853], [-0.27914261851479072, 0.93510728149470124, -0.53879075592924153]], 
[[0.35317039473411094, 0.27349223842850989, 0.17857703639642208], [0.081066152127427404, 0.24921389283355189, 
-0.32119802724234936]]], [[[-0.27122459045457448, 0.12374096359932163, 0.098555754962582176], [-0.41596820105662746, 
0.019195579641665184, 0.88766200247615701]], [[-0.83795262558010708, -0.35902431857403538, 0.51345730353821883], 
[-0.83514518115206204, -0.050293757317133148, 0.78578698137881475]], [[-0.68454273915749053, 0.14642155672621482, 
0.99350341498309569], [0.54993022925419965, 0.44994652845753191, -0.21859353390765168]]]])
        yy=numpy.array([[[[-0.027491010789976178, -0.46450717348026904, -0.56221633513377278], [0.018047509620586766, 
-0.13978778846573747, 0.84769317334474503]], [[-0.38773225710633019, -0.4051051385214226, 0.11472180950676747], 
[-0.22999877174573613, -0.4004318538387095, -0.38205998827680365]], [[0.18687006242101889, 0.010102610487439767, 
0.47239089777629206], [0.16754750105283844, -0.68815586743167967, 0.89856770585050905]]], [[[0.83763041748893463, 
0.85915342313244136, 0.80941578481784382], [0.74659393978578636, 0.68923567652727313, 0.66040268877849617]], 
[[-0.54099319040756289, -0.3166214582464586, 0.68790674304137256], [0.29805321430393206, 0.92873162705326973, 
-0.12494166101095749]], [[-0.77634845956752629, -0.79385764800099823, 0.19894119492679474], [-0.41527985438100945, 
0.46472664673828001, -0.87513387752767247]]], [[[0.35009305519532541, 0.66536388688305537, -0.65277330141728229], 
[-0.70087733725881485, -0.10363777314165779, 0.19605187918315892]], [[0.3970077435147672, 0.54037146065828012, 
0.57720057101979583], [-0.031711054828638119, 0.21974315080451934, 0.22073841984475018]], [[-0.18167464311883297, 
-0.48777817474380858, -0.61734397769991634], [0.93995177703054766, -0.92238679364816178, -0.14109889727097147]]], 
[[[-0.74109365170371744, -0.20438639696204719, -0.54524217171285794], [-0.82186449473202283, 0.82526741102274404, 
-0.13921852027733683]], [[0.53462559970445134, 0.0118375707861198, 0.68296130074917283], [0.65145988032251934, 
0.49281651559526951, 0.67745813014233747]], [[-0.98402219464417273, -0.23822161905022132, -0.32845093900687217], 
[0.38744745401507985, -0.56655506015045454, -0.72702342950889642]]], [[[0.64759736699322579, 0.3394049397726302, 
-0.72906698130929248], [-0.35061850711660592, 0.78976021858860368, 0.39321608495113192]], [[0.63527036415371674, 
0.92707354698601074, 0.48464603845777332], [0.0081520902227205649, 0.66042039253764417, 0.80542392255619033]], 
[[-0.9741275861648373, -0.14243780450252874, -0.20086847285446008], [-0.81748109359340382, 0.98205034294027294, 
-0.36428791951398165]]], [[[-0.76780965929495415, 0.4584787327557398, -0.14477527278969582], [-0.7596436527894721, 
-0.943254522587885, -0.58376329469989141]], [[0.19726082181968718, 0.9663800711371362, -0.29288226838298037], 
[-0.65419045069840753, 0.26306901413394357, -0.24709953334901957]], [[-0.35924445035475161, 0.22967782741931453, 
-0.38766115548356583], [0.79128393747834136, -0.43814174730721001, -0.35365651552027688]]]])
        ref=generalTensorTransposedProduct(xx,yy,axis_offset=4)
        res=Evaluator(z)(x=xx,y=yy)
        self.assertAlmostEqual(Lsup(res-ref), 0.0, self.TOL_DIGITS, "wrong result")

    def test_subs(self):
        #tests for preservation of data substitution 
        u=Symbol('u',(2,))
        a=Symbol('a',(2,))
        dat=Data([1,0],FunctionSpace())
        dat2=Data([0,1],FunctionSpace())
        u=u.subs(u,dat)
        a=a.subs(a,dat2)
        a[0]=u[0]
        self.assertEqual(u._subs, u[0]._subs, "indexing did not preserve data substitution")
        self.assertTrue(list(u[0]._subs.keys())[0] in a[0]._subs, "indexing did not preserve data substitution")

if __name__ == '__main__':
    run_tests(__name__, exit_on_failure=True)

