########################################################################
#
#       License: BSD
#       Created: September 4, 2002
#       Author:  Francesc Altet - faltet@carabos.com
#
#       $Source: /home/ivan/_/programari/pytables/svn/cvs/pytables/pytables/tables/Group.py,v $
#       $Id: Group.py 1086 2005-07-18 17:08:57Z faltet $
#
########################################################################

"""Here is defined the Group class.

See Group class docstring for more info.

Classes:

    Group
    RootGroup
    TransactionGroupG
    TransactionG
    MarkG

Functions:


Misc variables:

    __version__

"""

import sys
import warnings

import tables.hdf5Extension as hdf5Extension
import tables.utilsExtension as utilsExtension
from tables.constants import MAX_GROUP_WIDTH
from tables.registry import classNameDict, classIdDict
from tables.exceptions import NodeError, NoSuchNodeError, PerformanceWarning
from tables.utils import checkNameValidity, joinPath
from tables.Node import Node, HiddenNodeMixin
from tables.Leaf import Leaf, Filters
from tables.UnImplemented import UnImplemented, OldIndexArray
from tables.AttributeSet import AttributeSet



__version__ = "$Revision: 1086 $"

obversion = "1.0"



class Group(hdf5Extension.Group, Node):
    """This is the python counterpart of a group in the HDF5 structure.

    It provides methods to set properties based on information
    extracted from the HDF5 files and to walk throughout the
    tree. Every group has parents and children, which are all Group
    instances, except for the root group whose parent is a File
    instance.

    In Group instances, a special feature called "natural naming" is
    used, i.e. the instance attributes that represent HDF5 groups are
    the same as the names of the children. This offers the user a very
    convenient way to access nodes in tree by simply naming all the
    path from the root group.

    For this reason and in order to not pollute the children
    namespace, it is explicitely forbidden to assign "normal"
    attributes to Group instances, and the only ones allowed must
    start by "_c_" (for class variables), "_f_" (for methods), "_g_"
    (for private methods) or "_v_" (for instance variables) prefixes.

    Instance variables (in addition to those in `Node`):

    _v_nchildren
        The number of children hanging from this group.
    _v_children
        Dictionary with all nodes hanging from this group.
    _v_groups
        Dictionary with all groups hanging from this group.
    _v_leaves
        Dictionary with all leaves hanging from this group.
    _v_hidden
        Dictionary with all hidden nodes hanging from this group.
    _v_filters
        Default filter properties for child nodes --see `Filters`.  A
        shorthand for ``FILTERS`` attribute.

    Public methods (in addition to those in `Node`):

    * __delattr__(name)
    * __getattr__(name)
    * __setattr__(name, object)
    * __iter__()
    * __contains__(name)
    * _f_listNodes(classname)
    * _f_walkGroups()
    * _f_walkNodes(classname, recursive)
    * _f_copyChildren(dstgroup[, overwrite][, recursive][, **kwargs])
    """

    # Class identifier.
    _c_classId = 'GROUP'


    # <undo-redo support>
    _c_canUndoCreate = True  # Can creation/copying be undone and redone?
    _c_canUndoRemove = True  # Can removal be undone and redone?
    _c_canUndoMove   = True  # Can movement/renaming be undone and redone?
    # </undo-redo support>


    # <properties>

    # '_v_filters' is a direct read-write shorthand for the 'FILTERS' attribute
    # with the default Filters instance as a default value.
    def _g_getfilters(self):
        filters = getattr(self._v_attrs, 'FILTERS', None)
        if filters is None:
            filters = Filters()
        return filters

    def _g_setfilters(self, value):
        if not isinstance(value, Filters):
            raise TypeError(
                "value is not an instance of `Filters`: %r" % (value,))
        self._v_attrs.FILTERS = value

    def _g_delfilters(self):
        del self._v_attrs.FILTERS

    _v_filters = property(_g_getfilters, _g_setfilters, _g_delfilters,
                          "Default filter properties for child nodes.")

    # </properties>


    def __init__(self, title = "", new = 1, filters=None):
        """Create the basic structures to keep group information.

        title -- The title for this group
        new -- If this group is new or has to be read from disk
        filters -- A Filters instance

        """
        mydict = self.__dict__
        mydict["_v_new"] = new
        mydict["_v_new_title"] = title
        mydict["_v_new_filters"] = filters
        mydict["_v_groups"] = {}
        mydict["_v_leaves"] = {}
        mydict["_v_children"] = {}
        mydict["_v_nchildren"] = 0
        mydict["_v_hidden"] = {}  # the only place for hidden children
        mydict["_v_version"] = obversion


    def __iter__(self, classname=None, recursive=0):
        """Iterate over the children on self"""

        return self._f_walkNodes(classname, recursive)


    def __contains__(self, name):
        """
        Is there a child with that `name`?

        Returns ``True`` if the group has a child node (visible or
        hidden) with the given `name` (a string), ``False`` otherwise.
        """
        return name in self._v_children or name in self._v_hidden


    def _f_walkNodes(self, classname=None, recursive=True):
        """Iterate over the nodes of self

        If "classname" is supplied, only instances of this class
        are returned. If "recursive" is false, only children
        hanging immediately after the group are returned. If
        true, a recursion over all the groups hanging from it is
        performed. """

        # For compatibility with old default arguments.
        if classname == '':
            classname = None

        if not recursive:
            # Non-recursive algorithm
            for leaf in self._f_listNodes(classname):
                yield leaf
        else:
            if classname == "Group":
                # Recursive algorithm
                for group in self._f_walkGroups():
                    yield group
            else:
                for group in self._f_walkGroups():
                    for leaf in group._f_listNodes(classname):
                        yield leaf

    # This iterative version of _g_openFile is due to John Nielsen
    def _g_openFile(self):
        """Recusively read an HDF5 file and generate its object tree."""

        stack=[self]
        while stack:
            objgroup=stack.pop()
            pgroupId=objgroup._v_parent._v_objectID
            locId=objgroup._v_objectID
            (groups, leaves)=self._g_listGroup(pgroupId, locId,
                                               objgroup._v_hdf5name)
            for name in groups:
                classId = objgroup._g_getGChildAttr(name, 'CLASS')
                groupClass = Group  # default group class
                if classId in classIdDict:
                    groupClass = classIdDict[classId]
                elif self._v_file._isPTFile:
                    # Warn only in PyTables files, where 'CLASS' has meaning.
                    warnings.warn("""\
group ``%s`` has an unknown class ID ``%s``; \
it will become a standard ``Group`` node"""
                                  % (objgroup._g_join(name), classId))
                new_objgroup = groupClass(new = False)
                new_objgroup._g_putUnder(objgroup, name)
                if classId in ["DINDEX", "TINDEX"]:
                    # This recursive call is to priorize the putting of
                    # indexes in the object tree
                    new_objgroup._g_openFile()
                else:
                    stack.append(new_objgroup)

            for name in leaves:
                leafClass = objgroup._g_getLeafClass(objgroup, name)
                objleaf = leafClass()
#                 print "*************************************************"
#                 print "Group.py: Remember commenting out the lines bw **"
#                 objleaf._g_putUnder(objgroup, name)
#                 continue
#                 print "*************************************************"
                # Try if object can be loaded
                try:
                    # ************* Important *********************
                    # Whenever an error occurs and you want to know
                    # where exactly it is happening, get this line
                    # out of the try clause
                    # *********************************************
                    objleaf._g_putUnder(objgroup, name)
                except:  #XXX
                    (typerr, value, traceback) = sys.exc_info()
                    warnings.warn("""\
problems loading leaf ``%s``: %s; \
it will become an ``UnImplemented`` node""" % (objgroup._g_join(name), value))
                    # If not, associate an UnImplemented object to it
                    objleaf = UnImplemented()
                    objleaf._g_putUnder(objgroup, name)
        # Hook to tidy-up things after the object tree is completely
        # built.
        for node in self._f_walkNodes(recursive=True):
            node._g_afterOpen()

    def _g_getLeafClass(self, parent, name):
        """Return a proper Leaf class depending on the object to be opened."""

#         if self._v_file._isPTFile:
#             # We can call this only if we are certain than file has
#             # the attribute CLASS
#             classId = self._v_attrs._g_getChildSysAttr(name, "CLASS")
#         else:
#             classId = self._v_attrs._g_getChildAttr(name, "CLASS")
        # The next call is faster than creating a _v_attr
        classId = utilsExtension.read_g_leaf_attr( \
            self._v_objectID, name, "CLASS")
        if classId is None:
            # No CLASS attribute, try a guess
            classId = utilsExtension.whichClass(self._v_objectID, name)
            if classId == "UNSUPPORTED":
                warnings.warn("""\
leaf ``%s`` is of an unsupported type; \
it will become an ``UnImplemented`` node""" % (parent._g_join(name),))
                return UnImplemented
        if classId in classIdDict:
            return classIdDict[classId]
        else:
            warnings.warn("""\
leaf ``%s`` has an unknown class ID ``%s``; \
it will become an ``UnImplemented`` node""" % (parent._g_join(name), classId))
            return UnImplemented


    def _g_join(self, name):
        """Helper method to correctly concatenate a name child object
        with the pathname of this group."""

        if name == "/":
            # This case can happen when doing copies
            return self._v_pathname
        return joinPath(self._v_pathname, name)


    def _g_widthWarning(self):
        """Issue a `PerformanceWarning` on too many children."""

        warnings.warn("""\
group ``%s`` is exceeding the recommended maximum number of children (%d);\
be ready to see PyTables asking for *lots* of memory and possibly slow I/O"""
                      % (self._v_pathname, MAX_GROUP_WIDTH),
                      PerformanceWarning)


    def _g_refNode(self, node, name, validate=True):
        """
        Insert references to a `node` via a `name`.

        Checks that the `name` is valid and does not exist, then creates
        references to the given `node` by that `name`.  The validation
        of the name can be omitted by setting `validate` to a false
        value (this may be useful for adding already existing nodes to
        the tree).
        """

        # Check for name validity.
        if validate:
            checkNameValidity(name)
            node._g_checkName(name)

        # Check if there is already a child with the same name.
        if name in self._v_children or name in self._v_hidden:
            raise NodeError(
                "group ``%s`` already has a child node named ``%s``"
                % (self._v_pathname, name))

        # Check group width limits.
        if self._v_nchildren + len(self._v_hidden) >= MAX_GROUP_WIDTH:
            self._g_widthWarning()

        # Insert references to the new child.
        mydict = self.__dict__
        mydict[name] = node

        if not isinstance(node, HiddenNodeMixin):
            self._v_children[name] = node
            if isinstance(node, Leaf):
                self._v_leaves[name] = node
            elif isinstance(node, Group):
                self._v_groups[name] = node
            mydict['_v_nchildren'] += 1
        else:
            self._v_hidden[name] = node


    def _g_unrefNode(self, name):
        """
        Remove references to a node.

        Removes all references to the named node.
        """

        mydict = self.__dict__
        del mydict[name]

        if name in self._v_children:
            del self._v_children[name]
            self._v_leaves.pop(name, None)
            self._v_groups.pop(name, None)
            mydict['_v_nchildren'] -= 1
        else:
            del self._v_hidden[name]


    def _g_putUnder(self, parent, name):
        # All this will eventually end up in the node constructor.

        super(Group, self)._g_putUnder(parent, name)
        # Update class variables
        if self._v_new:
            self._g_create()
        else:
            self._g_open()
        if self._v_new:
            setAttr = self._v_attrs._g__setattr
            # Set the title, class and version attribute
            setAttr('TITLE', self._v_new_title)
            setAttr('CLASS', self._c_classId)
            setAttr('VERSION', self._v_version)
            # Set the filters object
            if self._v_new_filters is None:
                # If not filters has been passed in the constructor,
                filters = self._v_parent._v_filters
            else:
                filters = self._v_new_filters
            setAttr('FILTERS', filters)
        else:
            # We don't need to get attributes on disk. The most importants
            # are defined as properties
            pass


    def _g_move(self, newParent, newName):
        oldPathname = self._v_pathname

        # Move the node to the new location.
        super(Group, self)._g_move(newParent, newName)

        # Update location information in children.
        myself = True
        for group in self._f_walkGroups():
            if myself:
                # Do not change location information again for this group.
                myself = False
            else:
                # Update location information for the descendent group.
                group._g_updateLocation()
            # Update location information for its child leaves.
            for leaf in group._f_listNodes('Leaf'):
                leaf._g_updateLocation()


    def _g_copy(self, newParent, newName, recursive, **kwargs):
        # Compute default arguments.
        title = kwargs.get('title', self._v_title)
        filters = kwargs.get('filters', self._v_filters)
        stats = kwargs.get('stats', None)

        # Fix arguments with explicit None values for backwards compatibility.
        if title is None:  title = self._v_title
        if filters is None:  filters = self._v_filters

        # Create a copy of the object.
        newNode = self._v_file.createGroup(
            newParent, newName, title, filters, _log = False)

        # Copy user attributes if needed.
        if kwargs.get('copyuserattrs', True):
            self._v_attrs._g_copy(newNode._v_attrs)

        # Update statistics if needed.
        if stats is not None:
            stats['groups'] += 1

        if recursive:
            # Copy child nodes if a recursive copy was requested.
            # Some arguments should *not* be passed to children copy ops.
            kwargs = kwargs.copy()
            kwargs.pop('title', None)
            self._g_copyChildren(newNode, **kwargs)

        return newNode


    def _g_copyChildren(self, newParent, **kwargs):
        """Copy child nodes.

        Copies all nodes descending from this one into the specified
        `newParent`.  If the new parent has a child node with the same
        name as one of the nodes in this group, the copy fails with a
        `NodeError`, maybe resulting in a partial copy.  Nothing is
        logged.
        """
        # Recursive version of children copy.
        ##for (srcChildName, srcChild) in self._v_children.iteritems():
        ##    srcChild._g_copy(newParent, srcChildName, True, **kwargs)

        # Non-recursive version of children copy.
        parentStack = [(self, newParent)]  # [(source, destination), ...]
        while parentStack:
            (srcParent, dstParent) = parentStack.pop()
            for (srcChildName, srcChild) in srcParent._v_children.iteritems():
                dstChild = srcChild._g_copy(
                    dstParent, srcChildName, False, **kwargs)
                if isinstance(srcChild, Group):
                    parentStack.append((srcChild, dstChild))


    def _g_open(self):
        """Call the openGroup method in super class to open the existing
        group on disk. """

        # All this will eventually end up in the node constructor.

        # Call the superclass method to open the existing group
        self.__dict__["_v_objectID"] = self._g_openGroup()

    def _g_create(self):
        """Call the createGroup method in super class to create the group on
        disk. Also set attributes for this group. """

        # All this will eventually end up in the node constructor.

        # Call the superclass method to create a new group
        self.__dict__["_v_objectID"] = \
                     self._g_createGroup()


    def _f_listNodes(self, classname=None):
        """
        Return a list with children nodes.

        The list is alphanumerically sorted by node name.  If the name
        of a class derived from `Node` is supplied in the `classname`
        parameter, only instances of that class (or subclasses of it)
        will be returned.  `IndexArray` objects are not allowed to be
        listed.
        """

        # For compatibility with old default arguments.
        if classname == '':
            classname = None

        if classname is None:
            # Returns all the children alphanumerically sorted
            names = self._v_children.keys()
            names.sort()
            return [ self._v_children[name] for name in names ]
        elif classname == 'Group':
            # Returns all the groups alphanumerically sorted
            names = self._v_groups.keys()
            names.sort()
            return [ self._v_groups[name] for name in names ]
        elif classname == 'Leaf':
            # Returns all the leaves alphanumerically sorted
            names = self._v_leaves.keys()
            names.sort()
            return [ self._v_leaves[name] for name in names ]
        elif classname == 'IndexArray':
            raise TypeError(
                "listing ``IndexArray`` nodes is not allowed")
        else:
            if classname not in classNameDict:
                raise TypeError(
                    "there is no registered node class named ``%s``"
                    % (classname,))
            class_ = classNameDict[classname]

            children = self._v_children
            childNames = children.keys()
            childNames.sort()

            nodelist = []
            for childName in childNames:
                childNode = children[childName]
                if isinstance(childNode, class_):
                    nodelist.append(childNode)

            return nodelist


    def _f_walkGroups(self):
        """Iterate over the Groups (not Leaves) hanging from self.

        The groups are returned ordered from top to bottom, and
        alphanumerically sorted when in the same level.

        """

        stack = [self]
        yield self
        # Iterate over the descendants
        while stack:
            objgroup=stack.pop()
            groupnames = objgroup._v_groups.keys()
            # Sort the groups before delivering. This uses the groups names
            # for groups in tree (in order to sort() can classify them).
            groupnames.sort()
            for groupname in groupnames:
                stack.append(objgroup._v_groups[groupname])
                yield objgroup._v_groups[groupname]

    def __delattr__(self, name):
        """
        Remove a child from the hierarchy.

        Removal via this method is *never* recursive because of the
        potential damage it may cause when used inadvertently.  If a
        recursive behavior is needed, use the ``_f_remove()`` method of
        the child node.  See `Node._f_remove()`.
        """

        if name not in self._v_children:
            raise NoSuchNodeError(
                "group ``%s`` does not have a child named ``%s``"
                % (self._v_pathname, name))

        try:
            node = self._v_children[name]
            node._f_remove()
        except NodeError:
            # This error message is clearer than the original one
            # for this operation.
            raise NodeError("""\
child group ``%s`` has child nodes; \
please use ``child._f_remove(True)`` to remove it"""
                            % (node._v_pathname,))

    def __getattr__(self, name):
        """Get the object named "name" hanging from me."""

        if not self._v_file.isopen:
            raise ValueError, "You are trying to access to a closed file handler. Giving up!."

        if name in self._v_children:
            return self._v_children[name]
        elif name in self._v_hidden:
            return self._v_hidden[name]
        else:
            raise NoSuchNodeError(
                "group ``%s`` does not have a child named ``%s``"
                % (self._v_pathname, name))


    def __setattr__(self, name, value):
        """Attach new nodes to the tree.

        name -- The name of the new node
        value -- The new node object

        If "name" group already exists in "self", raise the NodeError
        exception. A ValueError is raised when the "name" starts
        by a reserved prefix is not a valid Python identifier.
        A TypeError is raised when "value" is not a PyTables node
        (a Group or a Leaf).

        """

        # This method will be eventually assimilated into Node.__init__.

        # Check if the object is a PyTables node.
        # Only PyTables nodes can be assigned into a Group.
        if not isinstance(value, Node):
            raise TypeError("assigned object is not a node: %r" % (value,))

        # Put value object with name "name" in object tree
        value._g_putUnder(self, name)


    def _f_flush(self):
        """ Flush this Group """
        self._g_flushGroup()

    def _g_closeNodes(self):
        """Recursively close all nodes in `self` and their descendents.

        This version correctly handles both visible and hidden nodes.
        """

        stack = [self]
        # Iterate over the descendants
        while stack:
            objgroup=stack.pop()
            stack.extend(objgroup._v_groups.values())
            # Collect any hidden group
            for node in objgroup._v_hidden.values():
                if isinstance(node, Group):
                    stack.append(node)
                else:
                    # If it is not a group, close it
                    node._f_close()
            # Close the visible leaves
            for leaf in objgroup._v_leaves.values():
                leaf._f_close()
            # Close the current group only if it is not myself to avoid
            # recursivity in case of calling from '/' group
            if objgroup is not self:
                objgroup._f_close()

    def _f_close(self):
        """
        Close this node in the tree.

        This method has the behavior described in `Node._f_close()`.  It
        should be noted that this operation disables access to nodes
        descending from this group.  Therefore, if you want to
        explicitly close them, you will need to walk the nodes hanging
        from this group *before* closing it.
        """

        self._g_closeGroup()
        super(Group, self)._f_close()


    def _g_remove(self, recursive = False):
        """Remove (recursively if needed) the Group.

        This version correctly handles both visible and hidden nodes.
        """
        if self._v_nchildren > 0:
            if recursive:
                # First close all the children hanging from this group
                self._g_closeNodes()
            else:
                raise NodeError("""\
group ``%s`` has child nodes; \
please state recursive removal to remove it"""
                                % (self._v_pathname,))
        # Finally, close this group
        self._f_close()
        self._g_deleteGroup()


    def _f_copy(self, newparent=None, newname=None,
                overwrite=False, recursive=False, **kwargs):
        """
        Copy this node and return the new one.

        This method has the behavior described in `Node._f_copy()`.  In
        addition, it recognizes the following keyword arguments:

        `title`
            The new title for the destination.  If omitted or ``None``,
            the original title is used.  This only applies to the
            topmost node in recursive copies.
        `filters`
            Specifying this parameter overrides the original filter
            properties in the source node.  If specified, it must be an
            instance of the `Filters` class.  The default is to copy the
            filter properties from the source node.
        `copyuserattrs`
            You can prevent the user attributes from being copied by
            setting this parameter to ``False``.  The default is to copy
            them.
        `stats`
            This argument may be used to collect statistics on the copy
            process.  When used, it should be a dictionary whith keys
            ``'groups'``, ``'leaves'`` and ``'bytes'`` having a numeric
            value.  Their values will be incremented to reflect the
            number of groups, leaves and bytes, respectively, that have
            been copied during the operation.
        """
        return super(Group, self)._f_copy(
            newparent, newname, overwrite, recursive, **kwargs)


    def _f_copyChildren(self, dstgroup, overwrite=False, recursive=False,
                        **kwargs):
        """
        Copy the children of this group into another group.

        Children hanging directly from this group are copied into
        `dstgroup`, which can be a `Group` object or its pathname in
        string form.

        The operation will fail with a `NodeError` if there is a child
        node in the destination group with the same name as one of the
        copied children from this one, unless `overwrite` is true; in
        this case, the former child node is recursively removed before
        copying the later.

        By default, nodes descending from children groups of this node
        are not copied.  If the `recursive` argument is true, all
        descendant nodes of this node are recursively copied.

        Additional keyword arguments may be passed to customize the
        copying process.  For instance, title and filters may be
        changed, user attributes may be or may not be copied, data may
        be subsampled, stats may be collected, etc.  Arguments unknown
        to nodes are simply ignored.  Check the documentation for
        copying operations of nodes to see which options they support.
        """

        dstParent = self._v_file.getNode(dstgroup)  # Does new parent exist?
        self._g_checkGroup(dstParent)  # Is it a group?

        if not overwrite:
            # Abort as early as possible when destination nodes exist
            # and overwriting is not enabled.
            for childName in self._v_children:
                if childName in dstParent:
                    raise NodeError("""\
destination group ``%s`` already has a node named ``%s``; \
you may want to use the ``overwrite`` argument"""
                                    % (dstParent._v_pathname, childName))

        for child in self._v_children.itervalues():
            child._f_copy(dstParent, None, overwrite, recursive, **kwargs)


    def __str__(self):
        """The string representation for this object."""
        # Get the associated filename
        filename = self._v_file.filename
        # The pathname
        pathname = self._v_pathname
        # Get this class name
        classname = self.__class__.__name__
        # The title
        title = self._v_title
        return "%s (%s) %r" % (pathname, classname, title)

    def __repr__(self):
        """A detailed string representation for this object."""

        rep = [ '%r (%s)' %  \
                (childname, child.__class__.__name__)
                for (childname, child) in self._v_children.items() ]
        childlist = '[%s]' % (', '.join(rep))

        return "%s\n  children := %s" % \
               (str(self), childlist)


# Special definition for group root
class RootGroup(Group):
    def __init__(self, file, ptname, h5name, new = True):
        super(RootGroup, self).__init__(new = new)

        mydict = self.__dict__

        # Explicitly set location-dependent attributes.
        # Calling _g_setLocation is not needed for the root group.
        mydict["_v_file"] = file
        mydict["_v_name"] = ptname
        mydict["_v_hdf5name"] = h5name
        mydict["_v_parent"] = file
        mydict["_v_pathname"] = ptname   # Can be h5name? I don't think so
        mydict["_v_depth"] = 0   # root is level 0
        # The version
        mydict["_v_version"] = obversion
        file._refNode(self, self._v_pathname)

        # hdf5Extension operations (do before setting an AttributeSet):
        #   Update node attributes.
        self._g_new(file, h5name)
        #   Get HDF5 identifier.
        mydict["_v_objectID"] = self._g_openGroup()

        if new:
            # Set the title
            mydict["_v_title"] = file.title
            # Set the filters instance
            mydict["_v_filters"] = file.filters
            # Save the RootGroup attributes on disk
            setAttr = self._v_attrs._g__setattr
            setAttr('TITLE', file.title)
            setAttr('CLASS', self._c_classId)
            setAttr('VERSION', self._v_version)
            setAttr('FILTERS', file.filters)
        else:
            # Get all the groups recursively (build the object tree)
            self._g_openFile()

    def _f_close(self):
        # Close all nodes in tree (except myself)
        self._g_closeNodes()

        # Delete the filter properties if assigned.
        mydict = self.__dict__
        if 'v_filters' in mydict:
            del mydict['_v_filters']

        # Finally, close myself.
        super(RootGroup, self)._f_close()


    def _f_rename(self, newname):
        raise NodeError("the root node can not be renamed")

    def _f_move(self, newparent = None, newname = None):
        raise NodeError("the root node can not be moved")

    def _f_remove(self, recursive = False):
        raise NodeError("the root node can not be removed")


    def _g_setLocation(self, parent, name):
        # The root group does not need to get location-dependent information.
        pass



class IndexesDescG(Group, HiddenNodeMixin):
    _c_classId = 'DINDEX'

    def _g_widthWarning(self):
        warnings.warn("""\
the number of indexed columns on a single description group is exceeding
the recommended maximum (%d); be ready to see PyTables asking for *lots*
of memory and possibly slow I/O"""
                      % (MAX_GROUP_WIDTH,), PerformanceWarning)


class IndexesTableG(Group, HiddenNodeMixin):
    _c_classId = 'TINDEX'

    def _g_widthWarning(self):
        warnings.warn("""\
the number of indexed columns on a single table is exceeding the
recommended maximum (%d); be ready to see PyTables asking for *lots*
of memory and possibly slow I/O"""
                      % (MAX_GROUP_WIDTH,), PerformanceWarning)

    def _g_checkName(self, name):
        if not name.startswith('_i_'):
            raise ValueError(
                "names of index groups must start with ``_i_``: %s" % name)


class IndexesColumnBackCompatG(Group, HiddenNodeMixin):
    """This is meant to hidden indexes of pre-PyTables 1.0 files.
    """
    _c_classId = 'INDEX'



class TransactionGroupG(Group, HiddenNodeMixin):
    _c_classId = 'TRANSGROUP'

    def _g_widthWarning(self):
        warnings.warn("""\
the number of transactions is exceeding the recommended maximum (%d);\
be ready to see PyTables asking for *lots* of memory and possibly slow I/O"""
                      % (MAX_GROUP_WIDTH,), PerformanceWarning)



class TransactionG(Group):
    _c_classId = 'TRANSG'

    def _g_widthWarning(self):
        warnings.warn("""\
transaction ``%s`` is exceeding the recommended maximum number of marks (%d);\
be ready to see PyTables asking for *lots* of memory and possibly slow I/O"""
                      % (self._v_pathname, MAX_GROUP_WIDTH),
                      PerformanceWarning)



class MarkG(Group):
    # Class identifier.
    _c_classId = 'MARKG'


    import re
    _c_shadowNameRE = re.compile(r'^a[0-9]+$')


    def _g_widthWarning(self):
        warnings.warn("""\
mark ``%s`` is exceeding the recommended maximum action storage (%d nodes);\
be ready to see PyTables asking for *lots* of memory and possibly slow I/O"""
                      % (self._v_pathname, MAX_GROUP_WIDTH),
                      PerformanceWarning)


    def _g_reset(self):
        """
        Empty action storage (nodes and attributes).

        This method empties all action storage kept in this node: nodes
        and attributes.
        """

        # Remove action storage nodes.
        for child in self._v_children.values():
            child._g_remove(True)

        # Remove action storage attributes.
        attrs = self._v_attrs
        shname = self._c_shadowNameRE
        for attrname in attrs._v_attrnamesuser[:]:
            if shname.match(attrname):
                attrs._g__delattr(attrname)



## Local Variables:
## mode: python
## py-indent-offset: 4
## tab-width: 4
## fill-column: 72
## End:
