#!/usr/bin/env python
# arch-tag: 36fb07e7-bd95-49f9-94bd-d190b30c14e2
# Copyright (C) 2005 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for pybaz.Factory."""

import sys
from unittest import TestCase

import pybaz as arch
import framework
import fixtures


class FactoryArchive(framework.NewTestCase):
    fullname = 'jdoe@example.com--9999'

    def test_from_string(self):
        """arch.factory.Archive works with a string."""
        archive = arch.factory.Archive(self.fullname)
        self.assertEqual(type(archive), arch.Archive)
        self.assertEqual(archive.fullname, self.fullname)
        self.assert_(arch.factory.isArchive(archive))

    def test_from_archive_fails(self):
        """arch.factory.Archive raises TypeError with an Archive."""
        archive = arch.Archive(self.fullname)
        self.assertRaises(TypeError, arch.factory.Archive, archive)


class FactoryVersion(framework.NewTestCase):
    fullname = 'jdoe@example.com--9999/foo--bar--0'

    def test_from_string(self):
        """arch.factory.Version works with a string."""
        version = arch.factory.Version(self.fullname)
        self.assertEqual(type(version), arch.Version)
        self.assertEqual(version.fullname, self.fullname)
        self.assert_(arch.factory.isVersion(version))

    def test_from_version_fails(self):
        """arch.factory.Version raises TypeError with a Version."""
        version = arch.Version(self.fullname)
        self.assertRaises(TypeError, arch.factory.Version, version)


class FactoryRevision(framework.NewTestCase):
    fullname = 'jdoe@example.com--9999/foo--bar--0--patch-1'

    def test_from_string(self):
        """arch.factory.Revision works with a string."""
        rev = arch.factory.Revision(self.fullname)
        self.assertEqual(type(rev), arch.Revision)
        self.assertEqual(rev.fullname, self.fullname)
        self.assert_(arch.factory.isRevision(rev))

    def test_from_revision_fails(self):
        """arch.factory.Revision raises TypeError with a Revision."""
        rev = arch.Revision(self.fullname)
        self.assertRaises(TypeError, arch.factory.Revision, rev)


class FactorySourceTree(framework.NewTestCase):
    fixture = fixtures.WorkingTreeFixture()

    def testIsSourceTree(self):
        tree = self.fixture.tree
        workingtree = arch.WorkingTree(str(tree))
        arch.factory.isSourceTree(workingtree)
        sourcetree = arch.ArchSourceTree(str(tree))
        self.assertEqual(arch.factory.isSourceTree(sourcetree), True)


class FactoryArchiveLocation(framework.NewTestCase):

    def testArchiveLocation(self):
        url = 'http://alice@example.com/archive'
        location = arch.factory.ArchiveLocation(url)
        self.assertEqual(type(location), arch.ArchiveLocation)
        self.assertEqual(location.url, url)
        self.assertEqual(arch.factory.isArchiveLocation(location), True)


framework.register(__name__)
