/*
 * main.cxx
 *
 * PWLib application source file for Voxilla
 *
 * A H.323 "net telephone" application.
 *
 * Copyright (c) 1993-1998 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the assistance of funding from
 * Vovida Networks, Inc. http://www.vovida.com.
 *
 * Portions are Copyright (C) 1993 Free Software Foundation, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *      Derek J Smithies, Indranet Technologies Ltd (derek@indranet.co.nz)
 *          ______________________________________.
 *
 * $Log: main.cxx,v $
 * Revision 1.46  2003/02/26 02:49:07  robertj
 * Fixed race condition when makeing calls.
 * Fixed caller hanging up issue, thanks Pietro Ravasio
 *
 * Revision 1.45  2003/01/06 06:23:39  robertj
 * Increased maximum possible jitter configuration to 10 seconds.
 * Fixed ability to set jitter size (condition wrong way round).
 * Added abort ability and removed 100% CPU problem diring DTMF digit
 *   gathering and awaiting connect.
 *
 * Revision 1.44  2002/12/11 10:48:36  rogerh
 * delete listener when exiting
 * Add NAT support with the --translate option (copied from ohphone)
 *
 * Revision 1.43  2002/11/10 08:12:42  robertj
 * Moved constants for "well known" ports to better place (OPAL change).
 *
 * Revision 1.42  2002/10/31 00:55:13  robertj
 * Enhanced jitter buffer system so operates dynamically between minimum and
 *   maximum values. Altered API to assure app writers note the change!
 * Removed some redundent code.
 *
 * Revision 1.41  2002/07/02 03:24:09  dereks
 * Add LOTS of PTrace statements..  Improve check on LID ringing.
 *
 * Revision 1.40  2002/06/17 23:26:13  dereks
 * Add fast start disable (-F) and H245 Tunnelling disable (-T) options.
 *
 * Revision 1.39  2002/03/20 06:09:39  robertj
 * Added options for setting volume, thanks Jason Spence
 * Bug fix for collecting digits time out, thanks David Rowe
 *
 * Revision 1.38  2002/01/27 14:37:10  rogerh
 * Add --listenport option
 *
 * Revision 1.37  2001/11/19 06:42:34  robertj
 * Added ability to collect digits before making H.323 connection, thanks David Rowe.
 *
 * Revision 1.36  2001/06/20 07:36:57  robertj
 * Added dial plan parameter, thanks Chih-Wei Huang
 *
 * Revision 1.35  2001/06/19 22:48:55  robertj
 * Gatekeeper supprt, multiple aliases and prefixes, thanks Chih-Wei Huang
 *
 * Revision 1.34  2001/06/06 00:16:15  robertj
 * Removed code to extract e164 address and use OpenH323 function instead.
 *
 * Revision 1.33  2001/06/05 03:14:41  robertj
 * Upgraded H.225 ASN to v4 and H.245 ASN to v7.
 *
 * Revision 1.32  2001/03/20 23:42:55  robertj
 * Used the new PTrace::Initialise function for starting trace code.
 *
 * Revision 1.31  2001/01/20 10:23:27  craigs
 * Added another fix to ring timer thanks to Chih-Wei Huang
 *
 * Revision 1.30  2001/01/12 22:32:55  craigs 
 * Added fix to ring timer thanks to Chih-Wei Huang
 * 
 * Revision 1.29  2001/01/01 22:57:47  craigs
 * Added correct handling of incoming ring detection
 *
 * Revision 1.28  2000/11/28 01:40:37  robertj
 * Added country code to command line options.
 *
 * Revision 1.27  2000/10/19 03:39:17  dereks
 * Fixed PTRACE report
 *
 * Revision 1.26  2000/10/19 03:36:30  dereks
 * Added argument to set jitter buffer length in pstngw.
 * Use -j delay    (or --jitter delay) where delay is 20<= 150(def) <= 10 000 ms
 *
 * Revision 1.25  2000/10/16 08:50:09  robertj
 * Added single function to add all UserInput capability types.
 *
 * Revision 1.24  2000/09/23 07:20:46  robertj
 * Fixed problem with being able to distinguish between sw and hw codecs in LID channel.
 *
 * Revision 1.23  2000/09/22 01:35:53  robertj
 * Added support for handling LID's that only do symmetric codecs.
 *
 * Revision 1.22  2000/08/21 04:39:54  dereks
 * add parameter to set number of audio packets in an ethernet frame
 *
 * Revision 1.21  2000/06/22 17:40:21  craigs
 * Fixed problem with ringing causing multiple connections
 *
 * Revision 1.20  2000/06/19 00:32:22  robertj
 * Changed functionf or adding all lid capabilities to not assume it is to an endpoint.
 *
 * Revision 1.19  2000/05/30 10:19:28  robertj
 * Added function to add capabilities given a LID.
 * Improved LID capabilities so cannot create one that is not explicitly supported.
 *
 * Revision 1.18  2000/05/25 13:25:47  robertj
 * Fixed incorrect "save" parameter specification.
 *
 * Revision 1.17  2000/05/25 12:06:17  robertj
 * Added PConfigArgs class so can save program arguments to config files.
 *
 * Revision 1.16  2000/05/23 12:57:37  robertj
 * Added ability to change IP Type Of Service code from applications.
 *
 * Revision 1.15  2000/05/23 11:32:37  robertj
 * Rewrite of capability table to combine 2 structures into one and move functionality into that class
 *    allowing some normalisation of usage across several applications.
 * Changed H323Connection so gets a copy of capabilities instead of using endponts, allows adjustments
 *    to be done depending on the remote client application.
 *
 * Revision 1.14  2000/04/19 02:07:29  robertj
 * Fixed reuse of variable.
 *
 * Revision 1.13  2000/04/05 04:06:05  robertj
 * Added ability to change order of codecs.
 * Added transfer of caller ID to H.323
 *
 * Revision 1.12  2000/03/27 18:21:03  robertj
 * Added IP access control and more improvements in calling tone translation into H.323 signals.
 *
 * Revision 1.11  2000/03/24 03:41:45  robertj
 * Fixed some GNU C++ warnings.
 *
 * Revision 1.10  2000/03/23 23:33:16  robertj
 * Added more calling tone detection functionality.
 *
 * Revision 1.9  2000/03/23 03:03:46  craigs
 * Added interface option, and improved semantics of forced dialling
 *
 * Revision 1.8  2000/03/23 02:42:33  robertj
 * Added detection of PSTN hang up when making outgoing call
 *
 * Revision 1.7  2000/02/17 22:12:48  craigs
 * Changed error to warning when no line interface devices
 *
 * Revision 1.6  2000/02/17 06:27:30  robertj
 * Added #ifdef for IxJack support code.
 *
 * Revision 1.5  2000/02/16 03:30:01  robertj
 * Fixed some references to VPB code that should be enclosed in #ifdef.
 *
 * Revision 1.4  2000/01/07 08:28:09  robertj
 * Additions and changes to line interface device base class.
 *
 * Revision 1.3  2000/01/04 00:16:25  craigs
 * Changed for new AnswerCall timing
 *
 * Revision 1.2  1999/12/23 23:02:35  robertj
 * File reorganision for separating RTP from H.323 and creation of LID for VPB support.
 *
 * Revision 1.1  1999/12/22 09:49:36  craigs
 * Initial version
 *
 *
 */

#include <ptlib.h>

#include <h323pdu.h>
#include <gsmcodec.h>
#include <lid.h>

#ifdef HAS_IXJ
#include <ixjlid.h>
#ifndef __GNUC__
#pragma message("Including IxJ support.")
#endif
#endif

#ifdef HAS_VPB
#include <vpblid.h>
#ifndef __GNUC__
#pragma message("Including VPB support.")
#endif
#endif

#include "main.h"
#include "version.h"


#if !defined(HAS_IXJ) && !defined(HAS_VPB)
#warning Must specify a Line Interface Device! Use either HAS_IXJ or HAS_VPB.
#endif



PCREATE_PROCESS(PSTNGw);

#define new PNEW


#if 0

static PMutex logMutex;
static PTextFile logFile;
static PFilePath logFilename = DEFAULT_CALL_LOG;

static void LogMessage(const PString & str)
{
  PTime now;
  PString msg = now.AsString("hh:mm:ss dd/MM/yyyy") & str;
  logMutex.Wait();

  if (!logFile.IsOpen()) {
    logFile.Open(logFilename, PFile::ReadWrite);
    logFile.SetPosition(0, PFile::End);
  }

  logFile.WriteLine(msg);

  logFile.Close();
  
  logMutex.Signal();
}

static void LogCall(const PFilePath & fn,
                    const PString & from,
                    const PString & user,
                    unsigned len,
                    const PString & codec,
                    const PString & product)
{
  PString addr = from;
  LogMessage(addr & "\"" + user + "\"" & PString(PString::Unsigned, len) & codec & "\"" + product + "\"" & "\"" + fn + "\"");
}

#endif



///////////////////////////////////////////////////////////////

PSTNGw::PSTNGw()
  : PProcess("OpenH323 Project", "PSTNGw",
             MAJOR_VERSION, MINOR_VERSION, BUILD_TYPE, BUILD_NUMBER)
{
}


void PSTNGw::Main()
{
  cout << GetName()
       << " Version " << GetVersion(TRUE)
       << " by " << GetManufacturer()
       << " on " << GetOSClass() << ' ' << GetOSName()
       << " (" << GetOSVersion() << '-' << GetOSHardware() << ")\n\n";

  PConfigArgs args(GetArguments());
  args.Parse(
#ifdef HAS_IXJ
             "A-aec:"
#endif
             "c-calling-tone."
             "C-country:"
	     "-collect."
             "d-dial:"
             "D-disable:"
             "h-help."
             "i-interface:"
             "f-force-dial."
             "F-fast-disable."
             "j-jitter:"
             "-listenport:"
             "n-numframes:"
#if PTRACING
             "o-output:"
#endif
             "P-prefer:"
#ifdef HAS_IXJ
             "q-quicknet:"
#endif
             "-save."
#if PTRACING
             "t-trace."
#endif
             "T-h245tunneldisable."
             "-tos:"
             "-translate:"
             "u-user:"
             "g-gatekeeper:"   
             "G-gatekeeper-id:"
             "-no-gatekeeper."
             "r-require-gatekeeper."
	     "S-disable-h245-in-setup."
             "s-prefix:"
#ifdef HAS_VPB
             "v-vpb:"
#endif
	     "-recordgain:"
	     "-playgain:"
          , FALSE);

#if PTRACING
  PTrace::Initialise(args.GetOptionCount('t'),
                     args.HasOption('o') ? (const char *)args.GetOptionString('o') : NULL,
		     PTrace::Blocks | PTrace::Timestamp | PTrace::Thread | PTrace::FileAndLine);
#endif

  if (args.HasOption('h')) {
    cout << "Usage : " << GetName() << " [options] [ line1addr ... ]\n"
            "Options:\n"
#ifdef HAS_IXJ
            "  -q --quicknet dev  : Use Quicknet LineJACK\n"
#endif
#ifdef HAS_VPB
            "  -v --vpb dev       : Use Voicetronix VPB4\n"
#endif
            "     --playgain vol  : Set VPB4 playback gain\n"
            "     --recordgain vol: Set VPB4 record gain\n"
            "  -u --user str      : Set local endpoint name\n"
            "  -n --numframes n   : Number of audio frames per ethernet packet\n"

            "  -i --interface ip  : Use specific interface\n"
            "  --listenport       : Port to listen on for incoming connections (default 1720)\n"
            "     --tos n         : Set IP Type of Service byte to n\n"
            "  -j --jitter delay  : Set jitter buffer to delay milliseconds\n"
            "  -d --dial phone    : Default number to call\n"
            "     --collect       : Collect digits before making call\n"
            "  -F --fast-disable  : Disable fast start\n"
            "  -f --force-dial    : Force dial number in call\n"
            "  -c --calling-tone  : Enable PSTN calling tones translation\n"
            "  -C --country code  : Set the country settings for PSTN card\n"
            "  -D --disable codec : Disable the specified codec (may be used multiple times)\n"
            "  -P --prefer codec  : Prefer the specified codec (may be used multiple times)\n"
	    "  -S --disable-h245-in-setup : Disable H245 in setup\n"
            "  -T --h245tunneldisable     : Disable H245 tunnelling\n"
            "  --translate up             : Set external IP address to up is masQueraded\n"
#ifdef HAS_IXJ
            "  -A --aec num       : Set AEC (default to medium)\n"
#endif
#if PTRACING
            "  -t --trace         : Enable trace, use multiple times for more detail\n"
            "  -o --output        : File for trace output, default is stderr\n"
#endif
            "\nGatekeeper options:\n"
            "  -g --gatekeeper host       : Specify gatekeeper host\n"
            "  -G --gatekeeper-id name    : Specify gatekeeper by ID\n"
            "     --no-gatekeeper         : Disable gatekeeper discovery\n"
            "  -r --require-gatekeeper    : Exit if gatekeeper discovery fails\n"
            "  -s --prefix str            : Set gateway supported prefix\n"
            "\n"
            "     --save          : Save arguments in configuration file\n"
            "  -h --help          : Display this help message\n";
    return;
  }

  args.Save("save");

  GatewayEndPoint endpoint;
  if (endpoint.Initialise(args)) {
    cout << "Press enter to terminate." << endl;
    cin.get();
  }
}


///////////////////////////////////////////////////////////////

GatewayEndPoint::GatewayEndPoint()
{
  terminalType = e_GatewayOnly;
}


GatewayEndPoint::~GatewayEndPoint()
{
  ClearAllCalls();
}


BOOL GatewayEndPoint::Initialise(PArgList & args)
{
  PINDEX i;

  autoDial         = args.GetOptionString('d');
  forceDial        = args.HasOption('f');
  convertTones     = args.HasOption('c');
  disableFastStart = args.HasOption('f');
  disableH245Tunneling = args.HasOption('T');
  disableH245inSetup = args.HasOption('S');

  if (args.HasOption('j')) {
    unsigned minJitter;
    unsigned maxJitter;
    PStringArray delays = args.GetOptionString('j').Tokenise(",-");
    if (delays.GetSize() == 2) {
      minJitter = delays[0].AsUnsigned();
      maxJitter = delays[1].AsUnsigned();
    }
    else {
      maxJitter = delays[0].AsUnsigned();
      minJitter = PMIN(GetMinAudioJitterDelay(), maxJitter);
    }
    if (minJitter >= 20 && minJitter <= maxJitter && maxJitter <= 10000)
      SetAudioJitterDelay(minJitter, maxJitter);
    else {
      cerr << "Jitter should be between 20 milliseconds and 10 seconds.\n"
              "Using default value of " << GetMinAudioJitterDelay() <<
              " to " << GetMaxAudioJitterDelay() << " ms." << endl;
    }
  }
  PTRACE(3, "GatewayEndPoint\t Jitter buffer is "
         << GetMinAudioJitterDelay() << " to " << GetMaxAudioJitterDelay() << " ms");


  if (args.HasOption("tos"))
    SetRtpIpTypeofService(args.GetOptionString("tos").AsUnsigned());

  if (forceDial && autoDial.IsEmpty()) {
    cout << "Must specify autodial number if forcedial option used" << endl;
    return FALSE;
  }

  // get local username
  if (args.HasOption('u')) {
    PStringArray aliases = args.GetOptionString('u').Lines();
    SetLocalUserName(aliases[0]);
    for (i = 1; i < aliases.GetSize(); i++)
      AddAliasName(aliases[i]);
  } else
    SetLocalUserName("OpenH323 PSTN Gateway");

  accessControl.LoadHostsAccess();

  H323ListenerTCP * listener;

  WORD listenPort = H323EndPoint::DefaultTcpPort;
  if (args.HasOption("listenport"))
    listenPort = (WORD)args.GetOptionString("listenport").AsInteger();

  if (args.GetOptionString('i').IsEmpty()) {
    PIPSocket::Address interfaceAddress(INADDR_ANY);
    listener  = new H323ListenerTCP(*this, interfaceAddress, listenPort);
  } else {
    PIPSocket::Address interfaceAddress(args.GetOptionString('i'));
    listener  = new H323ListenerTCP(*this, interfaceAddress, listenPort);
  }

  if (!StartListener(listener)) {
    cout <<  "Could not open H.323 listener port on "
         << listener->GetListenerPort() << endl;
    delete listener;
    return FALSE;
  }

  cout << "Listening on port " << listener->GetListenerPort() << endl;

  if (args.HasOption("translate")) {
    masqAddressPtr = new PIPSocket::Address(args.GetOptionString("translate"));
    behind_masq = TRUE;
    cout << "Masquerading as address " << *(masqAddressPtr) << endl;
  } else {
    behind_masq = FALSE;
  }


#ifdef HAS_IXJ
  BOOL usingIXJ = FALSE;
  if (args.HasOption('q')) {
    PStringArray devices = args.GetOptionString('q').Lines();
    for (i = 0; i < devices.GetSize(); i++) {
      OpalIxJDevice * lid = new OpalIxJDevice;
      if (lid->Open(devices[i])) {
        lids.Append(lid);
        usingIXJ = TRUE;
      }
      else {
        cout << "error: cannot open ixj device " << devices[i] << endl;
        return FALSE;
      }
      lid->EnableAudio(1, TRUE);
    }
  }
#endif

#ifdef HAS_VPB
  if (args.HasOption('v')) {
    PStringArray devices = args.GetOptionString('v').Lines();
    for (i = 0; i < devices.GetSize(); i++) {
      OpalVpbDevice * lid = new OpalVpbDevice;
      if (lid->Open(devices[i]))
        lids.Append(lid);
      else {
        cout << "error: cannot open VPB device " << devices[i] << endl;
        return FALSE;
      }
    }
  }
#endif

  OpalLineInterfaceDevice::AECLevels aecLevel;
  if (args.HasOption("aec"))
    aecLevel = (OpalLineInterfaceDevice::AECLevels)args.GetOptionString("aec").AsUnsigned();
  else
    aecLevel = OpalLineInterfaceDevice::AECMedium;
  PString country = args.GetOptionString('C');

  BOOL noLinesAvailable = TRUE;
  PINDEX argCount = 0;

  for (i = 0; i < lids.GetSize(); i++) {
    for (unsigned l = 0; l < lids[i].GetLineCount(); l++) {
      if (lids[i].IsLineTerminal(l))
        continue;

      lids[i].SetLineOnHook(l);
      PTRACE(2, "ep\tInitialise SetLineOnHook true");
      //if (!lids[i].IsLinePresent(l))
      //  cout << "No line on " << lids[i] << '-' << l << endl;
      //else {
        noLinesAvailable = FALSE;
	if (args.HasOption("playgain")) {
	    lids[i].SetPlayVolume(l, args.GetOptionString("playgain").AsUnsigned());
	}
	if (args.HasOption("recordgain")) {
	  lids[i].SetRecordVolume(l, args.GetOptionString("recordgain").AsUnsigned());
	}
        lids[i].SetAEC(l, aecLevel);
        if (!country)
          lids[i].SetCountryCodeName(country);

        PString h323Address;

        cout << "Using line " << lids[i] << '-' << l << ", Incoming calls ";
        if (argCount < args.GetCount()) {
          h323Address = args[argCount++];
          cout << "redirected to \"" << h323Address << '"';
        }
        else
          cout  << "ignored";
        cout << endl;

        H323_LIDCapability::AddAllCapabilities(lids[i], capabilities, 0, 0);
        lineMonitors.Append(new LineMonitor(*this, lids[i], l, h323Address, 
					    args.HasOption("collect")));
      //}
    }
  }

  if (noLinesAvailable) {
    cout << "No usable lines available\n";
    return FALSE;
  }

  SetCapability(0, 0, new H323_GSM0610Capability);
  SetCapability(0, 0, new H323_G711Capability(H323_G711Capability::muLaw));
  SetCapability(0, 0, new H323_G711Capability(H323_G711Capability::ALaw));

  H323_UserInputCapability::AddAllCapabilities(capabilities, 0, 1);

  capabilities.Remove(args.GetOptionString('D').Lines());
  capabilities.Reorder(args.GetOptionString('P').Lines());

  if (!args.GetOptionString('n').IsEmpty()) {
    int txFrames = args.GetOptionString('n').AsInteger();
    txFrames     = PMAX(1, PMIN(20,txFrames));
    cout <<"Audio frames of data per packet : "<<txFrames<<endl;
    for (i = 0; i < capabilities.GetSize(); i++) 
       capabilities[i].SetTxFramesInPacket(txFrames);
  }

  // Gatekeeper registration
  if (args.HasOption('s'))
    SupportedPrefix = args.GetOptionString('s').Lines();

  H323TransportUDP * rasChannel;
  if (args.GetOptionString('i').IsEmpty())
    rasChannel  = new H323TransportUDP(*this);
  else {
    PIPSocket::Address interfaceAddress(args.GetOptionString('i'));
    rasChannel  = new H323TransportUDP(*this, interfaceAddress);
  }   
  if (args.HasOption('g')) {
    PString gkName = args.GetOptionString('g');
    if (SetGatekeeper(gkName, rasChannel))
      cout << "Gatekeeper set: " << *gatekeeper << endl;
    else {
      cerr << "Error registering with gatekeeper at \"" << gkName << '"' << endl;
      return FALSE;
    }
  } else if (args.HasOption('G')) {
    PString gkIdentifier = args.GetOptionString('G');
    cout << "Searching for gatekeeper with id \"" << gkIdentifier << "\" ..." << endl;
    if (LocateGatekeeper(gkIdentifier, rasChannel))
      cout << "Gatekeeper set: " << *gatekeeper << endl;
    else {
      cerr << "Error registering with gatekeeper at \"" << gkIdentifier << '"' << endl;
      return FALSE;
    }
  } else if (!args.HasOption("no-gatekeeper") || args.HasOption('r')) {
    cout << "Searching for gatekeeper..." << endl;
    if (DiscoverGatekeeper(rasChannel))
      cout << "Gatekeeper found: " << *gatekeeper << endl;
    else {
      cerr << "No gatekeeper found." << endl;
      if (args.HasOption('r')) 
        return FALSE;
    }
  } else {
    // Since we didn't call any gatekeeper registration function,
    // we have to delete the unused rasChannel ourself.
    delete rasChannel;
  }

  cout << "Codecs (in preference order):\n" << setprecision(2) << capabilities
       << "\nWaiting for incoming calls for gateway \"" << GetLocalUserName() << '"' << endl;

  return TRUE;
}

void GatewayEndPoint::SetEndpointTypeInfo( H225_EndpointType & info ) const
{
  H323EndPoint::SetEndpointTypeInfo(info);
  info.m_gateway.IncludeOptionalField(H225_GatewayInfo::e_protocol);
  info.m_gateway.m_protocol.SetSize(1);
  H225_SupportedProtocols &protocol=info.m_gateway.m_protocol[0];
  protocol.SetTag(H225_SupportedProtocols::e_voice);
  PINDEX as=SupportedPrefix.GetSize();
  ((H225_VoiceCaps &)protocol).m_supportedPrefixes.SetSize(as);
  for (PINDEX p=0; p<as; p++)
    H323SetAliasAddress(SupportedPrefix[p], ((H225_VoiceCaps &)protocol).m_supportedPrefixes[p].m_prefix);
}

//
// Rewrite E164 number according to the dial plan
// Format:
//  prefix_match=strip,prefix[,suffix]
//
//         strip    number of digits to strip
//         prefix   prefix to insert
//         suffix   suffix to append
//
// Example:
//  09=1,886      0953378875 --> 8869378875
//  07=2,,0       07123      --> 1230
//
PString GatewayEndPoint::RewriteE164(const PString &dialedDigits)
{
  static PStringToString DialPlan(PConfig("DialPlan").GetAllKeyValues());
  return RewriteE164(dialedDigits, DialPlan);
}

PString GatewayEndPoint::RewriteE164(const PString &dialedDigits, const PStringToString &myplan)
{
  for (PINDEX i=dialedDigits.GetLength(); i>0; i--) {
    PString key = dialedDigits.Left(i);
    if (myplan.Contains(key)) {
      PStringArray rule = myplan[key].Tokenise(",");
      PString result = rule[1] + dialedDigits.Mid(rule[0].AsInteger());
      if (!rule[2])
	result += rule[2];
      PTRACE(1, "Prefix " << key << " matched, rewrite " << dialedDigits << " to " << result << endl);
      return result;
    }
  }
  // no match
  return dialedDigits;
}

H323Connection * GatewayEndPoint::CreateConnection(unsigned callReference)
{
 return new GatewayConnection(*this, callReference);
}


BOOL GatewayEndPoint::OnIncomingCall(H323Connection & connection,
                                    const H323SignalPDU & setupPDU,
                                    H323SignalPDU &)
{
  H323TransportAddress address = connection.GetControlChannel().GetRemoteAddress();
  PIPSocket::Address ip;
  WORD port;
  if (address.GetIpAndPort(ip, port) && !accessControl.IsAllowed(ip)) {
    cout << "Incoming H.323 call from " << ip << " is denied." << endl;
    PTRACE(2, "ep\tIncoming H.323 call from " << ip << " is denied.");
    return FALSE;
  }

  GatewayConnection & myConnection = (GatewayConnection &)connection;

  // Locate a free line
  PINDEX i;
  for (i = 0; i < lineMonitors.GetSize(); i++) {
    if (lineMonitors[i].IsAvailable()) {
      myConnection.line = &lineMonitors[i];
      break;
    }
  }

  if (myConnection.line == NULL) {
    cout << "No available lines for incoming H.323 call." << endl;
    PTRACE(2, "ep\tOnIncomingCall::No available lines for incoming H.323 call.");
    return FALSE;
  }

  myConnection.line->AttachConnection(connection);

  OpalLineInterfaceDevice & device = myConnection.line->GetDevice();
  unsigned line = myConnection.line->GetLineNumber();

  // Sieze the line
  device.SetLineOffHook(line);
  cout << "Setting line " << device << '-' << line << " off hook" << endl;
  PTRACE(2, "ep\tSetting line " << device << '-' << line << " off hook");

  PString number;

  if (forceDial || !setupPDU.GetDestinationE164(number))
    number = autoDial;

  if (!number) {
    // Wait for dial tone
    if (device.WaitForTone(line, OpalLineInterfaceDevice::DialTone, 5000)) {
      PTRACE(2, "ep\t Dial tone deteted.");
      cout << "Dial tone detected." << endl;
    } else {
      PTRACE(2, "ep\tNO Dial tone deteted.");
      cout << "No dial tone detected." << endl;
    }

    PString dialed(RewriteE164(number));
    PTRACE(2, "ep\tDialing: " << dialed);
    cout << "Dialling: " << dialed << endl;
    device.PlayDTMF(line, dialed);
  }

  return TRUE;
}

void GatewayEndPoint::TranslateTCPAddress(PIPSocket::Address &localAddr, const PIPSocket::Address &remoteAddr)
{

  if (this->behind_masq) {
    /* Check if the remote address is a private IP address.
     * RFC 1918 specifies the following private IP addresses
     * 10.0.0.0    - 10.255.255.255.255
     * 172.16.0.0  - 172.31.255.255
     * 192.168.0.0 - 192.168.255.255
     */

     BOOL remote_address_private =
       ( ((remoteAddr.Byte1() == 10))
       ||((remoteAddr.Byte1() == 172)
           && (remoteAddr.Byte2() >= 16)
           && (remoteAddr.Byte2() <= 31) )
       ||((remoteAddr.Byte1() == 192)
           && (remoteAddr.Byte2() == 168)) );
     /*
      * We are using NAT and the remote address is outside our LAN, so
      * replace the local address with the IP address of the NAT box.
      */
     if (!remote_address_private) {
       localAddr = *(this->masqAddressPtr);
     }
  }
  return;
}

H323Connection::AnswerCallResponse GatewayEndPoint::OnAnswerCall(H323Connection & connection,
                                                                 const PString & /*callerName*/,
                                                                 const H323SignalPDU & /*setupPDU*/,
                                                                 H323SignalPDU & /*connectPDU*/)
{
  if (!convertTones)
    return H323Connection::AnswerCallNow;

  GatewayConnection & myConnection = (GatewayConnection &)connection;
  OpalLineInterfaceDevice & device = myConnection.line->GetDevice();
  unsigned line = myConnection.line->GetLineNumber();

  switch (device.WaitForToneDetect(line, 3000)) {
    case OpalLineInterfaceDevice::RingTone :
      PTRACE(2, "ep\tRing tone deteted.");
      cout << "Ring tone detected." << endl;
      return H323Connection::AnswerCallNow; // Temporary code, not completed yet
//      return H323Connection::AnswerCallPending;

    case OpalLineInterfaceDevice::BusyTone :
      PTRACE(2, "ep\tBusy tone deteted, call is refused.");
      cout << "Busy tone detected, call being refused." << endl;
      break;

    default :
      PTRACE(2, "ep\tNo ring/busy tone deteted.");
      cout << "No ring/busy tone detected." << endl;
  }

  return H323Connection::AnswerCallDenied;
}


BOOL GatewayEndPoint::OpenAudioChannel(H323Connection & connection,
                                      BOOL /*isEncoding*/,
                                      unsigned,
                                      H323AudioCodec & codec)
{
  GatewayConnection & myConnection = (GatewayConnection &)connection;
  if (myConnection.line == NULL)
    return FALSE;

  codec.SetSilenceDetectionMode(H323AudioCodec::NoSilenceDetection);

  return codec.AttachChannel(new OpalLineChannel(myConnection.line->GetDevice(),
                                                 myConnection.line->GetLineNumber(),
                                                 codec));
}


void GatewayEndPoint::OnConnectionCleared(H323Connection & connection,
                                         const PString & /*token*/)
{
  GatewayConnection & myConnection = (GatewayConnection &)connection;
  if (myConnection.line == NULL)
    return;

  OpalLineInterfaceDevice & device = myConnection.line->GetDevice();
  unsigned lineNumber = myConnection.line->GetLineNumber();

  PTRACE(2, "ep\tConnection Cleared. Set line " << device << '-' << lineNumber << " on hook.");
  cout << "Setting line " << device << '-' << lineNumber << " on hook" << endl;
  device.SetLineOnHook(lineNumber);
}


///////////////////////////////////////////////////////////////

GatewayConnection::GatewayConnection(GatewayEndPoint & _ep, unsigned callReference)
  : H323Connection(_ep, callReference)
{
  line = NULL;
}


void GatewayConnection::OnUserInputString(const PString & value)
{
  if (line == NULL)
    return;

  OpalLineInterfaceDevice & device = line->GetDevice();
  unsigned lineNumber = line->GetLineNumber();

  cout << "Line " << device << '-' << lineNumber << " user input: " << value << endl;
  device.PlayDTMF(lineNumber, value);
}


BOOL GatewayConnection::OnStartLogicalChannel(H323Channel & channel)
{
  PStringStream message;
  message << "Line " << line->GetDevice() << '-' << line->GetLineNumber()
       << " started logical channel: ";

  switch (channel.GetDirection()) {
    case H323Channel::IsTransmitter :
      message << "sending ";
      break;

    case H323Channel::IsReceiver :
      message << "receiving ";
      break;

    default :
      break;
  }

  message << channel.GetCapability();
  cout << message << endl;
  PTRACE(2, "Connection\t" << message);

  return TRUE;
}


///////////////////////////////////////////////////////////////

LineMonitor::LineMonitor(GatewayEndPoint & ep,
                         OpalLineInterfaceDevice & dev,
                         unsigned line,
                         const PString & addr,
			 BOOL collectNum)
  : PThread(10000, NoAutoDeleteThread),
    endpoint(ep),
    device(dev),
    lineNumber(line),
    h323Address(addr),
    collectNumber(collectNum)
{
  Resume();
}


LineMonitor::~LineMonitor()
{
  shutdown.Signal();
  WaitForTermination();
}

static BOOL IsLIDRinging(OpalLineInterfaceDevice & lid, PINDEX line, PTimer & timer)
{
  if (timer.IsRunning())
    return TRUE;

  if (lid.IsLineOffHook(line))    // The Phone cannot be ringing, cause it is off hook.
    return FALSE;

  if (!lid.IsLineRinging(line))
    return FALSE;

  timer = 3000;

  return TRUE;
}


void LineMonitor::Main()
{
  PTimer ringTimer;

  PTRACE(2, "PSTNGW\tLine monitor thread for " << device << '-' << lineNumber << " started");

  while (!shutdown.Wait(500)) {

    // if the line is ringing, start a new call
    PTRACE(3, "PSTNGW\tLine Monitor, start of loop");
    if (IsLIDRinging(device, lineNumber, ringTimer)) {
      PTRACE(3, "PSTNGW\tLine Monitor, detected line is ringing");
      cout << "Line " << device << '-' << lineNumber << " is ringing, ";

      // ignore incoming calls when no IP address to dial
      if (h323Address.IsEmpty()) {
        cout << "ignoring";
        PString callerID;
        if (device.GetCallerID(lineNumber, callerID))
          cout << " caller " << callerID;
        cout << "  - waiting for ringing to stop." << endl;
        while (!shutdown.Wait(500) && IsLIDRinging(device, lineNumber, ringTimer))
          ;
        cout << "Ringing stopped on device " << device << "-" << lineNumber << endl;
	PTRACE(3, "PSTNGW\tRinging stopped on device " << device << "-" << lineNumber);
      }
      else {
	PTRACE(3, "PSTNGW\tLine Monitor. Call address " << h323Address);
        cout << "calling \"" << h323Address << "\"." << endl;

        // if still onhook, wait for the ringing to stop
	PStringStream message;
        if (HandleCall()) {
	  message << "Call from " << device << '-' << lineNumber << " to " << h323Address << " ended.";
	  PTRACE(3, "PSTNGW\tLine Monitor:" << message);
          cout << message << endl;
        } else {
	  message << "Call from " << device <<  '-' << lineNumber << " to " << h323Address << " failed.";
	  PTRACE(3, "PSTNGW\tLine Monitor: " << message);
	  cout << message << endl;

          while (!shutdown.Wait(500) && IsLIDRinging(device, lineNumber, ringTimer))
            ;
          cout << "Ringing stopped on device " << device << "-" << lineNumber << endl;
	  PTRACE(3, "PSTNGW\tLine Monitor, Ringing stopped on device" << device << '-' << lineNumber);
        }
      }
    }

    // if a call is active, but the line has been disconnected, hangup the call
    if (!callToken && device.IsLineDisconnected(lineNumber)) {
      cout << "Line " << device << '-' << lineNumber << " disconnected." << endl;
      PTRACE(3, "PSTNGW\tLine Monitor, Line " << device << '-' << lineNumber << " disconnected");
      endpoint.ClearCall(callToken);
      callToken = PString();
    }
  }

  PTRACE(2, "PSTNGW\tLine monitor thread for " << device << '-' << lineNumber << " stopped");
}


BOOL LineMonitor::HandleCall()
{
  PString   callerID;
  PString   exth323Address;
  const int max_digits = 10;
  char      number[max_digits];
  
  *number = 0;

  if (device.GetCallerID(lineNumber, callerID))
    cout << "Caller is " << callerID << endl;

  if (collectNumber) {
    // DR - collect number to dial before we make H323 call
    // - take lid port off-hook
    // - generate dial tone from lid
    // - collect digits until # pressed or 10 second timeout
    // - every new digit resets timer for another 10 seconds

    device.SetLineOffHook(lineNumber);
    device.PlayTone(lineNumber, OpalLineInterfaceDevice::DialTone);

    char      digit;
    int       num_digits = 0;

    PTimer collectTimer(10000);

    cout << "Enter number to dial at other end, then press #" << endl;
    do {
      digit = device.ReadDTMF(lineNumber);
      if (digit) {
	number[num_digits++] = digit;
	cout << "digit: " << digit << endl;
	collectTimer = 10000;
      }
      if (device.IsLineDisconnected(lineNumber)) {
        cerr << "Caller hung up. " << endl;
        return FALSE;
      }
      if (shutdown.Wait(50))
        return FALSE;
    } while((digit != '#') && (num_digits < max_digits) && 
	    collectTimer.IsRunning());

    device.StopTone(lineNumber);

    if (num_digits) {
      if (digit == '#')
	num_digits--;
      number[num_digits] = 0;
      cout << "num_digits: " << num_digits << " number: " << number << endl;
    }
    else {
      device.SetLineOnHook(lineNumber);
      return FALSE;
    }
  }

  // if we have collected a number concatenate with ip host
  if (*number) {
    exth323Address = PString(number) + PString("@") + h323Address;
  }
  else {
    exth323Address = h323Address;    
  }
  cout << "Make call to h323 Address " << exth323Address << endl;
  PTRACE(2, "Monitor\tMake call to h323 Address " << exth323Address);
  // now make call

  {
    GatewayConnection * myConnection = (GatewayConnection *)endpoint.MakeCallLocked(exth323Address, callToken);
    if (myConnection == NULL)
      return FALSE;

    myConnection->line = this;
    myConnection->SetLocalPartyName(callerID);
    myConnection->Unlock();
  }

  PTimer ringTimer(3000);

  while (!endpoint.IsConnectionEstablished(callToken)) {
    // If H.323 call has failed then ignore the call
    if (!endpoint.HasConnection(callToken)) {
      cerr << "endpoint does not have connection for " << callToken << endl;
      return FALSE;
    }

    if (!collectNumber) {
      // If the ringing stops before establishment, clear the call and exit
      if (!IsLIDRinging(device, lineNumber, ringTimer)) {
	cerr << "Lid is not ringing. " << endl;
	endpoint.ClearCall(callToken);
	return FALSE;
      } 
    }

    if (device.IsLineDisconnected(lineNumber)) {
      cerr << "Caller hung up. " << endl;
      endpoint.ClearCall(callToken);
      return FALSE;
    }

    if (shutdown.Wait(50))
      return FALSE;
  }

  if (!collectNumber) {
    cout << "Line " << device << '-' << lineNumber << " going off hook" <<endl;
    PTRACE(2, "Monitor\tLine " << device << '-' << lineNumber << " going off hook");
    // Have H.323 call answered, so answer the PSTN line
    device.SetLineOffHook(lineNumber);
  } 
  
  // Now wait for either party to hang up
  while (endpoint.HasConnection(callToken) && !device.IsLineDisconnected(lineNumber)) 
    if (shutdown.Wait(200))
      return TRUE;

  cout << "Line " << device << '-' << lineNumber << " disconnected." << endl;
  PTRACE(2, "Monitor\tLine " << device << '-' << lineNumber << " disconnected.");
  endpoint.ClearCall(callToken);
  return TRUE;
}


// End of File ///////////////////////////////////////////////////////////////






