/*
 *    (c) Copyright 2001  Uwe Steinmann.
 *    All rights reserved.
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the
 *    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *    Boston, MA 02111-1307, USA.
 */

/* $Id: ps_util.c,v 1.24 2004/06/07 12:00:06 steinm Exp $ */

#include <stdio.h>
#include <string.h>
#include "config.h"
#include "ps_intern.h"
#include "ps_memory.h"
#include "ps_fontenc.h"
#include "ps_inputenc.h"
#include "ps_error.h"

/* ps_ght_malloc() {{{
 * 
 * malloc routine which is passed to the hash table functions.
 */
void* ps_ght_malloc(size_t size, void *data) {
	PSDoc *psdoc = data;
	return(psdoc->malloc(psdoc, size, _("Allocate memory for hash table of char metrics")));
}
/* }}} */

/* ps_ght_free() {{{
 * 
 * free routine which is passed to the hash table functions.
 */
void ps_ght_free(void *ptr, void *data) {
	PSDoc *psdoc = data;
	psdoc->free(psdoc, ptr);
}
/* }}} */

/* ps_enter_scope() {{{
 */
void ps_enter_scope(PSDoc *psdoc, int scope) {
	if(psdoc->scopecount == (MAX_SCOPES-1)) {
		ps_error(psdoc, PS_RuntimeError, _("Maximum number of scopes reached."));
	} else {
		psdoc->scopecount++;
		psdoc->scopes[psdoc->scopecount] = scope;
	}
}
/* }}} */

/* ps_leave_scope() {{{
 */
void ps_leave_scope(PSDoc *psdoc, int scope) {
	if(scope != psdoc->scopes[psdoc->scopecount])
		ps_error(psdoc, PS_RuntimeError, _("Trying to leave scope %d but you are in %d."), scope, psdoc->scopes[psdoc->scopecount]);
	else
		psdoc->scopecount--;
}
/* }}} */

/* ps_current_scope() {{{
 */
int ps_current_scope(PSDoc *psdoc) {
	return(psdoc->scopes[psdoc->scopecount]);
}
/* }}} */

/* ps_check_scope() {{{
 */
ps_bool ps_check_scope(PSDoc *psdoc, int scope) {
	return(scope & psdoc->scopes[psdoc->scopecount]);
}
/* }}} */

/* ps_find_resource() {{{
 */
char *ps_find_resource(PSDoc *psdoc, const char *category, const char *resource) {
	PS_CATEGORY *cat;
	PS_RESOURCE *res;

	for(cat = dlst_first(psdoc->categories); cat != NULL; cat = dlst_next(cat)) {
		if (!strcmp(cat->name, category)) {
			for(res = dlst_first(cat->resources); res != NULL; res = dlst_next(res)) {
				if (!strcmp(res->name, resource))
					return(res->value);
			}
		}
	}
	return(NULL);
}
/* }}} */

/* ps_add_resource() {{{
 */
PS_RESOURCE * ps_add_resource(PSDoc *psdoc, const char *category, const char *resource, const char *filename, const char *prefix) {
	PS_CATEGORY *cat;
	PS_RESOURCE *res;

	if (strcmp("FontAFM", category) &&
			strcmp("FontOutline", category) &&
			strcmp("FontProtusion", category) &&
			strcmp("RightMarginKerning", category) &&
			strcmp("LeftMarginKerning", category)) {
		return(NULL);
	}

	for(cat = dlst_first(psdoc->categories); cat != NULL; cat = dlst_next(cat)) {
		if(!strcmp(cat->name, category))
			break;
	}

	if(cat == NULL) {
		if(NULL == (cat = (PS_CATEGORY *) dlst_newnode(psdoc->categories, sizeof(PS_CATEGORY)))) {
			return(NULL);
		}
		cat->name = ps_strdup(psdoc, category);
		cat->resources = dlst_init(psdoc->malloc, psdoc->realloc, psdoc->free);
		dlst_insertafter(psdoc->categories, cat, PS_DLST_HEAD(psdoc->categories));
	}

	// Check if resource already available
	for(res = dlst_first(cat->resources); res != NULL; res = dlst_next(res)) {
		if(!strcmp(res->name, resource))
			break;
	}
	if(res) {
		psdoc->free(psdoc, res->name);
		res->name = ps_strdup(psdoc, resource);
		psdoc->free(psdoc, res->value);
		res->value = ps_strdup(psdoc, filename);
	} else {
		if(NULL == (res = (PS_RESOURCE *) dlst_newnode(cat->resources, sizeof(PS_RESOURCE)))) {
			return(NULL);
		}
		res->name = ps_strdup(psdoc, resource);
		res->value = ps_strdup(psdoc, filename);
		dlst_insertafter(cat->resources, res, PS_DLST_HEAD(cat->resources));
	}

	return(res);
}
/* }}} */

/* ps_del_resources() {{{
 */
void ps_del_resources(PSDoc *psdoc) {
	PS_CATEGORY *cat, *nextcat;
	PS_RESOURCE *res, *nextres;
	if(NULL == psdoc->categories)
		return;
	cat = dlst_first(psdoc->categories);
	while(cat != NULL) {
		nextcat = dlst_next(cat);
		psdoc->free(psdoc, cat->name);
		res = dlst_first(cat->resources);
		while(res != NULL) {
			nextres = dlst_next(res);
			psdoc->free(psdoc, res->name);
			psdoc->free(psdoc, res->value);
			res = nextres;
		}
		dlst_kill(cat->resources, dlst_freenode);
		cat = nextcat;
	}
	dlst_kill(psdoc->categories, dlst_freenode);
	psdoc->categories = NULL;
}
/* }}} */

/* ps_del_parameters() {{{
 */
void ps_del_parameters(PSDoc *psdoc) {
	PS_PARAMETER *param, *nextparam;
	if(NULL == psdoc->parameters)
		return;
	param = dlst_first(psdoc->parameters);
	while(param != NULL) {
		nextparam = dlst_next(param);
		psdoc->free(psdoc, param->name);
		psdoc->free(psdoc, param->value);
		param = nextparam;
	}
	dlst_kill(psdoc->parameters, dlst_freenode);
	psdoc->parameters = NULL;
}
/* }}} */

/* ps_del_values() {{{
 */
void ps_del_values(PSDoc *psdoc) {
	PS_VALUE *param, *nextparam;
	if(NULL == psdoc->values)
		return;
	param = dlst_first(psdoc->values);
	while(param != NULL) {
		nextparam = dlst_next(param);
		psdoc->free(psdoc, param->name);
		param = nextparam;
	}
	dlst_kill(psdoc->values, dlst_freenode);
	psdoc->values = NULL;
}
/* }}} */

/* ps_del_bookmarks() {{{
 */
void ps_del_bookmarks(PSDoc *psdoc, DLIST *bookmarks) {
	PS_BOOKMARK *bm, *nextbm;
	if(NULL == bookmarks)
		return;
	bm = dlst_first(bookmarks);
	while(bm != NULL) {
		nextbm = dlst_next(bm);
		psdoc->free(psdoc, bm->text);
		ps_del_bookmarks(psdoc, bm->children);
		bm = nextbm;
	}
	dlst_kill(bookmarks, dlst_freenode);
}
/* }}} */

/* ps_add_word_spacing() {{{
 */
void ps_add_word_spacing(PSDoc *psdoc, PSFont *psfont, float value) {
	psfont->wordspace += value * 1000.0 / psfont->size;
}
/* }}} */

/* ps_set_word_spacing() {{{
 */
void ps_set_word_spacing(PSDoc *psdoc, PSFont *psfont, float value) {
	ADOBEINFO *ai;

	if(value != 0.0) {
		psfont->wordspace = value * 1000.0 / psfont->size;
	} else {
		ai = gfindadobe(psfont->metrics->gadobechars, "space");
		if(ai)
			psfont->wordspace = ai->width;
		else
			psfont->wordspace = 500.0;
	}
}
/* }}} */

/* ps_get_word_spacing() {{{
 */
float ps_get_word_spacing(PSDoc *psdoc, PSFont *psfont) {
	float ws;
	ws = ((float) psfont->wordspace) * psfont->size / 1000.0;
	return(ws);
}
/* }}} */

/* ps_check_for_lig() {{{
 * Checks if the characters in text form a ligature with the character
 * in ai. If that is the case the number of characters consumed in text
 * will be returned in offset and the name of the ligature will be returned
 * in newadobename.
 * Ligatures can be broken off by a ''. In such a case 1 will be returned
 * and offset is set to 1 in order to skip the ''. newadobename is set
 * to the name of the passed character.
 */
int ps_check_for_lig(ADOBEFONTMETRIC *metrics, ADOBEINFO *ai, const char *text, char **newadobename, int *offset) {
	ADOBEINFO *nextai;
	LIG *ligs;
	if((NULL == ai) || (NULL == ai->ligs))
		return 0;
	if(NULL == text || text[0] == '\0')
		return 0;

	ligs = ai->ligs;
	/*  breaks ligatures. The idea is, that every char followed by a ''
	 * forms a ligature which is the char itself. Setting offset to
	 * 1 will skip the '' */
	if(ligs && text[0] == '') {
		*offset += 1;
		*newadobename = ai->adobename;
		return 1;
	}
	if(ligs) {
		int _offset = 0;
		char *nextadobename;
		nextai = gfindadobe(metrics->gadobechars, inputencoding.vec[(unsigned char) (text[0])]);
		if(NULL == nextai)
			return 0;

		/* check if nextai and the following char form a ligature
		 * If that is the case then there is a change that we have a 3 char
		 * ligature. First check if current char and ligature of 2nd and 3th
		 * char for a ligature.
		 */
		if(ps_check_for_lig(metrics, nextai, &text[1], &nextadobename, &_offset)) {
			while(ligs) {
				if(0 == strcmp(ligs->succ, nextadobename)) {
					*offset += 1+_offset;
					*newadobename = ligs->sub;
					return 1;
				}
				ligs = ligs->next;
			}
		}

		/* If we haven't found a three char ligature at this point then check
		 * for a 2 char ligatur.
		 */
		ligs = ai->ligs;
		while(ligs) {
	//		printf("Ligature for %s is %s\n", ai->adobename, ligs->succ);
			if(!strcmp(ligs->succ, nextai->adobename)) {
				ADOBEINFO *ligai;
				*offset += 1;
				/* check if the ligature made of the first two chars form a ligature
				 * which the third char
				 */
				ligai = gfindadobe(metrics->gadobechars, ligs->sub);
				if(ligai) {
					if(ps_check_for_lig(metrics, ligai, &text[1], &nextadobename, offset)) {
						*newadobename = nextadobename;
						return 1;

					}
				}
				*newadobename = ligs->sub;
	//			printf("Found %s followed by %s replaced with %s\n", ai->adobename, nextai->adobename, ligs->sub);
				return 1;
			}
			ligs = ligs->next;
		}
	}
	return 0;
}
/* }}} */

/* ps_build_enc_hash() {{{
 * Builds the hash table of an encoding vector.
 * The glyhname is the the key, the position in the vector is the data.
 * The position of the glyph in the vector starts at 1 because 0 is reserved
 * by the hash table to indicate the end of the list.
 * If a glyphname exists more than once, only the first one will be
 * stored in the hash table. This would be the case for "" as glyphname,
 * but empty strings are not stored at all.
 */
ght_hash_table_t *ps_build_enc_hash(PSDoc *psdoc, ENCODING *enc) {
	int i;
	ght_hash_table_t *hashvec;
	hashvec = ght_create(512);
	if(hashvec) {
		ght_set_alloc(hashvec, ps_ght_malloc, ps_ght_free, psdoc);
		for(i=0; i<256; i++) {
			if(strlen(enc->vec[i]) > 0)
				ght_insert(hashvec, (void *) i+1, strlen(enc->vec[i])+1, enc->vec[i]);
		}
	}
	return(hashvec);
}
/* }}} */

/* ps_build_enc_from_font() {{{
 * Builds the hash table with a font encoding from the font itself.
 * This will only include those glyphs which have a number > 0.
 */
ght_hash_table_t *ps_build_enc_from_font(PSDoc *psdoc, ADOBEFONTMETRIC *metrics) {
	int i;
	ght_hash_table_t *hashvec;

	hashvec = ght_create(512);
	if(hashvec) {
		ght_iterator_t iterator;
		char *p_key;
		ADOBEINFO *p_e;

		ght_set_alloc(hashvec, ps_ght_malloc, ps_ght_free, psdoc);
		for(p_e = ght_first(metrics->gadobechars, &iterator, (void **) &p_key); p_e; p_e = ght_next(metrics->gadobechars, &iterator, (void **) &p_key)) {
			/* Do not add glyphs with a number <= 0 */
			if(p_e->adobenum > 0) {
				if(0 > ght_insert(hashvec, (void *) p_e->adobenum+1, strlen(p_e->adobename)+1, p_e->adobename))
				fprintf(stderr, "Could not insert entry %d->%s into font encoding hash table.\n", p_e->adobenum, p_e->adobename);
			}
		}
	}
	return(hashvec);
}
/* }}} */

/* ps_build_enc_vector() {{{
 * Builds the encoding vector from the hash table
 */
ENCODING *ps_build_enc_vector(PSDoc *psdoc, ght_hash_table_t *hashvec) {
	if(hashvec) {
		ght_iterator_t iterator;
		char *glyphname;
		int i;
		ENCODING *enc;

		enc = (ENCODING *) psdoc->malloc(psdoc, sizeof(ENCODING), _("Allocate memory for new encoding vector.")) ;
		if(NULL == enc) {
			ps_error(psdoc, PS_MemoryError, _("Could not allocate memory for encoding vector."));
			return NULL;
		}
		memset(enc, 0, sizeof(ENCODING));

		for(i = (int) ght_first(hashvec, &iterator, (void **) &glyphname); i; i = (int) ght_next(hashvec, &iterator, (void **) &glyphname)) {
//			printf("got %s = %d\n", glyphname, i-1);
			enc->vec[i-1] = ps_strdup(psdoc, glyphname);
		}

		return(enc);
	} else {
		return NULL;
	}
}
/* }}} */

/* ps_list_fontenc() {{{
 * 
 */
void ps_list_fontenc(PSDoc *psdoc, ght_hash_table_t *hashvec) {
	if(hashvec) {
		ght_iterator_t iterator;
		char *glyphname;
		int i;

		fprintf(stderr, "---- Font encoding vector -----\n");
		fprintf(stderr, "Has %d entries.\n", ght_size(hashvec));
		for(i = (int) ght_first(hashvec, &iterator, (void **) &glyphname); i; i = (int) ght_next(hashvec, &iterator, (void **) &glyphname)) {
			fprintf(stderr, "%s = %d\n", glyphname, i-1);
		}
	}
}
/* }}} */

/* ps_free_enc_vector() {{{
 * Frees all memory allocated for an encoding vector including the vector
 * itself.
 */
void ps_free_enc_vector(PSDoc *psdoc, ENCODING *enc) {
	int i;
	if(!enc)
		return;
	if(enc->name)
		psdoc->free(psdoc, enc->name);
	for(i=0; i<256; i++) {
		if(enc->vec[i])
			psdoc->free(psdoc, enc->vec[i]);
	}
	psdoc->free(psdoc, enc);
}
/* }}} */

/* ps_fontenc_code() {{{
 * Returns the index in the fontencoding vector
 */
unsigned char ps_fontenc_code(ght_hash_table_t *fontenc, char *adobename) {
	if(fontenc) {
		int code;
		code = ght_get(fontenc, strlen(adobename)+1, (void *) adobename);
		if(code)
			return((unsigned char) code -1);
		else {
			return '?';
		}
	} else {
		return '?';
	}
}
/* }}} */

/* ps_inputenc_name() {{{
 * Returns the adobe name for a char in the input encoding vector
 */
char *ps_inputenc_name(ENCODING *inputenc, unsigned char c) {
	if(inputenc) {
		return(inputenc->vec[c]);
	} else {
		return NULL;
	}
}
/* }}} */

/* ps_get_bool_parameter() {{{
 */
int ps_get_bool_parameter(PSDoc *psdoc, const char *name, int deflt) {
	const char *onoffstr;
	onoffstr = PS_get_parameter(psdoc, name, 0);
	if(NULL == onoffstr) {
		return(deflt);
	} else {
		if(!strcmp(onoffstr, "true"))
			return(1);
		else
			return(0);
	}
}
/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: sw=2 ts=2 fdm=marker
 * vim<600: sw=2 ts=2
 */
