/*
    ProjectType.h

    Protocol declaration of the ProjectType protocol for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import <Foundation/NSObject.h>

@class NSString, NSArray, NSDictionary, NSImage;

@class ProjectDocument;

/**
 * Identify the capabilities a project type requests from ProjectManager.
 * A project can have the following capabilities:
 *
 * - Files Capability: the project contains files which the user can
 *      create, organize and edit.
 *
 * - Frameworks Capability: the project supports linking against frameworks.
 *
 * - Build Capability: the project can be built using the make tool. The
 *      structure of the makefile and any additional supporting files (as
 *      well as the freedom of the project being something completely
 *      unrelated to software) is left up to the project, but the makefile
 *      must include at least the following rules:
 *              + make all
 *                  Should build the complete project.
 *              + make install
 *                  Should install the project. If the project doesn't
 *                  support installation, it output a message like
 *                  "This project can't be installed" and terminate with
 *                  an error.
 *              + make clean
 *                  Should delete files generated during the build process.
 *              + make distclean
 *                  Should delete absolutely all generated files and revert
 *                  the project to a "distributed" state.
 *
 *      Additionally, optional support for the following arguments can
 *      be provided:
 *              debug=yes
 *                  Should build a debugging version of the project. This
 *                  is mostly usable only for software projects.
 *              profile=yes
 *                  Should build a profiling version of the project.
 *
 * - Run Capability: the project's binary can be run.
 *
 * - Debugging Capability: the project's binary can be run in
 *      gdb for debugging.
 *
 * - Subprojects Capability: the project can have subprojects.
 */
enum {
  FilesProjectCapability =              0x01,
  FrameworksProjectCapability =         0x02,
  BuildProjectCapability =              0x04,
  RunProjectCapability =                0x08,
  DebugProjectCapability =              0x10,
  SubprojectsProjectCapability =        0x20,
  AllProjectCapabilities =              0x7fffffff
};

/**
 * Identifies various project binary type characteristics. A combination
 * of these flags is passed to the project in -pathToProjectBinaryOfType:.
 */
enum {
  DefaultBinaryType =           0x00,
  DebuggingBinaryType =         0x01,
  ProfilingBinaryType =         0x02,
  AnyBinaryType =               0xFF
};

/**
 * This protocol is adopted by project type support objects.
 *
 * Each project has a 'type' which is represented by it's
 * projectType object. This object defines what kind of file
 * categories exist in the project, how to construct the
 * makefile and a bunch of other things.
 */
@protocol ProjectType <NSObject>

// ======== Class Methods =========

/// Should return the project type ID.
+ (NSString *) projectTypeID;

/**
 * Should return a human-readable, perhaps localized descriptive
 * name of the project type.
 */
+ (NSString *) humanReadableProjectTypeName;

/// Should return a more detailed description of the project type.
+ (NSString *) projectTypeDescription;

/// Should return an icon of the project type (max 48x48 pixels).
+ (NSImage *) projectTypeIcon;

/**
 * Should return an OR'ed set of project capabilities provided by this
 * project type.
 */
+ (int) projectCapabilities;

/**
 * The project type should return a dictionary where keys are
 * project template names and values are descriptions of the
 * templates.
 */
+ (NSDictionary *) projectTemplateDescriptions;

/**
 * The receiver should return the path to the location of the
 * project template named `templateName'.
 */
+ (NSString *) pathToProjectTemplate: (NSString *) templateName;


// ========= Instance Methods ==========

- initWithOwner: (ProjectDocument *) aProject
 infoDictionary: (NSDictionary *) infoDict;

/**
 * Should return the project's info dictionary which is suitable for
 * being written into the project file when saving.
 */
- (NSDictionary *) infoDictionary;

/**
 * Should return a dictionary containing keys for names and values
 * for various views representing various project attributes. These
 * are then displayed in the project's window.
 */
- (NSDictionary *) projectAttributeViews;

/**
 * Invoked immediately before a build is started. The project type
 * should now (re)generate Makefiles as necessary.
 *
 * @return YES if preparation succeeded, otherwise NO.
 */
- (BOOL) prepareForBuild;

/**
 * The receiver should return a path to the directory where it holds
 * file templates for category `category'. If no file templates are
 * available for the specified category, `nil' should be returned instead.
 */
- (NSString *) pathToFileTemplatesDirectoryForCategory: (NSString *) category;

/**
 * Should check whether the user is allowed to add new categories to
 * `category'. This is used for example in the Application project
 * type, where adding categories or files to the root category is not
 * allowed, but is allowed in subcategories such as "/Class Files" or
 * "/Localized Resource Files".
 *
 * @return YES if the user is allows to add new categories to `category',
 * NO if he/she is not allowed to do so. 
 */
- (BOOL) canAddCategoriesToCategory: (NSString *) category;

/**
 * Should check whether the user is allowed to add files to `category'.
 *
 * @return YES if the user is allows to add new files to `category',
 * NO if he/she is not allowed to do so.
 *
 * @see -[ProjectType canAddCategoriesToCategory:].
 */
- (BOOL) canAddFilesToCategory: (NSString *) category;

/**
 * Should check whether the user is allowed to delete or rename `category'.
 *
 * @return YES if the user is allowed to delete category `category',
 * NO otherwise.
 */
- (BOOL) canDeleteCategory: (NSString *) category;

/**
 * Should return a path to the physical location of file `fileName'
 * which is located in category `category'. If `fileName' is nil, then
 * a path to the category's directory should be returned instead.
 */
- (NSString *) pathToFile: (NSString *) fileName
               inCategory: (NSString *) category;

/**
 * Should return a path to the directory where the project type
 * would like it's subprojects to reside.
 */
- (NSString *) pathToSubprojectsDirectory;

/**
 * Should return an iconic representation of the category.
 */
- (NSImage *) iconForCategory: (NSString *) category;

/**
 * Should return an array of file types which are allowed inside
 * the specified category or `nil' if any files are allowed.
 */
- (NSArray *) permissibleFileTypesInCategory: (NSString *) category;

/**
 * Should return a path to the project's binary of type `type'.
 */
- (NSString *) pathToProjectBinaryOfType: (int) type;

/**
 * Should return the debugger type which to use for debugging this
 * project type. Project types which are not debuggable may safely
 * return `nil' here.
 */
- (NSString *) debuggerType;

/**
 * Should return a list of frameworks of the project which can neither
 * be removed nor renamed. They are thus considered ``fixed'' or mandatory.
 */
- (NSArray *) fixedFrameworks;

@end
