/*
    Launcher.m

    Implementation of the Launcher class for the ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "Launcher.h"

#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#import <Foundation/NSBundle.h>
#import <Foundation/NSTask.h>
#import <Foundation/NSFileHandle.h>
#import <Foundation/NSProcessInfo.h>
#import <Foundation/NSNotification.h>

#import <AppKit/NSForm.h>
#import <AppKit/NSFormCell.h>
#import <AppKit/NSTextView.h>
#import <AppKit/NSOpenPanel.h>
#import <AppKit/NSFont.h>
#import <AppKit/NSColor.h>
#import <AppKit/NSImage.h>

#import "ProjectDocument.h"
#import "ProjectType.h"
#import "ArgumentsInspector.h"
#import "EnvironmentInspector.h"

@interface Launcher (Private)

- (void) clearState;

@end

@implementation Launcher (Private)

- (void) clearState
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  DESTROY(task);
  DESTROY(stdinHandle);
  DESTROY(stdoutHandle);
  DESTROY(stderrHandle);
}

@end

@implementation Launcher

- (void) dealloc
{
  if (task != nil && [task isRunning])
    {
      [task terminate];
    }

  [self clearState];

  TEST_RELEASE(arguments);
  TEST_RELEASE(environment);

  [super dealloc];
}

- init
{
  if ([super init])
    {
      arguments = [NSMutableArray new];
       // create a copy of our environment
      environment = [[[NSProcessInfo processInfo] environment] mutableCopy];

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) awakeFromNib
{
  [super awakeFromNib];

  stdin = [stdin cellAtIndex: 0];

  [stdout setFont: [NSFont userFixedPitchFontOfSize: 0]];
  [stderr setFont: [NSFont userFixedPitchFontOfSize: 0]];
  [stderr setTextColor: [NSColor whiteColor]];

  [launchButton setImage: [NSImage imageNamed: @"Tab_Run"]];
}

- (void) launch: (id)sender
{
  NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];

  if (task != nil)
    {
      [nc removeObserver: self];

      [task terminate];
      [self clearState];

      [launchButton setTitle: _(@"Launch")];
      [launchButton setImage: [NSImage imageNamed: @"Tab_Run"]];
    }
  else
    {
      NSPipe * stdinPipe, * stdoutPipe, * stderrPipe;

      task = [NSTask new];

      [task setLaunchPath: [[document projectType]
        pathToProjectBinaryOfType: AnyBinaryType]];
      [task setArguments: arguments];
      [task setEnvironment: environment];
      if ([[workingDirectory stringValue] length] > 0)
        {
          [task setCurrentDirectoryPath: [workingDirectory stringValue]];
        }

      [nc addObserver: self
             selector: @selector(taskTerminated)
                 name: NSTaskDidTerminateNotification
               object: task];

      stdinPipe = [NSPipe pipe];
      stdoutPipe = [NSPipe pipe];
      stderrPipe = [NSPipe pipe];

      [task setStandardInput: stdinPipe];
      [task setStandardOutput: stdoutPipe];
      [task setStandardError: stderrPipe];

      ASSIGN(stdinHandle, [stdinPipe fileHandleForWriting]);
      ASSIGN(stdoutHandle, [stdoutPipe fileHandleForReading]);
      ASSIGN(stderrHandle, [stderrPipe fileHandleForReading]);

      [nc addObserver: self
             selector: @selector(readStdout)
                 name: NSFileHandleDataAvailableNotification
               object: stdoutHandle];
      [nc addObserver: self
             selector: @selector(readStderr)
                 name: NSFileHandleDataAvailableNotification
               object: stderrHandle];

      NS_DURING
        [task launch];
      NS_HANDLER
        NSRunAlertPanel(_(@"Failed to launch"),
          _(@"Failed to launch the project.\n"
            @"Perhaps you need to build it first."),
          nil, nil, nil);

        [self clearState];

        return;
      NS_ENDHANDLER

      [stdout setString: @""];
      [stderr setString: @""];

      [stdoutHandle waitForDataInBackgroundAndNotify];
      [stderrHandle waitForDataInBackgroundAndNotify];

      [launchButton setTitle: _(@"Kill")];
      [launchButton setImage: [NSImage imageNamed: @"Kill"]];
    }
}


- (void) chooseWorkingDirectory: (id)sender
{
  NSOpenPanel * op = [NSOpenPanel openPanel];

  [op setCanChooseDirectories: YES];
  [op setCanChooseFiles: NO];

  if ([op runModalForTypes: nil] == NSOKButton)
    {
      [workingDirectory setStringValue: [op filename]];
    }
}


- (void) showArguments: (id)sender
{
  [[ArgumentsInspector shared]
    activateWithProject: [document projectName] arguments: arguments];
}


- (void) showEnvironment: (id)sender
{
  [[EnvironmentInspector shared]
    activateWithProject: [document projectName] environment: environment];
}

- (void) taskTerminated
{
  [stdout replaceCharactersInRange: NSMakeRange([[stdout string] length], 0)
                        withString:
    [NSString stringWithFormat: _(@"Terminated with code %i"), [task
    terminationStatus]]];

  [self clearState];
  [launchButton setTitle: _(@"Launch")];
  [launchButton setImage: [NSImage imageNamed: @"Tab_Run"]];
}

- (void) writeStdin: sender
{
  if (stdinHandle != nil)
    {
      NSString * string = [stdin stringValue];

      // append a trailing newline automagically
      string = [string stringByAppendingString: @"\n"];

      [stdinHandle writeData: [string
        dataUsingEncoding: NSUTF8StringEncoding]];
      [stdinHandle synchronizeFile];
    }
  else
    {
      NSRunAlertPanel(_(@"Not running"),
        _(@"The project is not running."),
        nil, nil, nil);
    }
}

- (void) readStdout
{
  NSString * string = [[[NSString alloc]
    initWithData: [stdoutHandle availableData] encoding: NSUTF8StringEncoding]
    autorelease];

  [stdout replaceCharactersInRange: NSMakeRange([[stdout string] length], 0)
                        withString: string];
  [stdout scrollRangeToVisible: NSMakeRange([[stdout string] length], 0)];
  [stdoutHandle waitForDataInBackgroundAndNotify];
}

- (void) readStderr
{
  NSString * string = [[[NSString alloc]
    initWithData: [stderrHandle availableData] encoding: NSUTF8StringEncoding]
    autorelease];

  [stderr replaceCharactersInRange: NSMakeRange([[stderr string] length], 0)
                        withString: string];
  [stderr scrollRangeToVisible: NSMakeRange([[stderr string] length], 0)];
  [stderrHandle waitForDataInBackgroundAndNotify];
}

@end
