/*
    FileTemplateChooser.m

    Interface declaration of the ArgumentsInspector class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "FileTemplateChooser.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSFileManager.h>

#import <AppKit/NSNibLoading.h>
#import <AppKit/NSForm.h>
#import <AppKit/NSScrollView.h>
#import <AppKit/NSWorkspace.h>
#import <AppKit/NSMatrix.h>
#import <AppKit/NSBrowser.h>
#import <AppKit/NSBrowserCell.h>
#import <AppKit/NSPanel.h>
#import <AppKit/NSApplication.h>

@implementation FileTemplateChooser

static FileTemplateChooser * shared = nil;

+ shared
{
  if (shared == nil)
    {
      shared = [self new];
    }

  return shared;
}

- init
{
  if ([super init])
    {
      [NSBundle loadNibNamed: @"FileTemplateChooser" owner: self];

      return self;
    }
  else
    {
      return nil;
    }
}

- (int) runModalWithTemplatesDirectory: (NSString *) templatesDir
{
  int code;

  ASSIGN(templatesDirectory, templatesDir);

  [panel selectKeyViewFollowingView: okButton];
  [browser reloadColumn: 0];
  [browser setPath: @"/"];

  code = [NSApp runModalForWindow: panel];
  [panel close];

  return code;
}

- (void) awakeFromNib
{
  fileName = [fileName cellAtIndex: 0];
  [[description enclosingScrollView] setHasVerticalScroller: NO];
}

- (NSString *) templatePath
{
  return [templatesDirectory stringByAppendingPathComponent: [browser path]];
}

- (NSString *) fileName
{
  return [fileName stringValue];
}

- (void) ok: sender
{
  if ([[fileName stringValue] length] == 0)
    {
      NSRunAlertPanel(_(@"No file name chosen"),
        _(@"You must choose a name for the new file."),
        nil, nil, nil);

      return;
    }

  if ([[browser selectedCell] isLeaf] == NO)
    {
      NSRunAlertPanel(_(@"No template chosen"),
        _(@"You must choose a template file to import."),
        nil, nil, nil);

      return;
    }

  [NSApp stopModalWithCode: NSOKButton];
}

- (void) cancel: sender
{
  [NSApp stopModalWithCode: NSCancelButton];
}

- (void) selectFile: sender
{
  NSString * string;
  NSString * path =
    [templatesDirectory stringByAppendingPathComponent:
    [[[browser path] stringByDeletingPathExtension]
    stringByAppendingPathExtension: @"pmdesc"]];
  NSFileManager * fm = [NSFileManager defaultManager];

  if ([fm fileExistsAtPath: path] &&
    (string = [NSString stringWithContentsOfFile: path]) != nil)
    {
      [description setString: string];
    }
  else
    {
      [description setString: @""];
    }
}

- (void)      browser: (NSBrowser *)sender
  createRowsForColumn: (int)column
             inMatrix: (NSMatrix *)matrix
{
  NSFileManager * fm = [NSFileManager defaultManager];
  NSString * filename;
  NSDirectoryEnumerator * de;
  NSWorkspace * ws = [NSWorkspace sharedWorkspace];

  if (templatesDirectory == nil)
    {
      return;
    }

  de = [fm enumeratorAtPath: [templatesDirectory
    stringByAppendingPathComponent: [browser path]]];
  while ((filename = [de nextObject]) != nil)
    {
      NSBrowserCell * cell;
      NSDictionary * fattrs;

      // ignore description files
      if ([[filename pathExtension] isEqualToString: @"pmdesc"])
        continue;

      fattrs = [de fileAttributes];
      if ([[fattrs fileType] isEqualToString: NSFileTypeDirectory])
        {
          [de skipDescendents];
        }

      [matrix addRow];
      cell = [matrix cellAtRow: [matrix numberOfRows] - 1 column: 0];
      [cell setTitle: [filename lastPathComponent]];
       // treat as directories only files without an extension - directories
       // with an extension (e.g. ".gorm") are in fact file packages.
      if ([[filename pathExtension] isEqualToString: @""] &&
        [[fattrs fileType] isEqualToString: NSFileTypeDirectory])
        {
          [cell setLeaf: NO];
        }
      else
        {
          [cell setLeaf: YES];
        }
      [cell setImage: [ws iconForFile: filename]];
    }
}

@end
