/**
 * @file tools.c
 * @brief handle memory allocation and file access
 * @created 1999-08-19
 * @date 2007-08-31
 */
/*
 * copyright (c) 1998-2007 TLK Games all rights reserved
 * $Id: tools.c,v 1.26 2007/08/31 20:46:43 gurumeditation Exp $
 *
 * Powermanga is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Powermanga is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#include <stdbool.h>
#include <string.h>
#include "powermanga.h"
#include "tools.h"
#include "config_file.h"
#include <errno.h>
#ifndef POWERMANGA_SDL
#include <sys/types.h>
#include <sys/stat.h>
#endif

/**
 * memory-block description structure
 */
typedef struct
{
  /** Pointer to memory zone */
  char *addr;
  /** Size of memory zone in bytes */
  Sint32 size;
}
mem_struct;
mem_struct *memory_list_base = NULL;
mem_struct *memory_list = NULL;
/** Size of memory table */
static Uint32 iTailleListe;
/** Maximum number of memory zones being able to be allocated */
static Uint32 mem_maxnumof_zones;
/** Number of currently memory zones */
Uint32 mem_numof_zones;
/** Total memory size currently allocated */
Uint32 mem_total_size;
/** Maximum number of memory zones reached */
static Uint32 mem_maxreached_zones;
Uint32 loops_counter;
#ifdef POWERMANGA_SDL
static Uint32 time_begin;
static Uint32 ticks_previous;
#else
static struct timeval time_begin;
static struct timeval ticks_previous;
#endif
static Uint16 little_endian_to_ushort (Uint16 * _pMem);
/** Prefixe where the data files are localised */
static const char prefix_dir[] = PREFIX;
float *precalc_sin = NULL;
float *precalc_cos = NULL;
float *precalc_sin128 = NULL;
float *precalc_cos128 = NULL;
/** Table used for displacement of the guided missile */
float depix[13][32];
/** Table used for displacement of the guided missile */
float depiy[13][32];

/**
 * Initialize the memory table for our malloc() wrapper
 * @param numofzones Maximum number of memory zones
 * @return Boolean value on success or failure
 */
bool
memory_init (Uint32 numofzones)
{
  Uint32 i;
  mem_struct *memlst = memory_list_base;
  /* clear number of memory zones reserved */
  mem_numof_zones = 0;
  /* maximum number of memory zones being able to be allocated */
  mem_maxnumof_zones = numofzones;
  mem_maxreached_zones = 0;
  iTailleListe = mem_maxnumof_zones * sizeof (mem_struct);
  mem_total_size = iTailleListe;
  memory_list_base = (mem_struct *) malloc (iTailleListe);
  if (memory_list_base == NULL)
    {
      fprintf (stderr, "tools.c/memory_init ()"
               "malloc() failed\n");
      return FALSE;
    }
  memory_list = memory_list_base;

  /* clear memory table */
  memlst = memory_list_base;
  for (i = 0; i < mem_maxnumof_zones; i++, memlst++)
    {
      memlst->addr = NULL;
      memlst->size = 0;
    }
  return TRUE;
}

/**
 * Allocate memory, malloc() wrapper
 * @param memsize Size in bytes to alloc
 * @return Pointer to the allocated memory or NULL if an error occurred
 */
char *
memory_allocation (Uint32 memsize)
{
  char *addr = NULL;
  if (mem_numof_zones >= mem_maxnumof_zones)
    {
      fprintf (stderr, "tools.c/memory_allocation():"
               "table overflow; size request %i bytes; total allocate: %i in %i zones\n",
               memsize, mem_total_size, mem_numof_zones);
      return NULL;
    }
  addr = (char *) malloc (memsize);
  if (addr == NULL)
    {
      fprintf (stderr, "tools.c/memory_allocation(): malloc() "
               "return NULL; size request %i bytes; total allocate: %i in %i zones\n",
               memsize, mem_total_size, mem_numof_zones);
      return NULL;
    }
  memset (addr, 0, sizeof (char) * memsize);
  mem_total_size += memsize;
  memory_list->addr = addr;
  memory_list->size = memsize;
  memory_list += 1;
  mem_numof_zones++;
  if (mem_numof_zones > mem_maxreached_zones)
    {
    mem_maxreached_zones = mem_numof_zones;
    }
  return addr;
}

/**
 * Deallocates the memory, free() wrapper
 * @param addr Pointer to memory
 */
void
free_memory (char *addr)
{
  mem_struct *memlist;
  mem_struct *memlist_src;
  Uint32 i;
  if (addr == NULL)
    {
      fprintf (stderr, "tools.c/free_memory(): "
               "try to release a null address!\n");
      return;
    }
  memlist = memory_list_base;
  for (i = 0; i < mem_numof_zones; i++, memlist++)
    {
      /* search address */
      if (memlist->addr == addr)
        {
          free (addr);
          memlist_src = memlist + 1;
          mem_total_size -= memlist->size;
          mem_numof_zones--;
          memory_list--;
          while (i < mem_numof_zones)
            {
              memlist->addr = memlist_src->addr;
              memlist->size = memlist_src->size;
              i++;
              memlist++;
              memlist_src++;
            }
          memlist->addr = NULL;
          memlist->size = 0;
          addr = NULL;
          break;
        }
    }
  if (addr != NULL)
    {
#ifdef __LP64__
      fprintf (stderr, "free_memory() : can't release the address %p\n",
               addr);
#else
      fprintf (stderr, "free_memory() : can't release the address %p\n",
               addr);
#endif
    }
}

/**
 * Releases all memory allocated
 * @param verbose Verbose level
 */
void
memory_releases_all (Sint32 verbose)
{
  Uint32 i;
  char *addr;
  mem_struct *memlist = memory_list_base;
#ifdef VERBOSE
  if (verbose)
    {
      fprintf (stdout, "> tools.c/memory_releases_all(): "
               "maximum of memory which were allocated during the game: %i\n",
               mem_maxreached_zones);
    }
#endif
  if (mem_numof_zones > 0)
    {
#ifdef VERBOSE
      if (verbose)
        {
          fprintf (stdout, "> tools.c/memory_releases_all(): "
                   "%i zones were not released.\n", mem_numof_zones);
        }
#endif
      for (i = 0; i < mem_numof_zones; i++, memlist++)
        {
          addr = memlist->addr;
          if (addr != NULL)
            {
              fprintf(stderr, "-> free(%p); size=%i\n",
                      memlist->addr, memlist->size);
              free (addr);
              memlist->addr = NULL;
              memlist->size = 0;
            }
        }
    }
  if (memory_list_base != NULL)
    {
      free (memory_list_base);
      memory_list_base = NULL;
    }
  mem_numof_zones = 0;
}

/**
 * Load and decompress a PCX file
 * @param filename Filename specified by path
 * @return Pointer to a bitmap_desc structure or null if an error occurred 
 */
bitmap_desc *
load_pcx (char *filename)
{
  Uint32 width, height, depth, size, ptr;
  Uint16 *ptr16;
  unsigned char numof_bytes;
  unsigned char val;
  Uint32 i, j, total;
  unsigned char *filedata, *pixel;
  bitmap_desc *bmp;
  filedata = (unsigned char *) loadfile (filename, &size);
  if (filedata == NULL)
    {
      return NULL;
    }
  ptr16 = (Uint16 *) filedata;
  width = (little_endian_to_ushort (ptr16 + 4) - little_endian_to_ushort (ptr16 + 2)) + 1;
  height = (little_endian_to_ushort (ptr16 + 5) - little_endian_to_ushort (ptr16 + 3)) + 1;
  /* bits per pixel */
  depth = filedata[3];
  
  /* allocate bitmap description structure memory */
  bmp = (bitmap_desc *) memory_allocation (sizeof (bitmap_desc));
  if (bmp == NULL)
    {
      fprintf (stderr, "tools.c/load_pcx(): "
               "not enough memory to allocate 'bmp'\n");
      return NULL;
    }
  bmp->width = width;
  bmp->height = height;
  bmp->depth = depth;
  bmp->size = width * height * (depth >> 3);
  /* allocate bitmap memory */
  bmp->pixel = memory_allocation (bmp->size);
  if (bmp->pixel == NULL)
    {
      fprintf (stderr,
               "(!)tools.c/load_pcx(%s): "
               "height=%i / width=%i \n",
               filename, width, height);
      fprintf (stderr, "(!)tools.c/load_pcx(): "
               "not enough memory to allocate %i bytes\n",
               bmp->size);
      free_memory ((char *) bmp);
      return NULL;
    }
  /* decompress rle */
  pixel = (unsigned char *) bmp->pixel;
  total = 0;
  i = size - 768;
  ptr = 128;
  while (ptr < i)
    {
      if ((filedata[ptr] & 0xC0) == 0xC0)
        {
          numof_bytes = filedata[ptr] & 0x3F;
          ptr++;
        }
      else
        {
          numof_bytes = 1;
        }
      val = filedata[ptr];
      /* bug fixed by Samuel Hocevar */
      total += numof_bytes;
      if (total >= bmp->size)
        {
          break;
        }
      for (j = 0; j < numof_bytes; j++)
        {
          *pixel = val;
          pixel++;
        }
      ptr++;
    }
  free_memory ((char *) filedata);
#ifdef VERBOSE
  if (power_conf->verbose)
    {
      fprintf (stdout,
               "> tools.c/load_pcx(\"%s\"): "
               " height=%i; width=%i; size=%i bytes\n",
               filename, width, height, bmp->size);
    }
#endif
  return bmp;
}

/**
 * Read a little-endian 16-bit signed short
 * @param pointer to a 16-bit signed short
 * @return the big or little endian 16-bit signed short value
 */
Sint16
little_endian_to_short (Sint16 *addr)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  Sint16 val = 0;
  unsigned char *ptr = (unsigned char *) addr;
  val = ptr[1];
  val <<= 8;
  val += ptr[0];
  return val;
#else
  return *addr;
#endif
}

/**
 * Read a little-endian 16-bit unsigned short
 * @param pointer to a 16-bit unsigned short
 * @return the big or little endian 16-bit unsigned short value
 */
static Uint16
little_endian_to_ushort (Uint16 * addr)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  Sint16 val = 0;
  unsigned char *ptr = (unsigned char *) addr;
  val = ptr[1];
  val <<= 8;
  val += ptr[0];
  return val;
#else
  return *addr;
#endif
}

/**
 * Read a little-endian 32-bit signed long
 * @param pointer to a 32-bit signed long
 * @return the big or little endian 32-bit signed long value
 */
Sint32
little_endian_to_int (Sint32 * addr)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  return (((*addr & 0xFF000000) >> 24) | ((*addr & 0x00FF0000) >> 8) |
          ((*addr & 0x0000FF00) << 8) | ((*addr & 0x000000FF) << 24));
#else
  return *addr;
#endif
}

/**
 * Write a little-endian 32-bit signed long
 * @param pointer to a 32-bit signed long
 * @return the big or little endian 32-bit signed long value
 */
void
int_to_little_endian (Sint32 value, Sint32 * addr)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  *addr = (((value & 0xFF000000) >> 24) | ((value & 0x00FF0000) >> 8) |
          ((value & 0x0000FF00) << 8) | ((value & 0x000000FF) << 24));
#else
  *addr = value;
#endif
}

/**
 * Convert a 32-bit zone memory
 * @param pointer to a 4 bytes in memory
 */
void
convert32bits_2bigendian (unsigned char *memory)
{
  unsigned char b0, b1, b2, b3;
  b0 = memory[1];
  b1 = memory[0];
  b2 = memory[3];
  b3 = memory[2];
  memory[0] = b2;
  memory[1] = b3;
  memory[2] = b0;
  memory[3] = b1;
}

/**
 * Creates a string representing an integer number
 * @param value the integer value to be converted
 * @param padding length of the string
 * @param str the string representation of the number
 */
void
integer_to_ascii (Sint32 value, Uint32 padding, char *str) 
{
  char* ptr = str + padding - 1 ;
  bool neg = (value < 0);
  if(neg) 
    {
      value = -value;
      --padding;
    }
  do
    {
      *ptr-- = (value % 10) + '0';
      value /= 10;
      --padding;
    } while(value && padding > 0);
  for(; padding > 0; --padding)
    {
      *ptr-- = '0';
    }
  if(neg)
    {
      *ptr-- = '-';
    }
}

/** Directory list to locate a file */
static const char
*data_directories[] =
{
  /* normally unused, except when running from the source directory... */
  ".",
  /* special value meaning "$(PREFIX)/share/games/powermanga/" */
  /* also marks end of list */
  0
};

/**
 * Locate a file under one of the data directories
 * @param name Name of file relative to data directory
 * @return Pointer to a malloc'd buffer containing the name under which the
 * file was found (free()-ing the buffer is the responsibility of the caller.)
 * or NULL if could not locate file (not found, or not enough memory, or the
 * name given was absolute)
 * @author Andre Majorel
 */
char *
locate_data_file (const char *const name)
{
  const char **p;
  char *pathname;
#ifdef WIN32
  struct _stat s;
#else
  struct stat s;
#endif
  const char *subdir = "/share/games/powermanga/";

  if (name == NULL)
    {
      fprintf (stderr, "(!)tools.c/locate_data_file(): " 
        "NULL pointer was passed as an argument!\n");
      return NULL;
    }
  /* if absolute path, return a pointer to a duplicate string */
  if (*name == '/')
    {
      return string_duplicate (name);
    }
  /* process each folder of the list */
  for (p = data_directories;; p++)
    {
      if (*p == 0)
        {
          pathname = memory_allocation (strlen(prefix_dir) + strlen(subdir) + strlen(name) + 1);
          if (pathname == NULL)
            {
              fflush (stdout);
              fprintf(stderr, "(!)tools.c/locate_data_file(): "
                      "not enough memory\n");
              return NULL;
            }
          strcpy (pathname, prefix_dir);
          strcat (pathname, subdir);
          strcat (pathname, name);
        }
      /* not user anymore */
      else if (**p == '~')
        {
          static const char bogus = '\0';
          static const char *home_dir = &bogus;
          if (home_dir == &bogus)
            {
              home_dir = getenv ("HOME");
            }
          if (home_dir == 0)
            {
              /* $HOME not set. Skip this directory */
              continue;
            }
          pathname = memory_allocation (strlen(home_dir) + 1 + strlen(*p + 1) + 1 + strlen(name) + 1);
          if (pathname == NULL)
            {
              fflush (stdout);
              fprintf(stderr, "(!)tools.c/locate_data_file(): "
                      "not enough memory\n");
              return NULL;
            }
          strcpy (pathname, home_dir);
          strcat (pathname, *p + 1);
          strcat (pathname, "/");
          strcat (pathname, name);
        }
      else
        {
          /* check if the file is located in current directory */
          pathname = memory_allocation (strlen(*p) + 1 + strlen(name) + 1);
          if (pathname == NULL)
            {
              fflush (stdout);
              fprintf(stderr, "(!)tools.c/locate_data_file(): "
                      "not enough memory\n");
              return NULL;
            }
          strcpy (pathname, *p);
          strcat (pathname, "/");
          strcat (pathname, name);
        }
/*
      puts (pathname);
*/
#ifdef WIN32
      if (_stat (pathname, &s) == 0 && !(s.st_mode & _S_IFDIR))
        {
          return pathname;
        }

#else
      if (stat (pathname, &s) == 0 && !S_ISDIR (s.st_mode))
        {
          return pathname;
        }
#endif
      free_memory (pathname);
      if (*p == 0)
        {
          break;
        }
    }
  /* not found */
  return NULL;
}

/**
 * Allocate memory and load a file
 * @param filename Filename specified by path
 * @return File data buffer pointer
 */
char *
load_file (const char *const filename)
{
  Uint32 filesize;
  return loadfile (filename, &filesize);
}

/**
 * Allocate memory and load a file (filename with a language code)
 * @param filename Filename specified by path
 * @param fsize Pointer on the size of file which will be loaded
 * @return Pointer to the file data
 */
char *
loadfile_with_lang (const char *const filename, Uint32 * const fsize)
{
  char *data, *fname, *lang;
  if(filename == NULL || strlen (filename) == 0)
    {
      fprintf (stderr, "(!)tools.c/loadfile_with_lang(): "
               "NULL string\n");
      return NULL;
    }

  fname = memory_allocation (strlen (filename) + 1);
  if (fname == NULL)
    {
      fprintf (stdout, "(!)tools.c/loadfile_with_lang(): "
               "'%s' out of memory\n", filename);
      return NULL;
    }
  strcpy (fname, filename);
  lang = configfile_get_lang ();
  sprintf (fname, filename, lang);
#ifdef VERBOSE
  if (power_conf->verbose > 0)
    {
      fprintf (stdout, "- tools.c/loadfile_with_lang(): "
               "file %s was loaded in memory\n",
               fname);
    }
#endif
  data = loadfile (fname, fsize);
  free_memory (fname);
  return data;
}

/**
 * Allocate memory and load a file (filename with a number)
 * @param filename Filename specified by path
 * @param num Interger to convert in string
 * @return Pointer to the file data
 */
char *
loadfile_num (const char *const filename, Sint32 num)
{
  char *data, *fname;

  if(filename == NULL || strlen (filename) == 0)
    {
      fprintf (stderr, "(!)tools.c/loadfile_num(): "
               "NULL string\n");
      return NULL;
    }

  fname = memory_allocation (strlen (filename) + 1);
  if (fname == NULL)
    {
      fprintf (stderr, "(!)tools.c/loadfile_num (%s, %i): "
        "not enough memory to allocate %i bytes\n",
        filename, num, strlen (filename) + 1);
      return NULL ;
    }
  sprintf (fname, filename, num);
/*
#ifdef VERBOSE
  if (power_conf->verbose > 0)
    fprintf (stdout, "tools.c/loadfile_num: file %s was loaded in memory\n",
             fname);
#endif
*/
  data = load_file (fname);
  free_memory (fname);
  return data;
}

/**
 * Allocate memory and load a file there
 * @param filename the file which should be loaded
 * @param fsize pointer on the size of file which will be loaded
 * @return file data buffer pointer
 */
char *
loadfile (const char *const filename, Uint32 * const fsize)
{
  char *buffer;
  char *pathname = locate_data_file (filename);
  if (pathname == NULL)
    {
      fprintf (stderr, "(!)tools.c/loadfile(): can't locate file %s\n",
               filename);
      return NULL;
    }
  buffer = load_absolute_file (pathname, fsize);
  if (buffer == NULL)
    {
      free_memory (pathname);
      return NULL;
    }
  free_memory (pathname);
  return buffer;
}

/**
 * Allocate memory and load a file there
 * @param filename the file which should be loaded
 * @param fsize pointer on the size of file which will be loaded
 * @return file data buffer pointer
 */
char*
load_absolute_file (const char *const filename, Uint32 * const fsize)
{
  Sint32 fhandle;
  char *buffer;
  struct stat sb;
#ifdef WIN32
  fhandle = open (filename, O_RDONLY | O_BINARY, 0);
#else
  fhandle = open (filename, O_RDONLY, 0);
#endif
  if (fhandle == -1)
    {
      fprintf (stderr, "(!)tools.c/load_absolute_file() "
               "can't open file  %s (%s)\n",
               filename, strerror (errno));
      return NULL;
    }
  if (fstat (fhandle, &sb))
    {
      fprintf (stderr, "(!)tools.c/load_absolute_file() "
               "can't stat file  %s (%s)\n",
               filename, strerror (errno));
      close (fhandle);
      return NULL;
    }
  (*fsize) = sb.st_size;
  if (sb.st_size == 0)
    {
      close (fhandle);
      fprintf (stderr, "(!)tools.c/load_absolute_file() "
               "file %s is empty!\n",
               filename);
      return NULL;
    }
  buffer = memory_allocation (sb.st_size);
  if (buffer == NULL)
    {
      fprintf (stderr, "(!)tools.c/load_absolute_file(%s) "
               "not enough memory to allocate %i bytes!\n",
               filename,
               (Sint32)sb.st_size);
      close (fhandle);
      return NULL;
    }
  if (read (fhandle, buffer, sb.st_size) !=
      sb.st_size)
    {
      free_memory (buffer);
      fprintf (stderr, "(!)tools.c/load_absolute_file() "
               "can't read file %s (%s)\n",
               filename, strerror (errno));
      close (fhandle);
      return NULL;
    }
  close (fhandle);
#ifdef VERBOSE
  if (power_conf->verbose > 1)
    {
      fprintf (stdout, "- tools.c/load_absolute_file() "
               "file %s was loaded in memory\n",
               filename);
    }
#endif
  return buffer;
}

/**
 * Load a file in memory buffer already allocated
 * @param filename the file which should be loaded
 * @param buffer pointer to the buffer where data are stored
 * @return boolean value on success or failure
 */
bool
loadfile_into_buffer (const char *const filename, char *const buffer)
{
  Sint32 fhandle;
  struct stat sb;
  char *pathname = locate_data_file (filename);
  if (pathname == NULL)
    {
      fprintf (stderr, "(!)tools.c/loadfile_into_buffer(): "
               "can't locate file: %s\n", filename);
      return FALSE;
    }
#ifdef WIN32
  fhandle = open (pathname, O_RDONLY | O_BINARY, 0);
#else
  fhandle = open (pathname, O_RDONLY, 0);
#endif
  if (fhandle == -1)
    {
      fprintf (stderr, "(!)tools.c/loadfile_into_buffer(): "
               "can't open file: %s (%s)\n", pathname, strerror (errno));
      free_memory (pathname);
      return FALSE;
    }
  if (fstat (fhandle, &sb))
    {
      fprintf (stderr,  "(!)tools.c/loadfile_into_buffer(): "
               "can't stat file %s: %s\n", pathname, strerror (errno));
      close (fhandle);
      free_memory (pathname);
      return FALSE;
    }
  if (sb.st_size == 0)
    {
      close (fhandle);
      fprintf (stderr, "(!)tools.c/tools.c/loadfile_into_buffer() "
               "file %s is empty!\n",
               filename);
      return FALSE;
    }
  if (read (fhandle, buffer, sb.st_size) != sb.st_size)
    {
      fprintf (stderr, "(!)tools.c/loadfile_into_buffer(): "
               "can't read file %s: %s\n", pathname, strerror (errno));
      close (fhandle);
      free_memory (pathname);
      return FALSE;
    }
  close (fhandle);
#ifdef VERBOSE
  if (power_conf->verbose > 1)
    {
      fprintf (stdout, "- tools.c/loadfile_into_buffer(): "
               "file %s was loaded in memory\n", pathname);
    }
#endif
  free_memory (pathname);
  return TRUE;
}

/**
 * Initialize ticks counters
 */
void
fps_init (void)
{
#ifdef POWERMANGA_SDL
  time_begin = SDL_GetTicks ();
  ticks_previous = SDL_GetTicks ();
#else
  gettimeofday (&time_begin, NULL);
  gettimeofday (&ticks_previous, NULL);
#endif
  loops_counter = 0;
}

/**
 * Draw informations on framerate and Linux system
 */
void
fps_print (void)
{
#ifdef POWERMANGA_SDL
  double fps;
  unsigned long duration;
  Sint32 time_end;
  time_end = SDL_GetTicks ();
  duration = time_end - time_begin;
  fps = (1000.0 * loops_counter) / duration;
  if (power_conf->verbose)
    {
      printf ("> tools.c/number of loops : %i\n", loops_counter);
      printf ("> tools.c/running time    : %li\n", duration);
      printf
        ("> tools.c/fps_print()   : frames per seconde : %g \n",
         fps);
    }
#else
  struct utsname kernel;
  struct stat statmem;
  unsigned long duration;
  double fps;
  double mem;
  char os_name[32];
  char os_vers[32];
  char cpu[64];
  char freq[32];
  FILE *cpuinfo;
  char txt[256];
  static struct timeval time_end;

  /* total time of execution */
  gettimeofday (&time_end, NULL);
  duration =
    (time_end.tv_sec - time_begin.tv_sec) * 1000 + (time_end.tv_usec -
                                                    time_begin.tv_usec) /
    1000;
  fps = (1000.0 * loops_counter) / duration;

  /* Linux kernel version */
  if (uname (&kernel) < 0)
    {
      strcpy (os_name, "?");
      strcpy (os_vers, "");
    }
  else
    {
      strncpy (os_name, kernel.sysname, 32);
      strncpy (os_vers, kernel.release, 32);
    }
  /* CPU and physic memory */
  stat ("/proc/kcore", &statmem);
  mem = ((float) statmem.st_size) / 1024 / 1024;
  strcpy (cpu, "Unknown");
  strcpy (freq, "Unknown");
  cpuinfo = fopen ("/proc/cpuinfo", "r");
  if (cpuinfo != NULL)
    {
      while (fgets (txt, 255, cpuinfo))
        {
          if (!strncmp (txt, "model", 5))
            {
              int i = 0;
              while (txt[i] != ':')
                i++;
              i += 2;
              for (int j = 0; j < 64;)
                {
                  if (txt[i++] != '\n')
                    cpu[j++] = txt[i - 1];
                }
            }
          if (!strncmp (txt, "cpu MHz", 7))
            {
              int i = 0;
              while (txt[i] != ':')
                i++;
              i += 2;
              sprintf (freq, "%d", atoi (txt + i));
            }
        }
    }
  if (power_conf->verbose)
    {
      printf ("operating system   : %s %s\n", os_name, os_vers);
      printf ("processor          : %s at %s Mhz with %.0f RAM\n", cpu, freq,
              mem);
      printf ("number of loops    : %i\n", loops_counter);
      printf ("running time       : %li\n", duration);
      printf ("frames per seconde : %g \n", fps);
    }
#endif
}

/**
 * Sleep for a time interval
 * @input delay
 * @return
 */
Sint32
wait_next_frame (Sint32 delay, Sint32 max)
{
#ifdef POWERMANGA_SDL
  if (delay > max)
    {
      delay = max;
    }
  if (delay > 0)
    {
      SDL_Delay (delay);
    }
  return (delay > 0 ? delay : 0);
#else
  struct timeval temps;
  if (delay > max)
    {
      delay = max;
    }
  if (delay > 0)
    {
      temps.tv_usec = delay % (unsigned long) 1000000;
      temps.tv_sec = delay / (unsigned long) 1000000;
      select (0, NULL, NULL, NULL, &temps);
    }
  return (delay > 0 ? delay : 0);
#endif
}

#ifdef POWERMANGA_SDL
/**
 * Calculate diffence between 2 times
 * @return difference
 */
Sint32
get_time_difference ()
{
  Sint32 diff;
  Uint32 current_ticks;
  current_ticks = SDL_GetTicks ();
  diff = current_ticks - ticks_previous;
  ticks_previous = current_ticks;
  return diff;
}
#else
/**
 * Calculate diffence between 2 times
 * @return difference
 */
Sint32
get_time_difference ()
{
  struct timeval current_ticks;
  Sint32 diff;
  gettimeofday (&current_ticks, NULL);
  diff =
    (1000000 * (current_ticks.tv_sec - ticks_previous.tv_sec)) +
    (current_ticks.tv_usec - ticks_previous.tv_usec);
  ticks_previous = current_ticks;
  return diff;
}
#endif

/**
 * Check if a value is null, signed or unsigned
 * @return 0 if value is null, -1 if signed, or 1 otherwise
 */
Sint16
sign(float val)
{
  if(val == 0)
    {
      return 0;
    }
  if(val < 0)
    {
      return -1;
    }
  return 1;
}

/**
 * Calculate a shot angle from two points
 * @param pxs X coordinate of the source
 * @param pys Y coordinate of the source
 * @param pxd X coordinate of the destination 
 * @param pyd Y coordinate of the destination
 * @return angle
 */
float
calc_target_angle(Sint16 pxs, Sint16 pys, Sint16 pxd, Sint16 pyd)
{
  double length, result, dx, dy, angle;
  /* calculate distance between source and destination */
  dx = pxd - pxs;
  dy = pyd - pys;
  result = (dx * dx) + (dy * dy);
  length = sqrt(result);
  if(length == 0)
    {
      /* avoid division by zero */
      return 0.0;
    }
  result = dx / length;
  angle = acos(result);
  result = dy / length;
  if(asin(result) < 0)
    {
      angle = -angle;
    }
  return (float)angle;
}

/** 
 * Calculate a new angle from another angle and some deftness
 * @param old_angle
 * @param new_angle
 * @param deftness
 * @return New angle
 */
float
get_new_angle(float old_angle, float new_angle, float deftness)
{
  float delta_angle;
  delta_angle = old_angle - new_angle;
  if(((delta_angle < 0) && (delta_angle > -3.14)) || (delta_angle > 3.14))
  {
    old_angle = old_angle + deftness;
    if(old_angle > 3.14)
      {
        old_angle = old_angle - 6.28f;
      }
  }
  if(((delta_angle > 0) && (delta_angle < 3.14)) || (delta_angle < -3.14))
  {
    old_angle = old_angle - deftness;
    if(old_angle < -3.14)
      {
        old_angle = old_angle + 6.28f;
      }
  }
  return old_angle;
}

/**
 * Allocates memory and coopies into it the string addressed
 * @param Null-terminated string to duplicate.
 * @return Pointer to a newly allocated copy of the string or NULL 
 */
char*
string_duplicate(register const char* str)
{
  register char* new_str;
  register Uint32 size;
  size = strlen(str) + 1;
  new_str = memory_allocation (size);
  if (new_str == NULL)
    {
      fprintf (stderr, "(!)tools.c/string_duplicate() "
               "not enough memory to allocate %i bytes\n",
               size);
      return NULL;
    }
  memcpy (new_str, str, size);
  return new_str;
}

/**
 * Allocate and precalculate sinus and cosinus curves 
 * @return TRUE if it completed successfully or FALSE otherwise
 */
bool 
alloc_precalulate_sinus (void)
{
  Uint32 i;
  double a, step, pi;
  
  if (precalc_sin128 == NULL)
    {
      i = 128 * sizeof(float) * 2;
      precalc_sin128 = (float *) memory_allocation (i); 
      if (precalc_sin128 == NULL)
        {
          fprintf (stderr,
                   "(!)tools.c/alloc_precalulate_sinus(): "
                   "not enough memory to allocate %i bytes\n", i);
          return FALSE;
        }
      precalc_cos128 = precalc_sin128 + 128;
    }
  pi = 4 * atan (1.0);
  a = 0.0;
  step = (pi * 2) / 128;
  for (i = 0; i < 128; i++)
    {
     precalc_sin128[i] = (float)sin(a);
     precalc_cos128[i] = (float)cos(a);
     a += step;
    }
  if (precalc_sin == NULL)
    {
      i = 32 * sizeof(float) * 2;
      precalc_sin = (float *) memory_allocation (i); 
      if (precalc_sin == NULL)
        {
          fprintf (stderr,
                   "(!)tools.c/alloc_precalulate_sinus(): "
                   "not enough memory to allocate %i bytes\n", i);
          return FALSE;
        }
      precalc_cos = precalc_sin + 32;
    }

  step = (pi * 2) / 32;
  a = 0.0;
  for (i = 0; i < 32; i++)
    {
     precalc_sin[i] = (float)sin(a);
     precalc_cos[i] = (float)cos(a);
     a += step;
    }
  /* table user for guided missile */
  for (i = 0; i < 32; i++)
    {
      depix[0][i] = precalc_cos[i] * 0.5f;
      depix[1][i] = precalc_cos[i] * 1.0f;
      depix[2][i] = precalc_cos[i] * 1.5f;
      depix[3][i] = precalc_cos[i] * 2.0f;
      depix[4][i] = precalc_cos[i] * 2.5f;
      depix[5][i] = precalc_cos[i] * 3.0f;
      depix[6][i] = precalc_cos[i] * 3.5f;
      depix[7][i] = precalc_cos[i] * 4.0f;
      depix[8][i] = precalc_cos[i] * 4.5f;
      depix[9][i] = precalc_cos[i] * 5.0f;
      depix[10][i] = precalc_cos[i] * 5.5f;
      depix[11][i] = precalc_cos[i] * 6.0f;
      depix[12][i] = precalc_cos[i] * 6.5f;
      depiy[0][i] = precalc_sin[i] * 0.5f;
      depiy[1][i] = precalc_sin[i] * 1.0f;
      depiy[2][i] = precalc_sin[i] * 1.5f;
      depiy[3][i] = precalc_sin[i] * 2.0f;
      depiy[4][i] = precalc_sin[i] * 2.5f;
      depiy[5][i] = precalc_sin[i] * 3.0f;
      depiy[6][i] = precalc_sin[i] * 3.5f;
      depiy[7][i] = precalc_sin[i] * 4.0f;
      depiy[8][i] = precalc_sin[i] * 4.5f;
      depiy[9][i] = precalc_sin[i] * 5.0f;
      depiy[10][i] = precalc_sin[i] * 5.5f;
      depiy[11][i] = precalc_sin[i] * 6.0f;
      depiy[12][i] = precalc_sin[i] * 6.5f;
    }
  return TRUE;
}

/**
 * Release precalculded sinus and cosinus curves
 */
void 
free_precalulate_sinus (void)
{
  if (precalc_sin != NULL)
    {
      free_memory ((char *)precalc_sin);
      precalc_sin = NULL;
      precalc_cos = NULL;
    }
  if (precalc_sin128 != NULL)
    {
      free_memory ((char *)precalc_sin128);
      precalc_sin128 = NULL;
      precalc_cos128 = NULL;
    }
}
