/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <sys/sysinfo.h>

#include <glib.h>
#include <gio/gio.h>

#include "powerd-internal.h"
#include "powerd-dbus.h"
#include "log.h"

#define USECS_PER_SEC 1000000
#define NSECS_PER_USEC 1000

struct client_stats {
    const char *dbus_name;
    GSList *sys_stats;
    GSList *disp_stats;
};

struct sys_request_stats {
    const char *name;
    unsigned active_count;
    guint64 active_time;
    guint64 max_active_time;
    guint64 active_since;
};

GHashTable *stats_hash;

static guint64 get_usecs(void)
{
    struct timespec ts;

    if (clock_gettime(CLOCK_MONOTONIC, &ts)) {
        powerd_error("Could not get monotonic time: %s", strerror(errno));
        return 0;
    }
    return (guint64)ts.tv_sec * USECS_PER_SEC +
           (ts.tv_nsec + NSECS_PER_USEC / 2) / NSECS_PER_USEC;
}

static struct client_stats *alloc_client(const char *dbus_name)
{
    struct client_stats *client = calloc(1, sizeof(*client));
    if (!client) {
        powerd_error("Error allocating client for statistics");
        return NULL;
    }

    client->dbus_name = strdup(dbus_name);
    if (!client->dbus_name) {
        powerd_error("Error duplicating client dbus name for statistics");
        free(client);
        return NULL;
    }

    return client;
}

static struct sys_request_stats *alloc_sys_stats(const char *name)
{
    struct sys_request_stats *stats = calloc(1, sizeof(*stats));
    if (!stats) {
        powerd_error("Error allocating memory for system request statistics");
        return NULL;
    }

    stats->name = strdup(name);
    if (!stats->name) {
        powerd_error("Error duplicating request name for statistics");
        free(stats);
        return NULL;
    }

    return stats;
}

static gint sys_stats_comp(gconstpointer a, gconstpointer b)
{
    const struct sys_request_stats *stats = a;
    const char *name = b;
    return strcmp(stats->name, name);
}

static struct client_stats *lookup_client(const char *dbus_name,
                                          gboolean alloc)
{
    struct client_stats *client;

    if (!stats_hash)
        return NULL;

    client = g_hash_table_lookup(stats_hash, dbus_name);
    if (!client && alloc) {
        client = alloc_client(dbus_name);
        if (client)
            g_hash_table_insert(stats_hash, (gpointer)client->dbus_name,
                                client);
    }
    return client;
}

void powerd_account_request_sys_state(const char *dbus_name, const char *name)
{
    struct client_stats *client;
    struct sys_request_stats *stats;
    GSList *item;

    client = lookup_client(dbus_name, TRUE);
    if (!client)
        return;

    item = g_slist_find_custom(client->sys_stats, name, sys_stats_comp);
    if (item) {
        stats = item->data;
    } else {
        stats = alloc_sys_stats(name);
        if (!stats)
            return;
        client->sys_stats = g_slist_prepend(client->sys_stats, stats);
    }

    stats->active_count++;
    stats->active_since = get_usecs();
}

void powerd_account_clear_sys_state(const char *dbus_name, const char *name)
{
    struct client_stats *client;
    struct sys_request_stats *stats;
    GSList *item;
    guint64 duration;

    if (!stats_hash)
        return;

    client = lookup_client(dbus_name, FALSE);
    if (!client)
        return;
    item = g_slist_find_custom(client->sys_stats, name, sys_stats_comp);
    if (!item)
        return;
    stats = item->data;

    duration = get_usecs() - stats->active_since;
    stats->active_since = 0;
    stats->active_time += duration;
    if (duration > stats->max_active_time)
        stats->max_active_time = duration;
}

static void log_sys_req_stats(struct client_stats *client)
{
    const char *dbus_name = client->dbus_name;
    GSList *cur;
    guint64 us;

    us = get_usecs();
    for (cur = client->sys_stats; cur; cur = g_slist_next(cur)) {
        struct sys_request_stats *stats = cur->data;
        guint64 active_time, max_active_time;
       
        /*
         * Aggregate currently held requests into active_time, and
         * consider whether this is greater than the current
         * max_active_time.
         */
        active_time = stats->active_time;
        max_active_time = stats->max_active_time;
        if (stats->active_since != 0) {
            guint64 duration = us - stats->active_since;
            active_time += duration;
            if (duration > max_active_time)
                max_active_time = duration;
        }
        powerd_info("  Name: %s", stats->name);
        powerd_info("    Owner: %s", dbus_name);
        powerd_info("    Active Count: %u", stats->active_count);
        powerd_info("    Active Time: %f", (double)active_time / 1000000.0f);
        powerd_info("    Max Active Time: %f",
                    (double)max_active_time / 1000000.0f);
        powerd_info("    Active Since: %f",
                    (double)stats->active_since / 1000000.0f);
    }
}

void powerd_log_stats(void)
{
    struct sysinfo si;
    GHashTableIter iter;
    gpointer key, value;

    if (!sysinfo(&si))
        powerd_info("Uptime: %ld", si.uptime);

    if (stats_hash) {
        powerd_info("System Request Statistics:");
        g_hash_table_iter_init(&iter, stats_hash);
        while (g_hash_table_iter_next(&iter, &key, &value))
            log_sys_req_stats(value);
    }
}

static void sys_stats_list_destroy(gpointer data)
{
    struct sys_request_stats *stats = data;
    free((void *)stats->name);
    free(stats);
}

static void stats_hash_destroy(gpointer data)
{
    struct client_stats *client = data;
    g_slist_free_full(client->sys_stats, sys_stats_list_destroy);
    free((void *)client->dbus_name);
    free(client);
}

int powerd_stats_init(void)
{
    stats_hash = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
                                       stats_hash_destroy);
    if (!stats_hash) {
        powerd_warn("Unable to allocate stats hash table");
        return -ENOMEM;
    }
    return 0;
}

void powerd_stats_deinit(void)
{
    if (stats_hash)
        g_hash_table_destroy(stats_hash);
}

/* Dbus interface support */

static int build_sys_request_list(GVariantBuilder *builder,
                                  struct client_stats *client)
{
    const char *dbus_name = client->dbus_name;
    GSList *cur;
    guint64 us;
    int count = 0;

    us = get_usecs();
    for (cur = client->sys_stats; cur; cur = g_slist_next(cur)) {
        struct sys_request_stats *stats = cur->data;
        guint64 active_time, max_active_time;
       
        /*
         * Aggregate currently held requests into active_time, and
         * consider whether this is greater than the current
         * max_active_time.
         */
        active_time = stats->active_time;
        max_active_time = stats->max_active_time;
        if (stats->active_since != 0) {
            guint64 duration = us - stats->active_since;
            active_time += duration;
            if (duration > max_active_time)
                max_active_time = duration;
        }
        g_variant_builder_add(builder, "(ssuttt)", dbus_name, stats->name,
                              stats->active_count, active_time,
                              max_active_time, stats->active_since);
        count++;
    }
    return count;
}

gboolean handle_get_sys_request_stats(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation)
{
    GVariantBuilder *builder;
    GVariant *list, *tuple;
    GHashTableIter iter;
    gpointer key, value;
    int count = 0;

    builder = g_variant_builder_new(G_VARIANT_TYPE("a(ssuttt)"));
    if (stats_hash) {
        g_hash_table_iter_init(&iter, stats_hash);
        while (g_hash_table_iter_next(&iter, &key, &value))
            count += build_sys_request_list(builder, value);
    }

    if (count == 0) {
        g_variant_builder_clear(builder);
        list = g_variant_new_array(G_VARIANT_TYPE("a(ssuttt)"), NULL, 0);
    } else {
        list = g_variant_builder_end(builder);
    }
    tuple = g_variant_new_tuple(&list, 1);
    g_dbus_method_invocation_return_value(invocation, tuple);
    g_variant_builder_unref(builder);

    return TRUE;
}

