/***************************************************************************
 *
 * pm-main.c : PowerManager DBUS Server
 *
 * Copyright (C) 2005 Richard Hughes, <richard@hughsie.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 **************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <dbus/dbus-glib.h>
#include "pm-main.h"
int daemon (int nochdir, int noclose);

typedef struct PMObject PMObject;
typedef struct PMObjectClass PMObjectClass;
GType pm_object_get_type (void);
struct PMObject {GObject parent;};
struct PMObjectClass {GObjectClass parent;};
#define PM_TYPE_OBJECT				(pm_object_get_type ())
#define PM_OBJECT(object)			(G_TYPE_CHECK_INSTANCE_CAST ((object), PM_TYPE_OBJECT, PMObject))
#define PM_OBJECT_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST ((klass), PM_TYPE_OBJECT, PMObjectClass))
#define PM_IS_OBJECT(object)		(G_TYPE_CHECK_INSTANCE_TYPE ((object), PM_TYPE_OBJECT))
#define PM_IS_OBJECT_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass), PM_TYPE_OBJECT))
#define PM_OBJECT_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS ((obj), PM_TYPE_OBJECT, PMObjectClass))
G_DEFINE_TYPE(PMObject, pm_object, G_TYPE_OBJECT)

gboolean pm_object_shutdown (PMObject *obj, gboolean *ret, GError **error);
gboolean pm_object_restart (PMObject *obj, gboolean *ret, GError **error);
gboolean pm_object_hibernate (PMObject *obj, gboolean *ret, GError **error);
gboolean pm_object_suspend (PMObject *obj, gboolean *ret, GError **error);
gboolean pm_object_is_active (PMObject *obj, gboolean *ret, GError **error);
gboolean pm_object_hdparm (PMObject *obj, gint timeout, const char *device, gboolean *ret, GError **error);

#include "power-manager-glue.h"

static void pm_object_init (PMObject *obj) { }
static void pm_object_class_init (PMObjectClass *klass) { }

/** run a program/script async.
 *
 *  @param  path		Script to run.
 *  @return				Success
 */
static gboolean
run_script (const char *path)
{
	g_debug ("Executing '%s'", path);
	if (!g_spawn_command_line_async (path, NULL)) {
		g_warning ("Couldn't execute '%s'.", path);
		return FALSE;
	}
	return TRUE;
}

/** hard-drives have a slightly odd calculation for the hard drive timeout ID,
 *  work them out them here.
 *
 *  @param  seconds		Number of seconds to spin down
 *  @return				Nearest ID to the time that is specified
 */
static int
get_hdparm_spindownID (int seconds)
{
	/*
	 * I.D.			| Timeout
	 * -------------|------------------------------------------------------
	 * 000			| off	
	 * 001-240		| 5sec	to	20min 		(multiple of 5 seconds)
	 * 241-251		| 30min	to	5.5hours	(multiple of 30 minutes)
	 * 252 			| 21 minutes.
	 * 253			| Vendor-defined timeout period between 8 and 12 hours
	 * 254			| reserved.
	 * 255			| 21 minutes plus 15 seconds.
	 */
	if (seconds < 5)
		return 0;
	else if (seconds >= 5 && seconds <= 1200)
		return seconds / 5;
	else if (seconds > 1200 && seconds < 1800)
		return 241;	/* Cannot achieve state */
	else if (seconds >= 1800 && seconds < 19800)
		return seconds / 1800;
	else if (seconds >= 19800)
		return 251;
	return 0;
}

/** do shutdown on object
 *
 *  @return	ret			Success
 */
gboolean
pm_object_shutdown (PMObject *obj, gboolean *ret, GError **error)
{
	g_debug ("pm_object_shutdown");
	*ret = run_script (SBINDIR "/pmi action shutdown");
	return TRUE;
}

/** do restart on object
 *
 *  @return	ret			Success
 */
gboolean
pm_object_restart (PMObject *obj, gboolean *ret, GError **error)
{
	g_debug ("pm_object_restart");
	*ret = run_script (SBINDIR "/pmi action restart");
	return TRUE;
}

/** do hibernate on object
 *
 *  @return	ret			Success
 */
gboolean
pm_object_hibernate (PMObject *obj, gboolean *ret, GError **error)
{
	g_debug ("pm_object_hibernate");
	*ret = run_script (SBINDIR "/pmi action hibernate");
	return TRUE;
}

/** do suspend on object
 *
 *  @return	ret			Success
 */
gboolean
pm_object_suspend (PMObject *obj, gboolean *ret, GError **error)
{
	g_debug ("pm_object_suspend");
	*ret = run_script (SBINDIR "/pmi action suspend");
	return TRUE;
}

/** Find out if this service is active
 *
 *  @return	ret			Success
 */
gboolean
pm_object_is_active (PMObject *obj, gboolean *ret, GError **error)
{
	g_debug ("pm_object_is_active");
	*ret = TRUE;
	return TRUE;
}

/** do shutdown on object
 *
 *  @param	timeout		timout for spindown in seconds
 *  @param	device		the device, e.g. /dev/hda
 *  @return	ret			Success
 */
gboolean
pm_object_hdparm (PMObject *obj, gint timeout, const char *device, gboolean *ret, GError **error)
{
	g_debug ("pm_object_hdparm (%i, %s)", timeout, device);

	int hdparmID = get_hdparm_spindownID (timeout * 60);
	g_debug ("hdparm ID = %i", hdparmID);

	GString *gs = g_string_new ("bug");
	g_string_printf (gs, "%s/%s -S %i %s", SBINDIR, "hdparm", hdparmID, device);
	*ret = run_script (gs->str);
	g_string_free (gs, TRUE);

	return TRUE;
}

/** Prints program usage.
 *
 */
static void print_usage (void)
{
	g_print ("\nusage : PowerManager [options]\n");
	g_print ("\n"
		 " Options\n"
		 "    --help             Show this information and exit\n"
		 "    --nodaemon         Do not daemonise\n"
		 "\n");
}

/** Entry point
 *
 *  @param  argc		Number of arguments given to program
 *  @param  argv		Arguments given to program
 *  @return				Return code
 */
int
main (int argc, char **argv)
{
	DBusGConnection *bus;
	DBusGProxy *bus_proxy;
	GError *error = NULL;
	PMObject *obj;
	GMainLoop *mainloop;

	g_type_init ();

	dbus_g_object_type_install_info (PM_TYPE_OBJECT, &dbus_glib_pm_object_object_info);

	mainloop = g_main_loop_new (NULL, FALSE);

	bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (!bus)
		g_error ("Couldn't connect to session bus: %s", error->message);

	bus_proxy = dbus_g_proxy_new_for_name (bus, 
		DBUS_SERVICE_DBUS,
		DBUS_PATH_DBUS,
		DBUS_INTERFACE_DBUS);

	guint request_name_result;
	if (!dbus_g_proxy_call (bus_proxy, "RequestName", &error,
		G_TYPE_STRING, PM_DBUS_SERVICE,
		G_TYPE_UINT, DBUS_NAME_FLAG_PROHIBIT_REPLACEMENT,
		G_TYPE_INVALID,
		G_TYPE_UINT, &request_name_result,
		G_TYPE_INVALID))
	g_error ("Failed to acquire %s: %s", PM_DBUS_SERVICE, error->message);
	obj = g_object_new (PM_TYPE_OBJECT, NULL);
	dbus_g_connection_register_g_object (bus, PM_DBUS_PATH, G_OBJECT (obj));

	int a;
	gboolean daemonise = TRUE;
	for (a=1; a < argc; a++) {
		if (strcmp (argv[a], "--help") == 0) {
			print_usage ();
			return 1;
		} else if (strcmp (argv[a], "--nodaemon") == 0)
			daemonise = FALSE;
	}

	if (daemonise && !daemon (0,0))
		g_warning ("Unable to daemonize");

	g_message ("PowerManager service running");

	g_main_loop_run (mainloop);

	dbus_g_connection_unref (bus);
	return 0;
}
