#include <config.h>

#include <unistd.h>
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>

#include <pork.h>
#include <pork_util.h>
#include <pork_list.h>
#include <pork_queue.h>
#include <pork_inet.h>
#include <pork_acct.h>
#include <pork_proto.h>
#include <pork_io.h>
#include <pork_imwindow.h>
#include <pork_screen_cmd.h>
#include <pork_chat.h>
#include <pork_irc.h>

static void irc_event(int sock, u_int32_t cond, void *data) {
	if (cond & IO_COND_READ)
		irc_input_dispatch(data);

	irc_flush_outq(data);
}

static void irc_connected(int sock, u_int32_t cond, void *data) {
	int ret;
	irc_session_t *session = data;
	struct pork_acct *acct = session->data;

	pork_io_del(data);

	ret = sock_is_error(sock);
	if (ret != 0) {
		screen_err_msg("Fail");
	} else {
		session->sock = sock;
		sock_setflags(sock, 0);
		pork_io_add(sock, IO_COND_READ, data, data, irc_event);
		irc_callback_add_defaults(session);
		irc_send_login(session);
	}
}

static int irc_init(struct pork_acct *acct) {
	irc_session_t *session = xcalloc(1, sizeof(*session));
	char *ircname;

	ircname = getenv("IRCNAME");
	if (ircname != NULL)
		acct->profile = xstrdup(ircname);
	else {
		if (acct->profile == NULL)
			acct->profile = xstrdup(DEFAULT_IRC_PROFILE);
	}

	irc_callback_init(session);
	session->outq = queue_new(0);
	session->inq = queue_new(0);

	session->data = acct;
	acct->data = session;
	return (0);
}

static int irc_free(struct pork_acct *acct) {
	irc_session_t *session = acct->data;

	free(session->server);
	free(session->chanmodes);
	free(session->chantypes);

	irc_callback_clear(session);

	queue_destroy(session->inq, free);
	queue_destroy(session->outq, free);

	free(session);
	return (0);
}

static int irc_update(struct pork_acct *acct) {
	return (0);
}

static int irc_read_config(struct pork_acct *acct) {
	return (0);
}

static int irc_write_config(struct pork_acct *acct) {
	return (0);
}

static int irc_do_connect(struct pork_acct *acct, char *args) {
	irc_session_t *session = acct->data;
	char *server;
	char *port;
	int sock;
	int ret;

	if (args == NULL)
		return (-1);

	server = strsep(&args, " ");
	if (server == NULL)
		return (-1);

	if (args != NULL)
		port = args;
	else
		port = DEFAULT_IRC_PORT;

	ret = irc_connect(server, port, &sock);
	if (ret == 0)
		irc_connected(sock, 0, session);
	else if (ret == -EINPROGRESS)
		pork_io_add(sock, IO_COND_WRITE, session, session, irc_connected);
	else
		return (-1);

	return (0);
}

static int irc_join(struct pork_acct *acct, char *chan) {
	return (irc_send_join(acct->data, chan, NULL));
}

static int irc_privmsg(struct pork_acct *acct, char *dest, char *msg) {
	char *p;

	p = strchr(dest, ',');
	if (p != NULL)
		*p = '\0';

	return (irc_send_privmsg(acct->data, dest, msg));
}

static int irc_chat_send(	struct pork_acct *acct,
							struct chatroom *chat,
							char *msg)
{
	return (irc_send_privmsg(acct->data, chat->title, msg));
}

static int chat_find_compare_cb(void *l, void *r) {
	char *str = l;
	struct chatroom *chat = r;

	return (strcasecmp(str, chat->title));
}

static struct chatroom *irc_find_chat(struct pork_acct *acct, char *chat) {
	dlist_t *node;

	node = dlist_find(acct->chat_list, chat, chat_find_compare_cb);
	if (node == NULL)
		return (NULL);

	return (node->data);
}

static int irc_whois(struct pork_acct *acct, char *dest) {
	char *p;

	while ((p = strsep(&dest, ",")) != NULL)
		irc_send_whois(acct->data, p);

	return (0);
}

static int irc_whowas(struct pork_acct *acct, char *dest) {
	char *p;

	while ((p = strsep(&dest, ",")) != NULL)
		irc_send_whowas(acct->data, p);

	return (0);
}

static int irc_chat_get_name(const char *str, char *buf, size_t len) {
	int ret;
	char *p;

	if (xstrncpy(buf, str, len) == -1)
		return (-1);

	p = strchr(buf, ',');
	if (p != NULL)
		*p = '\0';

	return (0);
}

static int irc_change_nick(struct pork_acct *acct, char *nick) {
	irc_session_t *irc = acct->data;

	if (irc->nick_len) {
		if (strlen(nick) > irc->nick_len) {
			screen_err_msg("Error: Nick is too long. Maximum length is %d",
				irc->nick_len);

			return (-1);
		}
	}

	if (!acct->connected) {
		free(acct->username);
		acct->username = xstrdup(nick);
	}

	return (irc_send_nick(acct->data, nick));
}

static int irc_part(struct pork_acct *acct, struct chatroom *chat) {
	return (irc_send_part(acct->data, chat->title));
}

static int irc_chat_users(struct pork_acct *acct, struct chatroom *chat) {
	return (irc_send_names(acct->data, chat->title));
}

static int irc_chat_kick(	struct pork_acct *acct,
							struct chatroom *chat,
							char *user,
							char *reason)
{
	if (reason == NULL)
		reason = "No reason given";

	return (irc_send_kick(acct->data, chat->title, user, reason));
}

static int irc_chat_notice(	struct pork_acct *acct,
							struct chatroom *chat,
							char *msg)
{
	return (irc_send_notice(acct->data, chat->title, msg));
}

static int irc_notice(struct pork_acct *acct, char *dest, char *msg) {
	return (irc_send_notice(acct->data, dest, msg));
}

static int irc_who(struct pork_acct *acct, char *str) {
	return (irc_send_who(acct->data, str));
}

static int irc_quit(struct pork_acct *acct, char *reason) {
	if (acct->connected)
		return (irc_send_quit(acct->data, reason));

	return (-1);
}

static int irc_is_chat(struct pork_acct *acct, char *str) {
	irc_session_t *irc = acct->data;
	char *p;

	if (irc->chantypes == NULL) {
		if (*str == '#' || *str == '&')
			return (1);
		return (0);
	}

	for (p = irc->chantypes ; p != '\0' ; p++) {
		if (*str == *p)
			return (1);
	}

	return (0);
}

static int irc_quote(struct pork_acct *acct, char *str) {
	char *p = str;

	if (str == NULL)
		return (-1);

	while (*p == ' ')
		p++;

	if (!strncasecmp(p, "NICK ", 5)) {
		screen_err_msg("Use the /nick command.");
		return (-1);
	}

	return (irc_send(acct->data, str));
}

static int irc_away(struct pork_acct *acct, char *msg) {
	return (irc_set_away(acct->data, msg));
}

static int irc_back(struct pork_acct *acct) {
	return (irc_set_away(acct->data, NULL));
}

static int irc_topic(	struct pork_acct *acct,
						struct chatroom *chat,
						char *topic)
{
	return (irc_send_topic(acct->data, chat->title, topic));
}

char *irc_text_filter(char *str) {
	static const char *mirc_fg_col = "wwbgrymyYGcCBMDW";
	static const char *mirc_bg_col = "ddbgrymyygccbmww";
	static const char *ansi_esc_col = "drgybmcwDRGYBMCW";
	size_t len = strlen(str) + 1024;
	char *ret;
	size_t i;
	int fgcol = 7;
	int bgcol = -1;

	ret = malloc(len);

	len--;
	for (i = 0 ; i < len && *str != '\0' ;) {
		switch (*str) {
			case '%':
				if (i + 2 >= len)
					goto out;
				memcpy(&ret[i], "%%", 2);
				i += 2;
				str++;
				break;

			/* ^B */
			case 0x02:
				if (i + 2 >= len)
					goto out;
				memcpy(&ret[i], "%1", 2);
				i += 2;
				str++;
				break;

			/* ^V */
			case 0x16:
				if (i + 2 >= len)
					goto out;
				memcpy(&ret[i], "%2", 2);
				i += 2;
				str++;
				break;

			/* ^_ */
			case 0x1f:
				if (i + 2 >= len)
					goto out;
				memcpy(&ret[i], "%3", 2);
				i += 2;
				str++;
				break;

			/* ^C - mirc color code */
			case 0x03:
			{
				int fgcol = -1;
				int bgcol = -1;
				char colbuf[4];

				if (!isdigit(str[1])) {
					if (i + 2 >= len)
						goto out;
					memcpy(&ret[i], "%x", 2);
					i += 2;
					break;
				}
				str++;

				memcpy(colbuf, str, 2);
				colbuf[2] = '\0';

				if (isdigit(colbuf[1]))
					str += 2;
				else
					str++;

				fgcol = strtol(colbuf, NULL, 10) % 16;

				if (*str == ',') {
					memcpy(colbuf, &str[1], 2);
					colbuf[2] = '\0';

					if (isdigit(colbuf[0])) {
						if (isdigit(str[2]))
							str += 3;
						else
							str += 2;

						bgcol = strtol(colbuf, NULL, 10) % 16;
					}
				}

				if (i + 2 >= len)
					goto out;
				ret[i++] = '%';
				ret[i++] = mirc_fg_col[fgcol];

				if (bgcol >= 0) {
					if (i + 2 >= len)
						goto out;
					ret[i++] = ',';
					ret[i++] = mirc_bg_col[bgcol];
				}
			}

			/* ^[ - ANSI escape sequence */
			case 0x1b:
			{
				char *end;
				char *p;
				int bold = 0;
				char buf[64];
				int slen;

				buf[0] = '\0';
				if (str[1] != '[')
					goto add;

				end = strchr(&str[2], 'm');
				if (end == NULL)
					goto add;
				*end++ = '\0';

				str += 2;
				while ((p = strsep(&str, ";")) != NULL) {
					char *n;
					int num;

					num = strtoul(p, &n, 10);
					if (*n != '\0')
						continue;

					switch (num) {
						/* foreground color */
						case 30 ... 39:
							fgcol = num - 30;
							break;

						/* background color */
						case 40 ... 49:
							bgcol = num - 40;
							break;

						/* bold */
						case 1:
							bold = 8;
							break;

						/* underscore */
						case 4:
							if (xstrncat(buf, "%3", sizeof(buf)) == -1)
								goto out;
							break;

						/* blink */
						case 5:
							if (xstrncat(buf, "%4", sizeof(buf)) == -1)
								goto out;
							break;

						/* reverse */
						case 7:
							if (xstrncat(buf, "%2", sizeof(buf)) == -1)
								goto out;
							break;

						/* clear all attributes */
						case 0:
							if (xstrncat(buf, "%x", sizeof(buf)) == -1)
								goto out;
							fgcol = -1;
							bgcol = -1;
							bold = 0;
							break;
					}
				}

				if (fgcol >= 0 || bgcol >= 0) {
					if (fgcol < 0)
						fgcol = 7;

					if (i + 2 >= len)
						goto out;

					ret[i++] = '%';
					ret[i++] = ansi_esc_col[fgcol + bold];

					if (bgcol >= 0) {
						if (i + 2 >= len)
							goto out;
						ret[i++] = ',';
						ret[i++] = ansi_esc_col[bgcol];
					}
				} else if (bold) {
					if (xstrncat(buf, "%1", sizeof(buf)) == -1)
						goto out;
				}

				slen = xstrncat(ret, buf, len - i);
				if (slen == -1)
					goto out;
				i += slen;
				str = end;
				break;
			}

			default:
			add:
				ret[i++] = *str++;
				break;
		}
	}
out:

	ret[i] = '\0';
	return (ret);
}

int irc_proto_init(struct pork_proto *proto) {
	proto->protocol = PROTO_IRC;
	xstrncpy(proto->name, "IRC", sizeof(proto->name));

	proto->chat_join = irc_join;
	proto->chat_send = irc_chat_send;
	proto->chat_find = irc_find_chat;
	proto->chat_name = irc_chat_get_name;
	proto->chat_leave = irc_part;
	proto->chat_users = irc_chat_users;
	proto->chat_kick = irc_chat_kick;
	proto->chat_send_notice = irc_chat_notice;
	proto->chat_set_topic = irc_topic;

	proto->get_profile = irc_whois;
	proto->connect = irc_do_connect;
	proto->free = irc_free;
	proto->init = irc_init;
	proto->who = irc_who;
	proto->whowas = irc_whowas;
	proto->send_notice = irc_notice;
	proto->signoff = irc_quit;
	proto->normalize = xstrncpy;
	proto->read_config = irc_read_config;
	proto->send_msg = irc_privmsg;
	proto->update = irc_update;
	proto->write_config = irc_write_config;
	proto->user_compare = strcasecmp;
	proto->change_nick = irc_change_nick;
	proto->filter_text = irc_text_filter;
	proto->quote = irc_quote;
	proto->is_chat = irc_is_chat;
	proto->set_away = irc_away;
	proto->set_back = irc_back;

	return (0);
}
