/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-ui-findrep-options.h"
#include "polyxmass-findrepopt.h"
#include "polyxmass-ui-findrep.h"
#include "polyxmass-ui-seqed-widget.h"
#include "polyxmass-pixbuf-rendering.h"

/*
  static gboolean is_finding;
  static gboolean is_replacing;
*/


GtkWidget *
polyxmass_findrep_opt_setup_wnd (GtkWidget *seqed_widget)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  
  GtkWidget *seqed_widget_find = NULL;
  GtkWidget *seqed_widget_replace = NULL;

  PxmFindrepOpt *findrepopt = NULL;
  
  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
    

  g_assert (seqed_widget != NULL);

  gui_file = 
    g_strdup_printf ("%s/polyxmass-find-replace.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "find_replace_options_wnd", 
		       PACKAGE);

  g_free (gui_file);

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "find_replace_options_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the find/replace "
		    "options window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }


  /* Most crucial datum to set right away.
   */
  g_object_set_data (G_OBJECT (window), "seqed_widget", seqed_widget);


  /* We'll certainly have to display messages, so here is our
     standard...
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), 
		     "messages_entry", widget);
  

  /* We'll need the options for the find/replace activities to be
     available throughout the life of the window.
  */
  findrepopt = polyxmass_findrepopt_new ();
  g_object_set_data_full (G_OBJECT (window), "findrepopt", findrepopt,
			  (GDestroyNotify) polyxmass_findrepopt_free);
    
  
  /* Make a string out of the seqed_widget pointer value.
   */  
  help = g_strdup_printf ("%p", widget);
  widget = glade_xml_get_widget (xml, "polseq_id_number_entry");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_id_number_entry", seqed_widget);
  
  /* Display that value and free the allocated string.
   */
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  /* Set the name of the polymer sequence in its GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);

  g_assert (PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->name != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->name);

  /* Set the code of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_code_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_code_entry", widget);

  g_assert (PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->code != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->code);




  /****** CREATION OF THE FIND SEQED_WIDGET AND RELATED STUFF *******/

  /* The vbox where we'll pack the seqed_widget_find PxmSeqedWidget object.
   */
  widget = glade_xml_get_widget (xml, "seqed_widget_holder_for_find_vbox");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "seqed_widget_holder_for_find_vbox", widget);


  seqed_widget_find = polyxmass_seqed_widget_new ();
  g_assert (seqed_widget_find != NULL);

  g_object_set_data (G_OBJECT (window), 
		     "seqed_widget_find", seqed_widget_find);

  gtk_container_add (GTK_CONTAINER (widget), seqed_widget_find);
  
  /* We should standardize on the fact that the window that uses that widget
     will set a datum to the widget with a pointer to itself under the 
     name "parent_wnd".
  */
  g_object_set_data (G_OBJECT (seqed_widget_find), "parent_wnd", window);
  

  /* We have to create a polymer object, that will belong to this
     specific seqed_widget_find instance, where all the monomers that
     are typed in are stored ! Note that we initialize the 'type' of
     the polymer with the value of the polymer in the seqed_widget:
     logically these should be identical!
  */
  PXM_SEQED_WIDGET (seqed_widget_find)->polymer = pxmchem_polymer_new ();

  pxmchem_polymer_plminfo_set_type (PXM_SEQED_WIDGET (seqed_widget_find)->polymer->plminfo,
				    PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->type);
  
  pxmchem_polymer_plminfo_set_name (PXM_SEQED_WIDGET (seqed_widget_find)->polymer->plminfo,
				    _("Not set"));
  
  pxmchem_polymer_plminfo_set_code (PXM_SEQED_WIDGET (seqed_widget_find)->polymer->plminfo,
				    _("Not set"));

  /* Since we do not allocate, each time we use it, a polchemdefctxt
     instance, we increment the reference count to let the program
     know it is used once more.
  */
  PXM_SEQED_WIDGET (seqed_widget_find)->polchemdefctxt = 
    PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  polyxmass_polchemdefctxt_ref (PXM_SEQED_WIDGET (seqed_widget_find)->polchemdefctxt);

  g_assert (TRUE == 
	    polyxmass_seqed_widget_post_pack_constructor (PXM_SEQED_WIDGET (seqed_widget_find)));


  findrepopt->findseqGPA = PXM_SEQED_WIDGET (seqed_widget_find)->polymer->monomerGPA;


  /**** END CREATION OF THE FIND SEQED_WIDGET AND RELATED STUFF ****/

  


  /****** CREATION OF THE REPLACE SEQED_WIDGET AND RELATED STUFF *******/

  /* The vbox where we'll pack the seqed_widget_replace PxmSeqedWidget object.
   */
  widget = glade_xml_get_widget (xml, "seqed_widget_holder_for_replace_vbox");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "seqed_widget_holder_for_replace_vbox", widget);


  seqed_widget_replace = polyxmass_seqed_widget_new ();
  g_assert (seqed_widget_replace != NULL);

  g_object_set_data (G_OBJECT (window), 
		     "seqed_widget_replace", seqed_widget_replace);

  gtk_container_add (GTK_CONTAINER (widget), seqed_widget_replace);
  
  /* We should standardize on the fact that the window that uses that widget
     will set a datum to the widget with a pointer to itself under the 
     name "parent_wnd".
  */
  g_object_set_data (G_OBJECT (seqed_widget_replace), "parent_wnd", window);  


  /* We have to create a polymer object, that will belong to this
     specific seqed_widget_replace instance, where all the monomers that
     are typed in are stored ! Note that we initialize the type of the
     polymer with the value of the polymer in the seqed_widget: logically
     these should be identical!
  */
  PXM_SEQED_WIDGET (seqed_widget_replace)->polymer = pxmchem_polymer_new ();

  pxmchem_polymer_plminfo_set_type (PXM_SEQED_WIDGET (seqed_widget_replace)->polymer->plminfo,
				    PXM_SEQED_WIDGET (seqed_widget)->polymer->plminfo->type);
  
  pxmchem_polymer_plminfo_set_name (PXM_SEQED_WIDGET (seqed_widget_replace)->polymer->plminfo,
				    _("Not set"));
  
  pxmchem_polymer_plminfo_set_code (PXM_SEQED_WIDGET (seqed_widget_replace)->polymer->plminfo,
				    _("Not set"));

  /* Since we do not allocate, each time we use it, a polchemdefctxt
     instance, we increment the reference count to let the program
     know it is used once more.
  */
  PXM_SEQED_WIDGET (seqed_widget_replace)->polchemdefctxt = 
    PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  polyxmass_polchemdefctxt_ref (PXM_SEQED_WIDGET (seqed_widget_replace)->polchemdefctxt);

  g_assert (TRUE == 
	    polyxmass_seqed_widget_post_pack_constructor (PXM_SEQED_WIDGET (seqed_widget_replace)));


  findrepopt->repseqGPA = PXM_SEQED_WIDGET (seqed_widget_replace)->polymer->monomerGPA;


  /**** END CREATION OF THE REPLACE SEQED_WIDGET AND RELATED STUFF ****/

  
  widget = glade_xml_get_widget (xml, "find_options_start_at_point_checkbutton");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_options_start_at_point_checkbutton", widget);
  
  widget = glade_xml_get_widget (xml, "find_options_backward_checkbutton");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_options_backward_checkbutton", widget);

 
  widget = glade_xml_get_widget (xml, "find_strictness_identical_radiobutton");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_strictness_identical_radiobutton", widget);
  

  widget = glade_xml_get_widget (xml, "find_strictness_subset_radiobutton");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_strictness_subset_radiobutton", widget);
  


  widget = glade_xml_get_widget (xml, "replace_strictness_identical_radiobutton");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "replace_strictness_identical_radiobutton", widget);
  

  widget = glade_xml_get_widget (xml, "replace_strictness_subset_radiobutton");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "replace_strictness_subset_radiobutton", widget);
  

  

  /* Actions buttons.
   */
  widget = glade_xml_get_widget (xml, "find_options_find_button");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_options_find_button", widget);
  
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxmass_findrep_opt_wnd_find_button),
		    window);

  widget = glade_xml_get_widget (xml, "find_options_next_button");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_options_next_button", widget);
  
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxmass_findrep_opt_wnd_next_button),
		    window);

  widget = glade_xml_get_widget (xml, "find_options_replace_button");
  g_assert (widget != NULL);  
  g_object_set_data (G_OBJECT (window), 
		     "find_options_replace_button", widget);
  
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxmass_findrep_opt_wnd_replace_button),
		    window);


  gtk_widget_show_all (GTK_WIDGET (window));
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  /* The signal of the window itself.
   */
  /* Signal / callback connections.
   */
  g_signal_connect 
    (G_OBJECT (window),
     "delete_event",
     G_CALLBACK (polyxmass_findrep_opt_wnd_delete_event), 
     NULL);

  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one cleavage options' window opened for
     a given polymer seqence editing window, and we do not want that
     the second window destroys the datum name of the first window, so
     we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("findrep_options_wnd-%p", window);
  
  g_object_set_data_full (G_OBJECT (seqed_widget),
			  help, GTK_WIDGET (window), 
			  (GDestroyNotify) polyxmass_findrep_opt_wnd_really_close);
  
  g_free (help);

  return window;
}



void
polyxmass_findrep_opt_wnd_find_button (GtkWidget *widget, gpointer data)
{
  GtkWidget *start_at_point_checkbutton = NULL;
  GtkWidget *backward_checkbutton = NULL;
  GtkWidget *button = NULL;
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;

  gint result = -1;
  
  PxmFindrepOpt *findrepopt = NULL;
  
  
  /* Certainly it would not make any sense to find an empty
     motif. Thence, it is required that the seqed_widget_find at least
     has one monomer in it !

     Whe the user clicks the "Find" button, that means that she wants
     to start the search from the beginning.
   */  
  
  g_assert (window != NULL);

  
  /* Note that the findrepopt object was allocated at construction of
     the window. It is reused each time a button is clicked because it
     contains values that have to be conserved through operation of
     the window button (like knowing where the last found sequence is
     located for the case in which the user clicks "Next", for
     example).
  */
  findrepopt = g_object_get_data (G_OBJECT (window), "findrepopt");
  g_assert (findrepopt != NULL);
  
  if (findrepopt->findseqGPA->len <= 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					 _("The find sequence is empty"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return;
    }



  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  backward_checkbutton = 
    g_object_get_data (G_OBJECT (window),
		       "find_options_backward_checkbutton");
  g_assert (backward_checkbutton != NULL);
  
  start_at_point_checkbutton =
    g_object_get_data (G_OBJECT (window),
		       "find_options_start_at_point_checkbutton");
  g_assert (start_at_point_checkbutton != NULL);
  


  /* Note that we are dealing with the Find button, and not the Next
     button. 
     
     That means that we have a number of different cases : 

     0. The user just wants to (re)start Find to the first monomer of
        the polymer sequence. Set findrepopt->current_idx to 0.

     1. If the user wants that the Find operation starts at a given
        monomer position in the sequence (typically the point, in
        emacs terminology, that is where the cursor is located), then
        she has checked find_options_start_at_point_checkbutton. In
        this case, we set findrepopt->current_idx to the index of
        point.

     2. If the user wants that the Find operation be backward, then
        she has checked find_options_backward_checkbutton. Of course,
        this only makes sense if the user also has checked
        find_options_start_at_point_checkbutton, because otherwise it
        would be impossible to go left of the first monomer in the
        sequence (remember we are in the Find button handler and not
        in the Next one).

	Below: remember that the last_point_1_idx used below is
	greater by one unit with respect to the true value (this is
	due to graphics necessities and is well documented elsewhere).
   */

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (start_at_point_checkbutton)))
    findrepopt->current_idx = PXM_SEQED_WIDGET (seqed_widget)->last_point_1_idx - 1;
  else
    findrepopt->current_idx = 0;
  

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (backward_checkbutton)))
    {
      /* We only find that requirement meaningful if the user also
	 asks that the Find be performed at cursor point.
      */
      if (FALSE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (start_at_point_checkbutton)))
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					    _("'Backward' is not allowed without 'Start At Point'"),
					    POLYXMASS_MEDI_MSG_TIMEOUT);
	  return;
	}
      else
	findrepopt->backward = TRUE;
    }
  else
    findrepopt->backward = FALSE;
      
  /* Strictness stuff related to the finding / replacing operations matches.
   */
  button = g_object_get_data (G_OBJECT (window),
			      "find_strictness_identical_radiobutton");
  g_assert (button != NULL);
  
  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    findrepopt->find_strictness |= PXM_FIND_STRICT_IDENTICAL;
  else
    findrepopt->find_strictness |= PXM_FIND_STRICT_SUBSET;
  

  button = g_object_get_data (G_OBJECT (window),
			      "replace_strictness_identical_radiobutton");
  g_assert (button != NULL);
  
  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    findrepopt->replace_strictness |= PXM_REPLACE_STRICT_IDENTICAL;
  else
    findrepopt->replace_strictness |= PXM_REPLACE_STRICT_SUBSET;
  

  /* We are in the find-only function handler.
   */
  findrepopt->find_only = TRUE;
  
  /* We are set to launch an actual find operation... When that
     operation will be finished, then the findrepopt object will
     contain anything required for the "Next" button to perform as
     expected.
   */
  result = polyxmass_findrep_find_in_seqed_widget_with_findrepopt (seqed_widget, findrepopt);
  
  if (result <= 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("No occurrence was found"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return;
    }
  else
    {
      /* We have to show the found sequence by selecting it and
	 making sure the canvas is scrolled so that it is indeed
	 visible.
      */
      if (FALSE == polyxmass_seqed_widget_select_sequence (seqed_widget,            
							   findrepopt->find_start_idx,
							   findrepopt->find_end_idx))
	polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					  _("Failed to select the found sequence"),
					  POLYXMASS_MEDI_MSG_TIMEOUT);
      
      return;
    }
}


void
polyxmass_findrep_opt_wnd_next_button (GtkWidget *widget, gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;
  GtkWidget *button = NULL;
  

  gint result = -1;

  PxmFindrepOpt *findrepopt = NULL;
  
  
  /* When the user clicks on the next button, that means that she
     wants to find next occurrence of a sequence motif. Just get the
     findrepopt structure and do the work! 

     Note that "next occurrence" is relative to the fact that
     find_options_backward_checkbutton might be checked or not. If it
     is checked, then the next occurrence, if any, will be at a
     polymer sequence index lower than present.
  */

  /* Certainly it would not make any sense to find an empty
     motif. Thence, it is required that the seqed_widget_find at least
     has one monomer in it !

     Whe the user clicks the "Find" button, that means that she wants
     to start the search from the beginning.
   */  



  g_assert (window != NULL);
  
  /* Note that the findrepopt object was allocated at construction of
     the window. It is reused each time a button is clicked because it
     contains values that have to be conserved through operation of
     the window button (like knowing where the last found sequence is
     located for the case in which the user clicks "Next", for
     example).
  */
  findrepopt = g_object_get_data (G_OBJECT (window), "findrepopt");
  g_assert (findrepopt != NULL);
  
  if (findrepopt->findseqGPA->len <= 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					 _("The find sequence is empty"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return;
    }

  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);


  button = g_object_get_data (G_OBJECT (window), "find_options_backward_checkbutton");
  g_assert (button != NULL);
  
  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
    {
      result = polyxmass_findrep_find_backward_in_seqed_widget_with_findrepopt (seqed_widget,
										findrepopt);
    }
  else
    {
      result = polyxmass_findrep_find_forward_in_seqed_widget_with_findrepopt (seqed_widget,
									       findrepopt);
    }
  
  if (result <= 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("No occurrence was found"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return;
    }
  else
    {
      /* We have to show the found sequence by selecting it and
	 making sure the canvas is scrolled so that it is indeed
	 visible.
      */
      if (FALSE == polyxmass_seqed_widget_select_sequence (seqed_widget,            
							   findrepopt->find_start_idx,
							   findrepopt->find_end_idx))
	{
	  g_critical (_("%s@%d: failed to select the found sequence\n"),
		      __FILE__, __LINE__);
	  
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					    _("Failed to select the found sequence"),
					    POLYXMASS_MEDI_MSG_TIMEOUT);

	  /* We have to set the different values to a non-found
	     situation because the situation here is faulty and we do
	     not want that a fake replace operation might take place.
	   */
	  findrepopt->find_start_idx = -1;
	  findrepopt->find_end_idx = -1;
	  findrepopt->current_idx = -1;
	}
      
      return;
    }

return;
}




void
polyxmass_findrep_opt_wnd_replace_button (GtkWidget *widget, gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;
  gint sel_start_idx = -1;
  gint sel_end_idx = -1;
  
  gint result = -1;
  
  PxmFindrepOpt *findrepopt = NULL;
  
  /* The replace operation only can happen once a sequence has been
     successfully found and shown by selection. Only at that precise
     moment, can a sequence be replaced. And only at that precise
     moment the findrepopt->find_start_idx and
     findrepopt->find_end_idx should be different to -1 and also
     findrepopt->current_idx. Because a sequence successfully found is
     selected, we have the possibility to ensure that we are doing a
     replace on a bona fide found sequence.
  */
  g_assert (window != NULL);

  findrepopt = g_object_get_data (G_OBJECT (window), "findrepopt");
  g_assert (findrepopt != NULL);

  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  

  /* Make sure we actually have some found sequence data...
   */
  if (findrepopt->find_start_idx == -1 
      || findrepopt->find_end_idx == -1
      || findrepopt->current_idx == -1)
    {
      g_warning (_("%s@%d: failed to replace a sequence not previously found\n"),
		 __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to replace a sequence not previously found"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return ;
    }
  
  /* When a sequence is found, it get automagically selected, so we
     really should get selection indices corresponding to the found sequence.
  */
  if (FALSE == polyxmass_seqed_widget_get_selection_indices (seqed_widget,
							     &sel_start_idx,
							     &sel_end_idx))
    {
      g_warning (_("%s@%d: failed to get found selection indices\n"),
		 __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to get found selection indices"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return ;
    }
      
  /* Below: remember that the end_idx returned by the function call
     above is greater by one unit with respect to the true value (this
     is due to graphics necessities and is well documented elsewhere).
   */
  if (sel_start_idx != findrepopt->find_start_idx
      || sel_end_idx -1 != findrepopt->find_end_idx)
    {
      g_warning (_("%s@%d: current selection is not the found sequence\n"),
		 __FILE__, __LINE__);
      
      /*
      debug_printf (("sel_start_idx =  %d ; findrepopt->find_start_idx = %d ;"
		     " sel_end_idx = %d ; findrepopt->find_end_idx = %d\n",
		     sel_start_idx, findrepopt->find_start_idx, 
		     sel_end_idx, findrepopt->find_end_idx));
      */

      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to get found selection indices"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return ;
    }
  
  /* At this point we certainly do have a properly set environment to
     be confident that we are replacing the right sequence !
   */
  result = polyxmass_findrep_replace_in_seqed_widget_with_findrepopt (seqed_widget, findrepopt);
  
  if (result <= 0)
    {
      g_critical (_("%s@%d: failed to perform the replacement\n"),
		  __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Failed to perform the replacement"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return ;
    }


  /* We ought to automatically perform a find operation so that the
     replacing operation performs smoothly.
   */
  polyxmass_findrep_opt_wnd_next_button (NULL, (gpointer) window);
  
  
  /* We have to make sure that the masses are updated and that the sequence gets redrawn.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (seqed_widget),
						      PXM_MASSCALC_TARGET_BOTH);
  
  polyxmass_seqed_widget_redraw_sequence (seqed_widget);
}



/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxmass_findrep_opt_wnd_really_close (GtkWidget *window)
{
  PxmSeqedWidget *seqed_widget = NULL;
  
  g_assert (window != NULL);
  

  /* This function is called as a GDestroyNotify callback function
     when the seqed_widget on behalf of which it was created gets
     destroyed.
  */
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /* Further, we have to remember that the the two seqed_widgets
     packed in this window have received the pointer of two allocated
     polymer instances.

     Equally important is that the polchemdefctxt of the polymers were
     reffed upon creation of the window, so we should unref this
     polchemdefctxt object once for each seqed_widget!
  */
  
  
  /* The find seqed_widget.
   */
  seqed_widget = PXM_SEQED_WIDGET (g_object_get_data (G_OBJECT (window),
						      "seqed_widget_find"));
  g_assert (seqed_widget != NULL);
  
  g_assert (seqed_widget->polymer != NULL);

  pxmchem_polymer_free (seqed_widget->polymer);
  
  g_assert (seqed_widget->polchemdefctxt != NULL);

  if (-1 == polyxmass_polchemdefctxt_unref (seqed_widget->polchemdefctxt))
    {
      g_critical ("%s@%d: failed to unref polymer chemistry definition"
		  "context from main global array: '%s'\n",
		  __FILE__, __LINE__, 
		  seqed_widget->polchemdefctxt->polchemdef->type);
    }



  /* The replace seqed_widget.
   */
  seqed_widget = PXM_SEQED_WIDGET (g_object_get_data (G_OBJECT (window),
						      "seqed_widget_replace"));
  g_assert (seqed_widget != NULL);
  
  g_assert (seqed_widget->polymer != NULL);

  pxmchem_polymer_free (seqed_widget->polymer);
  
  g_assert (seqed_widget->polchemdefctxt != NULL);

  if (-1 == polyxmass_polchemdefctxt_unref (seqed_widget->polchemdefctxt))
    {
      g_critical ("%s@%d: failed to unref polymer chemistry definition"
		  "context from main global array: '%s'\n",
		  __FILE__, __LINE__, 
		  seqed_widget->polchemdefctxt->polchemdef->type);
    }

  gtk_widget_destroy (window);
}




gboolean
polyxmass_findrep_opt_wnd_delete_event (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  gint count = 0;

  gchar *help = NULL;
    
  PxmSeqedWidget *seqed_widget = NULL;
  
  g_assert (window != NULL);
  
  
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  /*
    debug_printf (("count is %d\n", count));
  */


  /* Further, we have to remember that the the two seqed_widgets
     packed in this window have received the pointer of two allocated
     polymer instances.

     Equally important is that the polchemdefctxt of the polymers were
     reffed upon creation of the window, so we should unref this
     polchemdefctxt object once for each seqed_widget!
  */
  
  
  /* The find seqed_widget.
   */
  seqed_widget = PXM_SEQED_WIDGET (g_object_get_data (G_OBJECT (window),
						      "seqed_widget_find"));
  g_assert (seqed_widget != NULL);
  
  g_assert (seqed_widget->polymer != NULL);

  pxmchem_polymer_free (seqed_widget->polymer);
  
  g_assert (seqed_widget->polchemdefctxt != NULL);

  if (-1 == polyxmass_polchemdefctxt_unref (seqed_widget->polchemdefctxt))
    {
      g_critical ("%s@%d: failed to unref polymer chemistry definition"
		  "context from main global array: '%s'\n",
		  __FILE__, __LINE__, 
		  seqed_widget->polchemdefctxt->polchemdef->type);
    }



  /* The replace seqed_widget.
   */
  seqed_widget = PXM_SEQED_WIDGET (g_object_get_data (G_OBJECT (window),
						      "seqed_widget_replace"));
  g_assert (seqed_widget != NULL);
  
  g_assert (seqed_widget->polymer != NULL);

  pxmchem_polymer_free (seqed_widget->polymer);
  
  g_assert (seqed_widget->polchemdefctxt != NULL);

  if (-1 == polyxmass_polchemdefctxt_unref (seqed_widget->polchemdefctxt))
    {
      g_critical ("%s@%d: failed to unref polymer chemistry definition"
		  "context from main global array: '%s'\n",
		  __FILE__, __LINE__, 
		  seqed_widget->polchemdefctxt->polchemdef->type);
    }

  /* This window pointer was set as a full datum to the seqed_widget ,
     which means that we have to remove that pointer, without
     triggering the callback function call.
  */
  seqed_widget = PXM_SEQED_WIDGET (g_object_get_data (G_OBJECT (window),
						      "seqed_widget"));
  g_assert (seqed_widget != NULL);


  help = g_strdup_printf ("findrep_options_wnd-%p", window);
  
  window = 
    g_object_steal_data (G_OBJECT (seqed_widget),
			 help);
  
  g_free (help);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}

  
