/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "libpolyxmass-config.h"
#include "libpolyxmass-globals.h"


/* 
   Returns the directory name where the GNU polyxmass system configuration
   is located. On normal setups, that is /etc/polyxmass. That directory
   is configure in the configure.in file and "CONFDIR" is defined in
   config.h.
*/
gchar *
libpolyxmass_config_get_system_polyxmass_config_dir (void)
{
  gchar *confdir = NULL;
  
  confdir = g_strdup_printf ("%s", CONFDIR);
  
  g_assert (TRUE == g_file_test (confdir, G_FILE_TEST_IS_DIR));
  
  return confdir;
}


/* 
   Returns the directory name where the GNU polyxmass user's configuration
   is located. On normal setups, that is $HOME/.polyxmass.
*/
gchar *
libpolyxmass_config_get_user_polyxmass_config_dir (void)
{
  gchar *help = NULL;
  gchar *confdir = NULL;


  help = (gchar *) g_get_home_dir () ; /* not allocated ! */
  confdir = g_strdup_printf ("%s/.polyxmass", help);
  /*
    debug_printf (("confdir is %s\n", confdir));
  */

  /* 
     Make sure that this directory exists !
  */
  if (TRUE == g_file_test (confdir, G_FILE_TEST_IS_DIR))
    return confdir;
  
  /* Apparently the user configuration directory is not there, so we
     create it !
  */
  if (0 != g_mkdir (confdir, S_IRWXU))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	     _("%s@%d: failed to create directory: '%s' "
	       "with error: '%s'\n"),
	     __FILE__, __LINE__, confdir, strerror (errno));
      g_free (confdir);
    }
  else
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: created directory: '%s'\n"),
	     __FILE__, __LINE__, confdir);
    }
  /* At this point we can create the two other directories in the
     first one: atom-defs and polchem-defs. Use the help gchar *
     to that end:
  */
  help = g_strdup_printf ("%s/atom-defs", confdir);

  if (0 != g_mkdir (help, S_IRWXU))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	     _("%s@%d: failed to create directory: '%s' "
	       "with error: '%s'\n"),
	     __FILE__, __LINE__, help, strerror (errno));

      g_free (confdir);
      g_free (help);
    }
  else
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: created directory: '%s'\n"),
	     __FILE__, __LINE__, help);

      g_free (help);
    }
      
  help = g_strdup_printf ("%s/polchem-defs", confdir);
      
  if (0 != g_mkdir (help, S_IRWXU))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	     _("%s@%d: failed to create directory: '%s' "
	       "with error: '%s'\n"),
	     __FILE__, __LINE__, help, strerror (errno));

      g_free (confdir);
      g_free (help);
    }
  else
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: created directory: '%s'\n"),
	     __FILE__, __LINE__, help);

      g_free (help);
    }
  
  /*
    At this point we are sure that we are returning the name of an
    existing directory.
  */
  return confdir;
}


/*
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  The /etc/polyxmass/atom-defs/atom-defs-cat and similar files.
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
gchar *
libpolyxmass_config_get_system_atom_defs_cat_dir (void)
{
  gchar *catalogue_dir = NULL;
  
  catalogue_dir = g_strdup_printf ("%s/atom-defs", CONFDIR);
  
  if (FALSE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to test the existence of the directory: "
	       "'%s', please ensure that the GNU polyxmass software suite "
	       "is correctly installed. In particular, is the "
	       "GNU polyxmass-common package installed?\n"),
	     __FILE__, __LINE__, catalogue_dir);

      g_free (catalogue_dir);
      
      return NULL;
    }
    
  return catalogue_dir;
}


gchar *
libpolyxmass_config_get_user_atom_defs_cat_dir (void)
{
  gchar * config_dir = NULL;
  gchar * catalogue_dir = NULL;
  

  /* 
     Get the directory where all the atom definition catalogue files
     are located for the user.
  */
  config_dir = libpolyxmass_config_get_user_polyxmass_config_dir ();
  
  /* 
     Now calculate the filename of the catalogue directory.
  */
  catalogue_dir = g_strdup_printf ("%s/atom-defs", config_dir);
  
  /*
    Free config_dir as we do not need it anymore.
  */
  g_free (config_dir);
  
  /*
    Make sure that the catalogue directory is actually there !
  */
   if (FALSE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
    {
      /* 
	 The directory does not exist. Just tell the user that there
	 is no such catalogue file.
      */
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: catalogue directory not found: '%s'\n"),
	     __FILE__, __LINE__, catalogue_dir);
      
      return NULL;
    }
   
   return catalogue_dir;  
}


/*
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  The /etc/polyxmass/polchem-defs/polchem-defs-cat and similar files.
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
gchar *
libpolyxmass_config_get_system_polchem_defs_cat_dir (void)
{
  gchar *catalogue_dir = NULL;
  
  catalogue_dir = g_strdup_printf ("%s/polchem-defs", CONFDIR);

  if (FALSE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to test the existence of the directory: "
	       "'%s', please ensure that the GNU polyxmass software suite "
	       "is correctly installed. In particular, is the "
	       "GNU polyxmass-common package installed?\n"),
	     __FILE__, __LINE__, catalogue_dir);

      g_free (catalogue_dir);
      
      return NULL;
    }
    
  return catalogue_dir;
}

  
gchar *
libpolyxmass_config_get_user_polchem_defs_cat_dir (void)
{
  gchar * config_dir = NULL;
  gchar * catalogue_dir = NULL;
  

  /* 
     Get the directory where all the polchem-defs-cat polymer
     chemistry definition catalogue files are located for the user.
  */
  config_dir = libpolyxmass_config_get_user_polyxmass_config_dir ();
  
  /* 
     Now calculate the filename of the catalogue directory.
  */
  catalogue_dir = g_strdup_printf ("%s/polchem-defs", config_dir);
  
  /*
    Free config_dir as we do not need it anymore.
  */
  g_free (config_dir);
  
  /*
    Make sure that the catalogue directory is actually there !
  */
   if (FALSE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
    {
      /* 
	 The directory does not exist. Just tell the user that there
	 is no such catalogue file.
      */
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: catalogue directory not found: '%s'\n"),
	     __FILE__, __LINE__, catalogue_dir);
      
      return NULL;
    }
   
   return catalogue_dir;  
}


/*
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  The /etc/polyxmass/polchem-defs/polchem-defs-atom-defs-dic and
  similar files.
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
gchar *
libpolyxmass_config_get_system_polchem_defs_atom_defs_dic_dir (void)
{
  gchar *catalogue_dir = NULL;
  
  catalogue_dir = g_strdup_printf ("%s/polchem-defs", CONFDIR);
  
  if (FALSE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: Failed to test the existence of the directory: "
	       "%s, please ensure that the GNU polyxmass software suite "
	       "is correctly installed. In particular, is the "
	       "GNU polyxmass-common package installed?\n"),
	     __FILE__, __LINE__, catalogue_dir);

      g_free (catalogue_dir);
      
      return NULL;
    }
  
  return catalogue_dir;
}

  
gchar *
libpolyxmass_config_get_user_polchem_defs_atom_defs_dic_dir (void)
{
  gchar * config_dir = NULL;
  gchar * catalogue_dir = NULL;
  

  /* 
     Get the directory where all the atom definition catalogue files
     are located for the user. This directory name is allocated.
  */
  config_dir = libpolyxmass_config_get_user_polyxmass_config_dir ();
  
  /* 
     Now calculate the filename of the dictionary directory.
  */
  catalogue_dir = g_strdup_printf ("%s/polchem-defs", config_dir);
  
  /*
    Free config_dir as we do not need it anymore.
  */
  g_free (config_dir);
  
  /*
    Make sure that the catalogue directory is actually there !
  */
   if (FALSE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
    {
      /* 
	 The directory does not exist. Just tell the user that there
	 is no such catalogue file.
      */
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	     _("%s@%d: catalogue directory not found: '%s'\n"),
	     __FILE__, __LINE__, catalogue_dir);

      g_free (catalogue_dir);
      
      return NULL;
    }
   
   return catalogue_dir;  
}





