/* -*- c++ -*-
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#include "ugameStdAfx.h"
#ifdef WIN32
#include <cstdio>
# define snprintf _snprintf
#else
#include <ugame/BetSlider>
#endif

#include <iostream>

#include <libxml/xmlreader.h>

#include <osg/Notify>
#include <osg/Geode>
#include <osg/PositionAttitudeTransform>
#include <osg/Geometry>
#include <osg/LineWidth>
#include <osg/BlendFunc>

#include <osgText/Font>
#include <osgText/Text>

#include <osgDB/ReadFile>

using namespace betslider;

#define DEFAULT_CHARACTER_SIZE 12.f

//
// Serialization helpers
//

static bool readColorFromXml(xmlTextReaderPtr reader, osg::Vec4& color) {
  bool status = false;
  xmlChar* red = xmlTextReaderGetAttribute(reader, (const xmlChar*)"red");
  if(red) {
    color.x() = atoi((const char*)red) / 255.f;
    xmlFree(red);
    status = true;
  }
  xmlChar* green = xmlTextReaderGetAttribute(reader, (const xmlChar*)"green");
  if(green) {
    color.y() = atoi((const char*)green) / 255.f;
    xmlFree(green);
    status = true;
  }
  xmlChar* blue = xmlTextReaderGetAttribute(reader, (const xmlChar*)"blue");
  if(blue) {
    color.z() = atoi((const char*)blue) / 255.f;
    xmlFree(blue);
    status = true;
  }
  xmlChar* alpha = xmlTextReaderGetAttribute(reader, (const xmlChar*)"alpha");
  if(alpha) {
    color.w() = atof((const char*)alpha);
    xmlFree(alpha);
    status = true;
  } else {
    color.w() = 1.f;
  }
  return status;
}

static bool readImage(osg::Geode* geode, const char* path, float width, float height, const osgDB::ReaderWriter::Options* options) {
  osg::Image* image = osgDB::readImageFile(path, options);
  if(image && image->s() > 0 && image->t() > 0) {
    float x = width / 2.f;
    float y = height / 2.f;

    osg::Texture2D* texture = new osg::Texture2D;
    texture->setImage(image);

    osg::StateSet* state = new osg::StateSet;
    state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
    state->setTextureAttributeAndModes(0, texture);
    osg::BlendFunc* blendFunc = new osg::BlendFunc();
    blendFunc->setFunction(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    state->setAttributeAndModes(blendFunc);
    state->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);

    osg::Geometry* geometry = new osg::Geometry;
    geometry->setStateSet(state);

    osg::Vec3Array* vertexes = new osg::Vec3Array(4);
    (*vertexes)[0].set(-x, -y, 0.f);
    (*vertexes)[1].set(x, -y, 0.f);
    (*vertexes)[2].set(x, y, 0.f);
    (*vertexes)[3].set(-x, y, 0.f);
    geometry->setVertexArray(vertexes);

    osg::Vec2Array* texcoords = new osg::Vec2Array(4);
    (*texcoords)[0].set(0.f, 0.f);
    (*texcoords)[1].set(1.f, 0.f);
    (*texcoords)[2].set(1.f, 1.f);
    (*texcoords)[3].set(0.f, 1.f);
    geometry->setTexCoordArray(0, texcoords);

    osg::Vec4Array* colours = new osg::Vec4Array(1);
    (*colours)[0].set(1.0f,1.0f,1.0,1.0f);
    geometry->setColorArray(colours);
    geometry->setColorBinding(osg::Geometry::BIND_OVERALL);

    geometry->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::QUADS, 0, 4));

    geode->addDrawable(geometry);

    return true;
  }
  return false;
}

static bool readTextColorFromXml(xmlTextReaderPtr reader, const char* end_tag, int index, int row, BetSlider* betslider) {
  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT &&
       (!strcmp(end_tag, (const char*)xmlTextReaderConstName(reader))))
      return true;

    osg::Vec4 color;
    if(!strcmp("normal", (const char*)xmlTextReaderConstName(reader))) {
      readColorFromXml(reader, color);
      betslider->setNormalColor(row, index, color);
    } else if(!strcmp("selected", (const char*)xmlTextReaderConstName(reader))) {
      readColorFromXml(reader, color);
      betslider->setSelectedColor(row, index, color);
    }
  }
  return false;
}

static bool readTextIndexFromXml(xmlTextReaderPtr reader, const char* end_tag, int index, BetSlider* betslider, const osgDB::ReaderWriter::Options* options) {
  float font_size = DEFAULT_CHARACTER_SIZE;
  xmlChar* font_size_string = xmlTextReaderGetAttribute(reader, (const xmlChar*)"font_size");
  if(font_size_string) {
    font_size = atof((const char*)font_size_string);
    xmlFree(font_size_string);
  }
  xmlChar* font_path = xmlTextReaderGetAttribute(reader, (const xmlChar*)"font");
  if(font_path) {
    osg::Object* object = osgDB::readObjectFile((const char*)font_path, options);

    osgText::Font* font = dynamic_cast<osgText::Font*>(object);
    if(font) {
      betslider->setFont(index, font, font_size);
    } else {
      if (object && object->referenceCount()==0) object->unref();
    }
    xmlFree(font_path);
  }

  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT &&
       (!strcmp(end_tag, (const char*)xmlTextReaderConstName(reader))))
      return true;
    
    if(!strcmp("cancel", (const char*)xmlTextReaderConstName(reader)))
      readTextColorFromXml(reader, "cancel", index, BetSlider::ROW_CANCEL, betslider);
    else if(!strcmp("call", (const char*)xmlTextReaderConstName(reader)))
      readTextColorFromXml(reader, "call", index, BetSlider::ROW_CALL, betslider);
    else if(!strcmp("raise", (const char*)xmlTextReaderConstName(reader)))
      readTextColorFromXml(reader, "raise", index, BetSlider::ROW_RAISE, betslider);
    else if(!strcmp("raise_max", (const char*)xmlTextReaderConstName(reader)))
      readTextColorFromXml(reader, "raise_max", index, BetSlider::ROW_RAISE_MAX, betslider);
    else if(!strcmp("current_amount", (const char*)xmlTextReaderConstName(reader)))
      readTextColorFromXml(reader, "current_amount", index, BetSlider::ROW_CURRENT_AMOUNT, betslider);
  }
  return false;
}

static bool readTextFromXml(xmlTextReaderPtr reader, BetSlider* betslider, const osgDB::ReaderWriter::Options* options) {
  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT &&
       !strcmp("text", (const char*)xmlTextReaderConstName(reader)))
      return true;
    
    if(!strcmp("left", (const char*)xmlTextReaderConstName(reader)))
      readTextIndexFromXml(reader, "left", BetSlider::SIDE_LEFT, betslider, options);
    else if(!strcmp("right", (const char*)xmlTextReaderConstName(reader)))
      readTextIndexFromXml(reader, "right", BetSlider::SIDE_RIGHT, betslider, options);
  }
  return false;
}

static bool readSliderFromXml(xmlTextReaderPtr reader, BetSlider* betslider, const osgDB::ReaderWriter::Options* options) {
  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT &&
       (!strcmp("betslider", (const char*)xmlTextReaderConstName(reader))))
      return true;
    const char* name = (const char*)xmlTextReaderConstName(reader);

    if(!strcmp("cursor", name)) {
      xmlChar* image_file = xmlTextReaderGetAttribute(reader, (const xmlChar*)"image");
      xmlChar* width = xmlTextReaderGetAttribute(reader, (const xmlChar*)"width");
      xmlChar* height = xmlTextReaderGetAttribute(reader, (const xmlChar*)"height");
      if(image_file && width && height) {
	osg::Geode* geode = new osg::Geode;
	if(readImage(geode, (const char*)image_file, atof((const char*)width), atof((const char*)height), options)) {
	  betslider->setCursor(geode);
	}
      }
      if(image_file) xmlFree(image_file);
      if(width) xmlFree(width);
      if(height) xmlFree(height);
    } else if(!strcmp("separator", name)) {
      xmlChar* image_file = xmlTextReaderGetAttribute(reader, (const xmlChar*)"image");
      xmlChar* width = xmlTextReaderGetAttribute(reader, (const xmlChar*)"width");
      xmlChar* height = xmlTextReaderGetAttribute(reader, (const xmlChar*)"height");
      if(image_file && width && height) {
	osg::Geode* geode = new osg::Geode;
	if(readImage(geode, (const char*)image_file, atof((const char*)width), atof((const char*)height), options)) {
	  betslider->setSeparator(geode);
	}
      }
      if(image_file) xmlFree(image_file);
      if(width) xmlFree(width);
      if(height) xmlFree(height);
    } else if(!strcmp("background", name)) {
      xmlChar* side_string = xmlTextReaderGetAttribute(reader, (const xmlChar*)"side");
      int side = BetSlider::SIDE_RIGHT;
      if(side_string && !strcmp("left", (const char*)side_string)) {
	side = BetSlider::SIDE_LEFT;
	xmlFree(side_string);
      }
      osg::Vec4 color;
      if(readColorFromXml(reader, color))
	betslider->setBackgroundColor(side, color);
    } else if(!strcmp("middle", name)) {
      osg::Vec4 color;
      if(readColorFromXml(reader, color))
	betslider->setMiddleColor(color);
      xmlChar* width = xmlTextReaderGetAttribute(reader, (const xmlChar*)"width");
      if(width) {
	betslider->setMiddleWidth(atoi((const char*)width));
	xmlFree(width);
      }
    } else if(!strcmp("text", name)) {
      readTextFromXml(reader, betslider, options);
    } else if(!strcmp("motor", name)) {
      xmlChar* fixed_length = xmlTextReaderGetAttribute(reader, (const xmlChar*)"fixed_length");
      if(fixed_length) {
	betslider->setMotorFixedLength(atoi((const char*)fixed_length));
	xmlFree(fixed_length);
      }

      xmlChar* variable_length = xmlTextReaderGetAttribute(reader, (const xmlChar*)"variable_length");
      if(variable_length) {
	betslider->setMotorVariableLength(atoi((const char*)variable_length));
	xmlFree(variable_length);
      }
    }
  }
  return false;
}

static bool readbetsliderFromXml(xmlTextReaderPtr reader, BetSlider* betslider, const osgDB::ReaderWriter::Options* options) {

  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    //    const char* fileName = xmlTextReaderCurrentDoc(reader)->name;
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_ELEMENT) {

      const char* name = (const char*)xmlTextReaderConstName(reader);
      if(!strcmp("betslider", name)) {
	xmlChar* margin = xmlTextReaderGetAttribute(reader, (const xmlChar*)"margin");
	if(margin) {
	  betslider->setMargin(atof((const char*)margin));
	  xmlFree(margin);
	}
	readSliderFromXml(reader, betslider, options);
      }
    }
  }
  return status == 0;
}

BetSlider::BetSlider() :
  _middle_color(255.f/255.f, 240.f/255.f, 185.f/255.f, 1.f),
  _middle_width(5.f),
  _rows(ROW_COUNT),
  _height(0.f),
  _motor_span(0.f),
  _motor_position(0.f),
  _motor_fixed_length(10.f),
  _motor_variable_length(100.f),
  _cursor_position(0.f),
  _value_step(1)
{
  _background_color[SIDE_LEFT] = osg::Vec4(85.f/255.f, 85.f/255.f, 85.f/255.f, 1.f),
  _background_color[SIDE_RIGHT] = osg::Vec4(150.f/255.f, 150.f/255.f, 150.f/255.f, 1.f),
  _font_size[SIDE_LEFT] = DEFAULT_CHARACTER_SIZE;
  _font_size[SIDE_RIGHT] = DEFAULT_CHARACTER_SIZE;
  build();
}

BetSlider::BetSlider(const BetSlider& slider, const osg::CopyOp& copyop) :
  osg::Group(slider, copyop)  
{
}

BetSlider::~BetSlider()
{
}

void BetSlider::setFont(int side_index, osgText::Font* font, float font_size)
{
  if(side_index < 0 || side_index >= SIDE_COUNT) {
    osg::notify(osg::WARN) << "BetSlider::setFont: side_index out of range " << side_index << std::endl;
    return;
  }
    
  _font[side_index] = font;
  _font_size[side_index] = font_size;
  for(unsigned int i = 0; i < _rows.size(); i++) {
    if(_rows[i].valid()) {
      if(_rows[i]->_text[side_index].valid()) {
	_rows[i]->_text[side_index]->setFont(font);
	_rows[i]->_text[side_index]->setCharacterSize(font_size);
      }	
    }
  }

  layout();
}

void BetSlider::setNormalColor(int row_index, int side_index, const osg::Vec4& color)
{
  if(row_index < 0 || row_index >= ROW_COUNT) {
    osg::notify(osg::WARN) << "BetSlider::setNormalColor: row_index out of range " << row_index << std::endl;
    return;
  }

  if(side_index < 0 || side_index >= SIDE_COUNT) {
    osg::notify(osg::WARN) << "BetSlider::setNormalColor: side_index out of range " << side_index << std::endl;
    return;
  }

  if(_rows[row_index].valid()) {
    if(_rows[row_index]->_text[side_index].valid())
      _rows[row_index]->_text[side_index]->setColor(color);

    _rows[row_index]->_normal[side_index] = color;
  }
}

void BetSlider::setSelectedColor(int row_index, int side_index, const osg::Vec4& color)
{
  if(row_index < 0 || row_index >= ROW_COUNT) {
    osg::notify(osg::WARN) << "BetSlider::setSelectedColor: row_index out of range " << row_index << std::endl;
    return;
  }

  if(side_index < 0 || side_index >= SIDE_COUNT) {
    osg::notify(osg::WARN) << "BetSlider::setSelectedColor: side_index out of range " << side_index << std::endl;
    return;
  }

  if(_rows[row_index].valid()) {
    _rows[row_index]->_selected[side_index] = color;
  }
}

void BetSlider::setSeparator(osg::Geode* separator)
{
  _separator = new osg::PositionAttitudeTransform;
  _separator->addChild(separator);
  for(int i = 0; i < ROW_COUNT; i++) {
    _rows[ROW_CALL]->setSeparator(_separator.get());
    _rows[ROW_RAISE]->setSeparator(_separator.get());
    _rows[ROW_RAISE_MAX]->setSeparator(_separator.get());
  }
}

void BetSlider::setCursor(osg::Geode* cursor)
{
  _cursor = new osg::PositionAttitudeTransform;
  _cursor->addChild(cursor);
  addChild(_cursor.get());
  updateCursorPosition();
}

void BetSlider::setBackgroundColor(int side_index, const osg::Vec4& background_color)
{
  _background_color[side_index] = background_color;
  osg::Vec4Array* colors = dynamic_cast<osg::Vec4Array*>(_background->getColorArray());
  (*colors)[0] = _background_color[SIDE_LEFT];
  (*colors)[2] = _background_color[SIDE_RIGHT];
  _background->setColorArray(colors);
}


void BetSlider::setMiddleColor(const osg::Vec4& middle_color)
{
  _middle_color = middle_color;
  osg::Vec4Array* colors = dynamic_cast<osg::Vec4Array*>(_background->getColorArray());
  (*colors)[1] = _middle_color;
  _background->setColorArray(colors);
}

void BetSlider::setMiddleWidth(float middle_width)
{
  if(middle_width != _middle_width) {
    _middle_width = middle_width;
    _background->getOrCreateStateSet()->setAttributeAndModes(new osg::LineWidth(_middle_width));
    layout();
  }
}

void BetSlider::setMargin(float margin)
{
  if(margin != _margin) {
    _margin = margin;
    layout();
  }
}

void BetSlider::setLimits(unsigned int call, unsigned int raise, unsigned int raise_max, unsigned int all_in, unsigned int pot, unsigned int step)
{
  _value_step = step;

  char tmp[128];

  float cursor = 0.f;
  _height = 0.f;
  _motor_span = 0.f;
  for(Rows::iterator i = _rows.begin(); i != _rows.end(); i++)
    (*i)->remove();

  Row* row;
  // cancel
  row = _rows[ROW_CANCEL].get();
  row->add();
  row->setText("Cancel", "");
  row->setMotorRange(_motor_span, _motor_fixed_length);
  row->setCursorRange(_height, cursor, 5.f);
  row->setValueRange(0, 0);

  // call
  if(call > 0) {
    row = _rows[ROW_CALL].get();
    row->add();
    if(all_in <= call) {
      snprintf(tmp, 128, "%d", all_in);
      row->setText("Call All In", tmp);
      row->setValueRange(all_in, all_in);
    } else {
      snprintf(tmp, 128, "%d", call);
      row->setText("Call", tmp);
      row->setValueRange(call, call);
    }
    row->setMotorRange(_motor_span, _motor_fixed_length);
    row->setCursorRange(_height, cursor, 5.f);
  }

  if(all_in > call && raise > call) {
    // raise
    row = _rows[ROW_RAISE].get();
    row->add();
    if(all_in <= raise) {
      snprintf(tmp, 128, "%d", all_in);
      if(call > 0)
	row->setText("Raise All In", tmp);
      else
	row->setText("Bet All In", tmp);
      row->setValueRange(all_in, all_in);
    } else {
      snprintf(tmp, 128, "%d", raise);
      if(call > 0)
	row->setText("Raise", tmp);
      else
	row->setText("Bet", tmp);
      row->setValueRange(raise, raise);
    }
    row->setMotorRange(_motor_span, _motor_fixed_length);
    row->setCursorRange(_height, cursor, 5.f);

    if(all_in > raise && raise_max > raise) {
      // raise max
      row = _rows[ROW_RAISE_MAX].get();
      row->add();
      snprintf(tmp, 128, "%d", raise_max);
      if(all_in <= raise_max) {
	snprintf(tmp, 128, "%d", all_in);
	row->setValueRange(raise, all_in);
	if(all_in == pot) {
	  row->setText("Pot All In", tmp);
	} else {
	  row->setText("All In", tmp);
	}
      } else if(raise_max == pot) {
	row->setText("Pot", tmp);
	row->setValueRange(raise, raise_max);
      } else {
	row->setText("Max", tmp);
	row->setValueRange(raise, raise_max);
      }
      row->setMotorRange(_motor_span, _motor_variable_length);
      row->setCursorRange(_height, cursor, 100.f);
   
      Row* row_current = _rows[ROW_CURRENT_AMOUNT].get();
      row_current->add();
      snprintf(tmp, 128, "%d", raise_max); // use raise_max to make sure the maximum width is used
      row_current->setText(tmp, "");
      row_current->_text[SIDE_LEFT]->setPosition(osg::Vec3(0.f, (row->_cursor_start + row->_cursor_end) / 2, .1f));
    }
  }

  _cursor_position = 0.f;
  _motor_position = 0.f;
  _height += 20.f;
  layout();

  setMotorPosition((_rows[ROW_CALL]->_motor_start + _rows[ROW_CALL]->_motor_end) / 2.f);
}

unsigned int BetSlider::moveCursor(float delta)
{
  float motor_delta = delta * _motor_span;
  float motor_position = _motor_position + motor_delta;
  setMotorPosition(motor_position);
  return getCurrentValue();
}

unsigned int BetSlider::getCurrentValue()
{
  Row* row = getCurrentRow();
  if(row->_variable) {
    float fraction = (_cursor_position - row->_cursor_start) / (row->_cursor_end - row->_cursor_start);
    unsigned int value = row->_value_start + (unsigned int)((row->_value_end - row->_value_start) * fraction);
    value -= value % _value_step;
    return value > row->_value_end ? row->_value_end : value;
  } else {
    return row->_value_start;
  }
}

bool BetSlider::unserialize(struct _xmlDoc* doc, const osgDB::ReaderWriter::Options* options) {
  xmlTextReaderPtr reader = xmlReaderWalker(doc);
  if(reader == NULL) return false;
  bool status = readbetsliderFromXml(reader, this, options);
  xmlFreeTextReader(reader);
  return status;
}

bool BetSlider::unserialize(const std::string& fileName, const osgDB::ReaderWriter::Options* options) {
  xmlTextReaderPtr reader = xmlReaderForFile(fileName.c_str(), NULL, XML_PARSE_PEDANTIC|XML_PARSE_NONET);
  if(reader == NULL) return false;
  xmlDocPtr doc = xmlTextReaderCurrentDoc(reader);
  bool status = readbetsliderFromXml(reader, this, options);
  xmlFreeDoc(doc);
  xmlFreeTextReader(reader);
  return status;
}

bool BetSlider::serialize(const std::string& fileName) const {
  return true;
}

void BetSlider::layout()
{
  float left_width = 0.f;
  float right_width = 0.f;

  for(unsigned int i = 0; i < _rows.size(); i++) {
    if(_rows[i]->_active) {
      Row* row = _rows[i].get();
      if(row->_text[SIDE_LEFT].valid()) {
	const osg::BoundingBox& box = row->_text[SIDE_LEFT]->getBound();
	if(box.xMax() - box.xMin() > left_width)
	  left_width = box.xMax() - box.xMin();
      }
      if(row->_text[SIDE_RIGHT].valid()) {
	const osg::BoundingBox& box = row->_text[SIDE_RIGHT]->getBound();
	if(box.xMax() - box.xMin() > right_width)
	  right_width = box.xMax() - box.xMin();
      }
    }
  }

  float left_x = - ( _margin + left_width / 2.f );
  float right_x = _margin;
  
  for(unsigned int i = 0; i < _rows.size(); i++) {
    if(_rows[i]->_active) {
      Row* row = _rows[i].get();
      if(row->_text[SIDE_LEFT].valid()) {
	osg::Vec3 position = row->_text[SIDE_LEFT]->getPosition();
	position.x() = left_x;
	row->_text[SIDE_LEFT]->setPosition(position);
      }
      if(row->_text[SIDE_RIGHT].valid()) {
	osg::Vec3 position = row->_text[SIDE_RIGHT]->getPosition();
	position.x() = right_x;
	row->_text[SIDE_RIGHT]->setPosition(position);
      }
    }
  }

  left_width += _margin * 2.f;
  right_width += _margin * 2.f;

  osg::Vec3Array* vertexes = dynamic_cast<osg::Vec3Array*>(_background->getVertexArray());
  int index = 0;

  // left square
  float left = - (_middle_width / 2.f + left_width );
  float right = - _middle_width / 2.f;
  (*vertexes)[index++] = osg::Vec3(left, 0.f, 0.f);
  (*vertexes)[index++] = osg::Vec3(right, 0.f, 0.f);
  (*vertexes)[index++] = osg::Vec3(right, _height, 0.f);
  (*vertexes)[index++] = osg::Vec3(left, _height, 0.f);

  // middle line
  left = - _middle_width / 2.f;
  right = _middle_width / 2.f;
  (*vertexes)[index++] = osg::Vec3(left, 0.f, 0.f);
  (*vertexes)[index++] = osg::Vec3(right, 0.f, 0.f);
  (*vertexes)[index++] = osg::Vec3(right, _height, 0.f);
  (*vertexes)[index++] = osg::Vec3(left, _height, 0.f);

  // right square
  left = _middle_width / 2.f;
  right = _middle_width / 2.f + right_width;
  (*vertexes)[index++] = osg::Vec3(left, 0.f, 0.f);
  (*vertexes)[index++] = osg::Vec3(right, 0.f, 0.f);
  (*vertexes)[index++] = osg::Vec3(right, _height, 0.f);
  (*vertexes)[index++] = osg::Vec3(left, _height, 0.f);

  _background->setVertexArray(vertexes);
}

void BetSlider::build()
{
  if(getNumChildren() > 0)
    removeChild(0, getNumChildren());

  _geode = new osg::Geode;
  addChild(_geode.get());

  _background = new osg::Geometry;
  _geode->addDrawable(_background.get());

  const int num_primitives = 3;
  osg::Vec4Array* colors = new osg::Vec4Array(num_primitives);
  (*colors)[0] = _background_color[SIDE_LEFT];
  (*colors)[1] = _middle_color;
  (*colors)[2] = _background_color[SIDE_RIGHT];
  _background->setColorArray(colors);
  _background->setColorBinding(osg::Geometry::BIND_PER_PRIMITIVE_SET);

  osg::Vec3Array* vertexes = new osg::Vec3Array(4 * num_primitives);
  _background->setVertexArray(vertexes);

  _background->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::QUADS, 0, 4));
  _background->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::QUADS, 4, 4));
  _background->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::QUADS, 8, 4));

  _rows[ROW_CANCEL] = new Row(this, _geode.get(), true, false, 0);

  _rows[ROW_CALL] = new Row(this, _geode.get(), true, true, _separator.get());

  _rows[ROW_RAISE] = new Row(this, _geode.get(), true, true, _separator.get());

  _rows[ROW_RAISE_MAX] = new Row(this, _geode.get(), true, true, _separator.get());
  _rows[ROW_RAISE_MAX]->_variable = true;

  _rows[ROW_CURRENT_AMOUNT] = new Row(this, _geode.get(), true, false, 0);
}

BetSlider::Row* BetSlider::getCurrentRow()
{
  for(unsigned int i = 0; i < ROW_COUNT; i++) {
    Row* row = _rows[i].get();
    if(row->_defines_range) {
      if(row->_motor_start <= _motor_position && _motor_position < row->_motor_end)
	return row;
    }
  }
  osg::notify(osg::WARN) << "BetSlider::getCurrentRow: motor position " << _motor_position << " not matching any row" << std::endl;
  return 0;
}

void BetSlider::setMotorPosition(float motor_position)
{
  Row* row = getCurrentRow();
  if(row) row->unselected();
  if(motor_position < 0.f)
    _motor_position = 0.f;
  else if(motor_position >= _motor_span)
    _motor_position = _motor_span - 1;
  else
    _motor_position = motor_position;
  updateCursorPosition();
  updateCurrentValue();
}

void BetSlider::updateCursorPosition()
{
  Row* row = getCurrentRow();
  if(row && _cursor.valid()) {
    float position;
    if(row->_variable) {
      if(_motor_position >= row->_motor_end - 1) {
	position = row->_cursor_end;
      } else {
	float fraction = (_motor_position - row->_motor_start) / (row->_motor_end - row->_motor_start);
	position = row->_cursor_start + (row->_cursor_end - row->_cursor_start) * fraction;
      }
    } else {
      position = row->_cursor_end;
    }
    if(_motor_position < 0.f) position = 0.f;
    _cursor->setPosition(osg::Vec3(0.f, position, .2f));
    _cursor_position = position;
  }
}

void BetSlider::updateCurrentValue()
{
  Row* row = getCurrentRow();
  if(row) {
    bool current_active = false;
    unsigned int value = 0;
    if(row->_variable) {
      value = getCurrentValue();
      if(value < row->_value_end)
	current_active = true;
    }
    if(current_active) {
      _rows[ROW_CURRENT_AMOUNT]->add();
      char tmp[128];
      snprintf(tmp, 128, "%d", value);
      _rows[ROW_CURRENT_AMOUNT]->setText(tmp, "");
    } else {
      _rows[ROW_CURRENT_AMOUNT]->remove();
      row->selected();
    }
  }
}

BetSlider::Row::Row(osg::Group* group,
		    osg::Geode* geode,
		    bool left, bool right,
		    const osg::PositionAttitudeTransform* separator) :
  _defines_range(false),
  _active(false),
  _variable(false),
  _value_start(0),
  _value_end(0),
  _cursor_start(0.f),
  _cursor_end(0.f),
  _motor_start(0.f),
  _motor_end(0.f),
  _geode(geode),
  _group(group),
  _separator(separator ? dynamic_cast<osg::PositionAttitudeTransform*>(separator->clone(osg::CopyOp::SHALLOW_COPY)) : 0)
{
  for(int i = 0; i < SIDE_COUNT; i++) {
    _normal[i] = osg::Vec4(.7f, .7f, .7f, 1.f);
    _selected[i] = osg::Vec4(1.f, 1.f, 1.f, 1.f);
  }

  if(left) {
    osgText::Text* text = new osgText::Text;
    text->setColor(_normal[SIDE_LEFT]);
    text->setCharacterSize(DEFAULT_CHARACTER_SIZE);
    text->setAlignment(osgText::Text::CENTER_CENTER);
    _text[SIDE_LEFT] = text;
  }

  if(right) {
    osgText::Text* text = new osgText::Text;
    text->setColor(_normal[SIDE_RIGHT]);
    text->setCharacterSize(DEFAULT_CHARACTER_SIZE);
    text->setAlignment(osgText::Text::LEFT_CENTER);
    _text[SIDE_RIGHT] = text;
  }
}

BetSlider::Row::~Row() {}

void BetSlider::Row::setText(const std::string& left, const std::string& right)
{
  if(_text[SIDE_LEFT].valid())
    _text[SIDE_LEFT]->setText(left);
  if(_text[SIDE_RIGHT].valid())
    _text[SIDE_RIGHT]->setText(right);
}

void BetSlider::Row::setMotorRange(float& origin, float length)
{
  _defines_range = true;
  _motor_start = origin;
  _motor_end = origin + length;
  origin += length;
}

void BetSlider::Row::setCursorRange(float& origin, float& cursor, float height)
{
  _defines_range = true;
  osg::BoundingBox box = getBound();
  float text_height = box.yMax() - box.yMin();
  float position = origin + height + text_height;
  if(_variable) {
    _cursor_start = cursor;
    _cursor_end = position;
  } else {
    _cursor_end = _cursor_start = position;
  }
  cursor = position;
  if(_separator.valid())
    _separator->setPosition(osg::Vec3(0.f, position, .1f));
  for(int i = 0; i < SIDE_COUNT; i++)
    if(_text[i].valid())
      _text[i]->setPosition(osg::Vec3(0.f, position, .1f));
  origin = position;
}

void BetSlider::Row::setSeparator(osg::PositionAttitudeTransform* separator)
{
  osg::Vec3 position;
  if(_separator.valid())
    position = _separator->getPosition();

  if(_active && _separator.valid()) {
    _group->removeChild(_separator.get());
  }

  if(separator) {
    _separator = dynamic_cast<osg::PositionAttitudeTransform*>(separator->clone(osg::CopyOp::SHALLOW_COPY));
    _separator->setPosition(position);
  } else {
    _separator = 0;
  }

  if(_active && _separator.valid())
    _group->addChild(_separator.get());
}

osg::BoundingBox BetSlider::Row::getBound() const
{
  osg::BoundingBox box;
  for(int i = 0; i < SIDE_COUNT; i++)
    if(_text[i].valid())
      box.expandBy(_text[i]->getBound());
  return box;
}

void BetSlider::Row::add()
{
  if(!_active) {
    _active = true;
    for(int i = 0; i < SIDE_COUNT; i++)
      if(_text[i].valid())
	_geode->addDrawable(_text[i].get());
    if(_separator.valid())
      _group->addChild(_separator.get());
  }
}

void BetSlider::Row::remove()
{
  if(_active) {
    _defines_range = false;
    _active = false;
    for(int i = 0; i < SIDE_COUNT; i++)
      if(_text[i].valid())
	_geode->removeDrawable(_text[i].get());
    if(_separator.valid())
      _group->removeChild(_separator.get());
  }
}

void BetSlider::Row::selected()
{
  for(int i = 0; i < SIDE_COUNT; i++)
    if(_text[i].valid())
      _text[i]->setColor(_selected[i]);
}

void BetSlider::Row::unselected()
{
  for(int i = 0; i < SIDE_COUNT; i++)
    if(_text[i].valid())
      _text[i]->setColor(_normal[i]);
}
