/*
*
* Copyright (C) 2004 Mekensleep
*
*	Mekensleep
*	24 rue vieille du temple
*	75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*
* Authors:
*  Igor Kravtchenko <igor@mekensleep.com>
*
*/

#include "mafStdAfx.h"

#ifndef USE_VS_PCH
#include <maf/osghelper.h>
#include <osg/StateAttribute>
#include <osg/Light>
#include <osg/Plane>
#include <osg/Vec3>
#include <exg/exg_vector3.h>
#include <vector>
#include <osg/Array>
#endif


static void recurse_getNbLightSources(const osg::Group &_group, int &_nbLights)
{
	int nbChilds = _group.getNumChildren();
	for (int i = 0; i < nbChilds; i++) {
		const osg::Node *node = _group.getChild(i);
		const std::string &clsname = node->className();
		const osg::Group *group = node->asGroup();

		if (clsname == "LightSource")
			_nbLights++;

		if (group)
			recurse_getNbLightSources(*group, _nbLights);
	}
}

int OSGHelper_getNbLightSourcesOf(const osg::Group &_root)
{
	int nbLights = 0;
	recurse_getNbLightSources(_root, nbLights);
	return nbLights;
}

//
static void recurse_getNbGeodes(const osg::Group &_group, int &_nbGeodes)
{
	int nbChilds = _group.getNumChildren();
	for (int i = 0; i < nbChilds; i++) {
		const osg::Node *node = _group.getChild(i);
		const std::string &clsname = node->className();
		const osg::Group *group = node->asGroup();

		if (clsname == "Geode")
			_nbGeodes++;

		if (group)
			recurse_getNbGeodes(*group, _nbGeodes);
	}
}

int OSGHelper_getNbGeodesOf(const osg::Group &_root)
{
	int nbGeodes = 0;
	recurse_getNbGeodes(_root, nbGeodes);
	return nbGeodes;
}


//

static std::map<std::string, osg::Geode*> g_string2geode;

static void recurse_getGeodeByName(const osg::Group &_group)
{
	int nbChilds = _group.getNumChildren();
	for (int i = 0; i < nbChilds; i++) {
		const osg::Node *node = _group.getChild(i);
		const std::string &clsname = node->className();
		const std::string &name = node->getName();
		const osg::Group *group = node->asGroup();

		if (clsname == "Geode")
			g_string2geode[name] = (osg::Geode*) node;

		if (group)
			recurse_getGeodeByName(*group);
	}
}

osg::Geode* OSGHelper_getGeodeByName(const osg::Group &_root, const std::string &_name)
{
	g_string2geode.clear();
	recurse_getGeodeByName(_root); // TODO: do it once at some init time
	return g_string2geode[_name];
}

//

static std::map<std::string, osg::LightSource*> g_string2lightsource;

static void recurse_getLightSourceByName(const osg::Group &_group)
{
	int nbChilds = _group.getNumChildren();
	for (int i = 0; i < nbChilds; i++) {
		const osg::Node *node = _group.getChild(i);
		const std::string &clsname = node->className();
		const std::string &name = node->getName();
		const osg::Group *group = node->asGroup();

		if (clsname == "LightSource")
			g_string2lightsource[name] = (osg::LightSource*) node;

		if (group)
			recurse_getGeodeByName(*group);
	}
}

osg::LightSource* OSGHelper_getLightSourceByName(const osg::Group &_root, const std::string &_name)
{
	g_string2lightsource.clear();
	recurse_getLightSourceByName(_root); // TODO: do it once at some init time
	return g_string2lightsource[_name];
}


//

static std::map<std::string, osg::Node*> g_string2node;

static void recurse_getNodeByName(const osg::Group &_group)
{
	int nbChilds = _group.getNumChildren();
	for (int i = 0; i < nbChilds; i++) {
		const osg::Node *node = _group.getChild(i);
		//		const std::string &clsname = node->className();
		const std::string &name = node->getName();
		const osg::Group *group = node->asGroup();

		g_string2node[name] = (osg::Node*) node;

		if (group)
			recurse_getNodeByName(*group);
	}
}

osg::Node* OSGHelper_getNodeByName(const osg::Group &_root, const std::string &_name)
{
	g_string2node.clear();
	recurse_getNodeByName(_root); // TODO: do it once at some init time
	return g_string2node[_name];
}

//
void OSGHelper_turnOffLighting(osg::StateSet &_ss)
{
	_ss.setMode(GL_LIGHTING, osg::StateAttribute::OFF);
	_ss.removeAttribute(osg::StateAttribute::LIGHT);
}

void OSGHelper_turnOnLighting(osg::StateSet &_ss)
{
	_ss.setMode(GL_LIGHTING, osg::StateAttribute::ON);
}

void OSGHelper_turnLightOff(osg::StateSet &_ss, int _index)
{
	_ss.setMode(GL_LIGHT0 + _index, osg::StateAttribute::OFF);
//	_ss.setAttributeToInherit(osg::StateAttribute::LIGHT_0 + _index);
}

void OSGHelper_turnLightOn(osg::StateSet &_ss, int _index)
{
	_ss.setMode(GL_LIGHT0 + _index, osg::StateAttribute::ON);
}


void OSGHelper_lookAt(const osg::Vec3f &_from, const osg::Vec3f &_target, osg::Matrixf &_mat)
{
	// These calculs are based on the Robert Dunlop's method
	osg::Vec3f vec = _target - _from;
	vec.normalize();

	osg::Vec3f up(0.0f, 1.0f, 0.0f);
	up -= vec * vec.y();
	float len = up.length();
	if (len < 1e-6f) {
		up = osg::Vec3f(0.0f, 0.0f, 1.0f) - vec * vec.z();
		len = up.length();
	}
	up /= len;

	osg::Vec3f right = up ^ vec;
	_mat(0, 0) = right.x();	_mat(1, 0) = up.x();	_mat(2, 0) = vec.x();
	_mat(0, 1) = right.y();	_mat(1, 1) = up.y();	_mat(2, 1) = vec.y();
	_mat(0, 2) = right.z(); _mat(1, 2) = up.z();	_mat(2, 2) = vec.z();
}

osg::Matrix OSGHelper_getRotationMatrixFromVectorToVector(const osg::Vec3 &_dir, const osg::Vec3 &_wished_dir)
{
	osg::Vec3 ndir = _dir;
	ndir.normalize();
	osg::Vec3 nwdir = _wished_dir;
	nwdir.normalize();

	// calcul the axis of rotation
	osg::Vec3 axisrot = ndir ^ nwdir;

	float dot = fabs(ndir * nwdir);
	float rot = dot;

	osg::Plane plane( osg::Vec3(0, 0, 0), ndir, axisrot);
	float d = plane.distance(nwdir);
	if (d < 0)
		rot = -dot;

	return osg::Matrix::rotate(rot, axisrot);
}

void OSGHelper_getPointsEqualTo(const osg::Vec3Array &_array, const osg::Vec3 &_cmp, std::vector<int> &_res, float _tolerance)
{
	int size = _array.size();
	for (int i = 0; i < size; i++) {
		const osg::Vec3 &v = _array[i];

		float difX = v.x() - _cmp.x();
		float difY = v.y() - _cmp.y();
		float difZ = v.z() - _cmp.z();

		if (difX < _tolerance && difX > -_tolerance &&
			difY < _tolerance && difY > -_tolerance &&
			difZ < _tolerance && difZ > -_tolerance) {
			_res.push_back(i);
			}
	}
}

void OSGHelper_getPointsWithXOf(const osg::Vec3Array &_array, float _x, std::vector<int> &_res, float _tolerance)
{
	int size = _array.size();
	for (int i = 0; i < size; i++) {
		const osg::Vec3 &v = _array[i];
		float dif = v.x() - _x;
		if (dif < _tolerance && dif > -_tolerance)
			_res.push_back(i);
	}
}

void OSGHelper_getPointsWithYOf(const osg::Vec3Array &_array, float _y, std::vector<int> &_res, float _tolerance)
{
	int size = _array.size();
	for (int i = 0; i < size; i++) {
		const osg::Vec3 &v = _array[i];
		float dif = v.y() - _y;
		if (dif < _tolerance && dif > -_tolerance)
			_res.push_back(i);
	}
}

void OSGHelper_getPointsWithZOf(const osg::Vec3Array &_array, float _z, std::vector<int> &_res, float _tolerance)
{
	int size = _array.size();
	for (int i = 0; i < size; i++) {
		const osg::Vec3 &v = _array[i];
		float dif = v.z() - _z;
		if (dif < _tolerance && dif > -_tolerance)
			_res.push_back(i);
	}
}

osg::Matrixf convert(const exg::Matrix4f &_from)
{
	osg::Matrixf res;
	for (int j = 0; j < 4; j++) {
		for (int i = 0; i < 4; i++) {
			res(j, i) = _from[j][i];
		}
	}
	return res;
}

exg::Matrix4f convert(const osg::Matrixf &_from)
{
	exg::Matrix4f res;
	for (int j = 0; j < 4; j++) {
		for (int i = 0; i < 4; i++) {
			res[j][i] = _from(j, i);
		}
	}
	return res;
}
