/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Henry Precheur <henry at precheur dot org>
 *  Igor kravtchenko <igor@obraz.net>
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/maferror.h>
#include <maf/camera.h>
#include <maf/data.h>
#include <maf/wnc_desktop.h>
#include <maf/wnc_source.h>

#include <sys/types.h>
#include <signal.h>
#include <errno.h>
#include <set>
#include <vector>

#include <glib.h>

#include <vorbis/vorbisfile.h>

#include <cal3d/cal3d.h>

#include <osgDB/Registry>
#include <osgDB/ReadFile>
#include <osgDB/WriteFile>
#include <osgText/Font>
#include <osg/Geode>
#include <osg/Drawable>
#include <osg/MatrixTransform>
#include <osg/LightSource>
#include <osg/PositionAttitudeTransform>
#include <osgCal/CoreModel> // remove
#include <osg/Geometry>
#include <osg/NodeVisitor>
#include <osgText/Font>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <maf/wnc_window.h>

#include <maf/osghelper.h>

#endif

// MAFVisionData

MAFCameraController* MAFVisionData::GetCamera(const std::string& name)
{
  if(mCameras.find(name) == mCameras.end())
    throw new MAFError(UNDERWARE_MAF_ERROR_CAMERA, "MAFVisionData::GetCamera: camera %s is not known", name.c_str());

  return mCameras[name].get();
}

MAFCameraController* MAFVisionData::getFirstCamera()
{
	if (mCameras.size() == 0)
		return NULL;
	std::map<std::string, osg::ref_ptr<MAFCameraController> >::iterator it = mCameras.begin();
	osg::ref_ptr<MAFCameraController> &cam = (*it).second;
	return cam.get();
}


// MAFVisionData

osg::ref_ptr<osg::LightSource> MAFVisionData::getLightByIndex(int _index) const
{
	return mLights[_index];
}


// MAFOSGData

MAFData* MAFOSGData::Clone(void)
{
  MAFOSGData* copy = new MAFOSGData;
  copy->mGroup = dynamic_cast<osg::Group*>(mGroup->clone(osg::CopyOp::DEEP_COPY_ALL));
  g_assert(copy->mGroup != 0);
  copy->mCameras = mCameras;
  return copy;
}

bool MAFOSGData::Load(const std::string& path, osgDB::ReaderWriter::Options* options)
{
  osg::Node* node = osgDB::readNodeFile(path, options);
  if(!node)
    throw new MAFError(UNDERWARE_MAF_ERROR_DATALOAD, "MAFOSGData::Load: osgDB::readNodeFile(%s) failed", path.c_str()); 

  mGroup = node->asGroup();

  if(!mGroup) {
    g_warning("MAFOSGData::Load: root node of %s is not a osg::Group", path.c_str());
    return false;
  }

  int dot = path.rfind('.');
  std::string lcfile = path;
  std::transform(lcfile.begin(), lcfile.end(), lcfile.begin(), tolower);
  std::string suffix = lcfile.substr(dot);
  return true;
}

osg::Node* MAFOSGData::GetNode(const std::string& name) {
  return ::GetNode(mGroup.get(), name);
}

void MAFOSGData::GroupAnchors(osg::Group* group, const std::vector<std::string>& names)
{
  osg::Group* parent = 0;
  std::vector<std::string>::const_iterator name;
  for(name = names.begin();
      name != names.end();
      name++) {
    MAFAnchor* anchor = GetAnchor(*name);
    if(parent == 0)
      parent = anchor->getParent(0);
    if(parent != anchor->getParent(0))
      g_error("MAFOSGData::GroupAnchors: %s ... can't group anchors with different parents", name->c_str());
    group->addChild(anchor);
    parent->removeChild(anchor);
  }

  if(!parent)
    g_critical("MAFOSGData::GroupAnchors: %s ... no anchor found", (*(names.begin()+0)).c_str());
  else
    parent->addChild(group);
}

class AnchorVisitor : public osg::NodeVisitor {
public:
  AnchorVisitor(const std::string& name) : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN), mAnchor(0), mName(name) {
    setNodeMaskOverride(0xffffffff);
  }

  virtual void apply(osg::Transform& node) {
    if(node.getName().find(mName) != std::string::npos) {
      mAnchor = dynamic_cast<MAFAnchor*>(&node);
      if(node.getName() != mName)
	g_critical("anchor searched %s but found anchor %s (AMBIGUOUS MUST BE FIXED)", mName.c_str(), node.getName().c_str());
    } else
      traverse(node);
  }

  MAFAnchor* mAnchor;
  const std::string& mName;
};

MAFAnchor* MAFOSGData::GetAnchor(const std::string& name)
{
  AnchorVisitor visitor(name);
  mGroup->accept(visitor);
  if(visitor.mAnchor == 0)
    throw new MAFError(UNDERWARE_MAF_ERROR_ANCHOR, "MAFOSGData::GetAnchor: %s not found", name.c_str()); 
  return visitor.mAnchor;
}

osg::BoundingBox MAFOSGData::GetBound()
{
  g_assert(mGroup.valid());

  osg::Geode* geode = GetGeode(mGroup.get());
  g_assert(geode != 0);
  osg::BoundingBox bb;
  unsigned int num_drawables = geode->getNumDrawables();
  for(unsigned int i = 0; i < num_drawables; i++) {
    osg::Drawable* drawable = geode->getDrawable(i);
    osg::BoundingBox f = drawable->getBound();
    bb.expandBy(drawable->getBound());
  }

  return bb;
}

void MAFOSGData::SetDescription(const std::string& name)
{
  g_assert(mGroup.valid());

  mGroup->addDescription(name);
}

// MAFCal3dData

MAFCal3dData::MAFCal3dData() :
  mConfigurable(false),
  mDocument(NULL),
  mXpath(NULL)
{
}

MAFCal3dData::~MAFCal3dData() {
  if(mXpath != NULL)
    xmlXPathFreeContext(mXpath); 

  if (mDocument != NULL)
    xmlFreeDoc(mDocument);

  //  g_assert(mCoreModel->referenceCount() == 1);
}

#define LINE_BUFFER_SIZE 4096

/* This method is not present in osgCal, however it is sorely
 * needed to have a usable loader.
 */

bool MAFCal3dData::Load(const std::string& dir, osgDB::ReaderWriter::Options* options) {
  std::string path = dir + "/cal3d.xml";
  FILE* file = fopen(path.c_str(), "r");
  if(file) {
    fclose(file);
    LoadXML(dir, path);
  } else {
    path = dir + "/cal3d.cfg";
    file = fopen(path.c_str(), "r");
    if(file) {
      fclose(file);
      LoadCFG(dir, path);
    }
	else
		return false;
  }
  return true;
}

MAFCal3dData::Outfit* MAFCal3dData::GetOutfit(const std::string& outfitName)
{
  Outfit* outfit = new Outfit;
  if(mConfigurable) {
    g_message("Loading outfit %s from %s", outfitName.c_str(), mPath.c_str());
    if(!GetOutfit(outfit, "/cal3d/outfits/outfit[@name='" + outfitName + "']/item/mesh") ||
       !GetOutfit(outfit, "/cal3d/outfits/common/mesh")) {
      delete outfit;
      outfit = 0;
    }
  } else {
    CalCoreModel* calCoreModel = mCoreModel->getCalCoreModel();
    std::map<int,std::string> coreMaterialId2Name;
    for(int coreMaterialId = 0; coreMaterialId < calCoreModel->getCoreMaterialCount(); coreMaterialId++) {
      const std::string& name = calCoreModel->getCoreMaterial(coreMaterialId)->getName();
      coreMaterialId2Name[coreMaterialId] = name;
    }

    for(int coreMeshId = 0; coreMeshId < calCoreModel->getCoreMeshCount(); coreMeshId++) {
      const std::string& meshName = calCoreModel->getCoreMesh(coreMeshId)->getName();
      int coreMaterialId = coreMeshId;
      MeshDescription description;
      description["name"] = meshName;
      if(coreMaterialId2Name.find(coreMaterialId) == coreMaterialId2Name.end())
	g_critical("MAFCal3dData::GetOutfit: no material with id %d for mesh %s", coreMaterialId, meshName.c_str());
      else
	description["material0"] = coreMaterialId2Name[coreMaterialId];
      if(meshName.substr(0, 2) == "__")
	description["collision"] = "yes";
      outfit->push_back(description);
    }
  }

  return outfit;
}

bool MAFCal3dData::GetOutfit(Outfit* outfit, const std::string& xpath)
{
  std::set<std::string> duplicates;
  bool status = true;
  xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)xpath.c_str(), mXpath);
  if(xpathObj == NULL)
    g_error("MAFCal3dData::GetOutfit: eval %s file in %s", xpath.c_str(), mPath.c_str());

  xmlNodeSetPtr nodes = xpathObj->nodesetval;
  if(nodes && nodes->nodeNr >= 1) {
    for(int i = 0; i < nodes->nodeNr; i++) {
      std::string line;
      xmlNodePtr node = nodes->nodeTab[i];
      switch(node->type) {
      case XML_ELEMENT_NODE:
	{
	  MeshDescription meshDescription;
	  xmlAttr* attribute;
	  for(attribute = node->properties; attribute; attribute = attribute->next) {
	    const char* value = (const char*)xmlNodeGetContent((xmlNode*)attribute);
	    const char* variable = (const char*)attribute->name;
	    meshDescription[variable] = value;
	    line += variable + std::string("=") + value + " ";
	    xmlFree((void*)value);
	  }
	  if(meshDescription.find("lib") == meshDescription.end()) {
	    if(meshDescription.find("name") == meshDescription.end()) {
	       g_critical("MAFCal3dData::GetOutfit: lib and name attribute not set or empty at occurence %d of %s in %s (ignored)", i, xpath.c_str(), mPath.c_str());
	       status = false;
	    } else {
	      meshDescription["lib"] = meshDescription["name"];
	    }
	  } 
	  g_message("  outfit %s", meshDescription["lib"].c_str());
	  if(duplicates.find(line) != duplicates.end()) {
	    g_debug("MAFCal3dData::GetOutfit: %s%s duplicate mesh %s, ignored", mPath.c_str(), xpath.c_str(), line.c_str());
	  } else {
	    duplicates.insert(line);
	    outfit->push_back(meshDescription);
	  }
	}
	break;
      default:
	g_critical("MAFCal3dData::GetOutfit: %s%s expected ELEMENT node got %d", mPath.c_str(), xpath.c_str(), node->type);
	status = false;
	break;
      }
    }
  } else {
    g_critical("MAFCal3dData::GetOutfit: %s not found in %s", xpath.c_str(), mPath.c_str());
    status = false;
  }
  xmlXPathFreeObject(xpathObj);

  return status;
}

int MAFCal3dData::getNbOutfits() const
{
	xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)"/cal3d/outfits/outfit", mXpath);
	xmlNodeSetPtr nodes = xpathObj->nodesetval;
	return nodes->nodeNr;
}

const char* MAFCal3dData::getOutfitNameByIndex(int _index) const
{
	xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)"/cal3d/outfits/outfit", mXpath);
	xmlNodeSetPtr nodes = xpathObj->nodesetval;
	if (unsigned(_index) >= unsigned(nodes->nodeNr))
		return NULL;
	xmlNodePtr node = nodes->nodeTab[_index];
	xmlAttr* attribute = node->properties;
	const char* value = (const char*)xmlNodeGetContent((xmlNode*)attribute);
	return value;
}

void MAFCal3dData::LoadXML(const std::string& dir, const std::string& path) {
  mCoreModel = new osgCal::CoreModel;
  CalCoreModel* calCoreModel = GetCalCoreModel();
  g_message("Loading cal3d character from '%s'...\n", path.c_str());

  mConfigurable = true;
  mDocument = xmlParseFile(path.c_str());
  mXpath = xmlXPathNewContext(mDocument);
  mPath = path;
  if(mXpath == NULL)
    g_error("MAFCal3dData::LoadXML: unable to create new XPath context");

  {
    char* xpath = "/cal3d/skeleton/@name";
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)xpath, mXpath);
    if(xpathObj == NULL) g_error("MAFCal3dData::LoadXML: eval %s file in %s", xpath, path.c_str());
  
    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes) {
      if(nodes->nodeNr == 1) {
	xmlNodePtr node = nodes->nodeTab[0];
	switch(node->type) {
	case XML_ATTRIBUTE_NODE:
	  {
	    const char* name = (const char*)xmlNodeGetContent(node);

	    std::string skeleton = dir + "/" + name;
	    xmlFree((void*)name);
	    g_message("   skeleton %s", skeleton.c_str());
	    if(!calCoreModel->loadCoreSkeleton(skeleton))
	      g_critical("MAFCal3dData::LoadXML: loadCoreSkeleton(%s) %s at %s in %s", skeleton.c_str(), CalError::getLastErrorDescription().c_str(), xpath, path.c_str());
	  }
	  break;
	default:
	  throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s%s expected ATTRIBUTE node got %d", path.c_str(), xpath, node->type);
	  break;
	}
      }
    } else {
      throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s not found in %s", xpath, path.c_str()); 
    }

    xmlXPathFreeObject(xpathObj);
  }
  
  {
    char* xpath = "/cal3d/library/mesh";
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)xpath, mXpath);
    if(xpathObj == NULL) g_error("MAFCal3dData::LoadXML: eval %s file in %s", xpath, path.c_str());

    osgCal::CoreModel::Name2Filename meshName2Filename;
    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes && nodes->nodeNr >= 1) {
      for(int i = 0; i < nodes->nodeNr; i++) {
	std::string filename;
	std::string name;

	xmlNodePtr node = nodes->nodeTab[i];
	switch(node->type) {
	case XML_ELEMENT_NODE:
	  {
	    xmlAttr* attribute;
	    for(attribute = node->properties; attribute; attribute = attribute->next) {
	      const char* value = (const char*)xmlNodeGetContent((xmlNode*)attribute);
	      const char* variable = (const char*)attribute->name;
	      if(variable == std::string("file"))
		filename = dir + "/" + value;
	      else if(variable == std::string("name"))
		name = value;
	      else if(variable == std::string("version"))
		;
	      else
		g_critical("MAFCal3dData::LoadXML: unexpected attribute %s=\"%s\" at %s in %s", variable, value, xpath, path.c_str());
	      xmlFree((void*)value);
	    }
	  }
	  break;
	default:
	  throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s%s expected ELEMENT node got %d", path.c_str(), xpath, node->type);
	  break;
	}
	if(filename == "")
	  g_critical("MAFCal3dData::LoadXML: filename attribute not set or empty at occurence %d of %s in %s", i, xpath, path.c_str());
	if(name == "")
	  g_critical("MAFCal3dData::LoadXML: name attribute not set or empty at occurence %d of %s in %s", i, xpath, path.c_str());
	meshName2Filename[name] = filename;
      }
    } else {
      throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s not found in %s", xpath, path.c_str()); 
    }
    xmlXPathFreeObject(xpathObj);
    mCoreModel->setMeshName2Filename(meshName2Filename);
  }

  {
    char* xpath = "/cal3d/library/material";
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)xpath, mXpath);
    if(xpathObj == NULL) g_error("MAFCal3dData::LoadXML: eval %s file in %s", xpath, path.c_str());

    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes && nodes->nodeNr >= 1) {
      for(int i = 0; i < nodes->nodeNr; i++) {
	std::string filename;
	std::string name;

	xmlNodePtr node = nodes->nodeTab[i];
	switch(node->type) {
	case XML_ELEMENT_NODE:
	  {
	    xmlAttr* attribute;
	    for(attribute = node->properties; attribute; attribute = attribute->next) {
	      const char* value = (const char*)xmlNodeGetContent((xmlNode*)attribute);
	      const char* variable = (const char*)attribute->name;
	      if(variable == std::string("file"))
		filename = dir + "/" + value;
	      else if(variable == std::string("name"))
		name = value;
	      else if(variable == std::string("version"))
		;
	      else
		g_critical("MAFCal3dData::LoadXML: unexpected attribute %s=\"%s\" at %s in %s", variable, value, xpath, path.c_str());
	      xmlFree((void*)value);
	    }
	  }
	  break;
	default:
	  throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s%s expected ELEMENT node got %d", path.c_str(), xpath, node->type);
	  break;
	}
	if(filename == "")
	  g_critical("MAFCal3dData::LoadXML: filename attribute not set or empty at occurence %d of %s in %s", i, xpath, path.c_str());
	if(name == "")
	  g_critical("MAFCal3dData::LoadXML: name attribute not set or empty at occurence %d of %s in %s", i, xpath, path.c_str());
	g_message("   material %s", filename.c_str());
	int materialId = calCoreModel->loadCoreMaterial(filename, name);
	if(materialId < 0) {
	  g_critical("MAFCal3dData::LoadXML: loadCoreMaterial(%s,%s) %s at %s in %s", filename.c_str(), name.c_str(), CalError::getLastErrorDescription().c_str(), xpath, path.c_str());
	} else {
    char* dir2=0;
	  dir2 = g_path_get_dirname(filename.c_str());
	  // create a single set (0), each material belongs to a single thread named after its
	  // material id.
	  calCoreModel->createCoreMaterialThread(materialId); 
	  calCoreModel->setCoreMaterialId(materialId, 0, materialId);
	  CalCoreMaterial* material = calCoreModel->getCoreMaterial(materialId);
	  for(std::vector<CalCoreMaterial::Map>::iterator i2 = material->getVectorMap().begin();
	      i2 != material->getVectorMap().end();
	      i2++) {
	    i2->strFilename = std::string(dir2) + "/" + i2->strFilename;
	  }
	  g_free(dir2);
	}
      }
    } else {
      throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s not found in %s", xpath, path.c_str()); 
    }
    xmlXPathFreeObject(xpathObj);
  }

  {
    char* xpath = "/cal3d/library/animation";
    xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)xpath, mXpath);
    if(xpathObj == NULL) g_error("MAFCal3dData::LoadXML: eval %s file in %s", xpath, path.c_str());

    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    if(nodes && nodes->nodeNr >= 1) {
      for(int i = 0; i < nodes->nodeNr; i++) {
	std::string filename;
	std::string name;

	xmlNodePtr node = nodes->nodeTab[i];
	switch(node->type) {
	case XML_ELEMENT_NODE:
	  {
	    xmlAttr* attribute;
	    for(attribute = node->properties; attribute; attribute = attribute->next) {
	      const char* value = (const char*)xmlNodeGetContent((xmlNode*)attribute);
	      const char* variable = (const char*)attribute->name;
	      if(variable == std::string("file"))
		filename = dir + "/" + value;
	      else if(variable == std::string("name"))
		name = value;
	      else if(variable == std::string("version"))
		;
	      else
		g_critical("MAFCal3dData::LoadXML: unexpected attribute %s=\"%s\" at %s in %s", variable, value, xpath, path.c_str());
	      xmlFree((void*)value);
	    }
	  }
	  break;
	default:
	  throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s%s expected ELEMENT node got %d", path.c_str(), xpath, node->type);
	  break;
	}
	if(filename == "")
	  g_critical("MAFCal3dData::LoadXML: filename attribute not set or empty at occurence %d of %s in %s", i, xpath, path.c_str());
	if(name == "")
	  g_critical("MAFCal3dData::LoadXML: name attribute not set or empty at occurence %d of %s in %s", i, xpath, path.c_str());
	g_message("   animation %s", filename.c_str());
	int coreAnimationId = calCoreModel->loadCoreAnimation(filename, name);
	if(coreAnimationId < 0)
	  g_critical("MAFCal3dData::LoadXML: loadCoreAnimation(%s,%s) %s at %s in %s", filename.c_str(), name.c_str(), CalError::getLastErrorDescription().c_str(), xpath, path.c_str());
      }
    } else {
      throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "MAFCal3dData::LoadXML: %s not found in %s", xpath, path.c_str()); 
    }
    xmlXPathFreeObject(xpathObj);
  }

}

void MAFCal3dData::LoadCFG(const std::string& dir, const std::string& path) {
  mCoreModel = new osgCal::CoreModel;
  CalCoreModel* calCoreModel = GetCalCoreModel();
  g_message("Loading cal3d character from %s", path.c_str());
  FILE* f = fopen(path.c_str(), "r");
  if(!f)
      throw new MAFError(UNDERWARE_MAF_ERROR_DATALOAD, "MAFCal3dData::Load: fopen(%s) failed: %s", path.c_str(), strerror(errno)); 

  char buffer[LINE_BUFFER_SIZE];

  while (fgets(buffer, LINE_BUFFER_SIZE,f))
  {
    // Ignore comments or empty lines
    if (*buffer == '#' || *buffer == 0)
      continue;

    char* equal = strchr(buffer, '=');
    if (equal)
    {
      // Terminates first token
      *equal++ = 0;
      // Removes ending newline (CR & LF)
      {
        int last = strlen(equal) - 1;
        if (equal[last] == '\n') equal[last] = 0;
        if (last > 0 && equal[last-1] == '\r') equal[last-1] = 0;
      }
      std::string fullpath = dir + "/" + std::string(equal);

      if (!strcmp(buffer, "scale")) { g_message("  unhandled 'scale'"); continue; }

      g_message("  subloading %-9s: '%s'", buffer, fullpath.c_str());
      if (!strcmp(buffer, "skeleton")) {
	if(!calCoreModel->loadCoreSkeleton(fullpath))
	  g_critical("MAFCal3dData::LoadCFG: %s", CalError::getLastErrorDescription().c_str());
      } else if (!strcmp(buffer, "animation")) {
	if(calCoreModel->loadCoreAnimation(fullpath, equal) < 0)
	  g_critical("MAFCal3dData::LoadCFG: %s", CalError::getLastErrorDescription().c_str());
      } else if (!strcmp(buffer, "mesh")) {
	if(calCoreModel->loadCoreMesh(fullpath, equal) < 0)
	  g_critical("MAFCal3dData::LoadCFG: %s", CalError::getLastErrorDescription().c_str());
      } else if (!strcmp(buffer, "material"))  {
	int materialId = calCoreModel->loadCoreMaterial(fullpath, equal);
	if(materialId < 0) {
	  g_critical("MAFCal3dData::LoadCFG: %s", CalError::getLastErrorDescription().c_str());
	} else {
	  CalCoreMaterial* material = calCoreModel->getCoreMaterial(materialId);
	  for(std::vector<CalCoreMaterial::Map>::iterator i = material->getVectorMap().begin();
	      i != material->getVectorMap().end();
	      i++) {
	    i->strFilename = dir + "/" + i->strFilename;
	  }
	}
      }
    }
  }

  fclose(f);
}

// MAFAudioData

MAFAudioData::~MAFAudioData()
{
}

void MAFAudioData::Error(const openalpp::Error& error)
{
  std::ostringstream str;
#ifndef WIN32 // help me. It does not compile with it on windows
  error.put(str);
#else
  str.str("can't find error");
#endif
  g_critical("MAFAudioData::Error: %s", str.str().c_str());
  ALenum alerror = alGetError();
  if(alerror != AL_NO_ERROR)
    g_critical("MAFAudioData::Error: alGetError() = %s", alGetString(alerror));
}

bool MAFAudioDataWAV::Load(const std::string& path, osgDB::ReaderWriter::Options* options) {
  try {
    mSoundData = new openalpp::Sample(path.c_str());
  } catch(openalpp::Error error) {
    Error(error);
  }
  return true;
}


#if 1
#define BUFFER_OGG 32768

bool MAFAudioDataOGG::Load(const std::string& path, osgDB::ReaderWriter::Options* options) {
  int endian = 0;                         // 0 for Little-Endian, 1 for Big-Endian
  int bitStream;
  long bytes;
  char array[BUFFER_OGG];                // Local fixed size array
  std::vector<char> buffer;
  ALsizei  freq;
  ALenum   format;

  FILE *file = fopen(path.c_str(), "rb");
  g_debug("Open %s for reading", (char*)path.c_str());

  if(file == NULL) {
    g_debug("Cannot open %s for reading", (char*)path.c_str());
    return false;
  }

  vorbis_info *pInfo;
  OggVorbis_File oggFile;

  if(ov_open(file, &oggFile, NULL, 0) != 0) {
    g_debug("ov_open failed for %s", (char*)path.c_str());
    fclose(file);
    return false;
  }

  pInfo = ov_info(&oggFile, -1);

  if (pInfo->channels == 1)
    format = AL_FORMAT_MONO16;
  else
    format = AL_FORMAT_STEREO16;

  freq = pInfo->rate;

  do {
    bytes = ov_read(&oggFile, array, BUFFER_OGG, endian, 2, 1, &bitStream);

    if (bytes < 0) {
      ov_clear(&oggFile);
      g_debug("read failed for %s", (char*)path.c_str());
      fclose(file);
      return false;
    }
    buffer.insert(buffer.end(), array, array + bytes);
  } while(bytes > 0);

  ov_clear(&oggFile);

  void* vb=&buffer.front();
  int sz=buffer.size();

  try {
	mSoundData = new openalpp::Sample(format,vb,sz,freq);
  } catch (...) {
	g_debug("Can't play ogg no sound");
  }
  return true;
}
#else

bool MAFAudioDataOGG::Load(const std::string& path, osgDB::ReaderWriter::Options* options) {
  try {
    mSoundData = new openalpp::FileStream(path.c_str());
  } catch(openalpp::Error error) {
    Error(error);
  }
  return true;
}
#endif

// MAFXmlData

MAFXmlData::MAFXmlData() : mDocument(NULL) {
}

MAFXmlData::~MAFXmlData() {
  if (mDocument != NULL)
    xmlFreeDoc(mDocument);
}

bool MAFXmlData::Load(const std::string &path, osgDB::ReaderWriter::Options* options) {
  mDocument = xmlParseFile(path.c_str());
  if (!mDocument)
	  return false;
  return true;
}


std::string MAFXmlData::Get(const std::string& path)
{
  const std::list<std::string> &result = GetList(path);
  return result.size() > 0 ? result.front() : "";
}

std::list<std::string> MAFXmlData::GetList(const std::string& path)
{
  
  std::list<std::string> result;

  
  xmlDocPtr header = mDocument;
  /* Create xpath evaluation context */
  xmlXPathContextPtr xpathCtx = xmlXPathNewContext(header);
  if(xpathCtx == NULL)
    g_error("MAFXmlData::GetList: unable to create new XPath context");

  /* Evaluate xpath expression */
  xmlXPathObjectPtr xpathObj = xmlXPathEvalExpression((xmlChar*)path.c_str(), xpathCtx);
  if(xpathObj == NULL) {
    g_error("Error: unable to evaluate xpath expression %s", path.c_str());
    xmlXPathFreeContext(xpathCtx); 
    throw this;
  }
  
  xmlNodeSetPtr nodes = xpathObj->nodesetval;
  if(nodes && nodes->nodeNr >= 1) {
    for(int i = 0; i < nodes->nodeNr; i++) {
      xmlNodePtr node = nodes->nodeTab[i];
      switch(node->type) {
      case XML_ELEMENT_NODE:
      case XML_ATTRIBUTE_NODE:
	{
	  const char* content = (const char*)xmlNodeGetContent(node);
	  result.push_back(content);
	  xmlFree((void*)content);
	}
	break;
      default:
	break;
      }
    }
  }
  
  xmlXPathFreeObject(xpathObj);
  xmlXPathFreeContext(xpathCtx); 
  return result;
}


MAFCursorData::MAFCursorData() : mCursor(NULL)
{
}

MAFCursorData::~MAFCursorData()
{
  SDL_FreeCursor(mCursor);
}

bool MAFCursorData::Load(const std::string &path, osgDB::ReaderWriter::Options* options)
{
  FILE *f = fopen(path.c_str(), "r");
  if (!f)
	  return false;
  int result = fread(mData, sizeof(char), 256, f);
  assert(result == 256);
  fclose(f);
  return true;
}

SDL_Cursor *MAFCursorData::GetCursor()
{
  return mCursor;
}

SDL_Cursor *MAFCursorData::CreateCursor()
{
  mCursor = SDL_CreateCursor(mData, mData+128, 32, 32, 0, 0);
  return mCursor;
}

SDL_Cursor *MAFCursorData::GetOrCreateCursor()
{
  if (mCursor == NULL)
    CreateCursor();
  return mCursor;
}

// MAFRepositoryData

MAFRepositoryData::~MAFRepositoryData() {
  for(std::map<std::string, MAFVisionData*>::iterator i = mVisionMap.begin();
      i != mVisionMap.end();
      i++)
    delete i->second;

  for(std::map<std::string, MAFAudioData*>::iterator j = mAudioMap.begin();
      j != mAudioMap.end();
      j++)
    delete j->second;
  for(std::map<std::string, MAFXmlData*>::iterator j = mXmlMap.begin();
      j != mXmlMap.end();
      j++)
    delete j->second;
  for(std::map<std::string, MAFCursorData*>::iterator j = mCursorMap.begin();
      j != mCursorMap.end();
      j++)
    delete j->second;
  if(mDesktop) delete mDesktop;
}

MAFVisionData* MAFRepositoryData::GetVision(const std::string& name)  {
  if(mVisionMap.find(name) == mVisionMap.end())
    throw new MAFError(UNDERWARE_MAF_ERROR_DATALOAD, "MAFVisionData::GetVision: no %s", name.c_str());
  
  return mVisionMap[name];
}

void MAFRepositoryData::Load(const std::string& dir) {
	GError* error = 0;
	GDir* dir_descriptor = g_dir_open(dir.c_str(), 0, &error);

	osgDB::ReaderWriter::Options* options = new osgDB::ReaderWriter::Options;
	options->setObjectCacheHint(osgDB::ReaderWriter::Options::CACHE_ALL);

	g_assert((error == 0 && dir_descriptor != 0) || (error != 0 && dir_descriptor == 0));

	if(!dir_descriptor) throw new MAFError(error);

	const gchar* file_char;
	while((file_char = g_dir_read_name(dir_descriptor))) {
		std::string file(file_char);
		if(file == "." || file == "..")
			continue;
		int dot = file.rfind('.');
		if(dot >= 0) {
			std::string lcfile = file;
			std::transform(lcfile.begin(), lcfile.end(), lcfile.begin(), tolower);
			std::string suffix = lcfile.substr(dot);
			std::string path = dir + "/" + file;
			if(suffix == ".wav") {
				g_message("loading wav '%s'", path.c_str());
				MAFAudioDataWAV* data = new MAFAudioDataWAV;
				if(data->Load(path, options))
				  mAudioMap[file] = data;
				else
				  delete data;
			} else if(suffix == ".ogg") {
				g_message("loading ogg '%s'", path.c_str());
				MAFAudioDataOGG* data = new MAFAudioDataOGG;
				if(data->Load(path, options))
				  mAudioMap[file] = data;
				else
				  delete data;
			} else if(suffix == ".cal3d") {
				g_message("loading cal3d '%s'", path.c_str());
				MAFCal3dData* data = new MAFCal3dData;
				if(data->Load(path, options))
				  mVisionMap[file] = data;
				else
				  delete data;
			} else if(suffix == ".osg") {
				g_message("loading osg '%s'", path.c_str());
				MAFOSGData* data = new MAFOSGData;
				if(data->Load(path, options))
				  mVisionMap[file] = data;
				else
				  delete data;
			} else if(suffix == ".escn") {
				g_message("loading escn '%s'", path.c_str());
				MAFESCNData * data = new MAFESCNData();
				if(data->Load(path, dir, file, options))
				  mVisionMap[file] = data;
				else
				  delete data;
			} else if (suffix == ".xml") {
				g_message("loading xml '%s'", path.c_str());
				MAFXmlData* data = new MAFXmlData;
				if(data->Load(path, options))
				  mXmlMap[file] = data;
				else
				  delete data;
			}	else if(suffix ==  ".cursor") {
				g_message("loading cursor '%s'", path.c_str());
				MAFCursorData *data = new MAFCursorData();
				if(data->Load(path, options))
				  mCursorMap[file] = data;
				else
				  delete data;
			} else if(suffix != ".jpg" && suffix != ".tga") {
				g_message("unknown resource %s", path.c_str());
			}
		}
	}

	g_dir_close(dir_descriptor);
}

void MAFRepositoryData::XwncConnect(const std::string& url) {
  g_debug("MAFRepositoryData::XwncConnect");
  mDesktop = new XwncDesktop((char*)url.c_str(), "foldable");
}

osgText::Font*	MAFLoadFont(const std::string& filename, osgDB::ReaderWriter::Options* options)
{
  osgText::Font*	font = dynamic_cast<osgText::Font*>
    (osgDB::readObjectFile(filename, options));
  if (!font)
    g_critical("unable to load font %s", filename.c_str());
  return font; // null font is the default font
}

osg::Image*	MAFLoadImage(const std::string& filename, osgDB::ReaderWriter::Options* options)
{
  osg::Image*	img = osgDB::readImageFile(filename, options);
  if (!img)
    g_critical("unable to load image file %s", filename.c_str());
  return img;
}
