/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Henry Precheur <henry@precheur.org>
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#include <glib.h>

#include <osg/Geode>
#include <osg/Material>
#include <osg/BlendFunc>

#include <maf/animate2d.h>
#include <maf/wnc_window.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#endif

#define VISIBLE_PIXELS 10

void MAFApplication2DAnimate::Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen) {
  for(MAFApplication2DAnimateActionList::iterator i = begin(); i != end(); i++) {
    if((*i).valid()) {
      (*i)->Configure(group, window, origin, geometry, screen);
    }
  }
}

void MAFApplication2DAnimate::Destroy(osg::Group* group, osg::MatrixTransform* window) {
  for(MAFApplication2DAnimateActionList::iterator i = begin(); i != end(); i++) {
    if((*i).valid()) {
      (*i)->Destroy(group, window);
    }
  }
}

void MAFApplication2DAnimate::Unmap(osg::Group* group) {
  for(MAFApplication2DAnimateActionList::iterator i = begin(); i != end(); i++) {
    if((*i).valid()) {
      (*i)->Unmap(group);
    }
  }
}

void MAFApplication2DAnimate::Map(osg::Group* group) {
  for(MAFApplication2DAnimateActionList::iterator i = begin(); i != end(); i++) {
    if((*i).valid()) {
      (*i)->Map(group);
    }
  }
}

void MAFApplication2DAnimate::Update(bool mouseOver, double delta) {
  for(MAFApplication2DAnimateActionList::iterator i = begin(); i != end(); i++) {
    if((*i).valid()) {
      (*i)->Update(mouseOver, delta);
    }
  }
}

MAFApplication2DSlide::MAFApplication2DSlide(int direction) :
  mDirectionFlags(direction),
  mMouseTrigger(false),
  mVisible(false),
  mSwitching(false),
  mSwitchingTime(0.f),
  mDelay(1000.f),
  mDestroyped(true)
{
}

MAFApplication2DSlide::~MAFApplication2DSlide() {}

void MAFApplication2DSlide::Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen)
{
  mOrigin = origin;

  if(!(mDirectionFlags & AXIS_MASK)) {
    if((mDirectionFlags & DIRECTION_MASK) == DIRECTION_NEGATIVE) {
      mDisplacement.x() = - (origin.x() + geometry.x());
      if(mMouseTrigger)
	mDisplacement.x() += VISIBLE_PIXELS;
    } else {
      mDisplacement.x() = screen.x() - origin.x();
      if(mMouseTrigger)
	mDisplacement.x() -= VISIBLE_PIXELS;
    }
  } else {
    if((mDirectionFlags & DIRECTION_MASK) == DIRECTION_NEGATIVE) {
      mDisplacement.y() = - (origin.y() + geometry.y());
      if(mMouseTrigger)
	mDisplacement.y() += VISIBLE_PIXELS;
    } else {
      mDisplacement.y() = screen.y() - origin.y();
      if(mMouseTrigger)
	mDisplacement.y() -= VISIBLE_PIXELS;
    }
  }
  
  if(mDestroyped && mWindow.valid()) {
    g_assert(dynamic_cast<XwncWindow*>(mWindow.get()) == 0);
    g_assert(mWindow->getParent(0) == group);
    group->removeChild(mWindow.get());
    mWindow = 0;
  }
  
  mWindow = window;
  mDestroyped = false;
  mSwitching = false;
  mSwitchingTime = 0.f;

  if(!mVisible)
    Slide(mDisplacement);
}

void MAFApplication2DSlide::Destroy(osg::Group* group, osg::MatrixTransform* window)
{
  mWindow = 0;
  mDestroyped = true;
  mSwitching = false;
  mSwitchingTime = 0.f;
}

void MAFApplication2DSlide::Update(bool mouseOver, double delta)
{
  if(!mWindow.valid()) return;

  if(mMouseTrigger) SetVisible(mouseOver);

  if(mSwitching) {
    osg::Vec2 offset;
    mSwitchingTime += delta;
    if(mSwitchingTime >= mDelay) {
      mSwitching = false;
      mSwitchingTime = 0.f;
      offset = mVisible ? osg::Vec2(0.f, 0.f) : mDisplacement;
    } else {
      float ratio = mSwitchingTime / mDelay;
      if(mVisible)
	ratio = 1.f - ratio;
      offset = mDisplacement * ratio;
    }
    Slide(offset);
  } else if(mDestroyped) {
    XwncWindow* xwncWindow = dynamic_cast<XwncWindow*>(mWindow.get());
    g_assert(xwncWindow == 0);
    osg::Group* parent = mWindow->getParent(0);
    parent->removeChild(mWindow.get());
    mWindow = 0;
  }
}

void MAFApplication2DSlide::Slide(const osg::Vec2& offset)
{
  if(mWindow.valid()) {
    XwncWindow* xwncWindow = dynamic_cast<XwncWindow*>(mWindow.get());
    if(xwncWindow)
      xwncWindow->Slide(offset);
    else
      mWindow->setMatrix(osg::Matrix::translate(mOrigin.x() + offset.x(), mOrigin.y() + offset.y(), 0.f));
  }
}

void MAFApplication2DSlide::SetVisible(bool visible)
{
  if(visible != mVisible) {
    if(mSwitching)
      mSwitchingTime = mDelay - mSwitchingTime;
    mSwitching = true;
    mVisible = !mVisible;
  }
}

MAFApplication2DSlideInOut::MAFApplication2DSlideInOut(int directionFlags) :
  MAFApplication2DSlide(directionFlags) {
}

void MAFApplication2DSlideInOut::Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen)
{
  mVisible = false;
  MAFApplication2DSlide::Configure(group, window, origin, geometry, screen);
  SetVisible(true);
}

void MAFApplication2DSlideInOut::Destroy(osg::Group* group, osg::MatrixTransform* window)
{
  if(group) {
    if(!mDestroyped) {
      group->addChild(window);
      mWindow = window;
      mDestroyped = true;
      SetVisible(false);
    } else {
      g_critical("MAFApplication2DSlideInOut::Destroy: unexpected multiple call to Destroy");
    }
  } else {
    g_critical("MAFApplication2DSlideInOut::Destroy: unexpected null group");
  }
}

void MAFApplication2DSlideInOut::Unmap(osg::Group* group)
{
  group->addChild(mWindow.get());
  SetVisible(false);
}

void MAFApplication2DSlideInOut::Map(osg::Group* group)
{
  group->removeChild(mWindow.get());
  SetVisible(true);
}

void MAFApplication2DAlpha::Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen)
{
  osg::StateSet* state = window->getOrCreateStateSet();
  osg::BlendFunc*	blend = new osg::BlendFunc(osg::BlendFunc::SRC_ALPHA,
						   osg::BlendFunc::ONE_MINUS_SRC_ALPHA);
  state->setAttributeAndModes(blend);
  if(!osg::equivalent(mAlpha, 1.f)) {
    osg::Material* material = new osg::Material();
    osg::Vec4 white(1.f, 1.f, 1.f, mAlpha);
    material->setColorMode(osg::Material::DIFFUSE);
    material->setDiffuse(osg::Material::FRONT_AND_BACK, white);
    material->setAmbient(osg::Material::FRONT_AND_BACK, white);
    material->setSpecular(osg::Material::FRONT_AND_BACK, white);
    material->setEmission(osg::Material::FRONT_AND_BACK, white);
    state->setAttributeAndModes(material);
  }
}

MAFApplication2DDecorate::MAFApplication2DDecorate() 
{
}

MAFApplication2DDecorate::~MAFApplication2DDecorate() 
{
}

void MAFApplication2DDecorate::Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen)
{
  if(mDecoration.valid())
    group->removeChild(mDecoration.get());
  else
    mDecoration = new osg::Geode;
  group->insertChild(0, mDecoration.get());
}

void MAFApplication2DDecorate::Destroy(osg::Group* group, osg::MatrixTransform* window)
{
  group->removeChild(mDecoration.get());
  mDecoration = 0;
}

void MAFApplication2DDecorate::Unmap(osg::Group* group)
{
  mDecoration->setNodeMask(0);
}

void MAFApplication2DDecorate::Map(osg::Group* group)
{
  mDecoration->setNodeMask(0xFFFFFFFF);
}

void MAFApplication2DDecorateSquare::Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen) 
{
  MAFApplication2DDecorate::Configure(group, window, origin, geometry, screen);
  if(mDecoration->getNumDrawables() > 0)
    mDecoration->removeDrawable(0, mDecoration->getNumDrawables());

  float left = origin.x() - mLeft;
  float bottom = origin.y() - mBottom;
  float right = origin.x() + geometry.x() + mRight;
  float top = origin.y() + geometry.y() + mTop;

  osg::Geometry* decoration = new osg::Geometry;

  float z = -1.f;

  osg::Vec3Array* square = new osg::Vec3Array(8);
  (*square)[0] = osg::Vec3(left, bottom, z);
  (*square)[1] = osg::Vec3(right, bottom, z);
  (*square)[2] = osg::Vec3(right, top, z);
  (*square)[3] = osg::Vec3(left, top, z);

  if(mBorder > 0) {
    (*square)[4] = osg::Vec3(left - mBorder, bottom - mBorder, z);
    (*square)[5] = osg::Vec3(right + mBorder, bottom - mBorder, z);
    (*square)[6] = osg::Vec3(right + mBorder, top + mBorder, z);
    (*square)[7] = osg::Vec3(left - mBorder, top + mBorder, z);
  }
  decoration->setVertexArray(square);

  osg::Vec4Array* colors = new osg::Vec4Array(2);
  (*colors)[0] = mBackgroundColor;
  if(mBorder > 0) {
    (*colors)[1] = mBorderColor;
  }
  decoration->setColorArray(colors);
  decoration->setColorBinding(osg::Geometry::BIND_PER_PRIMITIVE_SET);

  osg::DrawElementsUShort* background = new osg::DrawElementsUShort;
  background->setMode(osg::PrimitiveSet::QUADS);
  background->push_back(0);
  background->push_back(1);
  background->push_back(2);
  background->push_back(3);
  decoration->addPrimitiveSet(background);

  if(mBorder > 0) {
    osg::DrawElementsUShort* border = new osg::DrawElementsUShort;
    border->setMode(osg::PrimitiveSet::QUADS);
    // bottom
    border->push_back(0);
    border->push_back(4);
    border->push_back(5);
    border->push_back(1);
    // right
    border->push_back(1);
    border->push_back(5);
    border->push_back(6);
    border->push_back(2);
    // top
    border->push_back(2);
    border->push_back(6);
    border->push_back(7);
    border->push_back(3);
    // left
    border->push_back(3);
    border->push_back(7);
    border->push_back(4);
    border->push_back(0);
    decoration->addPrimitiveSet(border);
  }

  mDecoration->addDrawable(decoration);

  osg::StateSet* state = mDecoration->getOrCreateStateSet();
  osg::BlendFunc*	blend = new osg::BlendFunc(osg::BlendFunc::SRC_ALPHA,
						   osg::BlendFunc::ONE_MINUS_SRC_ALPHA);
  state->setAttributeAndModes(blend);
}
