/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   profile.h
   *
   * @brief   Profile system
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson, Based from article on Gems Game programming 3
   *	    Original authors GregHjelstrom and Byon Garrabrant
   *
   * @date    Created 2002/11
   *
   * @version $Id: profile.h,v 1.3 2004/09/30 16:28:34 izidor79 Exp $
   *
   ****************************************************************************/

#ifndef MAF_PROFILE_H
#define MAF_PROFILE_H

#ifdef USE_PROFILE

class ProfileNode;


  /**
   *  Main class for the profile system. Example of how to use the profile system
   *  imagine you have a function :
   *
   *  int Test()
   *  {
   *
   *    TestA();
   *    TestB();
   *    TestC();
   *
   *  }
   *
   *  if you want to profile the entire function you should do it like that:
   *
   *  int Test()
   *  {
   *    PRF_SAMPLE("Sample Test");
   *
   *    TestA();
   *    TestB();
   *    TestC();
   *
   *  }
   *
   *
   *  Imagine now you want to sample only a part of Test like only TestA and TestB
   *
   *  int Test()
   *  {
   *
   *    { 
   *     PRF_SAMPLE("TestA and TestB");
   *     TestA();
   *     TestB();
   *    }
   *
   *    TestC();
   *
   *  }
   *
   *  So with these example you can sample all functions you want
   */
  class Profile
  {

    /// root node
    ProfileNode* mRoot;

    /// current node
    ProfileNode* mCurrentNode;

    /// level
    int mLevel;

  public:


    /// Default constructor
    Profile();


    ~Profile();



    /** 
     *  Return Root node
     *
     *  @return ProfileNode* root node
     */
    ProfileNode* GetRootNode() { return mRoot;}



    /** 
     *  Start profile. It's create the node the first time, after it update the profile
     *
     *  @param const char* name of node to create or to update
     */
    void StartProfile(const char* _name);



    /** 
     *  Stop the profile
     */
    void StopProfile();



    /** 
     *  Set the limit subnode to examine. It's call Filter of the root node
     *
     *  @param int the limit depth
     */
    void SetFilter(int _level) { mRoot->SetFilter(_level);}



    /** 
     *  Update total time
     *
     *  @param double use a global time to compute good percent ...
     */
    void SetGlobalTime(double _globalTime) { mRoot->SetGlobalTime(_globalTime);}


  };



  /**
   *  This class is used to make a sample, that will be updated each time this sample 
   *  is executed  by your program. See the macro PRF_SAMPLE to make sample
   */
  class Sample
  {

  public:

    /** First entry to the profile sample*/
    Sample(const char *name);

    /** Stop the Profile sample*/
    ~Sample();

  };



/**
 *  Return the profiler
 */
Profile* GetProfiler();


/** 
 *  Use this macro to profile a function
 */
#define PROFILE_SAMPLE(a) Sample P__LINE__(a)

#else

#define PROFILE_SAMPLE(a)

#endif

#endif
