/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifndef _mafinterpolator_
#define _mafinterpolator_

#ifndef MAF_USE_VS_PCH
#include <iostream>
#include <osg/Quat>
#include <osg/Vec2>
#endif

#include <mth_hermite.h>
#include <maf/bezier.h>
#include <maf/data.h>

struct MAFInterpolatorConf
{
  typedef float Precision;
};

struct util
{
  typedef MAFInterpolatorConf::Precision F;  
  struct lerp
  {
    template<typename T>
    static inline void doit(T &out, const T& from, const T &to, F t)
    {
      out = from + ((to - from)*t);
    }
  };
  struct slerp
  {
    static inline void doit(osg::Quat &out, const osg::Quat& from, const osg::Quat& to, F t)
    {
      out.slerp(t, from, to);
    }
  };
  struct multislerp
  {
    static inline void doit(osg::Quat &out, const osg::Quat& from, const osg::Quat& to, F t, F scale)
    {
      double omega, cosomega, sinomega, scale_from, scale_to ;
      
      osg::Quat quatTo(to);
      
      cosomega = from.asVec4() * to.asVec4();
      
      if ( cosomega <0.0 )
	{ 
	  cosomega = -cosomega; 
	  quatTo = -to;
	}
      
      omega= acos(cosomega);
      sinomega = sin(omega);
      double phase = omega*(scale - 1.0)*t;
      
      scale_from = sin((1.0-t)*omega-phase)/sinomega ;
      scale_to = sin(t*omega+phase)/sinomega ;

      out = (from*scale_from) + (quatTo*scale_to);
    }
  };
};

template<typename T, typename D, typename F = MAFInterpolatorConf::Precision>
struct MAFInterpolator
{
  typedef D Derived;
  T mFrom;
  T mTo;
  void Init(const T& from, const T& to)
  {
    mFrom = from;
    mTo = to;
  }
  void Get(T &out, F t) const
  {
    F t1 = reinterpret_cast<const D *>(this)->GetT(t);
    Derived::Method::doit(out, mFrom, mTo, t1);
  }
  void Get(T &out, F t, F scale) const
  { 
    F t1 = reinterpret_cast<const D *>(this)->GetT(t);
    Derived::Method::doit(out, mFrom, mTo, t1, scale);
  }
};

template<typename T, typename M = util::lerp, typename F = MAFInterpolatorConf::Precision>
struct MAFLinearInterpolator : MAFInterpolator<T, MAFLinearInterpolator<T, M> >
{
  typedef T Type;
  typedef M Method;
  F GetT(F t) const {return t;}
};

template<typename T, typename M = util::lerp, typename F = MAFInterpolatorConf::Precision>
struct MAFSquareInterpolator : MAFInterpolator<T, MAFSquareInterpolator<T, M> >
{
  typedef T Type;
  typedef M Method;
  F GetT(F t) const {return t*t;}
};


template<typename T, int N>
struct metapow
{
  static inline T doit(const T &value)
  {
    return value * metapow<T, N - 1>::doit(value);
  }
};

template<typename T>
struct metapow<T, 0>
{
  static inline T doit(const T &value)
  {
    return 1;
  }
};

template<typename T, int N, typename M = util::lerp, typename F = MAFInterpolatorConf::Precision>
struct MAFPowInterpolator : MAFInterpolator<T, MAFPowInterpolator<T, N, M> >
{
  typedef T Type;
  typedef M Method;
  F GetT(F t) const {return metapow<F, N>::doit(t);}
};

template<typename T, typename M = util::lerp, typename F = MAFInterpolatorConf::Precision>
struct MAFHermiteInterpolator : MAFInterpolator<T, MAFHermiteInterpolator<T, M> >
{
  typedef T Type;
  typedef M Method;
  typedef MAFInterpolator<T, MAFHermiteInterpolator<T, M> > Base;
  typedef mth::Hermite_T<F, F> Hermite;
  typedef typename Hermite::Key_t Key;

  Hermite mHermite;
  std::vector<Key> mKeys;

  void InitKeys(const std::vector<osg::Vec2> &points)
  {
    mKeys.clear();
    int size = points.size();
    mKeys.resize(size);
    for (int i = 0; i < size; i++)
      {
	const osg::Vec2 &v = points[i];
	mKeys[i].Init(v.x(), v.y());
	mHermite.InsertKey(&mKeys[i]);
      }
  }
  F GetT(F t) const
  {
    F data;
    mHermite.GetData(t, data);
    return data;
  }
};

template<typename T, typename M = util::lerp, typename F = MAFInterpolatorConf::Precision>
struct MAFBezierInterpolator : MAFInterpolator<T, MAFBezierInterpolator<T, M> >
{
  typedef T Type;
  typedef M Method;
  typedef MAFInterpolator<T, MAFBezierInterpolator<T, M> > Base;
  typedef MAFBezier<osg::Vec2> Bezier;

  Bezier mBezier;

  void InitKeys(const std::vector<osg::Vec2> &points)
  {
    int size = points.size();
    for (int i = 0; i+4 <= size; i+=4)
      mBezier.Insert(points[i], points[i+1], points[i+2], points[i+3]);
  }
  F GetT(F t) const
  {
    osg::Vec2 result;
    mBezier.Get(result, t);
    return result.y();
  }
};

template<typename F = MAFInterpolatorConf::Precision>
struct MAFInterpolatorTimer
{
  MAFInterpolatorTimer() : mCurrent(0), mDuration(0) {}
  F mCurrent;
  F mDuration;
  void Init(F duration)
  {
    mCurrent = 0;
    mDuration = duration;
  }
  void AddTime(F delta)
  {
    F newTime = mCurrent + delta;
    
    if (newTime < mDuration)
      mCurrent  = (newTime < F(0) ? F(0) : newTime);
    else
      mCurrent = mDuration;
  }
  template<typename T, typename I>
    void Get(T &out, const I& interpolator) const
  {
    interpolator.Get(out, mCurrent/mDuration);
  }
  template<typename T, typename I>
    void Get(T &out, const I& interpolator, F scale) const
  {
    interpolator.Get(out, mCurrent/mDuration, scale);
  }
  bool Finished()
  {
    return mCurrent == mDuration;
  }
  F GetDuration() const { return mDuration; }
  F GetCurrentTime() const { return mCurrent; }
};

struct MAFInterpolatorHelper
{
  template<typename T>
  static void LoadSpline(T &interpolator, MAFXmlData *data, const std::string &name)
  {
    std::vector<osg::Vec2> keys;
    LoadKeys(keys, data, name);
    interpolator.InitKeys(keys);
  }
  static void LoadKeys(std::vector<osg::Vec2> &keys, MAFXmlData *data, const std::string &name)
  {
    if (data != NULL)
      {
	const std::list<std::string> &xResultList = data->GetList("/splines/" + name + "/list/entry/@xvalue");
	const std::list<std::string> &yResultList = data->GetList("/splines/" + name + "/list/entry/@yvalue");
	
	g_assert(xResultList.size() == yResultList.size());
	typedef std::list<std::string>::const_iterator It;
	It xbegin = xResultList.begin();
	It xend = xResultList.end();
	
	It ybegin = yResultList.begin();
	It yend = yResultList.end();
	
	while(xbegin != xend) {
	  keys.push_back(osg::Vec2(atof((*xbegin).c_str()), atof((*ybegin).c_str())));
	  xbegin++;
	  ybegin++;
   	}
      }
  }
};

#endif
