/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 */

#ifndef __CMULTIPLEANIMATIONPATHCALLBACK_H
#define __CMULTIPLEANIMATIONPATHCALLBACK_H

#ifndef MAF_USE_VS_PCH
#include <osg/Matrixf>
#include <osg/Matrixd>
#include <osg/Quat>
#include <osg/NodeCallback>
#include <osg/AnimationPath>

#include <maf/mafexport.h>
#include <map>
#include <istream>
#include <float.h>
#include <vector>
#endif

namespace osg
{
  typedef ref_ptr<AnimationPath> AnimationPathPTR;

  class MAF_EXPORT MultipleAnimationPathCallback : public osg::NodeCallback
  {
  public:

    MultipleAnimationPathCallback()
      : _pivotPoint(0.0f,0.0f,0.0f),
	_useInverseMatrix(false),
	_timeOffset(0.0),
	_timeMultiplier(1.0),
	_firstTime(DBL_MAX),
	_latestTime(0.0),
	_pause(false),
	_pauseTime(0.0)
    {

    }

    MultipleAnimationPathCallback(const MultipleAnimationPathCallback& apc,const CopyOp& copyop)
      : NodeCallback(apc,copyop),
	_pivotPoint(apc._pivotPoint),
	_useInverseMatrix(apc._useInverseMatrix),
	_timeOffset(apc._timeOffset),
	_timeMultiplier(apc._timeMultiplier),
	_firstTime(apc._firstTime),
	_latestTime(apc._latestTime),
	_pause(apc._pause),
	_pauseTime(apc._pauseTime),
        m_lAnimationPaths(apc.m_lAnimationPaths),
	m_name2AnimationPath(apc.m_name2AnimationPath)
    {

    }


    META_Object(osg,MultipleAnimationPathCallback);

    MultipleAnimationPathCallback(AnimationPath* ap,double timeOffset=0.0f,double timeMultiplier=1.0f)
      : _pivotPoint(0.0f,0.0f,0.0f),
	_useInverseMatrix(false),
	_timeOffset(timeOffset),
	_timeMultiplier(timeMultiplier),
	_firstTime(DBL_MAX),
	_latestTime(0.0),
	_pause(false),
	_pauseTime(0.0)
    {
		
    }


    //		void setAnimationPath(AnimationPath* path) { _animationPath = path; }
    AnimationPath* getAnimationPath() { return m_currentAnimationPath.get(); }
    const AnimationPath* getAnimationPath() const { return m_currentAnimationPath.get(); }
    AnimationPath* getAnimationPath(const std::string& name) { return m_name2AnimationPath[name].get(); }
    const AnimationPath* getAnimationPath(const std::string& name) const { std::map<std::string, AnimationPathPTR>::const_iterator i = m_name2AnimationPath.find(name); return (i == m_name2AnimationPath.end()) ? 0 : i->second.get(); }

    inline void setPivotPoint(const Vec3d& pivot) { _pivotPoint = pivot; }
    inline const Vec3d& getPivotPoint() const { return _pivotPoint; }

    void setUseInverseMatrix(bool useInverseMatrix) { _useInverseMatrix = useInverseMatrix; }
    bool getUseInverseMatrix() const { return _useInverseMatrix; }

    void setTimeOffset(double offset) { _timeOffset = offset; }
    double getTimeOffset() const { return _timeOffset; }

    void setTimeMultiplier(double multiplier) { _timeMultiplier = multiplier; }
    double getTimeMultiplier() const { return _timeMultiplier; }

    double getAnimationDuration() const { return getAnimationPath() ? getAnimationPath()->getPeriod() : 0.0;}
    double getAnimationDuration(const std::string& name) const { const AnimationPath* path = getAnimationPath(name); return path ? path->getPeriod() : 0.0;}

    void reset();

    void setPause(bool pause);

    /** get the animation time that is used to specify the position along the AnimationPath.
     * Animation time is computed from the formula ((_latestTime-_firstTime)-_timeOffset)*_timeMultiplier.*/
    double getAnimationTime() const;

    /** implements the callback*/
    virtual void operator()(Node* node, NodeVisitor* nv);

    void update(osg::Node& node);

    inline void addAnimationPath(const std::string& name, AnimationPath* ap)
    {
      m_name2AnimationPath[name] = ap;
      m_lAnimationPaths.push_back(ap);
    }

    inline void setCurrentAnimationPath(int nCurrentAnimationPath)
    {
      m_currentAnimationPath = m_lAnimationPaths[nCurrentAnimationPath];
    }

    inline void setCurrentAnimationPath(const std::string& name)
    {
      std::map<std::string, AnimationPathPTR>::iterator i = m_name2AnimationPath.find(name);
      if(i == m_name2AnimationPath.end())
	m_currentAnimationPath = 0;
      else
	m_currentAnimationPath = i->second.get();
    }

    inline bool isPlaying() {
      double time=getAnimationTime();
      if (time>=1e-4 && time<(getAnimationDuration()-1e-4))
	return true;
      return false;
    }

  public:

    osg::ref_ptr<AnimationPath> m_currentAnimationPath;

    //		ref_ptr<AnimationPath>  _animationPath;
    osg::Vec3d              _pivotPoint;
    bool                    _useInverseMatrix;
    double                  _timeOffset;
    double                  _timeMultiplier;
    double                  _firstTime;
    double                  _latestTime;
    bool                    _pause;
    double                  _pauseTime;

    std::vector<AnimationPathPTR>	m_lAnimationPaths;
    std::map<std::string, AnimationPathPTR> m_name2AnimationPath;

  protected:

    ~MultipleAnimationPathCallback()
    {
		
    }

  };
}

#endif // __CMULTIPLEANIMATIONPATHCALLBACK_H

