/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2003 Robert Osfield 
 *
 * This library is open source and may be redistributed and/or modified under  
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or 
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * OpenSceneGraph Public License for more details.
 */

#ifndef exg_nodevisitor_h
#define exg_nodevisitor_h

#include <exg/exg_object.h>

namespace exg {

  class VectorObjectPointer;
  class MapObjectPointer;
  class Material;
  class Polygon;
  class Vertex;
  class Point;
  class File;
  class Mesh;

  typedef std::vector<Object*> NodePath;
  typedef std::map<Object*,int> NodeTrace;

  /** Visitor for type safe operations on osg::Node's.
      Based on GOF's Visitor pattern. The NodeVisitor 
      is useful for developing type safe operations to nodes
      in the scene graph (as per Visitor pattern), and adds to this
      support for optional scene graph traversal to allow
      operations to be applied to whole scenes at once. The Visitor
      pattern uses a technique of double dispatch as a mechanism to
      called the appropriate apply(..) method of the NodeVisitor.  To
      use this feature one must use the Node::accept(NodeVisitor) which
      is extended in each Node subclass, rather than the NodeVisitor
      apply directly.  So use root->accept(myVisitor); instead of
      myVisitor.apply(*root).  The later method will bypass the double
      dispatch and the appropriate NodeVisitor::apply(..) method will
      not be called. */ 
  class EXG_EXPORT Visitor
  {
   public:

    enum TraverseType {
      ONCE,
      ALL};

    Visitor(TraverseType type=ALL):mTraverseType(type){ }
        
        
    virtual ~Visitor(){}

    /** Method to call to reset visitor. Useful for your visitor accumulates
        state during a traversal, and you plan to reuse the visitor. 
        To flush that state for the next traversal than call reset() prior
        to each traversal.*/
    virtual void Reset() {
      mNodeTrace.clear();
    }



    /** Method for handling traversal of a nodes.
        If you intend to use the visitor for actively traversing 
        the scene graph then make sure the accept() methods call
        this method unless they handle traversal directly.*/
    inline void Traverse(Object& node) {
      node.Traverse(*this);
    }

        
    /** Method called by osg::Node::accept() method before
     * a call the Visitor::apply(..).  The back of the list will,
     * therefore, be the current node being visited inside the apply(..),
     * and the rest of the list will be the parental sequence of nodes 
     * from the top most node applied down the graph to the current node.
     * Note, the user does not typically call pushNodeOnPath() as it
     * will be called automatically by the Node::accept() method.*/
    inline void PushOntoNodePath(Object* node) { mNodePath.push_back(node); }
        
    /** Method callby osg::Node::accept() method after
     * a call the Visitor::apply(..).
     * Note, the user does not typically call pushNodeOnPath() as it
     * will be called automatically by the Node::accept() method.*/
    inline void PopFromNodePath()            { mNodePath.pop_back(); }
        
    /** Get the non const NodePath from the top most node applied down
     * to the current Node being visited.*/
    NodePath& GetNodePath() { return mNodePath; }

    /** Get the const NodePath from the top most node applied down
     * to the current Node being visited.*/
    const NodePath& GetNodePath() const { return mNodePath; }
        

    inline int CheckAndTagNode(Object& node) {
      if (mTraverseType==ONCE) {
        if (mNodeTrace.find(&node)!=mNodeTrace.end())
          return 0;
        mNodeTrace[&node]++;
      }
      return 1;
    }

    /**
     *  Method that users can redefine
     */
    virtual void Apply(Object& node)              { Traverse(node);}
    virtual void Apply(Material& node)            { Apply((Object&)node); }
    virtual void Apply(Polygon& node)             { Apply((Object&)node); }
    virtual void Apply(Vertex& node)              { Apply((Object&)node); }
    virtual void Apply(Point& node)               { Apply((Object&)node); }
    virtual void Apply(Mesh& node)                { Apply((Object&)node); }
    virtual void Apply(File& node)                { Apply((Object&)node); }

   protected:

    TraverseType mTraverseType;    
    NodePath mNodePath;
    NodeTrace mNodeTrace;
  };


}

#endif
