/**
    Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 ****************************************************************************
 * @file   exg_vector4.h
 *
 * @brief   4 dimensional vectors
 *
 *****************************************************************************
 *
 * @author  Cedric Pinson
 *
 * @date    Created 2001/04
 *
 * @version $Id: exg_vector4.h,v 1.5 2004/07/15 10:23:55 sylvain Exp $
 *
 ****************************************************************************/

#ifndef exg_vector4_h
#define exg_vector4_h


#include <cmath>
#include <exg/exg_vector3.h>


/** 4D vector
 *
 * Can be accessed through the [] operator. Standard operation such as
 * +, -, Dot, Dist (distance to another vector), Mag (magnitude),  ...
 * can be performed.
 *
 * @warning The default constructor does not initialize the coordinates.
 *
 */


namespace exg {

  
  template <class T> class Vector4_T {

   protected:

    T c[4];
   
   public:


    /// do nothing
    Vector4_T() {}


    /// constructor with scalar
    inline Vector4_T( T _a, T _b, T _c, T _d=0) {
      Init(_a,_b,_c,_d);
    }


    /// Initialize the vector with another vector
    inline Vector4_T(const Vector3_T<T>& _v,T _a=0) {
      Init(_v[0],_v[1],_v[2],_a); }

	virtual ~Vector4_T(void)
	{
	}


    /// Initialize the vector with another vector
    template <class C> explicit Vector4_T(const Vector4_T<C>& _v) {
      c[0]=(T)_v[0]; c[1]=(T)_v[1]; c[2]=(T)_v[2]; c[3]=(T)_v[3];
    }


    /// Set the vector to 0
    inline void Init() {
      c[0]=0; c[1]=0; c[2]=0; c[3]=0;
    }


    /// Set the vector to 0
    inline void Init( T _a, T _b, T _c, T _d) {
      c[0]=_a; c[1]=_b; c[2]=_c; c[3]=_d;
    }


    /// Initialize the vector with another vector
    template <class C> void Init(const Vector4_T<C>& _v) {
      c[0]=(T)_v.c[0]; c[1]=(T)_v.c[1]; c[2]=(T)_v.c[2]; c[3]=(T)_v.c[3];
    }


    /// Initialize the vector with another vector
    template <class C> void Init(const Vector3_T<C>& _v, T _a=0) {
      c[0]=(T)_v[0]; c[1]=(T)_v[1]; c[2]=(T)_v[2]; c[3]=_a;
    }



    /// Compute the dot product against another vector
    inline T Dot(const Vector4_T& _v) const {
      return c[0]*_v.c[0]+c[1]*_v.c[1]+c[2]*_v.c[2]+c[3]*_v.c[3];
    }



    /// Multiply by row
    inline Vector4_T Multiply(const Vector4_T& _v) const {
      return Vector4_T(c[0]*_v.c[0],c[1]*_v.c[1],c[2]*_v.c[2],c[3]*_v.c[3]);
    }



    /// Return the squared magnitude (length) of the vector
    inline T MagSqr() const { 
      return c[0]*c[0]+c[1]*c[1]+c[2]*c[2]+c[3]*c[3];
    }


    /// Return the magnitude (length) of the vector
    inline T Mag() const { return (T)sqrt(MagSqr()); }


    
    /// Return the squared distance between two vectors
    inline T DistSqr(const Vector4_T& _v) const {
      return (*this-_v).MagSqr();}


    /// Return the distance between two vectors
    inline T Dist(const Vector4_T& _v) const {
      return (*this-_v).Mag(); }


    
    /** Normalize this vector
     *
     */
    inline void Normalize() {
      T a = MagSqr();
      *this *= (T)(1./sqrt(a));
    }


    /// Coordinates accessor operator
    inline T& operator [] (int _i) { return c[_i]; }

    
    /// Const coordinates accessor operator
    inline const T& operator [] (int _i) const { return c[_i]; }


    /// Operator * (dot product)
    inline T operator * (const Vector4_T& _v) const {
      return this->Dot(_v);}


    /// Addition of two vectors
    inline Vector4_T operator + (const Vector4_T& _v) const {
      return Vector4_T(c[0]+_v.c[0],c[1]+_v.c[1],c[2]+_v.c[2],c[3]+_v.c[3]);
    }


    /// Substract of two vectors
    inline Vector4_T operator - (const Vector4_T& _v) const {
      return Vector4_T(c[0]-_v.c[0],c[1]-_v.c[1],c[2]-_v.c[2],c[3]-_v.c[3]);
    }


    /// Comparaison operator
    bool operator == ( const Vector4_T& _v ) const {
      return c[0]==_v.c[0] && c[1]==_v.c[1] && c[2]==_v.c[2] && c[3]==_v.c[3];
    }


    /// Comparaison operator
    bool operator != ( const Vector4_T& _v ) const {
      return !(*this==_v);}


    /// += operator
    inline const Vector4_T& operator += ( const Vector4_T& _v ) {
      c[0]+=_v.c[0]; c[1]+=_v.c[1]; c[2]+=_v.c[2]; c[3]+=_v.c[3];
      return *this;}


    /// -= operator
    inline const Vector4_T& operator -= ( const Vector4_T& _v ) {
      c[0]-=_v.c[0]; c[1]-=_v.c[1]; c[2]-=_v.c[2]; c[3]-=_v.c[3];
      return *this;}


    /// *= operator
    inline const Vector4_T& operator *= ( T _v ) {
      c[0]*=_v; c[1]*=_v; c[2]*=_v; c[3]*=_v;
      return *this;}


    /// /= operator
    inline const Vector4_T& operator /= ( T _v ) {
      T a=(T)1.0/_v;
      c[0]*=a; c[1]*=a; c[2]*=a; c[3]*=a;
      return *this;}



    /// Operator -
    inline Vector4_T operator - () const {
      return Vector4_T(-c[0],-c[1],-c[2],-c[3]);
    }


    /// Mutliplication by a scalar value (to the right)
    inline Vector4_T operator * ( T _a) const {
      return Vector4_T(c[0]*_a,c[1]*_a,c[2]*_a,c[3]*_a);
    }


    /// Division by a scalar value
    inline Vector4_T operator / (T _v) const {
      T a=(T)1.0/_v;
      return Vector4_T(c[0]*a,c[1]*a,c[2]*a,c[3]*a);
    }

  };


  template <class T> inline Vector4_T<T> operator * (T _a, const Vector4_T<T>& _v) {
    return Vector4_T<T>(_a*_v[0], _a*_v[1], _a*_v[2],_a*_v[3]); }


  typedef Vector4_T<float> Vector4f;
  typedef Vector4_T<double> Vector4d;

}
#endif
