from xmlstream import XMLStream
import elements, utils
from twisted.protocols.basic import LineReceiver
from twisted.internet import protocol, reactor, defer
from twisted.python import log, failure, dispatch
import time, random, sys
import twisted

# some name shortcuts
get_text=elements.get_text
get_child=elements.get_child
ezel=elements.ezel
roster_item=elements.roster_item

#debug related functions
encoding='ISO-8859-15'
debug=1

#Namespaces URIs
ns_xmlstream='http://etherx.jabber.org/streams'
ns_jabberclient='jabber:client'
ns_jabberserver=''
ns_tls='urn:ietf:params:xml:ns:xmpp-tls'
ns_sasl='urn:ietf:params:xml:ns:xmpp-sasl'
ns_resource_binding='urn:ietf:params:xml:ns:xmpp-bind'
ns_session='urn:ietf:params:xml:ns:xmpp-session'
ns_session_iq='urn:ietf:params:xml:ns:xmpp-session'
ns_dialback='jabber:server:dialback'

#Events generated by the factory
JC_CONNECTION_LOST="//event/jabber/client/connection/lost"
JC_CONNECTION_FAILED="//event/jabber/client/connection/failed"

class XMPPStream(XMLStream, dispatch.EventDispatcher):
    """ Protocol Class handling XMPP Events """
   
    def _dummyHandler(self, tag):
        if debug:
            utils.debug(tag.toxml().encode(encoding, 'ignore'), xml=1)
        
    def __getattr__(self, k):
        """ """
        if k.find('_handle')==0:
            # return the dummy handler
            return self._dummyHandler
        else:
            raise AttributeError('%s not found'%(k,))
    
    def __init__(self):
        XMLStream.__init__(self)
        dispatch.EventDispatcher.__init__(self)

        # handling iq deferreds
        self._id_count=0
        self._pending_iqs={} #storing deferreds for iqs
    
        self._iq_handlers={}
        
        # XXX not all these elements are first level stanzas (i.e. SASL mechanisms)
        self._handlers = {
            #  Streams namespace
            ns_xmlstream: {
                'stream'    : self._handleStreamOpened,
                'features'  : self._handleFeatures,
                'error'     : self._handleError,
                },
        
            # TLS Namespace
            ns_tls: {
                'starttls'  : self._handleStartTLS,
                'required'  : self._handleTLSRequired,
                'proceed'   : self._handleTLSProceed,
                'failure'   : self._handleTLSFailure, 
                },
        
            # SASL namespace
            ns_sasl: {
                'mechanisms': self._handleSASLMechanisms, #XXX
                'auth'      : self._handleSASLAuth,
                'challenge' : self._handleSASLChallenge,
                'response'  : self._handleSASLResponse,
                'abort'     : self._handleSASLAbort,
                'success'   : self._handleSASLSuccess,
                'failure'   : self._handleSASLFailure,
                },
        
            # Resource binding namespace
            ns_resource_binding: {
                'bind'      : self._handleBind,
                },

            # Dialback namespace
            ns_dialback: {
                'result'    : self._handleDialbackResult,
                'verify'    : self._handleDialbackVerify,
                },

            # Default namespace
            '' : {
                'iq'        : self._handleIq,
                'message'   : self._handleMessage,
                'presence'  : self._handlePresence,
                }
                
            }
        
    # These two methods may be used in order to dynamically plug-in handlers
    # for XMPP stanzas
    def setHandler(self, ns, element, handler):
        """ Set the handler for ns:element """
        h=self._handlers.setdefault(ns,{})
        h[element]=handler
    
    def removeHandler(self, ns, element):
        """ Remove the handler for ns:element """
        del self._handlers[ns][element]        
    
    def setIqHandler(self, ns, element, handler):
        """ Set the iq handler for ns:element """
        h=self._iq_handlers.setdefault(ns,{})
        h[element]=handler
    
    def removeIqHandler(self, ns, element):
        """ Remove the iq handler for ns:element """
        del self._iq_handlers[ns][element]        
    
    def getId(self):
        """ return the stream id """
        return self._id
 
    def streamOpened(self, tag):
        """ Received the opening stream event, do all the init stuff
        """
        
        f=self.factory
        if f.debug:
            utils.debug(tag.toxml(), xml=1)
            
        #get the stream version
        self.version=tag.getAttribute('version')
        #Read the name of the peer
        self.peer=tag.getAttribute('from')
        #Get the id
        self._id=tag.getAttribute('id')
        
        #if f.role in ['c2s server', 's2s receiving']:
        #    self.openStream()
   
    def openStream(self):
        """ Make a stream opening header and send it 
        Subsequently, """
    
        f=self.factory
        ch=self.transport
        
        if f.xmlns: s_xmlns="xmlns='%s' "%(f.xmlns)
        else: s_xmlns="" 
        s_xmlns+=' '.join(["xmlns:%s='%s'"%(x,y) for x,y in f.namespaces.items()])
        if f.role=='c2s client':
            version="version='%s'"%(f.version)
            s="""<?xml version='1.0' encoding='UTF-8'?>
<stream:stream to='%s' %s %s>"""%(ch.getPeer()[1], s_xmlns, version)
        elif f.role=='c2s server':
            self._id=str(random.randint(0,sys.maxint-1))
            s="""<?xml version='1.0' encoding='UTF-8'?>
<stream:stream from='%s' id='%s' %s version='1.0'>"""%(ch.hostname, self._id, s_xmlns)
        elif f.role=='s2s receiving':
            self._id=str(random.randint(0,sys.maxint-1))
            s="""<?xml version='1.0' encoding='UTF-8'?>
<stream:stream id='%s' %s version='1.0'>"""%(self._id, s_xmlns)
        elif f.role in ['s2s open', 's2s dialback']:
            s="""<?xml version='1.0' encoding='UTF-8'?>
<stream:stream %s version='1.0'>"""%(s_xmlns)
            
        if f.debug:
            utils.debug(s, xml=1)

        self.writeData(s)
        
    def closeStream(self):
        self.transport.write("</stream:stream>")
        self.transport.loseConnection()
        
    def elementReceived(self, tag):
        """ xmpp element received, route the packets """

        if debug:
            utils.debug(tag.toxml().encode(encoding, 'ignore'), xml=1)
            utils.debug(("""  namespace: %s\n  name: %s"""%(tag.namespaceURI, tag.localName)).encode(encoding, 'ignore'))
        
        #this is often useful here ;)
        #if tag.localName=='route':
        #    import pdb; pdb.set_trace()

        # check the default ns
        if tag.namespaceURI==self.factory.xmlns: ns=''
        else: ns=tag.namespaceURI
        
        # call the handler
        try: h=self._handlers[ns][tag.localName]
        except KeyError: self.gotUnknownElement(tag)
        else: h(tag)
    
    def _handleIq(self, tag):
        """ Handle <iq/> stanzas """
        iq=elements.Iq(element=tag)
        _id=iq.getID()
        if  _id in self._pending_iqs:
            d=self._pending_iqs[_id]
            del self._pending_iqs[_id]
            type=iq.getType()
            if type=='result':
                #callbacks for pending iqs
                d.callback(iq)
            elif type=='error':
                d.errback(failure.Failure(iq,'iq'))
        else:
            # XXX Is it possible to have more than one child?
            # and do we always have one child?
            for c in tag.childNodes:
                if c.nodeType==c.ELEMENT_NODE: break
            try: h=self._iq_handlers[c.namespaceURI][c.localName]
            except KeyError: self.gotIq(iq) #generic iq
            else: h(iq) #specialized handler
                
    def _handlePresence(self, tag):
        """ Handle <presence/> stanzas """
        presence=elements.Presence(element=tag)
        self.gotPresence(presence)

    def _handleMessage(self, tag):
        """ Handle <message/> stanzas """
        message=elements.Message(element=tag)
        self.gotMessage(message)
        
    
    def _handleFeatures(self, tag):
        """ """
        self.gotFeatures(tag)
            
    #--------------------
    #XMPP regular stanzas
    
    def gotIq(self, iq):
        """ Got an iq element """
        #override
    
    def gotMessage(self, message):
        """ Got a message element """
        #override
        
    def gotPresence(self, presence):
        """ Got a presence element """
        #override
    
    def gotUnknownElement(self, tag):
        """ Got an unknown element """
        if debug:
           utils.debug(tag.toxml().encode(encoding,'ignore'), xml=1)
           
    def _makeupIq(self, iq, timeout=300):
        """ make the iq """
        
        #set an id if absent
        id=iq.getID()
        if not id:
            id=self._id_count
            iq.setID(str(id))
            self._id_count+=1
            
        d=None
        if iq.getType() in ['set', 'get']:
            d=defer.Deferred()
            d.setTimeout(timeout,  self._gotIqTimeOut, str(id))
            self._pending_iqs[str(id)]=d
            
        return d
           
    def sendIq(self, iq, timeout=300):
        """ send an iq packet and return a deferred for the anwser (only for
        iqs of type set and get) """
        
        d=self._makeupIq(iq)
        
        if debug:
            utils.debug(iq.toxml().encode(encoding, 'ignore'), xml=1)
        self.sendStanza(iq)
        # XXX here we should add an errback intercepting errors which are not 
        # Jabber errors (define a JabberException?)
        return d
    
    def _gotIqTimeOut(self, fail, id):
        """ Iq timeout """
        #just be sure to delete the deferred
        d=self._pending_iqs[id]
        del self._pending_iqs[id]
        d.errback(defer.TimeoutError())
        
    def sendPacket(self, tag):
        """ send a generic packet """
        if debug:
            utils.debug(tag.toxml().encode(encoding, 'ignore'), xml=1)
        self.transport.write(tag.toxml().encode('UTF-8'))
    
    def sendStanza(self, tag):
        """ send a generic packet """
        if debug:
            utils.debug(tag.toxml().encode(encoding, 'ignore'), xml=1)
        self.transport.write(tag.toxml().encode('UTF-8'))
        
    def sendMessage(self, to, subject='', body='', thread='', ttype=''):
        """send a message """
        #XXX Rewrite the method signature in order to accept Messages
        m=elements.Message(to=to, subject=subject, body=body, thread=thread, ttype=ttype)
        self.sendStanza(m)
        
    def gotFeatures(self, tag):
        """ Got a stream features tag """
        #save the features
        self.features_tag=tag
        
    def sendPresence(self, show='', status='', priority='5', to=''):
        """ """
        #XXX change the method signature in order to accept presence stanzas
        priority=str(priority)
        p=elements.Presence(to=to, show=show, status=status, priority=priority)
        self.sendStanza(p)
   
    # This fixes a bug in python.dispatch
    def publishEvent(self, name, *args, **kwargs):
        # here the original publishEvent rises an exception if no listerner
        # is registered
        for cb in self.callbacks.get(name,[]):  
            cb(*args, **kwargs)
   
class XMPPBaseFactory(dispatch.EventDispatcher):
    """ Base class for XMPP factories; it must derived together with a real 
    factory """
    
    def __init__(self):
        """  """
        dispatch.EventDispatcher.__init__(self)
        self._listeners=[]
        
    def _bindListeners(self, protocol):
        for e,l in self._listeners:
            protocol.registerHandler(e, l)
    
    def buildProtocol(self, addr):
        raise NotImplemented('buildProtocol must be implemented')
        
    def listenEvent(self, event, callable):
        """ Register an event listener, in order to be called when the 
        connection is dropped or authentication is made, and etc. """
        if event in [JC_CONNECTION_FAILED, JC_CONNECTION_LOST]:
            self.registerHandler(event, callable)
        self._listeners.append((event, callable))

    def clientConnectionFailed(self, connector, reason):
        """ """
        self.publishEvent(JC_CONNECTION_FAILED, self, reason)
        
    def clientConnectionLost(self, connector, reason):
        """ """
        self.publishEvent(JC_CONNECTION_LOST, self)
    
    # This fixes a bug in python.dispatch
    def publishEvent(self, name, *args, **kwargs):
        # here the original publishEvent rises an exception if no listerner
        # is registered
        for cb in self.callbacks.get(name,[]):  
            cb(*args, **kwargs)
        
#XXX this class should be moved outside, to the examples section
class JabberConsole(LineReceiver):
    """ a very very stupid console: this is just an example of how to connect
    a console to the client jabber protocol """
    
    delimiter = '\n'
    
    def __init__(self, jabber_client):
        """ """
        self.jc=jabber_client
    
    def lineReceived(self, line):
        line = line.rstrip() # this sacrifices trailing white space and/or a possible '\r' if on windows?
        if line.lower() in ('quit', 'q'):
            reactor.stop()
        print "You have just typed: %s" % (line,)
        m=elements.Message(to='sciasbat@cselt2.polito.it', body=line.decode('ISO-8859-1'))
        
        print "--->",m.toxml().encode(encoding)
        self.jc.sendPacket(m)
        
