"""
Jabber elements handler 
(c) 2003 Fabio Forno fabio.forno@polito.it
This software is Opensource
version 0.1
"""

from types import StringTypes
from xml.dom import minidom


#NS_AUTH       = "jabber:iq:auth"
#NS_AUTH_ERROR = "jabber:iq:auth:error"
#NS_CLIENT     = "jabber:client"
#NS_SERVER     = "jabber:server"
#NS_ROSTER     = "jabber:iq:roster"

xmlstream_uri='http://etherx.jabber.org/streams'
jabberclient_uri='jabber:client'
jabberserver_uri='jabber:server'
jabbercomponent_accept_uri='jabber:component:accept'
sasl_param_uri='urn:ietf:params:xml:ns:xmpp-sasl'
iq_auth_uri='jabber:iq:auth'
iq_roster_uri='jabber:iq:roster'
dialback_uri='jabber:server:dialback'

from xmlutils import *

#Utility functions to build quickly jabber elements
#XXX Are these functions still called somewhere?
def db_node(tagname, origin, destination, id='', ttype=''):
    """ make a db node """
    #use lmx for fast xml writing
    db=ezel('db:%s'%(tagname))
    db['to']=destination
    db['from']=origin
    if ttype: db['type']=ttype
    if id: db.text(id)
    return db.node
    
def roster_item(contact, **kw):
    """ build a roster item """
    
    #remove group
    if kw.get('group', None):
        group=Element(tagName='group', text=kw['group'])
        del kw['group']
    else: group=None
    
    d={}
    for k,v in kw.items():
        if v: d[k]=v
    d['jid']=contact
    
    node=Element(tagName='item', attributes=d)
    if group: node.appendChild(group)
    return node

def iq_auth_error(code):
    
    error=Element(tagName='error', attributes={'code':'%d'%(code), 'class':'app'})
    if code==401:
        condition=Element(tagName='auth-condition', attributes={'xmlns':NS_AUTH_ERROR})
        condition.appendChild(Element(tagName='user-unauthorized'))
        error.appendChild(condition)
    return error

def presence(attrs, children):
    """ make a presence element 
    attrs and children are two dictionaries describing how to compose
    the presence element
    """
    element=ezel('presence')
    #Add all attributes passed in kw
    for attr in ['to','from','id','type','error']:
        if attrs.has_key(attr):
            element[attr]=attrs[attr]
    #Add all child nodes passed in kw
    for child in ['show', 'status', 'priority', 'error']:
        if children.has_key(child):
            element.add(child).text(children[child])
    return element.node

def vcard(**kw):
    """ make a vcard 
    pass all items as kw arguments
    """
    #TODO some vcard fields are nested with others: fix
    el=ezel('vcard')
    el['prodid']='-//HandGen//NONSGML vGen v1.0//EN'
    el['xmlns']='vcard-temp'
    el['version']='2.0'
    for k,v in kw.items():
        el.add(k).text(v)
    return el.node

   
class JabberElementBase:
    """ Generic Jabber Element """
    
    def __init__(self, element, to='', ttype=None, ffrom='', id=''):
        """ Build a jabber root element 
        If element is a string, it is interpreted as the element name
        Otherwise it must be a minidom element
        """
        if type(element) in StringTypes:
            self._el=Element(tagName=element)
        else:
            self._el=element

        if to: self.setTo(to)
        if ffrom: self.setFrom(ffrom)
        if ttype: self.setType(ttype)
        if id: self.setID(id)

    def toxml(self):
        """ Convert to XML """
        return self._el.toxml()
    
    __str__=toxml
    __repr__=__str__
    
    def getName(self):
        """ return the name of the element (i.e. message, iq, presence) """
        return self._el.localName
        
    def getDOM(self):
        """ return the DOM element """
        return self._el
        
    def getTo(self):
        """Returns the 'to' attribute as a JID object."""
        try: return JID(self._el.getAttribute('to'))
        except: return None
    
    def getFrom(self):
        """Returns the 'from' attribute as a JID object."""
        try: return JID(self._el.getAttribute('from'))
        except: return None

    def getType(self):
        """Returns the 'type' attribute of the protocol element."""
        try: return self._el.getAttribute('type')
        except: return None


    def getID(self):
        """Returns the 'id' attribute of the protocol element."""
        try: return self._el.getAttribute('id')
        except: return None

    def setTo(self,val):
        """Sets the 'to' element to the given JID."""
        self._el.setAttribute('to', str(val))


    def setFrom(self,val):
        """Sets the 'from' element to the given JID."""
        self._el.setAttribute('from', str(val))


    def setType(self,val):
        """Sets the 'type' attribute of the protocol element"""
        self._el.setAttribute('type', val)

    def setID(self, val):
        """Sets the ID of the protocol element"""
        self._el.setAttribute('id', val)
    
    def swapToFrom(self):
        """ Swaps the to and from fields """
        to=self.getTo()
        self.setTo(self.getFrom())
        self.setFrom(to)
    
    def appendChild(self, node):
        self._el.appendChild(node)

    def getChild(self, name):
        for c in self._el.childNodes:
            if c.localName==name: return c
        return None
    
    def getChildNS(self, name, ns):
        for c in self._el.childNodes:
            if c.localName==name and c.namespaceURI==ns: return c
        return None
        
    def getChildren(self, name):
        return [c for c in self._el.childNodes if c.localName==name]
    
    def getChildrenNS(self, name, ns):
        return [c for c in self._el.childNodes if c.localName==name and c.namespaceURI==ns]
        
    def hasChild(self, name):
        return has_child(self._el, name)
    
    def removeChild(self, name):
        n=self.getChild(name)
        if n: self._el.removeNode(n)
        
    def setChildWithText(self, name, text):
        """ Set a child element containg text. This child must be 
        unique inside the first level element
        """
        self.removeChild(name)
        e=Element(name,text=text)
        self.appendChild(e)
        return e
        
    def getChildText(self, name):
        """ Get the text of a child """
        n=self.getChild(name)
        if n: return get_text(n)
        else: return ''
    
    def appendXML(self, s):
        """ append XML directly: this string must contain a root node """
        doc=minidom.parseString(s)
        self.appendChild(doc.firstChild)
        
class Iq(JabberElementBase):
    
    def __init__(self, to='', ttype=None, ffrom='', id='', element=None):
        if element:
            apply(JabberElementBase.__init__, (self, element, to, ttype, ffrom, id))
        else:
            apply(JabberElementBase.__init__, (self, 'iq', to, ttype, ffrom, id))

    def getError(self):
        """Returns the Iq's error string, if any"""
        try:
            error=get_text(get_child(self._el, 'error'))
            return error
        except:
            return None


    def getErrorCode(self):
        """Returns the Iq's error code, if any"""
        try: return get_child(self._el, 'error').getAttribute('code')
        except: return None


    def setError(self, val, code):
        """Sets an Iq's error string and code"""
        err = get_child(self._el,'error')
        if not err:
            err = Element(tagName='error')
            self._el.appendChild(err)
        set_text(err, val)
        err.setAttribute('code', str(code))
        
    def getQueryNS(self):
        """returns the query namespace"""
        tag=get_child(self._el, 'query')
        if tag and tag.hasAttribute('xmlns'):
            return tag.getAttribute('xmlns')
        else:
            return None
    
    def setQueryNS(self,namespace):
        """set the query namespace"""
        tag=get_child(self._el, 'query')
        if tag is None:
            tag=ezel('query').dom()
            self._el.appendChild(tag)
        tag.setAttribute('xmlns', namespace)
        return tag
    
    def getQueryNode(self):
        return get_child(self._el, 'query')
    
    
    def makeResultFrom(self, type='result'):
        """ """
        return Iq(to=self.getFrom(), id=self.getID(), ttype=type, ffrom=self.getTo())
    
class Message(JabberElementBase):

    def __init__(self, to='', ttype=None, ffrom='', id='', element=None, **kw):
        """ """
        if element:
            apply(JabberElementBase.__init__, (self, element, to, ttype, ffrom, id))
        else:
            apply(JabberElementBase.__init__, (self, 'message', to, ttype, ffrom, id))

        # XXX ever used? 
        for k,v in kw.items():
            if v:
                fname='set'+k[0].capitalize()+k[1:]
                if hasattr(self, fname):
                    apply(getattr(self, fname), (v,))

    def setBody(self, body):
        """ Set the body element """
        return self.setChildWithText('body', body)
        
    def getBody(self):
        """ Get the body text """
        return self.getChildText('body')

    def setSubject(self, subject):
        """ Set the subject element """
        return self.setChildWithText('subject', subject)
    
    def getSubject(self):
        """ Get the subject text """
        return self.getChildText('subject')
        
    def setThread(self, thread):
        """ Set the thread element  """
        return self.setChildWithText('thread', thread)
    
    def getThread(self):
        """ Get the thread text """
        return self.getChildText('thread')

class ChatInvite(Message):

    def __init__(self, to, chat, ttype='normal'):
        Message.__init__(self, chat, ttype, '', '', None)
        xtag = Element('x', None, { 'xmlns': 'http://jabber.org/protocol/muc#user'})
        self.appendChild(xtag)

        invite = Element('invite', None, { 'to': to })
        xtag.appendChild(invite)

        invite.appendChild(Element('reason', 'Come here'))

class ChatKick(Iq):
    
    def __init__(self, kicker, kicked, chat, reason=''):
        Iq.__init__(self, chat, "set", kicker, "kick", None)
        query = Element('query', None, { 'xmlns': 'http://jabber.org/protocol/muc#admin'})
        self.appendChild(query)

        item = Element('item', None, { 'nick': kicked, 'role': 'none' })
        query.appendChild(item)

        reason = Element('reason', "get out!", {} )
        item.appendChild(reason)

class Presence(JabberElementBase):    
    
    def __init__(self, to='', ttype=None, ffrom='', id='', element=None, show='', status='', priority=''):
        """ """
        if element:
            apply(JabberElementBase.__init__, (self, element, to, ttype, ffrom, id))
        else:
            apply(JabberElementBase.__init__, (self, 'presence', to, ttype, ffrom, id))
            
        if show: self.setShow(show)
        if status: self.setStatus(status)
        if priority: self.setPriority(priority)
        
    def setShow(self, show):
        """ Set the show element """
        return self.setChildWithText('show', show)
        
    def getShow(self):
        """ Get the show text """
        return self.getChildText('show')
    
    def setStatus(self, status):
        """ Set the status element """
        return self.setChildWithText('status', status)
        
    def getStatus(self):
        """ Get the status text """
        return self.getChildText('status')

    def setPriority(self, priority):
        """ Set the priority element """
        return self.setChildWithText('priority', priority)
        
    def getPriority(self):
        """ Get the priority text """
        return self.getChildText('priority')

    def setXNS(self,namespace):
        """Set the x namespace"""
        tag=get_child(self._el, 'x')
        if tag is None:
            tag=ezel('x').dom()
            self._el.appendChild(tag)
        tag.setAttribute('xmlns', namespace)
        return tag

    def getXNS(self):
        return self.getChild('x')

class ChatPresence(Presence):
    
    def __init__(self, to='', ttype=None, ffrom='', id='', element=None, show='', status='', priority=''):
        """ """
        Presence.__init__(self, to, ttype, ffrom, id, element, show, status, priority)
        self.setXNS('http://jabber.org/protocol/muc')

# Route elements are for the component protocol
# XXX I do not know where to get a copy of the XML Schema for http://jabberd.jabberstudio.org/ns/component/1.0 , therefore this class is based on examples
# XXX WORK in progess (not used at present )
class Route(JabberElementBase):
    """ """
    
    def __init__(self, to='', ffrom='', element=None, error='', **kw):
        """ """
        if element:
            apply(JabberElementBase.__init__, (self, element, to, ffrom))
        else:
            apply(JabberElementBase.__init__, (self, 'route', to, ffrom))

    def setError(self, error):
        """ Set the error element """
        return self.setChildWithText('error', error)
        
    def getError(self):
        """ Get the error text """
        return self.getChildText('error')
 
    def getElement(self, name=''):
        """ Get the transported element """
        names = (name and [name] ) or ['iq', 'message', 'presence']
        for c in self._element.childNodes:
            if c.namespaceURI == 'jabber:client':
                if c.localName in names:
                    pass

#Stolen from jabberpy http://jabberpy.sf.net
class JID:
    """A Simple class for managing jabber users id's """
    def __init__(self, jid='', node='', domain='', resource=''):
        if jid:
            if jid.find('@') == -1:
                self.node = ''
            else:
                bits = jid.split('@',1)
                self.node = bits[0]
                jid = bits[1]
                
            if jid.find('/') == -1:
                self.domain = jid
                self.resource = ''
            else:
                self.domain, self.resource = jid.split('/',1) 
        else:
            self.node = node
            self.domain = domain
            self.resource = resource


    def __str__(self):
        try:
            jid_str = ''
            if self.node: jid_str = jid_str + self.node + '@'
            if self.domain: jid_str = jid_str + self.domain
            if self.resource: jid_str = jid_str +'/'+ self.resource
            return jid_str
        except:
            return ''

    __repr__ = __str__


    def getNode(self):
        """Returns JID Node as string"""
        return self.node


    def getDomain(self):
        """Returns JID domain as string"""
        return self.domain


    def getResource(self):
        """Returns JID resource as string"""
        return self.resource


    def setNode(self,val):
        """Sets JID Node from string"""
        self.node = val


    def setDomain(self,val):
        """Sets JID domain from string"""
        self.domain = val


    def setResource(self,val):
        """Sets JID resource from string"""
        self.resource = val


    def getStripped(self):
        """Returns a jid string with no resource"""
        jid_str = ''
        if self.node: jid_str = jid_str + self.node + '@'
        if self.domain: jid_str = jid_str + self.domain
        return jid_str    

    def match(self, jid, weak=1):
        """ Match two jids 
        If weak is set to 1 the resource is not compared
        """
        return self.node==jid.node and \
            self.domain==jid.domain and \
            (weak or self.resource==jid.resource)
