"""
XMPP Client library
(c) 2004 Fabio Forno - email: fabio.forno_at_polito.it
Politecnico di Torino

Basic Usage:
 
 If you are familiar with twisted, the usage of the XMPPClient protocol will be
 trivial, otherwise see the numerous examples in the documentation of twisted.
 In order to use twibber, you must:

 - Build your own protocol deriving it from  XMPPClient (or XMPPBaseClient)
    
    class MyClient(XMPPClient):
        ...
   
   (note that the MyClient object will be created by a factory, which does not 
   pass any parameters to its constructor; thus there are only two ways for 
   customizing the object:
   - overriding the buildProtocol in the factory (more cumbersome), in order to     call the constructor directly;
   - stuffing the factory itself with the parameters, and access them through 
     the the factory element of the object,
     
   
 - Override the events you want to handle; the most significative are
   
    gotMessage
    gotIq
    gotPresence
    
    authSuccess (this may be handy when you want to make some action once the
    client has authenticated)
 
 - Use these methods in order to send XMPP stanzas
    sendMessage
    sendIq (this method returns a deferred to which callbacks for the answer 
            or the error iq may be associated)
    sendPresence
    
 - The elements and xmlutils modules contain many convenience methods suitable
    for more easily manipulating XML stanzas
    
 - Create a client factory for your protocol and connect to a TCP port with
   the reactor
   
    f = XMPPClientFactory(c.server, c.username, c.password)
    f.protcol = MyClient
    reactor.connectTCP(c.server, 5222, f)

 - Extensions (like FileTransfer) may be added using the the extensions
   parameter of the factory (see the example below)
 
"""

import xmppstream as xmpp
from xmppstream import XMPPStream, XMPPBaseFactory
from roster import RosterMixIn
from xmlutils import * 
from elements import Iq, JID
import sasl
import utils
from disco import DiscoMixIn
from filetransfer import FileTransfer, StreamInitiation, SOCKSv5ByteStream
from legacy_auth import LegacyLoginManager
from inband_reg import InBandRegistrationManager
import sys, binascii
from twisted.internet import reactor, protocol

#utility functions
encoding='ISO-8859-15'
 
#EVENTS notified by XMPPClient
JC_AUTH_SUCCESS="//event/jabber/client/auth/success"
JC_AUTH_ERROR="//event/jabber/client/auth/error"
JC_REGISTRATION_SUCCESS="//event/jabber/client/registration/success"
JC_REGISTRATION_ERROR="//event/jabber/client/registration/error"
JC_CONNECTION_LOST="//event/jabber/client/connection/lost"
JC_CONNECTION_FAILED="//event/jabber/client/connection/failed"

class LocalPasswordManager(sasl.PasswordManager):

    def __init__(self, stream):
        self._stream=stream
    
    def get_serv_type(self):
        return "xmpp"
        
    def get_serv_host(self):
	return self._stream.factory.server
                
    def get_serv_name(self):
        return ''#self._stream.factory.server
    
    def choose_realm(self, realms):
        realm=self._stream.factory.realm
        if realm in realms: return realm
        else: return realms[0]

# TODO
# make sure not to send whitespaces during SASL
class XMPPBaseClient(XMPPStream):
    """ XMPP Client without roster capabilities """
     
    def __init__(self):
        """ """
        XMPPStream.__init__(self)

        self._state=None
        self._authenticated=0
        self.stream_features={}
    
    def getJID(self):
        f=self.factory
        return JID(node=f.username, domain=f.server, resource=f.resource)
    
    def connectionMade(self):
        """ Send the stream opening """
        self.openStream()
        
    def openStream(self):
        """ Open the stream """
        f=self.factory
        ch=self.transport
        
        if float(f.version)>=1.0:
            version="version='%s'"%(f.version)
        else:
            version=''
            
        s="""<?xml version='1.0' encoding='UTF-8'?>
<stream:stream to='%s' xmlns='%s' xmlns:stream='%s' %s>"""%(ch.getPeer()[1], f.xmlns, xmpp.ns_xmlstream, version)
        
        if f.debug:
            utils.debug(s, xml=1)
            
        self.writeData(s)

    def streamOpened(self, tag):
        """ Received the first stream event """
        XMPPStream.streamOpened(self, tag)
         
        f=self.factory
        try: version=float(self.version)
        except: version=0.1
        
        # Hooks for inband registration and legacy login
        if f.auth=='inband registration':
            if hasattr(f, 'registration_fields'): fields=f.registration_fields
            else: fields={}
            rm=InBandRegistrationManager(self, username=f.username, password=f.password, **fields)
            rm.start()
            
        elif version<1.0 or f.auth=='legacy digest':        
            lm=f.LegacyLoginManager(self, username=f.username, secret=f.password, resource=f.resource)
            lm.start()

    #----------------------------------
    # Registration/Authorization events
    def authSuccess(self):
        """ Called when the client has authenticated """
        self.publishEvent(JC_AUTH_SUCCESS, self)
        
    def authFailed(self, error=None):
        """ Authentication failed """
        if self.factory.debug:
            utils.debug(str(error))
            
        self.publishEvent(JC_AUTH_ERROR, self)
        self.closeStream()

    def registrationSuccess(self):
        """ Registration success """
        self.publishEvent(JC_REGISTRATION_SUCCESS, self)
        
    def registrationError(self, error):
        """ Registration error """
        self.publishEvent(JC_REGISTRATION_ERROR, self, error)

    # ----------------------
    # SASL and TLS Related functions
    def gotFeatures(self, tag):
        """ Got a stream features tag """
        
        #XXX I really do not like this method as it is! (it is not clear
        # which features are mandatory)
        
        f=self.factory
        self.features_tag=tag
        self.stream_features['starttls']=get_childNS(tag, xmpp.ns_tls, 'starttls')
        
        if self.stream_features['starttls']:
            if f.use_tls=='never':
                if get_child(self.stream_features['starttls'], 'mandatory'):
                    self.authFailed('You must use TLS for this connection')
                    return
                #if we get here we can proceed with SASL
            if f.use_tls in ['mandatory', 'ifavail']:
                self._startTLS()
                return
        elif f.use_tls=='mandatory':
            self.authFailed('No TLS available')
            return
        
        #get the sasl mechanisms
        self.stream_features['sasl-mechanisms']=get_childNS(tag, xmpp.ns_sasl,
            'mechanisms')
        if f.auth=='sasl' and self.stream_features['sasl-mechanisms']:
            self.startSASL()
            return
        
        # look for bind and session features
        self.stream_features['bind']=get_childNS(tag, xmpp.ns_resource_binding,
            'bind')
        self.stream_features['session']=get_childNS(tag, xmpp.ns_session, 'session')
        
        if self.stream_features['bind']:
            self.startBinding()
        elif self.stream_features['session']:
            self.startSession()
        elif self._authenticated:
            self.authSuccess()
        
    # -------------------------------------------   
    # TLS 
    def _startTLS(self):
        """ Start TLS negotiation """
        starttls=ezel('startls', xmlns=xmpp.ns_tls)
        self.sendPacket(starttls.dom())
    
    def _handleTLSProceed(self, tag):
        """ Got the TLS proceed stanza """
        from OpenSSL import SSL
        from twisted.internet import ssl
        # start TLS and reopen the stream XXX not sure about this
        self.transport.startTLS(ssl.ClientContextFactory())
        self.newParser()
        self.openStream()
        
    def _handleTLSFailure(self, tag):
        """ Got a TLS error stanza """
        self.authFailed(tag)
        
    # -------------------------------------------
    # SASL
    def startSASL(self):
        """ Begin SASL authentication """
        f=self.factory
        self.pwd_manager=LocalPasswordManager(self)
        
        #select the best authetication mechanism
        mechs=[get_text(x) for x in
            get_children(self.stream_features['sasl-mechanisms'], 'mechanism')]
        
        if 'DIGEST-MD5' in mechs: mech='DIGEST-MD5'
        elif 'PLAIN' in mechs: mech='PLAIN'
        else:
            raise Exception('Unsupported mechansims')
            
        self.sasl_authenticator=sasl.ClientAuthenticator(mech, 
                self.pwd_manager)
        response=self.sasl_authenticator.start(f.username, f.authzid,
            f.password)
        el=ezel('auth', xmlns=xmpp.ns_sasl, mechanism=mech)
        if response.data:
            el.text(response.base64())
        self.sendPacket(el.dom())
                
    def _handleSASLFailure(self, tag):
        """ Received SASL failure """
        self.authFailed(tag.toxml())
        
    def _handleSASLChallenge(self, tag):
        """ handle the challenge of the sasl authentication process """    
        data=binascii.a2b_base64(get_text(tag))
        response=self.sasl_authenticator.challenge(data)
        if isinstance(response,sasl.Response):
            el=ezel('response', xmlns=xmpp.ns_sasl)
            if self.factory.debug:
                utils.debug(response.data)
	    el.text(response.base64())
            self.sendPacket(el.dom())
	else:
            el=ezel('abort', xmlns=xmpp.ns_sasl)
            self.sendPacket(el.dom())
            self.authFailed()
    
    def _handleSASLFailure(self, tag):
        """ SASL failed """
        self.authFailed(tag.toxml())
        
    def _handleSASLSuccess(self, tag):
        """ Finish of the SASL authetication """
        self._authenticated=1
        # reopen the stream
        self.newParser() #reset the XML parser
        self.openStream()
        
    # -------------------------------------------
    # Resource binding
    def startBinding(self):
        """ do the binding """
        f=self.factory
        iq=Iq(ttype='set')
        bind=ezel('bind', xmlns=xmpp.ns_resource_binding)
        bind.add('resource').text(f.resource)
        iq.appendChild(bind.dom())
        d=self.sendIq(iq)
        d.addCallback(self.gotBindingResult)  
        d.addErrback(self.gotBindingError)
        
    def gotBindingResult(self, iq):
        """ """
        if self.stream_features.get('session'):
            self.startSession()
        else:
            self.authSuccess()
        
    def gotBindingError(self, iq):
        """ """
        self.authFailed('Binding Error')
        
    # -------------------------------------------
    # Session
    # Nothing at present
    def startSession(self):
        """ """
        iq=Iq(ttype='set')
        session=ezel('session', xmlns=xmpp.ns_session_iq)
        iq.appendChild(session.dom())
        d=self.sendIq(iq)
        d.addCallback(self.gotSessionResult)  
        d.addErrback(self.gotSessionError)

    def gotSessionResult(self, iq):
        """ """
        self.authSuccess()

    def gotSessionError(self, iq):
        """ """
        self.authFailed('Session Error')

# Additional features besides the basic XMPP protocol are plugged in as MixIns
class XMPPClient(XMPPBaseClient, RosterMixIn, DiscoMixIn):
    """ XMPP Client with Roster and Disco capabilities """
    
    def __init__(self):
        XMPPBaseClient.__init__(self)
        RosterMixIn.__init__(self)
        DiscoMixIn.__init__(self)
        
        # init all the mixins added at runtime by the factory
        # (XXX temporary commented out)
        # for c in self.__bases__:
        #    c.__init__(self)
        
    def authSuccess(self):
        """ Called when the client has authenticated """
        XMPPBaseClient.authSuccess(self)
        self.getRoster()   

#_extensions=[('stream_initiation', StreamInitiation, (), {}),
#    ('s5_bytestream', SOCKSv5ByteStream, (8321,), {})]
#now file trasnfer is capable of registering automatically the needed extensions
#_extensions=[('file_transfer', FileTransfer, (), {'downloaddir': 'dowload', 'port':8321})]
_extensions=[]

class XMPPClientFactory(XMPPBaseFactory, protocol.ReconnectingClientFactory):
    
    protocol=XMPPClient
    LegacyLoginManager=LegacyLoginManager

    xmlns='jabber:client'
    version='1.0'
    debug=0
    
    def __init__(self, server, username, password, resource='twibber', realm=None, auth='sasl', tls='never', mixins=[], extensions=_extensions):
        """ auth types: 'sasl' (default)'
                        'legacy digest' - pre XMPP legacy authetication
                        'inband registration' - register with server
            tls: never, ifvail, mandatory
            mixins
            extensions: a list of tuples containing: (name, factory, args, kw)
        """
        XMPPBaseFactory.__init__(self)
        
        self.username=username
        self.password=password
        self.resource=resource
        self.server=server
        self.realm=realm or server
        self.use_tls=tls
        self.auth=auth
        self.authzid="%s@%s"%(self.username, self.server)
        self.extensions=extensions

        # tricky part: add further mixins
        #self.protocol.__bases__+=tuple(mixins)
            
    def buildProtocol(self, addr):
        p=protocol.ReconnectingClientFactory.buildProtocol(self, addr)
        self._addExtensions(p)
        self._bindListeners(p)
        return p
        
    def _addExtensions(self, p):
        for name, e, args, kw in self.extensions:
            ext=e(p, *args, **kw)
            setattr(p, name, ext)
            
def _test():
    """ try the connection with a server """
    from utils import Config
    c=Config(['client2.cfg'])
    c.username='test'
    c.password='test'
    c.server='jabber.example.org'
    if len(sys.argv)>1: c.load(sys.argv[1])
    else: c.load(sys.argv[1])
    
    f=XMPPClientFactory(c.server, c.username, c.password)
    f.debug=1
    
    reactor.connectTCP(c.server, 5222, f)
    reactor.run()
    
if __name__=='__main__':
    _test()
