#include "pokerStdAfx.h"

#ifndef WIN32
#	include "perlin_noise.h"
#	include <sys/time.h>
#endif // WIN32

#include <ctime>

#ifdef WIN32
#	define random rand
#endif

/*
 *
 * videoSpace/image/processing/noise/PerlinNoise.cxx --
 *
 * Copyright (C) Nicolas Roussel
 *
 * See the file LICENSE for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

/* coherent noise function over 1, 2 or 3 dimensions */
/* (copyright Ken Perlin) */

#include <math.h>
#include <cstdlib>

#define B 0x100
#define BM 0xff

#define N 0x1000
#define NP 12   /* 2^N */
#define NM 0xfff

#define s_curve(t) ( t * t * (3.0f - 2.0f * t) )

#define lerp(t, a, b) ( a + t * (b - a) )

#define setup(coord,b0,b1,r0,r1)\
	t = coord + N;\
	b0 = ((int)t) & BM;\
	b1 = (b0+1) & BM;\
	r0 = t - (int)t;\
	r1 = r0 - 1.0f;

// -------------------------------------------------------------------

static void
init_random(void) {
#ifdef WIN32
	time_t myTime;
	time(&myTime);
	srand((unsigned int)myTime);
#else
  struct timeval tv ;
  gettimeofday(&tv, 0) ;
  unsigned int i = (tv.tv_sec+tv.tv_usec) % 737 ;
  srandom(i) ;
#endif 
}

// -------------------------------------------------------------------

PerlinNoise1D::PerlinNoise1D(void) {
  init_random() ;

  p = new int [B+B+2] ;
  g1 = new float [(B+B+2)] ;

  int i, j, k;
  
  for (i = 0 ; i < B ; i++) {
    p[i] = i; 
    g1[i] = (float)((random() % (B + B)) - B) / B;
  }

  while (--i) {
    k = p[i];
    p[i] = p[j = random() % B];
    p[j] = k;
  }

  for (i = 0 ; i < B + 2 ; i++) {
    p[B + i] = p[i];
    g1[B + i] = g1[i];
  }
}

PerlinNoise1D::~PerlinNoise1D() {
  delete [] p ;
  delete [] g1 ;
}

float
PerlinNoise1D::noise(float x) {
  int bx0, bx1;
  float rx0, rx1, sx, t, u, v ;

  setup(x, bx0,bx1, rx0,rx1);
  
  sx = s_curve(rx0);
  
  u = rx0 * g1[ p[ bx0 ] ];
  v = rx1 * g1[ p[ bx1 ] ];
  
  return lerp(sx, u, v);
}

// -------------------------------------------------------------------

static inline void
normalize2(float v[2]) {
  float s;

  s = sqrt(v[0] * v[0] + v[1] * v[1]);
  v[0] = v[0] / s;
  v[1] = v[1] / s;
}

PerlinNoise2D::PerlinNoise2D(void) {
  init_random() ;

  p = new int [B+B+2] ;
  g2 = new pt2D [B+B+2] ;

  int i, j, k;

  for (i = 0 ; i < B ; i++) {
    p[i] = i;

    for (j = 0 ; j < 2 ; j++)
	 g2[i][j] = (float)((random() % (B + B)) - B) / B;
    normalize2(g2[i]);
  }

  while (--i) {
    k = p[i];
    p[i] = p[j = random() % B];
    p[j] = k;
  }

  for (i = 0 ; i < B + 2 ; i++) {
    p[B + i] = p[i];
    for (j = 0 ; j < 2 ; j++)
	 g2[B + i][j] = g2[i][j];
  }
}

PerlinNoise2D::~PerlinNoise2D() {
  delete [] p ;
  delete [] g2 ;
}

float
PerlinNoise2D::noise(float x, float y) {
  int bx0, bx1, by0, by1, b00, b10, b01, b11;
  float rx0, rx1, ry0, ry1, *q, sx, sy, a, b, t, u, v;
  register int i, j;

  setup(x, bx0,bx1, rx0,rx1);
  setup(y, by0,by1, ry0,ry1);

  i = p[ bx0 ];
  j = p[ bx1 ];

  b00 = p[ i + by0 ];
  b10 = p[ j + by0 ];
  b01 = p[ i + by1 ];
  b11 = p[ j + by1 ];

  sx = s_curve(rx0);
  sy = s_curve(ry0);

#define at2(rx,ry) ( rx * q[0] + ry * q[1] )

  q = g2[ b00 ] ; u = at2(rx0,ry0);
  q = g2[ b10 ] ; v = at2(rx1,ry0);
  a = lerp(sx, u, v);

  q = g2[ b01 ] ; u = at2(rx0,ry1);
  q = g2[ b11 ] ; v = at2(rx1,ry1);
  b = lerp(sx, u, v);

  return lerp(sy, a, b);
}

// -------------------------------------------------------------------

static inline void
normalize3(float v[3]) {
  float s;

  s = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
  v[0] = v[0] / s;
  v[1] = v[1] / s;
  v[2] = v[2] / s;
}

PerlinNoise3D::PerlinNoise3D(void) {
  init_random() ;

  p = new int [B+B+2] ;
  g3 = new pt3D [B+B+2] ;

  int i, j, k;

  for (i = 0 ; i < B ; i++) {
    p[i] = i;

    for (j = 0 ; j < 3 ; j++)
	 g3[i][j] = (float)((random() % (B + B)) - B) / B;
    normalize3(g3[i]);
  }

  while (--i) {
    k = p[i];
    p[i] = p[j = random() % B];
    p[j] = k;
  }

  for (i = 0 ; i < B + 2 ; i++) {
    p[B + i] = p[i];
    for (j = 0 ; j < 3 ; j++)
	 g3[B + i][j] = g3[i][j];
  }
}

PerlinNoise3D::~PerlinNoise3D() {
  delete [] p ;
  delete [] g3 ;
}

float
PerlinNoise3D::noise(float x, float y, float z) {
  int bx0, bx1, by0, by1, bz0, bz1, b00, b10, b01, b11;
  float rx0, rx1, ry0, ry1, rz0, rz1, *q, sy, sz, a, b, c, d, t, u, v;
  register int i, j;

  setup(x, bx0,bx1, rx0,rx1);
  setup(y, by0,by1, ry0,ry1);
  setup(z, bz0,bz1, rz0,rz1);

  i = p[ bx0 ];
  j = p[ bx1 ];

  b00 = p[ i + by0 ];
  b10 = p[ j + by0 ];
  b01 = p[ i + by1 ];
  b11 = p[ j + by1 ];

  t  = s_curve(rx0);
  sy = s_curve(ry0);
  sz = s_curve(rz0);

#define at3(rx,ry,rz) ( rx * q[0] + ry * q[1] + rz * q[2] )

  q = g3[ b00 + bz0 ] ; u = at3(rx0,ry0,rz0);
  q = g3[ b10 + bz0 ] ; v = at3(rx1,ry0,rz0);
  a = lerp(t, u, v);

  q = g3[ b01 + bz0 ] ; u = at3(rx0,ry1,rz0);
  q = g3[ b11 + bz0 ] ; v = at3(rx1,ry1,rz0);
  b = lerp(t, u, v);

  c = lerp(sy, a, b);

  q = g3[ b00 + bz1 ] ; u = at3(rx0,ry0,rz1);
  q = g3[ b10 + bz1 ] ; v = at3(rx1,ry0,rz1);
  a = lerp(t, u, v);

  q = g3[ b01 + bz1 ] ; u = at3(rx0,ry1,rz1);
  q = g3[ b11 + bz1 ] ; v = at3(rx1,ry1,rz1);
  b = lerp(t, u, v);

  d = lerp(sy, a, b);

  return lerp(sz, c, d);
}
