/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "pokerStdAfx.h"
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
# define snprintf _snprintf
#endif

#include <osg/Geode>
#include <osg/AutoTransform>
#include <osg/Material>
#include <osg/Group>
#include <osg/MatrixTransform>
#include <osgDB/WriteFile>

#ifndef WIN32
#	include "PokerError.h"
#	include "PokerApplication.h"
#	include "PokerShowdown.h"
#endif // WIN32

PokerShowdownModel::PokerShowdownModel(PokerApplication* game, MAFOSGData* data) : mGame(game), mSetCount(0) {
  std::string sideNames[CARDS_SIZE];
  sideNames[HIGH] = "high";
  sideNames[LOW] = "low";

  char tmp[128];

  std::string glow_template = game->HeaderGet("sequence", "/sequence/bestHand/@glow");

  std::string anchor_template = game->HeaderGet("sequence", "/sequence/bestHand/@anchor");
  std::string url = game->HeaderGet("sequence", "/sequence/bestHand/@url");

  std::vector<std::string> anchors;
  std::string count_string = game->HeaderGet("sequence", "/sequence/bestHand/@count");
  int count = atoi(count_string.c_str());
  for(int i = 0; i < CARDS_SIZE; i++) {
    CardsGroup& cardsGroup = mCardsGroup[i];

    char anchorSide = i == HIGH ? 'p' : 'q';
    for(int j = 0; j < count; j++) {
      PokerCardController* card = new PokerCardController(game, url);
      card->Visible(true);
      card->Fold();
      snprintf(tmp, sizeof(tmp), anchor_template.c_str(), anchorSide, j + 1);
      anchors.push_back(tmp);
      cardsGroup.mCards.push_back(card);

      MAFAnchor* anchor = data->GetAnchor(tmp);
      card->Anchor(anchor);
    }

    std::map<std::string,std::string> color_string = game->HeaderGetProperties("sequence", "/sequence/bestHand/" + sideNames[i] + "Color");
    mColors[i].x() = atof(color_string["red"].c_str()) / 255;
    mColors[i].y() = atof(color_string["green"].c_str()) / 255;
    mColors[i].z() = atof(color_string["blue"].c_str()) / 255;
    mColors[i].w() = 1.f;
    g_debug("PokerShowdownModel::PokerShowdownModel: red %s, green %s, blue %s", color_string["red"].c_str(), color_string["green"].c_str(), color_string["blue"].c_str());
    cardsGroup.mColor = mColors[i];
    snprintf(tmp, sizeof(tmp), glow_template.c_str(), anchorSide);
    osg::Node* glow = data->GetNode(tmp);
    g_assert(glow != 0);
    glow->setNodeMask(0);
    g_assert(glow != 0);
    cardsGroup.mGlow = glow;
  }

  osg::Node* group = data->GetNode("autotransform_showdown");
  group->setName("PokerShowdown");
  g_assert(group != 0);
  g_assert(group->asGroup() != 0);
  g_assert(group->asGroup()->asTransform() != 0);

  SetNode(group);

  {
    std::map<std::string,std::string> colors = game->HeaderGetProperties("sequence", "/sequence/bestHand/boardColor");
    mShowdownCardColor.x() = atof(colors["red"].c_str()) / 255;
    mShowdownCardColor.y() = atof(colors["green"].c_str()) / 255;
    mShowdownCardColor.z() = atof(colors["blue"].c_str()) / 255;
    mShowdownCardColor.w() = 1.f;
    g_debug("PokerShowdownModel::PokerShowdownModel: boardColor red %s, green %s, blue %s", colors["red"].c_str(), colors["green"].c_str(), colors["blue"].c_str());
  }

  const std::string& scale_showdown = game->HeaderGet("sequence", "/sequence/bestHand/@scale_showdown");
  if (scale_showdown.empty())
    g_error("PokerShowdownModel::PokerShowdownModel /sequence/bestHand/@scale_showdown not found");
  float scale=atof(scale_showdown.c_str());

  // rescale card if needed
  osg::Group* myGrp=group->asGroup();
  g_assert(myGrp);
  osg::MatrixTransform* tr=new osg::MatrixTransform;
  tr->setMatrix(osg::Matrix::scale(scale,scale,scale));
  while (myGrp->getNumChildren()) {
    tr->addChild(myGrp->getChild(0));
    myGrp->removeChild(0,1);
  }
  myGrp->addChild(tr);

}

PokerShowdownModel::~PokerShowdownModel() {
}

void PokerShowdownModel::Init(void) { }

PokerShowdownController::PokerShowdownController(PokerApplication* game, MAFOSGData* data) {
  SetModel(new PokerShowdownModel(game, data));
  Init();
  SetSelectable(true);
}

PokerShowdownController::~PokerShowdownController() {
}

bool PokerShowdownController::Update(MAFApplication* application) {
  PokerShowdownModel* model = GetModel();
  if(model->mSetCount > 1) {
    UGAMEArtefactController::Update(application);
    if(model->GetSelected()) {
      CardsGroup& first = model->mCardsGroup[0];
      CardsGroup& second = model->mCardsGroup[1];
      for(unsigned int i = 0; i < first.mCards.size(); i++) {
	int value = first.mCards[i]->GetValue();
	first.mCards[i]->SetValue(second.mCards[i]->GetValue());
	second.mCards[i]->SetValue(value);
      }

      osg::Vec4 color = first.mColor;
      first.mColor = second.mColor;
      SetColor(first.mGlow.get(), first.mColor);
      
      second.mColor = color;
      SetColor(second.mGlow.get(), second.mColor);
      
      model->SetSelected(false);
    }
  }
  return true;
}

void PokerShowdownController::Reset(void) {
  PokerShowdownModel* model = GetModel();

  for(int i = 0; i < PokerShowdownModel::CARDS_SIZE; i++) {
    CardsGroup& cardsGroup = model->mCardsGroup[i];
    for(unsigned int j = 0; j < cardsGroup.mCards.size(); j++) {
      cardsGroup.mCards[j]->Fold();
      if (cardsGroup.mCards[j]->GetModel()->GetArtefact())
	SetColor(cardsGroup.mCards[j]->GetModel()->GetArtefact(),osg::Vec4(1.0,1.0,1.0,1.0));
    }
    UnsetColor(cardsGroup.mGlow.get());
    cardsGroup.mColor = model->mColors[i];
  }
  if (GetModel()->mHandValue.get())
    GetModel()->mHandValue->setText("");
  model->mSetCount = 0;
}

void PokerShowdownController::Set(const std::string& side, std::vector<int> values, std::vector<int> board, const std::string& handval) {
  PokerShowdownModel* model = GetModel();
  PokerShowdownModel::Side side_int;

  if (model->mHandValue.get())
    model->mHandValue->setText(handval);

  if(side == "low")
    side_int = PokerShowdownModel::LOW;
  else 
    side_int = PokerShowdownModel::HIGH;

  CardsGroup& cardsGroup = model->mCardsGroup[side_int];
  std::vector<osg::ref_ptr<PokerCardController> >& cards = cardsGroup.mCards;
  for(unsigned int i = 0; i < cards.size(); i++) {
    if(i < values.size()) {
      cards[i]->Receive();
      cards[i]->SetValue(values[i]);
      for(unsigned int j = 0; j < board.size(); j++) {
	if(board[j] == values[i]) {
	  SetColor(cards[i]->GetModel()->GetArtefact(), GetModel()->mShowdownCardColor);
	  break;
	}
      }

    } else {
      cards[i]->Fold();
    }
  }

  if(side != "") {
    SetColor(cardsGroup.mGlow.get(), cardsGroup.mColor);
    model->mSetCount++;
  }

}

class AlterMaterialColor : public osg::NodeVisitor {
public:
  AlterMaterialColor(const osg::Vec4& color) : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN), mColor(color) {
    setNodeMaskOverride(0xffffffff);
  }

  virtual void apply(osg::Geode& geode) {
    unsigned int num_drawables = geode.getNumDrawables();
    g_assert(num_drawables == 1);
    osg::Drawable* drawable = geode.getDrawable(0);
    osg::StateSet* state = drawable->getStateSet();
    g_assert(state != 0);
    if(state) {
      osg::Material* material = dynamic_cast<osg::Material*>(state->getAttribute(osg::StateAttribute::MATERIAL));
      if(!material) material = new osg::Material;
      material->setColorMode(osg::Material::DIFFUSE);
      material->setDiffuse(osg::Material::FRONT_AND_BACK, mColor);
      state->setAttributeAndModes(material, osg::StateAttribute::ON);
    }
  }

private:
  osg::Vec4 mColor;
};

void PokerShowdownController::SetColor(osg::Node* node, const osg::Vec4& color) {
  AlterMaterialColor alter(color);
  node->accept(alter);
  node->setNodeMask(0xffffffff);
}

void PokerShowdownController::UnsetColor(osg::Node* node) {
  node->setNodeMask(0x0);
}
