/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifndef _poker_camera_h
#define _poker_camera_h

#include <osg/Vec2>

#include <maf/camera.h>
#include <maf/data.h>

#include <mth_hermite.h>
#include <maf/interpolator.h>

struct DCJCurve
{
  struct lerp
  {
    template<typename T>
    static inline void doit(T &out, const T& from, const T &to, float t)
    {
      out = from + ((to - from)*t);
    }
  };
  std::vector<osg::Vec3> ps;
  DCJCurve()
  { 
  }
  void add(const osg::Vec3 p)
  {
    ps.push_back(p);
  }
  void clear()
  {
    ps.clear();
  }
  void get(osg::Vec3 &out, float t)
  {
    std::vector<osg::Vec3> result;
    result.resize(ps.size() - 1);
    process(result, ps, t);
    while(result.size() != 1)
      {
	process(result, result, t);
	result.resize(result.size() - 1);
      }
    out = result[0];
  }
  void process(std::vector<osg::Vec3> &out, const std::vector<osg::Vec3> &in, float t)
  {
    int count = (in.size() - 1);
    for (int i = 0; i < count; i++)
      {
	lerp::doit(out[i], in[i], in[i+1], t);
      }
  }
};

class PokerApplication;

class PokerCameraModel : public MAFCameraModel
{
public:
  class PointFunctor {
  public:
    PointFunctor(float duration) : mDuration(duration) {}

    virtual void init(const osg::Vec3& from, const osg::Vec3 &to, int pointCount) const {
      mSegment = to - from;
      mFrom = from;
      mPointCount = pointCount;
    }
    
    virtual void process(int index, osg::Vec3& position, float& when) const = 0;
    mutable osg::Vec3 mSegment;
    mutable osg::Vec3 mFrom;
    mutable int mPointCount;
    mutable float mDuration;
  };

  class PointFunctorPow : public PointFunctor {
  public:
    PointFunctorPow(float duration, float factor = 2.f, bool inverse = false) : PointFunctor(duration), mFactor(factor), mInverse(inverse) {}

    virtual void init(const osg::Vec3& from, const osg::Vec3 &to, int pointCount) const {
      PointFunctor::init(from, to, pointCount);
      mTimeStep = mDuration / pointCount;
    }

    virtual void process(int index, osg::Vec3& where, float& when) const;

    mutable float mTimeStep;
    mutable float mFactor;
    mutable bool mInverse;
  };

  class PointFunctorSin : public PointFunctor {
  public:
    PointFunctorSin(float duration, float factor = 1.f, bool inverse = false) : PointFunctor(duration), mFactor(factor), mInverse(inverse) {}

    virtual void init(const osg::Vec3& from, const osg::Vec3 &to, int pointCount) const {
      PointFunctor::init(from, to, pointCount);
      mTimeStep = mDuration / pointCount;
    }

    virtual void process(int index, osg::Vec3& where, float& when) const;

    mutable float mTimeStep;
    mutable float mFactor;
    mutable bool mInverse;
  };

  class PointFunctorHermite : public PointFunctor {
  public:
    PointFunctorHermite(float duration, std::vector<osg::Vec2> &keys) : PointFunctor(duration)
      {
	int count = keys.size();
	mKeys.resize(count);
	for (int i = 0; i < count; i++)
	  {
	    //std::cout << keys[i].x() << "," << keys[i].y() << std::endl;
	    mKeys[i].Init(keys[i].x(), keys[i].y());
	    mCurve.InsertKey(&mKeys[i]);
	  }
      }
    virtual ~PointFunctorHermite()
      {
      }
    virtual void init(const osg::Vec3 &from, const osg::Vec3 &to, int pointCount) const {
      PointFunctor::init(from, to, pointCount);
      mTimeStep = mDuration / pointCount;
    }
    virtual void process(int index, osg::Vec3 &where, float &when) const;
  private:
    typedef mth::Hermite_T<float, float> Curve;
    typedef Curve::Key_t KeyFrame;
    mutable float mTimeStep;
    Curve	mCurve;
    std::vector<KeyFrame> mKeys;
  };
  
  PokerCameraModel();
  ~PokerCameraModel();
  
  void StartInterpolation(float timeout);
  void SetupTargetInterpolator(const osg::Vec3 &target);
  void SetupLengthInterpolator(const osg::Vec3 &position, const osg::Vec3 &target);
  void SetupFovInterpolator(float fov);
  void SetupRotationInterpolator(const osg::Quat &attitude);
  void SetupReadjustInterpolator(const osg::Quat &attitude);

  void SetupInterpolators(const osg::Vec3 &position, const osg::Vec3 &target, const osg::Vec3 &up, float fovTo, float timeout);
  void SetupInterpolators(const osg::Vec3 &position, const osg::Vec3 &target, const osg::Quat& attitude, float fovTo, float timeout);
  void Readjust();

  bool GetIsMoving(void);
  void SetIsMoving(bool isMoving);

  const osg::Vec3& GetFrom(void) { return mFrom; }
  const osg::Vec3& GetTo(void) { return mTo; }

  void Update(float delta);
  void LoadKeys(std::vector<osg::Vec2> &keys, MAFXmlData *data, const std::string &name);
  template<typename T>
    void LoadSpline(T &interpolator, MAFXmlData *data, const std::string &name)
    {
      std::vector<osg::Vec2> keys;
      LoadKeys(keys, data, name);
      interpolator.InitKeys(keys);
    }
private:  
  typedef mth::Hermite_T<osg::Vec3> Spline;
  typedef Spline::Key_t KeyFrame;

  std::vector<KeyFrame*> mKeys;
  std::vector<Spline> mSpline;
  std::vector<bool> mIsMoving;
  osg::Vec3 mFrom;
  osg::Vec3 mTo;
 public:
  osg::Quat mCamAttitude;
  osg::Quat mCamPrevAttitude;
  osg::Quat mCamNextAttitude;
  osg::Vec3 mCamPosition;
  osg::Vec3 mCamPrevPosition;
  osg::Vec3 mCamNextPosition;
  osg::Vec3 mCamTarget;
  osg::Vec3 mCamNextTarget;
  osg::Vec3 mCamPrevTarget;
  float mCamPrevFov;
  float mCamNextFov;

  bool mGameMode;
  float mRotateX;
  float mRotateY;
  float mDistanceFromTarget;
  bool  mFirstTime;
  MAFBezierInterpolator<osg::Quat, util::slerp> mRotationBInterpolator;
  MAFBezierInterpolator<osg::Quat, util::multislerp> mReadjustBInterpolator;
  MAFBezierInterpolator<osg::Vec3> mTargetBInterpolator;
  MAFBezierInterpolator<float> mLengthBInterpolator;
  MAFBezierInterpolator<float> mFovBInterpolator;  
  MAFInterpolatorTimer<> mTimer;
  // state ?
  bool mNeedReadjust;
  bool mReadjusting;
  float mReadjustTimeout;
  float mReadjustScale;

 public:
  enum Mode {
    CAMERA_FREE_MODE,
    CAMERA_ENTER_MODE,
    CAMERA_DIRECT_MODE,
    CAMERA_GAME_MODE,
    CAMERA_LEAVE_MODE,
  };

  Mode GetMode() const {return mMode;}
  void SetMode(Mode mode) {mMode = mode; mModeChanged = true;}

  bool GetModeChanged() const {return mModeChanged;}
  void SetModeChanged(bool modeChanged) {mModeChanged = modeChanged;}

  float GetAnglePixelRatio(void) { return mAnglePixelRatio; }
  void SetAnglePixelRatio(float anglePixelRatio) { mAnglePixelRatio = anglePixelRatio; }

 private:
  Mode mMode;
  bool mModeChanged;
  float mAnglePixelRatio;
};

class PokerCameraController : public MAFCameraController {
public:
  PokerCameraController(PokerApplication* game);
  virtual ~PokerCameraController() {}

  virtual void Init(void);

  PokerCameraModel* GetModel() { return dynamic_cast<PokerCameraModel*>(MAFController::GetModel()); }
  const PokerCameraModel* GetModel() const { return dynamic_cast<const PokerCameraModel*>(MAFController::GetModel()); }

  bool Update(MAFApplication* application);

  void MoveTo(const osg::Vec3& destination, const PokerCameraModel::PointFunctor& pointGenerator);
  void MoveTo(const MAFCameraModel& target, 
	      const PokerCameraModel::PointFunctor& positionPointGenerator);
  void MoveTo(const MAFCameraModel& target, 
	      const PokerCameraModel::PointFunctor& positionPointGenerator,
	      const PokerCameraModel::PointFunctor& targetPointGenerator);

  void MoveTo(const osg::Vec3 &position, const osg::Vec3 &target, float fov, float timeout);
  void MoveToPrevious(float timeout);

  bool GetRespondToEvents(void) { return mRespondToEvents; }
  void SetRespondToEvents(bool respondToEvents) { mRespondToEvents = respondToEvents; }
  void SetConstraint(float (&constraint)[4]);
  void UnsetConstraint();

  void SetMode(PokerCameraModel::Mode mode);
  PokerCameraModel::Mode GetMode() const;
  bool ModeChanged() const;
  void ConsumeMode();

 private:
  void ApplyConstraint(float &dx, float &dy);
  void Rotate(float dx, float dy);
  void RotateFreeMode(float dx, float dy);
  void RotateGameMode(float dx, float dy);

public:
	PokerApplication* mGame;

 private:
  bool mConstraintSet;
  float mDeltaConstraint[4];
  float mDeltaCurrent[2];
  bool mRespondToEvents;
  int mCurrent;
  bool mApplicationStealFocus;
};

#endif 
