/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *  Igor Kravtchenko <igor@obraz.net>
 *
 */

#ifndef CAL_SCHEDULER_H
#define CAL_SCHEDULER_H

#include "ucal3d_export.h"

#define HACK_SOUND_IN_CAL3D

#ifdef HACK_SOUND_IN_CAL3D
#include <map>
#include <osg/ref_ptr>
class MAFAudioController;
#endif

#include <list>
#include <iostream>
//#include "cal3d/cal3d_export.h"

#include "cal3d/global.h"
#include "cal3d/animationalt.h"
#include "cal3d/blender.h"
#include "cal3d/mixer.h"

/**
 * CalScheduler runs and stops animations according to the time
 * specifications of the caller. In the most simple case, it starts
 * running an animation immediately and forgets about it when it
 * finishes. It may also remember to run (or stop) an animation at
 * a given point in time. 
 *
 * When asked to run animation, CalScheduler is provided a
 * CalCoreAnimation id and it will create a CalAnimationAlt instance
 * pointing to it. This instance holds all the informations necessary
 * to manage the state of the running animation. The caller can tune
 * the CalAnimationAlt in various ways:
 *
 * - specify the weight of the animation
 *
 * - specify the total length of the animation in seconds (may be forever)
 * 
 * - ask for a callback to be called when the animation finishes
 *
 * - register a function that modifies the weight of the animation as
 *   it runs (default functions are provided for fade in and fade out)
 *
 * - register a function that modifies the time perception of the 
 *   animation (such as slowing it down or accelerating it)
 *
 * At a given moment, the CalScheduler relies on a CalBlender instance
 * to blend all active animations together. The blending policy of
 * CalBlender is to provide two channels, foreground and background,
 * where the foreground channel has precedence over the background
 * channel. When N animations are running simultaneously in a channel,
 * they are granted a weight of 1/N, which may be further altered by
 * the CalAnimationAlt weight (1/N * weight).
 * 
 */
class CalScheduler : public CalAbstractMixer
{
private:
  friend class Entry;
  friend class EntryStart;
  friend class EntryStop;

  typedef std::list<class Entry*> Entries;
  Entries m_entries;


  typedef std::list<CalAnimationAlt*> ActiveAnimations;
  ActiveAnimations m_activeAnimations;

  typedef std::map<int,CalAnimationAlt*> Id2Animation;
  Id2Animation m_id2animation;

  double m_time;
  CalModel* m_pModel;
  CalBlender* m_pBlender;

protected:
  int createAnimation(int coreAnimationId);


public:
  typedef CalAnimationAlt::TimeFunction TimeFunction;
  typedef CalAnimationAlt::WeightFunction WeightFunction;
  typedef CalAnimationAlt::FadeInOut FadeInOut;
  typedef CalAnimationAlt::FadeIn FadeIn;
  typedef CalAnimationAlt::FadeOut FadeOut;
  typedef CalAnimationAlt::StopCallback StopCallback;
  // Convenience alias. MUST mimic the corresponding enum in CalBlender
  enum Channel {
    FOREGROUND=0,
    BACKGROUND=1,
  };


  UCAL3D_API static const int ALL;
  UCAL3D_API static const float ONCE;
  UCAL3D_API static const float FOREVER;
  UCAL3D_API static const float NOW;

  UCAL3D_API CalScheduler();
  virtual ~CalScheduler();

  UCAL3D_API CalAnimationAlt* run(Channel channel,
		       int coreAnimationId,
		       float length,
		       float weight = 1.f,
		       WeightFunction *pWeightFunction = 0,
		       float delay = 0.f
		       );

  UCAL3D_API CalAnimationAlt* stop(int anyId,
			WeightFunction *pWeightFunction = 0,
			float delay = 0.f
			);

  virtual void updateAnimation(float deltaInSeconds);
  virtual void updateSkeleton();
  virtual bool create(CalModel *pModel);
  virtual void destroy();

  UCAL3D_API CalModel* getModel() { return m_pModel; }
  UCAL3D_API CalBlender* getBlender() { return m_pBlender; }
  UCAL3D_API CalAnimationAlt* getAnimation(int anyId);
  UCAL3D_API void getAnimationIdsFromAnyId(int anyId, std::vector<int>& animationIds);
  UCAL3D_API bool isAnimationId(int anyId);
  UCAL3D_API bool isAnimationActive(int anyId);

#ifdef HACK_SOUND_IN_CAL3D
  std::map<int,osg::ref_ptr<MAFAudioController> >* mAnimation2Sounds;
#endif

};


#endif 
