/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *  Igor Kravtchenko <igor@obraz.net>
 *
 */

#ifndef CAL_ANIMATIONALT_H
#define CAL_ANIMATIONALT_H

//****************************************************************************//
// Includes                                                                   //
//****************************************************************************//

#include "cal3d/coreanimation.h"
#include "cal3d/coretrack.h"
#include "cal3d/blender.h"

//****************************************************************************//
// Forward declarations                                                       //
//****************************************************************************//

//****************************************************************************//
// Class declaration                                                          //
//****************************************************************************//

 /*****************************************************************************/
/** CalAnimationAlt holds the information necessary for a
 * CalCoreAnimation to run (fade in/fade out, weight, etc). It is
 * created on the basis of a CalCoreAnimation that provides the
 * animation tracks with a list of bones and the rotation/translation
 * that must be applied to them when the animation runs.
 *
 * CalAnimationAlt instances should be created as a side effect of
 * calling CalScheduler::run.
 *
 *****************************************************************************/

#include "ucal3d_export.h"

class CalAnimationAlt
{
public:
  class Function
  {
  public:
    UCAL3D_API Function() { }
    virtual ~Function() {}

    virtual Function* clone() = 0;

  };

  /**
   *  Can be subclassed to control the time perception for an
   *  animation.
   */
  class TimeFunction : public Function
  {
    float m_time;

  public:
    UCAL3D_API TimeFunction() { }
    virtual ~TimeFunction() {}

    virtual Function* clone() { return new TimeFunction(*this); }

    virtual float process(CalAnimationAlt* pAnimation)
    {
      return function(pAnimation->getCumulatedTime());
    }

    virtual float function(float _time) { return _time; }
  };

  /**
   *  Base classe for weight functions. To modify weight over time.
   *  it permits the user to modify the weight with a function over time
   *  @see FadeInOut
   */
  class WeightFunction : public Function
  {
  public:
    UCAL3D_API WeightFunction() {}
    virtual ~WeightFunction() {}

    virtual Function* clone() { return new WeightFunction(*this); }

    virtual float process(CalAnimationAlt* pAnimation)
    {
      return function(pAnimation->m_info.m_weight);
    }
  
    virtual float function(float weight) { return weight; }
  };

  /**
   *  Fade in fade out class of weight
   */
  class FadeInOut : public WeightFunction
  {
  protected:

    float m_fadeIn; // in second
    float m_fadeOut; // in second

  public:
    UCAL3D_API FadeInOut(float fadeIn, float fadeOut) : m_fadeIn(fadeIn), m_fadeOut(fadeOut) {}
    virtual ~FadeInOut() {}

    virtual Function* clone() { return new FadeInOut(*this); }

    virtual bool checkAndRepair(CalAnimationAlt* pAnimation)
    {
      return true;
    }

    virtual float process(CalAnimationAlt* pAnimation);
  };

  class FadeIn : public FadeInOut
  {
  public:
    UCAL3D_API FadeIn(float fadeIn) : FadeInOut(fadeIn, 0.f) {}
    virtual ~FadeIn() {}
  };

  class FadeOut : public FadeInOut
  {
  public:
    UCAL3D_API FadeOut(float fadeOut) : FadeInOut(0.f, fadeOut) {}
    virtual ~FadeOut() {}
  };

  /**
   *  Base class for callback at the end of animation.
   *  The callback will be called before destructing animation
   */

  class StopCallback
    {
    public:
      UCAL3D_API StopCallback() {}
      virtual ~StopCallback() {}

      virtual void process(CalModel* model, CalAnimationAlt* animation) = 0;
    };

  enum State { RUNNING, WANT_TO_STOP, STOPPING, STOPPED };

  struct Info {
    UCAL3D_API Info();

    State m_state;

    // channel type to apply instance in mixer(FOREGROUND or BACKGROUND)
    CalBlender::Channel m_channel;

    // call back called at the end of animation
    StopCallback* m_pStopCallback;

    // weight function
    WeightFunction* m_pWeightFunction;  
  
    // time function
    TimeFunction* m_pTimeFunction;  

    // id of animation to use (CalCoreAnimation)
    int m_coreAnimationId;

    // id of animation to use
    int m_animationId;

    // total length of the animation
    float m_length;

    // reference weight from which the actual weight is calculated
    float m_weight;
  };

private:
  Info m_info;

  int m_loopCount;

  // cumulated time since the beginning of the animation in seconds
  float m_cumulatedTime;

  // time local to the animation in seconds
  float m_time;

  // the weight of the animation as calculated as a function of m_time
  float m_weight;

  CalCoreAnimation* m_pCoreAnimation;

  // fix keyframe time if not equal to duration
  void FixAnimation();

  // cache of upper bound for each core track
  std::vector<unsigned int> m_cacheKeyframe;

public:
  UCAL3D_API CalAnimationAlt();
  UCAL3D_API CalAnimationAlt(CalCoreAnimation* coreAnimation);

  virtual ~CalAnimationAlt();

  // update the time weight ...
  UCAL3D_API int update(float deltaTimeInSecond);

  UCAL3D_API CalAnimationAlt& setState(State state) { m_info.m_state = state; return *this; }
  UCAL3D_API State getState() { return m_info.m_state; }

  // channel accessors 
  UCAL3D_API CalAnimationAlt& setChannel(CalBlender::Channel channel) { m_info.m_channel =channel; return *this; }
  UCAL3D_API CalBlender::Channel getChannel() const { return m_info.m_channel; }

  // end callback accessors
  UCAL3D_API CalAnimationAlt& setStopCallback(StopCallback* pStopCallback) { m_info.m_pStopCallback = pStopCallback; return *this; }
  UCAL3D_API StopCallback* getStopCallback() { return m_info.m_pStopCallback; }

  // set weight function
  UCAL3D_API CalAnimationAlt& setWeightFunction(WeightFunction* pWeightFunction);
  UCAL3D_API WeightFunction* getWeightFunction() { return m_info.m_pWeightFunction; }

  // set time function
  UCAL3D_API CalAnimationAlt& setTimeFunction(TimeFunction* pTimeFunction);
  UCAL3D_API TimeFunction* getTimeFunction() { return m_info.m_pTimeFunction; }
  
  // core animation accessors (maybe we could forget that)
  UCAL3D_API CalAnimationAlt& setCoreAnimationId(int id) { m_info.m_coreAnimationId = id; return *this; }
  UCAL3D_API int getCoreAnimationId() { return m_info.m_coreAnimationId; }

  UCAL3D_API CalAnimationAlt& setAnimationId(int id) { m_info.m_animationId = id; return *this; }
  UCAL3D_API int getAnimationId() { return m_info.m_animationId; }

  // set length
  UCAL3D_API CalAnimationAlt& setLength(float length) { if(length >= 0.f && length < 0.0001f) m_info.m_length = getDuration(); else m_info.m_length = length; return *this; }
  UCAL3D_API float getLength() const { return m_info.m_length; }

  // reference weight
  UCAL3D_API void setReferenceWeight(float weight) { m_info.m_weight=weight;}
  UCAL3D_API float getReferenceWeight() const { return m_info.m_weight;}

  // weight accessors
  UCAL3D_API void setWeight(float weight) { m_weight = weight; }
  UCAL3D_API float getWeight() const { return m_weight; }

  // time accessors
  UCAL3D_API void setTime(float _time) { m_time = _time;}
  UCAL3D_API float getTime() const { return m_time; }
  
  // core animation accessors
  UCAL3D_API void setCoreAnimation(CalCoreAnimation* pCoreAnimation);
  UCAL3D_API CalCoreAnimation* getCoreAnimation() { return m_pCoreAnimation;}

  UCAL3D_API float getDuration() { return getCoreAnimation()->getDuration(); }

  UCAL3D_API int getLoopCount() { return m_loopCount; }
  UCAL3D_API void setLoopCount(int loopCount) { m_loopCount = loopCount; }

  UCAL3D_API float getCumulatedTime() { return m_cumulatedTime; }

  UCAL3D_API bool getState(CalCoreTrack* track,int indexOfCoreTrack,float time, CalVector& translation, CalQuaternion& rotation);
  UCAL3D_API void getUpperBound(CalCoreTrack* track,int indexOfCoreTrack,float time,std::vector<CalCoreKeyframe*>::iterator& result);

};


#endif 
