/*
 * $Id: grayfont.c,v 1.13 2003/10/04 16:09:44 nordstrom Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <PalmOS.h>
#include <FntGlue.h>
#include <TxtGlue.h>

#include "os.h"
#include "util.h"
#include "prefsdata.h"
#define NO_GRAY_FONT_SUBSTITUTION
#include "grayfont.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define MAX_FONT_ID                             255
#define HIGH_DENSITY_FEATURE_SET_VERSION        4
#define RESOURCE_NAME_ID             'G'
#define RESOURCE_NAME_IDLETTER       0
#define RESOURCE_NAME_ORIENTATION    1
#define RESOURCE_NAME_BITMAP_VERSION 2
#define RESOURCE_NAME_DEPTH          3

#define RED_BITS                     5
#define GREEN_BITS                   6
#define BLUE_BITS                    5
#define MAX_RED                      ( ( 1 << RED_BITS ) - 1 )
#define MAX_GREEN                    ( ( 1 << GREEN_BITS ) - 1 )
#define MAX_BLUE                     ( ( 1 << BLUE_BITS ) - 1 )
#define palmCompressed               0x80



/***********************************************************************
 *
 *      Internal types
 *
 ***********************************************************************/
typedef enum {
        palmPixelFormatIndexed = 0,
        palmPixelFormat565,
        palmPixelFormat565LE,
        palmPixelFormatIndexedLE,
} PalmPixelFormatType;

typedef struct {
        Int16  width;
        Int16  height;
        UInt16 rowBytes;
        UInt8  flags;
        UInt8  reserved0;
        UInt8  pixelSize;
        UInt8  version;
} PalmBitmapType;

typedef struct {
        Int16  width;
        Int16  height;
        UInt16 rowBytes;
        UInt8  flags;
        UInt8  reserved0;
        UInt8  pixelSize;
        UInt8  version;
        UInt16 nextDepthOffset;
        UInt16 reserved[2];
} PalmBitmapTypeV1;

typedef struct {
        Int16  width;
        Int16  height;
        UInt16 rowBytes;
        UInt8  flags;
        UInt8  reserved0;
        UInt8  pixelSize;
        UInt8  version;
        UInt16 nextDepthOffset;
        UInt8  transparentIndex;
        UInt8  compressionType;
        UInt16 reserved;
} PalmBitmapTypeV2;

typedef struct {
       Int16  width;
       Int16  height;
       UInt16 rowBytes;
       UInt8  flags;
       UInt8  reserved0;
       UInt8  pixelSize;
       UInt8  version;
       UInt8  size;
       UInt8  pixelFormat;
       UInt8  unused;
       UInt8  compressionType;
       UInt16 density;
       UInt32 transparentValue;
       UInt32 nextBitmapOffset;
} PalmBitmapTypeV3;

typedef struct {
    UInt16 firstGlyph;   /* first glyph found in resource */
    UInt16 lastGlyph;    /* last glyph found in resource */
    UInt16 resourceID;   /* the ID of the resource */
    UInt16 reserved;     /* should be zero for now */
} GrayFontBitmapsInfo;

typedef struct {
    UInt16 fontVersion;
    UInt16 firstChar;
    UInt16 lastChar;
    UInt16 fRectWidth;
    UInt16 fRectHeight;
    Int16  ascent;
    Int16  descent;
    Int16  leading;
    UInt16 numberOfBitmapResources;
    UInt16 bitmapResourceTableOffset;
    UInt16 glyphInfoTableOffset;
} GrayFontType;

typedef struct {
    UInt16 offset;
    UInt16 length;
} GrayFontResourceIndexEntry;

typedef struct {
    Int16   leftKerning;
    Int16   advance;
    UInt16  bitmapWidth;
    UInt16  resourceNumber;
    UInt16  positionInResourceIndex;
    UInt16  reserved;
} GrayFontGlyphInfo;

typedef struct {
    GrayFontType         header;
    GrayFontGlyphInfo*   glyphList;
    GrayFontBitmapsInfo* bitmapsResourceIndex;
} GrayFontInfo;

typedef union {
    DmResType resType;
    Char      string[4];
} ResourceType;

typedef enum {
    palmBitmapCompressionTypeScanLine = 0,
    palmBitmapCompressionTypeRLE,
    palmBitmapCompressionTypePackBits,
    palmBitmapCompressionTypeEnd,
    palmBitmapCompressionTypeBest = 0x64,
    palmBitmapCompressionTypeNone = 0xFF
} PalmBitmapCompressionType;



/***********************************************************************
 *
 *      Local functions
 *
 ***********************************************************************/
static void GrayFntDelete ( FontID font ) GRAYFONT_SECTION;
static GrayFontGlyphInfo* GetGlyph ( WChar ch ) GRAYFONT_SECTION;
static UInt16 TempBitmapSize ( UInt16 width, UInt16 height ) GRAYFONT_SECTION;
static void ColorizeBitmapSetColorMap ( RGBColorType* color, Boolean invert )
    GRAYFONT_SECTION;
static void ColorizeBitmap4 ( PalmBitmapType* dest, PalmBitmapType* src,
    Boolean scanLine ) GRAYFONT_SECTION;
static void InvertBitmap4 ( PalmBitmapType* dest, PalmBitmapType* src,
    Boolean scanLine )
    GRAYFONT_SECTION;



/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static ResourceType    resource;
static GrayFontInfo*   grayFont[ MAX_FONT_ID + 1];
static GrayFontInfo*   currentFontPtr = NULL;
static FontID          currentFont;
static Boolean         uses8BitChars;
static Boolean         havePalmHiRes;
static UInt8           substitutionList[ MAX_FONT_ID + 1 ];
static Boolean         active = false;
static PalmBitmapType* tempBitmap = NULL;
static UInt16          tempBitmapSize = 0;
static UInt16          tempBitmapHeaderSize;
static UInt16          colorMap[ 16 ];


/* Set a map for colorizing a bitmap */
static void ColorizeBitmapSetColorMap
      (
      RGBColorType*  color,
      Boolean        invert
      )
{
    UInt16  i;
    for ( i = 0 ; i < 16 ; i++ ) {
         UInt8  r;
         UInt8  g;
         UInt8  b;
         UInt16 rgb;
         r = ( ( ( 15 - i ) * 255 + i * ( color->r ) ) / 15 ) >> ( 8 - RED_BITS );
         g = ( ( ( 15 - i ) * 255 + i * ( color->g ) ) / 15 ) >> ( 8 - GREEN_BITS );
         b = ( ( ( 15 - i ) * 255 + i * ( color->b ) ) / 15 ) >> ( 8 - BLUE_BITS );

         rgb = ( r << ( BLUE_BITS + GREEN_BITS ) ) | ( g << BLUE_BITS ) | b;
         if ( invert )
             colorMap[ 15 - i ] = rgb;
         else
             colorMap[ i ] = rgb;
    }
}



/* Colorize a 4-bit bitmap */
/* Input bitmap: type 1 or 3.  Output: type 2 or 3 */
/* No colormaps permitted */
static void ColorizeBitmap4
      (
      PalmBitmapType*   destBitmap,
      PalmBitmapType*   srcBitmap,
      Boolean           srcScanLineCompress
      )
{
    UInt16  x;
    UInt16  y;
    UInt8*  in;
    UInt16* out;
    UInt8*  src;
    UInt16* dest;
    UInt16  destRowWords;
    UInt16  srcRowBytes;
    UInt16  width;
    UInt16  height;

    srcRowBytes        = srcBitmap->rowBytes;
    width              = srcBitmap->width;
    height             = srcBitmap->height;
    destBitmap->width  = width;
    destBitmap->height = height;
    destBitmap->pixelSize = 16;

    destRowWords          = width;
    destBitmap->rowBytes = 2 * destRowWords;
    if ( srcBitmap->version < 3 ) {
        /* note that sizeof( PalmBitmapTypeV1 ) is the same as
           sizeof( PalmBitmapTypeV2 ) */
        src = ( UInt8* )srcBitmap + sizeof( PalmBitmapTypeV1 );
    }
    else {
        src = ( UInt8* )srcBitmap + sizeof( PalmBitmapTypeV3 );
    }
    if ( destBitmap->version < 3 ) {
        dest = ( UInt16* )( ( UInt8* )destBitmap +
                              sizeof( PalmBitmapTypeV2 ) );
    }
    else {
        dest = ( UInt16* )( ( UInt8* )destBitmap +
                              sizeof( PalmBitmapTypeV3 ) );
    }

    if ( ! srcScanLineCompress ) {
        for ( y = 0 ; y < height ; y ++ ) {
            in  = src + y * srcRowBytes;
            out = dest + y * destRowWords;
            for ( x = 0 ; x < width ; x += 2 ) {
                 *out++ = colorMap[ *in >> 4 ];
                 /* The next operation may overwrite the next
                    row if width is odd, but the output bitmap
                    is allocated with two spare bytes so this
                    is not a problem */
                 *out++ = colorMap[ *in & 0xF ];
                 in++;
            }
        }
    }
    else {
        UInt16* lastRow;
        UInt16  srcRowNibbles;

        srcRowNibbles = 2 * srcRowBytes;

        /* To prevent crashes due to bad fonts */
        lastRow = dest;
        /* skip length data */
        if ( srcBitmap->version < 3 )
            in = src + 2;
        else
            in = src + 4;
        for ( y = 0 ; y < height ; y++ ) {
            UInt16* startOut;
            out      = dest + y * destRowWords;
            startOut = out;
            x        = 0;
            while ( x < srcRowNibbles ) {
                UInt8   octupleMarker;
                UInt8   mask;

                octupleMarker = *in++;
                mask          = 0x80;
                do {
                    if ( mask & octupleMarker ) {
                        if ( x < width ) {
                            *out++ = colorMap[ *in >> 4 ];
                            *out++ = colorMap[ *in & 0xF ];
                        }
                        in++;
                    }
                    else if ( x < width ) {
                        *out++ = lastRow[ x ];
                        *out++ = lastRow[ x + 1 ];
                    }
                    mask >>= 1;
                    x     += 2;
                } while ( mask != 0 && x < srcRowNibbles );
            }
            lastRow = startOut;
        }
    }
}



/* Invert a 4-bit bitmap */
/* Input bitmap: type 1 or 3.  Output bitmap: same type */
/* No colormaps permitted */
void InvertBitmap4
      (
      PalmBitmapType*   destBitmap,
      PalmBitmapType*   srcBitmap,
      Boolean           scanLineCompress
      )
{
    UInt16  x;
    UInt16  y;
    UInt8*  in;
    UInt8*  out;
    UInt8*  src;
    UInt8*  dest;
    UInt16  destRowBytes;
    UInt16  srcRowBytes;
    UInt16  width;
    UInt16  height;

    srcRowBytes        = srcBitmap->rowBytes;
    width              = srcBitmap->width;
    height             = srcBitmap->height;
    destBitmap->width  = width;
    destBitmap->height = height;
    destBitmap->pixelSize = 4;

    destRowBytes = ( width + 1 ) / 2;
    if ( destRowBytes % 2 != 0 )
        destRowBytes++;
    destBitmap->rowBytes = destRowBytes;

    if ( srcBitmap->version < 3 ) {
        /* note that sizeof( PalmBitmapTypeV1 ) is the same as
           sizeof( PalmBitmapTypeV2 ) */
        src  = ( UInt8* )srcBitmap + sizeof( PalmBitmapTypeV1 );
        dest = ( UInt8* )destBitmap + sizeof( PalmBitmapTypeV1 );
    }
    else {
        src  = ( UInt8* )srcBitmap + sizeof( PalmBitmapTypeV3 );
        dest = ( UInt8* )destBitmap + sizeof( PalmBitmapTypeV3 );
    }

    if ( ! scanLineCompress ) {
        /* the following two lines work on V1, V2 and V3 */
        ( ( PalmBitmapTypeV2* )destBitmap )->flags = 0;
        ( ( PalmBitmapTypeV2* )destBitmap )->compressionType = 0;
        for ( y = 0 ; y < height ; y ++ ) {
            in  = ( UInt8* ) src + y * srcRowBytes;
            out = ( UInt8* ) dest + y * destRowBytes;
            for ( x = 0 ; x < width ; x += 2 ) {
                 *out = ( ( 15 - ( *in >> 4 ) ) << 4 ) | ( 15 - ( *in & 0xF ) );
                 in++;
                 out++;
            }
        }
    }
    else {
        UInt32 length;

        /* the following two fields match up on V1, V2 and V3 */
        ( ( PalmBitmapTypeV2* )destBitmap )->flags |= palmCompressed;
        ( ( PalmBitmapTypeV2* )destBitmap )->compressionType =
                                                  palmBitmapCompressionTypeScanLine;
        src    = ( UInt8* )srcBitmap;
        dest   = ( UInt8* )destBitmap;

        if ( 3 <= srcBitmap->version ) {
            length = ( ( UInt32 )( ( ( UInt16* )srcBitmap )[ 0 ] ) << 16 ) |
                     ( UInt32 )( ( ( UInt16* )srcBitmap )[ 1 ] );
            MemMove( dest, src, 4 );
            src  += 4;
            dest += 4;
        }
        else {
            length = ( ( UInt16* )srcBitmap )[ 0 ];
            MemMove( dest, src, 2 );
            src  += 2;
            dest += 2;
        }
        while ( 0 < length ) {
            UInt8  octupleMarker;
            UInt8  mask;
            octupleMarker = *src;
            mask          = 0x80;
            *dest++       = *src++;
            length--;
            while ( 0 < length && 0 != mask ) {
                if ( mask & octupleMarker ) {
                    *dest = ( ( 15 - ( *src >> 4 ) ) << 4 ) | ( 15 - ( *src & 0xF ) );
                    src++;
                    dest++;
                    length--;
                }
                mask >>= 1;
                length--;
            }
        }
    }
}


static void GrayFntDelete
       (
       FontID font
       )
{
   if ( ! active || grayFont[ font ] == NULL ) {
       return;
   }
   if ( substitutionList[ currentFont ] == font ) {
       currentFont    = stdFont;
       currentFontPtr = NULL;
       FntSetFont( currentFont );
   }
   if ( grayFont[ font ]->glyphList != NULL )
       SafeMemPtrFree( grayFont[ font ]->glyphList );
   if ( grayFont[ font ]->bitmapsResourceIndex != NULL )
       SafeMemPtrFree( grayFont[ font ]->bitmapsResourceIndex );
   SafeMemPtrFree( grayFont[ font ] );
   grayFont[ font ] = NULL;
}



static GrayFontGlyphInfo* GetGlyph
      (
      WChar ch
      )
{
    UInt16 index;
    if ( ch < currentFontPtr->header.firstChar ||
         currentFontPtr->header.lastChar < ch  ||
         currentFontPtr->glyphList[ ch -
                    currentFontPtr->header.firstChar ].resourceNumber == 0
       )
        index = currentFontPtr->header.lastChar -
                    currentFontPtr->header.firstChar;
    else
        index = ch - currentFontPtr->header.firstChar;
    return &( currentFontPtr->glyphList[ index ] );
}



/* Start the gray-scale font functions */
void GrayFntStart( void )
{
    UInt16  i;
    UInt32  charEncoding;
    UInt32  version;
    Err     err;
    if ( grayFont != NULL )
        GrayFntStop();
    for ( i = 0 ; i <= MAX_FONT_ID ; i++ ) {
        grayFont[ i ]         = NULL;
        substitutionList[ i ] = i;
    }
    currentFont    = FntGetFont();
    currentFontPtr = NULL;
    err = FtrGet( sysFtrCreator, sysFtrNumEncoding, &charEncoding );
    if ( err == errNone )
        uses8BitChars = ( charEncoding <= charEncodingPalmLatin );
    else
        uses8BitChars = true;
    err = FtrGet( sysFtrCreator, sysFtrNumWinVersion, &version );
    havePalmHiRes = ( HIGH_DENSITY_FEATURE_SET_VERSION <= version );
    resource.string[ RESOURCE_NAME_IDLETTER ] = RESOURCE_NAME_ID;
    resource.string[ RESOURCE_NAME_ORIENTATION ] = GRAY_FONT_NORMAL;
    resource.string[ RESOURCE_NAME_BITMAP_VERSION ] =
        havePalmHiRes ? '3' : '1';
    resource.string[ RESOURCE_NAME_DEPTH ] = '4';
    active = true;
}



/* Stop them and clear memory */
void GrayFntStop( void )
{
    UInt16 i;
    if ( active ) {
        for ( i = 0 ; i <= MAX_FONT_ID ; i++ ) {
            GrayFntDelete( i );
            grayFont[ i ] = NULL;
        }
        active = false;
    }
    if ( tempBitmap != NULL ) {
        SafeMemPtrFree( tempBitmap );
        tempBitmap     = NULL;
        tempBitmapSize = 0;
    }
}



/* size of temporary bitmap for font */
static UInt16 TempBitmapSize
        (
        UInt16 width,
        UInt16 height
        )
{
    UInt16  rowBytes;
    UInt16  extra;
    if ( havePalmHiRes )
        tempBitmapHeaderSize = sizeof( PalmBitmapTypeV3 );
    else
        tempBitmapHeaderSize = sizeof( PalmBitmapTypeV2 );
    if ( 16 <= GetMaxBitDepth() ) {
        rowBytes = width * 2;
        extra    = 2; /* the colorization algorithm needs extra space */
    }
    else {
        rowBytes = ( width + 1 ) / 2;
        if ( rowBytes % 2 != 0 )
            rowBytes++;
        extra    = 0;
    }
    return tempBitmapHeaderSize + extra + height * rowBytes;
}



Err GrayFntDefineFont
       (
       FontID font,
       void*  fontP
       )
{
    UInt16 numGlyphs;
    UInt16 newTempSize;
    UInt16 verticalLayoutSize;
    GrayFontInfo* newFont;

    GrayFntDelete( font );
    if ( * ( UInt8* ) fontP & 0x80 ) {
        return FntDefineFont( font, fontP );
    }
    grayFont[ font ] = SafeMemPtrNew( sizeof( GrayFontInfo ) );
    newFont = grayFont[ font ];
    if ( newFont == NULL )
        return memErrNotEnoughSpace;
    MemMove( &( newFont->header ), fontP, sizeof( GrayFontType ) );
    numGlyphs = newFont->header.lastChar - newFont->header.firstChar + 1;
    newFont->bitmapsResourceIndex = NULL;
    newFont->glyphList = SafeMemPtrNew( sizeof( GrayFontGlyphInfo ) *
                                                numGlyphs );
    if ( newFont->glyphList == NULL ) {
        GrayFntDelete( font );
        return memErrNotEnoughSpace;
    }
    newFont->bitmapsResourceIndex =
        SafeMemPtrNew( sizeof( GrayFontBitmapsInfo ) *
                      newFont->header.numberOfBitmapResources );
    if ( newFont->bitmapsResourceIndex == NULL ) {
        GrayFntDelete( font );
        return memErrNotEnoughSpace;
    }

    MemMove( newFont->glyphList,
             ( UInt8* ) fontP + newFont->header.glyphInfoTableOffset,
             sizeof( GrayFontGlyphInfo ) * numGlyphs );

    MemMove( newFont->bitmapsResourceIndex,
             ( UInt8* ) fontP +
                  newFont->header.bitmapResourceTableOffset,
             sizeof( GrayFontBitmapsInfo ) *
                    newFont->header.numberOfBitmapResources );

    if ( substitutionList[ currentFont ] == font ) {
        currentFontPtr = newFont;
    }

    newTempSize        = TempBitmapSize( newFont->header.fRectWidth,
                              newFont->header.fRectHeight );
    verticalLayoutSize = TempBitmapSize( newFont->header.fRectHeight,
                              newFont->header.fRectWidth );

    if ( newTempSize < verticalLayoutSize )
        newTempSize = verticalLayoutSize;
    if ( tempBitmapSize < newTempSize || tempBitmap == NULL ) {
        if ( tempBitmap != NULL ) {
            SafeMemPtrFree( tempBitmap );
            tempBitmap     = NULL;
            tempBitmapSize = 0;
        }
        tempBitmap = SafeMemPtrNew( newTempSize );
        if ( tempBitmap != NULL ) {
            tempBitmapSize = newTempSize;
            MemSet( tempBitmap, tempBitmapSize, 0 );
            if ( havePalmHiRes ) {
                (( PalmBitmapTypeV3* )tempBitmap)->size    
                    = sizeof( PalmBitmapTypeV3 );
                (( PalmBitmapTypeV3* )tempBitmap)->density
                    = PalmGetDensity();
                tempBitmap->version = 3;
            }
            else {
                tempBitmap->version = 2;
            }
        }
    }
    return errNone;
}




FontID GrayFntGetFont( void )
{
    return currentFont;
}



FontID GrayFntSetFont
        (
        FontID font
        )
{
    FontID    oldFont;
    oldFont        = currentFont;
    currentFont    = font;
    currentFontPtr = grayFont[ substitutionList[ font ] ];
    if ( grayFont[ font ] == NULL )
        FntSetFont( substitutionList[ font ] );
    return oldFont;
}



Int16 GrayFntLineHeight( void )
{
    if ( currentFontPtr == NULL )
        return FntLineHeight();
    else {
        Coord height;
        height = currentFontPtr->header.leading +
                   currentFontPtr->header.fRectHeight;
        HiResAdjustNativeToCurrent( &height );
        return height;
    }
}



Int16 GrayFntCharHeight( void )
{
    if ( currentFontPtr == NULL )
        return FntCharHeight();
    else {
        Coord height;
        height = currentFontPtr->header.fRectHeight;
        HiResAdjustNativeToCurrent( &height );
        return height;
    }
}



Int16 GrayFntWCharWidth
       (
       WChar ch
       )
{
    if ( currentFontPtr == NULL )
        return FntGlueWCharWidth( ch );
    else {
        Coord width;
        width = GetGlyph( ch )->advance;
        HiResAdjustNativeToCurrent( &width );
        return width;
    }
}



Int16 GrayFntCharWidth
       (
       Char ch
       )
{
    if ( currentFontPtr == NULL )
        return FntCharWidth( ch );
    else {
        Coord width;
        width = GetGlyph( ( UInt8 )ch )->advance;
        HiResAdjustNativeToCurrent( &width );
        return width;
    }
}



Int16 GrayFntCharsWidth
       (
       Char const* chars,
       Int16 length
       )
{
    Int16  width;
    UInt8 const* p;
    if ( currentFontPtr == NULL )
        return FntCharsWidth( chars, length );
    width = 0;
    if ( uses8BitChars ) {
        p = ( UInt8* )chars;
        while ( 0 < length-- ) {
            width += GetGlyph( *p++ )->advance;
        }
    }
    else {
        WChar  ch;
        UInt32 inOffset;
        inOffset = 0;
        while ( inOffset < length ) {
            inOffset += TxtGlueGetNextChar( chars, inOffset, &ch );
            if ( length < inOffset )
                break;
            width += GetGlyph( ch )->advance;
        }
    }
    HiResAdjustNativeToCurrent( &width );
    return width;
}



Char GrayFntSetOrientation
      (
      Char o
      )
{
    Char oldOrientation;
    oldOrientation = resource.string[ RESOURCE_NAME_ORIENTATION ];
    resource.string[ RESOURCE_NAME_ORIENTATION ] = o;
    return oldOrientation;
}



void GrayWinDrawCharsGeneral
      (
      Char const* chars,
      Int16   length,
      Coord   x,
      Coord   y,
      Boolean invert
      )
{
    Coord*  toIncrement;
    Int16   incrementSign;
    UInt32  inOffset;
    Boolean incrementFirst;
    Boolean doMapColors;
    Boolean doInvert;
    UInt16  lastResourceNumber;
    MemHandle bitmapHandle   = NULL;
    UInt8*    bitmapResource = NULL;
    UInt16  prevCoordSys;

    if ( currentFontPtr == NULL ) {
        if ( invert )
            WinDrawInvertedChars( chars, length, x, y );
        else
            WinDrawChars( chars, length, x, y );
        return;
    }

    HiResAdjustCurrentToNative( &x );
    HiResAdjustCurrentToNative( &y );

    prevCoordSys   = PalmSetCoordinateSystem( NATIVE );

    switch( resource.string[ RESOURCE_NAME_ORIENTATION ] )
    {
        case GRAY_FONT_LEFT:
            y++;
            toIncrement    = &y;
            incrementSign  = -1;
            incrementFirst = true;
            break;
        case GRAY_FONT_RIGHT:
            toIncrement    = &y;
            incrementSign  = 1;
            incrementFirst = false;
            x -= currentFontPtr->header.fRectHeight;
            break;
        default:
            toIncrement    = &x;
            incrementSign  = 1;
            incrementFirst = false;
            break;
    }
    doMapColors = false;
    doInvert    = false;
    if ( Prefs()->screenDepth == 16 ) {
        RGBColorType color;
        WinSetTextColorRGB( NULL, &color );
        if ( ( color.r != 0 || color.g != 0 || color.b != 0 || invert ) &&
             tempBitmap != NULL ) {
            ColorizeBitmapSetColorMap( &color, invert );
            if ( havePalmHiRes ) {
                ( ( PalmBitmapTypeV3* )tempBitmap )->pixelFormat =
                    palmPixelFormat565;
                tempBitmap->version = 3;
            }
            else {
                tempBitmap->version = 2;
            }
            doMapColors = true;
        }
    }
    else if ( invert && tempBitmap != NULL ) {
        doInvert = true;
        if ( havePalmHiRes ) {
            ( ( PalmBitmapTypeV3* )tempBitmap )->pixelFormat =
                palmPixelFormatIndexed;
            tempBitmap->version = 3;
        }
        else {
            tempBitmap->version = 1;
        }
    }

    inOffset           = 0;
    lastResourceNumber = 0;
    while ( inOffset < length ) {
        GrayFontGlyphInfo*     glyph;
        WChar                  ch;
        UInt16                 resourceNumber;

        inOffset += TxtGlueGetNextChar( chars, inOffset, &ch );
        if ( length < inOffset )
            break;
        glyph = GetGlyph( ch );
        if ( incrementFirst )
            *toIncrement += incrementSign * glyph->advance;

        /* FIXME: handle kerning and advance correctly */

        resourceNumber = glyph->resourceNumber;
        if ( resourceNumber != lastResourceNumber ) {
            UInt16  resourceID;
            if ( bitmapHandle != NULL ) {
                MemHandleUnlock( bitmapHandle );
                DmReleaseResource( bitmapHandle );
            }
            resourceID     = currentFontPtr->bitmapsResourceIndex[
                                 resourceNumber - 1 ].resourceID;
            bitmapHandle   = DmGetResource( resource.resType, resourceID );
            if ( bitmapHandle != NULL )
                bitmapResource = MemHandleLock( bitmapHandle );
            lastResourceNumber = resourceNumber;
        }

        if ( bitmapHandle != NULL ) {
            PalmBitmapType* bitmap;

            bitmap = ( PalmBitmapType* ) ( bitmapResource +
                        ( ( GrayFontResourceIndexEntry* )bitmapResource )[
                                 glyph->positionInResourceIndex ].offset );
            if ( doInvert ) {
                Boolean scanLineCompress;
                if ( bitmap->flags & palmCompressed )
                    scanLineCompress = true;
                else
                    scanLineCompress = false;
                InvertBitmap4( tempBitmap, bitmap, scanLineCompress );
                WinDrawBitmap( ( BitmapType* )tempBitmap, x, y );
            }
            else if ( doMapColors ) {
                Boolean scanLineCompress;
                if ( bitmap->flags & palmCompressed )
                    scanLineCompress = true; 
                else
                    scanLineCompress = false;
                ColorizeBitmap4( tempBitmap, bitmap, scanLineCompress );
                WinDrawBitmap( ( BitmapType* )tempBitmap, x, y );
            }
            else {
                WinDrawBitmap( ( BitmapType* )bitmap, x, y );
            }
        }
        if ( ! incrementFirst )
            *toIncrement += incrementSign * glyph->advance;
    }
    if ( bitmapHandle != NULL ) {
        MemHandleUnlock( bitmapHandle );
        DmReleaseResource( bitmapHandle );
    }

    PalmSetCoordinateSystem( prevCoordSys );
}




void GrayWinDrawChar
      (
      WChar ch,
      Coord x,
      Coord y
      )
{
    Char  line[80];
    Int16 length;
    if ( currentFontPtr == NULL ) {
        WinDrawChar( ch, x, y );
        return;
    }
    length = TxtGlueSetNextChar( line, 0, ch );
    GrayWinDrawChars( line, length, x, y );
}



Boolean GrayFntIsCurrentGray( void )
{
    return currentFontPtr != NULL;
}



/* Handle font substitution */
void GrayFntSubstitute
      (
      FontID oldFont,
      FontID newFont
      )
{
    substitutionList[ oldFont ] = newFont;
    if ( currentFont == oldFont )
        GrayFntSetFont( currentFont );
}



void GrayFntClearSubstitutionList( void )
{
    Int16 i;
    for ( i = 0 ; i <= MAX_FONT_ID ; i++ )
         substitutionList[ i ] = i;
    GrayFntSetFont( currentFont );
}

