/*
 * $Id: fiveway.c,v 1.9 2003/10/07 20:57:15 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2002, Mark Ian Lillywhite and Michael Nordstrom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "os.h"
#include "control.h"
#include "debug.h"
#include "prefsdata.h"
#include "util.h"
#include "rotate.h"

#include "fiveway.h"


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Int16 selectedRow;
static Int8 currentRow;



void FiveWayResetValues( void )
{
    selectedRow = NO_ROW;
    currentRow  = 0;
}


/* update currentRow -- currentRow uses absolute row values */
void FiveWaySetRow( Int8 row )
{
    currentRow = row;
}


Int8 FiveWayGetRow( void )
{
    return currentRow;
}


Int8 FiveWaySelectedRow( void )
{
    return selectedRow;
}


void FiveWayHighlightRow( Boolean enable )
{
    if ( selectedRow == NO_ROW )
        return;

    if ( enable )
        selectedRow = currentRow - LibraryGetFirstVisibleRow();

    HighlightRectangle( LibraryGetDisplayListBounds( selectedRow ), 0,
        enable, BUTTON);

    if ( ! enable )
        selectedRow = NO_ROW;
}


/* Handler to process any FiveWay requests in the library form */
Boolean FiveWayLibraryHandler
    (
    EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean        handled;
    Boolean        scroll;
    Int16          firstVisibleRow;

    handled         = true;
    scroll          = false;
    firstVisibleRow = LibraryGetFirstVisibleRow();

    if ( FiveWayCenterPressed( event )) {
        if ( selectedRow != NO_ROW ) {
            OpenNewDocument( LibraryGetDisplayListIndex( selectedRow ) );
            selectedRow = NO_ROW;
        }
        else {
            selectedRow = currentRow - firstVisibleRow;
            HighlightRectangle( LibraryGetDisplayListBounds( selectedRow ),
                0, true, BUTTON );
        }
    }
    else if ( FiveWayDirectionPressed( event, Left )) {
        if ( selectedRow != NO_ROW ) {
            HighlightRectangle( LibraryGetDisplayListBounds( selectedRow ),
                0, false, BUTTON);
            selectedRow = NO_ROW;
        }
        else {
            EventType stopEvent;

            stopEvent.eType = appStopEvent;
            EvtAddEventToQueue(&stopEvent);
        }
    }
    else if ( FiveWayDirectionPressed( event, Right )) {
        if ( selectedRow != NO_ROW ) {
            LibrarySetLastIndexForRow( currentRow - firstVisibleRow );
            IconPopupList( currentRow - firstVisibleRow );
        }
    }
    else {
        if ( LibraryGetNumberOfRows() == 0 ) {
            handled = false;
            return handled;
        }

        if ( FiveWayDirectionPressed( event, Up )) {
            if ( selectedRow != NO_ROW ) {
                if ( 0 < currentRow ) {
                    HighlightRectangle(
                        LibraryGetDisplayListBounds( selectedRow ), 0,
                        false, BUTTON );
                    if ( currentRow == firstVisibleRow )
                        scroll = true;
                    currentRow--;
                }
                else {
                    return handled;
                }
            }
            if ( scroll ) {
                ScrollUp( ONE_ROW );
            }
            else {
                selectedRow = currentRow - firstVisibleRow;
                HighlightRectangle( LibraryGetDisplayListBounds( selectedRow ),
                    0, true, BUTTON);
            }
        }
        else if ( FiveWayDirectionPressed( event, Down )) {
            if ( selectedRow != NO_ROW ) {
                if ( currentRow < LibraryGetNumberOfRows() - 1) {
                    HighlightRectangle(
                        LibraryGetDisplayListBounds( selectedRow ), 0,
                        false, BUTTON );
                    if ( currentRow == LibraryGetVisibleRows() +
                                   firstVisibleRow - 1)
                        scroll = true;
                    currentRow++;
                }
                else {
                    return handled;
                }
            }
            if ( scroll ) {
                ScrollDown( ONE_ROW );
            }
            else {
                selectedRow = currentRow - firstVisibleRow;
                HighlightRectangle( LibraryGetDisplayListBounds(selectedRow),
                    0, true, BUTTON);
            }
        }
        else {
            handled = false;
        }
    }
    return handled;
}



/* Handler to process any FiveWay requests in the main form */
Boolean FiveWayMainHandler
    (
    EventType* event
    )
{
    Boolean handled;
    Boolean didLeft;
    Boolean didRight;
    Boolean didUp;
    Boolean didDown;

    handled = false;

    if ( ! Prefs()->arrowKeys || ! HaveFiveWay() || ! IsFiveWayEvent( event ))
    {
        return handled;
    }

    didLeft  = FiveWayKeyPressed( event, Left );
    didRight = FiveWayKeyPressed( event, Right );
    didUp    = FiveWayKeyPressed( event, Up );
    didDown  = FiveWayKeyPressed( event, Down );

    if ( RotSelect( didLeft, didDown, didUp ) &&
         Prefs()->arrowMode[ LEFT_ARROW ] != SELECT_NONE ) {
        DoSelectTypeAction( Prefs()->arrowMode[ LEFT_ARROW ] );
        handled = true;
    }
    else if ( RotSelect( didRight, didUp, didDown ) &&
         Prefs()->arrowMode[ RIGHT_ARROW ] != SELECT_NONE ) {
        DoSelectTypeAction( Prefs()->arrowMode[ RIGHT_ARROW ] );
        handled = true;
    }
    else if ( RotSelect( didUp, didLeft, didRight ) &&
         Prefs()->arrowMode[ UP_ARROW ] != SELECT_NONE ) {
        DoSelectTypeAction( Prefs()->arrowMode[ UP_ARROW ] );
        handled = true;
    }
    else if ( RotSelect( didDown, didRight, didLeft ) &&
         Prefs()->arrowMode[ DOWN_ARROW ] != SELECT_NONE ) {
        DoSelectTypeAction( Prefs()->arrowMode[ DOWN_ARROW ] );
        handled = true;
    }
    else if ( FiveWayCenterPressed( event ) &&
         Prefs()->arrowMode[ SELECT_ARROW ] != SELECT_NONE ) {
        DoSelectTypeAction( Prefs()->arrowMode[ SELECT_ARROW ]);
        handled = true;
    }

    return handled;
}

