/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file gui_layer.cpp
 * \brief Implementation of the ptb::gui_layer class.
 * \author Julien Jorge
 */
#include "ptb/layer/gui_layer.hpp"

#include "engine/font_factory.hpp"
#include "engine/game.hpp"
#include "text/font.hpp"

#include <sstream>
#include <claw/png.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param layers The sub layers managed by this layer.
 */
ptb::gui_layer::gui_layer( const std::vector<bear::engine::gui_layer*>& layers )
  : m_sub_layers(layers),
    m_fps_text( NULL,
                bear::engine::font_factory::create
                ("font/fixed_white-7x12.tga") ),
    m_fps_count(0), m_its_count(0), m_show_fps(false), m_levelshot_ratio(0),
    m_levelshot_key_count(0), m_fps_key(bear::input::keyboard::kc_F2),
    m_screenshot_key(bear::input::keyboard::kc_F5),
    m_fullscreen_key(bear::input::keyboard::kc_F12),
    m_level_map_key(bear::input::keyboard::kc_F11),
    m_time_step_key(bear::input::keyboard::kc_F10)
{
  m_fps_text.set_auto_size(true);
  m_fps_text.set_text("0");
  m_fps_text.set_position
    ( m_fps_text.height(), get_size().y - 2 * m_fps_text.height() );

  m_last_fps_check.set();
} // gui_layer::gui_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::gui_layer::~gui_layer()
{
  for (unsigned int i=0; i!=m_sub_layers.size(); ++i)
    delete m_sub_layers[i];
} // gui_layer::~gui_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the sub layers.
 */
void ptb::gui_layer::start()
{
  for (unsigned int i=0; i!=m_sub_layers.size(); ++i)
    m_sub_layers[i]->start();
} // gui_layer::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one step in the progression of the sub layers.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::gui_layer::progress( bear::universe::time_type elapsed_time )
{
  for (unsigned int i=0; i!=m_sub_layers.size(); ++i)
    m_sub_layers[i]->progress(elapsed_time);

  ++m_its_count;
} // gui_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the visibles components of the sub layers on a screen.
 * \param screen The screen on which we draw.
 */
void ptb::gui_layer::render( bear::visual::screen& screen )
{
  for (unsigned int i=0; i!=m_sub_layers.size(); ++i)
    m_sub_layers[i]->render( screen );

  ++m_fps_count;
  render_fps( screen );
} // gui_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a key had been pressed.
 * \param key The value of the pressed key.
 * \remark The message is sent to the sub layers until one processes it.
 */
bool ptb::gui_layer::key_pressed( bear::input::keyboard::key_code key )
{
  bool result = false;
  unsigned int i = m_sub_layers.size();

  if ( m_levelshot_key_count != 0 )
    result = read_levelshot_key( key );

  if ( !result )
    {
      result = true;

      if ( key == m_fps_key )
        m_show_fps = !m_show_fps;
      else if ( key == m_screenshot_key )
        screenshot();
      else if ( key == m_fullscreen_key )
        bear::engine::game::get_instance().toggle_fullscreen();
      else if ( key == m_level_map_key )
        {
          m_levelshot_key_count = 1;
          m_levelshot_ratio = 0;
        }
      else if ( key == m_time_step_key )
        bear::engine::game::get_instance().toggle_time_step();
      else
        result = false;
    }

  while ( (i != 0) && !result )
    {
      --i;
      result = m_sub_layers[i]->key_pressed(key);
    }

  return result;
} // gui_layer::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a joystick button had been pressed.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 * \remark The message is sent to the sub layers until one processes it.
 */
bool ptb::gui_layer::button_pressed
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = false;
  unsigned int i = m_sub_layers.size();

  while ( (i != 0) && !result )
    {
      --i;
      result = m_sub_layers[i]->button_pressed(button, joy_index);
    }

  return result;
} // gui_layer::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been pressed.
 * \param pos The current position of the cursor.
 * \param key The value of the pressed button.
 * \remark The message is sent to the sub layers until one processes it.
 */
bool ptb::gui_layer::mouse_pressed
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  unsigned int i = m_sub_layers.size();

  while ( (i != 0) && !result )
    {
      --i;
      result = m_sub_layers[i]->mouse_pressed(key, pos);
    }

  return result;
} // gui_layer::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button is maintained.
 * \param pos The current position of the cursor.
 * \param key The value of the maintained button.
 * \remark The message is sent to the sub layers until one processes it.
 */
bool ptb::gui_layer::mouse_maintained
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  unsigned int i = m_sub_layers.size();

  while ( (i != 0) && !result )
    {
      --i;
      result = m_sub_layers[i]->mouse_maintained(key, pos);
    }

  return result;
} // gui_layer::mouse_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been released.
 * \param pos The current position of the cursor.
 * \param key The value of the released button.
 * \remark The message is sent to the sub layers until one processes it.
 */
bool ptb::gui_layer::mouse_released
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  unsigned int i = m_sub_layers.size();

  while ( (i != 0) && !result )
    {
      --i;
      result = m_sub_layers[i]->mouse_released(key, pos);
    }

  return result;
} // gui_layer::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that the mouse has been moved.
 * \param pos The new position of the mouse.
 * \remark The message is sent to the sub layers until one processes it.
 */
bool ptb::gui_layer::mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;
  unsigned int i = m_sub_layers.size();

  while ( (i != 0) && !result )
    {
      --i;
      result = m_sub_layers[i]->mouse_move(pos);
    }

  return result;
} // gui_layer::mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read a key that set the ratio of the level shot picture.
 * \param key The value of the pressed key.
 */
bool ptb::gui_layer::read_levelshot_key( bear::input::keyboard::key_code key )
{
  bool result = true;

  m_levelshot_ratio *= 10;
  ++m_levelshot_key_count;

  switch ( key )
    {
    case bear::input::keyboard::kc_0:
    case bear::input::keyboard::kc_keypad_0:
      break;
    case bear::input::keyboard::kc_1:
    case bear::input::keyboard::kc_keypad_1:
      m_levelshot_ratio += 1;
      break;
    case bear::input::keyboard::kc_2:
    case bear::input::keyboard::kc_keypad_2:
      m_levelshot_ratio += 2;
      break;
    case bear::input::keyboard::kc_3:
    case bear::input::keyboard::kc_keypad_3:
      m_levelshot_ratio += 3;
      break;
    case bear::input::keyboard::kc_4:
    case bear::input::keyboard::kc_keypad_4:
      m_levelshot_ratio += 4;
      break;
    case bear::input::keyboard::kc_5:
    case bear::input::keyboard::kc_keypad_5:
      m_levelshot_ratio += 5;
      break;
    case bear::input::keyboard::kc_6:
    case bear::input::keyboard::kc_keypad_6:
      m_levelshot_ratio += 6;
      break;
    case bear::input::keyboard::kc_7:
    case bear::input::keyboard::kc_keypad_7:
      m_levelshot_ratio += 7;
      break;
    case bear::input::keyboard::kc_8:
    case bear::input::keyboard::kc_keypad_8:
      m_levelshot_ratio += 8;
      break;
    case bear::input::keyboard::kc_9:
    case bear::input::keyboard::kc_keypad_9:
      m_levelshot_ratio += 9;
      break;
    default:
      m_levelshot_key_count = 0;
      m_levelshot_ratio = 0;
      result = false;
    }

  if ( m_levelshot_key_count == 3 )
    {
      if ( m_levelshot_ratio == 0 )
        m_levelshot_ratio = 100;

      levelshot();

      m_levelshot_key_count = 0;
      m_levelshot_ratio = 0;
    }

  return result;
} // gui_layer::read_levelshot_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Take a picture of the screen.
 */
void ptb::gui_layer::screenshot() const
{
  try
    {
      bear::time_ref::time_reference current_time;
      current_time.set();

      std::ostringstream name;
      name << current_time << ".png";

      claw::graphic::image img;
      bear::engine::game::get_instance().screenshot(img);

      const std::string image_path
        ( bear::engine::game::get_instance().get_custom_game_file(name.str()) );
      std::ofstream f( image_path.c_str() );

      claw::graphic::png::writer writer(img, f);
    }
  catch(std::exception& e)
    {
      claw::logger << claw::log_error << "Screenshot: " << e.what()
                   << claw::lendl;
    }
} // gui_layer::screenshot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Take a picture of the level.
 */
void ptb::gui_layer::levelshot() const
{
  try
    {
      bear::time_ref::time_reference current_time;
      current_time.set();

      std::ostringstream name;
      name << "levelshot-" << current_time << ".png";

      claw::graphic::image img;
      bear::engine::game::get_instance().levelshot(img, m_levelshot_ratio);

      const std::string image_path
        ( bear::engine::game::get_instance().get_custom_game_file(name.str()) );
      std::ofstream f( image_path.c_str() );

      claw::graphic::png::writer writer(img, f);
    }
  catch(std::exception& e)
    {
      claw::logger << claw::log_error << "Screenshot: " << e.what()
                   << claw::lendl;
    }
} // gui_layer::screenshot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the fps text component.
 * \param screen The screen on which we draw.
 */
void ptb::gui_layer::render_fps( bear::visual::screen& screen )
{
  if ( m_show_fps )
    {
      bear::time_ref::time_reference current_time;
      current_time.set();

      if ( current_time - m_last_fps_check >= 1000 )
        {
          std::ostringstream oss;
          oss << m_fps_count << " fps - " << m_its_count << " its";

          m_fps_text.set_text( oss.str() );
          m_fps_count = 0;
          m_its_count = 0;
          m_last_fps_check.set();
        }

      m_fps_text.render( screen );
    }
} // gui_layer::render_fps()
