/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file state_wasp_attack.cpp
 * \brief Implementation of the ptb::state_wasp_attack class.
 * \author Sebastien Angibaud
 */

#include "ptb/item/wasp/state_wasp_attack.hpp"
#include "ptb/item/wasp/sting.hpp"
#include "universe/forced_translation.hpp"
#include "engine/game.hpp"
#include "engine/world.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param wasp_instance The pointer on the wasp.
 */
ptb::state_wasp_attack::state_wasp_attack( wasp* wasp_instance )
  : state_wasp(wasp_instance),  m_nb_iterations(0), m_index_sting(0)
{

} // state_wasp_attack::state_wasp_attack()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the name of the state.
 */
std::string ptb::state_wasp_attack::get_name() const
{
  return "attack";
} // state_wasp_attack::get_name()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::state_wasp_attack::progress( bear::universe::time_type elapsed_time )
{
  m_nb_iterations++;

  if( ( m_nb_iterations %
      ( wasp::s_offensive_time / wasp::s_nb_sting ) ) == wasp::s_first_attack )
    {
      m_index_sting++;
      create_sting();
    }

  if( m_nb_iterations == wasp::s_offensive_time )
      m_wasp_instance->set_state(wasp::fly_state);
} // state_wasp_attack::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialization of this state.
 */
void ptb::state_wasp_attack::start()
{
  m_wasp_instance->start_action_parent("attack");

  m_nb_iterations = 0;
  m_index_sting = 0;
} // state_wasp_attack::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Create a sting.
 */
void ptb::state_wasp_attack::create_sting() const
{
  bear::engine::world* w = m_wasp_instance->get_owner();
  sting* new_sting = new sting;

  bear::universe::speed_type speed;

  if ( m_index_sting == 1  ) 
    speed.x = 1;
  else if ( m_index_sting == 2  ) 
    speed.x = 5;
  else 
    speed.x = 14;

  if ( ! m_wasp_instance->get_positive_orientation() )
    speed.x = -speed.x;

  speed.y = (int)sqrt(300 - speed.x*speed.x);
  bear::universe::forced_movement* movement =
    new bear::universe::forced_translation(*new_sting, speed);
  new_sting->set_forced_movement( *movement );

  bear::universe::position_type pos = sting_position();
  new_sting->set_real_field("pos_x", pos.x - new_sting->get_width()/2 );
  new_sting->set_real_field("pos_y", pos.y - new_sting->get_height()/2 );
  new_sting->set_integer_field( "pos_z", 
				m_wasp_instance->get_z_position()-1 );

  double angle = orientate_sting(speed);
  new_sting->set_angle(angle);
  CLAW_ASSERT(new_sting->is_valid(), "The sting isn't correctly initialized" );

  w->register_item( new_sting );

  new_sting->start();

  claw::math::coordinate_2d<int> pos_sound;
  pos_sound.x = (int)m_wasp_instance->get_position().x;
  pos_sound.y = (int)m_wasp_instance->get_position().y;

  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  glob.get_sound("sound/sting.ogg").play( pos_sound );
} // state_wasp_attack::create_sting()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the sting position.
 */
bear::universe::position_type ptb::state_wasp_attack::sting_position() const
{
  bear::universe::position_type pos(m_wasp_instance->get_position());
  
  if ( m_wasp_instance->get_positive_orientation() ) 
     {
      if ( m_index_sting == 1 ) 
	{
	  pos.x += wasp::s_gap_width_1;
	  pos.y += wasp::s_gap_height_1;
	}
      else if ( m_index_sting == 2 ) 
	{
	  pos.x += wasp::s_gap_width_2;
	  pos.y += wasp::s_gap_height_2;
	}
      else
	{
	  pos.x += wasp::s_gap_width_3;
	  pos.y += wasp::s_gap_height_3;
	}
    }	
  else
    {
      if ( m_index_sting == 1 ) 
	{
	  pos.x += m_wasp_instance->get_width() - wasp::s_gap_width_1;
	  pos.y += wasp::s_gap_height_1;
	}
      else if ( m_index_sting == 2 ) 
	{
	  pos.x += m_wasp_instance->get_width() - wasp::s_gap_width_2;
	  pos.y += wasp::s_gap_height_2;
	}
      else
	{
	  pos.x += m_wasp_instance->get_width() - wasp::s_gap_width_3;
	  pos.y += wasp::s_gap_height_3;
	}
    }

  return pos;
} // state_wasp_attack::sting_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the angle of the sting.
 * \param speed Speed of the sting.
 */
double ptb::state_wasp_attack::orientate_sting
( bear::universe::speed_type speed ) const
{
  double angle = 0;
  speed.normalize();

  if ( speed.y >= 0 )
    angle = - acos(speed.x);
  else
    angle = acos(speed.x);

  return angle;
} // state_wasp_attack::orientate_sting()
