/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file state_start_throw.cpp
 * \brief Implementation of the ptb::state_start_throw class.
 * \author Sebastien Angibaud
 */
#include "ptb/item/plee/state_start_throw.hpp"

const bear::universe::time_type
ptb::state_start_throw::s_time_to_start_throw = 0.4;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::state_start_throw::state_start_throw( plee* plee_instance  )
  : state_plee(plee_instance), m_stop_wanted(false),
    m_throw_up(false), m_throw_down(false)
{

} // state_start_throw::state_start_throw()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the name of the state.
 */
std::string ptb::state_start_throw::get_name() const
{
  return "start_throw";
} // state_start_throw::get_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialization of this state.
 */
void ptb::state_start_throw::start()
{
  m_plee_instance->start_action_parent("start_throw");
  m_stop_wanted = false;
  //m_plee_instance->initialise_throw_time();
  m_throw_time = 0;
  m_throw_up = false;
  m_throw_down = false;
} // state_start_throw::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::state_start_throw::progress( bear::universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  m_throw_time += elapsed_time;

  if ( m_stop_wanted || (m_throw_time >= s_time_to_start_throw) )
    {
      m_plee_instance->set_state(plee::throw_state);
      m_plee_instance->create_stone
        ( get_stone_force(m_throw_up, m_throw_down) );
    }
} // state_start_throw::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee want start throw.
 */
void ptb::state_start_throw::do_start_throw()
{
  // Plee can't throw an other stone.
} // state_start_throw::do_start_throw()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee want throw.
 */
void ptb::state_start_throw::do_stop_throw()
{
  m_stop_wanted = true;
} // state_start_throw::do_stop_throw()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee look up.
 */
void ptb::state_start_throw::do_look_up()
{
  m_throw_up = true;
} // state_start_throw::do_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee continue to look up.
 */
void ptb::state_start_throw::do_continue_look_up()
{
  m_throw_up = true;
} // state_start_throw::do_continue_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee crouch.
 */
void ptb::state_start_throw::do_crouch()
{
  m_throw_down = true;
} // state_start_throw::do_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee continue to crouch.
 */
void ptb::state_start_throw::do_continue_crouch()
{
  m_throw_down = true;
} // state_start_throw::do_continue_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a stone.
 *
 * \param up Indicates if Plee throws up.
 * \param down Indicates if Plee throws down.
 */
bear::universe::force_type
ptb::state_start_throw::get_stone_force( bool up, bool down ) const
{
  unsigned int force = 3;
  bear::universe::force_type result;
  
  if ( m_throw_time > ( 2 * ( s_time_to_start_throw / 3 ) ) )
    force = 9;
  else if ( m_throw_time > ( s_time_to_start_throw / 3 ) )
    force = 6;

  if ( up && ! down)
    {
      result.x = 1;
      result.y = -5;
    }
  else if ( ! up && down)
    {
      result.x = 4;
      result.y = 1;
    }
  else
    {
      result.x = 3;
      result.y = -3;
    }
      
  if ( !m_plee_instance->has_positive_orientation() )
    result.x = -result.x;
      
  result.x = result.x * force + m_plee_instance->get_speed().x;
  result.y = result.y * force + m_plee_instance->get_speed().y;
  
  return result;
} // state_start_throw::get_stone_force()
