/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file state_start_jump.cpp
 * \brief Implementation of the ptb::state_start_jump class.
 * \author Sebastien Angibaud
 */
#include "ptb/item/plee/state_start_jump.hpp"

const bear::universe::time_type ptb::state_start_jump::s_time_to_jump = 1;
const bear::universe::size_type ptb::state_start_jump::s_max_halo_height = 64;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::state_start_jump::state_start_jump( plee* plee_instance)
  : state_plee(plee_instance), m_halo_animation(NULL)
{
  m_stop_wanted = false;
} // state_start_jump::state_start_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::state_start_jump::~state_start_jump()
{
  if ( m_halo_animation ) 
    delete m_halo_animation;
} // state_start_jump::state_start_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the name of the state.
 */
std::string ptb::state_start_jump::get_name() const
{
  return "start_jump";
} // state_start_jump::get_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialization of this state.
 */
void ptb::state_start_jump::start()
{
  m_plee_instance->start_action_parent("start_jump");
  m_stop_wanted = false;
  m_jump_time = 0;
  m_halo_animation->reset();
} // state_start_jump::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::state_start_jump::progress( bear::universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  m_halo_animation->next();

  if ( m_plee_instance->has_bottom_contact() )
    {
      if ( !test_walk() ) 
        {
          m_jump_time += elapsed_time;
          
          if ( m_stop_wanted || ( m_jump_time >= s_time_to_jump ) )
            {
              if ( m_jump_time >= s_time_to_jump )
                m_plee_instance->set_jump_time_ratio(1);
              else
                m_plee_instance->set_jump_time_ratio
                  (m_jump_time / s_time_to_jump);

              m_plee_instance->set_state(plee::vertical_jump_state);
            }
        }	
    }
  else
    test_in_sky();
} // state_start_jump::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void ptb::state_start_jump::get_visual
( std::list<bear::engine::scene_visual>& visuals ) const
{
  bear::universe::size_type height(s_max_halo_height * 
			     (m_jump_time / s_time_to_jump) );

  bear::visual::sprite current_sprite(m_halo_animation->get_sprite());
  current_sprite.set_height((unsigned int)height);

  visuals.push_front
    ( bear::engine::scene_visual
      ( m_plee_instance->get_bottom_middle().x - current_sprite.width() / 2,
	m_plee_instance->get_bottom() - current_sprite.height(),
	current_sprite,
        m_plee_instance->get_angle(), m_plee_instance->get_z_position()+1 ) );
} // ptb::state_start_jump::get_visual()

/*---------------------------------------------------------------------------*/
/**
 * \brief Progress the spot.
 */
void ptb::state_start_jump::progress_spot()
{
  claw::math::coordinate_2d<int> gap(0,-5);
  m_plee_instance->add_spot_gap(gap);
} // state_start_jump::progress_spot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee want an vertical jump.
 */
void ptb::state_start_jump::do_stop_vertical_jump()
{
  if ( m_plee_instance->has_bottom_contact() )
    m_stop_wanted = true;
} // state_start_jump::do_stop_vertical_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the plee to the left.
 */
void ptb::state_start_jump::do_move_left()
{
  // Plee can't move on the left in this state
} // state_start_jump::do_move_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the plee to the left.
 */
void ptb::state_start_jump::do_move_right()
{
  // Plee can't move on the right in this state
} // state_start_jump::do_move_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop an vertical jump.
 */
void ptb::state_start_jump::do_stop_look_up()
{
  m_plee_instance->set_state(plee::idle_state);
} // state_start_jump::do_stop_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set an animation.
 */
void ptb::state_start_jump::set_animation
( const std::string& name, bear::visual::animation* value)
{
  if ( name == "halo" )
    m_halo_animation = value;
  else
    super::set_animation(name,value);
} // state_start_jump::set_animation()
