/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_language.hpp
 * \brief This frame is the language configuration menu.
 * \author Julien Jorge
 */
#ifndef __PTB_FRAME_LANGUAGE_HPP__
#define __PTB_FRAME_LANGUAGE_HPP__

#include "ptb/frame/frame.hpp"
#include "ptb/frame/button.hpp"
#include "ptb/frame/message_box.hpp"
#include "ptb/frame/text_box.hpp"
#include "gui/picture.hpp"
#include "gui/static_text.hpp"

#include <vector>
#include <map>
#include <string>

namespace ptb
{
  /**
   * \brief This frame is the language configuration menu.
   * \author Julien Jorge
   */
  class frame_language:
    public frame
  {
  private:
    typedef button< bear::gui::picture, std::mem_fun_t<bool, frame_language> >
    button_with_picture;

    typedef button< bear::gui::static_text,
                    std::mem_fun_t<bool, frame_language> > button_with_text;

  public:
    frame_language( windows_layer* in_layer );

  private:
    bool on_key_press( bear::input::keyboard::key_code key );
    bool on_button_press
    ( bear::input::joystick::joy_code button, unsigned int joy_index );
    bool on_mouse_press( bear::input::mouse::mouse_code key,
                         const claw::math::coordinate_2d<unsigned int>& pos );
    bool on_mouse_move( const claw::math::coordinate_2d<unsigned int>& pos );

    void on_focus();

    bool on_ok();
    bool on_cancel();

    bool on_up();
    bool on_down();

    void select_default();
    void position_cursor();
    void show_selection();
    void show_flag();

    void set_language_file( const std::string& language_file ) const;
    void read_available_languages();

    void create_controls();
    void create_language_text( bear::gui::static_text::font_type f );
    unsigned int create_arrows();
    void create_flag( unsigned int x );
    void create_ok_cancel( bear::gui::static_text::font_type f );

  private:
    /** \brief All available languages. */
    std::vector<std::string> m_language;

    /** \brief For each language, the path to the language file. */
    std::map<std::string, std::string> m_language_path;

    /** \brief The cursor to select 'ok' or 'cancel'. */
    bear::gui::picture* m_cursor;

    /** \brief True is the cursor is on 'ok' (otherwise it is on 'cancel'). */
    bool m_cursor_on_ok;

    /** \brief The flag of the selected language. */
    bear::gui::picture* m_flag;

    /** \brief The text control that displays the selected language. */
    text_box* m_selected;

    /** \brief The text control that displays the language preceding the
        selected one. */
    button_with_text* m_previous;

    /** \brief The text control that displays the language after the selected
        one. */
    button_with_text* m_next;

    /** \brief The index of the selected language. */
    unsigned int m_selected_index;

    /** \brief 'Ok' button. */
    button_with_text* m_ok;

    /** \brief 'Cancel' button. */
    button_with_text* m_cancel;

    /** \brief The language file configured when the frame is created. */
    std::string m_saved_language_file;

    /** \brief Result of the messages box. */
    message_box::flags m_message_box_result;

    /** \brief The path to the file that contains the list of language files. */
    const std::string m_path_to_language_list;

    /** \brief The margin between the controls. */
    const unsigned int m_margin;

    /** \brief The prefix (folder path) of the flag name. */
    static const std::string s_flag_name_prefix;

    /** \brief The suffix appened to the language name to get the flag name. */
    static const std::string s_flag_name_suffix;

    /** \brief The name of the image of the default flag. */
    static const std::string s_default_flag_name;

  }; // class frame_language
} // namespace ptb

#endif // __PTB_FRAME_LANGUAGE_HPP__
