/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_pause.cpp
 * \brief Implementation of the ptb::frame_screen class.
 * \author Sebastien Angibaud
 */
#include "ptb/frame/frame_screen.hpp"

#include "engine/game.hpp"
#include "engine/string_base.hpp"
#include "ptb/config_file.hpp"

/*----------------------------------------------------------------------------*/
const unsigned int ptb::frame_screen::s_pos_full_screen = 0;
const unsigned int ptb::frame_screen::s_pos_ok = 1;
const unsigned int ptb::frame_screen::s_pos_cancel = 2;
const unsigned int ptb::frame_screen::s_margin = 10;
const unsigned int ptb::frame_screen::s_line_space = 6;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer onwing the window.
 */
ptb::frame_screen::frame_screen( windows_layer* owning_layer )
  : frame(owning_layer),
    m_menu(new menu(&get_frame(), s_margin, s_margin, s_line_space)),
    m_saved_mode(bear::engine::game::get_instance().get_fullscreen())
{
  create_controls();
} // frame_screen::frame_screen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool ptb::frame_screen::on_key_press( bear::input::keyboard::key_code key )
{
  bool result = true;

  switch( key )
    {
    case bear::input::keyboard::kc_new_line:
    case bear::input::keyboard::kc_keypad_enter:
      validate();
      break;
    default:
      result = false;
    }

  return result;
} // frame_screen::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::frame_screen::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  switch( button )
    {
    case bear::input::joystick::jc_button_1:
    case bear::input::joystick::jc_button_2:
    case bear::input::joystick::jc_button_3:
    case bear::input::joystick::jc_button_4:
    case bear::input::joystick::jc_button_5:
    case bear::input::joystick::jc_button_6:
    case bear::input::joystick::jc_button_7:
    case bear::input::joystick::jc_button_8:
    case bear::input::joystick::jc_button_9:
    case bear::input::joystick::jc_button_10:
    case bear::input::joystick::jc_button_11:
    case bear::input::joystick::jc_button_12:
    case bear::input::joystick::jc_button_13:
    case bear::input::joystick::jc_button_14:
    case bear::input::joystick::jc_button_15:
    case bear::input::joystick::jc_button_16:
      validate();
      break;
    default:
      result = false;
    }

  return result;
} // frame_screen::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button has been pressed.
 * \param key The code of the key.
 * \param pos The current position of the cursor.
 */
bool ptb::frame_screen::on_mouse_press
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;

  switch( key )
    {
    case bear::input::mouse::mc_left_button:
    case bear::input::mouse::mc_middle_button:
    case bear::input::mouse::mc_right_button:
      validate();
      break;
    default:
      result = false;
    }

  return result;
} // frame_screen::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Validate the selected action.
 */
void ptb::frame_screen::validate()
{
  switch ( m_menu->cursor_position() )
    {
    case s_pos_full_screen:
      bear::engine::game::get_instance().toggle_fullscreen();
      write_mode();
      break;
    case s_pos_ok:
      save();
      close_window();
      break;
    case s_pos_cancel:
      if ( m_saved_mode != bear::engine::game::get_instance().get_fullscreen() )
        bear::engine::game::get_instance().set_fullscreen( m_saved_mode );
      close_window();
      break;
    }
} // frame_screen::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the static text components.
 */
void ptb::frame_screen::create_controls()
{
  char* strings[] = { "fullscreen", "ok", "cancel", NULL };
  m_menu->make( strings );

  m_text = new text_box(this);
  m_text->set_position( m_menu->right(), s_margin );
  m_text->set_visible( true );
  m_text->set_size(50,20);

  write_mode();

  set_size( m_text->right() + s_margin, m_menu->get_size().y );
} // frame_screen::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write the full screen mode.
 */
void ptb::frame_screen::write_mode()
{
  std::string text;

  if ( bear::engine::game::get_instance().get_fullscreen() )
    text = "on";
  else
    text = "off";

  bear::engine::string_base::get_instance().get_string(text);
  m_text->set_text(text);
} // frame_screen::write_mode()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the screen configuration.
 */
void ptb::frame_screen::save()
{
  config_file config;
  config.set_fullscreen( bear::engine::game::get_instance().get_fullscreen() );
  config.save();
} // frame_screen::save()
