/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_player_controls.cpp
 * \brief Implementation of the ptb::frame_player_controls class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame_player_controls.hpp"

#include <sstream>

#include "engine/game.hpp"
#include "input/joystick_button.hpp"
#include "ptb/player_action.hpp"

/*----------------------------------------------------------------------------*/
bear::engine::player::action ptb::frame_player_controls::s_actions[] =
  { ptb::player_action::move_left, ptb::player_action::move_right,
    ptb::player_action::look_up, ptb::player_action::crouch,
    ptb::player_action::slap, ptb::player_action::throw_stone,
    ptb::player_action::drop, ptb::player_action::jump,
    ptb::player_action::speak, ptb::player_action::get_camera };

const unsigned int ptb::frame_player_controls::s_pos_back = 10;
const unsigned int ptb::frame_player_controls::s_pos_save = 11;

const unsigned int ptb::frame_player_controls::s_action_count = 10;

const unsigned int ptb::frame_player_controls::s_text_box_width = 200;
const unsigned int ptb::frame_player_controls::s_margin = 10;
const unsigned int ptb::frame_player_controls::s_line_space = 6;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer onwing the window.
 * \param player_index The index of the player for which we change the controls.
 */
ptb::frame_player_controls::frame_player_controls
( windows_layer* owning_layer, unsigned int player_index )
  : frame(owning_layer), 
    m_menu(new menu(&get_frame(), s_margin, s_margin, s_line_space)),
    m_key_codes(s_action_count), m_joy_codes(s_action_count),
    m_mouse_codes(s_action_count), m_player_index(player_index),
    m_edit_mode(false)
{
  // HACK: I don't know why but the compilator (g++ 4.0.3) output a link error
  // when using bear::input::keyboard::kc_not_a_key,
  // input::joystick::jc_invalid or bear::input::mouse::mc_invalid here.
  //
  // So I replace these constants with their values.
  // -- Julien 07.06.18

  std::fill( m_key_codes.begin(), m_key_codes.end(),
             312 /*bear::input::keyboard::kc_not_a_key*/ );
  std::fill( m_joy_codes.begin(), m_joy_codes.end(),
             joystick_pair( 0, 24 /*bear::input::joystick::jc_invalid*/ ) );
  std::fill( m_mouse_codes.begin(), m_mouse_codes.end(),
             15 /*bear::input::mouse::mc_invalid*/ );

  m_controller_layout =
    bear::engine::game::get_instance().get_rules().get_layout(player_index);

  set_input_priority( true );
  create_controls();
  show_key_values();
} // frame_player_controls::frame_player_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool
ptb::frame_player_controls::on_key_press( bear::input::keyboard::key_code key )
{
  bool result = true;

  if ( key == bear::input::keyboard::kc_escape )
    {
      if ( m_edit_mode )
        edit_mode_off();
      else
        result = false;
    }
  else if ( m_edit_mode )
    {
      set_key(key);
      edit_mode_off();
    }
  else
    switch( key )
      {
      case bear::input::keyboard::kc_new_line :
      case bear::input::keyboard::kc_keypad_enter :
        validate();
        break;
      default:
        result = false;
      }

  return result;
} // frame_player_controls::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::frame_player_controls::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  if ( m_edit_mode )
    {
      set_button( button, joy_index );
      edit_mode_off();
    }
  else if ( button >= bear::input::joystick::jc_button_1 )
    validate();
  else
    result = false;

  return result;
} // frame_player_controls::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button has been pressed.
 * \param key The code of the key.
 * \param pos The current position of the cursor.
 */
bool ptb::frame_player_controls::on_mouse_press
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;

  if ( m_edit_mode )
    {
      set_button( key );
      edit_mode_off();
    }
  else switch( key )
    {
    case bear::input::mouse::mc_left_button :
    case bear::input::mouse::mc_middle_button :
    case bear::input::mouse::mc_right_button :
      validate();
      break;
    default:
      result = false;
    }

  return result;
} // frame_player_controls::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Validate the selected action.
 */
void ptb::frame_player_controls::validate()
{
  switch ( m_menu->cursor_position() )
    {
    case s_pos_back : close_window(); break;
    case s_pos_save :
      save();
      close_window();
      break;
    default:
      edit_mode_on();
    }
} // frame_player_controls::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the static text components.
 */
void ptb::frame_player_controls::create_controls()
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect( 0, 40, 32, 24 );

  m_focus_box = new bear::gui::frame( this, NULL, NULL, NULL,
                                new bear::visual::sprite( img, rect ) );
  m_focus_box->set_visible( false );
  m_focus_box->set_size( s_text_box_width, 0 );

  char* strings[] = { "move_left", "move_right", "look_up", "crouch", "slap",
                      "throw", "drop", "jump", "speak", "get_camera", "back",
                      "save", NULL };

  m_menu->make( strings );

  m_focus_box->set_size( s_text_box_width, m_menu->item(0).get_size().y );

  for ( unsigned int i=0; i!=s_action_count; ++i )
    {
      text_box* text = new text_box(this);
      text->set_position( m_menu->right(),
                          m_menu->item(i).get_position().y );
      text->set_size( m_focus_box->get_size() );

      m_keys.push_back( text );
    }

  set_size( m_keys[0]->right() + s_margin, m_menu->get_size().y );

} // frame_player_controls::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the human readable values of the keys and store the actions in
 *        m_action_value.
 */
void ptb::frame_player_controls::show_key_values()
{
  for ( unsigned int i=0; i!=s_action_count; ++i )
    {
      m_key_codes[i] = bear::input::keyboard::kc_not_a_key;
      m_joy_codes[i].second = bear::input::joystick::jc_invalid;
      m_mouse_codes[i] = bear::input::mouse::mc_invalid;

      bear::input::keyboard::key_code key =
        m_controller_layout.find_key( s_actions[i] );
      bear::input::joystick_button joy =
        m_controller_layout.find_joystick_button( s_actions[i] );
      bear::input::mouse::mouse_code mouse =
        m_controller_layout.find_mouse( s_actions[i] );

      if ( key != bear::input::keyboard::kc_not_a_key )
        {
          m_keys[i]->set_text( bear::input::keyboard::get_name_of(key) );
          m_key_codes[i] = key;
        }
      else if ( joy.button != bear::input::joystick::jc_invalid )
        {
          m_keys[i]->set_text
            ( bear::engine::controller_layout::build_joystick_button_name
              (joy.button, joy.joystick_index) );
          m_joy_codes[i].first = joy.joystick_index;
          m_joy_codes[i].second = joy.button;
        }
      else if ( mouse != bear::input::mouse::mc_invalid )
        {
          m_keys[i]->set_text( bear::input::mouse::get_name_of(mouse) );
          m_key_codes[i] = mouse;
        }
    }
} // frame_player_controls::show_key_values()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop editing the selected action.
 * \pre m_edit_mode == true
 */
void ptb::frame_player_controls::edit_mode_off()
{
  CLAW_PRECOND( m_edit_mode );

  m_edit_mode = false;
  m_focus_box->set_visible( false );
  m_keys[ m_menu->cursor_position() ]->set_visible( true );
} // frame_player_controls::edit_mode_off()

/*----------------------------------------------------------------------------*/
/**
 * \brief Edit the selected action.
 * \pre (m_menu->cursor_position() < m_keys.size()) && !m_edit_mode
 */
void ptb::frame_player_controls::edit_mode_on()
{
  CLAW_PRECOND( m_menu->cursor_position() < m_keys.size() );
  CLAW_PRECOND( !m_edit_mode );

  m_edit_mode = true;
  m_focus_box->set_visible( true );
  m_focus_box->set_position
    ( m_keys[ m_menu->cursor_position() ]->get_position() );
  m_keys[ m_menu->cursor_position() ]->set_visible( false );
} // frame_player_controls::edit_mode_on()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a key for the currently edited action.
 * \param key The value of the key.
 * \pre (m_edit_mode == true) && (m_menu->cursor_position() < m_keys.size())
 */
void ptb::frame_player_controls::set_key( bear::input::keyboard::key_code key )
{
  CLAW_PRECOND( m_edit_mode );
  CLAW_PRECOND( m_menu->cursor_position() < m_keys.size() );

  for (unsigned int i=0; i!=m_key_codes.size(); ++i)
    if ( m_key_codes[i] == key )
      {
        m_key_codes[i] = bear::input::keyboard::kc_not_a_key;
        m_keys[i]->set_text("");
      }

  m_keys[ m_menu->cursor_position() ]->set_text
    ( bear::input::keyboard::get_name_of(key) );

  m_controller_layout.remove_key( m_key_codes[ m_menu->cursor_position() ] );
  m_key_codes[ m_menu->cursor_position() ] = key;
  m_controller_layout(key) = s_actions[ m_menu->cursor_position() ];
} // frame_player_controls::set_key()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a button for the currently edited action.
 * \param button The value of the button.
 * \param joy_index The index of the joytick.
 * \pre (m_edit_mode == true) && (m_menu->cursor_position() < m_keys.size())
 */
void ptb::frame_player_controls::set_button
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  CLAW_PRECOND( m_edit_mode );
  CLAW_PRECOND( m_menu->cursor_position() < m_keys.size() );

  for (unsigned int i=0; i!=m_joy_codes.size(); ++i)
    if ( (m_joy_codes[i].first == joy_index)
         && (m_joy_codes[i].second == button) )
      {
        m_joy_codes[i].second = bear::input::joystick::jc_invalid;
        m_keys[i]->set_text("");
      }

  m_keys[ m_menu->cursor_position() ]->set_text
    ( bear::engine::controller_layout::build_joystick_button_name
      (button, joy_index) );

  m_controller_layout.remove_joy
    ( m_joy_codes[ m_menu->cursor_position() ].first,
      m_joy_codes[ m_menu->cursor_position() ].second);

  m_joy_codes[ m_menu->cursor_position() ].first = joy_index;
  m_joy_codes[ m_menu->cursor_position() ].second = button;

  m_controller_layout(joy_index, button) = s_actions[m_menu->cursor_position()];
} // frame_player_controls::set_button()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a mouse button for the currently edited action.
 * \param button The value of the button.
 * \pre (m_edit_mode == true) && (m_menu->cursor_position() < m_keys.size())
 */
void
ptb::frame_player_controls::set_button( bear::input::mouse::mouse_code button )
{
  CLAW_PRECOND( m_edit_mode );
  CLAW_PRECOND( m_menu->cursor_position() < m_keys.size() );

  for (unsigned int i=0; i!=m_mouse_codes.size(); ++i)
    if ( m_mouse_codes[i] == button )
      {
        m_mouse_codes[i] = bear::input::mouse::mc_invalid;
        m_keys[i]->set_text("");
      }

  m_keys[ m_menu->cursor_position() ]->set_text
    ( bear::input::mouse::get_name_of(button) );

  m_controller_layout.remove_mouse(m_mouse_codes[ m_menu->cursor_position() ]);
  m_mouse_codes[ m_menu->cursor_position() ] = button;
  m_controller_layout(button) = s_actions[ m_menu->cursor_position() ];
} // frame_player_controls::set_button()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the controller_layout.
 */
void ptb::frame_player_controls::save() const
{
  bear::engine::game::get_instance().get_rules().set_layout
    ( m_player_index, m_controller_layout );
  bear::engine::game::get_instance().get_rules().save_controller_layout
    ( m_player_index );
} // frame_player_controls::save()
