/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_language.cpp
 * \brief Implementation of the ptb::frame_language class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame_language.hpp"

#include <claw/string_algorithm.hpp>
#include "ptb/config_file.hpp"
#include "engine/font_factory.hpp"
#include "engine/game.hpp"
#include "engine/resource_pool.hpp"
#include "engine/string_base.hpp"
#include "text/text_metric.hpp"

/*----------------------------------------------------------------------------*/
const std::string ptb::frame_language::s_flag_name_prefix( "gfx/flag/" );
const std::string ptb::frame_language::s_flag_name_suffix( ".png" );
const std::string ptb::frame_language::s_default_flag_name
( s_flag_name_prefix + "default" + s_flag_name_suffix );

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param in_layer The layer containing this frame.
 */
ptb::frame_language::frame_language( windows_layer* in_layer )
  : frame(in_layer), m_cursor_on_ok(true), m_selected_index(0),
    m_message_box_result(0), m_path_to_language_list("lang/list.txt"),
    m_margin(10)
{
  read_available_languages();
  create_controls();

  const config_file config;
  m_saved_language_file = config.get_language_file();

  select_default();
} // frame_language::frame_language()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool ptb::frame_language::on_key_press( bear::input::keyboard::key_code key )
{
  bool result = true;

  switch( key )
    {
    case bear::input::keyboard::kc_new_line:
    case bear::input::keyboard::kc_keypad_enter:
      if ( m_cursor_on_ok )
        result = on_ok();
      else
        result = on_cancel();
      break;
    case bear::input::keyboard::kc_left:
    case bear::input::keyboard::kc_right:
      m_cursor_on_ok = !m_cursor_on_ok;
      position_cursor();
      break;
    case bear::input::keyboard::kc_up:
      result = on_up();
      break;
    case bear::input::keyboard::kc_down:
      result = on_down();
      break;
    default:
      result = false;
    }

  return result;
} // frame_language::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::frame_language::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  switch( button )
    {
    case bear::input::joystick::jc_axis_left:
    case bear::input::joystick::jc_axis_right:
      m_cursor_on_ok = !m_cursor_on_ok;
      position_cursor();
      break;
    case bear::input::joystick::jc_axis_up:
      on_up();
      break;
    case bear::input::joystick::jc_axis_down:
      on_down();
      break;
    case bear::input::joystick::jc_button_1:
    case bear::input::joystick::jc_button_2:
    case bear::input::joystick::jc_button_3:
    case bear::input::joystick::jc_button_4:
    case bear::input::joystick::jc_button_5:
    case bear::input::joystick::jc_button_6:
    case bear::input::joystick::jc_button_7:
    case bear::input::joystick::jc_button_8:
    case bear::input::joystick::jc_button_9:
    case bear::input::joystick::jc_button_10:
    case bear::input::joystick::jc_button_11:
    case bear::input::joystick::jc_button_12:
    case bear::input::joystick::jc_button_13:
    case bear::input::joystick::jc_button_14:
    case bear::input::joystick::jc_button_15:
    case bear::input::joystick::jc_button_16:
      if ( m_cursor_on_ok )
        result = on_ok();
      else
        result = on_cancel();
      break;
    default:
      result = false;
    }

  return result;
} // frame_language::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when a click occurs on the button.
 * \param key The mouse button that was clicked.
 * \param pos The position of the click.
 */
bool ptb::frame_language::on_mouse_press
( bear::input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result;

  switch( key )
    {
    case bear::input::mouse::mc_wheel_up:
      result = on_up();
      break;
    case bear::input::mouse::mc_wheel_down:
      result = on_down();
      break;
    default:
      result = false;
    }

  return result;
} // frame_language::on_mouse_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when a mouse moves over the frame.
 * \param pos The position of the mouse.
 */
bool ptb::frame_language::on_mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = true;

  if ( m_ok->get_rectangle().includes(pos) )
    m_cursor_on_ok = true;
  else if ( m_cancel->get_rectangle().includes(pos) )
    m_cursor_on_ok = false;
  else
    result = false;

  if ( result )
    position_cursor();

  return result;
} // frame_language::on_mouse_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief The frame is focused. If this is after showing the message box, then
 *        check the result.
 */
void ptb::frame_language::on_focus()
{
  if ( m_message_box_result & message_box::s_cancel )
    {
      set_language_file( m_saved_language_file );
      select_default();
    }
  else if ( m_message_box_result & message_box::s_ok )
    {
      config_file config;
      config.set_language_file(m_language_path[ m_language[m_selected_index] ]);
      config.save();

      bear::engine::game::get_instance().set_waiting_level("title_screen");
    }
} // frame_language::on_ok()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the configuration.
 */
bool ptb::frame_language::on_ok()
{
  if ( !m_language.empty() )
    {
      if ( m_saved_language_file
           != m_language_path[ m_language[m_selected_index] ] )
        {
          set_language_file( m_language_path[ m_language[m_selected_index] ] );
          m_message_box_result = message_box::s_ok | message_box::s_cancel;
          show_window
            ( new message_box( m_owning_layer, "msg_language_need_restart",
                               &m_message_box_result ) );
        }
      else
        close_window();
    }
  else
    close_window();

  return true;
} // frame_language::on_ok()

/*----------------------------------------------------------------------------*/
/**
 * \brief Exit without saving.
 */
bool ptb::frame_language::on_cancel()
{
  close_window();
  return true;
} // frame_language::on_cancel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the selection to the previous language.
 */
bool ptb::frame_language::on_up()
{
  if ( !m_language.empty() )
    if ( m_selected_index != 0 )
      {
        --m_selected_index;
        show_selection();
      }

  return true;
} // frame_language::on_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the selection to the next language.
 */
bool ptb::frame_language::on_down()
{
  if ( !m_language.empty() )
    if ( m_selected_index + 1 != m_language.size() )
      {
        ++m_selected_index;
        show_selection();
      }

  return true;
} // frame_language::on_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
void ptb::frame_language::select_default()
{
  bool stop = false;

  for (unsigned int i=0; !stop && (i!=m_language.size()); ++i)
    if ( m_language_path[ m_language[i] ] == m_saved_language_file )
      {
        stop = true;
        m_selected_index = i;
      }

  show_selection();
  position_cursor();
} // frame_language::select_default()

/*----------------------------------------------------------------------------*/
/**
 * \brief Position the cursor near the selected item.
 */
void ptb::frame_language::position_cursor()
{
  unsigned int x;
  unsigned int y;

  if ( m_cursor_on_ok )
    x = m_ok->get_position().x;
  else
    x = m_cancel->get_position().x;

  x -= m_cursor->width() + m_margin;
  y = m_ok->get_position().y + (int)(m_ok->height() - m_cursor->height()) / 2;

  m_cursor->set_position(x, y);
} // frame_language::position_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the selected language.
 */
void ptb::frame_language::show_selection()
{
  if ( !m_language.empty() )
    {
      m_selected->set_text( m_language[m_selected_index] );

      if ( m_selected_index == 0 )
        m_previous->set_text( "" );
      else
        m_previous->set_text( m_language[m_selected_index-1] );

      if ( m_selected_index + 1 == m_language.size() )
        m_next->set_text( "" );
      else
        m_next->set_text( m_language[m_selected_index+1] );

      show_flag();
    }
} // frame_language::show_selection()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the flag of the selected language.
 */
void ptb::frame_language::show_flag()
{
  if ( !m_language.empty() )
    {
      bear::visual::sprite* new_flag = NULL;
      std::string lang_name = m_language_path[ m_language[m_selected_index] ];
      std::string::size_type pos = lang_name.find_last_of('/');

      if ( pos != std::string::npos )
        {
          lang_name = lang_name.substr( pos+1 );
          pos = lang_name.find_first_of('.');

          if ( pos != std::string::npos )
            lang_name = lang_name.substr(0, pos);
        }

      lang_name = s_flag_name_prefix + lang_name + s_flag_name_suffix;

      bear::engine::level_globals& glob =
        bear::engine::game::get_instance().current_level_globals();

      if ( glob.image_exists(lang_name) )
        {
          const bear::visual::image& img =
            glob.get_image(lang_name);
          new_flag = new bear::visual::sprite(img);
        }
      else if ( glob.image_exists(s_default_flag_name) )
        {
          const bear::visual::image& img =
            glob.get_image(s_default_flag_name);
          new_flag = new bear::visual::sprite(img);
        }

      m_flag->set_picture(new_flag);
    }
} // frame_language::show_flag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a language file.
 * \param language_file The path of the language file to load.
 */
void
ptb::frame_language::set_language_file( const std::string& language_file ) const
{
  if ( bear::engine::resource_pool::get_instance().exists(language_file) )
    {
      std::stringstream lang_stream;
      bear::engine::resource_pool::get_instance().get_file
        ( language_file, lang_stream );
      bear::engine::string_base::get_instance().load( lang_stream );
    }
} // frame_language::set_language_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the available languages.
 */
void ptb::frame_language::read_available_languages()
{
  if ( bear::engine::resource_pool::get_instance().exists
       ( m_path_to_language_list ) )
    {
      std::stringstream language_list;
      bear::engine::resource_pool::get_instance().get_file
        ( m_path_to_language_list, language_list );
      std::string line;

      while ( claw::text::getline(language_list, line) )
        if ( !line.empty() )
          {
            std::string::size_type pos = line.find_first_of(':');

            if ( pos != std::string::npos )
              {
                std::string lang_name = line.substr(0, pos);
                std::string lang_path = line.substr(pos+1);

                claw::text::trim( lang_name );
                claw::text::trim( lang_path );

                if ( bear::engine::resource_pool::get_instance().exists
                     (lang_path) )
                  {
                    m_language.push_back( lang_name );
                    m_language_path[lang_name] = lang_path;
                  }
              }
          }
    }
} // frame_language::read_available_languages()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls in this frame.
 */
void ptb::frame_language::create_controls()
{
  m_cursor = new bear::gui::picture( this, frame::get_cursor() );

  bear::gui::static_text::font_type font =
    bear::engine::font_factory::create("font/fixed_yellow-10x20.png");

  create_language_text( font );
  unsigned int r = create_arrows();
  create_flag( r + m_margin );
  create_ok_cancel( font );

  set_size( m_cancel->right() + m_margin, m_cancel->bottom() + m_margin );
} // frame_language::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls that display the languages.
 * \param f The font to use for the texts.
 */
void ptb::frame_language::create_language_text
( bear::gui::static_text::font_type f )
{
  m_previous = new button_with_text
    ( *this, std::mem_fun(&frame_language::on_up), this );
  m_next = new button_with_text
    ( *this, std::mem_fun(&frame_language::on_down), this );
  m_selected = new text_box( this );

  m_previous->set_font( f );
  m_next->set_font( f );

  // set the static text to the maximum required size()
  unsigned int w(0);
  for (unsigned int i=0; i!=m_language.size(); ++i)
    {
      bear::text::text_metric tm( m_language[i], *f );
      if ( tm.width() > w )
        w = tm.width();
    }

  m_previous->set_size( w, f->get_size().y );
  m_next->set_size( w, f->get_size().y );
  m_selected->set_size( w, f->get_size().y );

  m_previous->set_position( m_margin, m_margin );
  m_selected->set_position( m_margin, m_previous->bottom() + m_margin );
  m_next->set_position( m_margin, m_selected->bottom() + m_margin );
} // frame_language::create_language_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the arrow buttons.
 * \return The right position of the created controls.
 */
unsigned int ptb::frame_language::create_arrows()
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img =
    glob.get_image( "gfx/frame.tga" );

  claw::math::rectangle<unsigned int> rect(64, 0, 18, 20);

  bear::visual::sprite* arrow = new bear::visual::sprite( img, rect );

  button_with_picture* btn =
    new button_with_picture(*this, std::mem_fun(&frame_language::on_up), this);

  btn->set_size( arrow->get_size() );
  btn->set_picture( new bear::visual::sprite(*arrow) );
  btn->set_position
    ( m_previous->right() + m_margin, m_previous->get_position().y );

  arrow->flip(true);

  btn = new button_with_picture
    ( *this, std::mem_fun(&frame_language::on_down), this );

  btn->set_size( arrow->get_size() );
  btn->set_picture( new bear::visual::sprite(*arrow) );
  btn->set_position
    ( m_next->right() + m_margin, m_next->bottom() - arrow->height() );

  return btn->right();
}  // frame_language::create_arrows()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the component containing the flag of the selected language.
 * \param x The x-coordinate of the component.
 */
void ptb::frame_language::create_flag( unsigned int x )
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  m_flag = new bear::gui::picture( this );

  if ( glob.image_exists(s_default_flag_name) )
    {
      const bear::visual::image& img =
        glob.get_image( s_default_flag_name );

      m_flag->set_size( img.size() );
      m_flag->set_stretch( true );
      m_flag->set_picture( new bear::visual::sprite(img) );
    }

  m_flag->set_position
    ( x, ( m_previous->get_position().y + m_next->bottom()
           - m_flag->height() ) / 2 );
}  // frame_language::create_flag()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the "ok" and "cancel" buttons.
 * \param f The font to use for the texts.
 */
void
ptb::frame_language::create_ok_cancel( bear::gui::static_text::font_type f )
{
  std::string text("ok");

  m_ok =
    new button_with_text( *this, std::mem_fun(&frame_language::on_ok), this );

  m_ok->set_font( f );
  m_ok->set_auto_size( true );
  bear::engine::string_base::get_instance().get_string(text);
  m_ok->set_text(text);
  m_ok->set_position
    ( m_margin + m_cursor->width() + m_margin, m_next->bottom() + m_margin );

  m_cancel = new button_with_text
    ( *this, std::mem_fun(&frame_language::on_cancel), this );

  m_cancel->set_font( f );
  m_cancel->set_auto_size( true );
  text = "cancel";
  bear::engine::string_base::get_instance().get_string(text);
  m_cancel->set_text(text);

  unsigned int x = m_ok->right() + m_margin + m_cursor->width() + m_margin;
  unsigned int r = m_flag->right();

  if ( x + m_cancel->width() > r )
    r = x + m_cancel->width();

  m_cancel->set_position( r - m_cancel->width(), m_next->bottom() + m_margin );
} // frame_language::create_ok_cancel()
