/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file monster.cpp
 * \brief Implementation of the ptb::monster class.
 * \author S�bastien Angibaud
 */
#include "ptb/monster.hpp"

#include <iostream>
#include <sstream>
#include <string>

/*----------------------------------------------------------------------------*/
const unsigned int ptb::monster::s_attack_count = 4;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::monster::monster()
  : m_offensive_force(0), m_energy(0), m_offensive_phase(true),
    m_defensive_powers(s_attack_count+1, false),
    m_offensive_coefficients(s_attack_count+1, 0), m_invincible(false),
    m_is_injured(false), m_creator(NULL)
{
} // base_monster::base_monster()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::monster::~monster()
{

} // base_monster::~base_monster()

/*----------------------------------------------------------------------------*/
/**
 * \brief The item receive an attack.
 *
 * \param attacker The monster attacking me.
 */
void ptb::monster::receive_an_attack( monster& attacker )
{
  if( (!m_invincible) && (!m_is_injured) )
    if( is_vulnerable( attacker ) )
      {
        unsigned int energy = 0;
        
        for( unsigned int i=0; i!=m_defensive_powers.size(); ++i )
          if ( !m_defensive_powers[i] )
            energy += attacker.m_offensive_coefficients[i];
        
        energy *= attacker.m_offensive_force;
	
	if ( energy > 0 )
	  {
	    injure();
	    remove_energy(energy);
	    attacker.has_attacked();
	  }	
      }
} // monster::receive_an_attack()


/*----------------------------------------------------------------------------*/
/**
 * \brief The item has attacked.
 */
void ptb::monster::has_attacked()
{

} // monster::has_attacked()

/*----------------------------------------------------------------------------*/
/**
 * \brief We remove some energy of the item..
 *
 * \param energy The quantity of energy removed.
 */
void ptb::monster::remove_energy( unsigned int energy )
{
  if ( energy >= m_energy )
    m_energy = 0;
  else
    m_energy -= energy;
} // ptb::monster::remove_energy()


/*----------------------------------------------------------------------------*/
/**
 * \brief The monster is injure.
 */
void ptb::monster::injure()
{
  m_is_injured = true;
  m_injured_time.set();
} // ptb::monster::injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief The monster isn't injure any more.
 */
void ptb::monster::finish_injure()
{
  m_is_injured = false;
} // ptb::monster::finish_injure()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void ptb::monster::to_string( std::string& str ) const
{
  std::ostringstream oss;

  oss << "\noffensive_force: " << m_offensive_force;
  oss << "\nenergy: " << m_energy;
  oss << "\noffensive_phase: ";

  if ( m_offensive_phase )
    oss << "true";
  else
    oss << "false";

  oss << "\ndefensive: ";
  std::vector<bool>::const_iterator it;
  for(it = m_defensive_powers.begin(); it != m_defensive_powers.end(); ++it )
    oss << *it << " ";

  oss << "\noffensive_coef: ";
  std::vector<unsigned int>::const_iterator it2;
  for(it2 = m_offensive_coefficients.begin();
      it2 != m_offensive_coefficients.end(); ++it2 )
    oss << *it2 << " ";

  oss << "\ninvincible: ";
  if ( m_invincible )
    oss << "true";
  else
    oss << "false";

  oss << "\ninjured: ";
  if ( m_is_injured )
    oss << "true";
  else
    oss << "false";

  oss << "\n";

  str += oss.str();
} // monster::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the current item is vulnerable to an other item.
 *
 * \param attacker Type of the attacker.
 */
bool ptb::monster::is_vulnerable( monster& attacker ) const
{
  bool result;

  switch( m_monster_type )
    {
    case nature_monster: result = false; break;

    case player_monster:
      if ( attacker.get_monster_type() != stone_monster )
	result = true;
      else
	result = ( attacker.get_monster_creator() != this );
      break;

    case ennemy_monster: result =
	( attacker.get_monster_type() != ennemy_monster ); break;

    case stone_monster:
      if ( attacker.get_monster_type() != player_monster )
	result = true;
      else
	result = ( get_monster_creator() != &attacker );
      break;
    }

  return result;
} // monster::is_vulnerable()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the invincible statut.
 *
 * \param invincible The new invincible statut.
 */
void ptb::monster::set_invincible(const bool invincible)
{
  m_invincible = invincible;
} // ptb::monster::set_invincible()


/*----------------------------------------------------------------------------*/
/**
 * \brief Set the offensive phase statut.
 *
 * \param invincible The new offensive phase statut.
 */
void ptb::monster::set_offensive_phase(const bool offensive_phase)
{
  m_offensive_phase = offensive_phase;
} // ptb::monster::set_offensive_phase()


/*----------------------------------------------------------------------------*/
/**
 * \brief Set the defensive power statut.
 *
 * \param power Defensive index.
 * \param statut The new offensive phase statut.

 */
void ptb::monster::set_defensive_power(const attack_type index,
				       const bool statut)
{
  m_defensive_powers[index] = statut;
} // ptb::monster::set_defensive_power()


/*----------------------------------------------------------------------------*/
/**
 * \brief Return the energy of the monster.
 *
 */
unsigned int ptb::monster::get_energy() const
{
  return m_energy;
} // monster::get_energy()


/*----------------------------------------------------------------------------*/
/**
 * \brief Return the type of monster.
 *
 */
ptb::monster::monster_type ptb::monster::get_monster_type() const
{
  return m_monster_type;
} // monster::get_monster_type()


/*----------------------------------------------------------------------------*/
/**
 * \brief Return the type of monster.
 *
 */
ptb::monster* ptb::monster::get_monster_creator() const
{
  return m_creator;
} // monster::get_monster_creator()


/*----------------------------------------------------------------------------*/
/**
 * \brief Set of the creator.
 *
 */
void ptb::monster::set_monster_creator( ptb::monster* creator)
{
  m_creator = creator;
} // ptb::monster::set_monster_creator()
