/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \brief Implementation of the ptb::config_file class.
 * \author Julien Jorge
 */
#include "ptb/config_file.hpp"

#include <claw/configuration_file.hpp>
#include <claw/logger.hpp>
#include <sstream>
#include <fstream>

#include "engine/game.hpp"
#include "engine/resource_pool.hpp"
#include "engine/string_base.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::config_file::config_file()
  : m_fullscreen(false), m_sound_on(true), m_music_on(true),
    m_config_name("config")
{
  std::string full_config_path
    ( bear::engine::game::get_instance().get_custom_game_file(m_config_name) );
  std::ifstream f( full_config_path.c_str() );

  if (f)
    {
      claw::configuration_file config(f);

      str_to_bool( m_fullscreen, config("Video", "fullscreen") );
      str_to_bool( m_sound_on, config("Audio", "sound_on") );
      str_to_bool( m_music_on, config("Audio", "music_on") );

      m_language_file = config("i18n", "language_file");
    }
} // config_file::config_file

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the configuration to the game.
 */
void ptb::config_file::apply() const
{
  bear::engine::game::get_instance().set_fullscreen( m_fullscreen );

  if ( bear::engine::resource_pool::get_instance().exists( m_language_file ) )
    {
      std::stringstream language_file;
      bear::engine::resource_pool::get_instance().get_file
	( m_language_file, language_file );

      bear::engine::string_base::get_instance().load( language_file );
    }

  if ( !m_sound_on || !m_music_on )
    claw::logger << claw::log_warning << "Sound muting is not supported yet."
		 << claw::lendl;
} // config_file::apply()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the configuration.
 */
void ptb::config_file::save() const
{
  std::string full_config_path
    ( bear::engine::game::get_instance().get_custom_game_file(m_config_name) );
  std::ofstream f( full_config_path.c_str() );

  f << "# Configuration of the screen.\n"
    << "[Video]\n"
    << "# Do we use the fullscreen ?\n"
    << "fullscreen = " << bool_to_str(m_fullscreen) << '\n'
    << '\n'
    << "# Configuration of the sound system.\n"
    << "[Audio]\n"
    << "# Do we play the sounds ?\n"
    << "sound_on = " << bool_to_str(m_sound_on) << '\n'
    << "# Do we play the music ?\n"
    << "music_on = " << bool_to_str(m_music_on) << '\n'
    << '\n'
    << "# Internationalisation\n"
    << "[i18n]\n"
    << "# The language file to use to display the texts.\n"
    << "language_file = " << m_language_file << '\n'
    << std::endl;
} // config_file::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the fullscreen mode value.
 * \param b The new value.
 */
void ptb::config_file::set_fullscreen( bool b )
{
  m_fullscreen = b;
} // config_file::set_fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the sound mode value.
 * \param b The new value.
 */
void ptb::config_file::set_sound_on( bool b )
{
  m_sound_on = b;
} // config_file::set_sound_on()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the music mode value.
 * \param b The new value.
 */
void ptb::config_file::set_music_on( bool b )
{
  m_music_on = b;
} // config_file::set_music_on()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the path of the language file.
 * \param path The path to the file.
 */
void ptb::config_file::set_language_file( const std::string& path )
{
  m_language_file = path;
} // config_file::set_language_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the path of the language file.
 */
const std::string& ptb::config_file::get_language_file() const
{
  return m_language_file;
} // config_file::get_language_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a boolean value to its string respresentation.
 * \param b The boolean to convert.
 */
std::string ptb::config_file::bool_to_str( bool b ) const
{
  if (b)
    return "true";
  else
    return "false";
} // config_file::bool_to_str()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a boolean value from its string representation.
 * \param b The resulting boolean.
 * \param s The string representation.
 * \remark \a b is unchanged if \ s does not represent a boolean value.
 */
void ptb::config_file::str_to_bool( bool& b, const std::string& s ) const
{
  if ( s == "true" )
    b = true;
  else if ( s == "false" )
    b = false;
} // config_file::str_to_bool()
