/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file symbol_table.hpp
 * \brief User-defined symbols : constants.
 * \author Julien Jorge
 */
#ifndef __MC_SYMBOL_TABLE_HPP__
#define __MC_SYMBOL_TABLE_HPP__

#include <map>
#include <string>
#include "mc/custom_type.hpp"
#include "mc/image_type.hpp"
#include "mc/animation_type.hpp"

namespace mc
{
  /**
   * \brief User-defined symbols : constants.
   * \author Julien Jorge
   */
  class symbol_table
  {
  private:
    /**
     * \brief Generic class to store constants.
     */
    template<typename Type>
    class symbol_map : public std::map<std::string, Type> {};

  public:
    void clear();

    void add_u_integer( const std::string& name,
                        const u_integer_type& value );
    void add_real( const std::string& name, const real_type& value );
    void add_bool( const std::string& name, const bool_type& value );
    void add_string( const std::string& name, const string_type& value );
    void add_image( const std::string& name, const image_type& value );
    void add_animation( const std::string& name,
                        const animation_type& value );

    bool constant_exists( const std::string& name ) const;

    bool u_integer_exists( const std::string& name ) const;
    bool real_exists( const std::string& name ) const;
    bool bool_exists( const std::string& name ) const;
    bool string_exists( const std::string& name ) const;
    bool image_exists( const std::string& name ) const;
    bool animation_exists( const std::string& name ) const;

    const u_integer_type& get_u_integer( const std::string& name ) const;
    const real_type& get_real( const std::string& name ) const;
    const bool_type& get_bool( const std::string& name ) const;
    const string_type& get_string( const std::string& name ) const;
    const image_type& get_image( const std::string& name ) const;
    const animation_type& get_animation( const std::string& name ) const;

  private:
    /** \brief Constants of type unsigned integer. */
    symbol_map<u_integer_type> m_u_integer;

    /** \brief Constants of type real. */
    symbol_map<real_type> m_real;

    /** \brief Constants of type bool. */
    symbol_map<bool_type> m_bool;

    /** \brief Constants of type string. */
    symbol_map<string_type> m_string;

    /** \brief Constants of type image. */
    symbol_map<image_type> m_image;

    /** \brief Constants of type animation. */
    symbol_map<animation_type> m_animation;

  }; // class symbol_table
} // namespace mc

#endif // __MC_SYMBOL_TABLE_HPP__
