/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file node_compiler_frame_list.cpp
 * \brief Implementation of the mc::node_compiler_frame_list class.
 * \author Julien Jorge
 */
#include "mc/node_compiler/node_compiler_frame_list.hpp"

#include <claw/assert.hpp>
#include "mc/model_grammar.hpp"
#include "mc/node_compiler/node_compiler_image_type.hpp"
#include "mc/node_compiler/node_compiler_identifier.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a node of type frame_list.
 * \param frames (out) The list of the frames.
 * \param symbols The symbols of the model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_frame_list::compile_node
( std::list<image_type>& frames, const symbol_table& symbols,
  const model& the_model, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_frame_list );

  for (unsigned int i=0; i!=node.children.size(); ++i)
    {
      image_type frame;
      get_frame( frame, symbols, the_model, node.children[i] );

      frames.push_back(frame);
    }
} // node_compiler_frame_list::compile_node()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read an image.
 * \param frame (out) the frame read.
 * \param symbols The symbols of the model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_frame_list::get_frame
( image_type& frame, const symbol_table& symbols, const model& the_model,
  const tree_node& node ) const
{
  if ( node.value.id() == model_grammar::id_image_type )
    {
      node_compiler_image_type comp;
      comp.compile_node( frame, symbols, the_model, node );
    }
  else // node.value.id() == model_grammar::id_identifier
    {
      node_compiler_identifier comp;
      std::string id;

      comp.compile_node(id, node);

      if ( symbols.image_exists(id) )
        frame = symbols.get_image(id);
      else
        error( node.value.begin(), node.value.end(), "Not an image." );
    }
} // node_compiler_frame_list::get_frame()
