/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file node_compiler_common_image_anim.cpp
 * \brief Implementation of the mc::node_compiler_common_image_anim class.
 * \author Julien Jorge
 */
#include <claw/assert.hpp>

#include "mc/model_grammar.hpp"
#include "mc/node_compiler/node_compiler_identifier.hpp"
#include "mc/node_compiler/node_compiler_u_integer_type.hpp"
#include "mc/node_compiler/node_compiler_bool_type.hpp"
#include "mc/node_compiler/node_compiler_real_type.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the horizontal flip attribute.
 * \param val The current image or animation.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
template<class T>
void mc::node_compiler_common_image_anim<T>::set_flip_x
( T& val, const symbol_table& symbols, const tree_node& node ) const
{
  bool field_val;

  if ( get_value( field_val, symbols, node ) )
    val.set_flip_x(field_val);
  else
    error( node.value.begin(), node.value.end(), "Not a boolean value." );
} // node_compiler_common_image_anim::set_flip_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the vertical flip attribute.
 * \param val The current image or animation.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
template<class T>
void mc::node_compiler_common_image_anim<T>::set_flip_y
( T& val, const symbol_table& symbols, const tree_node& node ) const
{
  bool field_val;

  if ( get_value( field_val, symbols, node ) )
    val.set_flip_y(field_val);
  else
    error( node.value.begin(), node.value.end(), "Not a boolean value." );
} // node_compiler_common_image_anim::set_flip_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the alpha transparency attribute.
 * \param val The current image or animation.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
template<class T>
void mc::node_compiler_common_image_anim<T>::set_alpha
( T& val, const symbol_table& symbols, const tree_node& node ) const
{
  float field_val;

  if ( get_value( field_val, symbols, node ) )
    {
      if ( (field_val >= 0) && (field_val <= 1) )
        val.set_alpha(field_val);
      else
        error( node.value.begin(), node.value.end(),
               "Alpha must be in [0,1]." );
    }
  else
    error( node.value.begin(), node.value.end(), "Not a real value." );
} // node_compiler_common_image_anim::set_alpha()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a unsigned int value.
 * \param val (out) The value read.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
template<class T>
bool mc::node_compiler_common_image_anim<T>::get_value
( unsigned int& val, const symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_any_type );

  bool ok = false;
  const tree_node& child = node.children[0];

  if ( child.value.id() == model_grammar::id_u_integer_type )
    {
      node_compiler_u_integer_type comp;
      u_integer_type v;

      comp.compile_node( v, child );
      val = v.get_value();
      ok = true;
    }
  else if ( child.value.id() == model_grammar::id_real_type )
    {
      node_compiler_real_type comp;
      real_type v;

      comp.compile_node( v, child );

      if ( v.get_value() == (float)(unsigned int)v.get_value() )
        {
          val = (unsigned int)v.get_value();
          ok = true;
        }
    }
  else if ( child.value.id() == model_grammar::id_identifier )
    {
      node_compiler_identifier comp_id;
      std::string id;

      comp_id.compile_node( id, child );

      if ( symbols.u_integer_exists(id) )
        {
          val = symbols.get_u_integer(id).get_value();
          ok = true;
        }
      else if ( symbols.real_exists(id) )
        {
          float v = symbols.get_real(id).get_value();

          if ( v == (float)(unsigned int)v )
            {
              val = (unsigned int)v;
              ok = true;
            }
        }
    }

  return ok;
} // node_compiler_common_image_anim::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a float value.
 * \param val (out) The value read.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
template<class T>
bool mc::node_compiler_common_image_anim<T>::get_value
( float& val, const symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_any_type );

  bool ok = false;
  const tree_node& child = node.children[0];

  if ( child.value.id() == model_grammar::id_real_type )
    {
      node_compiler_real_type comp;
      real_type v;

      comp.compile_node( v, child );
      val = v.get_value();
      ok = true;
    }
  else if ( child.value.id() == model_grammar::id_u_integer_type )
    {
      node_compiler_u_integer_type comp;
      u_integer_type v;

      comp.compile_node( v, child );
      val = v.get_value();
      ok = true;
    }
  else if ( child.value.id() == model_grammar::id_identifier )
    {
      node_compiler_identifier comp_id;
      std::string id;

      comp_id.compile_node( id, child );

      if ( symbols.real_exists(id) )
        {
          val = symbols.get_real(id).get_value();
          ok = true;
        }
      else if ( symbols.u_integer_exists(id) )
        {
          val = symbols.get_u_integer(id).get_value();
          ok = true;
        }
    }

  return ok;
} // node_compiler_common_image_anim::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a boolean value.
 * \param val (out) The value read.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
template<class T>
bool mc::node_compiler_common_image_anim<T>::get_value
( bool& val, const symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_any_type );

  bool ok = false;
  const tree_node& child = node.children[0];

  if ( child.value.id() == model_grammar::id_bool_type )
    {
      node_compiler_bool_type comp;
      bool_type v;

      comp.compile_node( v, child );
      val = v.get_value();
      ok = true;
    }

  return ok;
} // node_compiler_common_image_anim::get_value()

