/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file node_compiler_animation_type.cpp
 * \brief Implementation of the mc::node_compiler_animation_type class.
 * \author Julien Jorge
 */
#include "mc/node_compiler/node_compiler_animation_type.hpp"

#include <claw/assert.hpp>
#include "mc/model_grammar.hpp"
#include "mc/node_compiler/node_compiler_frame_list.hpp"
#include "mc/node_compiler/node_compiler_identifier.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a node of type animation_type.
 * \param val The value read.
 * \param symbols The symbols of the model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_animation_type::compile_node
( animation_type& val, const symbol_table& symbols,  const model& the_model,
  const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_animation_type );
  CLAW_PRECOND( node.children.size() >= 1 );

  node_compiler_frame_list comp_frames;
  std::list< image_type > frames;

  comp_frames.compile_node( frames, symbols, the_model, node.children[0] );

  val.set_frames( frames );

  for ( unsigned int i=1; i!=node.children.size(); ++i )
    compile_body_item( val, symbols, node.children[i] );
} // node_compiler_animation_type::compile_node()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the body of an image.
 * \param val The current image.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
void mc::node_compiler_animation_type::compile_body_item
( animation_type& val, const symbol_table& symbols,
  const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_image_body_item );
  CLAW_PRECOND( node.children.size() == 2 );

  node_compiler_identifier comp_id;
  std::string field;

  comp_id.compile_node( field, node.children[0] );

  if ( field == "flip_x" )
    set_flip_x( val, symbols, node.children[1] );
  else if ( field == "flip_y" )
    set_flip_y( val, symbols, node.children[1] );
  else if ( field == "alpha" )
    set_alpha( val, symbols, node.children[1] );
  else if ( field == "loops" )
    set_loops( val, symbols, node.children[1] );
  else if ( field == "loop_back" )
    set_loop_back( val, symbols, node.children[1] );
  else
    error( node.value.begin(), node.value.end(), "Not a field of image_type." );
} // node_compiler_animation_type::compile_body_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the number of times the animation will be played.
 * \param val The current animation.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
void mc::node_compiler_animation_type::set_loops
( animation_type& val, const symbol_table& symbols,
  const tree_node& node ) const
{
  unsigned int field_val;

  if ( get_value(field_val, symbols, node) )
    val.set_loops(field_val);
  else
    error( node.value.begin(), node.value.end(), "Not a u_integer value." );
} // node_compiler_animation_type::set_loops()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the loop back attribute of the animation.
 * \param val The current animation.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
void mc::node_compiler_animation_type::set_loop_back
( animation_type& val, const symbol_table& symbols,
  const tree_node& node ) const
{
  bool field_val;

  if ( get_value(field_val, symbols, node) )
    val.set_loop_back(field_val);
  else
    error( node.value.begin(), node.value.end(), "Not a boolean value." );
} // node_compiler_animation_type::set_loop_back()

