/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/item_instance.hpp
 * \brief An instance of an item class.
 * \author Julien Jorge
 */
#ifndef __BF_ITEM_INSTANCE_HPP__
#define __BF_ITEM_INSTANCE_HPP__

#include <map>
#include <string>
#include <set>
#include <list>

#include "bf/sprite.hpp"
#include "bf/animation.hpp"
#include "bf/custom_type.hpp"
#include "bf/type_field.hpp"

#include "level_code_value.hpp"

namespace bf
{
  /**
   * \brief An instance of an item class.
   * \author Julien Jorge
   */
  class item_instance
  {
  public:
    item_instance();
    item_instance( const std::string& class_name );

    const std::string& get_class_name() const;

    const std::string& get_id() const;
    void set_id( const std::string& id );

    void set_value( const std::string& name, const integer_type& v );
    void set_value( const std::string& name, const u_integer_type& v );
    void set_value( const std::string& name, const real_type& v );
    void set_value( const std::string& name, const bool_type& v );
    void set_value( const std::string& name, const string_type& v );
    void set_value( const std::string& name, const sprite& v );
    void set_value( const std::string& name, const animation& v );
    void set_value( const std::string& name, const item_reference_type& v );

    void set_value( const std::string& name, const std::list<integer_type>& v );
    void
    set_value( const std::string& name, const std::list<u_integer_type>& v );
    void set_value( const std::string& name, const std::list<real_type>& v );
    void set_value( const std::string& name, const std::list<bool_type>& v );
    void set_value( const std::string& name, const std::list<string_type>& v );
    void set_value( const std::string& name, const std::list<sprite>& v );
    void set_value( const std::string& name, const std::list<animation>& v );
    void set_value
    ( const std::string& name, const std::list<item_reference_type>& v );

    void get_value( const std::string& name, integer_type& v ) const;
    void get_value( const std::string& name, u_integer_type& v ) const;
    void get_value( const std::string& name, real_type& v ) const;
    void get_value( const std::string& name, bool_type& v ) const;
    void get_value( const std::string& name, string_type& v ) const;
    void get_value( const std::string& name, sprite& v ) const;
    void get_value( const std::string& name, animation& v ) const;
    void get_value( const std::string& name, item_reference_type& v ) const;

    void get_value( const std::string& name, 
		    std::list<integer_type>& v ) const;
    void
    get_value( const std::string& name, std::list<u_integer_type>& v ) const;
    void get_value( const std::string& name, std::list<real_type>& v ) const;
    void get_value( const std::string& name, std::list<bool_type>& v ) const;
    void get_value( const std::string& name, std::list<string_type>& v ) const;
    void get_value( const std::string& name, std::list<sprite>& v ) const;
    void get_value( const std::string& name, std::list<animation>& v ) const;
    void get_value
    ( const std::string& name, std::list<item_reference_type>& v ) const;

    bool has_value( const type_field& f ) const;
    void delete_value( const type_field& f );

    bool get_fixed() const;
    void set_fixed( bool b );

    double get_pos_x() const;
    double get_pos_y() const;
    int get_pos_z() const;
    double get_width() const;
    double get_height() const;
    double get_angle() const;
    int get_gap_x() const;
    int get_gap_y() const;

    bool has_sprite() const;
    const sprite& get_sprite() const;

    void set_position( double x, double y );
    void set_size( double x, double y );

    void compile( compiled_file& f,
                  const std::map<std::string, unsigned int>& id_to_int ) const;

    bool check( std::set<std::string>& list_id, std::string& error_msg ) const;

  private:
    bool check_item_references(std::set<std::string>& list_id,
			       std::string& error_msg) const;
    bool check_required_fields(std::string& error_msg) const;

    bool has_sprite_in_sprites() const;
    bool has_sprite_in_animations() const;

    const sprite& get_sprite_in_sprites() const;
    const sprite& get_sprite_in_animations() const;

    void sort_fields( std::list<std::string>& fields ) const;
    void insert_field
    ( const std::string& field_name, std::list<std::string>& fields,
      std::set<std::string>& all_fields ) const;

    void compile_field
    ( compiled_file& f, const type_field& field,
      const std::map<std::string, unsigned int>& id_to_int ) const;
    void compile_field_single
    ( compiled_file& f, const type_field& field,
      const std::map<std::string, unsigned int>& id_to_int ) const;
    void compile_field_list
    ( compiled_file& f, const type_field& field,
      const std::map<std::string, unsigned int>& id_to_int ) const;

    bear::level_code_value::value_type
    get_code_value( const type_field& field ) const;

    template<typename MapType>
    void
    copy_field_names( const MapType& m, std::set<std::string>& fields ) const;

    template<typename T>
    void compile_list( compiled_file& f, const std::list<T>& v ) const;

  private:
    /** \brief The name of the class I am an instance of. */
    std::string m_class_name;

    /** \brief Tell if the item is fixed. */
    bool m_fixed;

    /** \brief The identifier of this item, for references with other items. */
    std::string m_id;

    /** \brief The values of the integer fields. */
    std::map<std::string, integer_type> m_int;

    /** \brief The values of the unsigned integer fields. */
    std::map<std::string, u_integer_type> m_u_int;

    /** \brief The values of the real fields. */
    std::map<std::string, real_type> m_real;

    /** \brief The values of the boolean fields. */
    std::map<std::string, bool_type> m_bool;

    /** \brief The values of the string fields. */
    std::map<std::string, string_type> m_string;

    /** \brief The values of the sprite fields. */
    std::map<std::string, sprite> m_sprite;

    /** \brief The values of the animation fields. */
    std::map<std::string, animation> m_animation;

    /** \brief The values of the item reference fields. */
    std::map<std::string, item_reference_type> m_item_reference;

    /** \brief The values of the integer fields. */
    std::map< std::string, std::list<integer_type> > m_int_list;

    /** \brief The values of the unsigned integer fields. */
    std::map< std::string, std::list<u_integer_type> > m_u_int_list;

    /** \brief The values of the real fields. */
    std::map< std::string, std::list<real_type> > m_real_list;

    /** \brief The values of the boolean fields. */
    std::map< std::string, std::list<bool_type> > m_bool_list;

    /** \brief The values of the string fields. */
    std::map< std::string, std::list<string_type> > m_string_list;

    /** \brief The values of the sprite fields. */
    std::map< std::string, std::list<sprite> > m_sprite_list;

    /** \brief The values of the animation fields. */
    std::map< std::string, std::list<animation> > m_animation_list;

    /** \brief The values of the item reference fields. */
    std::map< std::string, std::list<item_reference_type> >
    m_item_reference_list;

  }; // class item_instance
} // namespace bf

#include "bf/impl/item_instance.tpp"

#endif // __BF_ITEM_INSTANCE_HPP__
